/*
 * @las3LoadControlData.java Version 1.1 02/24/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratControlStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadControlData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 02/24/2010
 *  @author  John R. Victorine
 */

public class las3LoadControlData
{
  public static final int _NONE    = las3.las3Constants._NONE;
  // Plot Scale Depth Values
  public static final int _IQ_KGS  = 0;  // Profile Saved Data Indicator
  public static final int _IQSTRT  = 1;  // Starting Display Depth
  public static final int _IQSTOP  = 2;  // Ending Display Depth
  public static final int _IQSCLE  = 3;  // Scale Feet/Inch
  // Track Presentation Settings
  public static final int _IQGRP   = 4;  // LAS, ROCK or HORIZON
  public static final int _IQORDR  = 5;  // DEFAULT or MODIFIED
  // Shale Gamma Ray Values for Colorlith tracks
  public static final int _IQSHMIN = 6;  // Gamma Ray Minimum Value (Default)
  public static final int _IQSHLY  = 7;  // Gamma Ray Shaly Value (Default)
  public static final int _IQSH    = 8;  // Gamma Ray Shale Value (Default)
  public static final int _IQSHHOT = 9;  // Gamma Ray Hot Shale Value (Default)
  public static final int _IQSHMAX = 10; // Gamma Ray Maximum Value (Default)
  // Thin Porosity Track
  public static final int _IQTHN   = 11; // Thin Porosity Track Curve Mnemonic
  // Conductivity Image Track
  public static final int _IQOHM   = 12; // Conductivity Image Track Curve Mnemonic
  public static final int _IQOHM_L = 13; // Conductivity Image Track Minimum Value
  public static final int _IQOHM_U = 14; // Conductivity Image Track Maximum Value
  // Porosiity Image Track
  public static final int _IQPHI   = 15; // Porosity Image Track Curve Mnemonic
  public static final int _IQPHI_L = 16; // Porosity Image Track Minimum Value
  public static final int _IQPHI_U = 17; // Porosity Image Track Maximum Value
  // Gamma Ray Count to Gamma Ray API Conversion Values
  public static final int _GRNL    = 18; // % GR Count Lower Limit
  public static final int _GRNU    = 19; // % GR Count Upper Limit
  public static final int _GRL     = 20; // GR (API) Lower Limit
  public static final int _GRU     = 21; // GR (API) Upper Limit
  // Neutron Counts to Neutron Porosity Conversion Values
  public static final int _NEUTL   = 22; // % Neutron Count Upper PHI Limit
  public static final int _NEUTU   = 23; // % Neutron Count Lower PHI Limit
  public static final int _NPHIL   = 24; // Neutron Porosity Lower Limit
  public static final int _NPHIU   = 25; // Neutron Porosity Upper Limit

  public static final int _PARAM_ROWS  = 26;
  public static final int _PARAM_COLS  = 4;
  public static final String PARAM[][] = {
 // Mnemonic  Description                                  Units  Format
    { "IQKGS",   "Profile Web App Saved Data Indicator",       "",        ""  },
    // Plot Scale Depth Values
    { "IQSTRT",  "Starting Display Depth",                     "F",       "F" },
    { "IQSTOP",  "Ending Display Depth",                       "F",       "F" },
    { "IQSCLE",  "Plot Scale Feet/Inch",                       "FT/IN",   "F" },
    // Track Presentation Settings
    { "IQGRP",   "LAS, ROCK or HORIZON",                       "",        "S" },
    { "IQORDR",  "DEFAULT or MODIFIED",                        "",        "S" },
    // Shale Gamma Ray Values for Colorlith tracks
    { "IQSHMIN", "Gamma Ray Minimum Value (Default)",          "API",     "F" },
    { "IQSHLY",  "Gamma Ray Shaly Value (Default)",            "API",     "F" },
    { "IQSH",    "Gamma Ray Shale Value (Default)",            "API",     "F" },
    { "IQSHHOT", "Gamma Ray Hot Shale Value (Default)",        "API",     "F" },
    { "IQSHMAX", "Gamma Ray Maximum Value (Default)",          "API",     "F" },
    // Thin Porosity Track
    { "IQTHN",   "Thin Porosity Track Curve Mnemonic",         "",        "S" },
    // Conductivity Image Track
    { "IQOHM",   "Conductivity Image Track Curve Mnemonic",    "",        "S" },
    { "IQOHM_L", "Conductivity Image Track Minimum Value",     "MMHO/M",  "F" },
    { "IQOHM_U", "Conductivity Image Track Maximum Value",     "MMHO/M",  "F" },
    // Porosiity Image Track
    { "IQPHI",   "Porosity Image Track Curve Mnemonic",        "",        "S" },
    { "IQPHI_L", "Porosity Image Track Minimum Value",         "PU",      "F" },
    { "IQPHI_U", "Porosity Image Track Maximum Value",         "PU",      "F" },
    // Gamma Ray Count to Gamma Ray API Conversion Values
    { "GRNL",  "% GR Count Lower Limit",                       "%",       "F" },
    { "GRNU",  "% GR Count Upper Limit",                       "%",       "F" },
    { "GRL",   "GR (API) Lower Limit",                         "API",     "F" },
    { "GRU",   "GR (API) Upper Limit",                         "API",     "F" },
    // Neutron Counts to Neutron Porosity Conversion Values
    { "NEUTL", "% Neutron Count Upper PHI Limit",              "%",       "F" },
    { "NEUTU", "% Neutron Count Lower PHI Limit",              "%",       "F" },
    { "NPHIL", "Neutron Porosity Lower Limit",                 "PU",      "F" },
    { "NPHIU", "Neutron Porosity Upper Limit",                 "PU",      "F" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  public static final int _SOURCE  = iqstrat.iqstratTracksStruct._SOURCE;
  public static final int _TYPE    = iqstrat.iqstratTracksStruct._TYPE;
  public static final int _DESCR   = iqstrat.iqstratTracksStruct._DESCRIPTION;
  public static final int _STATUS  = iqstrat.iqstratTracksStruct._STATUS;

  /** Method getData()
   * <p> This method will return the Control Data Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP = the LAS 3 Data Structure
   * @return st  = the Control Data Structure
   */

  public static iqstratControlStruct getData( las3Struct stP )
  {
    int i,j,k;
    iqstratControlStruct st     = new iqstratControlStruct();
    int                  iParam = -1;
    int                  iValue = 0;
    String               sData  = "";

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][0]))
          {
            iParam = j;

            switch (iParam)
            {
              case _NONE:
                break;

                // Required Fields

              case _IQ_KGS:
                st.sKGS = new String("YES");
                break;
              case  _IQSTRT:  // Starting Display Depth
                if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                  st.depthStart =
                      cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                break;
              case  _IQSTOP:  // Ending Display Depth
                if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                  st.depthEnd =
                      cmn.cmnString.stringToDouble(stP.sParams[i][_VALUE]);
                break;
              case  _IQSCLE:  // Scale Feet/Inch
                if (cmn.cmnString.isNumeric(stP.sParams[i][_VALUE]))
                {
                  for (k=0; k<iqstrat.iqstratTracksStruct.SCALE.length; k++)
                  {
                    iValue = cmn.cmnString.stringToInt(stP.sParams[i][_VALUE]);
                    if (iValue == iqstrat.iqstratTracksStruct.SCALE[k])
                      st.iScale = k;
                  }
                }
                break;
              // Track Presentation Settings
              case  _IQGRP:  // LAS, ROCK or HORIZON
                sData = new String(stP.sParams[i][_VALUE]);

                if (sData.equals("ALL"))
                  st.iConfig = iqstrat.iqstratControlStruct._C_ALL;

                if (sData.equals("LAS"))
                  st.iConfig = iqstrat.iqstratControlStruct._C_LAS;

                if (sData.equals("ROCK"))
                  st.iConfig = iqstrat.iqstratControlStruct._C_CORE;

                if (sData.equals("HORIZON"))
                  st.iConfig = iqstrat.iqstratControlStruct._C_HORIZON;
                break;
              case  _IQORDR:  // DEFAULT or MODIFIED
                sData     = new String(stP.sParams[i][_VALUE]);
                st.iPanel = iqstrat.iqstratControlStruct._DEFAULT_ORDER;
                if (sData.equals("MODIFIED"))
                  st.iPanel = iqstrat.iqstratControlStruct._MODIFY_ORDER;
                break;
                // Shale Gamma Ray Values for Colorlith tracks
              case  _IQSHMIN:  // Gamma Ray Minimum Value (Default)
              case  _IQSHLY:   // Gamma Ray Shaly Value (Default)
              case  _IQSH:     // Gamma Ray Shale Value (Default)
              case  _IQSHHOT:  // Gamma Ray Hot Shale Value (Default)
              case  _IQSHMAX:  // Gamma Ray Maximum Value (Default)
                break;
                // Thin Porosity Track
              case  _IQTHN:  // Thin Porosity Track Curve Mnemonic
                sData    = new String(stP.sParams[i][_VALUE]);
                st.iThin = iqstrat.iqstratControlStruct._YES;

                if (sData.equals("NPHI"))
                  st.iTHINPHI = iqstrat.iqstratTracksStruct._THIN_NPHI;

                if (sData.equals("AVERAGE"))
                  st.iTHINPHI = iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI;

                if (sData.equals("DPHI"))
                  st.iTHINPHI = iqstrat.iqstratTracksStruct._THIN_DPHI;

                if (sData.equals("SPHI"))
                  st.iTHINPHI = iqstrat.iqstratTracksStruct._THIN_SPHI;
                break;

                // Conductivity Image Track
              case  _IQOHM:  // Conductivity Image Track Curve Mnemonic
                sData   = new String(stP.sParams[i][_VALUE]);
                st.iOHM = iqstrat.iqstratControlStruct._YES;
                for (k=0; k<las.lasStandardTools.TOTAL; k++)
                {
                  if (sData.equals(las.lasStandardTools.LAS_TOOLS[k][1]))
                    st.iOHMTool = k;
                }
                break;
              case  _IQOHM_L:  // Conductivity Image Track Minimum Value
                sData   = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric(sData))
                  st.dOHMMin = cmn.cmnString.stringToDouble(sData);
                break;
              case  _IQOHM_U:  // Conductivity Image Track Maximum Value
                sData   = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric(sData))
                  st.dOHMMax = cmn.cmnString.stringToDouble(sData);
                break;

                // Porosiity Image Track
              case  _IQPHI:  // Porosity Image Track Curve Mnemonic
                sData   = new String(stP.sParams[i][_VALUE]);
                st.iPHI = iqstrat.iqstratControlStruct._YES;
                for (k=0; k<las.lasStandardTools.TOTAL; k++)
                {
                  if (sData.equals(las.lasStandardTools.LAS_TOOLS[k][1]))
                    st.iPHITool = k;
                }
                break;
              case  _IQPHI_L:  // Porosity Image Track Minimum Value
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                 st.dPHIMin = cmn.cmnString.stringToDouble( sData );
               break;
              case  _IQPHI_U:  // Porosity Image Track Maximum Value
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dPHIMax = cmn.cmnString.stringToDouble( sData );
                break;

              case _GRNL:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dGRNMin = cmn.cmnString.stringToDouble( sData );
                break;
              case _GRNU:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dGRNMax = cmn.cmnString.stringToDouble( sData );
                break;
              case _GRL:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dGRCMin = cmn.cmnString.stringToDouble( sData );
                break;
              case _GRU:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dGRCMax = cmn.cmnString.stringToDouble( sData );
                break;

              case _NEUTL:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dNEUTMin = cmn.cmnString.stringToDouble( sData );
                break;
              case _NEUTU:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dNEUTMax = cmn.cmnString.stringToDouble( sData );
                break;
              case _NPHIL:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dNPHICMin = cmn.cmnString.stringToDouble( sData );
                break;
              case _NPHIU:
                sData = new String(stP.sParams[i][_VALUE]);
                if (cmn.cmnString.isNumeric( sData ))
                  st.dNPHICMax = cmn.cmnString.stringToDouble( sData );
                break;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Control Data Structure
   * @param  dNull      = LAS 3 Null Value
   * @param  st         = the LAS 3 Data Structure
   * @param  stCMN      = Common LAS Curve List Structure.
   * @param  stControl  = the Control Data Structure
   * @return stControl2 = the Control Data Structure
   */

  public static iqstratControlStruct loadData( double dNull,
                                               las3Struct st,
                                               cmnLASCurveListStruct stCMN,
                                               iqstratControlStruct  stControl )
  {
    iqstratControlStruct stControl2 = null;
    String            sKEY    = "0";
    int               iTool   = -1;

    if ((st != null) && (stControl != null) && (stCMN != null))
    {
      stControl2 = iqstrat.iqstratControlUtility.copy( stControl );

      stControl.delete();
      stControl = null;

      if (stControl2 == null)
        stControl2 = new iqstratControlStruct();

      if (st.iCurveRows > 0)
      {
        stControl2.iTracks = new int[st.iRows];
        stControl2.iCount  = 0;

        for (int i=0; i<st.iRows; i++)
        {
          for (int j=0; j<st.iColumns; j++)
          {
            iTool = -1;
            for (int k=0; k<stCMN.iCount; k++)
            {
              if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                      stCMN.stItem[k].sMnemonic))
              {
                iTool = k;
              }
            }

            if (iTool > -1)
            {
              stControl2 = addControlData( dNull,
                                           stCMN.stItem[iTool].sMnemonic,
                                           st.sData[i][j],
                                           stControl2 );
            }
          }
        }
      }
    }

    return (stControl2);
  }

  /** Method addControlData()
   * <p> This method will add the LAS 3 Data Structure to the
   *     Control Data Structure.
   * @param  dNULL     = LAS 3 Null Value
   * @param  sMnemonic = Mnemonic
   * @param  sData     = The data value for the mnemonic
   * @param  st        = the Control Data Structure
   * @return st        = the Control Data Structure
   */

  public static iqstratControlStruct addControlData( double dNULL,
                                                     String sMnemonic,
                                                     String sData,
                                                     iqstratControlStruct st )
  {
    int    iRow  = iqstrat.iqstratControlStandardTools.getRow(
                     iqstrat.iqstratControlStandardTools._C_MNEM, sMnemonic );
    String sTemp = "";
    String sNull = new String(""+dNULL);

    if (st != null)
    {
      switch (iRow)
      {
        case iqstrat.iqstratControlStandardTools._IQ_SRC: // Source
          break;
        case iqstrat.iqstratControlStandardTools._IQ_TYPE: // Type
          break;
        case iqstrat.iqstratControlStandardTools._IQ_DESC: // Description
          for (int i=0; i<iqstrat.iqstratTracksStruct.TOTAL_TRACKS; i++)
          {
            if (sData.equals(iqstrat.iqstratTracksStruct.TRACK[i][_DESCRIPTION]))
            {
              st.iTracks[st.iCount] = i;
              st.iCount++;
            }
          }
          break;
      }
    }

    return (st);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Control Data Structure.
   * @param  st    = the Control Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( iqstratControlStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_CONTROL;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Control Data Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( iqstratControlStruct st, las3Struct stT )
  {
    int i,j;
    String sData = "";

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_UNIT]        = new String( PARAM[i][2] );
        stT.sParams[i][_FORMAT]      = new String( PARAM[i][3] );

        switch (i)
        {
          case _IQ_KGS:
            break;
          case _IQSTRT:  // Starting Display Depth
            stT.sParams[i][_VALUE] = new String( "" + st.depthStart );
            break;
          case _IQSTOP:  // Ending Display Depth
            stT.sParams[i][_VALUE] = new String( "" + st.depthEnd );
            break;
          case _IQSCLE:  // Scale Feet/Inch
            stT.sParams[i][_VALUE] = new String(
                "" + iqstrat.iqstratTracksStruct.SCALE[st.iScale] );
            break;
            // Track Presentation Settings
          case _IQGRP:   // LAS, ROCK or HORIZON
            switch (st.iConfig)
            {
              case iqstrat.iqstratControlStruct._C_ALL:
                sData = new String("ALL");
                break;
              case iqstrat.iqstratControlStruct._C_LAS:
                sData = new String("LAS");
                break;
              case iqstrat.iqstratControlStruct._C_CORE:
                sData = new String("ROCK");
                break;
              case iqstrat.iqstratControlStruct._C_HORIZON:
                sData = new String("HORIZON");
                break;
            }

            stT.sParams[i][_VALUE] = new String( sData );
            break;
          case _IQORDR:  // DEFAULT or MODIFIED
            sData = "DEFAULT";
            if (st.iPanel != iqstrat.iqstratControlStruct._DEFAULT_ORDER)
              sData = "MODIFIED";

            stT.sParams[i][_VALUE] = new String( sData );
            break;
            // Shale Gamma Ray Values for Colorlith tracks
          case _IQSHMIN: // Gamma Ray Minimum Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.GAMMA_MIN );
            break;
          case _IQSHLY:  // Gamma Ray Shaly Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALY );
            break;
          case _IQSH:    // Gamma Ray Shale Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALE );
            break;
          case _IQSHHOT: // Gamma Ray Hot Shale Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.SHALE_HOT );
            break;
          case _IQSHMAX: // Gamma Ray Maximum Value (Default)
            stT.sParams[i][_VALUE] = new String(
                      "" + iqstrat.iqstratShaleStruct.GAMMA_MAX );
            break;
            // Thin Porosity Track
          case _IQTHN:   // Thin Porosity Track Curve Mnemonic
            if (st.iThin == cmn.cmnStruct._YES)
            {
              switch (st.iTHINPHI)
              {
                case iqstrat.iqstratTracksStruct._THIN_NPHI:
                  sData = "NPHI";
                  break;
                case iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI:
                  sData = "AVERAGE";
                  break;
                case iqstrat.iqstratTracksStruct._THIN_SPHI:
                  sData = "SPHI";
                  break;
                case iqstrat.iqstratTracksStruct._THIN_DPHI:
                  sData = "DPHI";
                  break;
                default:
                  sData = "";
                  break;
              }

              stT.sParams[i][_VALUE] = new String( sData );
            }
            break;
            // Conductivity Image Track
          case _IQOHM:   // Conductivity Image Track Curve Mnemonic
            if (st.iOHM == cmn.cmnStruct._YES)
            {
              sData = las.lasStandardTools.LAS_TOOLS[st.iOHMTool][1];
              stT.sParams[i][_VALUE]       = new String( sData );
              sData = las.lasStandardTools.LAS_TOOLS[st.iOHMTool][2];
              stT.sParams[i][_DESCRIPTION] = new String( sData );
            }
            break;
          case _IQOHM_L: // Conductivity Image Track Minimum Value
            if (st.iOHM == cmn.cmnStruct._YES)
            {
              stT.sParams[i][_VALUE] = new String( "" + st.dOHMMin );
            }
            break;
          case _IQOHM_U: // Conductivity Image Track Maximum Value
            if (st.iOHM == cmn.cmnStruct._YES)
            {
              stT.sParams[i][_VALUE] = new String( "" + st.dOHMMax );
            }
            break;
            // Porosiity Image Track
          case _IQPHI:   // Porosity Image Track Curve Mnemonic
            if (st.iPHI == cmn.cmnStruct._YES)
            {
              sData = las.lasStandardTools.LAS_TOOLS[st.iPHITool][1];
              stT.sParams[i][_VALUE]       = new String( sData );
              sData = las.lasStandardTools.LAS_TOOLS[st.iPHITool][2];
              stT.sParams[i][_DESCRIPTION] = new String( sData );
            }
            break;
          case _IQPHI_L: // Porosity Image Track Minimum Value
            if (st.iPHI == cmn.cmnStruct._YES)
            {
              stT.sParams[i][_VALUE] = new String( "" + st.dPHIMin );
            }
            break;
          case _IQPHI_U: // Porosity Image Track Maximum Value
            if (st.iPHI == cmn.cmnStruct._YES)
            {
              stT.sParams[i][_VALUE] = new String( "" + st.dPHIMax );
            }
            break;


          case _GRNL:
            stT.sParams[i][_VALUE] = new String( "" + st.dGRNMin );
            break;
          case _GRNU:
            stT.sParams[i][_VALUE] = new String( "" + st.dGRNMax );
            break;
          case _GRL:
            stT.sParams[i][_VALUE] = new String( "" + st.dGRCMin );
            break;
          case _GRU:
            stT.sParams[i][_VALUE] = new String( "" + st.dGRCMax );
            break;

          case _NEUTL:
            stT.sParams[i][_VALUE] = new String( "" + st.dNEUTMin );
            break;
          case _NEUTU:
            stT.sParams[i][_VALUE] = new String( "" + st.dNEUTMax );
            break;
          case _NPHIL:
            stT.sParams[i][_VALUE] = new String( "" + st.dNPHICMin );
            break;
          case _NPHIU:
            stT.sParams[i][_VALUE] = new String( "" + st.dNPHICMax );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the Control Data Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( iqstratControlStruct st,
                                      las3Struct           stT,
                                      double               dNull )
  {
    int    i,j;
    int    iCnt  = 0;
    String sData = "";

    if ((st != null) && (stT != null))
    {
      if (st.iTracks != null)
      {
        // Build Curve Definition Section

        stT.iCurveRows = iqstrat.iqstratControlStandardTools.TOTAL; // Total rows
        stT.iCurveCols = las3.las3Struct._TOTAL;                     // Total cols
        stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

        for (i=0; i<iqstrat.iqstratControlStandardTools.TOTAL; i++)
        {
          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              iqstrat.iqstratControlStandardTools.CURVES[i][
                iqstrat.iqstratControlStandardTools._C_MNEM] );

          stT.sCurves[iCnt][_UNIT] = new String( "" );

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              iqstrat.iqstratControlStandardTools.CURVES[i][
                iqstrat.iqstratControlStandardTools._C_NAME] );

          stT.sCurves[iCnt][_FORMAT] = new String( "S" );

          iCnt++;
        }

        // Build the Data Section

        stT.iRows    = st.iTracks.length;  // Total Number of Rows
        stT.iColumns = stT.iCurveRows;     // Total Number of Columns
        stT.sData    = new String[stT.iRows][stT.iColumns];

        // Load the Data Columns from the Control Data Structure

        for (i=0; i<st.iTracks.length; i++)
        {
          if (st.iTracks[i] != -1)
          {
            stT.sData[i][0] = new String(
                iqstrat.iqstratTracksStruct.TRACK[st.iTracks[i]][_SOURCE]);
            stT.sData[i][1] = new String(
                iqstrat.iqstratTracksStruct.TRACK[st.iTracks[i]][_TYPE]);
            stT.sData[i][2] = new String(
                iqstrat.iqstratTracksStruct.TRACK[st.iTracks[i]][_DESCR]);
          }
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 02/24/2010
 *  @author  John Victorine
 */
