/*
 * @las3LoadBrineData.java Version 1.1 06/02/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import iqstrat.iqstratHeadersStruct;
import brine.brineListStruct;
import brine.brineStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadBrineData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 06/02/2011
 *  @author  John R. Victorine
 */

public class las3LoadBrineData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  // ~PARAMETER Mnemonics in Core Data

  public static final int _IQ_KGS    = 0;  // Profile Saved Data Indicator
  public static final int _SOURCE    = 1;  // Source of Data
  public static final int _RECOVER   = 2;  // Date Recoverd
  public static final int _TOP       = 3;  // Top Depth of Sample
  public static final int _BASE      = 4;  // Base Depth of Sample
  public static final int _FORMATION = 5;  // Primary Formation
  public static final int _COMPANY   = 6;  // Analyzing Company
  public static final int _ANALYZED  = 7;  // Analyzing Date

  public static final int _PARAM_ROWS  = 8;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 //   Mnemonic  Description          Variations
    { "IQKGS",  "Profile Web App Saved Data Indicator",  "", "", "" },
    { "B_SRS",  "Source",            "", "", "" }, //   {S}
    { "B_DT",   "Recovery Date",     "", "", "" }, //   {D}
    { "B_TP",   "Top Depth",         "", "", "" }, //   {F}
    { "B_BS",   "Base Depth",        "", "", "" }, //   {F}
    { "B_FM",   "Primary Formation", "", "", "" }, //   {S}
    { "B_AC",   "Analyzing Company", "", "", "" }, //   {S}
    { "B_AD",   "Analysis Date",     "", "", "" }  //   {D}
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  public static final int _C_KEY  = brine.brineStandardTools._C_KEY;
  public static final int _C_MNEM = brine.brineStandardTools._C_MNEM;
  public static final int _C_DESC = brine.brineStandardTools._C_DESC;
  public static final int _C_UNIT = brine.brineStandardTools._C_UNIT;
  public static final int _C_TYPE = brine.brineStandardTools._C_TYPE;
  public static final int _C_VAL  = brine.brineStandardTools._C_VAL;
  public static final int _C_MOL  = brine.brineStandardTools._C_MOL;
  public static final int _C_SEL  = brine.brineStandardTools._C_SEL;

  /** Method getData()
   * <p> This method will return the Brine List Structure from the
   *     LAS 3 Data List Structure.
   * @param  stP   = the LAS 3 Data Structure
   * @return st    = the Brine List Structure
   */

  public static brineListStruct getData( las3Struct stP )
  {
    int i,j,k;
    brineListStruct st     = new brineListStruct();
    int             iParam = -1;

    if (stP != null)
    {
      for (i=0; i<stP.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stP.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _SOURCE:    // Core Source
                  st.source = new String( stP.sParams[i][_VALUE] );
                  break;
                case _RECOVER:   // Recovery Date
                  st.sRecovery = new String( stP.sParams[i][_VALUE] );
                  break;
                case _TOP:       // Core Top Depth
                  st.depthStart = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _BASE:      // Core Base Depth
                  st.depthEnd = stringToDouble( stP.sParams[i][_VALUE] );
                  break;
                case _FORMATION: // Primary Formation
                  st.sFormation = new String( stP.sParams[i][_VALUE] );
                  break;
                case _COMPANY:   // Analyzing Company
                  st.sCompany = new String( stP.sParams[i][_VALUE] );
                  break;
                case _ANALYZED:  // Analysis Date
                  st.sAnalyzed = new String( stP.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the Brine List
   *     Structure.
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stRock  = the Brine List Structure
   * @return stRock2 = the Brine List Structure
   */

  public static brineListStruct loadData( double                dNull,
                                          las3Struct            st,
                                          cmnLASCurveListStruct stCMN,
                                          brineListStruct       stBrine )
  {
    double depthStart  = 0.0;
    double depthEnd    = 0.0;

    if ((st != null) && (stBrine != null) && (stCMN != null))
    {
      stBrine = findCurves( stCMN, stBrine );
      stBrine = addBrineData( dNull, st, stBrine );

      stBrine = brine.brineUtility.computeDepthRange( stBrine );
      stBrine = brine.brineUtility.bubbleSort( stBrine, 0 );
      stBrine = brine.brineUtility.addBaseDepth( stBrine );
    }

    return (stBrine);
  }

  /** Method findCurves()
   * <p> This method will identify the Curves and the Brine Variables.
   * @param  stCurves = The LAS Curve Data List Structure
   * @param  st       = the Brine List Structure
   * @return st       = the Brine List Structure
   */

  public static brineListStruct findCurves( cmnLASCurveListStruct stCurves,
                                            brineListStruct       st )
  {
    if ((st != null) && (stCurves != null))
    {
      // Brine Data Identifiers

      st._KEY    = -1;  // KEY

      // AGE
      st._FORM   = -1;  // Primary Formation
      st._AGE    = -1;  // Age of Formation

      // Depth
      st._TOP    = -1;  // Top Depth
      st._BASE   = -1;  // Base Depth

      // Other Measured Well Data
      st._SPGR   = -1;  // Specific Gravity
      st._SIGMA  = -1;  // Specific Conductivity
      st._PH     = -1;  // PH
      st._DEG    = -1;  // Degree F
      st._OHM    = -1;  // Resistivity (Rw)
      st._OHM75  = -1; // Resistivity at 75 deg
      st._OHME   = -1; // Estimated Rw

      // .. Alkali Metal
      st._LI     = -1; // Lithium
      st._NA     = -1; // Sodium
      st._K      = -1; // Potassium
      st._RB     = -1; // Rubidium
      st._CS     = -1; // Cesium
      st._NA_K   = -1; // Sodium/Potasium

      // ..  Alkali Earth Metal
      st._BE     = -1; // Beryllium
      st._MG     = -1; // Magnesium
      st._CA     = -1; // Calcium
      st._SR     = -1; // Strontium
      st._BA     = -1; // Barium

      // ..  VIB
      st._CR_II  = -1; // Chromium (II)
      st._CR_III = -1; // Chromium (iii)

      // ..  VIIB
      st._MN_II  = -1; // Manganese (II)
      st._MN_III = -1; // Manganese (IV)

      // ..  VIIIB
      st._FE_II  = -1; // Iron (II)
      st._FE_III = -1; // Iron (III)
      st._CO_II  = -1; // Cobalt (II)
      st._CO_III = -1; // Cobalt (III)
      st._NI_II  = -1; // Nickel (II)
      st._NI_III = -1; // Nickel (III)

      // ..  IB
      st._CU_I   = -1; // Copper (I)
      st._CU_II  = -1; // Copper (II)
      st._AG     = -1; // Silver
      st._AU     = -1; // Gold
      st._AU_III = -1; // Gold (III)

      // ..  IIB
      st._ZN     = -1; // Zinc
      st._CD     = -1; // Cadmium
      st._HG     = -1; // Mercury (I)
      st._HG_II  = -1; // Mercury (II)

      // ..  IIIA
      st._AL     = -1; // Aluminium

      //..
      st._SB_III = -1; // Antimonous(III)
      st._SB_V   = -1; // Antimonic(V)
      st._BI_III = -1; // Bismuthous(III)
      st._BI_V   = -1; // Bismuth(V)

      // ..  IVA
      st._SN_II  = -1; // Tin (II)
      st._SN_IV  = -1; // Tin (IV)
      st._PB_II  = -1; // Lead (II)
      st._PB_IV  = -1; // Lead (IV)

      // .. Other
      st._NH4    = -1; // Ammonium

      // Common Anions

      // ..Halogen
      st._F      = -1; // Flouride
      st._CL     = -1; // Chloride
      st._BR     = -1; // Bromide
      st._I      = -1; // Iodide

      st._OH     = -1; // Hydroxide

      st._BO3    = -1; // Borate
      st._CO3    = -1; // Carbonate
      st._HCO3   = -1; // Bicarbonate

      st._CLO    = -1; // Hypochlorite
      st._CLO2   = -1; // Chlorite
      st._CLO3   = -1; // Chlorate
      st._CLO4   = -1; // Per chlorate

      st._CN     = -1; // Cyanide
      st._NCO    = -1; // Cyanate
      st._OCN    = -1; // Isocyanate
      st._SCN    = -1; // Thiocyanate

      st._N      = -1; // Nitride
      st._N3     = -1; // Azide
      st._NO2    = -1; // Nitrite
      st._NO3    = -1; // Nitrate

      st._CRO4   = -1; // Chromium (II)
      st._CR2O7  = -1; // Chromium (III)

      st._MNO4   = -1; // Permanganate

      st._P      = -1; // Phosphide
      st._PO3    = -1; // Phosphite
      st._PO4    = -1; // Phosphate
      st._HPO4   = -1; // Monohydrogen Phosphate
      st._H2PO4  = -1; // Dihydrogen Phosphate

      st._AS     = -1; // Arsinide
      st._SE     = -1; // Selenide

      st._S      = -1; // Sulfide
      st._HS     = -1; // Hydrogen Sulfide
      st._SO3    = -1; // Sulphite
      st._HSO3   = -1; // Hydrogen Sulphite
      st._S2O3   = -1; // Thiosulphate
      st._SO4    = -1; // Sulfate
      st._HSO4   = -1; // Bisulfate

      st._SOLID  = -1; // Total Solids
      st._TDS    = -1; // Total Solids Computed

      if (stCurves != null)
      {
        for (int i=0; i<stCurves.iCount; i++)
        {
          // -- Depth information

          // Get Top of Rock Depth Log Curve Number
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._KEY][0]))
          {
            if (st._KEY == -1) st._KEY = stCurves.stItem[i].iCurveNumber;
          }

          // AGE
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._FORM][0]))
          {
            if (st._FORM == -1) st._FORM = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AGE][0]))
          {
            if (st._AGE == -1) st._AGE = stCurves.stItem[i].iCurveNumber;
          }

          // Depth
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._TOP][0]))
          {
            if (st._TOP == -1) st._TOP = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BASE][0]))
          {
            if (st._BASE == -1) st._BASE = stCurves.stItem[i].iCurveNumber;
          }

          // Other Measured Well Data
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SPGR][0]))
          {
            if (st._SPGR == -1) st._SPGR = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SIGMA][0]))
          {
            if (st._SIGMA == -1) st._SIGMA = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._PH][0]))
          {
            if (st._PH == -1) st._PH = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._DEG][0]))
          {
            if (st._DEG == -1) st._DEG = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._OHM][0]))
          {
            if (st._OHM == -1) st._OHM = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._OHM75][0]))
          {
            if (st._OHM75 == -1) st._OHM75 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._OHME][0]))
          {
            if (st._OHME == -1) st._OHME = stCurves.stItem[i].iCurveNumber;
          }

          // .. Alkali Metal
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._LI][0]))
          {
            if (st._LI == -1) st._LI = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NA][0]))
          {
            if (st._NA == -1) st._NA = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._K][0]))
          {
            if (st._K == -1) st._K = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._RB][0]))
          {
            if (st._RB == -1) st._RB = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CS][0]))
          {
            if (st._CS == -1) st._CS = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NA_K][0]))
          {
            if (st._NA_K == -1) st._NA_K = stCurves.stItem[i].iCurveNumber;
          }

          // ..  Alkali Earth Metal
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BE][0]))
          {
            if (st._BE == -1) st._BE = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._MG][0]))
          {
            if (st._MG == -1) st._MG = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CA][0]))
          {
            if (st._CA == -1) st._CA = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SR][0]))
          {
            if (st._SR == -1) st._SR = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BA][0]))
          {
            if (st._BA == -1) st._BA = stCurves.stItem[i].iCurveNumber;
          }

          // ..  VIB
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CR_II][0]))
          {
            if (st._CR_II == -1) st._CR_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CR_III][0]))
          {
            if (st._CR_III == -1) st._CR_III = stCurves.stItem[i].iCurveNumber;
          }

          // ..  VIIB
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._MN_II][0]))
          {
            if (st._MN_II == -1) st._MN_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._MN_III][0]))
          {
            if (st._MN_III == -1) st._MN_III = stCurves.stItem[i].iCurveNumber;
          }

          // ..  VIIIB
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._FE_II][0]))
          {
            if (st._FE_II == -1) st._FE_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._FE_III][0]))
          {
            if (st._FE_III == -1) st._FE_III = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CO_II][0]))
          {
            if (st._CO_II == -1) st._CO_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CO_III][0]))
          {
            if (st._CO_III == -1) st._CO_III = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NI_II][0]))
          {
            if (st._NI_II == -1) st._NI_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NI_III][0]))
          {
            if (st._NI_III == -1) st._NI_III = stCurves.stItem[i].iCurveNumber;
          }

          // ..  IB
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CU_I][0]))
          {
            if (st._CU_I == -1) st._CU_I = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CU_II][0]))
          {
            if (st._CU_II == -1) st._CU_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AG][0]))
          {
            if (st._AG == -1) st._AG = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AU][0]))
          {
            if (st._AU == -1) st._AU = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AU_III][0]))
          {
            if (st._AU_III == -1) st._AU_III = stCurves.stItem[i].iCurveNumber;
          }

          // ..  IIB
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._ZN][0]))
          {
            if (st._ZN == -1) st._ZN = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CD][0]))
          {
            if (st._CD == -1) st._CD = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HG][0]))
          {
            if (st._HG == -1) st._HG = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HG_II][0]))
          {
            if (st._HG_II == -1) st._HG_II = stCurves.stItem[i].iCurveNumber;
          }

          // ..  IIIA
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AL][0]))
          {
            if (st._AL == -1) st._AL = stCurves.stItem[i].iCurveNumber;
          }

          //..
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SB_III][0]))
          {
            if (st._SB_III == -1) st._SB_III = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SB_V][0]))
          {
            if (st._SB_V == -1) st._SB_V = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BI_III][0]))
          {
            if (st._BI_III == -1) st._BI_III = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BI_V][0]))
          {
            if (st._BI_V == -1) st._BI_V  = stCurves.stItem[i].iCurveNumber;
          }

          // ..  IVA
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SN_II][0]))
          {
            if (st._SN_II == -1) st._SN_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SN_IV][0]))
          {
            if (st._SN_IV == -1) st._SN_IV = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._PB_II][0]))
          {
            if (st._PB_II == -1) st._PB_II = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._PB_IV][0]))
          {
            if (st._PB_IV == -1) st._PB_IV = stCurves.stItem[i].iCurveNumber;
          }

          // .. Other
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NH4][0]))
          {
            if (st._NH4 == -1) st._NH4 = stCurves.stItem[i].iCurveNumber;
          }

          // Common Anions

          // ..Halogen
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._F][0]))
          {
            if (st._F == -1) st._F = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CL][0]))
          {
            if (st._CL == -1) st._CL = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BR][0]))
          {
            if (st._BR == -1) st._BR = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._I][0]))
          {
            if (st._I == -1) st._I = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._OH][0]))
          {
            if (st._OH == -1) st._OH = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._BO3][0]))
          {
            if (st._BO3 == -1) st._BO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CO3][0]))
          {
            if (st._CO3 == -1) st._CO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HCO3][0]))
          {
            if (st._HCO3 == -1) st._HCO3 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CLO][0]))
          {
            if (st._CLO == -1) st._CLO = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CLO2][0]))
          {
            if (st._CLO2 == -1) st._CLO2 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CLO3][0]))
          {
            if (st._CLO3 == -1) st._CLO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CLO4][0]))
          {
            if (st._CLO4 == -1) st._CLO4 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CN][0]))
          {
            if (st._CN == -1) st._CN = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NCO][0]))
          {
            if (st._NCO == -1) st._NCO = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._OCN][0]))
          {
            if (st._OCN == -1) st._OCN = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SCN][0]))
          {
            if (st._SCN == -1) st._SCN = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._N][0]))
          {
            if (st._N == -1) st._N = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._N3][0]))
          {
            if (st._N3 == -1) st._N3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NO2][0]))
          {
            if (st._NO2 == -1) st._NO2 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._NO3][0]))
          {
            if (st._NO3 == -1) st._NO3 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CRO4][0]))
          {
            if (st._CRO4 == -1) st._CRO4 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._CR2O7][0]))
          {
            if (st._CR2O7 == -1) st._CR2O7 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._MNO4][0]))
          {
            if (st._MNO4 == -1) st._MNO4 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._P][0]))
          {
            if (st._P == -1) st._P = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._PO3][0]))
          {
            if (st._PO3 == -1) st._PO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._PO4][0]))
          {
            if (st._PO4 == -1) st._PO4 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HPO4][0]))
          {
            if (st._HPO4 == -1) st._HPO4 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._H2PO4][0]))
          {
            if (st._H2PO4 == -1) st._H2PO4 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._AS][0]))
          {
            if (st._AS == -1) st._AS = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SE][0]))
          {
            if (st._SE == -1) st._SE = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._S][0]))
          {
            if (st._S == -1) st._S = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HS][0]))
          {
            if (st._HS == -1) st._HS = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SO3][0]))
          {
            if (st._SO3 == -1) st._SO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HSO3][0]))
          {
            if (st._HSO3 == -1) st._HSO3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._S2O3][0]))
          {
            if (st._S2O3 == -1) st._S2O3 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SO4][0]))
          {
            if (st._SO4 == -1) st._SO4 = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._HSO4][0]))
          {
            if (st._HSO4 == -1) st._HSO4 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._SOLID][0]))
          {
            if (st._SOLID == -1) st._SOLID = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            brine.brineStandardTools.CURVES[brine.brineStandardTools._TDS][0]))
          {
            if (st._TDS == -1) st._TDS = stCurves.stItem[i].iCurveNumber;
          }

        }
      }
    }

    return (st);
  }

  /** Method addBrineData()
   * <p> This method will add the LAS 3 Data Structure to
   *     the Brine List Structure
   * @param  dNULL   = LAS 3 Null Value
   * @param  st      = LAS 3 Data Structure
   * @param  stBrine = Brine List Structure
   * @return stBrine = Brine List Structure
   */

  public static brineListStruct addBrineData( double          dNULL,
                                              las3Struct      st,
                                              brineListStruct stBrine )
  {
    double data = 0.0;
    String str  = "";
    String sKEY = new String( cmn.cmnString.UniqueName() );

    if ((stBrine != null) && (st !=null))
    {
      stBrine.iCount = st.iRows;  // Number of Rows
      stBrine.stItem = new brineStruct[st.iRows];

      for (int i=0; i<st.iRows; i++)
      {
        stBrine.stItem[i]         = new brineStruct();
        stBrine.stItem[i].sKID    = "";  // KID
        stBrine.stItem[i].sKEY    = new String( sKEY + "_" + i );

        // AGE
        stBrine.stItem[i].sFORM   = "";  // Primary Formation
        stBrine.stItem[i].sAGE    = "";  // Age of Formation

        // Depth
        stBrine.stItem[i].dTOP    = dNULL;  // Top Depth
        stBrine.stItem[i].dBASE   = dNULL;  // Base Depth

        // Other Measured Well Data
        stBrine.stItem[i].dSPGR   = dNULL;  // Specific Gravity
        stBrine.stItem[i].dSIGMA  = dNULL;  // Specific Conductivity
        stBrine.stItem[i].dPH     = dNULL;  // PH
        stBrine.stItem[i].dDEG    = dNULL;  // Degree F
        stBrine.stItem[i].dOHM    = dNULL;  // Resistivity (Rw)
        stBrine.stItem[i].dOHM75  = dNULL; // Resistivity at 75 deg
        stBrine.stItem[i].dOHME   = dNULL; // Estimated Rw

         // .. Alkali Metal
        stBrine.stItem[i].dLI     = dNULL; // Lithium
        stBrine.stItem[i].dNA     = dNULL; // Sodium
        stBrine.stItem[i].dK      = dNULL; // Potassium
        stBrine.stItem[i].dRB     = dNULL; // Rubidium
        stBrine.stItem[i].dCS     = dNULL; // Cesium
        stBrine.stItem[i].dNA_K   = dNULL; // Sodium/Potasium

        // ..  Alkali Earth Metal
        stBrine.stItem[i].dBE     = dNULL; // Beryllium
        stBrine.stItem[i].dMG     = dNULL; // Magnesium
        stBrine.stItem[i].dCA     = dNULL; // Calcium
        stBrine.stItem[i].dSR     = dNULL; // Strontium
        stBrine.stItem[i].dBA     = dNULL; // Barium

        // ..  VIB
        stBrine.stItem[i].dCR_II  = dNULL; // Chromium (II)
        stBrine.stItem[i].dCR_III = dNULL; // Chromium (III)

        // ..  VIIB
        stBrine.stItem[i].dMN_II  = dNULL; // Manganese (II)
        stBrine.stItem[i].dMN_III = dNULL; // Manganese (III)

        // ..  VIIIB
        stBrine.stItem[i].dFE_II  = dNULL; // Iron (II)
        stBrine.stItem[i].dFE_III = dNULL; // Iron (III)
        stBrine.stItem[i].dCO_II  = dNULL; // Cobalt (II)
        stBrine.stItem[i].dCO_III = dNULL; // Cobalt (III)
        stBrine.stItem[i].dNI_II  = dNULL; // Nickel (II)
        stBrine.stItem[i].dNI_III = dNULL; // Nickel (III)

        // ..  IB
        stBrine.stItem[i].dCU_I   = dNULL; // Copper (I)
        stBrine.stItem[i].dCU_II  = dNULL; // Copper (II)
        stBrine.stItem[i].dAG     = dNULL; // Silver
        stBrine.stItem[i].dAU     = dNULL; // Gold
        stBrine.stItem[i].dAU_III = dNULL; // Gold (III)

        // ..  IIB
        stBrine.stItem[i].dZN     = dNULL; // Zinc
        stBrine.stItem[i].dCD     = dNULL; // Cadmium
        stBrine.stItem[i].dHG     = dNULL; // Mercury (I)
        stBrine.stItem[i].dHG_II  = dNULL; // Mercury (II)

        // ..  IIIA
        stBrine.stItem[i].dAL     = dNULL; // Aluminium

        // ..
        stBrine.stItem[i].dSB_III = dNULL; // Antimonous(III)          (Sb+++)
        stBrine.stItem[i].dSB_V   = dNULL; // Antimonic(V)             (Sb++++)
        stBrine.stItem[i].dBI_III = dNULL; // Bismuthous(III)          (Bi+++)
        stBrine.stItem[i].dBI_V   = dNULL; // Bismuth(V)               (Bi++++)

        // ..  IVA
        stBrine.stItem[i].dSN_II  = dNULL; // Tin (II)
        stBrine.stItem[i].dSN_IV  = dNULL; // Tin (IV)
        stBrine.stItem[i].dPB_II  = dNULL; // Lead (II)
        stBrine.stItem[i].dPB_IV  = dNULL; // Lead (IV)

        // .. Other
        stBrine.stItem[i].dNH4    = dNULL; // Ammonium

        // Common Anions

        // ..Halogen
        stBrine.stItem[i].dF      = dNULL; // Flouride
        stBrine.stItem[i].dCL     = dNULL; // Chloride
        stBrine.stItem[i].dBR     = dNULL; // Bromide
        stBrine.stItem[i].dI      = dNULL; // Iodide

        stBrine.stItem[i].dOH     = dNULL; // Hydroxide

        stBrine.stItem[i].dBO3    = dNULL; // Borate
        stBrine.stItem[i].dCO3    = dNULL; // Carbonate
        stBrine.stItem[i].dHCO3   = dNULL; // Bicarbonate

        stBrine.stItem[i].dCLO    = dNULL; // Hypochlorite
        stBrine.stItem[i].dCLO2   = dNULL; // Chlorite
        stBrine.stItem[i].dCLO3   = dNULL; // Chlorate
        stBrine.stItem[i].dCLO4   = dNULL; // Per chlorate

        stBrine.stItem[i].dCN     = dNULL; // Cyanide
        stBrine.stItem[i].dNCO    = dNULL; // Cyanate
        stBrine.stItem[i].dOCN    = dNULL; // Isocyanate
        stBrine.stItem[i].dSCN    = dNULL; // Thiocyanate

        stBrine.stItem[i].dN      = dNULL; // Nitride
        stBrine.stItem[i].dN3     = dNULL; // Azide
        stBrine.stItem[i].dNO2    = dNULL; // Nitrite
        stBrine.stItem[i].dNO3    = dNULL; // Nitrate

        stBrine.stItem[i].dCRO4   = dNULL; // Chromium (II)
        stBrine.stItem[i].dCR2O7  = dNULL; // Chromium (III)

        stBrine.stItem[i].dMNO4   = dNULL; // Permanganate

        stBrine.stItem[i].dP      = dNULL; // Phosphide
        stBrine.stItem[i].dPO3    = dNULL; // Phosphite
        stBrine.stItem[i].dPO4    = dNULL; // Phosphate
        stBrine.stItem[i].dHPO4   = dNULL; // Monohydrogen Phosphate
        stBrine.stItem[i].dH2PO4  = dNULL; // Dihydrogen Phosphate

        stBrine.stItem[i].dAS     = dNULL; // Arsinide
        stBrine.stItem[i].dSE     = dNULL; // Selenide

        stBrine.stItem[i].dS      = dNULL; // Sulfide
        stBrine.stItem[i].dHS     = dNULL; // Hydrogen Sulfide
        stBrine.stItem[i].dSO3    = dNULL; // Sulphite
        stBrine.stItem[i].dHSO3   = dNULL; // Hydrogen Sulphite
        stBrine.stItem[i].dS2O3   = dNULL; // Thiosulphate
        stBrine.stItem[i].dSO4    = dNULL; // Sulfate
        stBrine.stItem[i].dHSO4   = dNULL; // Bisulfate

        stBrine.stItem[i].dSOLID  = dNULL; // Total Solids
        stBrine.stItem[i].dTDS    = dNULL; // Total Solids Computed

        for (int j=0; j<st.iColumns; j++)
        {
          data = dNULL;
          str  = new String( "" + dNULL );

          //if (cmn.cmnString.isNumeric( st.sData[i][j] ))
            data = stringToDouble( st.sData[i][j].trim(), dNULL );

          if (st.sData[i][j].trim().equals(str))
            str = new String( "" );
          else
            str = new String( st.sData[i][j].trim() );

          // Brine Data Identifiers

//          if (stBrine._KID    == j)  // KID
//            stBrine.stItem[i].sKID = new String( str );
          if (stBrine._KEY    == j)  // KEY
            stBrine.stItem[i].sKEY = new String( str );

          // AGE
          if (stBrine._FORM    == j)  // Primary Formation
            stBrine.stItem[i].sFORM = new String( str );
          if (stBrine._AGE     == j)  // Age of Formation
            stBrine.stItem[i].sAGE = new String( str );

          // -- Depth information

          if (stBrine._TOP     == j) // Top of Rock Depth
            stBrine.stItem[i].dTOP = data;
          if (stBrine._BASE    == j)  // Base Depth
            stBrine.stItem[i].dBASE = data;

          // Other Measured Well Data
          if (stBrine._SPGR    == j)  // Specific Gravity
            stBrine.stItem[i].dSPGR = data;
          if (stBrine._SIGMA   == j)  // Specific Conductivity
            stBrine.stItem[i].dSIGMA = data;
          if (stBrine._PH      == j)  // PH
            stBrine.stItem[i].dPH = data;
          if (stBrine._DEG     == j)  // Degree F
            stBrine.stItem[i].dDEG = data;
          if (stBrine._OHM     == j)  // Resistivity (Rw)
            stBrine.stItem[i].dOHM = data;
          if (stBrine._OHM75   == j) // Resistivity at 75 deg
            stBrine.stItem[i].dOHM75 = data;
          if (stBrine._OHME    == j) // Estimated Rw
            stBrine.stItem[i].dOHME = data;

          // .. Alkali Metal
          if (stBrine._LI      == j) // Lithium
            stBrine.stItem[i].dLI = data;
          if (stBrine._NA      == j) // Sodium
            stBrine.stItem[i].dNA = data;
          if (stBrine._K       == j) // Potassium
            stBrine.stItem[i].dK = data;
          if (stBrine._RB      == j) // Rubidium
            stBrine.stItem[i].dRB = data;
          if (stBrine._CS      == j) // Cesium
            stBrine.stItem[i].dCS = data;
          if (stBrine._NA_K    == j) // Sodium/Potasium
            stBrine.stItem[i].dNA_K = data;

          // ..  Alkali Earth Metal
          if (stBrine._BE      == j) // Beryllium
            stBrine.stItem[i].dBE = data;
          if (stBrine._MG      == j) // Magnesium
            stBrine.stItem[i].dMG = data;
          if (stBrine._CA      == j) // Calcium
            stBrine.stItem[i].dCA = data;
          if (stBrine._SR      == j) // Strontium
            stBrine.stItem[i].dSR = data;
          if (stBrine._BA      == j) // Barium
            stBrine.stItem[i].dBA = data;

          // ..  VIB
          if (stBrine._CR_II   == j) // Chromium (II)
            stBrine.stItem[i].dCR_II = data;
          if (stBrine._CR_III  == j) // Chromium (iii)
            stBrine.stItem[i].dCR_III = data;

          // ..  VIIB
          if (stBrine._MN_II   == j) // Manganese (II)
            stBrine.stItem[i].dMN_II = data;
          if (stBrine._MN_III   == j) // Manganese (III)
            stBrine.stItem[i].dMN_III = data;

          // ..  VIIIB
          if (stBrine._FE_II   == j) // Iron (II)
            stBrine.stItem[i].dFE_II = data;
          if (stBrine._FE_III  == j) // Iron (III)
            stBrine.stItem[i].dFE_III = data;
          if (stBrine._CO_II   == j) // Cobalt (II)
            stBrine.stItem[i].dCO_II = data;
          if (stBrine._CO_III  == j) // Cobalt (III)
            stBrine.stItem[i].dCO_III = data;
          if (stBrine._NI_II   == j) // Nickel (II)
            stBrine.stItem[i].dNI_II = data;
          if (stBrine._NI_III  == j) // Nickel (III)
            stBrine.stItem[i].dNI_III = data;

          // ..  IB
          if (stBrine._CU_I    == j) // Copper (I)
            stBrine.stItem[i].dCU_I = data;
          if (stBrine._CU_II   == j) // Copper (II)
            stBrine.stItem[i].dCU_II = data;
          if (stBrine._AG      == j) // Silver
            stBrine.stItem[i].dAG = data;
          if (stBrine._AU      == j) // Gold
            stBrine.stItem[i].dAU = data;
          if (stBrine._AU_III  == j) // Gold
            stBrine.stItem[i].dAU_III = data;

          // ..  IIB
          if (stBrine._ZN      == j) // Zinc
            stBrine.stItem[i].dZN = data;
          if (stBrine._CD      == j) // Zinc
            stBrine.stItem[i].dCD = data;
          if (stBrine._HG     == j) // Mercury (I)
            stBrine.stItem[i].dHG = data;
          if (stBrine._HG_II      == j) // Mercury (II)
            stBrine.stItem[i].dHG_II = data;

          // ..  IIIA
          if (stBrine._AL      == j) // Aluminium
            stBrine.stItem[i].dAL = data;

        //..
          if (stBrine._SB_III  == j) // Antimonous(III)
            stBrine.stItem[i].dSB_III = data;
          if (stBrine._SB_V    == j)   // Antimonic(V)
            stBrine.stItem[i].dSB_V = data;
          if (stBrine._BI_III  == j) // Bismuthous(III)
            stBrine.stItem[i].dBI_III = data;
          if (stBrine._BI_V    == j)   // Bismuth(V)
            stBrine.stItem[i].dBI_V = data;

          // ..  IVA
          if (stBrine._SN_II   == j) // Tin (II)
            stBrine.stItem[i].dSN_II = data;
          if (stBrine._SN_IV   == j) // Tin (IV)
            stBrine.stItem[i].dSN_IV = data;
          if (stBrine._PB_II   == j) // Lead (II)
            stBrine.stItem[i].dPB_II = data;
          if (stBrine._PB_IV   == j) // Lead (IV)
            stBrine.stItem[i].dPB_IV = data;

          // .. Other
          if (stBrine._NH4     == j) // Ammonium
            stBrine.stItem[i].dNH4 = data;

          // Common Anions

          // ..Halogen
          if (stBrine._F       == j) // Flouride
            stBrine.stItem[i].dF = data;
          if (stBrine._CL      == j) // Chloride
            stBrine.stItem[i].dCL = data;
          if (stBrine._BR      == j) // Bromide
            stBrine.stItem[i].dBR = data;
          if (stBrine._I       == j) // Iodide
            stBrine.stItem[i].dI = data;

          if (stBrine._OH      == j) // Hydroxide
            stBrine.stItem[i].dOH = data;

          if (stBrine._BO3     == j) // Borate
            stBrine.stItem[i].dBO3 = data;
          if (stBrine._CO3     == j) // Carbonate
            stBrine.stItem[i].dCO3 = data;
          if (stBrine._HCO3    == j) // Bicarbonate
            stBrine.stItem[i].dHCO3 = data;

          if (stBrine._CLO     == j) // Hypochlorite
            stBrine.stItem[i].dCLO = data;
          if (stBrine._CLO2    == j) // Chlorite
            stBrine.stItem[i].dCLO2 = data;
          if (stBrine._CLO3    == j) // Chlorate
            stBrine.stItem[i].dCLO3 = data;
          if (stBrine._CLO4    == j) // Per chlorate
            stBrine.stItem[i].dCLO4 = data;

          if (stBrine._CN      == j) // Cyanide
            stBrine.stItem[i].dCN = data;
          if (stBrine._NCO     == j) // Cyanate
            stBrine.stItem[i].dNCO = data;
          if (stBrine._OCN     == j) // Isocyanate
            stBrine.stItem[i].dOCN = data;
          if (stBrine._SCN     == j) // Thiocyanate
            stBrine.stItem[i].dSCN = data;

          if (stBrine._N       == j) // Nitride
            stBrine.stItem[i].dN = data;
          if (stBrine._N3      == j) // Azide
            stBrine.stItem[i].dN3 = data;
          if (stBrine._NO2     == j) // Nitrite
            stBrine.stItem[i].dNO2 = data;
          if (stBrine._NO3     == j) // Nitrate
            stBrine.stItem[i].dNO3 = data;

          if (stBrine._CRO4    == j) // Chromium (II)
            stBrine.stItem[i].dCRO4 = data;
          if (stBrine._CR2O7   == j) // Chromium (III)
            stBrine.stItem[i].dCR2O7 = data;

          if (stBrine._MNO4    == j) // Permanganate
            stBrine.stItem[i].dMNO4 = data;

          if (stBrine._P       == j) // Phosphide
            stBrine.stItem[i].dP = data;
          if (stBrine._PO3     == j) // Phosphite
            stBrine.stItem[i].dPO3 = data;
          if (stBrine._PO4     == j) // Phosphate
            stBrine.stItem[i].dPO4 = data;
          if (stBrine._HPO4    == j) // Monohydrogen Phosphate
            stBrine.stItem[i].dHPO4 = data;
          if (stBrine._H2PO4   == j) // Dihydrogen Phosphate
            stBrine.stItem[i].dH2PO4 = data;

          if (stBrine._AS      == j) // Arsinide
            stBrine.stItem[i].dAS = data;
          if (stBrine._SE      == j) // Selenide
            stBrine.stItem[i].dSE = data;

          if (stBrine._S       == j) // Sulfide
            stBrine.stItem[i].dS = data;
          if (stBrine._HS      == j) // Hydrogen Sulfide
            stBrine.stItem[i].dHS = data;
          if (stBrine._SO3     == j) // Sulphite
            stBrine.stItem[i].dSO3 = data;
          if (stBrine._HSO3    == j) // Hydrogen Sulphite
            stBrine.stItem[i].dHSO3 = data;
          if (stBrine._S2O3    == j) // Thiosulphate
            stBrine.stItem[i].dS2O3 = data;
          if (stBrine._SO4     == j) // Sulfate
            stBrine.stItem[i].dSO4  = data;
          if (stBrine._HSO4    == j) // Bisulfate
            stBrine.stItem[i].dHSO4 = data;

          if (stBrine._SOLID   == j) // Total Solids
            stBrine.stItem[i].dSOLID = data;
          if (stBrine._TDS     == j) // Total Solids Computed
            stBrine.stItem[i].dTDS = data;
	    }
      }
    }

    return (stBrine);
  }

  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Brine List Structure.
   * @param  st    = the Brine List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( brineListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_BRINE;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Brine List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( brineListStruct st, las3Struct stT)
  {
    int i,j;
    String sDate[] = { "", "" };

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = _PARAM_ROWS;           // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            stT.sParams[i][_VALUE] = new String( "YES" );
            break;
          case  _SOURCE:
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case  _RECOVER:
            stT.sParams[i][_VALUE]  = new String( st.sRecovery );
            sDate = cmn.cmnString.convertDate( stT.sParams[i][_VALUE] );
            stT.sParams[i][_VALUE]  = new String( sDate[0] );
            stT.sParams[i][_FORMAT] = new String( sDate[1] );
            break;
          case  _TOP:
            stT.sParams[i][_UNIT]   = new String( "F" );
            stT.sParams[i][_VALUE]  = new String( ""+st.depthStart );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _BASE:
            stT.sParams[i][_UNIT]   = new String( "F" );
            stT.sParams[i][_VALUE]  = new String( ""+st.depthEnd );
            stT.sParams[i][_FORMAT] = new String( "F" );
            break;
          case  _FORMATION:
            stT.sParams[i][_VALUE] = new String( st.sFormation );
            break;
          case  _COMPANY:
            stT.sParams[i][_VALUE] = new String( st.sCompany );
            break;
          case  _ANALYZED:
            stT.sParams[i][_VALUE] = new String( st.sAnalyzed );
            sDate = cmn.cmnString.convertDate( stT.sParams[i][_VALUE] );
            stT.sParams[i][_VALUE]  = new String( sDate[0] );
            stT.sParams[i][_FORMAT] = new String( sDate[1] );
            break;
        }
      }
    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curves Data to the LAS 3 File Data Structure
   * @param  st    = the Brine List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( brineListStruct st,
                                      las3Struct      stT,
                                      double          dNull )
  {
    int    i,j;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[brine.brineStandardTools.TOTAL];
      for (i=0; i<brine.brineStandardTools.TOTAL; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (j=0; j<st.iCount; j++)
      {
        if (st.stItem[j].sFORM.length() > 0) // Primary Formation
        {
		  iSelected[brine.brineStandardTools._FORM] = 1;
		}
        if (st.stItem[j].sAGE.length() > 0)  // Age of Formation
        {
		  iSelected[brine.brineStandardTools._AGE] = 1;
		}
	  }

      for (j=0; j<st.iCount; j++)
      {
        for (i=brine.brineStandardTools._TOP;
             i<brine.brineStandardTools.TOTAL; i++)
        {
//        if (st.checkData(i))
//          iSelected[i] = 1;
//          if (st.getData( i, j ) > -1.0)
          if (st.getData( i, j ) > 0.0)
          {
            iSelected[i] = 1;
		  }
        }
      }

      // Count the Number of Curves Populated

      for (i=0; i<brine.brineStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<brine.brineStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              brine.brineStandardTools.CURVES[i][_C_MNEM]);

          stT.sCurves[iCnt][_UNIT] = new String(
              brine.brineStandardTools.CURVES[i][_C_UNIT]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              brine.brineStandardTools.CURVES[i][_C_DESC]);

          stT.sCurves[iCnt][_FORMAT] = new String("F");
          if ((i == brine.brineStandardTools._FORM) ||
              (i == brine.brineStandardTools._AGE))
          {
            stT.sCurves[iCnt][_FORMAT] = new String("S");
          }

          iCnt++;
        }
      }

      // Build the Data Section

      stT.iRows    = st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Brine Data List Structure

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSum; j++)
        {
          if ((iCurves[j] == brine.brineStandardTools._FORM) ||
              (iCurves[j] == brine.brineStandardTools._AGE))
          {
            switch (iCurves[j])
            {
              case brine.brineStandardTools._FORM:
                stT.sData[i][j] = new String( st.stItem[i].sFORM );
                break;
              case brine.brineStandardTools._AGE:
                stT.sData[i][j] = new String( st.stItem[i].sAGE );
                break;
            }
          }
          else
            stT.sData[i][j] = new String( "" + st.getData( iCurves[j], i ) );
        }
      }
    }

    return (stT);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back
   * @param  str    = the string to be converted
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str )
  {
    double dValue = 0.0;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }

  /** Method stringToDouble()
   * <p> This method will convert the String to double.  This first checks to
   *     see if the string is numeric and then passes the data back, but uses
   *     the Null Value if it is not a valid number
   * @param  str    = the string to be converted
   * @param  dNull  = the LAS File Null Value
   * @return dValue = double value of the string passed in or Null value
   */

  public static double stringToDouble( String str, double dNull )
  {
    double dValue = dNull;

    if (str != null)
    {
      if (cmn.cmnString.isNumeric( str ))
        dValue = cmn.cmnString.stringToDouble( str );
    }

    return (dValue);
  }
}

/*
 *  @version 1.1 06/02/2011
 *  @author  John Victorine
 */
