/*
 * @las3LoadBioData.java Version 1.1 11/29//2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3;

import cmn.cmnLASCurveListStruct;
import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;

import las3.las3ListStruct;
import las3.las3Struct;

/** Class las3LoadBioData
 *  <p> This Class will provide load utilities for the LAS 3 Classes
 *
 *  @version 1.1 11/29//2011
 *  @author  John R. Victorine
 */

public class las3LoadBioData
{
  public static final int _NONE        = las3.las3Constants._NONE;

  public static final int _IQ_KGS      = 0;  // Profile Saved Data Indicator
  public static final int _BIOSRC      = 1;
  public static final int _BIODR       = 2;

  public static final int _PARAM_ROWS  = 3;
  public static final int _PARAM_COLS  = 5;
  public static final String PARAM[][] = {
 // Mnemonic  Description                        Variations
    { "IQKGS",  "Profile Web App Saved Data Indicator",  "", "", "" },
    { "BIOSRC", "Formation Source",                      "", "", "" },
    { "BIODR",  "Tops Depth Reference",                  "", "", "" },
  };

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  /** Method getData()
   * <p> This method will return the Bio-Stratigraphic Data List Structure
   *     from the LAS 3 Data List Structure.
   * @param  stT = the LAS 3 Data Structure
   * @return st  = the Bio-Stratigraphic Data List Structure
   */

  public static bioStratListStruct getData( las3Struct stT )
  {
    int i,j,k;
    int iParam = -1;
    bioStratListStruct st  = new bioStratListStruct();

    if (stT != null)
    {
      for (i=0; i<stT.iParamRows; i++)
      {
        iParam = _NONE;
        for (j=0; j<_PARAM_ROWS; j++)
        {
          for (k=0; k<_PARAM_COLS; k++)
          {
            if (stT.sParams[i][_MNEMONIC].toUpperCase().equals(PARAM[j][k]))
            {
              iParam = j;

              switch (iParam)
              {
                case _NONE:
                  break;

                // Required Fields

                case _IQ_KGS:
                  st.sKGS  = new String( "YES" );
                  break;
                case _BIOSRC:
                  st.source = new String( stT.sParams[i][_VALUE] );
                  break;
                case _BIODR:
                  st.sReference = new String( stT.sParams[i][_VALUE] );
                  break;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 File Data into the
   *     Bio-Stratigraphic Data List Structure
   * @param  dNull   = LAS 3 Null Value
   * @param  st      = the LAS 3 Data Structure
   * @param  stCMN   = Common LAS Curve List Structure.
   * @param  stBio   = the Bio-Stratigraphic Data List Structure
   * @return stBio2  = the Bio-Stratigraphic Data List Structure
   */

  public static bioStratListStruct loadData( double     dNull,
                                             las3Struct st,
                                             cmnLASCurveListStruct stCMN,
                                             bioStratListStruct    stBio )
  {
    bioStratListStruct stBio2 = null;
    bioStratStruct     stB    = null;
    int    iTool       = -1;
    String sMnemonic   = "";
    String sData       = "";
    int    iRow        = -1;
    String sTemp       = "";
    int    ino         = 0;
    int    iSpecies    = -1;
    String sNull       = new String(""+dNull); // NULL Value in String Form
    String sKID        = "0"; // Not Used
    String sKEY        = "0"; // Unique Identifier for Entry

    String sName       = "";  // Name (Genus species) Name of Fossil
    String sKEYa       = "0"; // Array of Unique Key for each Depth Sample
    int    iAbundance  = horizon.bio.bioStratStruct._PRESENT;
    double depthStart  = 0.0; // Array of Starting Depths
    double depthEnd    = 0.0; // Array if Ending Depths

  // Fossil Scientific Name

    String species     = "";  // Species

    String sGenus      = "";  // Genus
    String sTribe      = "";  // Tribe

    String subfamily   = "";  // Sub family
    String sFamily     = "";  // Family
    String superfamily = "";  // Super family

    String sInfraorder = "";  // Infra Order
    String suborder    = "";  // Sub Order
    String sOrder      = "";  // Order
    String superorder  = "";  // Super Order

    String subclass    = "";  // Class
    String sClass      = "";  // Sub Class
    String superclass  = "";  // Super Class

    String subphylum   = "";  // Sub Phylum
    String sPhylum     = "";  // Phylum
    String superphylum = "";  // Super Phylum

    String subkingdom  = "";  // Sub Kingdom
    String sKingdom    = "";  // Kingdom

    String sAuthor     = "";  // Author Credited

  // Generic Rock Fossil Identifier

    String sFossilID   = "10.2.1";       // Fossil identifier
    String sFossil     = "macrofossils"; // Generic Fossil Name,

  // -- Earliest geological time

    double dFrom       = 0.0; // From Age in Ma
    String sKey_e      = "";  // Stratigraphic Unit Unique KEY
    String system_e    = "";  // System Stratigraphic Unit Names
    String series_e    = "";  // Series Stratigraphic Unit Names
    String subSystem_e = "";  // Substage Stratigraphic Unit Names
    String subSeries_e = "";  // Subseries Stratigraphic Unit Names

  // -- Latest geological time

    double dTo         = 0.0; // To Age in Ma
    String sKey_l      = "";  // Stratigraphic Unit Unique KEY
    String system_l    = "";  // System Stratigraphic Unit Names
    String series_l    = "";  // Series Stratigraphic Unit Names
    String subSystem_l = "";  // Substage Stratigraphic Unit Names
    String subSeries_l = "";  // Subseries Stratigraphic Unit Names

    if ((st != null) && (stBio != null) && (stCMN != null))
    {
      if (stCMN.iCount > 0)
      {
        stBio2 = horizon.bio.bioStratUtility.copyList( stBio );

        stBio.delete();
        stBio = null;

        if (stBio2 == null)
          stBio2 = new bioStratListStruct();

        if (st.iRows > 0)
        {
          for (int i=0; i<st.iRows; i++)
          {
            for (int j=0; j<st.iColumns; j++)
            {
              iTool = -1;
              for (int k=0; k<stCMN.iCount; k++)
              {
                if (st.sCurves[j][las3.las3Struct._MNEMONIC].equals(
                        stCMN.stItem[k].sMnemonic))
                {
                  iTool = k;
                }
              }

              if (iTool > -1)
              {
                sMnemonic = new String( stCMN.stItem[iTool].sMnemonic );
                sData     = new String( st.sData[i][j] );
				iRow      = horizon.bio.bioStandardTools.getRow(
                     horizon.bio.bioStandardTools._C_MNEM, sMnemonic );

                switch (iRow)
                {
                  case horizon.bio.bioStandardTools._SP:   // Species",
                    if (!sData.equals(sNull))
                    {
                      species = new String(sData);
                      sName   = new String(sData);

                      iSpecies = -1;
    		          if (species.length() > 0)
		              {
		                iSpecies = horizon.bio.bioStratUtility.isSpecies(
							         species, stBio2 );
				      }
           	        }
                    break;
                  case horizon.bio.bioStandardTools._ABUN: // Abundance
                    if (!sData.equals(sNull))
// JRV
//                      iAbundance = parse.parseGeneraUtility.getAbundance(sData);
                      iAbundance = horizon.bio.bioStratUtility.getAbundance(sData);
                      if (iAbundance == horizon.bio.bioStratStruct._NONE)
                      {
						iAbundance = horizon.bio.bioStratStruct._PRESENT;
					  }
                    break;
                  case horizon.bio.bioStandardTools._STRT: // Starting Depth",
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                      {
                        depthStart = cmn.cmnString.stringToDouble(sData);
                        ino   = (int) depthStart;
						sKEYa = new String( cmn.cmnString.UniqueName() +
						                    "_" + i + "_" + ino );
					  }
					}
                    break;
                  case horizon.bio.bioStandardTools._END:  // Ending Depth",
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        depthEnd = cmn.cmnString.stringToDouble(sData);
					}
                    break;

                  // Species Classification

                  case horizon.bio.bioStandardTools._GENUS: // Genus",
                    if (!sData.equals(sNull))
                      sGenus = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._TRIBE: // Tribe",
                    if (!sData.equals(sNull))
                      sTribe = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._SBFAM: // Subfamily",
                    if (!sData.equals(sNull))
                      subfamily = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._FAM:   // Family",
                    if (!sData.equals(sNull))
                      sFamily = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SPFAM: // Superfamily",
                    if (!sData.equals(sNull))
                      superfamily = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._INORD: // Infraorder",
                    if (!sData.equals(sNull))
                      sInfraorder = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SBORD: // Suborder",
                    if (!sData.equals(sNull))
                      suborder = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._ORDER: // Order",
                    if (!sData.equals(sNull))
                      sOrder = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SPORD: // Superorder",
                    if (!sData.equals(sNull))
                      superorder = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._SBCLS: // Subclass",
                    if (!sData.equals(sNull))
                      subclass = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._CLASS: // Class",
                    if (!sData.equals(sNull))
                      sClass = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SPCLS: // Superclass",
                    if (!sData.equals(sNull))
                      superclass = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._SBPHY: // Subphylum",
                    if (!sData.equals(sNull))
                      subphylum = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._PHYLM:// Phylum",
                    if (!sData.equals(sNull))
                      sPhylum = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SPPHY: // Superphylum",
                    if (!sData.equals(sNull))
                      superphylum = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._SBKNG: // Subkingdom",
                    if (!sData.equals(sNull))
                      subkingdom = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._KNGDM: // Kingdom",
                    if (!sData.equals(sNull))
                      sKingdom = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._AUTHR: // Author",
                    if (!sData.equals(sNull))
                      sAuthor = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._GENID: // Generic Fossil ID",
                    if (!sData.equals(sNull))
                      sFossilID = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._GENRC: // Generic Fossil Name",
                    if (!sData.equals(sNull))
                      sFossil = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._FROM:  // Age From",
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dFrom = cmn.cmnString.stringToDouble(sData); // From Age in Ma
		            }
                    break;
                  case horizon.bio.bioStandardTools._KEYE:  // Strat Unit KEY
                    if (!sData.equals(sNull))
                      sKey_e = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SYSE:  // System
                    if (!sData.equals(sNull))
                      system_e = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SERE:  // Series
                    if (!sData.equals(sNull))
                      series_e = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SSYSE: // Subsystem
                    if (!sData.equals(sNull))
                      subSystem_e = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SSERE: // Subseries
                    if (!sData.equals(sNull))
                      subSeries_e = new String(sData);
                    break;

                  case horizon.bio.bioStandardTools._TO:    // Age To",
                    if (!sData.equals(sNull))
                    {
                      if (cmn.cmnString.isNumeric(sData))
                        dTo = cmn.cmnString.stringToDouble(sData);
		            }
                    break;
                  case horizon.bio.bioStandardTools._KEYL:  // Strat Unit KEY
                    if (!sData.equals(sNull))
                      sKey_l = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SYSL:  // System
                    if (!sData.equals(sNull))
                      system_l = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SERL:  // Series
                    if (!sData.equals(sNull))
                      series_l = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SSYSL: // Subsystem
                    if (!sData.equals(sNull))
                      subSystem_l = new String(sData);
                    break;
                  case horizon.bio.bioStandardTools._SSERL: // Subseries
                    if (!sData.equals(sNull))
                      subSeries_l = new String(sData);
                    break;
                }
              }
			}

            // Insert variables into bio-stratigraphy data list structure

            if (iSpecies > -1) // Species already in list just add depth info
            {
			  stBio2.stItem[iSpecies] =
			    horizon.bio.bioStratUtility.addDepth(
			  	  stBio2.stItem[iSpecies],
                  sKEYa,
                  iAbundance,
                  depthStart,
                  depthEnd );
			}
			else // New species save all information.
			{
			  stB      = new bioStratStruct();
              sKEY     = new String( cmn.cmnString.UniqueName() + "_" + i );
              stB.sKID  = new String( sKEY );
              stB.sKEY  = new String( sKEY );

              stB.sName = new String( sName );

              stB.iRows         = 1;
              stB.sKEYa         = new String[1];
              stB.iAbundance    = new int[1];
              stB.depthStart    = new double[1];
              stB.depthEnd      = new double[1];

              stB.sKEYa[0]      = new String( sKEY );
              stB.iAbundance[0] = iAbundance;
              stB.depthStart[0] = depthStart;
              stB.depthEnd[0]   = depthStart;
              if (depthEnd > depthStart)
                stB.depthEnd[0] = depthEnd;

              // Fossil Scientific Name

              stB.species     = new String( species );
              stB.sGenus      = new String( sGenus );
              stB.sTribe      = new String( sTribe );
              stB.subfamily   = new String( subfamily );
              stB.sFamily     = new String( sFamily );
              stB.superfamily = new String( superfamily );
              stB.sInfraorder = new String( sInfraorder );
              stB.suborder    = new String( suborder );
              stB.sOrder      = new String( sOrder );
              stB.superorder  = new String( superorder );
              stB.subclass    = new String( subclass );
              stB.sClass      = new String( sClass );
              stB.superclass  = new String( superclass );
              stB.subphylum   = new String( subphylum );
              stB.sPhylum     = new String( sPhylum );
              stB.superphylum = new String( superphylum );
              stB.subkingdom  = new String( subkingdom );
              stB.sKingdom    = new String( sKingdom );
              stB.sAuthor     = new String( sAuthor );

              // Generic Rock Fossil Identifier

              stB.sFossilID   = new String( sFossilID );
              stB.sFossil     = new String( sFossil );

              // Stratigraphic Unit Range

              // -- Earliest geological time

              stB.dFrom       = dFrom; // From Age in Ma
              stB.sKey_e      = new String( sKey_e );
              stB.system_e    = new String( system_e );
              stB.series_e    = new String( series_e );
              stB.subSystem_e = new String( subSystem_e );
              stB.subSeries_e = new String( subSeries_e );

              // -- Latest geological time

              stB.dTo         = dTo; // From Age in Ma
              stB.sKey_l      = new String( sKey_l );
              stB.system_l    = new String( system_l );
              stB.series_l    = new String( series_l );
              stB.subSystem_l = new String( subSystem_l );
              stB.subSeries_l = new String( subSeries_l );

			  stBio2 = horizon.bio.bioStratUtility.add( stB, stBio2 );
			}

            // Clear Variables for next entry

			if (stB != null)
			  stB.delete();
			stB = null;

            iSpecies    = -1;

            sKID        = new String( "0" ); // Not Used
            sKEY        = new String( "0" ); // Unique Identifier for Entry

            sName       = "";  // Name (Genus species) Name of Fossil
            sKEYa       = new String( "0" ); // Array of Unique Key for each Depth Sample
            iAbundance  = 2;   // Array of Abundance indicators
            depthStart  = 0.0; // Array of Starting Depths
            depthEnd    = 0.0; // Array if Ending Depths

            // Fossil Scientific Name

            species     = new String( "" );  // Species

            sGenus      = new String( "" );  // Genus
            sTribe      = new String( "" );  // Tribe

            subfamily   = new String( "" );  // Sub family
            sFamily     = new String( "" );  // Family
            superfamily = new String( "" );  // Super family

            sInfraorder = new String( "" );  // Infra Order
            suborder    = new String( "" );  // Sub Order
            sOrder      = new String( "" );  // Order
            superorder  = new String( "" );  // Super Order

            subclass    = new String( "" );  // Class
            sClass      = new String( "" );  // Sub Class
            superclass  = new String( "" );  // Super Class

            subphylum   = new String( "" );  // Sub Phylum
            sPhylum     = new String( "" );  // Phylum
            superphylum = new String( "" );  // Super Phylum

            subkingdom  = new String( "" );  // Sub Kingdom
            sKingdom    = new String( "" );  // Kingdom

            sAuthor     = new String( "" );  // Author Credited

            // Generic Rock Fossil Identifier

            sFossilID   = "10.2.1";       // Fossil identifier
            sFossil     = "macrofossils"; // Generic Fossil Name,

            // -- Earliest geological time

            dFrom       = 0.0; // From Age in Ma
            sKey_e      = new String( "" );  // Stratigraphic Unit Unique KEY
            system_e    = new String( "" );  // System Stratigraphic Unit Names
            series_e    = new String( "" );  // Series Stratigraphic Unit Names
            subSystem_e = new String( "" );  // Substage Stratigraphic Unit Names
            subSeries_e = new String( "" );  // Subseries Stratigraphic Unit Names

            // -- Latest geological time

            dTo         = 0.0; // To Age in Ma
            sKey_l      = new String( "" );  // Stratigraphic Unit Unique KEY
            system_l    = new String( "" );  // System Stratigraphic Unit Names
            series_l    = new String( "" );  // Series Stratigraphic Unit Names
            subSystem_l = new String( "" );  // Substage Stratigraphic Unit Names
            subSeries_l = new String( "" );  // Subseries Stratigraphic Unit Names
		  }
        }
      }
    }

    return (stBio2);
  }

  /** Method mergeData()
   * <p> This method will merge a bio-stratigraphic data list Structure
   *     with the new Bio-Stratigraphic Data List Structure
   * @param  dNULL = LAS 3 Null Value
   * @param  st    = Bio-Stratigraphic Data List Structure
   * @param  stBio = Merged Bio-Stratigraphic Data List Structure
   * @return stBio = Merged Bio-Stratigraphic Data List Structure
   *

  public static bioStratListStruct mergeData( double dNULL,
                                              bioStratListStruct st,
                                              bioStratListStruct stBio )
  {
    int i      = 0;
    int iStart = 0;

    if ((st != null) && (stBio != null))
    {
      for (i=0; i<stBio.iCount; i++)
      {
        if (stBio.stItem[i].depthStart != dNULL)
          iStart = i;
      }

      for (i=0; i<st.iCount; i++)
      {
        if (iStart < stBio.iCount)
        {
          stBio.stItem[iStart] =
              horizon.bio.bioStratUtility.copy( st.stItem[i] );
          iStart++;
        }
      }
    }

    return (stBio);
  }
*/
  /** Method setData()
   * <p> This method will return the LAS 3 Data List Structure from the
   *     Bio-Stratigraphic Data List Structure.
   * @param  st    = the Bio-Stratigraphic Data List Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 Data Structure
   */

  public static las3Struct setData( bioStratListStruct st, double dNull )
  {
    las3Struct stT = null;

    if (st != null)
    {
      stT        = new las3Struct();
      stT.iType  = las3.las3Constants._IQ_BIO;
      stT.iArray = 0;
      stT.sType  = new String(las3.las3Constants.SECTIONS[stT.iType]);

      stT        = addParameters( st, stT );    // Add the Required Parameters
      stT        = addCurves( st, stT, dNull ); // Add Curves
    }

    return (stT);
  }

  /** Method addParameters()
   * <p> This method will add the Parameter Data to the LAS 3 File Data Structure
   * @param  st    = the Bio-Stratigraphic Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addParameters( bioStratListStruct st, las3Struct stT )
  {
    int i,j;

    if ((st != null) && (stT != null))
    {
      stT.iParamRows  = 3;      // Total number of rows
      stT.iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
      stT.sParams = new String[stT.iParamRows][stT.iParamCols];

      for (i=0; i<stT.iParamRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
        {
          stT.sParams[i][j] = new String("");
        }
      }

      for (i=0; i<stT.iParamRows; i++)
      {
        stT.sParams[i][_MNEMONIC]    = new String( PARAM[i][0] );
        stT.sParams[i][_DESCRIPTION] = new String( PARAM[i][1] );
        stT.sParams[i][_FORMAT]      = new String( "S" );

        switch (i)
        {
          case _IQ_KGS:
            break;
          case _BIOSRC:
            stT.sParams[i][_VALUE] = new String( st.source );
            break;
          case _BIODR:
            stT.sParams[i][_VALUE] = new String( st.sReference );
            break;
        }
      }

    }

    return (stT);
  }

  /** Method addCurves()
   * <p> This method will add the Curve Data to the LAS 3 File Data Structure
   * @param  st    = the Bio-Stratigraphic Data List Structure
   * @param  stT   = the LAS 3 File Data Structure
   * @param  dNull = LAS 3 File Data Null Value
   * @return stT   = the LAS 3 File Data Structure
   */

  public static las3Struct addCurves( bioStratListStruct st,
                                      las3Struct         stT,
                                      double             dNull )
  {
    int    i,j,k;
    int    iSum        = 0;
    int    iCnt        = 0;
    int    iTotal      = 0;
    int    iSelected[] = null;
    int    iCurves[]   = null;
    String sData       = "";
    String sNull       = "" + dNull;

    if ((st != null) && (stT != null))
    {
      // Initialize the Curve Selected identifier

      iSelected = new int[horizon.bio.bioStandardTools.TOTAL];
      for (i=0; i<horizon.bio.bioStandardTools.TOTAL; i++)
      {
        iSelected[i] = -1;
      }

      // Check if columns are populated

      for (i=0; i<st.iCount; i++)
      {
        iSelected[horizon.bio.bioStandardTools._SP]++;      // Species

		for (k=0; k<st.stItem[i].iRows; k++)
		{
		  iTotal++;

          iSelected[horizon.bio.bioStandardTools._ABUN]++; // Abundance
          iSelected[horizon.bio.bioStandardTools._STRT]++; // Starting Depth
          iSelected[horizon.bio.bioStandardTools._END]++;  // Ending Depth
	    }

        // Species Classification

        iSelected[horizon.bio.bioStandardTools._GENUS]++; // Genus",

        if (st.stItem[i].sTribe.length() > 0.0) // Tribe
          iSelected[horizon.bio.bioStandardTools._TRIBE]++;

        if (st.stItem[i].subfamily.length() > 0.0) // Subfamily
          iSelected[horizon.bio.bioStandardTools._SBFAM]++;
        if (st.stItem[i].sFamily.length() > 0.0) // Family
          iSelected[horizon.bio.bioStandardTools._FAM]++;
        if (st.stItem[i].superfamily.length() > 0.0) // Superfamily
          iSelected[horizon.bio.bioStandardTools._SPFAM]++;

        if (st.stItem[i].sInfraorder.length() > 0.0) // Infraorder
          iSelected[horizon.bio.bioStandardTools._INORD]++;
        if (st.stItem[i].suborder.length() > 0.0) // Suborder
          iSelected[horizon.bio.bioStandardTools._SBORD]++;
        if (st.stItem[i].sOrder.length() > 0.0) // Order
          iSelected[horizon.bio.bioStandardTools._ORDER]++;
        if (st.stItem[i].superorder.length() > 0.0) // Superorder
          iSelected[horizon.bio.bioStandardTools._SPORD]++;

        if (st.stItem[i].subclass.length() > 0.0) // Subclass
          iSelected[horizon.bio.bioStandardTools._SBCLS]++;
        if (st.stItem[i].sClass.length() > 0.0) // Class
          iSelected[horizon.bio.bioStandardTools._CLASS]++;
        if (st.stItem[i].superclass.length() > 0.0) // Superclass
          iSelected[horizon.bio.bioStandardTools._SPCLS]++;

        if (st.stItem[i].subphylum.length() > 0.0) // Subphylum
          iSelected[horizon.bio.bioStandardTools._SBPHY]++;
        if (st.stItem[i].sPhylum.length() > 0.0) // Phylum
          iSelected[horizon.bio.bioStandardTools._PHYLM]++;
        if (st.stItem[i].superphylum.length() > 0.0) // Superphylum
          iSelected[horizon.bio.bioStandardTools._SPPHY]++;

        if (st.stItem[i].subkingdom.length() > 0.0) // Subkingdom
          iSelected[horizon.bio.bioStandardTools._SBKNG]++;
        if (st.stItem[i].sKingdom.length() > 0.0) // Kingdom
          iSelected[horizon.bio.bioStandardTools._KNGDM]++;

        if (st.stItem[i].sAuthor.length() > 0.0) // Author
          iSelected[horizon.bio.bioStandardTools._AUTHR]++;

        if (st.stItem[i].sFossilID.length() > 0.0) // Generic Fossil ID
          iSelected[horizon.bio.bioStandardTools._GENID]++;
        if (st.stItem[i].sFossil.length() > 0.0) // Generic Fossil Name
          iSelected[horizon.bio.bioStandardTools._GENRC]++;

        if (st.stItem[i].dFrom > 0.0) // Age From
          iSelected[horizon.bio.bioStandardTools._FROM]++;
        if (st.stItem[i].sKey_e.length() > 0.0) // Stratigraphic Unit KEY
          iSelected[horizon.bio.bioStandardTools._KEYE]++;
        if (st.stItem[i].system_e.length() > 0.0) // System stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SYSE]++;
        if (st.stItem[i].series_e.length() > 0.0) // Series stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SERE]++;
        if (st.stItem[i].subSystem_e.length() > 0.0) // Subsystem stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SSYSE]++;
        if (st.stItem[i].subSeries_e.length() > 0.0) // Subseries stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SSERE]++;

        if (st.stItem[i].dTo > 0.0) // Age To
          iSelected[horizon.bio.bioStandardTools._TO]++;
        if (st.stItem[i].sKey_l.length() > 0.0) // Stratigraphic Unit KEY
          iSelected[horizon.bio.bioStandardTools._KEYL]++;
        if (st.stItem[i].system_l.length() > 0.0) // System stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SYSL]++;
        if (st.stItem[i].series_l.length() > 0.0) // Series stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SERL]++;
        if (st.stItem[i].subSystem_l.length() > 0.0) // Subsystem stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SSYSL]++;
        if (st.stItem[i].subSeries_l.length() > 0.0) // Subseries stratigraphic unit name
          iSelected[horizon.bio.bioStandardTools._SSERL]++;
	  }

      // Count the Number of Curves Populated

      for (i=0; i<horizon.bio.bioStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
          iSum++;
      }

      // Build Curve Definition Section

      iCurves        = new int[iSum];

      stT.iCurveRows = iSum;                  // Total number of rows
      stT.iCurveCols = las3.las3Struct._TOTAL; // Total number of columns
      stT.sCurves    = new String[stT.iCurveRows][stT.iCurveCols];

      for (i=0; i<horizon.bio.bioStandardTools.TOTAL; i++)
      {
        if (iSelected[i] > -1)
        {
          iCurves[iCnt] = i;

          for (j=0; j<las3.las3Struct._TOTAL; j++)
          {
            stT.sCurves[iCnt][j] = new String("");
          }

          stT.sCurves[iCnt][_MNEMONIC] = new String(
              horizon.bio.bioStandardTools.CURVES[i][
                horizon.bio.bioStandardTools._C_MNEM]);

          stT.sCurves[iCnt][_UNIT] = new String(
              horizon.bio.bioStandardTools.CURVES[i][
                horizon.bio.bioStandardTools._C_UNIT]);

          stT.sCurves[iCnt][_DESCRIPTION] = new String(
              horizon.bio.bioStandardTools.CURVES[i][
                horizon.bio.bioStandardTools._C_NAME]);

          stT.sCurves[iCnt][_FORMAT] = new String("S");
          if (stT.sCurves[iCnt][_UNIT].length() > 0)
            stT.sCurves[iCnt][_FORMAT] = new String("F");

          iCnt++;
        }
      }

      // Build the Data Section

      stT.iRows    = iTotal; // st.iCount;  // Total Number of Rows
      stT.iColumns = iSum;       // Total Number of Columns ( iCurveRows )
      stT.sData    = new String[iTotal][stT.iColumns]; //[stT.iRows][stT.iColumns];

      // Initialize the Data Column values to Null

      for (i=0; i<iTotal; i++)
      {
        for (j=0; j<iSum; j++)
        {
          stT.sData[i][j] = new String("" + dNull);
        }
      }

      // Load the Data Columns from the Stratigraphic Units Data List Structure

      iCnt = 0;

      for (i=0; i<st.iCount; i++)
      {
		for (k=0; k<st.stItem[i].iRows; k++)
		{
		  if (iCnt < iTotal)
		  {
            for (j=0; j<iSum; j++)
            {
              switch (iCurves[j])
              {
                case horizon.bio.bioStandardTools._SP:   // Species",
                  if (st.stItem[i].species.length() > 1)
                  {
                    stT.sData[iCnt][j] = new String( st.stItem[i].species );
			      }
			      else
			      {
                    stT.sData[iCnt][j] = new String( st.stItem[i].sName );
			      }
                  break;
                case horizon.bio.bioStandardTools._ABUN: // Abundance
                  if (st.stItem[i].iAbundance[k] == horizon.bio.bioStratStruct._NONE)
                  {
		            st.stItem[i].iAbundance[k] = horizon.bio.bioStratStruct._PRESENT;
  		          }

                  stT.sData[iCnt][j] = new String(
			        horizon.bio.bioStratStruct.sAbundance[st.stItem[i].iAbundance[k]] );
                  break;
                case horizon.bio.bioStandardTools._STRT: // Starting Depth",
                  stT.sData[iCnt][j] = new String( "" + st.stItem[i].depthStart[k] );
                  break;
                case horizon.bio.bioStandardTools._END:  // Ending Depth",
                  stT.sData[iCnt][j] = new String( "" + st.stItem[i].depthEnd[k] );
                  break;

                  // Species Classification

                case horizon.bio.bioStandardTools._GENUS: // Genus",
                  if (st.stItem[i].sGenus.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sGenus );
                  break;
                case horizon.bio.bioStandardTools._TRIBE: // Tribe",
                  if (st.stItem[i].sTribe.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sTribe );
                  break;

                case horizon.bio.bioStandardTools._SBFAM: // Subfamily",
                  if (st.stItem[i].subfamily.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subfamily );
                  break;
                case horizon.bio.bioStandardTools._FAM:   // Family",
                  if (st.stItem[i].sFamily.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sFamily );
                  break;
                case horizon.bio.bioStandardTools._SPFAM: // Superfamily",
                  if (st.stItem[i].superfamily.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].superfamily );
                  break;

                case horizon.bio.bioStandardTools._INORD: // Infraorder",
                  if (st.stItem[i].sInfraorder.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sInfraorder );
                  break;
                case horizon.bio.bioStandardTools._SBORD: // Suborder",
                  if (st.stItem[i].suborder.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].suborder );
                  break;
                case horizon.bio.bioStandardTools._ORDER: // Order",
                  if (st.stItem[i].sOrder.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sOrder );
                  break;
                case horizon.bio.bioStandardTools._SPORD: // Superorder",
                  if (st.stItem[i].superorder.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].superorder );
                  break;

                case horizon.bio.bioStandardTools._SBCLS: // Subclass",
                  if (st.stItem[i].subclass.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subclass );
                  break;
                case horizon.bio.bioStandardTools._CLASS: // Class",
                  if (st.stItem[i].sClass.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sClass );
                  break;
                case horizon.bio.bioStandardTools._SPCLS: // Superclass",
                  if (st.stItem[i].superclass.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].superclass );
                  break;

                case horizon.bio.bioStandardTools._SBPHY: // Subphylum",
                  if (st.stItem[i].subphylum.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subphylum );
                  break;
                case horizon.bio.bioStandardTools._PHYLM:// Phylum",
                  if (st.stItem[i].sPhylum.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sPhylum );
                  break;
                case horizon.bio.bioStandardTools._SPPHY: // Superphylum",
                  if (st.stItem[i].superphylum.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].superphylum );
                  break;

                case horizon.bio.bioStandardTools._SBKNG: // Subkingdom",
                  if (st.stItem[i].subkingdom.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subkingdom );
                  break;
                case horizon.bio.bioStandardTools._KNGDM: // Kingdom",
                  if (st.stItem[i].sKingdom.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sKingdom );
                  break;

                case horizon.bio.bioStandardTools._AUTHR: // Author",
                  if (st.stItem[i].sAuthor.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sAuthor );
                  break;

                case horizon.bio.bioStandardTools._GENID: // Generic Fossil ID",
                  if (st.stItem[i].sFossilID.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sFossilID );
                  break;
                case horizon.bio.bioStandardTools._GENRC: // Generic Fossil Name",
                  if (st.stItem[i].sFossil.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sFossil );
                  break;

                case horizon.bio.bioStandardTools._FROM:  // Age From",
                  if (st.stItem[i].dFrom > 0.0)
                  {
                    stT.sData[iCnt][j] = new String( "" + st.stItem[i].dFrom );
				  }
				  else if (st.stItem[i].system_e.length() > 1)
				  {
                    stT.sData[iCnt][j] = new String( "" + st.stItem[i].dFrom );
				  }
                  break;
                case horizon.bio.bioStandardTools._KEYE:  // Strat Unit KEY
                  if (st.stItem[i].sKey_e.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sKey_e );
                  break;
                case horizon.bio.bioStandardTools._SYSE:  // System
                  if (st.stItem[i].system_e.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].system_e );
                  break;
                case horizon.bio.bioStandardTools._SERE:  // Series
                  if (st.stItem[i].series_e.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].series_e );
                  break;
                case horizon.bio.bioStandardTools._SSYSE: // Subsystem
                  if (st.stItem[i].subSystem_e.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subSystem_e );
                  break;
                case horizon.bio.bioStandardTools._SSERE: // Subseries
                  if (st.stItem[i].subSeries_e.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subSeries_e );
                  break;

                case horizon.bio.bioStandardTools._TO:    // Age To",
                  if (st.stItem[i].dTo > 0.0)
                  {
                    stT.sData[iCnt][j] = new String( "" + st.stItem[i].dTo );
				  }
				  else if (st.stItem[i].system_l.length() > 1)
				  {
                    stT.sData[iCnt][j] = new String( "" + st.stItem[i].dTo );
				  }
                  break;
                case horizon.bio.bioStandardTools._KEYL:  // Strat Unit KEY
                  if (st.stItem[i].sKey_l.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].sKey_l );
                  break;
                case horizon.bio.bioStandardTools._SYSL:  // System
                  if (st.stItem[i].system_l.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].system_l );
                  break;
                case horizon.bio.bioStandardTools._SERL:  // Series
                  if (st.stItem[i].series_l.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].series_l );
                  break;
                case horizon.bio.bioStandardTools._SSYSL: // Subsystem
                  if (st.stItem[i].subSystem_l.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subSystem_l );
                  break;
                case horizon.bio.bioStandardTools._SSERL: // Subseries
                  if (st.stItem[i].subSeries_l.length() > 1)
                    stT.sData[iCnt][j] = new String( st.stItem[i].subSeries_l );
                  break;
              }
            }

            iCnt++;
		  }
        }
      }
    }

    return (stT);
  }
}

/*
 *  @version 1.1 11/29/2011
 *  @author  John Victorine
 */
