/*
 * @las3WriteFile.java Version 1.1 02/10/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.io;

import java.io.*;
import java.awt.*;

import util.utilFileIO;
import cmn.cmnString;

import iqstrat.iqstratHeadersStruct;  // Well Headers Information Structure
import las3.las3ListStruct;            // LAS Version 3 File Data List Structure
import las3.las3Struct;                // LAS Version 3 File Data Structure
import las.lasFileDataStruct;         // LAS File Data Structure
import las3.io.las3ReadFile;           // Open, Read & Parse LAS Files

/** Class las3WriteFile
 *  <p> This class will open an create a LAS Version 3.0 with the data created
 *      This program will try to maintain the Original Data plus the added
 *      Profile Web App data..
 *
 *  @version 1.1 02/10/2010
 *  @author  John R. Victorine
 */

public class las3WriteFile
{
  private las3ListStruct       stLAS3   = null;
  private iqstratHeadersStruct stHeader = null;
  private lasFileDataStruct    stLAS    = null;

  private int    iError = util.utilFileIO.NOERROR;
  private String sError = new String("");

  private int SPACE = 15;

  // Columns

  public static final int _MNEMONIC    = las3.las3Struct._MNEMONIC;
  public static final int _UNIT        = las3.las3Struct._UNIT;
  public static final int _VALUE       = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT      = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION = las3.las3Struct._ASSOCIATION;

  // ~Well Information Constants

  public static final int _START_DEPTH = las3.las3Parameters._START_DEPTH;
  public static final int _END_DEPTH   = las3.las3Parameters._END_DEPTH;
  public static final int _STEP_BY     = las3.las3Parameters._STEP_BY;
  public static final int _NULL_VALUE  = las3.las3Parameters._NULL_VALUE;

  public static final int _COMPANY     = las3.las3Parameters._COMPANY;
  public static final int _WELL        = las3.las3Parameters._WELL;
  public static final int _FIELD       = las3.las3Parameters._FIELD;
  public static final int _SECTION     = las3.las3Parameters._SECTION;
  public static final int _TOWNSHIP    = las3.las3Parameters._TOWNSHIP;
  public static final int _RANGE       = las3.las3Parameters._RANGE;
  public static final int _LOCATION    = las3.las3Parameters._LOCATION;
  public static final int _LOCATION_1  = las3.las3Parameters._LOCATION_1;
  public static final int _LOCATION_2  = las3.las3Parameters._LOCATION_2;
  public static final int _COUNTY      = las3.las3Parameters._COUNTY;
  public static final int _STATE       = las3.las3Parameters._STATE;
  public static final int _COUNTRY     = las3.las3Parameters._COUNTRY;
  public static final int _PROVINCE    = las3.las3Parameters._PROVINCE;
  public static final int _SERVICE     = las3.las3Parameters._SERVICE;
  public static final int _LICENSE     = las3.las3Parameters._LICENSE;
  public static final int _DATE        = las3.las3Parameters._DATE;
  public static final int _API         = las3.las3Parameters._API;
  public static final int _UWI         = las3.las3Parameters._UWI;
  public static final int _LATI        = las3.las3Parameters._LATI;
  public static final int _LONG        = las3.las3Parameters._LONG;
  public static final int _GDAT        = las3.las3Parameters._GDAT;
  public static final int _X           = las3.las3Parameters._X;
  public static final int _Y           = las3.las3Parameters._Y;
  public static final int _HZCS        = las3.las3Parameters._HZCS;
  public static final int _UTM         = las3.las3Parameters._UTM;
  public static final int _WELL_TOTAL  = las3.las3Parameters._WELL_TOTAL;

  private String WELL[][] = las3.las3Parameters.WELL_PARAMETERS;

  /** Constructor las3WriteFile()
   *  <p> This is the constructor for this class
   *  @param stLAS3   = the LAS Version 3 File Data List Structure
   *  @param stHeader = the Header Information Data Structure
   *  @param stLAS    = the LAS File Data Structure
   */

  public las3WriteFile( las3ListStruct       stLAS3,
                        iqstratHeadersStruct stHeader,
                        lasFileDataStruct    stLAS )
  {
    this.stLAS3   = stLAS3;
    this.stHeader = stHeader;
    this.stLAS    = stLAS;
  }

  /** Method delete()
   * <p> This method will force Java to deallocate unused memory
   */

  public void delete()
  {
    sError   = null;
    stLAS3   = null;
    stHeader = null;
    stLAS    = null;
  }

  /** Method write()
   *  <p> This is the will open a LAS version 3.0 LAS File and write the
   *      Profile Web App Data Contents to the File
   *  @param sPath     = The Directory Path for the LAS 3 File
   *  @param sFilename = The Filename of the LAS 3 File.
   */

  public void write( String sPath, String sFilename )
  {
    int i=0;
    int        iLog     = 0;
    int        iArray[] = null;
    int        iParam[] = null;
    utilFileIO pIO      = new utilFileIO();
    las3Struct st       = null;
    String     sDelim   = ",";
/*
    if (stLAS.sFilename.equals(sFilename))
    {
      sFilename = new String("NEW_"+sFilename);
    }
*/
    if (stLAS3.dVersion == 3.0)
    {
      sDelim = new String(stLAS3.sDelim);
    }

    // Create the GEMINI Data directory path if it does not exist.

    gemini.io.geminiIO.mkDir(sPath);

    // IF the Directory Path Exists THEN

    if (gemini.io.geminiIO.exists(sPath))
    {
      // Open the File for write

      pIO.Open( util.utilFileIO.DIRECTORY,
                util.utilFileIO.WRITE,
                sPath,
                sFilename );

      // Count the number of Data Sections

      iArray = new int[las3.las3Constants._TOTAL];
      iParam = new int[las3.las3Constants._TOTAL];

      for (i=0; i<las3.las3Constants._TOTAL; i++)
      {
        iArray[i] = 0;
        iParam[i] = 0;
      }

      iArray[las3.las3Constants._VERSION] = 1;
      iArray[las3.las3Constants._WELL]    = 1;

      if (stLAS3 != null)
      {
        for (i=0; i<stLAS3.iCount; i++)
        {
          switch (stLAS3.stItem[i].iType)
          {
            case las3.las3Constants._LOG:  // Log Data Section
              iArray[las3.las3Constants._LOG]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._LOG]++;
              break;
            case las3.las3Constants._CORE:  // Core Data Section
              iArray[las3.las3Constants._CORE]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._CORE]++;
              break;
            case las3.las3Constants._DRILLING:  // Drilling Data Section
              iArray[las3.las3Constants._DRILLING]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._DRILLING]++;
              break;
            case las3.las3Constants._INCLINOMETRY:  // Inclinometry Data Section
              iArray[las3.las3Constants._INCLINOMETRY]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._INCLINOMETRY]++;
              break;
            case las3.las3Constants._TOPS:  // Tops Data Section
              iArray[las3.las3Constants._TOPS]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._TOPS]++;
              break;
            case las3.las3Constants._TEST:  // Test Data Section
              iArray[las3.las3Constants._TEST]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._TEST]++;
              break;
            case las3.las3Constants._PERFORATION:  // Perforation Data Section
              iArray[las3.las3Constants._PERFORATION]++;
              if (stLAS3.stItem[i].iParamRows > 0)
                iParam[las3.las3Constants._PERFORATION]++;
              break;
          }
        }
      }

      // Write the LAS Version 3.0 File

      // -- Write the ~Version Section

      Version( pIO );

      // -- Write the ~Well Section

//    Well_Parameters( pIO );

      if (stLAS3 != null)
      {
        // -- Write the ~Well Section

        Well_Parameters( pIO, stLAS3.stItem[0] );

        // -- Add Other Data Sections

        for (i=1; i<stLAS3.iCount; i++)
        {
          parameters( pIO, stLAS3.stItem[i],
                      iArray[stLAS3.stItem[i].iType],
                      iParam[stLAS3.stItem[i].iType] );
          curves( pIO, stLAS3.stItem[i],
                  iArray[stLAS3.stItem[i].iType],
                  iParam[stLAS3.stItem[i].iType] );

          if (stLAS3.stItem[i].iType != las3.las3Constants._LOG)
          {
            data( pIO, stLAS3.stItem[i], iArray[stLAS3.stItem[i].iType], sDelim );
          }
        }

        for (i=1; i<stLAS3.iCount; i++)
        {
          if ((stLAS3.stItem[i].iType == las3.las3Constants._LOG) ||
              (stLAS3.stItem[i].iType == las3.las3Constants._CORE))
          {
            curvesIQ( pIO, stLAS3.stItem[i], iArray[stLAS3.stItem[i].iType] );
          }
        }

        for (i=1; i<stLAS3.iCount; i++)
        {
          if (stLAS3.stItem[i].iType == las3.las3Constants._LOG)
          {
//            data( pIO, stLAS3.stItem[i], iArray[stLAS3.stItem[i].iType], sDelim );
            data( pIO, stLAS3.stItem[i],
                  iArray[stLAS3.stItem[i].iType], sDelim, stLAS );
          }
        }
      }

      // Close the File

      if (iError == util.utilFileIO.NOERROR)
        pIO.Close();
	}
  }

  /** Method Version()
   * <p> This method will build the LAS 3 File Output String for printing.
   *  @param  pIO   = Utility File IO Pointer
   */

  public void Version( utilFileIO pIO )
  {
    String str   = "~Version\r\n";
    String sMNEM = "";
    String sUNIT = "";
    String sVAL  = "";
    String sDES  = "";
    String sTemp = "";

    sMNEM = String.format("%-5s ", "#MNEM");
    sUNIT = String.format(".%-8s ", "UNIT");
    sVAL = String.format(" %"+SPACE+"s ", "VALUE");
    sDES = String.format(" : %-30s ", "DESCRIPTION");

    str = new String(str + sMNEM + sUNIT + sVAL + sDES + "\r\n");

    sMNEM = String.format(" %-4s ", "VERS");
    sUNIT = String.format(".%-8s ", "");
    sVAL = String.format(" %"+SPACE+"s ", "3.0");
    sDES = String.format(" : %-30s ", "CWLS LOG ASCII STANDARD - VERSION 3.0");

    str = new String(str + sMNEM + sUNIT + sVAL + sDES + "\r\n");

    sMNEM = String.format(" %-4s ", "WRAP");
    sUNIT = String.format(".%-8s ", "");
    sVAL = String.format(" %"+SPACE+"s ", "NO");
    sDES = String.format(" : %-30s ", "ONE LINE PER DEPTH STEP");

    str = new String(str + sMNEM + sUNIT + sVAL + sDES + "\r\n");

    sTemp = new String( "COMMA" );
    if (stLAS3 != null)
    {
      if (stLAS3.dVersion == 3.0)
      {
        if (stLAS3.sDelim.equals(" "))
          sTemp = new String( "SPACE" );
        else if (stLAS3.sDelim.equals(","))
          sTemp = new String( "COMMA" );
        else
          sTemp = new String( "TAB" );
      }
    }

    sMNEM = String.format(" %-4s ", "DLM");
    sUNIT = String.format(".%-8s ", "");
    sVAL = String.format(" %"+SPACE+"s ", sTemp);
    sDES = String.format(" : %-30s ", "DELIMITING CHARACTER (SPACE TAB OR COMMA)");

    str = new String(str + sMNEM + sUNIT + sVAL + sDES + "\r\n");

    str = new String(str + "\r\n");

    pIO.Write( str );
    iError = pIO.getErrorID();
    sError = pIO.getError();
  }

  /** Method Well_Parameters()
   * <p> This method will build the LAS 3 File Output String for printing.
   *  @param  pIO   = Utility File IO Pointer
   *  @param  st    = LAS Version 3.0 Data Structure
   */

  public void Well_Parameters( utilFileIO pIO, las3Struct st )
  {
    int i=0;
    String str    = "~Well\r\n";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;

    if (st != null)
    {
      if (st.iParamRows > 0)
      {
        SPACE = 10;

        if (st.iParamRows > 0)
        {
          for (i=0; i<st.iParamRows; i++)
          {
            len = st.sParams[i][_MNEMONIC].length();
            if (iMnem < len)   iMnem = len;

            len = st.sParams[i][_UNIT].length();
            if (iUnit < len)   iUnit = len;

            len = st.sParams[i][_VALUE].length();
            if (iValue < len)  iValue = len;
          }
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n");

        for (i=0; i<st.iParamRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.sParams[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.sParams[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.sParams[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.sParams[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.sParams[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.sParams[i][_FORMAT]);

          sASSOC = new String("");
          if (st.sParams[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.sParams[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n" );
        }

        str = new String(str + "\r\n");

        pIO.Write(str);
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }

  /** Method parameters()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  pIO   = Utility File IO Pointer
   *  @param  st    = LAS Version 3.0 Data Structure
   *  @param  iTTL  = Total Number of Data Types in the LAS File
   *  @param  iPTTL = Total Number of Parameter Sections in the Data Type
   */

  public void parameters( utilFileIO pIO, las3Struct st, int iTTL, int iPTTL )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    int    iCont  = 1;

    if (st != null)
    {
	  if ((st.iType == las3.las3Constants._LOG) &&
	      (iTTL > 1) &&
	      (st.iCurveRows == 0))
	  {
	    iCont = 0;
      }

      if ((st.iParamRows > 0) && (iCont == 1))
      {
        if (st.iType == las3.las3Constants._LOG)
          str = new String("~Parameter\r\n");
        else
          str = new String("~" + st.sType + "_Parameter\r\n");

//        iArray = st.iArray + 1;
        iArray = st.iArrayP + 1;

        if ((iTTL > 1) && (iPTTL > 1))
          str = new String("~" + st.sType + "_Parameter[" + iArray + "]\r\n");

        SPACE = 10;

        if (st.iParamRows > 0)
        {
          for (i=0; i<st.iParamRows; i++)
          {
            len = st.sParams[i][_MNEMONIC].length();
            if (iMnem < len)   iMnem = len;

            len = st.sParams[i][_UNIT].length();
            if (iUnit < len)   iUnit = len;

            len = st.sParams[i][_VALUE].length();
            if (iValue < len)  iValue = len;
          }
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n");

        for (i=0; i<st.iParamRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.sParams[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.sParams[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.sParams[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.sParams[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.sParams[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.sParams[i][_FORMAT]);

          sASSOC = new String("");
          if (st.sParams[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.sParams[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n" );
        }

        str = new String(str + "\r\n");

        pIO.Write(str);
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }

  /** Method curvesIQ()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  pIO   = Utility File IO Pointer
   *  @param  st    = LAS Version 3.0 Data Structure
   *  @param  iTTL  = Total Number of Data Types in the LAS File
   */

  public void curvesIQ( utilFileIO pIO, las3Struct st, int iTTL )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    String sType  = "";

    if (st != null)
    {
      if (st.iSelectRows > 0)
      {
        if (st.sType.equals("Log"))
          sType = "Las";
        else if (st.sType.equals("Core"))
          sType = "Rock";

        str = new String( "~IQ_" + sType + "_Parameter\r\n" );

//        iArray = st.iArray + 1;
        iArray = st.iArrayP + 1;

        if (iTTL > 1)
          str = new String( "~IQ_" + sType + "_Parameter[" + iArray + "]\r\n" );

        SPACE = 10;

        if (st.iSelectRows > 0)
        {
          for (i=0; i<st.iSelectRows; i++)
          {
            len = st.select[i][_MNEMONIC].length();
            if (iMnem < len)   iMnem = len;

            len = st.select[i][_UNIT].length();
            if (iUnit < len)   iUnit = len;

            len = st.select[i][_VALUE].length();
            if (iValue < len)  iValue = len;
          }
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n");

        for (i=0; i<st.iSelectRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.select[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.select[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.select[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.select[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.select[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.select[i][_FORMAT]);

          sASSOC = new String("");
          if (st.select[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.select[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n" );
        }

        str = new String(str + "\r\n");

        pIO.Write(str);
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }

  /** Method curves()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  pIO   = Utility File IO Pointer
   *  @param  st    = LAS Version 3.0 Data Structure
   *  @param  iTTL  = Total Number of Data Types in the LAS File
   *  @param  iPTTL = Total Number of Parameter Sections in the Data Type
   */

  public void curves( utilFileIO pIO, las3Struct st, int iTTL, int iPTTL )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;

    if (st != null)
    {
      if (st.iCurveRows > 0)
      {
        if (st.iType == las3.las3Constants._LOG)
          str = new String("~Curve\r\n");
        else
          str = new String("~" + st.sType + "_Definition\r\n");

//        iArray = st.iArray + 1;
        iArray = st.iArrayP + 1;

        if ((iTTL > 1) && (iPTTL > 1))
          str = new String("~" + st.sType + "_Definition[" + iArray + "]\r\n");
        else if (iTTL > 1)
          str = new String("~" + st.sType + "_Definition[" + iArray + "]\r\n");

        SPACE = 10;

        for (i=0; i<st.iCurveRows; i++)
        {
          len = st.sCurves[i][_MNEMONIC].length();
          if (iMnem < len)   iMnem = len;

          len = st.sCurves[i][_UNIT].length();
          if (iUnit < len)   iUnit = len;

          len = st.sCurves[i][_VALUE].length();
          if (iValue < len)  iValue = len;
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n");

        for (i=0; i<st.iCurveRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.sCurves[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.sCurves[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.sCurves[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.sCurves[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.sCurves[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.sCurves[i][_FORMAT]);

          sASSOC = new String("");
          if (st.sCurves[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.sCurves[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\r\n" );
        }

        str = new String(str + "\r\n");

        pIO.Write(str);
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }

  /** Method data()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  pIO    = Utility File IO Pointer
   *  @param  st     = LAS Version 3.0 Data Structure
   *  @param  iTTL   = Total Number of Data Types in the LAS File
   *  @param  sDelim = Delimiter
   */

  public void data( utilFileIO pIO, las3Struct st, int iTTL, String sDelim )
  {
    int i,j;
    String str    = "";
    int    iArray = 0;

    if (st != null)
    {
//      if (st.iRows > 0)
      if ((st.iRows > 0) && (st.iCurveRows > 0))
      {
        if (st.iType == las3.las3Constants._LOG)
          str = new String("~ASCII\r\n");
        else
          str = new String("~" + st.sType + "_Data | " +
                                 st.sType + "_Definition \r\n");

//        iArray = st.iArray+1;
        iArray = st.iArrayP + 1;

        if (iTTL > 1)
        {
          str = new String(
              "~" + st.sType + "_Data[" + iArray + "] | " +
                    st.sType + "_Definition[" + iArray + "] \r\n");
        }

        pIO.Write( str );

        str = new String("#");
        for (i=0; i<st.iCurveRows; i++)
        {
          str = new String( str + " " +st.sCurves[i][_MNEMONIC] );
        }
        str = new String(str + "\r\n");
        pIO.Write( str );

        for (i=0; i<st.iRows; i++)
        {
          str = new String(" ");
          for (j=0; j<st.iColumns; j++)
          {
            if (j!=0)
              str = new String(str + sDelim);

            if ((st.sCurves[j][_FORMAT].equals("S")) ||
                (st.sCurves[j][_FORMAT].equals("s")))
              str = new String( str + "\"" + st.sData[i][j] + "\"" );
            else
              str = new String( str + st.sData[i][j] );
          }

          str = new String(str + "\r\n");
          pIO.Write( str );
        }

        str = new String("\r\n");

        pIO.Write( str );
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }

  /** Method data()
   * <p> This method will build the log data output string for printing.
   *  @param  pIO    = Utility File IO Pointer
   *  @param  st     = LAS Version 3.0 Data Structure
   *  @param  iTTL   = Total Number of Data Types in the LAS File
   *  @param  sDelim = Delimiter
   *  @param  stLAS  = LAS File Data Structure
   */

  public void data( utilFileIO pIO, las3Struct st, int iTTL, String sDelim,
                    lasFileDataStruct stLAS )
  {
    int i,j;
    String str     = "";
    int    iArray  = 0;
    int    iFound  = -1;
    int    iType   = -1;
    String sData[] = null;

    if ((st != null) && (stLAS != null))
    {
//      if (st.iRows > 0)
//      if (stLAS.iRows > 0)
      if ((stLAS.iRows > 0) && (st.iCurveRows > 0))
      {
        if (st.iType == las3.las3Constants._LOG)
          str = new String("~ASCII\r\n");
        else
          str = new String("~" + st.sType + "_Data | " +
                                 st.sType + "_Definition \r\n");

//        iArray = st.iArray+1;
        iArray = st.iArrayP + 1;

        if (iTTL > 1)
        {
          str = new String(
              "~" + st.sType + "_Data[" + iArray + "] | " +
                    st.sType + "_Definition[" + iArray + "] \r\n");
        }

        pIO.Write( str );

        str = new String("#");
        for (i=0; i<st.iCurveRows; i++)
        {
          str = new String( str + " " +st.sCurves[i][_MNEMONIC] );
        }
        str = new String(str + "\r\n");
        pIO.Write( str );

        // Get Log Data from LAS File

//        iType = util.utilFileIO.DIRECTORY;
//        if (stLAS.sReadMethod.equals("URL"))
//          iType = util.utilFileIO.URL_FILE;

        las3ReadFile p3 = new las3ReadFile();
//        p3.Open( iType, stLAS.sDirectory, stLAS.sFilename ); // Open LAS File
        p3.Open( st.iTypeP, st.sDirectory, st.sFilename ); // Open LAS File
//        iFound = p3.ReadFile( las3.las3Constants._LOG, st.iArray, stLAS3.dVersion );
        iFound = p3.ReadFile( las3.las3Constants._LOG, st.iArray, st.dVersion );

        if (iFound == 1)        // IF Log Section is Found
        {
          while (iFound == 1) // While Data line is not null
          {
            sData = p3.getNextLine( st.sDelimP, st.sWrapP, st.dVersion, st.iColumns );
//            sData = p3.getNextLine(
//                    stLAS3.sDelim, stLAS3.sWrap, stLAS3.dVersion, st.iColumns);

            if (sData == null)
              iFound = -1;
            else
            {
              if (sData.length == st.iColumns)
              {
                str = new String(" ");
                for (j=0; j<st.iColumns; j++)
                {
                  if (j!=0)
                    str = new String(str + sDelim);

                  if ((st.sCurves[j][_FORMAT].equals("S")) ||
                      (st.sCurves[j][_FORMAT].equals("s")))
                    str = new String( str + "\"" + sData[j] + "\"" );
                  else
                    str = new String( str + sData[j] );
                }

                sData = null;
              }

              str = new String(str + "\r\n");
              pIO.Write( str );
            }
          }
        }
        p3.Close();                                           // Close LAS File

/*
        for (i=0; i<st.iRows; i++)
        {
          str = new String(" ");
          for (j=0; j<st.iColumns; j++)
          {
            if (j!=0)
              str = new String(str + sDelim);

            if ((st.sCurves[j][_FORMAT].equals("S")) ||
                (st.sCurves[j][_FORMAT].equals("s")))
              str = new String( str + "\"" + st.sData[i][j] + "\"" );
            else
              str = new String( str + st.sData[i][j] );
          }

          str = new String(str + "\r\n");
          pIO.Write( str );
        }
*/
        str = new String("\r\n");

        pIO.Write( str );
        iError = pIO.getErrorID();
        sError = pIO.getError();
      }
    }
  }
}

/*
 *  @version 1.1 02/10/2010
 *  @author  John Victorine
 */
