/*
 * @las3ReadFile.java Version 1.1 02/09/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.io;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import util.utilFileIO;
import las3.las3ListStruct;
import las3.las3Struct;
import las3.las3BuildListStruct;
import las3.las3BuildStruct;

/** Class las3ReadFile
 *  <p> This Class is designed to open an ascii text las file and read
 *      the contents and parse the data into the LAS Structure.
 *
 *  @version 1.1 02/09/2010
 *  @author  John R. Victorine
 */

public class las3ReadFile
{
  // Add or Update indicator

  public static final int _ADD    = 0;
  public static final int _MODIFY = 1;

  // Title Name of Data Section Identifiers

  private static final int _NONE          = las3.las3Constants._NONE;

  private static final int _VERSION       = las3.las3Constants._VERSION;
  private static final int _WELL          = las3.las3Constants._WELL;
  private static final int _LOG           = las3.las3Constants._LOG;
  private static final int _CORE          = las3.las3Constants._CORE;
  private static final int _DRILLING      = las3.las3Constants._DRILLING;
  private static final int _INCLINOMETRY  = las3.las3Constants._INCLINOMETRY;
  private static final int _TOPS          = las3.las3Constants._TOPS;
  private static final int _TEST          = las3.las3Constants._TEST;
  private static final int _PERFORATION   = las3.las3Constants._PERFORATION;

  private static final int _IQ_CONTROL    = las3.las3Constants._IQ_CONTROL;
  private static final int _IQ_LOG        = las3.las3Constants._IQ_LOG;
  private static final int _IQ_CORE       = las3.las3Constants._IQ_CORE;

  private static final int _IQ_SEQ_STRAT  = las3.las3Constants._IQ_SEQ_STRAT;

  private static final int _IQ_FLOW       = las3.las3Constants._IQ_FLOW;
  private static final int _IQ_PFEFFER    = las3.las3Constants._IQ_PFEFFER;

  private static final int _IQ_GEO_REPORT = las3.las3Constants._IQ_GEO_REPORT;

  private static final int _IQ_SHALE      = las3.las3Constants._IQ_SHALE;

  private static final int _IQ_IMAGES     = las3.las3Constants._IQ_IMAGES;
  private static final int _IQ_BRINE      = las3.las3Constants._IQ_BRINE;

  private static final int _IQ_BIO        = las3.las3Constants._IQ_BIO;
  private static final int _IQ_ENV        = las3.las3Constants._IQ_ENV;

  private static final int _TOTAL         = las3.las3Constants._TOTAL;

  private static final int _C_LOG         = 0;
  private static final int _C_CORE        = 1;
  private static final int _C_TOPS        = 2;
  private static final int _C_PERF        = 3;
  private static final int _C_DRILL       = 4;
  private static final int _C_INCLI       = 5;
  private static final int _C_TEST        = 6;
  private static final int _C_CONTROL     = 7;
  private static final int _C_SEQ         = 8;
  private static final int _C_FLOW        = 9;
  private static final int _C_PFEFFER     = 10;
  private static final int _C_REMARKS     = 17;
  private static final int _C_SHALE       = 18;
  private static final int _C_IMAGES      = 19;
  private static final int _C_BRINE       = 20;
  private static final int _C_BIO         = 21;
  private static final int _C_ENV         = 22;

  // Global Variables

  private utilFileIO          pLAS     = null;
  private las3ListStruct      stLAS    = null;
  private int                 iLAS[]   = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  private int                 iLog     = 0;
  private int                 iCore    = 0;
  private int                 iTops    = 0;
  private int                 iPerf    = 0;
  private int                 iDrill   = 0;
  private int                 inclin   = 0;
  private int                 iTest    = 0;
  private int                 iControl = 0;
  private int                 iBio     = 0;
  private int                 iEnv     = 0;
  private int                 iSeq     = 0;
  private int                 iFlow    = 0;
  private int                 iPfeffer = 0;
  private int                 iRemarks = 0;
  private int                 iShale   = 0;
  private int                 iImages  = 0;
  private int                 iBrine   = 0;

  private int                 iData    = _NONE;
  private int                 iSet     = _NONE;
  private int                 iList    = _NONE;

  private las3BuildListStruct stList   = null;

  private int    iError = util.utilFileIO.NOERROR;
  private String sError = new String("");

  /** Construct las3ReadFile()
   *  <p> This is the Constructor for this class.
   */

  public las3ReadFile() { }

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    if (pLAS != null)
      pLAS.delete();
    pLAS = null;

    if (stLAS != null)
      stLAS.delete();
    stLAS = null;

    iLAS     = null;

    iLog     = 0;
    iCore    = 0;
    iTops    = 0;
    iPerf    = 0;
    iDrill   = 0;
    inclin   = 0;
    iTest    = 0;
    iControl = 0;
    iBio     = 0;
    iEnv     = 0;
    iSeq     = 0;
    iFlow    = 0;
    iPfeffer = 0;
    iRemarks = 0;
    iShale   = 0;
    iImages  = 0;
    iBrine   = 0;

    iData = _NONE;
    iSet  = _NONE;
    iList = _NONE;

    if (stList != null)
      stList.delete();
    stList = null;

    iError = 0;
    sError = null;
  }

  /** Method Read()
   * <p> This method will read and parse the LAS File into its data structures.
   * @param iFileID    = To access the file as URL or FILE.
   * @param sDirectory = File URL or Directory Path
   * @param sFilename  = File Name
   */

  public void Read( int iFileID, String sDirectory, String sFilename )
  {
    int i,j   = 0;
    int iList = 0;

    stLAS  = new las3ListStruct();  // Initialize the LAS 3 List Data Structure

    // Determine the number of Data Sections in this LAS File

    Open( iFileID, sDirectory, sFilename );
    doSECTIONCount();
    Close();

    // Initialize the Build Data Section Data Structures

    if ((iLog > 0)     || (iCore > 0)    || (iTops > 0)  || (iPerf > 0) ||
        (iControl > 0) ||
        (iDrill > 0)   || (inclin > 0)   || (iTest > 0)  ||
        (iFlow > 0)    || (iPfeffer > 0) ||
        (iSeq > 0)     || (iRemarks > 0) || (iShale > 0) ||
        (iImages > 0)  || (iBrine > 0)   || (iBio > 0)   || (iEnv > 0))
    {
      stList        = new las3BuildListStruct();
      stList.iCount = iLog + iCore + iTops + iPerf + iControl + iSeq +
                      iDrill + inclin + iTest + iFlow + iPfeffer +
                      iRemarks + iShale + iImages + iBrine + iBio + iEnv;
      stList.stItem = new las3BuildStruct[stList.iCount];

      for (i=0; i<stList.iCount; i++)
        stList.stItem[i] = new las3BuildStruct();

      iList = 0;
      for (j=0; j<iLog; j++)
      {
        stList.stItem[iList].iType  = _C_LOG;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iCore; j++)
      {
        stList.stItem[iList].iType  = _C_CORE;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iTops; j++)
      {
        stList.stItem[iList].iType  = _C_TOPS;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iPerf; j++)
      {
        stList.stItem[iList].iType  = _C_PERF;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iDrill; j++)
      {
        stList.stItem[iList].iType  = _C_DRILL;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<inclin; j++)
      {
        stList.stItem[iList].iType  = _C_INCLI;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iTest; j++)
      {
        stList.stItem[iList].iType  = _C_TEST;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iFlow; j++)
      {
        stList.stItem[iList].iType  = _C_FLOW;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iPfeffer; j++)
      {
        stList.stItem[iList].iType  = _C_PFEFFER;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iControl; j++)
      {
        stList.stItem[iList].iType  = _C_CONTROL;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iBio; j++)
      {
        stList.stItem[iList].iType  = _C_BIO;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iEnv; j++)
      {
        stList.stItem[iList].iType  = _C_ENV;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iSeq; j++)
      {
        stList.stItem[iList].iType  = _C_SEQ;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iRemarks; j++)
      {
        stList.stItem[iList].iType  = _C_REMARKS;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iShale; j++)
      {
        stList.stItem[iList].iType  = _C_SHALE;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iImages; j++)
      {
        stList.stItem[iList].iType  = _C_IMAGES;
        stList.stItem[iList].iArray = j;
        iList++;
      }

      for (j=0; j<iBrine; j++)
      {
        stList.stItem[iList].iType  = _C_BRINE;
        stList.stItem[iList].iArray = j;
        iList++;
      }
    }

    // Count the number of rows within each data section

    Open( iFileID, sDirectory, sFilename );
    doASCIICount();
    Close();

    // Parse the LAS 3 File into the LAS File Data Structure

    Open( iFileID, sDirectory, sFilename );
    ReadLASFile();
    Close();
  }

  /* ====================================================================== *
   * -------------------------- LAS FILE METHODS -------------------------- *
   * ====================================================================== */

  /** Method Open( int iFileID, String sDirectory, String sFilename )
    * <p> This Method will open the digital LAS file.
    * @param iFileID    = To access the file as URL or FILE.
    * @param sDirectory = File URL or Directory Path
    * @param sFilename  = File Name
    */

  public void Open( int iFileID, String sDirectory, String sFilename )
  {
    pLAS = new utilFileIO();
    pLAS.Open(iFileID, util.utilFileIO.READ, sDirectory, sFilename);

    iError = pLAS.getErrorID();     // Get the Error Code ID
    sError = pLAS.getError();       // Get the Error
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close() { pLAS.Close(); }

  /** Method doASCIICount()
   * <p> This method will count the number of log data rows that are available
   */

  public void doSECTIONCount()
  {
    int    iSection   = las3.las3Constants._NONE;
    int    iGroup     = las3.las3Constants._NONE;
    int    iArray     = las3.las3Constants._NONE;
    String sTitle[]   = {"",""};
    String section[]  = {"","",""};

    iLog     = 0;
    iCore    = 0;
    iTops    = 0;
    iPerf    = 0;
    iDrill   = 0;
    inclin   = 0;
    iTest    = 0;
    iControl = 0;
    iBio     = 0;
    iEnv     = 0;
    iSeq     = 0;
    iFlow    = 0;
    iPfeffer = 0;
    iRemarks = 0;
    iShale   = 0;
    iImages  = 0;
    iBrine   = 0;

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pLAS.Next().equals(util.utilFileIO.EOF))
      {
        iError = pLAS.getErrorID();
        sError = pLAS.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (las3.las3Parse.isSectionTitle( pLAS.getString() ))
          {
            sTitle   = las3.las3Parse.splitTitle( pLAS.getString() );
            section  = las3.las3Parse.getSection( sTitle[0], stLAS.dVersion );
            iSection = las3.las3Utility.getSectionID( section[0] );
            iGroup   = las3.las3Utility.getGroupID( section[1] );
            iArray   = 0;
            if (cmn.cmnString.isNumeric( section[2] ))
              iArray = cmn.cmnString.stringToInt( section[2] );

            switch (iSection)
            {
              case _NONE:
              case _VERSION:       // Version Informaiton
              case _WELL:          // Well Information
                break;
              case _LOG:           // Log Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iLog++;
                iLAS[iSection] = iLog;
                break;
              case _CORE:          // Core Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iCore++;
                iLAS[iSection] = iCore;
                break;
              case _TOPS:          // Tops Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iTops++;
                iLAS[iSection] = iTops;
                break;
              case _PERFORATION:   // Perforation Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iPerf++;
                iLAS[iSection] = iPerf;
                break;

              case _DRILLING:      // Drilling Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iDrill++;
                iLAS[iSection] = iDrill;
                break;
              case _INCLINOMETRY:  // Inclinometry Data Section
                if (iGroup == las3.las3Constants._DATA)
                  inclin++;
                iLAS[iSection] = inclin;
                break;
              case _TEST:          // Test Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iTest++;
                iLAS[iSection] = iTest;
                break;

              case _IQ_CONTROL:    // Control Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iControl++;
                iLAS[iSection] = iControl;
                break;
              case _IQ_LOG:        // Selected Logs Curves
              case _IQ_CORE:       // Selected Core Curves
                break;
              case _IQ_BIO:        // Bio Stratigraphy
                if (iGroup == las3.las3Constants._DATA)
                  iBio++;
                iLAS[iSection] = iBio;
                break;
              case _IQ_ENV:        // Depositional Environment
                if (iGroup == las3.las3Constants._DATA)
                  iEnv++;
                iLAS[iSection] = iEnv;
                break;
              case _IQ_SEQ_STRAT:  // Sequence Stratigraphy
                if (iGroup == las3.las3Constants._DATA)
                  iSeq++;
                iLAS[iSection] = iSeq;
                break;
              case _IQ_FLOW:       // PfEFFER Flow Units Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iFlow++;
                iLAS[iSection] = iFlow;
                break;
              case _IQ_PFEFFER:    // PfEFFER Table Data Section
                if (iGroup == las3.las3Constants._DATA)
                  iPfeffer++;
                iLAS[iSection] = iPfeffer;
                break;
              case _IQ_GEO_REPORT: // Remarks/Comments/Notes
                if (iGroup == las3.las3Constants._DATA)
                  iRemarks++;
                iLAS[iSection] = iRemarks;
                break;
              case _IQ_SHALE:      // Shale Gamma Ray Limits
                if (iGroup == las3.las3Constants._DATA)
                  iShale++;
                iLAS[iSection] = iShale;
                break;
              case _IQ_IMAGES:     // Rock Images
                if (iGroup == las3.las3Constants._DATA)
                  iImages++;
                iLAS[iSection] = iImages;
                break;
              case _IQ_BRINE:      // Brine Data
                if (iGroup == las3.las3Constants._DATA)
                  iBrine++;
                iLAS[iSection] = iBrine;
                break;
            }
          }
        }

        if ((!las3.las3Parse.isSectionTitle( pLAS.getString() )) &&
            (!las3.las3Parse.isCommentLine( pLAS.getString().trim() )) &&
            (pLAS.getString().length() > 1) &&
            (iSection != las3.las3Constants._NONE))
        {
          switch (iSection)
          {
            case _NONE:
            case _VERSION:       // Version Informaiton
              addVersionLine( pLAS.getString() );
              break;
          }
        }

      }
    }
  }

  /** Method doASCIICount()
   * <p> This method will count the number of log data rows that are available
   */

  public void doASCIICount()
  {
    int    i = 0;
    int    icnt       = 0;
    int    iSection   = las3.las3Constants._NONE;
    int    iGroup     = las3.las3Constants._NONE;
    int    iArray     = las3.las3Constants._NONE;
    String sTitle[]   = {"",""};
    String section[]  = {"","",""};

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pLAS.Next().equals(util.utilFileIO.EOF))
      {
        iError = pLAS.getErrorID();
        sError = pLAS.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (las3.las3Parse.isSectionTitle( pLAS.getString() ))
          {
            sTitle   = las3.las3Parse.splitTitle( pLAS.getString() );
            section  = las3.las3Parse.getSection( sTitle[0], stLAS.dVersion );
            iSection = las3.las3Utility.getSectionID( section[0] );
            iGroup   = las3.las3Utility.getGroupID( section[1] );

            iArray   = 0;
            if (cmn.cmnString.isNumeric( section[2] ))
              iArray = cmn.cmnString.stringToInt( section[2] );

            if (iArray > 0) iArray--;
          }
        }

        if ((!las3.las3Parse.isSectionTitle( pLAS.getString() )) &&
            (!las3.las3Parse.isCommentLine( pLAS.getString().trim() )) &&
            (iSection != las3.las3Constants._NONE) &&
            (pLAS.getString().trim().length() > 0))
        {
          switch (iSection)
          {
            case _NONE:
            case _VERSION:      // Version Informaiton
            case _WELL:         // Well Information
              break;
            case _LOG:          // Log Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_LOG) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _CORE:         // Core Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_CORE) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _TOPS:         // Tops Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_TOPS) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _PERFORATION:  // Perforation Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_PERF) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;

            case _DRILLING:     // Drilling Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_DRILL) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _INCLINOMETRY: // Inclinometry Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_INCLI) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _TEST:         // Test Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_TEST) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;

            case _IQ_CONTROL:   // Control Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_CONTROL) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_LOG:
            case _IQ_CORE:
              break;
            case _IQ_BIO: // Bio Stratigraphy
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_BIO) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_ENV: // Depositional Environment
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_ENV) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_SEQ_STRAT: // Sequence Stratigraphy
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_SEQ) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;

            case _IQ_FLOW:   // PfEFFER Flow Units Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_FLOW) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_PFEFFER:   // PfEFFER Table Data Section
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_PFEFFER) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;

            case _IQ_GEO_REPORT: // Remarks/Comments/Notes
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_REMARKS) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_SHALE: // Shale
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_SHALE) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_IMAGES: // Rock Images
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_IMAGES) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
            case _IQ_BRINE: // Brine Data
              if (iGroup == las3.las3Constants._DATA)
              {
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == _C_BRINE) &&
                      (stList.stItem[i].iArray == iArray))
                  {
                    stList.stItem[i].iTotal++;
                  }
                }
              }
              break;
          }
        }
      }
    }
  }

  /** Method getNextLine()
   * <p> This method will retrieve the next line from the LAS File
   * @param  sDelim     = Delimiter used to separate data values
   * @param  sWrap      = Is the data line wrapped ( For LAS 2.0 Files )
   * @param  dVersion   = Version number of LAS File
   * @param  iCurveRows = Total number of curves ( For LAS 2.0 Files which wraps )
   * @return sData      = Array of string data values from LAS File
   */

  public String[] getNextLine( String sDelim,   String sWrap,
                               double dVersion, int iCurveRows )
  {
    int    i         = 0;
    String sData[]   = null;
    String sOrderd[] = { "", "", "", "", "", "" };
    int    iNext     = 0;
    int    ilocal    = 0;
    String sOther[]  = null;
    String str       = "";

    if (!pLAS.Next().equals(util.utilFileIO.EOF))
    {
      iError = pLAS.getErrorID();
      sError = pLAS.getError();

      if (iError == util.utilFileIO.NOERROR)
      {
        while ((!las3.las3Parse.isSectionTitle( pLAS.getString() )) &&
               ((las3.las3Parse.isCommentLine( pLAS.getString().trim() )) ||
                (pLAS.getString().length() < 2) ))
        {
          pLAS.Next();
        }

        if ((!las3.las3Parse.isSectionTitle( pLAS.getString() )) &&
            (!las3.las3Parse.isCommentLine( pLAS.getString().trim() )) &&
            (pLAS.getString().length() > 1) )
        {
          str = pLAS.getString();
          if (dVersion == 2.0)
          {
			str = new String( str.replace('\t', ' ') );
		  }

          sData = las3.las3Parse.splitDataLine( str, sDelim );

          if (sWrap.equals("YES"))
          {
            ilocal = 0;
            iNext  = 0;
            sOther = new String[iCurveRows];

            for (i=0; i<sData.length; i++)
            {
              if (iNext < iCurveRows)
              {
                sOther[iNext] = new String( sData[i] );
                iNext++;
              }
            }

            while (ilocal+sData.length < iCurveRows)
            {
              if (!pLAS.Next().equals(util.utilFileIO.EOF))
              {
                iError = pLAS.getErrorID();
                sError = pLAS.getError();
              }

              ilocal = ilocal+sData.length;
              if (!las3.las3Parse.isCommentLine( pLAS.getString().trim() ))
              {
                sData = las3.las3Parse.splitDataLine(
                          pLAS.getString().trim(), sDelim );
                for (i=0; i<sData.length; i++)
                {
                  if (iNext < iCurveRows)
                  {
                    sOther[iNext] = new String( sData[i] );
                    iNext++;
                  }
                }
              }
            }

            if (sOther.length > 0)
            {
              sData = new String[sOther.length];
              for (i=0; i<sOther.length; i++)
              {
                sData[i] = new String( sOther[i] );
              }
            }
          }

        }
      }
    }

    return (sData);
  }

  /** Method ReadFile()
   * <p> This method will read file until Data Section and Data Array matches.
   * @param  kSection = desired section type
   * @param  kArray   = desired data section array
   * @param  dVersion = Version number of LAS File
   * @return iFound   = Desired data section was found
   */

  public int ReadFile( int kSection, int kArray, double dVersion )
  {
    int    iSection   = las3.las3Constants._NONE;
    int    iGroup     = las3.las3Constants._NONE;
    int    iArray     = 0;
    int    iFound     = -1;
    String sTitle[]   = {"",""};
    String section[]  = {"","",""};
    String assoc[]    = {"","",""};

    if (iError == util.utilFileIO.NOERROR)
    {
      while (iFound == -1)
      {
        if (pLAS.Next().equals(util.utilFileIO.EOF))
        {
          iFound = 0;
        }
        else
        {
          iError = pLAS.getErrorID();
          sError = pLAS.getError();

          if (iError == util.utilFileIO.NOERROR)
          {
            if (las3.las3Parse.isSectionTitle( pLAS.getString() ))
            {
              sTitle   = las3.las3Parse.splitTitle( pLAS.getString() );
              section  = las3.las3Parse.getSection( sTitle[0], dVersion );

              iSection = las3.las3Utility.getSectionID( section[0] );
              iGroup   = las3.las3Utility.getGroupID( section[1] );
              iArray   = 0;
              if (cmn.cmnString.isNumeric( section[2] ))
                iArray = cmn.cmnString.stringToInt( section[2] );

              if (iArray > 0) { iArray--; }

              if ((iSection == kSection) &&
                  (iGroup   == las3.las3Constants._DATA)  &&
                  (iArray   == kArray))
              {
                iFound = 1;
              }
            }
          }
          else
            iFound = 0;
        }
      }
    }

    return (iFound);
  }

  /** Method ReadLASFile()
   * <p> This method will read and parse the LAS File into its data structures.
   */

  public void ReadLASFile()
  {
    int    i,j,k=0;
    int    iSection   = las3.las3Constants._NONE;
    int    iGroup     = las3.las3Constants._NONE;
    int    iArray     = 0;
    int    jSection   = las3.las3Constants._NONE;
    int    jGroup     = las3.las3Constants._NONE;
    int    jArray     = 0;
    int    iFound     = -1;
    int    iContinue  = 0;
    String sTitle[]   = {"",""};
    String section[]  = {"","",""};
    String assoc[]    = {"","",""};
    las3Struct st     = null;

    if (iError == util.utilFileIO.NOERROR)
    {
      while (!pLAS.Next().equals(util.utilFileIO.EOF))
      {
        iError = pLAS.getErrorID();
        sError = pLAS.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (las3.las3Parse.isSectionTitle( pLAS.getString() ))
          {
            sTitle   = las3.las3Parse.splitTitle( pLAS.getString() );
//System.out.println(sTitle[0]+" "+sTitle[1]);
            section  = las3.las3Parse.getSection( sTitle[0], stLAS.dVersion );
//if (section[0].length() > 0)
//System.out.println("   SEC:   "+section[0]+" "+section[1]+" "+section[2]);

            iSection = las3.las3Utility.getSectionID( section[0] );
            iGroup   = las3.las3Utility.getGroupID( section[1] );
            iArray   = 0;
            if (cmn.cmnString.isNumeric( section[2] ))
              iArray = cmn.cmnString.stringToInt( section[2] );

            if (iArray > 0) { iArray--; }

//System.out.println(iSection+" "+iGroup+" "+iArray);

            if (iSection > las3.las3Constants._NONE)
            {
              if (iGroup == las3.las3Constants._DATA)
              {
                switch (iSection)
                {
                  case _NONE:
                  case _VERSION: // Version Informaiton
                  case _WELL: // Well Information
                    break;
                  case _LOG: // Log Data Section
                    iData = _C_LOG;
                    iSet  = iArray;
                    break;
                  case _CORE: // Core Data Section
                    iData = _C_CORE;
                    iSet  = iArray;
                    break;
                  case _TOPS: // Tops Data Section
                    iData = _C_TOPS;
                    iSet  = iArray;
                    break;
                  case _PERFORATION: // Perforation Data Section
                    iData = _C_PERF;
                    iSet  = iArray;
                    break;

                  case _DRILLING: // Drilling Data Section
                    iData = _C_DRILL;
                    iSet  = iArray;
                    break;
                  case _INCLINOMETRY: // Inclinometry Data Section
                    iData = _C_INCLI;
                    iSet  = iArray;
                    break;
                  case _TEST: // Test Data Section
                    iData = _C_TEST;
                    iSet  = iArray;
                    break;

                  case _IQ_CONTROL:   // Control Data Section
                    iData = _C_CONTROL;
                    iSet  = iArray;
                    break;
                  case _IQ_LOG:
                  case _IQ_CORE:
                    break;
                  case _IQ_BIO:      // Bio Stratigraphy
                    iData = _C_BIO;
                    iSet  = iArray;
                    break;
                  case _IQ_ENV:      // Depositional Environment
                    iData = _C_ENV;
                    iSet  = iArray;
                    break;
                  case _IQ_SEQ_STRAT: // Sequence Stratigraphy
                    iData = _C_SEQ;
                    iSet  = iArray;
                    break;

                  case _IQ_FLOW:   // PfEFFER Flow Units Data Section
                    iData = _C_FLOW;
                    iSet  = iArray;
                    break;
                  case _IQ_PFEFFER:   // PfEFFER Table Data Section
                    iData = _C_PFEFFER;
                    iSet  = iArray;
                    break;

                  case _IQ_GEO_REPORT: // Remarks/Comments/Notes
                    iData = _C_REMARKS;
                    iSet  = iArray;
                    break;
                  case _IQ_SHALE: // Shale Gamma Ray Limits
                    iData = _C_SHALE;
                    iSet  = iArray;
                    break;
                  case _IQ_IMAGES: // Rock Images
                    iData = _C_IMAGES;
                    iSet  = iArray;
                    break;
                  case _IQ_BRINE: // Brine Data
                    iData = _C_BRINE;
                    iSet  = iArray;
                    break;
                }

                iList = -1;
                for (i=0; i<stList.iCount; i++)
                {
                  if ((stList.stItem[i].iType  == iData) &&
                      (stList.stItem[i].iArray == iSet))
                    iList = i;
                }
              }
            }

            jSection = las3.las3Constants._NONE;
            jGroup   = las3.las3Constants._NONE;
            jArray   = 0;

            if (sTitle[1].length() > 0)
            {
              assoc    = las3.las3Parse.getSection( sTitle[1] );
//if (assoc[0].length() > 0)
//System.out.println("   ASSOC: "+assoc[0]+" "+assoc[1]+" "+assoc[2]);

              jSection = las3.las3Utility.getSectionID( assoc[0] );
              jGroup   = las3.las3Utility.getGroupID( assoc[1] );
              if (cmn.cmnString.isNumeric( assoc[2] ))
                jArray = cmn.cmnString.stringToInt( assoc[2] );

              if (jArray > 0) { jArray--; }
            }
          }
        }

        if ((!las3.las3Parse.isSectionTitle( pLAS.getString() )) &&
            (!las3.las3Parse.isCommentLine( pLAS.getString().trim() )) &&
            (pLAS.getString().length() > 1) &&
            (iSection != las3.las3Constants._NONE))
        {
          switch (iSection)
          {
            case _NONE:
              break;
            case _VERSION:       // Version Informaiton
              addVersionLine( pLAS.getString() );
              break;
            case _WELL:          // Well Information
              addWellLine( pLAS.getString() );
              break;
            case _LOG:           // Log Data Section
            case _CORE:          // Core Data Section
            case _TOPS:          // Tops Data Section
            case _PERFORATION:   // Perforation Data Section
            case _DRILLING:      // Drilling Data Section
            case _INCLINOMETRY:  // Inclinometry Data Section
            case _TEST:          // Test Data Section
            case _IQ_CONTROL:    // Control Data Section
            case _IQ_SEQ_STRAT:  // Sequence Stratigraphy
            case _IQ_FLOW:       // PfEFFER Flow Units Data Section
            case _IQ_PFEFFER:    // PfEFFER Table Data Section
            case _IQ_GEO_REPORT: // Remarks/Comments/Notes
            case _IQ_SHALE:      // Shale Gamma Ray Limits
            case _IQ_IMAGES:     // Rock Images
            case _IQ_BRINE:      // Brine Data
            case _IQ_BIO:        // Bio Stratigraphy Data
            case _IQ_ENV:        // Depositional Environemnt Data
              addLine( pLAS.getString(), iSection, iGroup, iArray,
                                         jSection, jGroup, jArray );
              break;
            case _IQ_LOG:
              jSection = las3.las3Utility.getSectionID( "Log" );
              jGroup   = las3.las3Utility.getGroupID( "Definition" );
              jArray   = iArray;

              addLine0( pLAS.getString(), iSection, iGroup, iArray,
                                          jSection, jGroup, jArray );
              break;
            case _IQ_CORE:
              jSection = las3.las3Utility.getSectionID( "Core" );
              jGroup   = las3.las3Utility.getGroupID( "Definition" );
              jArray   = iArray;

              addLine0( pLAS.getString(), iSection, iGroup, iArray,
                                          jSection, jGroup, jArray );
              break;
          }
        }
      }
    }

    if ((stLAS != null) && (stList != null))
    {
      for (k=0; k<stList.iCount; k++)
      {
        iList    = -1;
        iSection = -1;

        switch (stList.stItem[k].iType)
        {
          case _C_LOG:
            iSection = las3.las3Constants._LOG;
            iList    = k;
            break;
          case _C_CORE:
            iSection = las3.las3Constants._CORE;
            iList    = k;
            break;
          case _C_TOPS:
            iSection = las3.las3Constants._TOPS;
            iList    = k;
            break;
          case _C_PERF:
            iSection = las3.las3Constants._PERFORATION;
            iList    = k;
            break;

          case _C_DRILL:
            iSection = las3.las3Constants._DRILLING;
            iList    = k;
            break;
          case _C_INCLI:
            iSection = las3.las3Constants._INCLINOMETRY;
            iList    = k;
            break;
          case _C_TEST:
            iSection = las3.las3Constants._TEST;
            iList    = k;
            break;

          case _C_CONTROL:
            iSection = las3.las3Constants._IQ_CONTROL;
            iList    = k;
            break;
          case _C_BIO:
            iSection = las3.las3Constants._IQ_BIO;
            iList    = k;
            break;
          case _C_ENV:
            iSection = las3.las3Constants._IQ_ENV;
            iList    = k;
            break;
          case _C_SEQ:
            iSection = las3.las3Constants._IQ_SEQ_STRAT;
            iList    = k;
            break;

          case _C_FLOW:
            iSection = las3.las3Constants._IQ_FLOW;
            iList    = k;
            break;
          case _C_PFEFFER:
            iSection = las3.las3Constants._IQ_PFEFFER;
            iList    = k;
            break;

          case _C_REMARKS:
            iSection = las3.las3Constants._IQ_GEO_REPORT;
            iList    = k;
            break;
          case _C_SHALE:
            iSection = las3.las3Constants._IQ_SHALE;
            iList    = k;
            break;
          case _C_IMAGES:
            iSection = las3.las3Constants._IQ_IMAGES;
            iList    = k;
            break;
          case _C_BRINE:
            iSection = las3.las3Constants._IQ_BRINE;
            iList    = k;
            break;
        }

        if ((iSection > -1) && (iSection != _LOG) && (iList > -1))
        {
          iArray = stList.stItem[iList].iArray;
          st     = las3.las3Utility.getData( iSection, iArray, stLAS );

          if ((st != null) && (stList.stItem[iList].sData != null))
          {
            st.iRows    = stList.stItem[iList].iTotal;
            st.iColumns = stList.stItem[iList].iColumns;
            st.sData    = new String[st.iRows][st.iColumns];

            for (i=0; i<st.iRows; i++)
            {
              for (j=0; j<st.iColumns; j++)
              {
                st.sData[i][j] = new String( stList.stItem[iList].sData[i][j] );
              }
            }

            stLAS = las3.las3Utility.modify( st, stLAS );
          }
        }
      }
    }
  }

  /* ====================================================================== *
   * ---------------------------- GET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method getData()
   * <p> This method will retrieve the LAS 3 File Data List Structure
   * @return stLAS = the LAS 3 File Data List Structure
   */

  public las3ListStruct getData() { return (stLAS); }

  /** Method getLAS()
   * <p> This method will retrieve the LAS 3 File Available Sections Count
   * @return iLAS = the LAS 3 File Available Sections Count
   */

  public int[] getLAS() { return (iLAS); }

  /* ====================================================================== *
   * ---------------------------- ADD METHODS ----------------------------- *
   * ====================================================================== */

  /** Method addVersionLine()
   * <p> This method will retrieve a line from the LAS File for the Version
   *     Section.
   * @param str = the string to be parsed
   */

  private void addVersionLine( String str )
  {
    String sLine[][] = null;
    String sMnem     = "";
    String sValue    = "";
    String sTemp     = "";

    if (str.length() > 0)
    {
      if (stLAS.dVersion == 2.0)
      {
        str = new String( str.replace('\t', ' ') );
      }

      sLine = las3.las3Parse.splitLine( str, stLAS.sDelim );

      if (sLine != null)
      {
        for (int i=0; i<sLine.length; i++)
        {
          if ((sLine[i][1].toUpperCase()).equals("MNEM"))
            sMnem = new String( sLine[i][0] );

          if ((sLine[i][1].toUpperCase()).equals("VALUE"))
            sValue = new String( sLine[i][0] );
        }

        if (sMnem.toUpperCase().equals("VERS"))
        {
          if (cmn.cmnString.isNumeric( sValue ))
          {
            stLAS.dVersion = cmn.cmnString.stringToDouble( sValue );
          }
        }

        if (sMnem.toUpperCase().equals("WRAP"))
          stLAS.sWrap = new String( sValue.toUpperCase() );

        if (sMnem.toUpperCase().equals("DLM"))
        {
          sTemp = new String( sValue.toUpperCase() );
          if (sTemp.equals("COMMA"))
            stLAS.sDelim = new String( "," );
          if (sTemp.equals("TAB"))
            stLAS.sDelim = new String( "\t" );
          if (sTemp.equals("SPACE"))
            stLAS.sDelim = new String( " " );
        }
      }
    }
  }

  /** Method addWellLine()
   * <p> This method will add the well information data structure.
   * @param str = the string to be parsed
   */

  private void addWellLine( String str )
  {
    int        iAction   = _ADD;
    String     sLine[][] = null;
    las3Struct st        = new las3Struct();
    String     sWell[]   = { "", "", "", "", "", "" };

    if (str.length() > 0)
    {
      if (stLAS.dVersion == 2.0)
      {
        str = new String( str.replace('\t', ' ') );
      }

      if (las3.las3Utility.isData( las3.las3Constants._WELL, 0, stLAS ))
        iAction = _MODIFY;

      st    = las3.las3Utility.getData( las3.las3Constants._WELL, 0, stLAS );
      sLine = las3.las3Parse.splitLine( str, stLAS.sDelim );

      if (sLine != null)
      {
        for (int i=0; i<sLine.length; i++)
        {
          for (int j=0; j<2; j++)
          {
            if ((sLine[i][1].toUpperCase()).equals("MNEM"))
            {
              sWell[las3.las3Struct._MNEMONIC] = new String( sLine[i][0] );
            }

            if ((sLine[i][1].toUpperCase()).equals("UNIT"))
            {
              sWell[las3.las3Struct._UNIT] = new String( sLine[i][0] );
            }

            if ((sLine[i][1].toUpperCase()).equals("VALUE"))
            {
              sWell[las3.las3Struct._VALUE] = new String( sLine[i][0] );
            }

            if ((sLine[i][1].toUpperCase()).equals("DESCRIPTION"))
            {
              sWell[las3.las3Struct._DESCRIPTION] = new String( sLine[i][0] );
            }

            if ((sLine[i][1].toUpperCase()).equals("FORMAT"))
            {
              sWell[las3.las3Struct._FORMAT] = new String( sLine[i][0] );
            }

            if ((sLine[i][1].toUpperCase()).equals("ASSOCIATION"))
            {
              sWell[las3.las3Struct._ASSOCIATION] = new String( sLine[i][0] );
            }
          }
        }

        st = las3.las3Utility.addParameters( sWell, st );

        switch (iAction)
        {
          case _ADD:
            stLAS = las3.las3Utility.add( st, stLAS );
            break;
          case _MODIFY:
            stLAS = las3.las3Utility.modify( st, stLAS );
            break;
        }
      }
    }
  }

  /** Method addLine()
   * <p> This method will add the LAS 3 File Line to the data structures.
   * @param str      = the string to be parsed
   * @param iSection = the section ID to be parsed
   * @param iGroup   = the Type of Data within the section to modify.
   * @param iArray   = Array Number of Data Type
   * @param jSection = the Association ID to be parsed
   * @param jGroup   = the Type of Data within the Association being modified.
   * @param jArray   = Array Number of Association Type
   */

  private void addLine( String str, int iSection, int iGroup, int iArray,
                                    int jSection, int jGroup, int jArray )
  {
    int i,j,k;
    int        ilocal    = 0;
    int        iAction   = _ADD;
    String     sLine[][] = null;
    String     sData[]   = null;
    las3Struct st        = new las3Struct();
    las3Struct st0       = new las3Struct();
    String     sOrderd[] = { "", "", "", "", "", "" };
    int        iNext     = 0;
    String     sOther[]  = null;

    if (str.length() > 0)
    {
      if (stLAS.dVersion == 2.0)
      {
        str = new String( str.replace('\t', ' ') );
      }

      if (las3.las3Utility.isData( iSection, iArray, stLAS ))
        iAction = _MODIFY;

      // Add Association Data to the iSection from the jGroup that is missing.

      st = las3.las3Utility.addAssociation( iSection, iGroup, iArray,
                                           jSection, jGroup, jArray,
                                           stLAS );
      switch (iGroup)
      {
        case las3.las3Constants._PARAMETER:
        case las3.las3Constants._DEFINITION:
          sLine = las3.las3Parse.splitLine( str, stLAS.sDelim );

          if (sLine != null)
          {
            for (i=0; i<sLine.length; i++)
            {
              for (j=0; j<2; j++)
              {
                if ((sLine[i][1].toUpperCase()).equals("MNEM"))
                {
                  if (sLine[i][0].trim().equals("IQKGS"))
                    st.iqstrat = 1;
                  sOrderd[las3.las3Struct._MNEMONIC] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("UNIT"))
                {
                  sOrderd[las3.las3Struct._UNIT] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("VALUE"))
                {
                  sOrderd[las3.las3Struct._VALUE] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("DESCRIPTION"))
                {
                  sOrderd[las3.las3Struct._DESCRIPTION] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("FORMAT"))
                {
                  sOrderd[las3.las3Struct._FORMAT] = new String( sLine[i][0] );
                  if ((iSection == las3.las3Constants._LOG) &&
                      (iGroup   == las3.las3Constants._DEFINITION) &&
                      (stLAS.dVersion == 2.0))
                    sOrderd[las3.las3Struct._FORMAT] = new String( "F" );
                }

                if ((sLine[i][1].toUpperCase()).equals("ASSOCIATION"))
                {
                  sOrderd[las3.las3Struct._ASSOCIATION] = new String( sLine[i][0] );
                }
              }
            }
          }
          break;

        case las3.las3Constants._DATA:
          if (iSection != las3.las3Constants._LOG)
          {
            sData = las3.las3Parse.splitDataLine(str, stLAS.sDelim);
          }
/*
          if (iSection == las3.las3Constants._LOG)
          {
            if (stLAS.sWrap.equals("YES"))
            {
              ilocal = 0;
              iNext  = 0;
              sOther = new String[st.iCurveRows];
              for (i=0; i<sData.length; i++)
              {
                if (iNext < st.iCurveRows)
                {
                  sOther[iNext] = new String( sData[i] );
                  iNext++;
                }
              }

              while (ilocal+sData.length < st.iCurveRows)
              {
                if (!pLAS.Next().equals(util.utilFileIO.EOF))
                {
                  iError = pLAS.getErrorID();
                  sError = pLAS.getError();
                }

                ilocal = ilocal+sData.length;
                if (!las3.las3Parse.isCommentLine( pLAS.getString().trim() ))
                {
                  stList.stItem[iList].iTotal--;
                  sData = las3.las3Parse.splitDataLine(
                            pLAS.getString().trim(), stLAS.sDelim );
                  for (i=0; i<sData.length; i++)
                  {
                    if (iNext < st.iCurveRows)
                    {
                      sOther[iNext] = new String( sData[i] );
                      iNext++;
                    }
                  }
                }
              }

              if (sOther.length > 0)
              {
                sData = new String[sOther.length];
                for (i=0; i<sOther.length; i++)
                {
                  sData[i] = new String( sOther[i] );
                }
              }
            }
          }
*/
          break;
      }

      switch (iGroup)
      {
        case las3.las3Constants._PARAMETER:
          st = las3.las3Utility.addParameters( sOrderd, st );
          break;

        case las3.las3Constants._DEFINITION:
          st = las3.las3Utility.addCurves( sOrderd, st );
          break;

        case las3.las3Constants._DATA:
          st.iColumns = st.iCurveRows;
          if (iSection != las3.las3Constants._LOG)
            addData( sData, st.iCurveRows );
          break;
      }

      switch (iAction)
      {
        case _ADD:
          stLAS = las3.las3Utility.add( st, stLAS );
          break;
        case _MODIFY:
          stLAS = las3.las3Utility.modify( st, stLAS );
          break;
      }
    }
  }

  /** Method addLine0()
   * <p> This method will add the log & core curve modifiers data structure.
   * @param str      = the string to be parsed
   * @param iSection = the section ID to be parsed
   * @param iGroup   = the Type of Data within the section to modify.
   * @param iArray   = Array Number of Data Type
   * @param jSection = the Association ID to be parsed
   * @param jGroup   = the Type of Data within the Association being modified.
   * @param jArray   = Array Number of Association Type
   */

  private void addLine0( String str, int iSection, int iGroup, int iArray,
                                     int jSection, int jGroup, int jArray )
  {
    int i,j;
    int        iAction   = _ADD;
    String     sLine[][] = null;
//    String     sData[]   = null;
    las3Struct st        = new las3Struct();
    String     sOrderd[] = { "", "", "", "", "", "" };

    if (str.length() > 0)
    {
      if (las3.las3Utility.isData( jSection, jArray, stLAS ))
        iAction = _MODIFY;

      // Add Association Data to the iSection from the jGroup that is missing.

      st = las3.las3Utility.addCurve0( jSection, jGroup, jArray, stLAS );

      switch (iGroup)
      {
        case las3.las3Constants._PARAMETER:
        case las3.las3Constants._DEFINITION:
          sLine = las3.las3Parse.splitLine( str, stLAS.sDelim );

          if (sLine != null)
          {
            for (i=0; i<sLine.length; i++)
            {
              for (j=0; j<2; j++)
              {
                if ((sLine[i][1].toUpperCase()).equals("MNEM"))
                {
                  if (sLine[i][0].trim().equals("IQKGS"))
                    st.iqstrat = 1;
                  sOrderd[las3.las3Struct._MNEMONIC] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("UNIT"))
                {
                  sOrderd[las3.las3Struct._UNIT] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("VALUE"))
                {
                  sOrderd[las3.las3Struct._VALUE] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("DESCRIPTION"))
                {
                  sOrderd[las3.las3Struct._DESCRIPTION] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("FORMAT"))
                {
                  sOrderd[las3.las3Struct._FORMAT] = new String( sLine[i][0] );
                }

                if ((sLine[i][1].toUpperCase()).equals("ASSOCIATION"))
                {
                  sOrderd[las3.las3Struct._ASSOCIATION] = new String( sLine[i][0] );
                }
              }
            }
          }
          break;

        case las3.las3Constants._DATA:
          break;
      }

      switch (iGroup)
      {
        case las3.las3Constants._PARAMETER:
        case las3.las3Constants._DEFINITION:
          st = las3.las3Utility.addIQCurves( sOrderd, st );
          break;

        case las3.las3Constants._DATA:
          break;
      }

      switch (iAction)
      {
        case _ADD:
          stLAS = las3.las3Utility.add( st, stLAS );
          break;
        case _MODIFY:
          stLAS = las3.las3Utility.modify( st, stLAS );
          break;
      }
    }
  }

  /** Method addData()
   * <p> This method will add the data to the temporary data string
   * @param str   = Array of Column Data
   * @param iCols = total columns
   */

  private void addData( String str[], int iCols )
  {
    if ((iList != -1) && (str != null))
    {
      if (stList.stItem[iList] != null)
      {
        if (stList.stItem[iList].iCount == 0)
        {
          stList.stItem[iList].iRows    = stList.stItem[iList].iTotal;
          stList.stItem[iList].iColumns = iCols;
           stList.stItem[iList].sData   =
               new String[stList.stItem[iList].iTotal][iCols];

           for (int i=0; i<stList.stItem[iList].iTotal; i++)
           {
             for (int j=0; j<stList.stItem[iList].iColumns; j++)
             {
               stList.stItem[iList].sData[i][j] = new String("");
             }
           }
        }

        if (stList.stItem[iList].iCount < stList.stItem[iList].iTotal)
        {
          for (int k=0; k<str.length; k++)
          {
            stList.stItem[iList].sData[stList.stItem[iList].iCount][k] =
                new String( str[k] );
          }
          stList.stItem[iList].iCount++;
        }
      }
    }
  }
}

/*
 *  @version 1.1 02/09/2010
 *  @author  John Victorine
 */
