/*
 * @las3Read.java Version 1.1 02/09/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.io;

import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import cmn.cmnStruct;                      // Global Common Class Data Structure
import iqstrat.iqstratStruct;              // Global IQSTRAT Data Structure

import las3.las3ListStruct;                // LAS Version 3 File Data List Structure
import las3.las3Struct;                    // LAS Version 3 File Data Structure
import las.lasFileDataStruct;              // LAS File Data Structure
import las3.io.las3ReadFile;               // Open, Read & Parse LAS Files
import las3.gui.las3CurvesFrame;           // Allow User to Change Selected Curves

import iqstrat.iqstratHeadersStruct;       // Well Headers Information Structure
import iqstrat.iqstratControlStruct;       // Control Data Structure
import iqstrat.iqstratRemarkListStruct;    // Remarks Data List Structure
import iqstrat.iqstratShaleListStruct;     // Shale Data List Structure

import brine.brineListStruct;              // Brine Data List Structure

import rock.rockDataListStruct;            // Rock Data List Structure
import rock.rockImagesListStruct;          // Rock Images Data List Structure

import horizon.strat.stratListStruct;      // Startigraphic Data List Structure
import horizon.seq.seqListStruct;          // Sequence Stratigraphy
import horizon.regions.regionsListStruct;  // Regions Data List Structure
import horizon.bio.bioStratListStruct;     // Bio-Startigraphic Data List Structure
import horizon.env.envListStruct;          // Depositonal Environment

import dst.dstListStruct;                  // DST Data List Structure

import pfeffer.pfefferDataListStruct;      // Pfeffer Data List Structure

import util.utilWaitAnimation;             // Utility Wait Animation Class
import util.SwingWorker;                   // Utility Swing Worker Class

/** Class las3Read
 *  <p> This Class is designed to open a LAS File either Version 2 or 3 and
 *      parse the file into Profile Data Structures
 *
 *  @version 1.1 02/09/2010
 *  @author  John R. Victorine
 */

public class las3Read implements Observer
{
  // Input Variables

  private Observable    pNotifier = null; // Top Observables Object
  private cmnStruct     stCMN     = null; // Global Common Data Structure
  private iqstratStruct stStruct  = null; // Global IQSTRAT Data Structure

  // Global Variables

  public static final int _LOAD_DATA = 0;
  public static final int _LOAD_LAS3 = 1;

  private int iLoad = _LOAD_DATA;

  private String sReadMethod  = "";  // File Read Method
  private String sDirectory   = "";  // Directory Path to the LAS File
  private String sFilename    = "";  // File name of LAS File

  // Required LAS 3.0 Data Sections default is all data.

  private int     iLAS[]      = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  private boolean bLog[]      = {true,true,true,true,true,
                                 true,true,true,true,true,
                                 true,true,true,true,true,
                                 true,true,true,true,true,
                                 true};

  private Observable      notifier = null; // Pfeffer Observables Object
  private las3ReadFile    p3       = null; // Read LAS File
  private las3CurvesFrame pCurves  = null; // Choose Log Curves

  // -- Well/Outcrop Data

  private iqstratHeadersStruct    stHeader    = null; // Headers Data Structure
  private iqstratControlStruct    stControl   = null; // Control Data Structure
  private las3ListStruct          stLAS3Data  = null; // LAS 3 Data List Struct
  private lasFileDataStruct    stLASFileData  = null; // LAS File Data Struct

  private stratListStruct         stTopsData  = null; // Tops Data List Struct
  private seqListStruct           stSeq       = null; // Sequence Strat List
  private regionsListStruct       stPerfData  = null; // Perf Data List Struct
  private bioStratListStruct      stBio       = null; // Bio Stratigraphy List Struct
  private envListStruct           stEnv       = null; // Environment List Structure

  private dstListStruct           stDST       = null; // DST Data List Structure

  private pfefferDataListStruct   stPfeffer   = null; // Pfeffer Data List Struct

  private brineListStruct         stBrine     = null; // Brine Data List Structure

  private rockDataListStruct      stRockData  = null; // Rock Data List Struct
  private rockImagesListStruct    stImages    = null; // Rock Images Data List Struct
  private iqstratRemarkListStruct stRemarks   = null; // Remarks Data List Struct

  private iqstratShaleListStruct  stShale    = null;  // Shale Data List Struct

  private double                  dStart      = 0.0;  // Starting LAS File Depth
  private double                  dEnd        = 0.0;  // Ending LAS File Depth

  /** Construct las3Read()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct  = Global IQSTRAT Data Structure
   *  @param bLog      = Indicator to identify which Data Sections to
   *                     make available ( required data sections )
   */

  public las3Read( Observable pNotifier, iqstratStruct stStruct )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;
      this.stCMN     = stStruct.stCMN;

      if (stStruct.bLAS != null)
        this.bLog    = stStruct.bLAS;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** Construct las3Read()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct  = Global IQSTRAT Data Structure
   *  @param iLoad     = User defined Load or not Load Profile data
   */

  public las3Read( Observable pNotifier, iqstratStruct stStruct, int iLoad )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;
      this.stCMN     = stStruct.stCMN;
      this.iLoad     = iLoad;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** Construct las3Read()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct  = Global IQSTRAT Data Structure
   *  @param iLoad     = User defined Load or not Load Profile data
   *  @param bLog      = Indicator to identify which Data Sections to
   *                     make available ( required data sections )
   */

  public las3Read( Observable pNotifier, iqstratStruct stStruct, int iLoad, boolean bLog[] )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;
      this.stCMN     = stStruct.stCMN;
      this.iLoad     = iLoad;
      this.bLog      = bLog;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    // Create an Observable

    notifier = new las3ReadObservable();

    // Add this object to the notifier as observer

    notifier.addObserver(this);
  }

  /* ===================================================================== *
   * ------------------------- DELETE METHODS ---------------------------- *
   * ===================================================================== */

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    // Input Variables

    pNotifier = null;  // Top Observables Object
    stCMN     = null;  // GLobal Common Data Structure
    stStruct  = null;  // Global IQSTRAT Data Structure

    // Global Variables

    notifier = null;   // Pfeffer Observables Object

    sReadMethod  = null;  // File Read Method
    sDirectory   = null;  // Directory Path to the LAS File
    sFilename    = null;  // File name of LAS File

    // Delete Data

    clearData();

//    stLAS = null;

    // Delete LAS 3 Read File

    if (p3 != null)
      p3.delete();
    p3 = null;

    // Delete Selected Log Curves Frame

    if (pCurves != null)
      pCurves.close();
    pCurves = null;
  }

  /** Method clearData()
   * <p> This method will set all the data structures to null to force
   *     java to free the allocated memory
   */

  public void clearData()
  {
    dStart = 0.0; // Starting LAS File Depth
    dEnd   = 0.0; // Ending LAS File Depth

    // Remove the LAS 3 File Data Structure

    if (stLAS3Data != null)
      stLAS3Data.delete();
    stLAS3Data = null;

    // Remove the Well Header Information Data Structure

    if (stHeader != null)
      stHeader.delete();
    stHeader = null;

    // Remove the LAS File Data Structure

    if (stLASFileData != null)
    {
      stLASFileData.delete();
      stLASFileData = null;
    }

    // Remove the Rock Data List Structure

    if (stRockData != null)
      stRockData.delete();
    stRockData = null;

    if (stBrine != null)
      stBrine.delete();
    stBrine = null;

    if (stImages != null)
      stImages.delete();
    stImages = null;

    if (stTopsData != null)
      stTopsData.delete();
    stTopsData = null;

    if (stPerfData != null)
      stPerfData.delete();
    stPerfData = null;

    if (stDST != null)
      stDST.delete();
    stDST = null;

    if (stControl != null)
      stControl.delete();
    stControl = null;

    if (stSeq != null)
      stSeq.delete();
    stSeq = null;

    if (stEnv != null)
      stEnv.delete();
    stEnv = null;

    if (stBio != null)
      stBio.delete();
    stBio = null;

    if (stPfeffer != null)
      stPfeffer.delete();
    stPfeffer = null;

    if (stRemarks != null)
      stRemarks.delete();
    stRemarks = null;

    if (stShale != null)
      stShale.delete();
    stShale = null;
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLAS()
   * <p> This method will retrieve the LAS 3 File Available Sections Count
   * @return iLAS = the LAS 3 File Available Sections Count
   */

  public int[] getLAS() { return (iLAS); }

  /** Method getPath()
   * <p> This method will search for a LAS File that is on the user's PC and
   *     load the LAS File into the data structures.
   */

  public void getPath()
  {
    FileDialog dialog    = null;
    Frame      frameTemp = new Frame();

    dialog = new FileDialog( frameTemp, "Select LAS File from your PC." );
    dialog.setVisible( true );
    frameTemp.setVisible( true );

    // Retrieve LAS File Location

    if ((dialog.getDirectory() != null) && (dialog.getFile() != null))
    {
      if (pNotifier != null)
        pNotifier.notifyObservers(new String("LAS 3 - Loading Data"));

      setPath( "FILE", dialog.getDirectory(), dialog.getFile() );
    }

    // Dispose of Temporary Dialog & Frame

    frameTemp.dispose();
    frameTemp = null;

    dialog.dispose();
    dialog = null;
  }

  /** Method getData()
   *  <p> This method will load the LAS 3.0 Data into the Data Structures
   */

  public void getData()
  {
    stLASFileData = las.lasFileDataUtility.transfer( pCurves.getLAS() );
    stLASFileData = las3.las3LoadLogData.modifyLogData( stLASFileData );
    stLASFileData = las.lasFileDataUtility.computeMath( stLASFileData );

    stControl  = iqstrat.iqstratControlUtility.transfer( pCurves.getControl() );

    stTopsData = horizon.strat.stratUtility.transfer( pCurves.getTops() );

    stRemarks  = iqstrat.iqstratRemarkUtility.transfer( pCurves.getRemarks() );

    if (pCurves != null)
      pCurves.close();
    pCurves = null;
  }

  /** Method getStartingDepth()
   * <p> This method will retrieve the starting depth of the LAS File.
   * @return dStart = the starting depth of the LAS File.
   */

  public double getStartingDepth() { return (dStart); }

  /** Method getEndingDepth()
   * <p> This method will retrieve the ending depth of the LAS File.
   * @return dStart = the ending depth of the LAS File.
   */

  public double getEndingDepth() { return (dEnd); }

  /** Method getLAS3Data()
   * <p> This method will retrieve the LAS 3 Raw Data List Structure
   *     This data structure contains all the data sections that are present
   *     in the LAS 3 File or just the Log Data Section for the LAS 2 File.
   * @return stLAS3Data = the LAS 3 Raw Data List Structure
   */

  public las3ListStruct getLAS3Data()
  {
    return (las3.las3Utility.remove( stLAS3Data, bLog ));
//    return ( stLAS3Data );
//    return (las3.las3Utility.removeIQSTRAT( stLAS3Data ));
  }

  /** Method getWellData()
   * <p> This method will retrieve the Well Information Data Structure
   * @return stHeader = the Well Information Data Structure
   */

  public iqstratHeadersStruct getWellData() { return (stHeader); }

  /** Method getControlData()
   * <p> This method will retrieve the Plot Control Data Structure
   * @return stControl = the Plot Control Data Structure
   */

  public iqstratControlStruct getControlData() { return (stControl); }

  /** Method getLASData()
   * <p> This method will retrieve the LAS File Data Structure
   * @return stLASFileData = the LAS File Data Structure
   */

  public lasFileDataStruct getLASData() { return (stLASFileData); }

  /** Method getRockData()
   * <p> This method will retrieve the Rock Data List Structure
   * @return stRockData = the Rock Data List Structure
   */

  public rockDataListStruct getRockData() { return (stRockData); }

  /** Method getBrine()
   * <p> This method will retrieve the Brine List Structure
   * @return stBrine = the Brine List Structure
   */

  public brineListStruct getBrine() { return (stBrine); }

  /** Method getImages()
   * <p> This method will retrieve the Rock Image Data List Structure
   * @return stImages = the Rock Image Data List Structure
   */

  public rockImagesListStruct getImages() { return (stImages); }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Unit Data List Structure
   * @return stTopsData = the Stratigraphic Unit Data List Structure
   */

  public stratListStruct getTops() { return (stTopsData); }

  /** Method getSeq()
   * <p> This method will retrieve the Sequence Stratigraphy List Structure
   * @return stSeq = the Sequence Stratigraphy Data List Structure
   */

  public seqListStruct getSeq() { return (stSeq); }

  /** Method getPerf()
   * <p> This method will retrieve the Perforations Data List Structure
   * @return stPerfData = the Perforations Data List Structure
   */

  public regionsListStruct getPerf() { return (stPerfData); }

  /** Method getDST()
   * <p> This method will retrieve the DST Data List Structure
   * @return stDST = the DST Data List Structure
   */

  public dstListStruct getDST() { return (stDST); }

  /** Method getBio()
   * <p> This method will retrieve the Bio-Stratigraphic Unit Data List Structure
   * @return stBio = the Bio-Stratigraphic Unit Data List Structure
   */

  public bioStratListStruct getBio() { return (stBio); }

  /** Method getEnv()
   * <p> This method will retrieve the Depositional Environment Data List Structure
   * @return stEnv = the Depositional Environment Data List Structure
   */

  public envListStruct getEnv() { return (stEnv); }

  /** Method getPfeffer()
   * <p> This method will retrieve the PfEFFER Data List Structure
   * @return stPfeffer = the PfEFFER Data List Structure
   */

  public pfefferDataListStruct getPfeffer() { return (stPfeffer); }

  /** Method getRemarks()
   * <p> This method will retrieve the Remarks Data List Structure
   * @return stRemarks = the Remarks Data List Structure
   */

  public iqstratRemarkListStruct getRemarks() { return (stRemarks); }

  /** Method getShale()
   * <p> This method will retrieve the Shale Data List Structure
   * @return stShale = the Shale Data List Structure
   */

  public iqstratShaleListStruct getShale() { return (stShale); }

  /* =========================== ========================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setPath()
   * <p> This method will search for a LAS File that is on the user's PC and
   *     load the LAS File into the data structures.
   * @param sMethod    = the type of file read method, i.e. "FILE" or "URL"
   * @param sDirectory = the Directory or URL Path to LAS File
   * @param sFilename  = the Filename of the LAS File
   */

  public void setPath( String sMethod, String sDirectory, String sFilename )
  {
    // Retrieve LAS File Location

    if ((sMethod != null) && (sDirectory != null) && (sFilename != null))
    {
      this.sReadMethod = new String( sMethod );
      this.sDirectory  = new String( sDirectory );
      this.sFilename   = new String( sFilename );

//System.out.println("["+sDirectory+"] ["+sFilename+"]");

      ReadLAS3(); //WaitReadLASFile();
    }
  }

  /* ===================================================================== *
   * -------------------------- READ METHODS ----------------------------- *
   * ===================================================================== */

  /** Method WaitReadLASFile()
   * <p> This method will display the Wait Animation for opening and parsing a
   *     digtial LAS File.
   */

  private void WaitReadLASFile()
  {
    java.net.URL url = null;

    // Get the URL Path to display the Wait Animation

    try
    {
      url = new java.net.URL(cmn.cmnStruct.IQSTRAT_BASE + "/");
    }
    catch(Exception ex)
    {
      System.out.println("Wrong Image Directory" + ex);
    }

    // Start the Wait Animation Dialog

    final utilWaitAnimation pDialog = new utilWaitAnimation(url);
    pDialog.startAnimation();

    final SwingWorker worker = new SwingWorker()
    {
      public Object construct()
      {
        ReadLAS3();

        return "OK";
      }

      //Runs on the event-dispatching thread.

      public void finished() { pDialog.stopAnimation(); }
    };

    // Start the Wait Animation Thread

    worker.start();  //required for SwingWorker
  }

  /** Method ReadLAS3()
   * <p> This method will load and print the LAS 3 File Data
   */

  private void ReadLAS3()
  {
    int iType = util.utilFileIO.DIRECTORY;

    if (this.sReadMethod.equals("URL"))
      iType = util.utilFileIO.URL_FILE;

    // Clear all Data Structures - for new LAS file Read

    clearData();

    // Open LAS 3 File and Read Data

    p3 = new las3ReadFile();

    p3.Read( iType, this.sDirectory, this.sFilename );
    stLAS3Data = las3.las3Utility.copyList( p3.getData() );

    iLAS = p3.getLAS();

    if (iLoad == _LOAD_DATA)
      loadLAS3( stLAS3Data );

    // Deallocate Java Memory

    if (p3 != null) // Close & clear memory for the LAS 3 File Read Class
      p3.delete();
    p3 = null;
  }

  /* ===================================================================== *
   * -------------------------- LOAD METHODS ----------------------------- *
   * ===================================================================== */

  /** Method loadLAS3()
   * <p> This method will load the LAS 3 Data List Structure into the Profile
   *     Data List Structures
   * @param st = the LAS 3 File Data List Structure
   */

  private void loadLAS3( las3ListStruct st )
  {
    double dTemp = 0.0;

    // Load Well Headers Data Structure

    stHeader = las3.las3LoadWellHeader.getData( st );

    // Load Required LAS File Data Structure Fields

    stLASFileData = las3.las3LoadLogData.getData( st );

    // Add Version Number, set default plot depth range

    if (stLASFileData != null)
    {
      stLASFileData.sReadMethod = new String( sReadMethod ); // Method
      stLASFileData.sDirectory  = new String( sDirectory );  // Directory
      stLASFileData.sFilename   = new String( sFilename );   // Filename
      stLASFileData.sVersion    = new String( "" + st.dVersion );  // Version No
      dStart                    = stLASFileData.depthStart; // Starting Depth
      dEnd                      = stLASFileData.depthEnd;   // Ending Depth

      if (dEnd < dStart)
      {
        dTemp  = dEnd;
        dEnd   = dStart;
        dStart = dTemp;
      }
    }

    // Set the missing depth variables depending on the source of data

    stHeader = las3.las3LoadWellHeader.normalizeDepth( stHeader, stLASFileData );
    stLASFileData = las3.las3LoadLogData.normalizeDepth( stHeader, stLASFileData );

    // Open Curve Selection Dialog to give the user a chance to change the
    // default curve selections.  This Dialog displays all PROFILE Web App
    // data curves that are read from a LAS File Version 2.0 or 3.0.

    pCurves = new las3CurvesFrame( notifier, stStruct, stLASFileData, st, bLog );
  }

  /* ===================================================================== *
   * ------------------------ OBSERVER ACTIONS --------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("LAS 3 - Curves Selected"))
    {
      stLASFileData = las.lasFileDataUtility.transfer( pCurves.getLAS() );
      stLASFileData = las3.las3LoadLogData.modifyLogData( stLASFileData );
      stLASFileData = las.lasFileDataUtility.computeMath( stLASFileData );

      stPfeffer     = pfeffer.pfefferDataUtility.transfer( pCurves.getPfeffer() );

      stTopsData    = horizon.strat.stratUtility.transfer( pCurves.getTops() );
      stPerfData    = horizon.regions.regionsUtility.transfer( pCurves.getPerf() );
      stSeq         = horizon.seq.seqUtility.transfer( pCurves.getSeq() );
      stBio         = horizon.bio.bioStratUtility.transfer( pCurves.getBio() );
      stEnv         = horizon.env.envUtility.transfer( pCurves.getEnv() );

      stDST         = dst.dstUtility.transfer( pCurves.getDST() );

      stControl     = iqstrat.iqstratControlUtility.transfer( pCurves.getControl() );

      stBrine       = brine.brineUtility.transfer( pCurves.getBrine() );

      stRockData    = rock.rockDataUtility.transfer( pCurves.getRockData() );
      stImages      = rock.rockImagesUtility.transfer( pCurves.getImages() );
      stRemarks     = iqstrat.iqstratRemarkUtility.transfer( pCurves.getRemarks() );
      stShale       = iqstrat.iqstratShaleUtility.transfer( pCurves.getShale() );

      if (pCurves != null)
        pCurves.close();
      pCurves = null;

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("LAS 3 - Curves Selected"));
    }
  }
}

/** CLASS las3ReadObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class las3ReadObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 02/09/2010
 *  @author  John Victorine
 */
