/*
 * @las3FileWellPanel.java Version 1.1 01/26/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import iqstrat.iqstratHeadersStruct;
import las.lasFileDataStruct;
import las3.las3Struct;

/** Class las3FileWellPanel
 *  <p> This Class will display the well parameter section of the LAS File
 *
 *  @version 1.1 01/26/2010
 *  @author  John R. Victorine
 */

public class las3FileWellPanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable notifier = null; // Observables Object

  // Global Variables

  private iqstratHeadersStruct stHeader = null; // Well Header Information
  private lasFileDataStruct    stLAS    = null; // LAS File Information
  private las3Struct           stLAS3   = null; // LAS 3 Well Parameters

  // Columns

  public static final int _MNEMONIC    = las3.las3Struct._MNEMONIC;
  public static final int _UNIT        = las3.las3Struct._UNIT;
  public static final int _VALUE       = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT      = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION = las3.las3Struct._ASSOCIATION;

  // Panel Constants

  public static final int _START_DEPTH = las3.las3Parameters._START_DEPTH;
  public static final int _END_DEPTH   = las3.las3Parameters._END_DEPTH;
  public static final int _STEP_BY     = las3.las3Parameters._STEP_BY;
  public static final int _NULL_VALUE  = las3.las3Parameters._NULL_VALUE;

  public static final int _COMPANY     = las3.las3Parameters._COMPANY;
  public static final int _WELL        = las3.las3Parameters._WELL;
  public static final int _FIELD       = las3.las3Parameters._FIELD;
  public static final int _SECTION     = las3.las3Parameters._SECTION;
  public static final int _TOWNSHIP    = las3.las3Parameters._TOWNSHIP;
  public static final int _RANGE       = las3.las3Parameters._RANGE;
  public static final int _LOCATION    = las3.las3Parameters._LOCATION;
  public static final int _LOCATION_1  = las3.las3Parameters._LOCATION_1;
  public static final int _LOCATION_2  = las3.las3Parameters._LOCATION_2;
  public static final int _COUNTY      = las3.las3Parameters._COUNTY;
  public static final int _STATE       = las3.las3Parameters._STATE;
  public static final int _COUNTRY     = las3.las3Parameters._COUNTRY;
  public static final int _PROVINCE    = las3.las3Parameters._PROVINCE;
  public static final int _SERVICE     = las3.las3Parameters._SERVICE;
  public static final int _LICENSE     = las3.las3Parameters._LICENSE;
  public static final int _DATE        = las3.las3Parameters._DATE;
  public static final int _API         = las3.las3Parameters._API;
  public static final int _UWI         = las3.las3Parameters._UWI;
  public static final int _LATI        = las3.las3Parameters._LATI;
  public static final int _LONG        = las3.las3Parameters._LONG;
  public static final int _GDAT        = las3.las3Parameters._GDAT;
  public static final int _X           = las3.las3Parameters._X;
  public static final int _Y           = las3.las3Parameters._Y;
  public static final int _HZCS        = las3.las3Parameters._HZCS;
  public static final int _UTM         = las3.las3Parameters._UTM;
  public static final int TOTAL        = las3.las3Parameters._WELL_TOTAL;

  // Well Information Textfields

  private String PARAM[][] = las3.las3Parameters.WELL_PARAMETERS;

  public static final int _TOWNSHIP_DIR = 0;
  public static final int _RANGE_DIR    = 1;
  public static final int TOTAL_DIR     = 2;

  public static final String DIR[][] = {
      { "S", "N"},
      { "E", "W"}
  };

  // Global Panel Widgets

  private JLabel     lbl_Mnem[]  = null;
  private JTextField txtUnit[]   = null;
  private JTextField txtValue[]  = null;
  private JTextField txtDesc[]   = null;
  private JTextField txtFormat[] = null;

  private JRadioButton rbLeft[]  = null;
  private JRadioButton rbRight[] = null;

  /** Constructor las3FileWellPanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public las3FileWellPanel( Observable notifier )
  {
    try
    {
      this.notifier = notifier;
      jbInit();
    }
    catch (Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop           = new JPanel();
    JPanel pnlCenter        = new JPanel();
    JPanel pnlWell          = new JPanel();

    JScrollPane scroll      = new JScrollPane();

    JPanel pnlA             = new JPanel();
    JLabel lblA_Mnem        = new JLabel();
    JPanel pnlA_a           = new JPanel();
    JPanel pnlA_al          = new JPanel();
    JPanel pnlA_ar          = new JPanel();

    JLabel lblA_Unit        = new JLabel();
    JLabel lblA_Value       = new JLabel();
    JLabel lblA_Colon       = new JLabel();
    JLabel lblA_Description = new JLabel();
    JLabel lblA_Format      = new JLabel();

    JPanel pnl[]            = null;
    JPanel pnl_a[]          = null;
    JPanel pnl_al[]         = null;
    JPanel pnl_ar[]         = null;
    JPanel pnl_Format[]     = null;

    JLabel lbl_Colon[]      = null;
    JLabel lbl_BraceL[]     = null;
    JLabel lbl_BraceR[]     = null;

    JPanel pnl_Value[]      = null;
    JPanel pnl_ValueDir[]   = null;
    ButtonGroup groupDir[]  = null;

    this.setLayout(new BorderLayout());

    pnlCenter.setLayout(new BorderLayout());
    pnlWell.setLayout(new GridLayout(TOTAL+1,1));
    pnlWell.setPreferredSize(new Dimension(620, (TOTAL+1)*20));
//    pnlWell.setLayout(new GridLayout(30, 1));
//    pnlWell.setPreferredSize(new Dimension(640, 900));

    // Well Information List Title Panel

    pnlA.setLayout(new BorderLayout());

    // -- Mnemonic Column Title Label

    lblA_Mnem.setFont(new java.awt.Font("Monospaced", 1, 11));
    lblA_Mnem.setText("MNEM ");

    // -- A - a Panel

    pnlA_a.setLayout(new GridLayout());

    // .. A - a Left Panel

    pnlA_al.setLayout(new BorderLayout());

    // .... Unit Column Title Label

    lblA_Unit.setFont(new java.awt.Font("Monospaced", 1, 11));
    lblA_Unit.setText(".UNIT   ");

    // .... Value Column Title Label

    lblA_Value.setFont(new java.awt.Font("Monospaced", 1, 11));
    lblA_Value.setHorizontalAlignment(SwingConstants.CENTER);
    lblA_Value.setText("VALUE");

    // .. A - a Right Panel

    pnlA_ar.setLayout(new BorderLayout());

    // .... Colon Separator

    lblA_Colon.setFont(new java.awt.Font("Dialog", 1, 11));
    lblA_Colon.setText(" : ");

    // .... Description Column Title

    lblA_Description.setFont(new java.awt.Font("Monospaced", 1, 11));
    lblA_Description.setHorizontalAlignment(SwingConstants.CENTER);
    lblA_Description.setText("DESCRIPTION");

    // .... Format Column Title

    lblA_Format.setFont(new java.awt.Font("Monospaced", 1, 11));
    lblA_Format.setText("{ Format }");

    // Attach Well Information Column Header

    pnlWell.add(pnlA,             null);
    pnlA.add(lblA_Mnem,           BorderLayout.WEST);
    pnlA.add(pnlA_a,              BorderLayout.CENTER);
    pnlA_a.add(pnlA_al,           null);
    pnlA_al.add(lblA_Unit,        BorderLayout.WEST);
    pnlA_al.add(lblA_Value,       BorderLayout.CENTER);
    pnlA_a.add(pnlA_ar,           null);
    pnlA_ar.add(lblA_Colon,       BorderLayout.WEST);
    pnlA_ar.add(lblA_Description, BorderLayout.CENTER);
    pnlA_ar.add(lblA_Format,      BorderLayout.EAST);

    /* ================================================================== */

    pnl        = new JPanel[TOTAL];
    pnl_a      = new JPanel[TOTAL];
    pnl_al     = new JPanel[TOTAL];
    pnl_ar     = new JPanel[TOTAL];
    pnl_Format = new JPanel[TOTAL];

    lbl_Colon  = new JLabel[TOTAL];
    lbl_BraceL = new JLabel[TOTAL];
    lbl_BraceR = new JLabel[TOTAL];

    lbl_Mnem   = new JLabel[TOTAL];
    txtUnit    = new JTextField[TOTAL];
    txtValue   = new JTextField[TOTAL];
    txtDesc    = new JTextField[TOTAL];
    txtFormat  = new JTextField[TOTAL];

    pnl_Value     = new JPanel[TOTAL_DIR];
    pnl_ValueDir  = new JPanel[TOTAL_DIR];
    rbLeft        = new JRadioButton[TOTAL_DIR];
    rbRight       = new JRadioButton[TOTAL_DIR];
    groupDir      = new ButtonGroup[TOTAL_DIR];

    for (int i=0; i<TOTAL; i++)
    {
      // Array of Well Information Central Row Panels

      pnl[i] = new JPanel();
      pnl[i].setLayout(new BorderLayout());

      // -- Array or Mnemonic Columns

      lbl_Mnem[i] = new JLabel();
      lbl_Mnem[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      lbl_Mnem[i].setText(PARAM[i][_MNEMONIC] + " .");

      // -- Array of a Panels

      pnl_a[i] = new JPanel();
      pnl_a[i].setLayout(new GridLayout());

      // .. Array of a Left Panels

      pnl_al[i] = new JPanel();
      pnl_al[i].setLayout(new BorderLayout());

      // .... Array of Unit Textfields

      txtUnit[i] = new JTextField();
      txtUnit[i].setEditable(false);
      txtUnit[i].setText(PARAM[i][_UNIT]);
      txtUnit[i].setColumns(3);
      txtUnit[i].addFocusListener(new las3FileWellPanelFocusAdapter(this));

      // .... Array of Value Textfields

      switch (i)
      {
        case _TOWNSHIP:
          pnl_Value[_TOWNSHIP_DIR] = new JPanel();
          pnl_Value[_TOWNSHIP_DIR].setLayout(new BorderLayout());

          pnl_ValueDir[_TOWNSHIP_DIR] = new JPanel();
          pnl_ValueDir[_TOWNSHIP_DIR].setLayout(new GridLayout());
          pnl_ValueDir[_TOWNSHIP_DIR].setPreferredSize(new Dimension(80, 20));

          groupDir[_TOWNSHIP_DIR] = new ButtonGroup();

          rbLeft[_TOWNSHIP_DIR] = new JRadioButton();
          rbLeft[_TOWNSHIP_DIR].setSelected(true);
          rbLeft[_TOWNSHIP_DIR].setText(DIR[_TOWNSHIP_DIR][0]);
          rbLeft[_TOWNSHIP_DIR].setFont(new java.awt.Font("Monospaced", 1, 11));
          rbLeft[_TOWNSHIP_DIR].addActionListener(this);

          rbRight[_TOWNSHIP_DIR] = new JRadioButton();
          rbRight[_TOWNSHIP_DIR].setText(DIR[_TOWNSHIP_DIR][1]);
          rbRight[_TOWNSHIP_DIR].setFont(new java.awt.Font("Monospaced", 1, 11));
          rbRight[_TOWNSHIP_DIR].addActionListener(this);
          break;
        case _RANGE:
          pnl_Value[_RANGE_DIR] = new JPanel();
          pnl_Value[_RANGE_DIR].setLayout(new BorderLayout());

          pnl_ValueDir[_RANGE_DIR] = new JPanel();
          pnl_ValueDir[_RANGE_DIR].setLayout(new GridLayout());
          pnl_ValueDir[_RANGE_DIR].setPreferredSize(new Dimension(80, 20));

          groupDir[_RANGE_DIR] = new ButtonGroup();

          rbLeft[_RANGE_DIR] = new JRadioButton();
          rbLeft[_RANGE_DIR].setSelected(true);
          rbLeft[_RANGE_DIR].setText(DIR[_RANGE_DIR][0]);
          rbLeft[_RANGE_DIR].setFont(new java.awt.Font("Monospaced", 1, 11));
          rbLeft[_RANGE_DIR].addActionListener(this);

          rbRight[_RANGE_DIR] = new JRadioButton();
          rbRight[_RANGE_DIR].setText(DIR[_RANGE_DIR][1]);
          rbRight[_RANGE_DIR].setFont(new java.awt.Font("Monospaced", 1, 11));
          rbRight[_RANGE_DIR].addActionListener(this);
          break;
      }

      txtValue[i] = new JTextField();
      if (i==_NULL_VALUE)
        txtValue[i].setEditable(false);
      txtValue[i].setText(PARAM[i][_VALUE]);
      txtValue[i].setHorizontalAlignment(SwingConstants.TRAILING);
      txtValue[i].addFocusListener(new las3FileWellPanelFocusAdapter(this));

      // .. Array of a Left Panels

      pnl_ar[i] = new JPanel();
      pnl_ar[i].setLayout(new BorderLayout());

      // .... Array of Colon Separators

      lbl_Colon[i] = new JLabel();
      lbl_Colon[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      lbl_Colon[i].setText(" : ");

      // .... Array of Descriptions Textfields

      txtDesc[i] = new JTextField();
      txtDesc[i].setText(PARAM[i][_DESCRIPTION]);
      txtDesc[i].setHorizontalAlignment(SwingConstants.LEADING);
      txtDesc[i].addFocusListener(new las3FileWellPanelFocusAdapter(this));

      // .... Array of Formats Panels

      pnl_Format[i] = new JPanel();
      pnl_Format[i].setLayout(new BorderLayout());

      lbl_BraceL[i] = new JLabel();
      lbl_BraceL[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      lbl_BraceL[i].setText("{");

      txtFormat[i] = new JTextField();
      txtFormat[i].setEditable(false);
      txtFormat[i].setText(PARAM[i][_FORMAT]);
      txtFormat[i].setColumns(2);
      txtFormat[i].setHorizontalAlignment(SwingConstants.CENTER);
      txtFormat[i].addFocusListener(new las3FileWellPanelFocusAdapter(this));

      lbl_BraceR[i] = new JLabel();
      lbl_BraceR[i].setFont(new java.awt.Font("Monospaced", 1, 11));
      lbl_BraceR[i].setText("}");

      // Add Array Widgets to Panel

      pnlWell.add(pnl[i],       null);
      pnl[i].add(lbl_Mnem[i],   BorderLayout.WEST);
      pnl[i].add(pnl_a[i],      BorderLayout.CENTER);
      pnl_a[i].add(pnl_al[i],   null);
      pnl_al[i].add(txtUnit[i], BorderLayout.WEST);

      switch (i)
      {
        case _TOWNSHIP:
          pnl_al[i].add(pnl_Value[_TOWNSHIP_DIR],   BorderLayout.CENTER);
          pnl_Value[_TOWNSHIP_DIR].add(txtValue[i], BorderLayout.CENTER);
          pnl_Value[_TOWNSHIP_DIR].add(pnl_ValueDir[_TOWNSHIP_DIR],
                                       BorderLayout.EAST);
          pnl_ValueDir[_TOWNSHIP_DIR].add(rbLeft[_TOWNSHIP_DIR],  null);
          pnl_ValueDir[_TOWNSHIP_DIR].add(rbRight[_TOWNSHIP_DIR], null);

          groupDir[_TOWNSHIP_DIR].add(rbLeft[_TOWNSHIP_DIR]);
          groupDir[_TOWNSHIP_DIR].add(rbRight[_TOWNSHIP_DIR]);
          break;
        case _RANGE:
          pnl_al[i].add(pnl_Value[_RANGE_DIR],   BorderLayout.CENTER);
          pnl_Value[_RANGE_DIR].add(txtValue[i], BorderLayout.CENTER);
          pnl_Value[_RANGE_DIR].add(pnl_ValueDir[_RANGE_DIR], BorderLayout.EAST);
          pnl_ValueDir[_RANGE_DIR].add(rbLeft[_RANGE_DIR],  null);
          pnl_ValueDir[_RANGE_DIR].add(rbRight[_RANGE_DIR], null);

          groupDir[_RANGE_DIR].add(rbLeft[_RANGE_DIR]);
          groupDir[_RANGE_DIR].add(rbRight[_RANGE_DIR]);
          break;
        default:
          pnl_al[i].add(txtValue[i],   BorderLayout.CENTER);
          break;
      }

      pnl_a[i].add(pnl_ar[i],          null);
      pnl_ar[i].add(lbl_Colon[i],      BorderLayout.WEST);
      pnl_ar[i].add(txtDesc[i],        BorderLayout.CENTER);
      pnl_ar[i].add(pnl_Format[i],     BorderLayout.EAST);
      pnl_Format[i].add(lbl_BraceL[i], BorderLayout.WEST);
      pnl_Format[i].add(txtFormat[i],  BorderLayout.CENTER);
      pnl_Format[i].add(lbl_BraceR[i], BorderLayout.EAST);
    }

    /* ================================================================== */

    // Add Widgets to Panel

    this.add(pnlTop,              BorderLayout.NORTH);

    this.add(pnlCenter,           BorderLayout.CENTER);
    pnlCenter.add(scroll,         BorderLayout.CENTER);
    scroll.getViewport().add(pnlWell, null);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getHeaders()
   * <p> This method will return the Well Headers Information data structure
   * @return stHeader = the Well Headers Information data structure
   */

  public iqstratHeadersStruct getHeaders() { return (stHeader); }

  /** Method getLAS()
   * <p> This method will return the LAS File Dependant Variables
   * @return stLAS = the LAS File Data Structure
   */

  public lasFileDataStruct getLAS() { return (stLAS); }

  /** Method getLAS3()
   * <p> This method will return the LAS 3 Well Information Data Structure
   * @return stLAS3 = the LAS 3 Well Information Data Structure
   */

  public las3Struct getLAS3()
  {
    String sDir = "";

    if (stLAS3 != null)
      stLAS3.delete();
    stLAS3 = null; // LAS 3 File Information

    stLAS3 = new las3Struct();
    stLAS3.iType = las3.las3Constants._WELL;
    stLAS3.iArray = 0;
    stLAS3.sType =
        new String(las3.las3Constants.SECTIONS[las3.las3Constants._WELL]);

    stLAS3.iParamRows = TOTAL;
    stLAS3.iParamCols = las3.las3Struct._TOTAL;
    stLAS3.sParams = new String[TOTAL][las3.las3Struct._TOTAL];

    for (int i=0; i<TOTAL; i++)
    {
      for (int j=0; j<las3.las3Struct._TOTAL; j++)
      {
        stLAS3.sParams[i][j] = new String( PARAM[i][j].trim() );
        if (j==_VALUE)
        {
          sDir = "";
          switch (i)
          {
            case _TOWNSHIP:
              if (stLAS3.sParams[i][j].length() > 0)
              {
                if (rbLeft[_TOWNSHIP_DIR].isSelected())
                  sDir = rbLeft[_TOWNSHIP_DIR].getText();
                else
                  sDir = rbRight[_TOWNSHIP_DIR].getText();
                stLAS3.sParams[i][j] = new String(stLAS3.sParams[i][j] + sDir);
              }
              break;
            case _RANGE:
              if (stLAS3.sParams[i][j].length() > 0)
              {
                if (rbLeft[_RANGE_DIR].isSelected())
                  sDir = rbLeft[_RANGE_DIR].getText();
                else
                  sDir = rbRight[_RANGE_DIR].getText();
                stLAS3.sParams[i][j] = new String(stLAS3.sParams[i][j] + sDir);
              }
              break;
          }
        }
      }
    }

    return (stLAS3);
  }

  /** Method getChange()
   * <p> This method will set the data structures depending on the data
   * @param iRow   = The Row Number being modified
   * @param sValue = The Value Column of the Row
   */

  private void getChange( int iRow, String sValue )
  {
    PARAM[iRow][_VALUE] = new String( sValue );

    if (stLAS != null)
    {
      switch (iRow)
      {
        case _START_DEPTH:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stLAS.depthStart = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _END_DEPTH:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stLAS.depthEnd = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _STEP_BY:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stLAS.depthStep =  cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _NULL_VALUE:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stLAS.dNull = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
      }

      if (notifier != null)
        notifier.notifyObservers(new String("LAS - WELL Information"));
    }

    if (stHeader != null)
    {
      switch (iRow)
      {
        case _COMPANY:
          stHeader.sOperator = new String( PARAM[iRow][_VALUE] );
          break;
        case _WELL:
          stHeader.sName     = new String( PARAM[iRow][_VALUE] );
          break;
        case _FIELD:
          stHeader.sField    = new String( PARAM[iRow][_VALUE] );
          break;
        case _SECTION:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.iSection  = cmn.cmnString.stringToInt(PARAM[iRow][_VALUE]);
          break;
        case _TOWNSHIP:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.iTownship = cmn.cmnString.stringToInt(PARAM[iRow][_VALUE]);
          break;
        case _RANGE:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.iRange    = cmn.cmnString.stringToInt(PARAM[iRow][_VALUE]);
          break;
        case _LOCATION:
          stHeader.sLocation = new String( PARAM[iRow][_VALUE] );
          break;
        case _LOCATION_1:
          stHeader.sLocation1 = new String( PARAM[iRow][_VALUE] );
          break;
        case _LOCATION_2:
          stHeader.sLocation2 = new String( PARAM[iRow][_VALUE] );
          break;
        case _COUNTY:
          stHeader.sCounty   = new String( PARAM[iRow][_VALUE] );
          break;
        case _STATE:
          stHeader.state     = new String( PARAM[iRow][_VALUE] );
          break;
        case _COUNTRY:
          stHeader.sCountry  = new String( PARAM[iRow][_VALUE] );
          break;
        case _PROVINCE:
          stHeader.sProvince = new String( PARAM[iRow][_VALUE] );
          break;
        case _SERVICE:
          stHeader.service   = new String( PARAM[iRow][_VALUE] );
          break;
        case _LICENSE:
          stHeader.sLicense  = new String( PARAM[iRow][_VALUE] );
          break;
        case _DATE:
          stHeader.sDate     = new String( PARAM[iRow][_VALUE] );
          break;
        case _API:
          stHeader.sAPI      = new String( PARAM[iRow][_VALUE] );
          break;
        case _UWI:
          stHeader.sUWI      = new String( PARAM[iRow][_VALUE] );
          break;
        case _LATI:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.dLatitude = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _LONG:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.dLongitude = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _GDAT:
          stHeader.sGDAT = new String( PARAM[iRow][_VALUE] );
          break;
        case _X:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.dUTMx = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _Y:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.dUTMy = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
        case _HZCS:
          stHeader.sHZCS = new String( PARAM[iRow][_VALUE] );
          break;
        case _UTM:
          if (cmn.cmnString.isNumeric(PARAM[iRow][_VALUE]))
            stHeader.dZone = cmn.cmnString.stringToDouble(PARAM[iRow][_VALUE]);
          break;
      }

      if (notifier != null)
        notifier.notifyObservers(new String("HEADER - WELL Information"));
    }
  }

  /*========================================================================*
   * ---------------------------- SET METHODS ------------------------------*
   *========================================================================*/

  /** Method setData()
   * <p> This method will set the Headers & LAS File Data Structures
   *  @param stHeader = the Well Header Data Structure
   *  @param stLAS    = the LAS File Data Structure
   */

  public void setData( iqstratHeadersStruct stHeader, lasFileDataStruct stLAS )
  {
    int i=0, j=0;

    this.stHeader = stHeader;
    this.stLAS    = stLAS;

    setData(); // Load the PARAMS Data Structure

    for (i=0; i<TOTAL; i++)
    {
      txtValue[i].setText(PARAM[i][_VALUE]);
    }

    for (i=0; i<TOTAL_DIR; i++)
    {
      rbLeft[i].setSelected(false);
      rbRight[i].setSelected(false);
    }

    if (stHeader.sTownship.equals(DIR[_TOWNSHIP_DIR][0]))
      rbLeft[_TOWNSHIP_DIR].setSelected(true);
    else if (stHeader.sTownship.equals(DIR[_TOWNSHIP_DIR][1]))
      rbRight[_TOWNSHIP_DIR].setSelected(true);

    if (stHeader.sRange.equals(DIR[_RANGE_DIR][0]))
      rbLeft[_RANGE_DIR].setSelected(true);
    else if (stHeader.sRange.equals(DIR[_RANGE_DIR][1]))
      rbRight[_RANGE_DIR].setSelected(true);
  }

  /** Method setData()
   * <p> This method will set the data
   */

  private void setData()
  {
    String sDate[] = { "", "" };

    if (stLAS != null)
    {
      for (int i=0; i<TOTAL; i++)
      {
        switch (i)
        {
          case _START_DEPTH:
            PARAM[i][_VALUE] = new String( "" + stLAS.depthStart );
            break;
          case _END_DEPTH:
            PARAM[i][_VALUE] = new String( "" + stLAS.depthEnd );
            break;
          case _STEP_BY:
            PARAM[i][_VALUE] = new String( "" + stLAS.depthStep );
            break;
          case _NULL_VALUE:
            PARAM[i][_VALUE] = new String( "" + stLAS.dNull );
            break;
        }
      }
    }

    if (stHeader != null)
    {
      for (int i=0; i<TOTAL; i++)
      {
        switch (i)
        {
          case _COMPANY:  // Operator
            PARAM[i][_VALUE] = new String( stHeader.sOperator );
            break;
          case _WELL:    // Well Name or Outcrop Name
            PARAM[i][_VALUE] = new String( stHeader.sName );
            break;
          case _FIELD:   // Field Name
            PARAM[i][_VALUE] = new String( stHeader.sField );
            break;
          case _SECTION: // Section
            PARAM[i][_VALUE] = new String( "" + stHeader.iSection );
            break;
          case _TOWNSHIP: // Township
            PARAM[i][_VALUE] = new String( "" + stHeader.iTownship );
            break;
          case _RANGE: // Range
            PARAM[i][_VALUE] = new String( "" + stHeader.iRange );
            break;
          case _LOCATION:    // location of outcrop or stream cut
            PARAM[i][_VALUE] = new String( stHeader.sLocation );
            break;
          case _LOCATION_1:  // location 1
            PARAM[i][_VALUE] = new String( stHeader.sLocation1 );
            break;
          case _LOCATION_2:  // location 2
            PARAM[i][_VALUE] = new String( stHeader.sLocation2 );
            break;
          case _COUNTY:  // County Name
            PARAM[i][_VALUE] = new String( stHeader.sCounty );
            break;
          case _STATE:   // State Name
            PARAM[i][_VALUE] = new String( stHeader.state );
            break;
          case _COUNTRY:  // Country
            PARAM[i][_VALUE] = new String( stHeader.sCountry );
            break;
          case _PROVINCE: // Province
            PARAM[i][_VALUE] = new String( stHeader.sProvince );
            break;
          case _SERVICE:  // Service Company
            PARAM[i][_VALUE] = new String( stHeader.service );
            break;
          case _LICENSE: // License
            PARAM[i][_VALUE] = new String( stHeader.sLicense );
            break;
          case _DATE:  // Date Logged
            PARAM[i][_VALUE] = new String( stHeader.sDate );
            sDate = cmn.cmnString.convertDate( PARAM[i][_VALUE] );
            PARAM[i][_VALUE]  = new String( sDate[0] );
            PARAM[i][_FORMAT] = new String( sDate[1] );
            break;
          case _API:   // API-Number
            PARAM[i][_VALUE] = new String( stHeader.sAPI );
            break;
          case _UWI:   // UWI
            PARAM[i][_VALUE] = new String( stHeader.sUWI );
            break;
          case _LATI:  // Latitude of Location
            PARAM[i][_VALUE] = new String( "" + stHeader.dLatitude );
            break;
          case _LONG:  // Longitude of Location
            PARAM[i][_VALUE] = new String( "" + stHeader.dLongitude );
            break;
          case _GDAT:  // Geodetic Datum
            if (stHeader.sGDAT.length() > 0)
              PARAM[i][_VALUE] = new String( stHeader.sGDAT );
            break;
          case _X:     // UTM X-Position
            PARAM[i][_VALUE] = new String( "" + stHeader.dUTMx );
            break;
          case _Y:     // UTM Y-Position
            PARAM[i][_VALUE] = new String( "" + stHeader.dUTMy );
            break;
          case _HZCS:  // Horizontal Co-ordinate System
            PARAM[i][_VALUE] = new String( stHeader.sHZCS );
            break;
          case _UTM:   // UTM Zone
            PARAM[i][_VALUE] = new String( "" + stHeader.dZone );
            break;
        }
      }
    }

    if (notifier != null)
      notifier.notifyObservers(new String("LAS 3 WEll Data"));
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    int i=0;

    // Input Variables

    notifier = null; // Observables Object

    // Global Variables

    stHeader = null; // Well Header Information
    stLAS    = null; // LAS File Information

    if (stLAS3 != null)
      stLAS3.delete();
    stLAS3   = null; // LAS 3 File Information

    PARAM = null;

    // Global Panel Widgets

    for (i=0; i<TOTAL; i++)
    {
      lbl_Mnem[i]  = null;
      txtUnit[i]   = null;
      txtValue[i]  = null;
      txtDesc[i]   = null;
      txtFormat[i] = null;
    }

    for (i=0; i<TOTAL_DIR; i++)
    {
      rbLeft[i]    = null;
      rbRight[i]   = null;
    }
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    for (int i=0; i<TOTAL_DIR; i++)
    {
      if (event.getSource() == rbLeft[i])
      {
        switch (i)
        {
          case _TOWNSHIP_DIR:
            stHeader.sTownship = "S";
            break;
          case _RANGE_DIR:
            stHeader.sRange = "E";
            break;
        }
      }

      if (event.getSource() == rbRight[i])
      {
        switch (i)
        {
          case _TOWNSHIP_DIR:
            stHeader.sTownship = "N";
            break;
          case _RANGE_DIR:
            stHeader.sRange = "W";
            break;
        }
      }
    }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    i = 0;
    int    iFound   = -1;
    int    iNumeric = 0;
    String sMessage = "";
    String sTemp    = "";

    for (i=0; i<TOTAL; i++)
    {
      if (e.getSource() == txtValue[i])
      {
        iFound = i;
      }
    }

    if (iFound > -1)
    {
      sTemp = txtValue[iFound].getText();

      switch (iFound)
      {
        case _START_DEPTH:
          iNumeric = 1;
          sMessage = new String(
              "The Starting Depth Value is a Numeric Field " +
              "and it is in units of feet");
          break;
        case _END_DEPTH:
          iNumeric = 1;
          sMessage = new String("The Ending Depth Value is a Numeric Field " +
                                "and it is in units of feet");
          break;
        case _STEP_BY:
          iNumeric = 1;
          sMessage = new String("The Step Depth Value is a Numeric Field " +
                                "and it is in units of feet");
          break;
        case _NULL_VALUE:
          iNumeric = 1;
          sMessage = new String("The Null Value is a Numeric Field");
          break;

        case _COMPANY:
        case _WELL:
        case _FIELD:
          break;
        case _SECTION:
          iNumeric = 1;
          sMessage = new String(
              "The Section Value is a Numeric Integer Field ");
          break;
        case _TOWNSHIP:
          iNumeric = 1;
          sMessage = new String(
              "The Township Value is a Numeric Integer Field ");
          break;
        case _RANGE:
          iNumeric = 1;
          sMessage = new String("The Range Value is a Numeric Integer Field ");
          break;
        case _LOCATION:
        case _LOCATION_1:
        case _LOCATION_2:
        case _COUNTY:
        case _STATE:
        case _COUNTRY:
        case _PROVINCE:
        case _SERVICE:
        case _LICENSE:
        case _DATE:
        case _API:
        case _UWI:
          break;
        case _LATI:
          iNumeric = 1;
          sMessage = new String("The Latitude Value is a Numeric Field " +
                                "and it is in units of decimal degrees");
          break;
        case _LONG:
          iNumeric = 1;
          sMessage = new String("The Longitude Value is a Numeric Field " +
                                "and it is in units of decimal degrees");
          break;
        case _GDAT:
          break;
        case _X:
          iNumeric = 1;
          sMessage = new String("The UTM-X Value is a Numeric Field ");
          break;
        case _Y:
          iNumeric = 1;
          sMessage = new String("The UTM-Y Value is a Numeric Field ");
          break;
        case _HZCS:
          break;
        case _UTM:
          iNumeric = 1;
          sMessage = new String(
              "The UTM Zone Value is a Numeric Integer Field ");
          break;
      }

      if (iNumeric == 1)
      {
        if (!cmn.cmnString.isNumeric(sTemp))
        {
          JOptionPane.showMessageDialog( (Component)null,
                                        sMessage,
                                        "ERROR",
                                        JOptionPane.ERROR_MESSAGE);
        }
      }
      else
      {
        getChange(iFound, txtValue[iFound].getText());
      }
    }
  }
}

/** CLASS las3FileWellPanelFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class las3FileWellPanelFocusAdapter extends java.awt.event.FocusAdapter
{
  las3FileWellPanel adaptee;

  las3FileWellPanelFocusAdapter(las3FileWellPanel adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 01/26/2010
 *  @author  John Victorine
 */
