 /*
 * @las3FilePanel.java Version 1.1 02/17/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import las3.las3Struct;

/** Class las3FilePanel
 *  <p> This Class will display the parameter & curve sections of the Data
 *      Section
 *
 *  @version 1.1 02/17/2010
 *  @author  John R. Victorine
 */

public class las3FilePanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable notifier = null; // Top Observables Object
  private las3Struct st3LAS   = null; // LAS 3 File Data Structure
  private double     dVersion = 3.0;  // LAS Version Number

  // Columns

  public static final int _MNEMONIC    = las3.las3Struct._MNEMONIC;
  public static final int _UNIT        = las3.las3Struct._UNIT;
  public static final int _VALUE       = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT      = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION = las3.las3Struct._ASSOCIATION;

  // Global Panel Widgets

  private int        iParameters   = 0;
  private JLabel     lbl_Mnem_E[]   = null;
  private JTextField txtUnit_E[]    = null;
  private JTextField txtValue_E[]   = null;
  private JTextField txtDesc_E[]    = null;
  private JTextField txtFormat_E[]  = null;
  private JTextField txtAssoc_E[]   = null;

  private int        iCurves        = 0;
  private JLabel     lbl_Mnem_L[]   = null;
  private JTextField txtUnit_L[]    = null;
  private JTextField txtValue_L[]   = null;
  private JTextField txtDesc_L[]    = null;
  private JTextField txtFormat_L[]  = null;
  private JTextField txtAssoc_L[]   = null;

  /** Constructor las3FilePanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param st3LAS   = LAS 3 File Data Structure
   *  @param dVersion = Version of the LAS File
   */

  public las3FilePanel( Observable notifier, las3Struct st3LAS, double dVersion )
  {
    try
    {
      this.notifier = notifier;
      this.st3LAS   = st3LAS;
      this.dVersion = dVersion;

      jbInit();
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int    iSize   = 0;
    int    iSum    = 0;
    int    jcurves = 0;
    int    jparams = 0;
    String sType   = "{ ? }";

    JPanel      pnlCurves        = new JPanel();
    JPanel      pnlParameters    = new JPanel();
    JPanel      pCurves          = new JPanel();
    JPanel      params           = new JPanel();
    JScrollPane scrollParameters = new JScrollPane();
    JScrollPane scrollCurves     = new JScrollPane();

    if (st3LAS != null)
    {
      sType = new String( st3LAS.sType );
    }

    TitledBorder titledBorderCurve = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "~" + sType + "_Definition");

    TitledBorder titledBorderParam = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "~" + sType + "_Parameters");

    titledBorderCurve.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderParam.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setLayout(new BorderLayout());

    if (st3LAS != null)
    {
      jparams = st3LAS.iParamRows;
      jcurves = st3LAS.iCurveRows;
      iSum    = jcurves;
    }

    if (jparams == 0)
      iSize = 0;
    else if (jcurves == 0)
      iSize = 600;
    else if (jparams+jcurves+2 < 30)
      iSize = (jparams+3) * 20;
    else if (jparams+jcurves+2 > 30)
    {
      if (jcurves+1 < 15)
        iSize = 600 - (jcurves+3)*20;
      else
        iSize = 300;
    }

    // Parameters Panel

    if (iSize > 0)
    {
      pnlParameters.setPreferredSize(new Dimension(10, iSize));
      pnlParameters.setBorder(titledBorderParam);
    }
    pnlParameters.setLayout(new BorderLayout());

    // -- Build Parameters Data Panel

    params = parameters( iSize );

    // Curves Panel

    if (jcurves > 0)
      pnlCurves.setBorder(titledBorderCurve);
    pnlCurves.setLayout(new BorderLayout());

    // -- Build Curves Data Panel

    pCurves = curves( iSize );

    // Attach Widgets to Panel

    if (jparams > 0)
    {
      this.add(pnlParameters,             BorderLayout.NORTH);
      pnlParameters.add(scrollParameters, BorderLayout.CENTER);
      scrollParameters.getViewport().add(params, null);
    }

    if (jcurves > 0)
    {
      this.add(pnlCurves,                 BorderLayout.CENTER);
      pnlCurves.add(scrollCurves,         BorderLayout.CENTER);
      scrollCurves.getViewport().add(pCurves, null);
    }
  }

  /** Method parameters()
   * <p> This method will build the Parameters panel
   * @param  iScreen = The Screen Panel Size
   * @return panel   = Parameters Panel
   */

  private JPanel parameters( int iScreen )
  {
    int    i      = 0;
    int    iCount = 0;
    int    iSum   = 0;
    int    iSize  = 0;
    int    iSizeU = 0;
    String sTemp  = "";

    JPanel panel            = new JPanel();

    JPanel pnlA             = new JPanel();
    JLabel lblA_Mnem        = new JLabel();
    JPanel pnlA_a           = new JPanel();
    JPanel pnlA_al          = new JPanel();
    JPanel pnlA_ar          = new JPanel();

    JLabel lblA_Unit        = new JLabel();
    JLabel lblA_Value       = new JLabel();
    JLabel lblA_Colon       = new JLabel();
    JLabel lblA_Description = new JLabel();
    JLabel lblA_Format      = new JLabel();

    JPanel pnl[]            = null;
    JPanel pnl_a[]          = null;
    JPanel pnl_al[]         = null;
    JPanel pnl_ar[]         = null;
    JPanel pnl_Area[]       = null;
    JPanel pnl_Format[]     = null;

    JLabel lbl_Colon[]      = null;
    JLabel lbl_BraceL[]     = null;
    JLabel lbl_BraceR[]     = null;
    JLabel lbl_Bar[]        = null;

    if (st3LAS != null)
    {
      iSum        = st3LAS.iParamRows;
      iParameters = iSum;
      iSize       = iScreen / 20;

      panel.setLayout(new GridLayout(iSum+1, 1));
      panel.setPreferredSize(new Dimension(620, (iSum+1) * 20));

      // Well Information List Title Panel

      pnlA.setLayout(new BorderLayout());

      // -- Mnemonic Column Title Label

      lblA_Mnem.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Mnem.setText("MNEM           ");

      // -- A - a Panel

      pnlA_a.setLayout(new BorderLayout());

      // .. A - a Left Panel

      pnlA_al.setLayout(new BorderLayout());
      pnlA_al.setPreferredSize(new Dimension(180, 10));

      // .... Unit Column Title Label

      lblA_Unit.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Unit.setText(".UNIT   ");

      // .... Value Column Title Label

      lblA_Value.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Value.setHorizontalAlignment(SwingConstants.CENTER);
      lblA_Value.setText("VALUE");

      // .. A - a Right Panel

      pnlA_ar.setLayout(new BorderLayout());

      // .... Colon Separator

      lblA_Colon.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Colon.setText(" : ");

      // .... Description Column Title

      lblA_Description.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Description.setHorizontalAlignment(SwingConstants.CENTER);
      lblA_Description.setText("DESCRIPTION");

      // .... Format Column Title

      lblA_Format.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Format.setText("{ Format } | Association");

      // Attach Well Information Column Header

      panel.add(pnlA,               null);
      pnlA.add(lblA_Mnem,           BorderLayout.WEST);
      pnlA.add(pnlA_a,              BorderLayout.CENTER);
      pnlA_a.add(pnlA_al,           BorderLayout.WEST);
      pnlA_al.add(lblA_Unit,        BorderLayout.WEST);
      pnlA_al.add(lblA_Value,       BorderLayout.CENTER);
      pnlA_a.add(pnlA_ar,           BorderLayout.CENTER);
      pnlA_ar.add(lblA_Colon,       BorderLayout.WEST);
      pnlA_ar.add(lblA_Description, BorderLayout.CENTER);
      pnlA_ar.add(lblA_Format,      BorderLayout.EAST);

      // Parameters Array

      pnl        = new JPanel[iSum];
      pnl_a      = new JPanel[iSum];
      pnl_al     = new JPanel[iSum];
      pnl_ar     = new JPanel[iSum];
      pnl_Format = new JPanel[iSum];
      pnl_Area   = new JPanel[iSum];

      lbl_Colon  = new JLabel[iSum];
      lbl_BraceL = new JLabel[iSum];
      lbl_BraceR = new JLabel[iSum];
      lbl_Bar    = new JLabel[iSum];

      lbl_Mnem_E  = new JLabel[iSum];
      txtUnit_E   = new JTextField[iSum];
      txtValue_E  = new JTextField[iSum];
      txtDesc_E   = new JTextField[iSum];
      txtFormat_E = new JTextField[iSum];
      txtAssoc_E  = new JTextField[iSum];

      iSize  = 0;
      iSizeU = 4;
      for (i=0; i<st3LAS.iParamRows; i++)
      {
        if (st3LAS.sParams[i][_MNEMONIC].length() > iSize)
          iSize  = st3LAS.sParams[i][_MNEMONIC].length();

        if (st3LAS.sParams[i][_UNIT].length() > iSizeU)
          iSizeU = st3LAS.sParams[i][_UNIT].length();
      }

      for (i=0; i<st3LAS.iParamRows; i++)
      {
        if (iCount < iSum)
        {
          // Array of Well Information Central Row Panels

          pnl[iCount] = new JPanel();
          pnl[iCount].setLayout(new BorderLayout());

          // -- Array or Mnemonic Columns

          sTemp = setPadding( st3LAS.sParams[i][_MNEMONIC], iSize );

          lbl_Mnem_E[iCount] = new JLabel();
          lbl_Mnem_E[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Mnem_E[iCount].setText( sTemp + " ." );
//          lbl_Mnem_E[iCount].setText( st3LAS.sParams[i][_MNEMONIC] + " ."  );

          // -- Array of a Panels

          pnl_a[iCount] = new JPanel();
          pnl_a[iCount].setLayout(new BorderLayout());

          // .. Array of a Left Panels

          pnl_al[iCount] = new JPanel();
          pnl_al[iCount].setLayout(new BorderLayout());
          pnl_al[iCount].setPreferredSize(new Dimension(180, 10));

          // .... Array of Unit Textfields

          txtUnit_E[iCount] = new JTextField();
          txtUnit_E[iCount].setText( st3LAS.sParams[i][_UNIT] );
          txtUnit_E[iCount].setEditable(false);
//          txtUnit_E[iCount].setColumns(4);
          txtUnit_E[iCount].setColumns(iSizeU);
          txtUnit_E[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // .... Array of Value Textfields

          txtValue_E[iCount] = new JTextField();
          txtValue_E[iCount].setText( st3LAS.sParams[i][_VALUE] );
          txtValue_E[iCount].setCaretPosition(0);
          txtValue_E[iCount].setHorizontalAlignment(SwingConstants.TRAILING);
          txtValue_E[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // .. Array of a Left Panels

          pnl_ar[iCount] = new JPanel();
          pnl_ar[iCount].setLayout(new BorderLayout());

          // .... Array of Colon Separators

          lbl_Colon[iCount] = new JLabel();
          lbl_Colon[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Colon[iCount].setText(" : ");

          // .... Array of Descriptions Textfields

          txtDesc_E[iCount] = new JTextField();
          txtDesc_E[iCount].setText( st3LAS.sParams[i][_DESCRIPTION] );
          txtDesc_E[iCount].setCaretPosition(0);
          txtDesc_E[iCount].setHorizontalAlignment(SwingConstants.LEADING);
          txtDesc_E[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // --- Arrea Panel

          pnl_Area[iCount] = new JPanel();
          pnl_Area[iCount].setLayout(new BorderLayout());
          pnl_Area[iCount].setPreferredSize(new Dimension(150, 10));

          // .... Array of Formats Panels

          pnl_Format[iCount] = new JPanel();
          pnl_Format[iCount].setLayout(new BorderLayout());

          lbl_BraceL[iCount] = new JLabel();
          lbl_BraceL[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_BraceL[iCount].setText("{");

          if (st3LAS.sParams[i][_FORMAT].length() == 0)
            st3LAS.sParams[i][_FORMAT] = new String("S");

          if (((txtUnit_E[iCount].getText().length() > 0) ||
               (cmn.cmnString.isNumeric( txtValue_E[iCount].getText() ))) &&
              (!st3LAS.sParams[i][_UNIT].equals("DATE")))
          {
            st3LAS.sParams[i][_FORMAT] = new String("F");
          }

          if (!st3LAS.sParams[i][_FORMAT].equals("S"))
            txtUnit_E[iCount].setEditable(true);

          txtFormat_E[iCount] = new JTextField();
          txtFormat_E[iCount].setText( st3LAS.sParams[i][_FORMAT] );
          txtFormat_E[iCount].setColumns(4);
          txtFormat_E[iCount].setHorizontalAlignment(SwingConstants.CENTER);
          txtFormat_E[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          lbl_BraceR[iCount] = new JLabel();
          lbl_BraceR[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_BraceR[iCount].setText("}");

          lbl_Bar[iCount] = new JLabel();
          lbl_Bar[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Bar[iCount].setText("|");

          txtAssoc_E[iCount] = new JTextField();
          txtAssoc_E[iCount].setText( st3LAS.sParams[i][_ASSOCIATION] );
          txtAssoc_E[iCount].setCaretPosition(0);
          txtAssoc_E[iCount].setColumns(6);
          txtAssoc_E[iCount].setHorizontalAlignment(SwingConstants.LEFT);
          txtAssoc_E[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // Add Array Widgets to Panel

          panel.add(pnl[iCount],                      null);
          pnl[iCount].add(lbl_Mnem_E[iCount],         BorderLayout.WEST);
          pnl[iCount].add(pnl_a[iCount],              BorderLayout.CENTER);
          pnl_a[iCount].add(pnl_al[iCount],           BorderLayout.WEST);
          pnl_al[iCount].add(txtUnit_E[iCount],       BorderLayout.WEST);
          pnl_al[iCount].add(txtValue_E[iCount],      BorderLayout.CENTER);

          pnl_a[iCount].add(pnl_ar[iCount],           BorderLayout.CENTER);
          pnl_ar[iCount].add(lbl_Colon[iCount],       BorderLayout.WEST);
          pnl_ar[iCount].add(txtDesc_E[iCount],       BorderLayout.CENTER);
          pnl_ar[iCount].add(pnl_Area[iCount],        BorderLayout.EAST);
          pnl_Area[iCount].add(pnl_Format[iCount],    BorderLayout.WEST);
          pnl_Format[iCount].add(lbl_BraceL[iCount],  BorderLayout.WEST);
          pnl_Format[iCount].add(txtFormat_E[iCount], BorderLayout.CENTER);
          pnl_Format[iCount].add(lbl_BraceR[iCount],  BorderLayout.EAST);

          if (st3LAS.sParams[i][_ASSOCIATION].length() > 0)
          {
            pnl_Area[iCount].add(lbl_Bar[iCount], BorderLayout.CENTER);
            pnl_Area[iCount].add(txtAssoc_E[iCount], BorderLayout.EAST);
          }
          iCount++;
        }
      }
    }

    return (panel);
  }

  /** Method curves()
   * <p> This method will build the Curves panel
   * @param  iScreen = The Screen Panel Size
   * @return panel   = Curves Panel
   */

  private JPanel curves( int iScreen )
  {
    int    i      = 0;
    int    iCount = 0;
    int    iSum   = 0;
    int    iSize  = 0;
    int    iSizeU = 0;
    String sTemp  = "";

    JPanel panel            = new JPanel();

    JPanel pnlA             = new JPanel();
    JLabel lblA_Mnem        = new JLabel();
    JPanel pnlA_a           = new JPanel();
    JPanel pnlA_al          = new JPanel();
    JPanel pnlA_ar          = new JPanel();

    JLabel lblA_Unit        = new JLabel();
    JLabel lblA_Value       = new JLabel();
    JLabel lblA_Colon       = new JLabel();
    JLabel lblA_Description = new JLabel();
    JLabel lblA_Format      = new JLabel();

    JPanel pnl[]            = null;
    JPanel pnl_a[]          = null;
    JPanel pnl_al[]         = null;
    JPanel pnl_ar[]         = null;
    JPanel pnl_Area[]       = null;
    JPanel pnl_Format[]     = null;

    JLabel lbl_Colon[]      = null;
    JLabel lbl_BraceL[]     = null;
    JLabel lbl_BraceR[]     = null;
    JLabel lbl_Bar[]        = null;

    if (st3LAS != null)
    {
      iSum    = st3LAS.iCurveRows;
      iCurves = iSum;
      iSize   = (600 - iScreen-40) / 20;

      if (iSum+1 < iSize)
      {
        panel.setLayout(new GridLayout(iSize, 1));
        panel.setPreferredSize(new Dimension(620, iSize * 20));
      }
      else
      {
        panel.setLayout(new GridLayout(iSum+1, 1));
        panel.setPreferredSize(new Dimension(620, (iSum+1) * 20));
      }

      // Well Information List Title Panel

      pnlA.setLayout(new BorderLayout());

      // -- Mnemonic Column Title Label

      lblA_Mnem.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Mnem.setText("MNEM           ");

      // -- A - a Panel

      pnlA_a.setLayout(new BorderLayout());

      // .. A - a Left Panel

      pnlA_al.setLayout(new BorderLayout());
      pnlA_al.setPreferredSize(new Dimension(180, 10));

      // .... Unit Column Title Label

      lblA_Unit.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Unit.setText(".UNIT   ");

      // .... Value Column Title Label

      lblA_Value.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Value.setHorizontalAlignment(SwingConstants.CENTER);
      lblA_Value.setText("VALUE");

      // .. A - a Right Panel

      pnlA_ar.setLayout(new BorderLayout());

      // .... Colon Separator

      lblA_Colon.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Colon.setText(" : ");

      // .... Description Column Title

      lblA_Description.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Description.setHorizontalAlignment(SwingConstants.CENTER);
      lblA_Description.setText("DESCRIPTION");

      // .... Format Column Title

      lblA_Format.setFont(new java.awt.Font("Monospaced", 1, 11));
      lblA_Format.setText("{ Format } | Association");

      // Attach Well Information Column Header

      panel.add(pnlA,               null);
      pnlA.add(lblA_Mnem,           BorderLayout.WEST);
      pnlA.add(pnlA_a,              BorderLayout.CENTER);
      pnlA_a.add(pnlA_al,           BorderLayout.WEST);
      pnlA_al.add(lblA_Unit,        BorderLayout.WEST);
      pnlA_al.add(lblA_Value,       BorderLayout.CENTER);
      pnlA_a.add(pnlA_ar,           BorderLayout.CENTER);
      pnlA_ar.add(lblA_Colon,       BorderLayout.WEST);
      pnlA_ar.add(lblA_Description, BorderLayout.CENTER);
      pnlA_ar.add(lblA_Format,      BorderLayout.EAST);

      // Parameters Array

      pnl        = new JPanel[iSum];
      pnl_a      = new JPanel[iSum];
      pnl_al     = new JPanel[iSum];
      pnl_ar     = new JPanel[iSum];
      pnl_Format = new JPanel[iSum];
      pnl_Area   = new JPanel[iSum];

      lbl_Colon  = new JLabel[iSum];
      lbl_BraceL = new JLabel[iSum];
      lbl_BraceR = new JLabel[iSum];
      lbl_Bar    = new JLabel[iSum];

      lbl_Mnem_L  = new JLabel[iSum];
      txtUnit_L   = new JTextField[iSum];
      txtValue_L  = new JTextField[iSum];
      txtDesc_L   = new JTextField[iSum];
      txtFormat_L = new JTextField[iSum];
      txtAssoc_L  = new JTextField[iSum];

      iSize  = 0;
      iSizeU = 4;
      for (i=0; i<st3LAS.iCurveRows; i++)
      {
        if (st3LAS.sCurves[i][_MNEMONIC].length() > iSize)
          iSize = st3LAS.sCurves[i][_MNEMONIC].length();

        if (st3LAS.sCurves[i][_UNIT].length() > iSizeU)
          iSizeU = st3LAS.sCurves[i][_UNIT].length();
      }

      for (i=0; i<st3LAS.iCurveRows; i++)
      {
        if (iCount < iSum)
        {
          // Array of Well Information Central Row Panels

          pnl[iCount] = new JPanel();
          pnl[iCount].setLayout(new BorderLayout());

          // -- Array or Mnemonic Columns

          sTemp = setPadding( st3LAS.sCurves[i][_MNEMONIC], iSize );

          lbl_Mnem_L[iCount] = new JLabel();
          lbl_Mnem_L[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Mnem_L[iCount].setText( sTemp + " ." );
//          lbl_Mnem_L[iCount].setText( st3LAS.sCurves[i][_MNEMONIC] + " ." );

          // -- Array of a Panels

          pnl_a[iCount] = new JPanel();
          pnl_a[iCount].setLayout(new BorderLayout());

          // .. Array of a Left Panels

          pnl_al[iCount] = new JPanel();
          pnl_al[iCount].setLayout(new BorderLayout());
          pnl_al[iCount].setPreferredSize(new Dimension(180, 10));

          // .... Array of Unit Textfields

          txtUnit_L[iCount] = new JTextField();
          txtUnit_L[iCount].setText( st3LAS.sCurves[i][_UNIT] );
//          txtUnit_L[iCount].setEditable(false);
//          txtUnit_L[iCount].setColumns(4);
          txtUnit_L[iCount].setColumns(iSizeU);
          txtUnit_L[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // .... Array of Value Textfields

          txtValue_L[iCount] = new JTextField();
          txtValue_L[iCount].setText( st3LAS.sCurves[i][_VALUE] );
          txtValue_L[iCount].setCaretPosition(0);
          txtValue_L[iCount].setHorizontalAlignment(SwingConstants.TRAILING);
          txtValue_L[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // .. Array of a Left Panels

          pnl_ar[iCount] = new JPanel();
          pnl_ar[iCount].setLayout(new BorderLayout());

          // .... Array of Colon Separators

          lbl_Colon[iCount] = new JLabel();
          lbl_Colon[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Colon[iCount].setText(" : ");

          // .... Array of Descriptions Textfields

          txtDesc_L[iCount] = new JTextField();
          txtDesc_L[iCount].setText( st3LAS.sCurves[i][_DESCRIPTION] );
          txtDesc_L[iCount].setCaretPosition(0);
          txtDesc_L[iCount].setHorizontalAlignment(SwingConstants.LEADING);
          txtDesc_L[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // --- Arrea Panel

          pnl_Area[iCount] = new JPanel();
          pnl_Area[iCount].setLayout(new BorderLayout());
          pnl_Area[iCount].setPreferredSize(new Dimension(150, 10));

          // .... Array of Formats Panels

          pnl_Format[iCount] = new JPanel();
          pnl_Format[iCount].setLayout(new BorderLayout());

          lbl_BraceL[iCount] = new JLabel();
          lbl_BraceL[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_BraceL[iCount].setText("{");

//          st3LAS.sCurves[i][_FORMAT] = new String("S");
//          if ((txtUnit_L[iCount].getText().length() > 0) ||
//              (cmn.cmnString.isNumeric( txtValue_L[iCount].getText() )))
//          {
//            st3LAS.sCurves[i][_FORMAT] = new String("F");
//            txtUnit_L[iCount].setEditable(true);
//          }

          txtFormat_L[iCount] = new JTextField();
          txtFormat_L[iCount].setText( st3LAS.sCurves[i][_FORMAT] );
          txtFormat_L[iCount].setColumns(4);
          txtFormat_L[iCount].setHorizontalAlignment(SwingConstants.CENTER);
          txtFormat_L[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          lbl_BraceR[iCount] = new JLabel();
          lbl_BraceR[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_BraceR[iCount].setText("}");

          lbl_Bar[iCount] = new JLabel();
          lbl_Bar[iCount].setFont(new java.awt.Font("Monospaced", 1, 11));
          lbl_Bar[iCount].setText("|");

          txtAssoc_L[iCount] = new JTextField();
          txtAssoc_L[iCount].setText( st3LAS.sCurves[i][_ASSOCIATION] );
          txtAssoc_L[iCount].setCaretPosition(0);
          txtAssoc_L[iCount].setColumns(6);
          txtAssoc_L[iCount].setHorizontalAlignment(SwingConstants.LEFT);
          txtAssoc_L[iCount].addFocusListener(new las3FilePanelFocusAdapter(this));

          // Add Array Widgets to Panel

          panel.add(pnl[iCount],                      null);
          pnl[iCount].add(lbl_Mnem_L[iCount],         BorderLayout.WEST);
          pnl[iCount].add(pnl_a[iCount],              BorderLayout.CENTER);
          pnl_a[iCount].add(pnl_al[iCount],           BorderLayout.WEST);
          pnl_al[iCount].add(txtUnit_L[iCount],       BorderLayout.WEST);
          pnl_al[iCount].add(txtValue_L[iCount],      BorderLayout.CENTER);

          pnl_a[iCount].add(pnl_ar[iCount],           BorderLayout.CENTER);
          pnl_ar[iCount].add(lbl_Colon[iCount],       BorderLayout.WEST);
          pnl_ar[iCount].add(txtDesc_L[iCount],       BorderLayout.CENTER);
          pnl_ar[iCount].add(pnl_Area[iCount],        BorderLayout.EAST);
          pnl_Area[iCount].add(pnl_Format[iCount],    BorderLayout.WEST);
          pnl_Format[iCount].add(lbl_BraceL[iCount],  BorderLayout.WEST);
          pnl_Format[iCount].add(txtFormat_L[iCount], BorderLayout.CENTER);
          pnl_Format[iCount].add(lbl_BraceR[iCount],  BorderLayout.EAST);

          if (st3LAS.sCurves[i][_ASSOCIATION].length() > 0)
          {
            pnl_Area[iCount].add(lbl_Bar[iCount], BorderLayout.CENTER);
            pnl_Area[iCount].add(txtAssoc_L[iCount], BorderLayout.EAST);
          }
          iCount++;
        }
      }
    }

    return (panel);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getTypeID()
   * <p> This method will return the LAS 3 File Data Type
   * @return iType = the LAS 3 File Data Type
   */

  public int getTypeID()
  {
    int iType = -1;

    if (st3LAS != null)
      iType = st3LAS.iType;

    return (iType);
  }

  /** Method getType()
   * <p> This method will return the LAS 3 File Data Type String
   * @return iType = the LAS 3 File Data Type
   */

  public String getType()
  {
    String sType = "";

    if (st3LAS != null)
      sType = new String( st3LAS.sType );

    return (sType);
  }

  /** Method getArray()
   * <p> This method will return the LAS 3 File Array Number
   * @return iArray = the LAS 3 File Array Number
   */

  public int getArray()
  {
    int iArray = 0;

    if (st3LAS != null)
      iArray = st3LAS.iArray;

    return (iArray);
  }

  /** Method getData()
   * <p> This method will return the LAS 3 File Data Structure
   * @return st3LAS = the LAS 3 File Data Structure
   */

  public las3Struct getData() { return (st3LAS); }

  /** Method getEChange()
   * <p> This method will set the data structures depending on the data
   * @param iRow    = The Row Number being modified
   * @param iCol    = The Column being modified
   * @param sValue  = The Value Column of the Row
   */

  private void getEChange( int iRow, int iCol, String sValue )
  {
    if (st3LAS != null)
    {
      st3LAS.sParams[iRow][iCol] = new String( sValue );

      if (notifier != null)
        notifier.notifyObservers(new String("LAS 3 File Parameters Modified"));
    }
  }

  /** Method getLChange()
   * <p> This method will set the data structures depending on the data
   * @param iRow    = The Row Number being modified
   * @param iCol    = The Column being modified
   * @param sValue  = The Value Column of the Row
   */

  private void getLChange( int iRow, int iCol, String sValue )
  {
    if (st3LAS != null)
    {
      st3LAS.sCurves[iRow][iCol] = new String( sValue );

      if (notifier != null)
        notifier.notifyObservers(new String("LAS 3 File Curves Modified"));
    }
  }

  /*========================================================================*
   * ---------------------------- SET METHODS ------------------------------*
   *========================================================================*/

  /** Method setPadding()
   * <p> This method will pad spaces in order to fit the Mnemonic equally spaced
   * @param  str = original string
   * @param  len = desired length
   * @return str = padded string
   */

  private String setPadding( String str, int len )
  {
    int iSpace = 0;
    if (str.length() < len)
    {
      iSpace = len-str.length();
      for (int i=0; i<iSpace; i++)
        str = new String( str + " ");
    }

    return (str);
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    int i=0;

    // Input Variables

    notifier = null; // Top Observables Object
    st3LAS   = null; // LAS 3 File Data Structure
    dVersion = 0.0;  // LAS Version Number

    // Global Panel Widgets

    for (i=0; i<iParameters; i++)
    {
      lbl_Mnem_E[i]  = null;
      txtUnit_E[i]   = null;
      txtValue_E[i]  = null;
      txtDesc_E[i]   = null;
      txtFormat_E[i] = null;
      txtAssoc_E[i]  = null;
    }
    iParameters  = 0;

    for (i=0; i<iCurves; i++)
    {
      lbl_Mnem_L[i]  = null;
      txtUnit_L[i]   = null;
      txtValue_L[i]  = null;
      txtDesc_L[i]   = null;
      txtFormat_L[i] = null;
      txtAssoc_L[i]  = null;
    }
    iCurves      = 0;
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event) { }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    i = 0;
    int    iRow     = -1;
    int    iColumn  = -1;
    int    iRowE    = -1;
    int    iColumnE = -1;
    int    iRowL    = -1;
    int    iColumnL = -1;
    int    iNumeric = 0;
    String sMessage = "";
    String sTemp    = "";

    for (i=0; i<iParameters; i++)
    {
      if (e.getSource() == txtUnit_E[i])
      {
        iRowE    = i;
        iColumnE = _UNIT;
      }

      if (e.getSource() == txtValue_E[i])
      {
        iRowE    = i;
        iColumnE = _VALUE;
      }

      if (e.getSource() == txtDesc_E[i])
      {
        iRowE    = i;
        iColumnE = _DESCRIPTION;
      }

      if (e.getSource() == txtFormat_E[i])
      {
        iRowE    = i;
        iColumnE = _FORMAT;
      }

      if (e.getSource() == txtAssoc_E[i])
      {
        iRowE    = i;
        iColumnE = _ASSOCIATION;
      }
    }

    for (i=0; i<iCurves; i++)
    {
      if (e.getSource() == txtUnit_L[i])
      {
        iRowL    = i;
        iColumnL = _UNIT;
      }

      if (e.getSource() == txtValue_L[i])
      {
        iRowL    = i;
        iColumnL = _VALUE;
      }

      if (e.getSource() == txtDesc_L[i])
      {
        iRowL    = i;
        iColumnL = _DESCRIPTION;
      }

      if (e.getSource() == txtFormat_L[i])
      {
        iRowL    = i;
        iColumnL = _FORMAT;
      }

      if (e.getSource() == txtAssoc_L[i])
      {
        iRowL    = i;
        iColumnL = _ASSOCIATION;
      }
    }

    if (iColumnE > -1)
    {
      switch (iColumnE)
      {
        case _MNEMONIC:
          break;
        case _UNIT:
          getEChange(iRowE, iColumnE, txtUnit_E[iRowE].getText());
          break;
        case _VALUE:
          getEChange(iRowE, iColumnE, txtValue_E[iRowE].getText());
          break;
        case _DESCRIPTION:
          getEChange(iRowE, iColumnE, txtDesc_E[iRowE].getText());
          break;
        case _FORMAT:
          getEChange(iRowE, iColumnE, txtFormat_E[iRowE].getText());
          break;
        case _ASSOCIATION:
          getEChange(iRowE, iColumnE, txtAssoc_E[iRowE].getText());
          break;
      }
    }

    if (iColumnL > -1)
    {
      switch (iColumnL)
      {
        case _MNEMONIC:
          break;
        case _UNIT:
          getLChange(iRowL, iColumnL, txtUnit_L[iRowL].getText());
          break;
        case _VALUE:
          getLChange(iRowL, iColumnL, txtValue_L[iRowL].getText());
          break;
        case _DESCRIPTION:
          getLChange(iRowL, iColumnL, txtDesc_L[iRowL].getText());
          break;
        case _FORMAT:
          getLChange(iRowL, iColumnL, txtFormat_L[iRowL].getText());
          break;
        case _ASSOCIATION:
          getLChange(iRowL, iColumnL, txtAssoc_L[iRowL].getText());
          break;
      }
    }
  }
}

/** CLASS las3FilePanelFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class las3FilePanelFocusAdapter extends java.awt.event.FocusAdapter
{
  las3FilePanel adaptee;

  las3FilePanelFocusAdapter(las3FilePanel adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 02/17/2010
 *  @author  John Victorine
 */
