/*
 * @las3FileFrame.java Version 1.1 01/26/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.gui;

import java.io.*;
import java.net.*;
import java.lang.*;
import java.util.*;
import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import gemini.gui.geminiFindPath;         // Search for Directory Path Dialog

import cmn.cmnLASFileListStruct;          // Common LAS File List Data Structure

import iqstrat.iqstratStruct;             // Global IQSTRAT Data Structure
import iqstrat.iqstratHeadersStruct;      // Header Data Structure
import iqstrat.iqstratControlStruct;      // Plot Control Data Structure
import iqstrat.iqstratRemarkListStruct;   // Remarks/Comments/Notes List Structure
import iqstrat.iqstratShaleListStruct;    // Shale Data List Structure

import las.lasFileDataStruct;             // LAS File Data Structure
import las3.las3ListStruct;               // LAS File Data List Structure
import las3.las3Struct;                   // LAS 3 Data Structure
import las3.gui.las3FileWellPanel;        // LAS 3 File Well Panel
import las3.gui.las3FilePanel;            // LAS 3 File Panel
import las3.io.las3WriteFile;             // LAS 3 Write File
import las3.io.las3Read;                  // LAS 3 Read File

import rock.rockDataListStruct;           // Rock Data List Structure
import rock.rockImagesListStruct;         // Rock Images List Structure

import horizon.strat.stratListStruct;     // Stratigraphic Units List Structure
import horizon.seq.seqListStruct;         // Sequence Strat List Structure
import horizon.regions.regionsListStruct; // Perforations Data List Structure
import horizon.bio.bioStratListStruct;    // Bio-Stratigraphic Units List Structure
import horizon.env.envListStruct;         // Environment List Structure

import pfeffer.pfefferDataListStruct;     // PfEFFER Data List Structure

import brine.brineListStruct;             // Brine Data List Structure

import dst.dstListStruct;                 // DST Data List Structure

/** Class las3FileFrame
 *  <p> This Class will display the parameters sections of the LAS 3 File
 *
 *  @version 1.1 01/26/2010
 *  @author  John R. Victorine
 */

public class las3FileFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable              pNotifier   = null; // Top Observables Object
  private iqstratStruct           stStruct    = null; // Global IQSTRAT Data Struct
  private cmnLASFileListStruct    stFiles     = null; // LAS Files Information
  private las3ListStruct          stLAS3      = null; // LAS 3 Well Parameters
  private iqstratHeadersStruct    stHeader    = null; // Well Header Information
  private lasFileDataStruct       stLAS       = null; // LAS File Information
  private rockDataListStruct      stRock      = null; // Rock Data List Structure
  private brineListStruct         stBrine     = null; // Brine Data List Structure
  private rockImagesListStruct    stImages    = null; // Rock Images List Structure
  private stratListStruct         stTops      = null; // Stratigraphic List Structure
  private seqListStruct           stSeq       = null; // Sequence Strat List Structure
  private envListStruct           stEnv       = null; // Environment List Structure
  private regionsListStruct       stPerf      = null; // Perforation List Struct
  private bioStratListStruct      stBio       = null; // Bio-Stratigraphic List Struct
  private iqstratControlStruct    stControl   = null; // Control Data Structure
  private pfefferDataListStruct   stPfeffer   = null; // Pfeffer List Structure
  private iqstratRemarkListStruct stRemarks   = null; // Remarks List Structure
  private iqstratShaleListStruct  stShale     = null; // Shale List Structure
  private dstListStruct           stDST       = null; // DST Data List Structure

  private boolean bLog[]      = {true,true,true,true,true,
                                 true,true,true,true,true,
                                 true,true,true,true,true,
                                 true,true,true,true,true,
                                 true};

  // Global Variables

  private Observable        notifier = null;    // Observables Object

//  private double     dVersion = 2.0;     // Version Number
  private double            dVersion = 0.0;     // Version Number
  private double            dNull    = -999.95; // LAS 3 Null Value
  private int               iArray[] = null;    // Array of data section counts

  private las3FileWellPanel pWell = null; // Well Information Panel
  private int               iPnl  = 2;    // Total Number of Panel Types
  private las3FilePanel     pnl[] = null; // Array of Data Sections

  private geminiFindPath    pFind = null; // Search for File path Dialog

//  private las3FileLogPanel  pLog     = null; // Log Section Panel

  private String            sKEY       = "0";

  // Columns

  public static final int _MNEMONIC    = las3.las3Struct._MNEMONIC;
  public static final int _UNIT        = las3.las3Struct._UNIT;
  public static final int _VALUE       = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT      = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION = las3.las3Struct._ASSOCIATION;

  // Log Parameters Textfields

  private String PARAM[][] = las3.las3Parameters.LOG_PARAMETERS;

  // -- LAS 3 File Data Structures

  private las3Struct  st3Log       = null; // LAS 3 Log Data Structure
  private las3Struct  st3Control   = null; // LAS 3 Control Data Structure
  private las3Struct  st3Rock      = null; // LAS 3 Rock Data Structure
  private las3Struct  st3Brine     = null; // LAS 3 Brine Data Structure
  private las3Struct  st3Images    = null; // LAS 3 Rock Images Data Structure
  private las3Struct  st3Tops      = null; // LAS 3 Tops Data Structure
  private las3Struct  st3Seq       = null; // LAS 3 Sequence Strat Data Structure
  private las3Struct  st3Env       = null; // LAS 3 Envionment Data Structure
  private las3Struct  st3Perf      = null; // LAS 3 Perforation Data Structure
  private las3Struct  st3Bio       = null; // LAS 3 Bio-Stratigraphy  Data Structure
  private las3Struct  st3Flow      = null; // LAS 3 PfEFFER Flow Units Data Struct
  private las3Struct  st3Pfeffer   = null; // LAS 3 PfEFFER Data Structure
  private las3Struct  st3Remarks   = null; // LAS 3 Remarks Data Structure
  private las3Struct  st3Shale     = null; // LAS 3 Shale Data Structure

  private las3ListStruct st3DST    = null; // LAS 3 DST Data List Structure

  // Global Frame Widgets

  private JTabbedPane tabbedPane = new JTabbedPane();

  private JMenuItem   mSave      = null;  // Save as LAS 3 File
  private JMenuItem   mExit      = null;  // Exit Profile Plot

  /** Constructor las3FileFrame()
   *  <p> This is the Constructor for this class.
   *      Note: bLog is an indicator to identify which Data Sections to
   *            make available ( required data sections )
   *  @param pNotifier   = Observable
   *  @param stStruct    = Global IQSTRAT Data Structure
   *  @param stFiles     = LAS Files Information
   *  @param stHeader    = the Well Header Data Structure
   *  @param stLAS       = the LAS File Data Structure
   *  @param stRock      = the Rock Data List Structure
   *  @param stBrine     = Brine Data List Structure
   *  @param stImages    = Rock Image Data List Structure
   *  @param stTops      = Stratigraphic Units Data List Structure
   *  @param stSeq       = Sequence Stratigraphy Data List Structure
   *  @param stEnv       = Depositional Environment Data List Structure
   *  @param stPerf      = Perforation Data List Structure
   *  @param stBio       = Bio-Stratigraphy Data List Structure
   *  @param stDST       = DST Data List Structure
   *  @param stControl   = Control Data Structure
   *  @param stShale     = Shale Data List Structure
   *  @param stPfeffer   = PfEFFER Data List Structure
   *  @param stRemarks   = Remarkds Data List Structure
   */

  public las3FileFrame( Observable              pNotifier,
                        iqstratStruct           stStruct,
                        cmnLASFileListStruct    stFiles,
                        iqstratHeadersStruct    stHeader,
                        lasFileDataStruct       stLAS,
                        // Rock & Brine Data Structues
                        rockDataListStruct      stRock,
                        brineListStruct         stBrine,
                        rockImagesListStruct    stImages,
                        // Horizon Data Structures
                        stratListStruct         stTops,
                        seqListStruct           stSeq,
                        envListStruct           stEnv,
                        regionsListStruct       stPerf,
                        bioStratListStruct      stBio,
                        // Misc Data Structures
                        dstListStruct           stDST,
                        iqstratControlStruct    stControl,
                        iqstratShaleListStruct  stShale,
                        pfefferDataListStruct   stPfeffer,
                        iqstratRemarkListStruct stRemarks )
  {
    try
    {
      this.pNotifier   = pNotifier;
      this.stStruct    = stStruct;
      this.stFiles     = stFiles;
      this.stHeader    = stHeader;
      this.stLAS       = stLAS;
      this.stRock      = stRock;
      this.stBrine     = stBrine;
      this.stImages    = stImages;
      this.stTops      = stTops;
      this.stSeq       = stSeq;
      this.stEnv       = stEnv;
      this.stPerf      = stPerf;
      this.stBio       = stBio;
      this.stDST       = stDST;
      this.stControl   = stControl;
      this.stShale     = stShale;
      this.stPfeffer   = stPfeffer;
      this.stRemarks   = stRemarks;

      if (stStruct.bLAS != null)
      {
        this.bLog      = stStruct.bLAS;
	  }

      loadData(); // Initialize the LAS 3 Panel Arrays

      // Create the LAS 3 File Frame

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor las3FileFrame()
   *  <p> This is the Constructor for this class.
   *      Note: bLog is an indicator to identify which Data Sections to
   *            make available ( required data sections )
   *  @param pNotifier   = Observable
   *  @param stStruct    = Global IQSTRAT Data Structure
   *  @param stFiles     = LAS Files Information
   *  @param stHeader    = the Well Header Data Structure
   *  @param stLAS       = the LAS File Data Structure
   *  @param stBrine     = Brine Data List Structure
   *  @param stControl   = Control Data Structure
   */

  public las3FileFrame( Observable              pNotifier,
                        iqstratStruct           stStruct,
                        cmnLASFileListStruct    stFiles,
                        iqstratHeadersStruct    stHeader,
                        lasFileDataStruct       stLAS,
                        // Brine Data Structues
                        brineListStruct         stBrine,
                        // Misc Data Structures
                        iqstratControlStruct    stControl )
  {
    try
    {
      this.pNotifier   = pNotifier;
      this.stStruct    = stStruct;
      this.stFiles     = stFiles;
      this.stHeader    = stHeader;
      this.stLAS       = stLAS;
      this.stBrine     = stBrine;
      this.stControl   = stControl;

      if (stStruct.bLAS != null)
      {
        this.bLog      = stStruct.bLAS;
	  }

      loadData(); // Initialize the LAS 3 Panel Arrays

      // Create the LAS 3 File Frame

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Method loadData()
   * <p> This method will load the LAS 3 Data
   */

  private void loadData()
  {
    int i, j, k;
    int            iCont    = 1;
    int            iType    = -1;
    String         sMethod  = "";
    String         sFile    = "";
    String         sDir     = "";
    las3ListStruct st       = null;
    las3Read       p3       = null;
    las3Struct     st3Temp  = null; // LAS 3 DST Data Structure
    int            iLAS[]   = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};

    iArray = new int[las3.las3Constants._TOTAL];
    for (int iq=0; iq<las3.las3Constants._TOTAL; iq++)
    {
	  iArray[iq] = -1;
	}

    // Read each LAS File

    if (stFiles != null)
    {
	  for (i=0; i<stFiles.iCount; i++)
	  {
	    if (stFiles.stItem[i] != null)
	    {
          sMethod = new String( "FILE" );
		  if (stFiles.stItem[i].iType == util.utilFileIO.URL_FILE)
		    sMethod = new String( "URL" );

		  sFile = new String( stFiles.stItem[i].sFilename );
		  sDir  = new String( stFiles.stItem[i].sDirectory );

          stStruct.sDirectory = new String( stFiles.stItem[i].sDirectory );
          stStruct.sName      = new String( stFiles.stItem[i].sFilename );

          p3 = new las3Read( notifier, stStruct, las3.io.las3Read._LOAD_LAS3, bLog );
          p3.setPath( sMethod, sDir, sFile );
          st = las3.las3Utility.copyList( p3.getLAS3Data() );

          iLAS = p3.getLAS();

          if (p3 != null)
            p3.delete();
          p3 = null;

          if (st != null)
          {
			if (st.iCount > 0)
			{
              for (j=0; j<st.iCount; j++)
              {
			    if (st.stItem[j] != null)
			    {
                  iType                   = st.stItem[j].iType;
                  iArray[iType]           = iArray[iType]+1;
                  st.stItem[j].iArrayP    = iArray[iType];

                  st.stItem[j].sWrapP     = new String( st.sWrap );
                  st.stItem[j].sDelimP    = new String( st.sDelim );

                  st.stItem[j].iTypeP     = stFiles.stItem[i].iType;
                  st.stItem[j].sDirectory = new String( sDir );
                  st.stItem[j].sFilename  = new String( sFile );
                  st.stItem[j].dVersion   = st.dVersion;

                  iCont = 1;
                  if ((i > 0) && (st.stItem[j].iType == las3.las3Constants._WELL))
                    iCont =  0;

                  if (iCont == 1)
                  {
                    stLAS3 = las3.las3Utility.add( st.stItem[j], stLAS3 );
			      }
                }
	          }
	        }
          }

          if (st != null)
            st.delete();
          st = null;
		}
	  }
	}

    // IF the LAS File Data Structure does not exist THEN
    //   Create a LAS File Data Structure
    //   Load the LAS 3.0 Data Structure for this data
    // END IF

    if (stLAS == null)
    {
      stLAS            = new lasFileDataStruct();
      stLAS.depthStart = stStruct.depthStart;
      stLAS.depthEnd   = stStruct.depthEnd;
      stLAS.depthStep  = stStruct.depthStep;
      stLAS.sENG       = new String( stHeader.source );
      stLAS            = las3.las3LoadLogData.normalizeDepth( stHeader, stLAS );

      st3Log = las3.las3LoadLogData.setData( stLAS, dNull );
      if (st3Log != null)
      {
        st3Log.iArray = iArray[las3.las3Constants._LOG];
        iArray[las3.las3Constants._LOG]++;
      }
    }

    if (stLAS != null)
      dNull = stLAS.dNull;

    // Initialize the Array of Data Sections

    iArray = new int[las3.las3Constants._TOTAL];
    for (i=0; i<las3.las3Constants._TOTAL; i++)
    {
      iArray[i] = 0;
    }

    iArray[las3.las3Constants._VERSION] = 1;
    iArray[las3.las3Constants._WELL]    = 1;

    // Count the Data Sections

    if (stLAS3 != null)
    {
      dVersion = stLAS3.dVersion;

      for (i=0; i<stLAS3.iCount; i++)
      {
//System.out.println(i+" "+las3.las3Constants.SECTIONS[stLAS3.stItem[i].iType] + " " + stLAS3.stItem[i].iqstrat);
        switch (stLAS3.stItem[i].iType)
        {
          case las3.las3Constants._LOG:  // Log Data Section
            iArray[las3.las3Constants._LOG]++;
            if (stLAS.iRows > 0)
              stHeader.iLAS++;
            break;
          case las3.las3Constants._CORE:  // Core Data Section
            iArray[las3.las3Constants._CORE]++;
            stHeader.iCore++;
            break;
          case las3.las3Constants._DRILLING:  // Drilling Data Section
            iArray[las3.las3Constants._DRILLING]++;
            break;
          case las3.las3Constants._INCLINOMETRY:  // Inclinometry Data Section
            iArray[las3.las3Constants._INCLINOMETRY]++;
            break;
          case las3.las3Constants._TOPS:  // Tops Data Section
            iArray[las3.las3Constants._TOPS]++;
            stHeader.iTops++;
            break;
          case las3.las3Constants._TEST:  // Test Data Section
            iCont = 1;
            if (stDST != null)
            {
              if (stDST.iCount > 0)
                iCont = 0;
			}

            if (iCont == 1)
            {
              iArray[las3.las3Constants._TEST]++;
              stHeader.iDST++;
		    }
            break;
          case las3.las3Constants._PERFORATION:  // Perforation Data Section
            iArray[las3.las3Constants._PERFORATION]++;
            stHeader.iPerf++;
            break;

          // IQ Data Types

          case las3.las3Constants._IQ_BIO:        // Bio Stratigraphy
            iArray[las3.las3Constants._IQ_BIO]++;
            break;
          case las3.las3Constants._IQ_ENV:        // Depositional Environment
            iArray[las3.las3Constants._IQ_ENV]++;
            break;
          case las3.las3Constants._IQ_SEQ_STRAT:  // Sequence Stratigraphy
            iArray[las3.las3Constants._IQ_SEQ_STRAT]++;
            break;
          case las3.las3Constants._IQ_FLOW:       // PfEFFER Flow Units Data Section
            iArray[las3.las3Constants._IQ_FLOW]++;
            stHeader.iPfeffer++;
            break;
          case las3.las3Constants._IQ_PFEFFER:    // PfEFFER Table Data Section
            iArray[las3.las3Constants._IQ_PFEFFER]++;
            break;
          case las3.las3Constants._IQ_GEO_REPORT: // Remarks/Comments/Notes
            iArray[las3.las3Constants._IQ_GEO_REPORT]++;
            stHeader.iGeo++;
            break;
          case las3.las3Constants._IQ_SHALE:      // Shale Gamma Ray Limits
            iArray[las3.las3Constants._IQ_SHALE]++;
            break;
          case las3.las3Constants._IQ_IMAGES:     // Rock Images
            iArray[las3.las3Constants._IQ_IMAGES]++;
            stHeader.images++;
            break;
          case las3.las3Constants._IQ_BRINE:      // Brine Data
            iArray[las3.las3Constants._IQ_BRINE]++;
            break;
        }
      }
    }

    setData();

    // IF the Plot Control LAS 3 Exists THEN
    //   Get LAS 3 Data Structure for Profile Plot Control

    if (stControl != null)
    {
      st3Control = las3.las3LoadControlData.setData( stControl, dNull );

      if (st3Control != null)
      {
        st3Control.iArray  = iArray[las3.las3Constants._IQ_CONTROL];
        st3Control.iArrayP = iArray[las3.las3Constants._IQ_CONTROL];
        iArray[las3.las3Constants._IQ_CONTROL]++;
      }
    }

    // IF the Shale Data List Structure Exists THEN
    //   Get LAS 3 Data Structure for the Shale Data

    if (stShale != null)
    {
      st3Shale = las3.las3LoadShaleData.setData( stShale, dNull );
      if (st3Shale != null)
      {
        st3Shale.iArray  = iArray[las3.las3Constants._IQ_SHALE];
        st3Shale.iArrayP = iArray[las3.las3Constants._IQ_SHALE];
        iArray[las3.las3Constants._IQ_SHALE]++;
	  }
    }

    // IF the Rock Data List Structure Exists and this is NOT version 3.0 THEN
    //   Get LAS 3 Data Structure for the Rock Data

//    if ((stRock != null) && (dVersion != 3.0))
    if (stRock != null)
    {
//	  if (stRock.iSource != las3.las3Constants._SOURCE_LAS)
	  {
        st3Rock = las3.las3LoadCoreData.setData( stRock, dNull );
        if (st3Rock != null)
        {
          st3Rock.iArrayP = iArray[las3.las3Constants._CORE];
          iArray[las3.las3Constants._CORE]++;
          stHeader.iCore++;
	    }
      }
    }

    // IF the Profile Brine Data Exists THEN
    //   Get LAS 3 Data Structure for Profile Brine Data

    if (stBrine != null)
    {
      st3Brine = las3.las3LoadBrineData.setData( stBrine, dNull );
      if (st3Brine != null)
      {
        st3Brine.iArray  = iArray[las3.las3Constants._IQ_BRINE];
        st3Brine.iArrayP = iArray[las3.las3Constants._IQ_BRINE];
        iArray[las3.las3Constants._IQ_BRINE]++;
      }
    }

    // IF the Profile Stratigraphic Units Exists THEN
    //   Get LAS 3 Data Structure for Profile Stratigraphic Units

    if (stTops != null)
    {
      st3Tops = las3.las3LoadTopsData.setData( stTops, dNull );
      if (st3Tops != null)
      {
        st3Tops.iArray  = iArray[las3.las3Constants._TOPS];
        st3Tops.iArrayP = iArray[las3.las3Constants._TOPS];
        iArray[las3.las3Constants._TOPS]++;
        stHeader.iTops++;
      }
    }

    // IF the Profile Sequence Stratigraphy Units Exists THEN
    //   Get LAS 3 Data Structure for Profile Sequence Stratigraphy Units

    if (stSeq != null)
    {
      st3Seq = las3.las3LoadSeqData.setData( stSeq, dNull );
      if (st3Seq != null)
      {
        st3Seq.iArray  = iArray[las3.las3Constants._IQ_SEQ_STRAT];
        st3Seq.iArrayP = iArray[las3.las3Constants._IQ_SEQ_STRAT];
        iArray[las3.las3Constants._IQ_SEQ_STRAT]++;
      }
    }

    // IF the Profile Depositonal Environment Units Exists THEN
    //   Get LAS 3 Data Structure for Profile epositonal Environment Units

    if (stEnv != null)
    {
      st3Env = las3.las3LoadEnvData.setData( stEnv, dNull );
      if (st3Env != null)
      {
        st3Env.iArray  = iArray[las3.las3Constants._IQ_ENV];
        st3Env.iArrayP = iArray[las3.las3Constants._IQ_ENV];
        iArray[las3.las3Constants._IQ_ENV]++;
      }
    }

    // IF the Perforation Data List Structure Exists THEN
    //   Get LAS 3 Data Structure for the Perforation Data

    if (stPerf != null)
    {
      st3Perf = las3.las3LoadPerfData.setData( stPerf, dNull );
      if (st3Perf != null)
      {
        st3Perf.iArray  = iArray[las3.las3Constants._PERFORATION];
        st3Perf.iArrayP = iArray[las3.las3Constants._PERFORATION];
        iArray[las3.las3Constants._PERFORATION]++;
        stHeader.iPerf++;
	  }
    }

    // IF the Bio Stratigraphy Units Exists THEN
    //   Get LAS 3 Data Structure for Bio Stratigraphy Units

    if (stBio != null)
    {
      st3Bio = las3.las3LoadBioData.setData( stBio, dNull );
      if (st3Bio != null)
      {
        st3Bio.iArray  = iArray[las3.las3Constants._IQ_BIO];
        st3Bio.iArrayP = iArray[las3.las3Constants._IQ_BIO];
        iArray[las3.las3Constants._IQ_BIO]++;
      }
    }

    // IF the DST Data Exists THEN
    //   Get LAS 3 Data Structure for DST Data

    if (stDST != null)
    {
	  for (k=0; k<stDST.iCount; k++)
	  {
        st3Temp = las3.las3LoadDSTData.setData( stDST.stItem[k], dNull );
        if (st3Temp != null)
        {
          st3Temp.iArray  = iArray[las3.las3Constants._TEST];
          st3Temp.iArrayP = iArray[las3.las3Constants._TEST];
          iArray[las3.las3Constants._TEST]++;

          st3DST = las3.las3Utility.add( st3Temp, st3DST );
          stHeader.iDST++;

          st3Temp.delete();
          st3Temp = null;
        }
      }
    }

    // IF the PfEFFER Data List Structure Exists THEN
    //   Get LAS 3 Data Structure for the PfEFFER Data
    if (stPfeffer != null)
    {
      st3Flow = las3.las3LoadPfefferData.setFlowData(
                    stPfeffer, dNull, stStruct.stSymbols );
      if (st3Flow != null)
      {
        st3Flow.iArray  = iArray[las3.las3Constants._IQ_FLOW];
        st3Flow.iArrayP = iArray[las3.las3Constants._IQ_FLOW];
        iArray[las3.las3Constants._IQ_FLOW]++;
        stHeader.iPfeffer++;
      }

      st3Pfeffer = las3.las3LoadPfefferData.setPfefferData( stPfeffer, dNull );
      if (st3Pfeffer != null)
      {
        st3Pfeffer.iArray  = iArray[las3.las3Constants._IQ_PFEFFER];
        st3Pfeffer.iArrayP = iArray[las3.las3Constants._IQ_PFEFFER];
        iArray[las3.las3Constants._IQ_PFEFFER]++;
      }
    }

    // IF the Remarks Data List Structure Exists THEN
    //   Get LAS 3 Data Structure for the Remarks Data

    if (stRemarks != null)
    {
      st3Remarks = las3.las3LoadRemarksData.setData( stRemarks, dNull );
      if (st3Remarks != null)
      {
        st3Remarks.iArray  = iArray[las3.las3Constants._IQ_GEO_REPORT];
        st3Remarks.iArrayP = iArray[las3.las3Constants._IQ_GEO_REPORT];
        iArray[las3.las3Constants._IQ_GEO_REPORT]++;
        stHeader.iGeo++;
	  }
    }


    // IF the Rock Images Data List Structure Exists and this is NOT version 3.0 THEN
    //   Get LAS 3 Data Structure for the Rock Images

    if (stImages != null)
    {
      st3Images = las3.las3LoadImageData.setData( stImages, dNull );
      if (st3Images != null)
      {
        st3Images.iArray  = iArray[las3.las3Constants._IQ_IMAGES];
        st3Images.iArrayP = iArray[las3.las3Constants._IQ_IMAGES];
        iArray[las3.las3Constants._IQ_IMAGES]++;
        stHeader.images++;
      }
    }

    stLAS3 = las3.las3Utility.bubbleSort( stLAS3 );

    for (i=2; i<las3.las3Constants._TOTAL; i++)
    {
      iPnl = iPnl + iArray[i];
    }

//    if (st != null)
//      st.delete();
//    st = null;
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int    i,j,k;
    int    icnt = 0;
    String sTab = "";
    int    rows = 0;
    int    cols = 0;

    // Frame Menu

    JMenuBar menuBar    = new JMenuBar();
    JMenu    menuFile   = new JMenu("File");

    mSave = new JMenuItem("Save as LAS Version 3.0 Output");
    mSave.addActionListener(this);

    mExit = new JMenuItem("Exit");
    mExit.addActionListener(this);

    menuFile.add( mSave );
    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Create an Observable

    notifier = new las3FileFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Create Frame

    this.setTitle("LAS File Data Types");
    this.getContentPane().setLayout(new BorderLayout());
    tabbedPane.setFont(new java.awt.Font("Dialog", 1, 11));

    pWell = new las3FileWellPanel( notifier );
    pWell.setData( stHeader, stLAS );

    this.getContentPane().add(tabbedPane, BorderLayout.CENTER);
    pWell.setName("~WELL");
    tabbedPane.add( pWell, 0 );

    if (iPnl > 0)
    {
      pnl = new las3FilePanel[iPnl];

      if (stLAS3 != null)
      {
        for (i=1; i<stLAS3.iCount; i++)
        {
          if (icnt < iPnl)
          {
            if ((dVersion != 3.0) && (stLAS3.stItem[i].sType.equals("Log")))
            {
              stLAS3.stItem[i].iParamRows = las3.las3Parameters._LOG_TOTAL;
              stLAS3.stItem[i].sParams =
                new String[las3.las3Parameters._LOG_TOTAL][las3.las3Struct._TOTAL];
              stLAS3.stItem[i].sParams = PARAM;
            }

            if (stLAS3.stItem[i].sType.equals("Log"))
            {
              if (stLAS != null)
              {
                if (stLAS.iLogs > 0)
                {
                  if (stLAS.stLAS3[stLAS3.stItem[i].iArrayP] != null)
                  {
                    stLAS3.stItem[i] = las3.las3Utility.copyCurves(
                        stLAS.stLAS3[stLAS3.stItem[i].iArrayP], stLAS3.stItem[i] );
                  }
                }
              }
            }

            if (stLAS3.stItem[i].sType.equals("Core"))
            {
              if (stRock != null)
              {
                if (stRock.iLogs > 0)
                {
                  if (stRock.stLAS3[stLAS3.stItem[i].iArrayP] != null)
                  {
                    stLAS3.stItem[i] = las3.las3Utility.copyCurves(
                        stRock.stLAS3[stLAS3.stItem[i].iArrayP], stLAS3.stItem[i] );
                  }
                }
              }
            }

            pnl[icnt] = new las3FilePanel( notifier,  stLAS3.stItem[i], dVersion);
            sTab = new String( "~" + stLAS3.stItem[i].sType );
            if (stLAS3.stItem[i].iArrayP > 0)
              sTab = new String( "~" + stLAS3.stItem[i].sType +
                                 "[" + stLAS3.stItem[i].iArrayP + "]" );
            pnl[icnt].setName(sTab);
            tabbedPane.add( pnl[icnt], icnt+1 );
            icnt++;
          }
        }
      }
    }

    if (st3Log != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Log, dVersion );

        sTab = new String( "~" + st3Log.sType );
        if ( st3Log.iArray > 0 )
          sTab = new String( "~" + st3Log.sType + "[" + st3Log.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Rock != null)
    {
      if (icnt < iPnl)
      {
        st3Rock.iArray = iArray[las3.las3Constants._CORE-1];
        pnl[icnt] = new las3FilePanel( notifier, st3Rock, dVersion );

        sTab = new String( "~" + st3Rock.sType );
        if ( st3Rock.iArray > 0 )
          sTab = new String( "~" + st3Rock.sType + "[" + st3Rock.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Brine != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Brine, dVersion );

        sTab = new String( "~" + st3Brine.sType );
        if ( st3Brine.iArray > 0 )
          sTab = new String( "~" + st3Brine.sType +
                             "[" + st3Brine.iArray + "]" );
        pnl[icnt].setName( sTab );
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Tops != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Tops, dVersion );

        sTab = new String( "~" + st3Tops.sType );
        if ( st3Tops.iArray > 0 )
          sTab = new String( "~" + st3Tops.sType + "[" + st3Tops.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Seq != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Seq, dVersion );

        sTab = new String( "~" + st3Seq.sType );
        if ( st3Seq.iArray > 0 )
          sTab = new String( "~" + st3Seq.sType + "[" + st3Seq.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Env != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Env, dVersion );

        sTab = new String( "~" + st3Env.sType );
        if ( st3Env.iArray > 0 )
          sTab = new String( "~" + st3Env.sType + "[" + st3Env.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Perf != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Perf, dVersion );

        sTab = new String( "~" + st3Perf.sType );
        if ( st3Perf.iArray > 0 )
          sTab = new String( "~" + st3Perf.sType + "[" + st3Perf.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Bio != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Bio, dVersion );

        sTab = new String( "~" + st3Bio.sType );
        if ( st3Bio.iArray > 0 )
          sTab = new String( "~" + st3Bio.sType + "[" + st3Bio.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3DST != null)
    {
	  for (k=0; k<st3DST.iCount; k++)
	  {
        if (icnt < iPnl)
        {
          pnl[icnt] = new las3FilePanel( notifier, st3DST.stItem[k], dVersion );

          sTab = new String( "~" + st3DST.stItem[k].sType );
          if ( st3DST.stItem[k].iArray > 0 )
            sTab = new String( "~" + st3DST.stItem[k].sType +
                               "[" + st3DST.stItem[k].iArray + "]" );
          pnl[icnt].setName(sTab);
          tabbedPane.add( pnl[icnt], icnt+1 );
          icnt++;
	    }
      }
    }

    if (st3Control != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Control, dVersion );

        sTab = new String( "~" + st3Control.sType );
        if ( st3Control.iArray > 0 )
          sTab = new String( "~" + st3Control.sType +
                             "[" + st3Control.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Shale != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Shale, dVersion );

        sTab = new String( "~" + st3Shale.sType );
        if ( st3Shale.iArray > 0 )
          sTab = new String( "~" + st3Shale.sType +
                             "[" + st3Shale.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Flow != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Flow, dVersion );

        sTab = new String( "~" + st3Flow.sType );
        if ( st3Flow.iArray > 0 )
          sTab = new String( "~" + st3Flow.sType +
                             "[" + st3Flow.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Pfeffer != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Pfeffer, dVersion );

        sTab = new String( "~" + st3Pfeffer.sType );
        if ( st3Pfeffer.iArray > 0 )
          sTab = new String( "~" + st3Pfeffer.sType +
                             "[" + st3Pfeffer.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Remarks != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Remarks, dVersion );

        sTab = new String( "~" + st3Remarks.sType );
        if ( st3Remarks.iArray > 0 )
          sTab = new String( "~" + st3Remarks.sType +
                             "[" + st3Remarks.iArray + "]" );
        pnl[icnt].setName(sTab);
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    if (st3Images != null)
    {
      if (icnt < iPnl)
      {
        pnl[icnt] = new las3FilePanel( notifier, st3Images, dVersion );

        sTab = new String( "~" + st3Images.sType );
        if ( st3Images.iArray > 0 )
          sTab = new String( "~" + st3Images.sType +
                             "[" + st3Images.iArray + "]" );
        pnl[icnt].setName( sTab );
        tabbedPane.add( pnl[icnt], icnt+1 );
        icnt++;
      }
    }

    // Display the Frame

    this.setSize(new Dimension(680, 700));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getHeaders()
   * <p> This method will return the Well Headers Information data structure
   * @return stHeader = the Well Headers Information data structure
   */

  public iqstratHeadersStruct getHeaders() { return (stHeader); }

  /** Method getLAS()
   * <p> This method will return the LAS File Dependant Variables
   * @return stLAS = the LAS File Data Structure
   */

  public lasFileDataStruct getLAS() { return (stLAS); }

  /** Method getLAS3()
   * <p> This method will return the LAS 3 Data List Structure
   * @return stLAS3 = the LAS 3 Data List Structure
   */

  public las3ListStruct getLAS3() { return (stLAS3); }

  /** Method getLAS3Data()
   * <p> This method will build the LAS 3 Data List Structure from the LAS 3
   *     panels
   * @return st = the LAS 3 Data List Structure
   */

  public las3ListStruct getLAS3Data()
  {
    las3ListStruct st = new las3ListStruct();

    st.dVersion = 3.0;  // Version Number of LAS File
    st.sWrap    = "No"; // Multiple lines per row
    st.sDelim   = ",";  // Delimiting Character between data

    if (stLAS3 != null)
    {
      st.dVersion = stLAS3.dVersion;              // Version Number of LAS File
      st.sWrap    = new String( stLAS3.sWrap );   // Multiple lines per row
      st.sDelim   = new String( stLAS3.sDelim );  // Delimiting Character
    }

    if (pWell != null)
    {
      st = las3.las3Utility.add( pWell.getLAS3(), st );
    }

    if (iPnl > 0)
    {
      for (int i=0; i<iPnl; i++)
      {
        if (pnl[i] != null)
        {
          st = las3.las3Utility.add( pnl[i].getData(), st );
        }
      }
    }

    st = las3.las3Utility.bubbleSort(st);

    return (st);
  }

  /*========================================================================*
   * ---------------------------- SET METHODS ------------------------------*
   *========================================================================*/

  /** Method setData()
   * <p> This method will set the data
   */

  private void setData()
  {
    String sDate[] = { "", "" };

    if (stLAS != null)
    {
      for (int i=0; i<las3.las3Parameters._LOG_TOTAL; i++)
      {
        switch (i)
        {
          case las3.las3Parameters._PDAT:  // Permanent Data
            PARAM[i][_VALUE] = new String( stLAS.sPDAT );
            break;
          case las3.las3Parameters._APD:   // Above Permanent Data
            PARAM[i][_VALUE] = new String( "" + stLAS.dAPD );
            break;
          case las3.las3Parameters._DREF:  // Depth Reference (KB,DF,CB)
            PARAM[i][_VALUE] = new String( stLAS.sDREF );
            break;
          case las3.las3Parameters._EREF:  // "Elevation of Depth Reference"
            PARAM[i][_VALUE] = new String( "" + stLAS.dElevation );
            break;
          case las3.las3Parameters._RUN:
            break;

          case las3.las3Parameters._TDL:  // Loggers Depth
            if (stLAS.dTLD > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dTLD );
            break;
          case las3.las3Parameters._TDD:  // Drillers Depth
            if (stLAS.dTDD > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dTDD );
            break;

          case las3.las3Parameters._CSGL:  // Casing Bottom Logger
            if (stLAS.dCSGL > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dCSGL );
            if (stLAS.sCSGL.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sCSGL );
            break;
          case las3.las3Parameters._CSGD:  // Casing Bottom Driller
            if (stLAS.dCSGD > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dCSGD );
            if (stLAS.sCSGD.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sCSGD );
            break;
          case las3.las3Parameters._CSGS:  // Casing Size
            if (stLAS.dCSGS > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dCSGS );
            if (stLAS.sCSGS.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sCSGS );
            break;
          case las3.las3Parameters._CSGW:  // Casing Weight
            if (stLAS.dCSGW > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dCSGW );
            if (stLAS.sCSGW.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sCSGW );
            break;
          case las3.las3Parameters._BS:    // Bit Size
            if (stLAS.dBS > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dBS );
            if (stLAS.sBS.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sBS );
            break;

          case las3.las3Parameters._MUD:
            if (stLAS.sMUD.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sMUD );
            break;
          case las3.las3Parameters._MUDS:
            if (stLAS.sMUDS.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sMUDS );
            break;
          case las3.las3Parameters._MUDD:
            if (stLAS.dMUDD > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dMUDD );
            if (stLAS.sMUDD.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sMUDD );
            break;
          case las3.las3Parameters._MUDV:
            if (stLAS.dMUDV > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dMUDV );
            if (stLAS.sMUDV.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sMUDV );
            break;

          case las3.las3Parameters._FL:
            if (stLAS.dFL > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dFL );
            if (stLAS.sFL.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sFL );
            break;
          case las3.las3Parameters._PH:
            if (stLAS.dPH > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dPH );
            break;

          case las3.las3Parameters._RM:
            if (stLAS.dRM > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRM );
            if (stLAS.sRM.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRM );
            break;
          case las3.las3Parameters._RMT:
            if (stLAS.dRMT > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRMT );
            if (stLAS.sRMT.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRMT );
            break;
          case las3.las3Parameters._RMF:
            if (stLAS.dRMF > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRMF );
            if (stLAS.sRMF.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRMF );
            break;
          case las3.las3Parameters._RMFT:
            if (stLAS.dRMFT > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRMFT );
            if (stLAS.sRMFT.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRMFT );
            break;
          case las3.las3Parameters._RMC:
            if (stLAS.dRMC > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRMC );
            if (stLAS.sRMC.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRMC );
            break;
          case las3.las3Parameters._RMCT:
            if (stLAS.dRMCT > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dRMCT );
            if (stLAS.sRMCT.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sRMCT );
            break;
          case las3.las3Parameters._TMAX:
            if (stLAS.dTMAX > 0.0)
              PARAM[i][_VALUE] = new String( "" + stLAS.dTMAX );
            if (stLAS.sTMAX.length() > 0)
              PARAM[i][_UNIT]  = new String( stLAS.sTMAX );
            break;

          case las3.las3Parameters._TIMC:
            PARAM[i][_FORMAT] = new String( "D/M/YYY" );
            if (stLAS.sTIMC.length() > 0)
            {
              PARAM[i][_VALUE] = new String( stLAS.sTIMC );
              sDate = cmn.cmnString.convertDate( stLAS.sTIMC );
              PARAM[i][_VALUE]  = new String( sDate[0] );
              PARAM[i][_FORMAT] = new String( sDate[1] );
            }
            break;
          case las3.las3Parameters._TIML:
            PARAM[i][_FORMAT] = new String( "D/M/YYY" );
            if (stLAS.sTIML.length() > 0)
            {
              PARAM[i][_VALUE] = new String( stLAS.sTIML );
              sDate = cmn.cmnString.convertDate( stLAS.sTIML );
              PARAM[i][_VALUE]  = new String( sDate[0] );
              PARAM[i][_FORMAT] = new String( sDate[1] );
            }
            break;
          case las3.las3Parameters._T_UNIT:
            if (stLAS.sUNIT.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sUNIT );
            break;
          case las3.las3Parameters._BASE:
            if (stLAS.sBASE.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sBASE );
            break;
          case las3.las3Parameters._ENG:
            if (stLAS.sENG.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sENG );
            break;
          case las3.las3Parameters._WIT:
            if (stLAS.sWIT.length() > 0)
              PARAM[i][_VALUE] = new String( stLAS.sWIT );
            break;
        }
      }
    }
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method find()
   * <p> This method will allow the user to search the PC to place the LAS File
   */

  private void find()
  {
    if (pFind != null)
      pFind.close();
    pFind = null;

    sKEY  = new String( iqstrat.iqstratHeadersUtility.getKEY( stHeader ) );

    pFind = new geminiFindPath( notifier );
    pFind.setKEY(sKEY);
  }

    /** Method save()
     * <p> This method will save the data as a LAS version 3.0 file
     */

  public void save()
  {
    las3ListStruct st      = null;
    las3WriteFile  p       = null;
    String         sPath   = "";
    String         sFile   = "";
    boolean        bGEMINI = false;

    if (pFind != null)
    {
      sPath   = new String( pFind.getDirectory() );
      sFile   = new String( pFind.getFilename() );
      bGEMINI = pFind.getGEMINI();
//System.out.println(bGEMINI);

      pFind.close();
      pFind = null;
    }

    st = getLAS3Data();
    if (st != null)
    {
	  // IF the data is saved to GEMINI Data Directory THEN

	  if (bGEMINI)
	  {
		// ADD User Primary KEY to Header Data Structure

        stHeader.sKEY = new String(sKEY);

        // Add this well to the Gemini Data Well Headers List XML File

		gemini.io.geminiIOHeaders.save( stHeader );
	  }

      // Save Well Data as a LAS 3.0 File

      p = new las3WriteFile( st, stHeader, stLAS );

      p.write( sPath, sFile );
      p.delete();
      p = null;
    }

    close();
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
	// Input Variables

    pNotifier   = null; // Top Observables Object

    stStruct    = null; // Global IQSTRAT Data Structure

    stFiles     = null; // LAS Files Information
    stLAS3      = null; // LAS 3 Well Parameters
    stHeader    = null; // Well Header Information
    stLAS       = null; // LAS File Information
    stRock      = null; // Rock Data List Structure
    stBrine     = null; // Brine Data List Structure
    stImages    = null; // Rock Images Data List Structure
    stTops      = null; // Stratigraphic List Structure
    stSeq       = null; // Sequence Stratigrapy Data List Structure
    stPerf      = null; // Perforation List Structure
    stBio       = null; // Bio-Stratigraphic List Structure
    stEnv       = null; // Depositonal Environment List Structure

    stDST       = null; // DST Data List Structure

    stControl   = null; // Control Plot Data Structure
    stShale     = null; // Shale Data List Structure
    stPfeffer   = null; // Global IQSTRAT Data Structure

    stRemarks   = null; // Remarks List Structure

    // Global Variables

    notifier    = null; // Observables Object

    if (pFind != null)
      pFind.close();
    pFind = null;

    if (pWell != null)
      pWell.close();
    pWell    = null; // Well Information Panel

    st3Log       = null; // LAS 3 Log Data Structure
    st3Control   = null; // LAS 3 Control Data Structure
    st3Rock      = null; // LAS 3 Rock Data Structure
    st3Brine     = null; // LAS 3 Brine Data Structure
    st3Images    = null; // LAS 3 Rock Images Data Structure
    st3Tops      = null; // LAS 3 Tops Data Structure
    st3Seq       = null; // LAS 3 Sequence Strat Data Structure
    st3Env       = null; // LAS 3 Environment Data Structure
    st3Perf      = null; // LAS 3 Perforation Data Structure
    st3Bio       = null; // LAS 3 Bio-Stratigraphic Data Structure
    st3Flow      = null; // LAS 3 PfEFFER Flow Units Data Struct
    st3DST       = null; // LAS 3 DST Data List Structure
    st3Pfeffer   = null; // LAS 3 PfEFFER Data Structure
    st3Remarks   = null; // LAS 3 Remarks Data Structure
    st3Shale     = null; // LAS 3 Shale Data Structure

    iArray       = null;    // Array of data section counts

    sKEY         = null;

    if (iPnl > 0)
    {
      for (int i=0; i<iPnl; i++)
      {
        if (pnl[i] != null)
          pnl[i].close();
        pnl[i] = null; // Array of Data Sections
      }
      iPnl = 0; // Total Number of Panel Types
    }

    // Global Frame Widgets

    tabbedPane = null;
    mSave      = null;  // Save as LAS Version 3.0 Output
    mExit      = null;  // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == mSave) { find();  }
    if (event.getSource() == mExit) { close();  }
  }

  /* ===================================================================== *
   * ------------------------ OBSERVER ACTIONS --------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("LAS - WELL Information"))
    {
      if (pWell != null)
      {
        stLAS  = pWell.getLAS();
      }

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("LAS - WELL Information"));
    }

    if (sArg.equals("HEADER - WELL Information"))
    {
      if (pWell != null)
      {
        stHeader = pWell.getHeaders();
      }

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("HEADER - WELL Information"));
    }

    if (sArg.equals("Get Directory & Filename"))
    {
      save();
    }
  }
}

/** CLASS las3FileFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class las3FileFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 01/26/2010
 *  @author  John Victorine
 */
