/*
 * @las3CurvesLogPanel.java Version 1.1 01/27/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import cmn.cmnStruct;                     // Global Common Data Structure
import cmn.cmnLASCurveListStruct;         // Common LAS Curve Data List Structure
import cmn.cmnLASCurveStruct;             // Common LAS Curve Data Structure

import iqstrat.iqstratStruct;             // Global IQSTRAT Data Structure
import iqstrat.iqstratControlStruct;      // Plot Control Data Structure
import iqstrat.iqstratRemarkListStruct;   // Remarks Data List Structure
import iqstrat.iqstratShaleListStruct;    // Shale Data List Structure

import las.lasFileDataStruct;             // LAS File Data Structure

import las3.las3Struct;                   // LAS 3 Data Sturcture
import las3.gui.las3ToolsFrame;           // LAS Tools Frame
import las3.io.las3ReadFile;              // Open, Read & Parse LAS Files

import horizon.strat.stratListStruct;     // Stratigraphic Units List Structure
import horizon.seq.seqListStruct;         // Sequence Stratigraphy List Struct
import horizon.regions.regionsListStruct; // Perforations Data List Structure
import horizon.bio.bioStratListStruct;    // Bio-Stratigraphic Units List Structure
import horizon.env.envListStruct;         // Depositional Environment List Struct

import pfeffer.pfefferDataListStruct;     // PfEFFER Data List Structure

import rock.rockDataListStruct;           // Rock Data List Structure
import rock.rockImagesListStruct;         // Rock Images List Structure

import brine.brineListStruct;             // Brine Data List Structure

import dst.dstStruct;                     // DST Data Structure

/** Class las3CurvesLogPanel
 *  <p> This Class will display the Curves sections of the LAS 3 File and allow
 *      the user to select the curves they wish to display on the KGS web apps
 *      and to identify unknown curves with KGS Standard Curves.
 *
 *  @version 1.1 01/27/2010
 *  @author  John R. Victorine
 */

public class las3CurvesLogPanel extends JPanel implements ActionListener
{
  // Input Variables

  private Observable        notifier = null;  // Observables Object
  private iqstratStruct     stStruct = null;  // Global IQSTRAT Data Structure
  private cmnStruct         stCMN    = null;  // Common Data Structure
  private las3Struct        stLAS3   = null;  // LAS 3 File Data Structure
  private lasFileDataStruct stLAS    = null;  // LAS File Data Structure
  private double            dVersion = 0.0;   // Version Number of LAS File
  private int               iqstrato = -1;    // Profile Saved Data with type

  // Global Variables

  private static final int OFF  = cmn.cmnStruct.OFF;
  private static final int ON   = cmn.cmnStruct.ON;

  private int    iAction = ON;

  private int    iCurve = -1;

  private int    iArray = 0;                       // Array Number
  private int    iType  = las3.las3Constants._NONE; // Type of Data Identifier
  private String sType  = "";                      // Type of Data Name Set
  private String sLabel = "";

  private int iTotal      = 0;    // Total Number of Log Curves

  private int iStandard   = 0;    // Total Number Standard Curves
  private int iSelected[] = null; // Array of Selected Standard Curves

  private cmnLASCurveListStruct stCurves = null; // LAS File Curves
  private rockDataListStruct    stRock   = null; // Rock Data List Structure
  private las3ToolsFrame         pTools   = null; // Standard Tools List Frame

  public static final int _IGNORE        = 0;
  public static final int _CHECK         = 1;

  public static final int _LAS_TYPE      = las3.gui.las3ToolsTable._LAS_TYPE;
  public static final int _CORE_TYPE     = las3.gui.las3ToolsTable._CORE_TYPE;
  public static final int _TOPS_TYPE     = las3.gui.las3ToolsTable._TOPS_TYPE;
  public static final int _PERF_TYPE     = las3.gui.las3ToolsTable._PERF_TYPE;
  public static final int _CNTRL_TYPE    = las3.gui.las3ToolsTable._CNTRL_TYPE;
  public static final int _SEQ_TYPE      = las3.gui.las3ToolsTable._SEQ_TYPE;
  public static final int _FLOW_TYPE     = las3.gui.las3ToolsTable._FLOW_TYPE;
  public static final int _GEO_TYPE      = las3.gui.las3ToolsTable._GEO_TYPE;
  public static final int _SHALE_TYPE    = las3.gui.las3ToolsTable._SHALE_TYPE;
  public static final int _IMAGES_TYPE   = las3.gui.las3ToolsTable._IMAGES_TYPE;
  public static final int _BRINE_TYPE    = las3.gui.las3ToolsTable._BRINE_TYPE;
  public static final int _BIO_TYPE      = las3.gui.las3ToolsTable._BIO_TYPE;
  public static final int _ENV_TYPE      = las3.gui.las3ToolsTable._ENV_TYPE;
  public static final int _DST_TYPE      = las3.gui.las3ToolsTable._DST_TYPE;

  private int iData = _LAS_TYPE;   // Type of Standard Tools List to Use

  public static final int _NONE          = las3.las3Constants._NONE;

  public static final int _VERSION       = las3.las3Constants._VERSION;
  public static final int _WELL          = las3.las3Constants._WELL;
  public static final int _LOG           = las3.las3Constants._LOG;
  public static final int _CORE          = las3.las3Constants._CORE;
  public static final int _DRILLING      = las3.las3Constants._DRILLING;
  public static final int _INCLINOMETRY  = las3.las3Constants._INCLINOMETRY;
  public static final int _TOPS          = las3.las3Constants._TOPS;
  public static final int _TEST          = las3.las3Constants._TEST;
  public static final int _PERFORATION   = las3.las3Constants._PERFORATION;
  public static final int _IQ_CONTROL    = las3.las3Constants._IQ_CONTROL;
  public static final int _IQ_LOG        = las3.las3Constants._IQ_LOG;
  public static final int _IQ_CORE       = las3.las3Constants._IQ_CORE;
  public static final int _IQ_SEQ_STRAT  = las3.las3Constants._IQ_SEQ_STRAT;
  public static final int _IQ_FLOW       = las3.las3Constants._IQ_FLOW;
  public static final int _IQ_GEO_REPORT = las3.las3Constants._IQ_GEO_REPORT;
  public static final int _IQ_SHALE      = las3.las3Constants._IQ_SHALE;
  public static final int _IQ_IMAGES     = las3.las3Constants._IQ_IMAGES;
  public static final int _IQ_BRINE      = las3.las3Constants._IQ_BRINE;
  public static final int _IQ_BIO        = las3.las3Constants._IQ_BIO;
  public static final int _IQ_ENV        = las3.las3Constants._IQ_ENV;

  public static final int _MNEMONIC      = las3.las3Struct._MNEMONIC;
  public static final int _UNIT          = las3.las3Struct._UNIT;
  public static final int _VALUE         = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION   = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT        = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION   = las3.las3Struct._ASSOCIATION;

  // Global Panel Widgets

  private JLabel    lblCurve    = new JLabel();
  private JTextArea txtSelected = new JTextArea();

  private JCheckBox cb[]        = null;
  private JButton   btnMnem[]   = null;
  private JLabel    lblUnit[]   = null;
  private JLabel    lblDesc[]   = null;

  private JRadioButton rbOFF    = new JRadioButton();
  private JRadioButton rbON     = new JRadioButton();

  /** Constructor las3CurvesLogPanel()
   *  <p> This is the Constructor for this class.
   * @param notifier = Observables Object
   * @param stStruct = Global IQSTRAT Data Structure
   * @param stLAS    = LAS File Data Structure
   * @param dVersion = The Version Number of LAS File
   * @param iqstrato = indicator that the data type has a profile saved data
   * @param stLAS3   = LAS 3 File Data Structure
   */

  public las3CurvesLogPanel( Observable        notifier,
                             iqstratStruct     stStruct,
                             lasFileDataStruct stLAS,
                             double            dVersion,
                             int               iqstrato,
                             las3Struct        stLAS3 )
  {
    try
    {
      this.notifier = notifier;
      this.stStruct = stStruct;
      this.stCMN    = stStruct.stCMN;
      this.stLAS    = stLAS;
      this.stLAS3   = stLAS3;
      this.dVersion = dVersion;
      this.iqstrato = iqstrato;

      if (stLAS3 != null)
      {
        if (stLAS3.iqstrat > -1) this.iqstrato = -1;
      }

//      loadData( las3.las3Utility.copy( stLAS3 ) );  // Load Data Structure
      loadData( stLAS3 );  // Load Data Structure

      jbInit();

      // Set Widgets

      setButtons();      // Enable or Disable buttons depending on data
      setDefault();      // Find & Load Default Curves into Data
      setDefinition();   // Build & Display ~{}_Definitions Section Text
    }
    catch(Exception ex)
    {
      ex.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    String sKID          = "0";
    String sMnem         = "";
    String sUnit         = "";
    String sDesc         = "";

    String start         = "0.0";
    String sEnd          = "0.0";
    String step          = "0.0";
    String sNull         = "0.0";

    JPanel pnlList       = new JPanel();
    JPanel pnlListTop    = new JPanel();

    JPanel pnlListRange  = new JPanel();
    JPanel pnlStart      = new JPanel();
    JPanel pnlNull       = new JPanel();
    JPanel pnlStep       = new JPanel();
    JPanel pnlEnd        = new JPanel();

    JPanel pnlListLabels = new JPanel();
    JPanel pnlListCurves = new JPanel();

    JPanel pnlCurves     = new JPanel();
    JPanel pnlCurvesTop  = new JPanel();
    JPanel pnlCurvesTopL = new JPanel();
    JPanel pnlCurvesTopU = new JPanel();

    JLabel lblCMnem      = new JLabel();
    JLabel lblCDesc      = new JLabel();
    JLabel lblCAssoc     = new JLabel();

    JPanel pnlCurvesText   = new JPanel();
    JScrollPane scrollText = new JScrollPane();

    JPanel pnlSelect     = new JPanel();
    JPanel pnlColumns    = new JPanel();
    JPanel pnl1          = new JPanel();

    JLabel lblLX         = new JLabel();
    JLabel lblLMnem      = new JLabel();
    JLabel lblLUnit      = new JLabel();
    JLabel lblLDesc      = new JLabel();

    JPanel pnlTable        = new JPanel();
    JScrollPane scrollList = new JScrollPane();

    JPanel pnl[]     = null;
    JPanel pnla[]    = null;
    JPanel pnlb[]    = null;
    JPanel pnlMnem[] = null;

    JTextField txtStart = new JTextField();
    JTextField txtEnd   = new JTextField();
    JTextField txtStep  = new JTextField();
    JTextField txtNull  = new JTextField();

    JPanel pnlAction    = new JPanel();
    ButtonGroup group   = new ButtonGroup();

    // Build Data Panel

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Start Depth:");

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "End Depth");

    TitledBorder titledBorderStep = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Step Depth");

    TitledBorder titledBorderNull = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Null Value");

    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderStep.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderNull.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setLayout(new BorderLayout());

    // Curve List Panel

    pnlList.setPreferredSize(new Dimension(350, 10));
    pnlList.setLayout(new BorderLayout());

    // -- Curve List Top Panel

    pnlListTop.setLayout(new BorderLayout());

    // -- Turn Data On or OFF

    pnlAction.setLayout(new GridLayout());

    rbOFF.setFont(new java.awt.Font("Dialog", 1, 11));
    rbOFF.setHorizontalAlignment(SwingConstants.CENTER);
    rbOFF.setText( "Do NOT Add this Data" );
    rbOFF.addActionListener(this);

    rbON.setFont(new java.awt.Font("Dialog", 1, 11));
    rbON.setHorizontalAlignment(SwingConstants.CENTER);
    rbON.setSelected(true);
    rbON.setText( "Add this Data" );
    rbON.addActionListener(this);

    // -- Depth Range List Panel

    pnlListRange.setLayout(new GridLayout());

    if (stLAS != null)
    {
      start = new String( "" + stLAS.depthStart );
      sEnd  = new String( "" + stLAS.depthEnd );
      step  = new String( "" + stLAS.depthStep );
      sNull = new String( "" + stLAS.dNull );
    }

    // .. Starting Depth Panel

    pnlStart.setLayout(new BorderLayout());
    pnlStart.setBorder(titledBorderStart);

    txtStart.setEditable(false);
    txtStart.setText(start);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);

    // .. Ending Depth Panel

    pnlEnd.setLayout(new BorderLayout());
    pnlEnd.setBorder(titledBorderEnd);

    txtEnd.setEditable(false);
    txtEnd.setText(sEnd);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);

    // .. Step Depth Panel

    pnlStep.setLayout(new BorderLayout());
    pnlStep.setBorder(titledBorderStep);

    txtStep.setEditable(false);
    txtStep.setText(step);
    txtStep.setHorizontalAlignment(SwingConstants.TRAILING);

    // .. Null Value Panel

    pnlNull.setLayout(new BorderLayout());
    pnlNull.setBorder(titledBorderNull);

    txtNull.setEditable(false);
    txtNull.setText(sNull);
    txtNull.setHorizontalAlignment(SwingConstants.TRAILING);

    // -- Curve List Top Labels Panel

    pnlListLabels.setLayout(new BorderLayout());

    // .. Curve List Top Select Panel

    pnlSelect.setPreferredSize(new Dimension(21, 25));

    lblLX.setFont(new java.awt.Font("Dialog", 1, 11));
    lblLX.setPreferredSize(new Dimension(7, 15));
    lblLX.setText("X");

    // .. Curve List Top Columns Panel

    pnlColumns.setLayout(new BorderLayout());

    lblLMnem.setFont(new java.awt.Font("Dialog", 1, 11));
    lblLMnem.setPreferredSize(new Dimension(120, 15));
    lblLMnem.setHorizontalAlignment(SwingConstants.CENTER);
    lblLMnem.setText("MNEM   ");

    // .. Curve List Top Columns 2 Panel

    pnl1.setLayout(new BorderLayout());

    lblLUnit.setFont(new java.awt.Font("Dialog", 1, 11));
    lblLUnit.setText(".UNITS   ");

    lblLDesc.setFont(new java.awt.Font("Dialog", 1, 11));
    lblLDesc.setText(": DESCRIPTION");

    // -- Curve List Table Panel

    pnlListCurves.setLayout(new BorderLayout());

    // ***********************************************************
    // -- Curves List Table Array

    iTotal       = 0;

    if (stCurves != null)
    {
      if (stLAS3.iArray > 0)
        sLabel = new String(
            "~" + stLAS3.sType  + "_Definition[" + stLAS3.iArray + "]" );
      else
        sLabel = new String( "~" + stLAS3.sType  + "_Definition" );

      iTotal = stCurves.iCount;
      lblCurve.setText( sLabel );

      switch (iData)
      {
        case _LAS_TYPE:
          iStandard = las.lasStandardTools.TOTAL;
          break;
        case _CORE_TYPE:
          iStandard = rock.rockStandardTools.TOTAL_DATA;
          break;
        case _TOPS_TYPE:
          iStandard = horizon.strat.stratStandardTools.TOTAL;
          break;
        case _SEQ_TYPE:
          iStandard = horizon.seq.seqStandardTools.TOTAL;
          break;
        case _ENV_TYPE:
          iStandard = horizon.env.envStandardTools.TOTAL;
          break;
        case _PERF_TYPE:
          iStandard = horizon.regions.regionsStandardTools.TOTAL;
          break;
        case _DST_TYPE:
          iStandard = dst.dstStandardTools.TOTAL;
          break;
        case _CNTRL_TYPE:
          iStandard = iqstrat.iqstratControlStandardTools.TOTAL;
          break;
        case _BIO_TYPE:
          iStandard = horizon.bio.bioStandardTools.TOTAL;
          break;
        case _FLOW_TYPE:
          iStandard = pfeffer.pfefferStandardTools.TOTAL_FLOW;
          break;
        case _GEO_TYPE:
          iStandard = iqstrat.iqstratRemarkStandardTools.TOTAL;
          break;
        case _SHALE_TYPE:
          iStandard = iqstrat.iqstratShaleStandardTools.TOTAL;
          break;
        case _IMAGES_TYPE:
          iStandard = rock.rockImagesStandardTools.TOTAL;
          break;
        case _BRINE_TYPE:
          iStandard = brine.brineStandardTools.TOTAL;
          break;
      }

      iSelected = new int[iStandard];
      for (int j=0; j<iStandard; j++)
        iSelected[j] = -1;
    }

    if (iTotal > 0)
    {
      if (iTotal > 11)
      {
        pnlTable.setLayout(new GridLayout(iTotal, 1));
        pnlTable.setPreferredSize(new Dimension(240, 30 * iTotal));
      }
      else
      {
        pnlTable.setLayout(new GridLayout(11, 1));
        pnlTable.setPreferredSize(new Dimension(240, 330));
      }

      pnl     = new JPanel[iTotal];
      pnla    = new JPanel[iTotal];
      pnlMnem = new JPanel[iTotal];
      pnlb    = new JPanel[iTotal];
      lblUnit = new JLabel[iTotal];
      lblDesc = new JLabel[iTotal];
      cb      = new JCheckBox[iTotal];
      btnMnem = new JButton[iTotal];

      for (int i=0; i<iTotal; i++)
      {
        if (stCurves != null)
        {
          sKID  = stCurves.stItem[i].sToolKID;
          sMnem = stCurves.stItem[i].sMnemonic;
          sUnit = stCurves.stItem[i].sCurveUnits;
          sDesc = stCurves.stItem[i].sCurveDescription;
        }

        // .. Main Curve Row Panel

        pnl[i]     = new JPanel();
        pnl[i].setLayout(new BorderLayout());

        // .... Check Box to display the curve to the Profile Plot

        cb[i]      = new JCheckBox();
        cb[i].setText("");
        if (i==0)
        {
          if (stLAS3.iType == las3.las3Constants._LOG)
            cb[i].setSelected(true);
        }
        cb[i].addActionListener(this);

        // .... Panel a

        pnla[i]    = new JPanel();
        pnla[i].setLayout(new BorderLayout());

        // .... .. Mnemonics Button Panel

        pnlMnem[i] = new JPanel();
        pnlMnem[i].setPreferredSize(new Dimension(120, 35));

        btnMnem[i] = new JButton();
        btnMnem[i].setPreferredSize(new Dimension(120, 25));
        btnMnem[i].setFont(new java.awt.Font("Dialog", 0, 11));
        if ((iData == _LAS_TYPE) && (i==0))
        {
          btnMnem[i].setEnabled(false);
          btnMnem[i].setText( "DEPT" );
        }
        else if (sKID.length() > 1)
          btnMnem[i].setText( sMnem );
        else
          btnMnem[i].setText( "? ( " + sMnem + " )"  );
        btnMnem[i].addActionListener(this);

        // .... .. Panel b

        pnlb[i]    = new JPanel();
        pnlb[i].setLayout(new BorderLayout());

        // .... .... Curve Unit Label

        lblUnit[i] = new JLabel();
        lblUnit[i].setFont(new java.awt.Font("Dialog", 0, 11));
        lblUnit[i].setText("." + sUnit);

        // .... .... Curve Description Label

        lblDesc[i] = new JLabel();
        lblDesc[i].setFont(new java.awt.Font("Dialog", 0, 11));
        lblDesc[i].setHorizontalAlignment(SwingConstants.LEADING);
        lblDesc[i].setText( " : " + sDesc );

        // Attach Widgets to the Table

        pnlTable.add(pnl[i],       null);
        pnl[i].add(cb[i],          BorderLayout.WEST);

        pnl[i].add(pnla[i],        BorderLayout.CENTER);
        pnla[i].add(pnlMnem[i],    BorderLayout.WEST);
        pnlMnem[i].add(btnMnem[i], null);

        pnla[i].add(pnlb[i],       BorderLayout.CENTER);
        pnlb[i].add(lblUnit[i],    BorderLayout.WEST);
        pnlb[i].add(lblDesc[i],    BorderLayout.CENTER);

        sKID  = "0";
        sMnem = "";
        sUnit = "";
        sDesc = "";
      }
    }

    // ***********************************************************

    // Selected Curves Panel

    pnlCurves.setBorder(BorderFactory.createEtchedBorder());
    pnlCurves.setLayout(new BorderLayout());

    // -- Selected Curves Top Panel

    pnlCurvesTop.setLayout(new BorderLayout());

    // .. Selected Curves Top Upper Panel

    pnlCurvesTopU.setLayout(new BorderLayout());

    lblCurve.setFont(new java.awt.Font("Dialog", 1, 11));
    lblCurve.setHorizontalAlignment(SwingConstants.LEADING);
//    lblCurve.setText("~{}");

    // .. Selected Curves Top Lower Panel

    pnlCurvesTopL.setLayout(new BorderLayout());

    lblCMnem.setFont(new java.awt.Font("Dialog", 1, 11));
    lblCMnem.setText("MNEM . UNITS");

    lblCDesc.setFont(new java.awt.Font("Dialog", 1, 11));
    lblCDesc.setHorizontalAlignment(SwingConstants.CENTER);
    lblCDesc.setText(" : DESCRIPTION");

    lblCAssoc.setFont(new java.awt.Font("Dialog", 1, 11));
    lblCAssoc.setText(" | ASSOCIATIONS");

    // -- Selected Curves Text Panel

    pnlCurvesText.setLayout(new BorderLayout());

    txtSelected.setText("");

    // Attach Widgets to Panel

    this.add(pnlList,                BorderLayout.WEST);
    pnlList.add(pnlListTop,          BorderLayout.NORTH);
    pnlListTop.add(pnlListRange,     BorderLayout.NORTH);
    pnlListRange.add(pnlStart,       null);
    pnlStart.add(txtStart,           BorderLayout.CENTER);
    pnlListRange.add(pnlEnd,         null);
    pnlEnd.add(txtEnd,               BorderLayout.CENTER);
    pnlListRange.add(pnlStep,        null);
    pnlStep.add(txtStep,             BorderLayout.CENTER);
    pnlListRange.add(pnlNull,        null);
    pnlNull.add(txtNull,             BorderLayout.CENTER);

    pnlListTop.add(pnlAction,        BorderLayout.CENTER);
    pnlAction.add(rbOFF,             null);
    pnlAction.add(rbON,              null);

    group.add(rbOFF);
    group.add(rbON);

    pnlListTop.add(pnlListLabels,    BorderLayout.SOUTH);
    pnlListLabels.add(pnlSelect,     BorderLayout.WEST);
    pnlSelect.add(lblLX,             null);

    pnlListLabels.add(pnlColumns,    BorderLayout.CENTER);
    pnlColumns.add(lblLMnem,         BorderLayout.WEST);
    pnlColumns.add(pnl1,             BorderLayout.CENTER);
    pnl1.add(lblLUnit,               BorderLayout.WEST);
    pnl1.add(lblLDesc,               BorderLayout.CENTER);

    pnlList.add(pnlListCurves,       BorderLayout.CENTER);
    pnlListCurves.add(scrollList,    BorderLayout.CENTER);
    scrollList.getViewport().add(pnlTable, null);

    this.add(pnlCurves,              BorderLayout.CENTER);
    pnlCurves.add(pnlCurvesTop,      BorderLayout.NORTH);
    pnlCurvesTop.add(pnlCurvesTopU,  BorderLayout.NORTH);
    pnlCurvesTopU.add(lblCurve,      BorderLayout.WEST);

    pnlCurvesTop.add(pnlCurvesTopL,  BorderLayout.SOUTH);
    pnlCurvesTopL.add(lblCMnem,      BorderLayout.WEST);
    pnlCurvesTopL.add(lblCDesc,      BorderLayout.CENTER);
    pnlCurvesTopL.add(lblCAssoc,     BorderLayout.EAST);

    pnlCurves.add(pnlCurvesText,     BorderLayout.CENTER);
    pnlCurvesText.add(scrollText,    BorderLayout.CENTER);
    scrollText.getViewport().add(txtSelected, null);
  }

  /** Method loadData()
   * <p> This method will load the data structure
   * @param st = LAS 3 Data Structure
   */

  private void loadData( las3Struct st )
  {
	String sXML = cmn.cmnStruct.LAS_STANDARD_TOOLS_XML;
	if (stStruct.stCMN.bStandalone)
	{
	  sXML = stStruct.stCMN.sPath + "/las_standard_tools.xml";
	}

    if (st != null)
    {
      iArray = st.iArray;              // Array Number
      iType  = st.iType;               // Type of Data Identifier
      sType  = new String( st.sType ); // Type of Data Name Set

      switch (st.iType)
      {
        case _NONE:
        case _VERSION:       // Version Informaiton
        case _WELL:          // Well Information
          break;
        case _LOG:           // Log Data Section
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( stCMN.sPath + "/las_standard_tools.xml", stCurves );
//          stCurves = las3.las3Map.mapToKGS( cmn.cmnStruct.LAS_STANDARD_TOOLS_XML, stCurves );
//                         stCMN.sPath + "/las_standard_tools.xml", stCurves );
          iData    = _LAS_TYPE;
          break;
        case _CORE:          // Core Data Section
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _CORE, stCurves );
          iData    = _CORE_TYPE;
          break;
        case _TOPS:          // Tops Data Section
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _TOPS, stCurves );
          iData    = _TOPS_TYPE;
          break;
        case _PERFORATION:   // Perforation Data Section
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _PERFORATION, stCurves );
          iData    = _PERF_TYPE;
          break;
        case _DRILLING:      // Drilling Data Section
        case _INCLINOMETRY:  // Inclinometry Data Section
          break;
        case _TEST:          // Test Data Section
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _TEST, stCurves );
          iData    = _DST_TYPE;
          break;
        case _IQ_CONTROL:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_CONTROL, stCurves );
          iData    = _CNTRL_TYPE;
          break;
        case _IQ_BIO:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_BIO, stCurves );
          iData    = _BIO_TYPE;
          break;
        case _IQ_SEQ_STRAT:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_SEQ_STRAT, stCurves );
          iData    = _SEQ_TYPE;
          break;
        case _IQ_ENV:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_ENV, stCurves );
          iData    = _ENV_TYPE;
          break;
        case _IQ_FLOW:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_FLOW, stCurves );
          iData    = _FLOW_TYPE;
        case _IQ_LOG:
        case _IQ_CORE:
          break;
        case _IQ_GEO_REPORT:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_GEO_REPORT, stCurves );
          iData    = _GEO_TYPE;
          break;
        case _IQ_SHALE:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_SHALE, stCurves );
          iData    = _SHALE_TYPE;
          break;
        case _IQ_IMAGES:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_IMAGES, stCurves );
          iData    = _IMAGES_TYPE;
          break;
        case _IQ_BRINE:
          stCurves = las3.las3Map.getLASCurves( st );
          stCurves = las3.las3Map.mapToKGS( _IQ_BRINE, stCurves );
          iData    = _BRINE_TYPE;
          break;
      }
    }
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getArray()
   * <p> This method will get the Array Number of the LAS 3 Data Type
   * @return iArray = the Array Number of the LAS 3 Data Type
   */

  public int getArray() { return (iArray); }

  /** Method getTypeID()
   * <p> This method will get the LAS 3 Data Type ID
   * @return iType = the LAS 3 Data Type ID
   */

  public int  getTypeID() { return (iType); }

  /** Method getType()
   * <p> This method will get the LAS 3 Data Type
   * @return iType = the LAS 3 Data Type
   */

  public String getType() { return (sType); }

  /** Method getLabel()
   * <p> This method will get the label for the tabbed Pane
   * @return sLabel = the label for the tabbed Pane
   */

  public String getLabel() { return (sLabel); }

  /** Method getDataType()
   * <p> This method will return the data type
   * @return iData = Data Type
   */

  public int getDataType() { return (iData); }

  /** Method getLAS3()
   * <p> This method will pass the LAS 3 File data structure with the selected
   *     curves and mnemonics.
   * @return stLAS3 = modified LAS 3 File data structure
   */

  public las3Struct getLAS3()
  {
    int i,j;
    String sTemp = "";

    if (stLAS3 != null)
    {
      stLAS3.iSelectRows = stLAS3.iCurveRows;
      stLAS3.select      = new String[stLAS3.iCurveRows][las3.las3Struct._TOTAL];

      for (i=0; i<stLAS3.iCurveRows; i++)
      {
        for (j=0; j<las3.las3Struct._TOTAL; j++)
          stLAS3.select[i][j] = new String("");

        if (stLAS3.select != null)
        {
          stLAS3.select[i][_MNEMONIC] = new String( btnMnem[i].getText() );
          sTemp = new String( removeChar( lblUnit[i].getText(), '.' ).trim() );
          stLAS3.select[i][_UNIT]  = new String( sTemp );
          stLAS3.select[i][_VALUE] = new String("");
          if (cb[i].isSelected())
            stLAS3.select[i][_VALUE] = new String("SELECTED");
          sTemp = new String( removeChar( lblDesc[i].getText(), ':' ).trim() );
          stLAS3.select[i][_DESCRIPTION] = new String( sTemp );
        }
      }
    }

    return (stLAS3);
  }

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @param  sDelim   = Delimiter used to separate data values
   * @param  sWrap    = Is the data line wrapped ( For LAS 2.0 Files )
   * @param  dVersion = Version number of LAS File
   * @param  iLog     = Log Number
   * @param  st       = the LAS File Data Structure
   * @return st       = the LAS File Data Structure
   */

  public lasFileDataStruct getLAS( double dVersion, int iLog, lasFileDataStruct st )
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    lasFileDataStruct     stC   = null;
    String sKIDo   = "0";
    String sKID    = "0";
    String sData[] = null;
    int    iFound  = -1;
    int    iType   = -1;

    if (iData == _LAS_TYPE)
    {
      if (dVersion == 2.0)
      {
        // LAS 3 File Parameters

        st.iParamRows = stLAS3.iParamRows;    // Total number of rows
        st.iParamCols = stLAS3.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stLAS3.sParams[i][j] );
          }
        }
      }

      // Load the SELECTED Curves Settings into the LAS 3 structure

      getLAS3();

      // Copy the LAS 3 Curves & Parameters to the LAS File Data Structure

      st.stLAS3[iLog] = las3.las3Utility.copyWOData( stLAS3 );
    }

    return (st);
  }

  /** Method getLASData()
   * <p> This method will retrieve the LAS File Data Structure
   * @param  sDelim   = Delimiter used to separate data values
   * @param  sWrap    = Is the data line wrapped ( For LAS 2.0 Files )
   * @param  dVersion = Version number of LAS File
   * @param  iLog     = Log Number
   * @param  st       = the LAS File Data Structure
   * @return st       = the LAS File Data Structure
   */

  public lasFileDataStruct getLASData( String sDelim, String sWrap, double dVersion,
                                       int iLog, lasFileDataStruct st )
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    lasFileDataStruct     stC   = null;
//    lasFileDataStruct     st    = null;
    String sKIDo   = "0";
    String sKID    = "0";
    String sData[] = null;
    int    iFound  = -1;
    int    iType   = -1;
    double data    = 0.0;
    int    iStart  = 0;
    double dStart  = 0.0;
    double dEnd    = 0.0;
    double dStep   = 0.0;
    int    iStep   = 0;
    int    iRows   = 0;
    int    iRow    = 0;
    las3ReadFile p3 = null;

    if (iData == _LAS_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            if (cb[i].isSelected())
            {
              iTool = -1;
              for (j=0; j<iStandard; j++)
              {
                if (iSelected[j] > -1)
                {
                  sKIDo = las.lasStandardTools.LAS_TOOLS[j][0];
                  if (sKID.equals(sKIDo))
                  {
                    iTool = j;
                  }
                }
              }

              if ((iTool > -1) || (i==0))
              {
                if (iCnt < iSum)
                {
				  stCurves.stItem[i].iCurveNumber = i; // JRV 11/28/2011
                  stNew.stItem[iCnt] =
                      cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                  iCnt++;
                }
              }
            }
          }
        }
      }

      // Get the LAS Data Structure

      stC   = new lasFileDataStruct();
      stC   = las.lasFileDataUtility.findCurves( stNew, stC );
      stLAS = las.lasFileDataUtility.findCurves( stNew, stLAS );
      st    = las.lasFileDataUtility.findCurves( stNew, st );

//System.out.println(      st.depthStart+" "+st.depthEnd+" "+st.depthStep);

      // Get Log Data from LAS File

      iType = util.utilFileIO.DIRECTORY;
      if (st.sReadMethod.equals("URL"))
        iType = util.utilFileIO.URL_FILE;

      // Open and compute the depth Range & Step
/*
      p3 = new las3ReadFile();
      p3.Open( iType, st.sDirectory, st.sFilename );        // Open LAS File
      iFound = p3.ReadFile( las3.las3Constants._LOG, iLog, dVersion );

      if (iFound == 1)        // IF Log Section is Found
      {
        while (iFound == 1) // While Data line is not null
        {
          sData = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);
          if (sData == null)
            iFound = -1;
          else
          {
            data = las3.las3LoadLogData.stringToDouble( sData[0], stLAS.dNull );

            if (data != stLAS.dNull)
            {
			  if (iStart == 0)
			  {
			    dStart = data;
			    dEnd   = data;

			    sData  = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);
			    if (sData != null)
			    {
                  data  = las3.las3LoadLogData.stringToDouble( sData[0], stLAS.dNull );
                  iStep = (int) ( Math.abs( data - dStart ) * 100.0 );
                  dStep = (double) iStep / 100.0;
			    }

			    iStart = 1;
			  }

			  if (dEnd < data)
			    dEnd = data;
			}
		  }
        }
      }

      // Compute the Data Structrue

      iRows = (int) ( Math.abs(dEnd - dStart) / dStep ) + 1;

      st.depthStart = dStart;
      st.depthEnd   = dEnd;
      st.depthStep  = dStep;

      st = las3.las3LoadLogData.initData( iRows, st );

      p3.Close();                                           // Close LAS File

      if (p3 != null) // Close & clear memory for the LAS 3 File Read Class
        p3.delete();
      p3 = null;
*/

      p3 = new las3ReadFile();
      p3.Open( iType, st.sDirectory, st.sFilename );        // Open LAS File
      iFound = p3.ReadFile( las3.las3Constants._LOG, iLog, dVersion );

      if (iFound == 1)        // IF Log Section is Found
      {
        while (iFound == 1) // While Data line is not null
        {
          sData = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);

          if (sData == null)
            iFound = -1;
          else
          {
            data = las3.las3LoadLogData.stringToDouble( sData[0], stLAS.dNull );

            if (data != stLAS.dNull)
            {
			  if (iStart == 0)
			  {
			    dStart = data;
			    dEnd   = data;

			    sData  = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);
			    if (sData != null)
			    {
                  data  = las3.las3LoadLogData.stringToDouble( sData[0], stLAS.dNull );
                  dStep = data - dStart;
//                  iStep = (int) ( Math.abs( data - dStart ) * 100.0 );
//                  dStep = (double) iStep / 100.0;
			    }

			    iStart = 1;
			  }

			  if (dEnd < data)
			    dEnd = data;
			}
		  }
        }
      }

      p3.Close();                                           // Close LAS File

      if (p3 != null) // Close & clear memory for the LAS 3 File Read Class
        p3.delete();
      p3 = null;

//      iRows = (int) ( Math.abs(st.depthEnd - st.depthStart) / st.depthStep ) + 1;
      iRows = (int) ( Math.abs(dEnd - dStart) / dStep ) + 1;
      st.depthStart = dStart;
      st.depthEnd   = dEnd;
      st.depthStep  = dStep;

      st    = las3.las3LoadLogData.initData( iRows, st );

      p3 = new las3ReadFile();
      p3.Open( iType, st.sDirectory, st.sFilename );        // Open LAS File
      iFound = p3.ReadFile( las3.las3Constants._LOG, iLog, dVersion );

      if (iFound == 1)        // IF Log Section is Found
      {
		iRow = 0;
        while (iFound == 1) // While Data line is not null
        {
          sData = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);

          if (sData == null)
          {
			iFound = -1;
		  }
          else
          {
            data = las3.las3LoadLogData.stringToDouble(sData[0], stLAS.dNull);

            if (iRow < st.iRows)
            {
              st.depths[iRow] = data;
              iRow++;
			}
		  }
        }
      }

      p3.Close();                                           // Close LAS File

      if (p3 != null) // Close & clear memory for the LAS 3 File Read Class
        p3.delete();
      p3 = null;

      // Now Add the Log Data to the Data Structure
//System.out.println(iRows+" "+dStart+" "+dEnd+" "+dStep);

      p3 = new las3ReadFile();
      p3.Open( iType, st.sDirectory, st.sFilename );        // Open LAS File
      iFound = p3.ReadFile( las3.las3Constants._LOG, iLog, dVersion );

      if (iFound == 1)        // IF Log Section is Found
      {
        while (iFound == 1) // While Data line is not null
        {
          sData = p3.getNextLine(sDelim, sWrap, dVersion, stLAS3.iColumns);
          if (sData == null) iFound = -1;
          st = las3.las3LoadLogData.addData( sData, stLAS3.iColumns, stC, st );
        }
      }

      p3.Close();                                           // Close LAS File
      stC.delete();
      stC = null;

      // Deallocate Java Memory

      if (p3 != null) // Close & clear memory for the LAS 3 File Read Class
        p3.delete();
      p3 = null;

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getRockData()
   * <p> This method will retrieve the Rock Data List Structure
   * @return st = the Rock Data List Structure
   */

  public rockDataListStruct getRockData()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    rockDataListStruct    st    = new rockDataListStruct();
    String sKIDo  = "0";
    String sKID   = "0";

    if (iData == _CORE_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = rock.rockStandardTools.ROCK_TOOLS[j][6];
                if (sKID.equals(sKIDo))
                  iTool = j;
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the LAS Data Structure

      st = las3.las3LoadCoreData.getData( stLAS3 );
      st = las3.las3LoadCoreData.loadData( stLAS.dNull, stLAS3, stNew, st );

      if (dVersion == 2.0)
      {
        // LAS 3 File Parameters

        st.iParamRows = stLAS3.iParamRows;    // Total number of rows
        st.iParamCols = stLAS3.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stLAS3.sParams[i][j] );
          }
        }
      }

      getLAS3(); // Load the SELECTED Curves Settings into the LAS 3 structure

      // Copy the LAS 3 Curves & Parameters to the LAS File Data Structure

      st.stLAS3 = new las3Struct[1];
      st.iLogs  = 1;
      st.stLAS3[0] = las3.las3Utility.copyWOData( stLAS3 );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getBrine()
   * <p> This method will retrieve the Brine List Structure
   * @return st = the Brine Data List Structure
   */

  public brineListStruct getBrine()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    brineListStruct       st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _BRINE_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = brine.brineStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Brine Data List Structure

      st = las3.las3LoadBrineData.getData( stLAS3 );
      st = las3.las3LoadBrineData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getImages()
   * <p> This method will retrieve the Rock Images List Structure
   * @return st = the Rock Images Data List Structure
   */

  public rockImagesListStruct getImages()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    rockImagesListStruct  st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _IMAGES_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = rock.rockImagesStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Rock Images Data List Structure

      st = las3.las3LoadImageData.getData( stLAS3 );
      st = las3.las3LoadImageData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Units Data Structure
   * @return st = the Stratigraphic Units Data List Structure
   */

  public stratListStruct getTops()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    stratListStruct       st    = null;
    String sKIDo = "0";
    String sKID  = "0";

    if (iData == _TOPS_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;
            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = horizon.strat.stratStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Stratigraphic Units Data List Structure

      st = las3.las3LoadTopsData.getData( stLAS3 );
      st = las3.las3LoadTopsData.loadData( stLAS.dNull, stLAS3, stNew, st );

      if (dVersion == 2.0)
      {
        // LAS 3 File Parameters

        st.iParamRows = stLAS3.iParamRows;    // Total number of rows
        st.iParamCols = stLAS3.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stLAS3.sParams[i][j] );
          }
        }
      }

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getPerf()
   * <p> This method will retrieve the Perforations Data Structure
   * @return st = the Perforations Data List Structure
   */

  public regionsListStruct getPerf()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    regionsListStruct     st    = null;
    String sKIDo = "0";
    String sKID  = "0";

    if (iData == _PERF_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = horizon.regions.regionsStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Perforations Data List Structure

      st = las3.las3LoadPerfData.getData( stLAS3 );
      st = las3.las3LoadPerfData.loadData( stLAS.dNull, stLAS3, stNew, st );

      if (dVersion == 2.0)
      {
        // LAS 3 File Parameters

        st.iParamRows = stLAS3.iParamRows;    // Total number of rows
        st.iParamCols = stLAS3.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stLAS3.sParams[i][j] );
          }
        }
      }

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getDST()
   * <p> This method will retrieve the DST Data Structure
   * @return st = the DST Data Structure
   */

  public dstStruct getDST()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct stNew = null;
    dstStruct             st    = null;
    String sKIDo = "0";
    String sKID  = "0";

    if (iData == _DST_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = dst.dstStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Perforations Data List Structure

      st = las3.las3LoadDSTData.getData( stLAS3 );
      st = las3.las3LoadDSTData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getControl()
   * <p> This method will retrieve the Control Data Structure
   * @return st = the Control Data Structure
   */

  public iqstratControlStruct getControl()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    iqstratControlStruct  st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _CNTRL_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = iqstrat.iqstratControlStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Control Data Structure

      st = las3.las3LoadControlData.getData( stLAS3 );
      st = las3.las3LoadControlData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getBio()
   * <p> This method will retrieve the Bio Stratigraphy List Structure
   * @return st = the Bio Stratigraphy Data List Structure
   */

  public bioStratListStruct getBio()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    bioStratListStruct    st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _BIO_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = horizon.bio.bioStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Bio Stratigraphy Data List Structure

      st = las3.las3LoadBioData.getData( stLAS3 );
      st = las3.las3LoadBioData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getSeq()
   * <p> This method will retrieve the Sequence Stratigraphy List Structure
   * @return st = the Sequence Stratigraphy Data List Structure
   */

  public seqListStruct getSeq()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    seqListStruct         st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _SEQ_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = horizon.seq.seqStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Sequence Stratigraphy Data List Structure

      st = las3.las3LoadSeqData.getData( stLAS3 );
      st = las3.las3LoadSeqData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getSeq()
   * <p> This method will retrieve the Depositional Environment List Structure
   * @return st = the Depositional Environment Data List Structure
   */

  public envListStruct getEnv()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    envListStruct         st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _ENV_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = horizon.env.envStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Sequence Stratigraphy Data List Structure

      st = las3.las3LoadEnvData.getData( stLAS3 );
      st = las3.las3LoadEnvData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getPfeffer()
   * <p> This method will retrieve the PfEFFER Data List Structure
   * @return st = the PfEFFER Data List Structure
   */

  public pfefferDataListStruct getPfeffer()
  {
    int i,j;
    int                   iSum  = 1;
    int                   iCnt  = 0;
    int                   iTool = -1;
    cmnLASCurveListStruct stNew = null;
    pfefferDataListStruct st    = null;
    String                sKIDo = "0";
    String                sKID  = "0";

    if (iData == _FLOW_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = pfeffer.pfefferStandardTools.P_CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Control Data Structure

      st = las3.las3LoadPfefferData.getData( stLAS3 );
      st = las3.las3LoadPfefferData.loadData(
              stLAS.dNull, stLAS3, stNew, stLAS, stStruct.stSymbols, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getRemarks()
   * <p> This method will retrieve the Remarks Data List Structure
   * @return st = the Remarks Data List Structure
   */

  public iqstratRemarkListStruct getRemarks()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct   stNew = null;
    iqstratRemarkListStruct st    = null;
    String sKIDo = "0";
    String sKID  = "0";

    if (iData == _GEO_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = iqstrat.iqstratRemarkStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Perforations Data List Structure

      st = las3.las3LoadRemarksData.getData( stLAS3 );
      st = las3.las3LoadRemarksData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method getShale()
   * <p> This method will retrieve the Shale Data List Structure
   * @return st = the Shale Data List Structure
   */

  public iqstratShaleListStruct getShale()
  {
    int i,j;
    int iSum  = 1;
    int iCnt  = 0;
    int iTool = -1;
    cmnLASCurveListStruct  stNew = null;
    iqstratShaleListStruct st    = null;
    String sKIDo = "0";
    String sKID  = "0";

    if (iData == _SHALE_TYPE)
    {
      for (j=0; j<iStandard; j++)
      {
        if (iSelected[j] > -1)
        {
          iSum++;
        }
      }

      if (stCurves != null)
      {
        if (stCurves.iCount > 0)
        {
          stNew          = new cmnLASCurveListStruct();
          stNew.iCount   = iSum;
          stNew.stItem   = new cmnLASCurveStruct[iSum];

          for (i=0; i<stCurves.iCount; i++)
          {
            sKID   = stCurves.stItem[i].sToolKID;

            iTool = -1;
            for (j=0; j<iStandard; j++)
            {
              if (iSelected[j] > -1)
              {
                sKIDo = iqstrat.iqstratShaleStandardTools.CURVES[j][0];
                if (sKID.equals(sKIDo))
                {
                  iTool = j;
                }
              }
            }

            if (iTool > -1)
            {
              if (iCnt < iSum)
              {
                stNew.stItem[iCnt] =
                    cmn.cmnLASCurveUtility.copy( stCurves.stItem[i] );
                iCnt++;
              }
            }
          }

          stNew.iCount   = iCnt;
        }
      }

      // Get the Perforations Data List Structure

      st = las3.las3LoadShaleData.getData( stLAS3 );
      st = las3.las3LoadShaleData.loadData( stLAS.dNull, stLAS3, stNew, st );

      // Delete the Common LAS Curves Data List Structure

      if (stNew != null)
        stNew.delete();
      stNew = null;
    }

    return (st);
  }

  /** Method isCurve()
   * <p> This method will determine if the curve was changed or not
   * @return bValue = true the curve is not -1 and false if it is
   */

  public boolean isCurve()
  {
    boolean bValue = false;

    if (iCurve > -1)
      bValue = true;

    return (bValue);
  }

  /** Method getCurve()
   * <p> This method will retrieve the curve id and populate the Curve Row data.
   */

  public void getCurve()
  {
    cmnLASCurveStruct st = new cmnLASCurveStruct();

    if (isCurve())
    {
      if (pTools != null)
      {
        st = cmn.cmnLASCurveUtility.copy( pTools.getCurve( st) );
//        stCurves.stItem[iCurve] = cmn.cmnLASCurveUtility.copy(
//                                    pTools.getCurve( stCurves.stItem[iCurve]) );

        cb[iCurve].setSelected(false);

        if ((st.sCurveUnits.equals("UNI")) || (st.sCurveUnits.equals("UNL")))
        {
		  stCurves.stItem[iCurve].sToolKID    = new String( st.sToolKID );
		  stCurves.stItem[iCurve].sCurveName  =
		    new String( "("+stCurves.stItem[iCurve].sCurveUnits+") "+
		                stCurves.stItem[iCurve].sCurveName );
		  stCurves.stItem[iCurve].sCurveUnits = new String( st.sCurveUnits );
          stCurves.stItem[iCurve].sMnemonic   = resetMnemonic( stCurves.stItem[iCurve].sMnemonic );

          btnMnem[iCurve].setText( stCurves.stItem[iCurve].sMnemonic );
          lblUnit[iCurve].setText( "." + stCurves.stItem[iCurve].sCurveUnits );
          lblDesc[iCurve].setText( " : " + stCurves.stItem[iCurve].sCurveName );
		}
        else
        {
		  stCurves.stItem[iCurve] = cmn.cmnLASCurveUtility.copy( st );

          btnMnem[iCurve].setText( stCurves.stItem[iCurve].sMnemonic );
          lblUnit[iCurve].setText( "." + stCurves.stItem[iCurve].sCurveUnits );
          lblDesc[iCurve].setText( " : " + stCurves.stItem[iCurve].sCurveName );
	    }
      }

      pTools.close();
      pTools = null;
      iCurve = -1;
    }

    setButtons();      // Enable or Disable buttons depending on data
  }

  /** Method resetMnemonic()
   * <p> This method will remove the '?', '(' and ')' around the String
   * @param  str  = The string to be converted
   * @return sNew = The text converted
   */

  public static String resetMnemonic( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if ((c[i] != '?') && (c[i] != '(') && (c[i] != ')'))
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method getAction()
   *  <p> This method will retrieve the question is this data to be included?
   */

  public int getAction() { return (iAction); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setButtons()
   * <p> This method will Disable or Enable Buttons depending on the data type
   */

  private void setButtons()
  {
    String sKIDi  = "0";

    if (cb != null)
    {
      if (cb.length > 0)
		cb[0].setEnabled( false );
    }

    if (btnMnem != null)
    {
	  if (btnMnem.length > 0)
	    btnMnem[0].setEnabled( false );
	}

    setCBColor(); // Set CheckBox Color

    for (int i=1; i<iTotal; i++)
    {
      if (stCurves != null)
      {
        sKIDi = stCurves.stItem[i].sToolKID;
      }

      cb[i].setEnabled( false );
      btnMnem[i].setEnabled( false );

      if (iAction == ON)
      {
		if (iData != _LAS_TYPE)
		{
  		  cb[0].setEnabled( true );
	      btnMnem[0].setEnabled( true );
		}

        btnMnem[i].setEnabled( true );
        if (sKIDi.length() > 1)
        {
          cb[i].setEnabled( true );
        }
      }
    }
  }

  /** Method setCBColor()
   *  <p> This method will set the CheckBox Color depending if the data is selected or not
   */

  private void setCBColor()
  {
    int    iTool  = -1;
    String str    = "";
    String sKIDo  = "0";
    String sKID   = "0";
    String sMnem  = "";
    String sUnit  = "";
    String sDesc  = "";
    String sAssoc = "";

    for (int i=0; i<iTotal; i++)
    {
      if (stCurves != null)
      {
        sKID   = stCurves.stItem[i].sToolKID;
        sMnem  = stCurves.stItem[i].sMnemonic;
        sUnit  = stCurves.stItem[i].sCurveUnits;
        sDesc  = stCurves.stItem[i].sCurveDescription;

        iTool = -1;
        for (int j=0; j<iStandard; j++)
        {
          switch (iData)
          {
            case _LAS_TYPE:
              sKIDo = las.lasStandardTools.LAS_TOOLS[j][0];
              break;
            case _CORE_TYPE:
              sKIDo = rock.rockStandardTools.ROCK_TOOLS[j][6];
              break;
            case _TOPS_TYPE:
              sKIDo = horizon.strat.stratStandardTools.CURVES[j][0];
              break;
            case _SEQ_TYPE:
              sKIDo = horizon.seq.seqStandardTools.CURVES[j][0];
              break;
            case _ENV_TYPE:
              sKIDo = horizon.env.envStandardTools.CURVES[j][0];
              break;
            case _PERF_TYPE:
              sKIDo = horizon.regions.regionsStandardTools.CURVES[j][0];
              break;
            case _DST_TYPE:
              sKIDo = dst.dstStandardTools.CURVES[j][0];
              break;
            case _CNTRL_TYPE:
              sKIDo = iqstrat.iqstratControlStandardTools.CURVES[j][0];
              break;
            case _BIO_TYPE:
              sKIDo = horizon.bio.bioStandardTools.CURVES[j][0];
              break;
            case _FLOW_TYPE:
              sKIDo = pfeffer.pfefferStandardTools.P_CURVES[j][0];
              break;
            case _GEO_TYPE:
              sKIDo = iqstrat.iqstratRemarkStandardTools.CURVES[j][0];
              break;
            case _SHALE_TYPE:
              sKIDo = iqstrat.iqstratShaleStandardTools.CURVES[j][0];
              break;
            case _IMAGES_TYPE:
              sKIDo = rock.rockImagesStandardTools.CURVES[j][0];
              break;
           case _BRINE_TYPE:
              sKIDo = brine.brineStandardTools.CURVES[j][0];
              break;
          }

          if (sKID.equals(sKIDo))
            iTool = j;
        }
      }

      if (cb[i].isSelected())   // JRV
      {
		// Resistivity, Spontaneous Potential
		if ((sUnit.equals("OHM-M")) || (sUnit.equals("OHMM")) || (sUnit.equals("MV")))
          cb[i].setBackground( new Color( 255, 165, 0 ) );   // orange
        // Porosity
		else if (sUnit.equals("PU"))
          cb[i].setBackground( new Color( 0, 255, 255 ) ); // cyan
        // Photoelectric
		else if (sUnit.equals("BARNS/E"))
          cb[i].setBackground( new Color( 173, 255, 47 ) ); // green yellow
        // Grain Density
		else if ((sUnit.equals("GM/CC"))  || (sUnit.equals("GMCC")))
          cb[i].setBackground( new Color( 0, 255, 0 ) ); // green
        // Acoustic transit time
		else if (sUnit.equals("USEC/FT"))
          cb[i].setBackground( new Color( 34, 139, 34 ) ); // forest green
        // Radioactive
		else if ((sUnit.equals("PPM")) || (sUnit.equals("%")) || (sUnit.equals("API")))
          cb[i].setBackground( new Color( 255, 0, 0 ) );     // red
        // Permeability
		else if (sUnit.equals("MD"))
          cb[i].setBackground( new Color( 0, 0, 255 ) ); // blue
        // Feet Inches
		else if ((sUnit.equals("F")) || (sUnit.equals("FT")) || (sUnit.equals("IN")))
          cb[i].setBackground( new Color( 165, 42, 42 ) ); // brown
        // Fraction
		else if (sUnit.equals("FRAC"))
          cb[i].setBackground( new Color( 255, 255, 128 ) ); // middle yellow
        // Unknown Linear Curve
		else if (sUnit.equals("UNI"))
          cb[i].setBackground( new Color( 148, 0, 211 ) ); // dark violet
        // Unknown Log Curve
		else if (sUnit.equals("UNL"))
          cb[i].setBackground( new Color( 148, 130, 238 ) ); // medium violet
        // Default
        else
          cb[i].setBackground( new Color( 255, 255, 0 ) ); // middle yellow
	  }
	  else
	  {
        cb[i].setBackground( new Color( 220, 220, 220 ) );
	  }
    }
  }

  /** Method setDefault()
   * <p> This method will automatically load the selected curves initially
   */

  private void setDefault()
  {
    String  sKIDo  = "0";
    int     iCont  = 0;

    for (int i=0; i<iTotal; i++)
    {
      if (stCurves != null)
      {
        for (int j=0; j<iStandard; j++)
        {
          switch (iData)
          {
            case _LAS_TYPE:
              sKIDo = las.lasStandardTools.LAS_TOOLS[j][0];
              break;
            case _CORE_TYPE:
              sKIDo = rock.rockStandardTools.ROCK_TOOLS[j][6];
              break;
            case _TOPS_TYPE:
              sKIDo = horizon.strat.stratStandardTools.CURVES[j][0];
              break;
            case _SEQ_TYPE:
              sKIDo = horizon.seq.seqStandardTools.CURVES[j][0];
              break;
            case _ENV_TYPE:
              sKIDo = horizon.env.envStandardTools.CURVES[j][0];
              break;
            case _PERF_TYPE:
              sKIDo = horizon.regions.regionsStandardTools.CURVES[j][0];
              break;
            case _DST_TYPE:
              sKIDo = dst.dstStandardTools.CURVES[j][0];
              break;
            case _CNTRL_TYPE:
              sKIDo = iqstrat.iqstratControlStandardTools.CURVES[j][0];
              break;
            case _BIO_TYPE:
              sKIDo = horizon.bio.bioStandardTools.CURVES[j][0];
              break;
            case _FLOW_TYPE:
              sKIDo = pfeffer.pfefferStandardTools.P_CURVES[j][0];
              break;
            case _GEO_TYPE:
              sKIDo = iqstrat.iqstratRemarkStandardTools.CURVES[j][0];
              break;
            case _SHALE_TYPE:
              sKIDo = iqstrat.iqstratShaleStandardTools.CURVES[j][0];
              break;
            case _IMAGES_TYPE:
              sKIDo = rock.rockImagesStandardTools.CURVES[j][0];
              break;
           case _BRINE_TYPE:
              sKIDo = brine.brineStandardTools.CURVES[j][0];
              break;
          }

          if ((stCurves.stItem[i].sToolKID.equals(sKIDo)) && (iqstrato == -1))
          {
            iCont = 1;
            if (stLAS3.iSelectRows > 0)
            {
              if (!stLAS3.select[i][_VALUE].equals("SELECTED"))
                iCont = 0;
            }

            if (iCont == 1)
            {
              cb[i].setSelected(true);
              setCheckBox( i, _IGNORE );
            }
          }
        }
      }
    }

    setButtons();
  }

  /** Method isLogCurve()
   * <p> This method will determine if the log curve type is already selected
   * @param  iBox = The Check Box that is being checked
   * @return bValue = true if the log curve type is already selected,
   *                  false otherwise
   */

  private boolean isLogCurve( int iBox )
  {
    boolean bValue = false;

    if (stCurves != null)
    {
      for (int i=1; i<iTotal; i++)
      {
        if (i != iBox)
        {
          if (stCurves.stItem[iBox].sToolKID.equals(
                stCurves.stItem[i].sToolKID))
          {
            if (cb[i].isSelected())
              bValue = true;
          }
        }
      }
    }

    return (bValue);
  }

  /** Method setCheckBox()
   * <p> This method will check to see if the Check box is selected or not
   *     and enable or disable other similar curves.
   *  @param iBox     = the Check Box was selected
   *  @param iDisplay = Display Error message if the log curve type was selected
   *                    or not.  The Set Default Method calls this method when
   *                    the data is loaded initially, so the display is set to
   *                    off.
   */

  private void setCheckBox( int iBox, int iDisplay )
  {
    if (isLogCurve( iBox ))
    {
      cb[iBox].setSelected(false);

      if (iDisplay == _CHECK)
      JOptionPane.showMessageDialog((Component) null,
                                     "Duplicate Curve Type Selected",
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);
    }
  }

  /** Method setDefinition()
   * <p> This method will set the Curve Definition section text area
   */

  private void setDefinition()
  {
    int    iTool  = -1;
    String str    = "";
    String sKIDo  = "0";
    String sKID   = "0";
    String sMnem  = "";
    String sUnit  = "";
    String sDesc  = "";
    String sAssoc = "";

    if (stCurves != null)
    {
      for (int j=0; j<iStandard; j++)
        iSelected[j] = -1;
    }

    for (int i=0; i<iTotal; i++)
    {
      if (stCurves != null)
      {
        sKID   = stCurves.stItem[i].sToolKID;
        sMnem  = stCurves.stItem[i].sMnemonic;
        sUnit  = stCurves.stItem[i].sCurveUnits;
        sDesc  = stCurves.stItem[i].sCurveDescription;

        sAssoc = "";
        if (stCurves.stItem[i].iAssoc > 0)
        {
          sAssoc = new String("| ");
          for (int j=0; j<stCurves.stItem[i].iAssoc; j++)
          {
            if (j>0)
              sAssoc = new String(sAssoc + ", ");
            sAssoc = new String(sAssoc + stCurves.stItem[i].sAssoc[j]);
          }
        }

        iTool = -1;
        for (int j=0; j<iStandard; j++)
        {
          switch (iData)
          {
            case _LAS_TYPE:
              sKIDo = las.lasStandardTools.LAS_TOOLS[j][0];
              break;
            case _CORE_TYPE:
              sKIDo = rock.rockStandardTools.ROCK_TOOLS[j][6];
              break;
            case _TOPS_TYPE:
              sKIDo = horizon.strat.stratStandardTools.CURVES[j][0];
              break;
            case _SEQ_TYPE:
              sKIDo = horizon.seq.seqStandardTools.CURVES[j][0];
              break;
            case _ENV_TYPE:
              sKIDo = horizon.env.envStandardTools.CURVES[j][0];
              break;
            case _PERF_TYPE:
              sKIDo = horizon.regions.regionsStandardTools.CURVES[j][0];
              break;
            case _DST_TYPE:
              sKIDo = dst.dstStandardTools.CURVES[j][0];
              break;
            case _CNTRL_TYPE:
              sKIDo = iqstrat.iqstratControlStandardTools.CURVES[j][0];
              break;
            case _BIO_TYPE:
              sKIDo = horizon.bio.bioStandardTools.CURVES[j][0];
              break;
            case _FLOW_TYPE:
              sKIDo = pfeffer.pfefferStandardTools.P_CURVES[j][0];
              break;
            case _GEO_TYPE:
              sKIDo = iqstrat.iqstratRemarkStandardTools.CURVES[j][0];
              break;
            case _SHALE_TYPE:
              sKIDo = iqstrat.iqstratShaleStandardTools.CURVES[j][0];
              break;
            case _IMAGES_TYPE:
              sKIDo = rock.rockImagesStandardTools.CURVES[j][0];
              break;
           case _BRINE_TYPE:
              sKIDo = brine.brineStandardTools.CURVES[j][0];
              break;
          }

          if (sKID.equals(sKIDo))
            iTool = j;
        }
      }

      if (cb[i].isSelected())
      {
        if (iTool > -1) iSelected[iTool] = i;
        str = new String( str + sMnem  + " ."      +
                                sUnit  + "     : " +
                                sDesc  + "  {F}  " +
                                sAssoc + "\n" );
      }
    }

    txtSelected.setText( str );
  }

  /** Method setCurve(int iFile, int iCurve)
   * <p> This method will display the Standard Curves List Frame
   * @param iCurve = The LAS File Curve Selected
   */

  private void setCurve( int iCurve )
  {
    this.iCurve = iCurve;

    if (pTools != null)
      pTools.close();
    pTools = null;

    pTools = new las3ToolsFrame( notifier, iData );
  }

  /** Method removeQuotes()
   * <p> This method will remove the Quotes around the String
   * @param  str  = The string to be converted
   * @param  ch   = character to be removed from string
   * @return sNew = The text converted
   */

  public static String removeChar( String str, char ch )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if (c[i] != ch)
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    // Input Variables

    notifier = null; // Observables Object
    stStruct = null; // Global IQSTRAT Data Structure
    stCMN    = null; // Common Data Structure
    stLAS3   = null; // LAS 3 File Data Structure
    stLAS    = null; // LAS File Data Structure

    // Global Variables

    iArray = 0;                       // Array Number
    iType  = las3.las3Constants._NONE; // Type of Data Identifier
    sType  = null;                    // Type of Data Name Set
    sLabel = null;

    iSelected = null;

    if (stCurves != null)
      stCurves.delete();
    stCurves = null;

    if (stRock != null)
      stRock.delete();
    stRock = null;

    if (pTools != null)
      pTools.close();
    pTools = null;

    // Global Panel Widgets

    lblCurve    = null;
    txtSelected = null;

    for (int i=0; i<iTotal; i++)
    {
      cb[i]      = null;
      btnMnem[i] = null;
      lblUnit[i] = null;
      lblDesc[i] = null;
    }
    iTotal    = 0;
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {

	if (event.getSource() == rbOFF) { iAction = OFF; }
	if (event.getSource() == rbON)  { iAction = ON; }

    for (int i=0; i<iTotal; i++)
    {
      if (event.getSource() == cb[i]) { setCheckBox( i, _CHECK ); }
      if (event.getSource() == btnMnem[i]) { setCurve( i ); }
    }

    setButtons();      // Enable or Disable buttons depending on data
    setDefinition();   // Build & Display ~{}_Definitions Section Text

    if (pTools != null) { pTools.toFront(); }
  }
}

/*
 *  @version 1.1 01/27/2010
 *  @author  John Victorine
 */
