/*
 * @las3CurvesFrame.java Version 1.1 01/27/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las3.gui;

import java.io.*;
import java.util.Observer;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;

import cmn.cmnStruct;                     // Global Common Data Structure

import iqstrat.iqstratStruct;             // Global IQSTRAT Data Structure
import iqstrat.iqstratControlStruct;      // Plot Control Data Structure
import iqstrat.iqstratRemarkListStruct;   // Remarks/Comments/Notes List Structure

import las.lasFileDataStruct;             // LAS File Data Structure

import las3.las3ListStruct;               // LAS 3 Data List Structure
import las3.las3Struct;                   // LAS 3 Data Structure
import las3.gui.las3CurvesLogPanel;       // LAS 3 Curves Log Panel

import rock.rockDataListStruct;           // Rock Data List Structure
import rock.rockDataStruct;               // Rock Data Structure

import brine.brineListStruct;             // Brine Data List Structure
import brine.brineStruct;                 // Brine Data Structure

import rock.rockImagesListStruct;         // Rock Images Data List Structure
import rock.rockImagesStruct;             // Rock Images Data Structure

import horizon.strat.stratListStruct;     // Stratigraphic Units List Structure
import horizon.strat.stratStruct;         // Stratigraphic Units Data Structure

import horizon.seq.seqListStruct;         // Sequence Stratigraphy List Structure
import horizon.seq.seqStruct;             // Sequence Stratigraphy Data Structure

import horizon.env.envListStruct;         // Sequence Stratigraphy List Structure
import horizon.env.envStruct;             // Sequence Stratigraphy Data Structure

import horizon.regions.regionsListStruct; // Perforations List Structure
import horizon.regions.regionsStruct;     // Perforations Data Structure

import horizon.bio.bioStratListStruct;    // Bio-Stratigraphic Units List Structure
import horizon.bio.bioStratStruct;        // Bio-Stratigraphic Units Data Structure

import pfeffer.pfefferDataListStruct;     // PfEFFER Data List Structure

import iqstrat.iqstratShaleListStruct;    // Shale Data List Structure
import iqstrat.iqstratShaleStruct;        // Shale Data Structure

import dst.dstListStruct;                 // DST Data List Structure
import dst.dstStruct;                     // DST Data Structure

/** Class las3CurvesFrame
 *  <p> This Class will display the Curves sections of the LAS 3 File and allow
 *      the user to select the curves they wish to display on the KGS web apps
 *      and to identify unknown curves with KGS Standard Curves.
 *
 *  @version 1.1 01/27/2010
 *  @author  John R. Victorine
 */

public class las3CurvesFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable         pNotifier = null; // Top Observables Object
  private iqstratStruct      stStruct  = null; // Global IQSTRAT Data Structure
  private cmnStruct          stCMN     = null; // Common Data Structure
  private las3ListStruct     stLAS3    = null; // LAS 3 File Data List Structure
  private lasFileDataStruct  stLAS     = null; // LAS File Data Structure

  // Required LAS 3.0 Data Sections default is all data.

  private boolean            bLog[]    = {true,true,true,true,true,
                                          true,true,true,true,true,
                                          true,true,true,true,true,
                                          true,true,true,true,true,
                                          true};

  // Global Variables

  private Observable              notifier    = null; // Observables Object
  private int                     iqstrato    = -1;

  private String                  sReadMethod = "";  // File Read Method
  private String                  sDirectory  = "";  // Directory Path
  private String                  sFilename   = "";  // File name of LAS File

  private int                     iLogs       = 0;
  private lasFileDataStruct       stLogs[]    = null; // LAS File Data Structure

  private int                     iRock       = 0;
  private rockDataListStruct      stRock[]    = null; // Rock Data List Structure

  private int                     iBrine      = 0;
  private brineListStruct         stBrine[]   = null; // Brine Data List Structure

  private int                     iImages     = 0;
  private rockImagesListStruct    stImages[]  = null; // Rock Images Data List Structure

  private int                     iIQTops     = -1;
  private int                     iTops       = 0;
  private stratListStruct         stTops[]    = null; // Stratigraphic Units List

  private int                     iPerf       = 0;
  private regionsListStruct       stPerf[]    = null; // Perforations List Struct

  private int                     iDST        = 0;
  private dstStruct               stDST[]     = null;  // DST Data Structure

  private int                     iBio        = 0;
  private bioStratListStruct      stBio[]     = null; // Bio-Stratigraphic Units List

  private int                     iControl    = 0;
  private iqstratControlStruct    stControl[] = null; // Control Data Structure

  private int                     iSeq        = 0;
  private seqListStruct           stSeq[]     = null; // Sequence Stratigraphy

  private int                     iEnv        = 0;
  private envListStruct           stEnv[]     = null; // Depositional Environment

  private int                     iFlow       = 0;
  private pfefferDataListStruct   stPfeffer[] = null; // Pfeffer List Structure

  private int                     iRemarks    = 0;
  private iqstratRemarkListStruct stRemarks[] = null; // Remarks List Structure

  private int                     iShale      = 0;
  private iqstratShaleListStruct  stShale[]   = null; // Shale List Structure

  private int                     iTotal      = 0;    // Total Number of Panels
  private las3CurvesLogPanel      pCurves[]   = null; // Curve Selection Panel
  private String                  sCurves[]   = null;
  private int                     iData[][]   = null; // LAS 3 Order, Type, Array

  // Use either step depth
  public static final int _SAME   = 0; // Both LAS Files have same step value
  // One Step Depth is greater than the other use the greater value as step depth
  public static final int _STEP_1 = 1; // Step value for 1 is less than 2
  public static final int _STEP_2 = 2; // Step value for 2 is less than 1
  // Both have 1 ft depth step, but one starts at 1/2 foot and the other
  // starts at a 1 foot depth value select the second file for interpolation.
  public static final int _SAME_I = 3; // Both LAS Files have same step value
  // Both have .5 or less step size, but one starts at a different depth than
  // the other need to normalize the depth
  public static final int _SAME_S = 4; // Both LAS Files have same step value

  private int    iStep  = _SAME;

  public static final int _NONE          = las3.las3Constants._NONE;

  public static final int _VERSION       = las3.las3Constants._VERSION;
  public static final int _WELL          = las3.las3Constants._WELL;
  public static final int _LOG           = las3.las3Constants._LOG;
  public static final int _CORE          = las3.las3Constants._CORE;
  public static final int _DRILLING      = las3.las3Constants._DRILLING;
  public static final int _INCLINOMETRY  = las3.las3Constants._INCLINOMETRY;
  public static final int _TOPS          = las3.las3Constants._TOPS;
  public static final int _TEST          = las3.las3Constants._TEST;
  public static final int _PERFORATION   = las3.las3Constants._PERFORATION;
  public static final int _IQ_CONTROL    = las3.las3Constants._IQ_CONTROL;
  public static final int _IQ_LOG        = las3.las3Constants._IQ_LOG;
  public static final int _IQ_CORE       = las3.las3Constants._IQ_CORE;
  public static final int _IQ_SEQ_STRAT  = las3.las3Constants._IQ_SEQ_STRAT;
  public static final int _IQ_FLOW       = las3.las3Constants._IQ_FLOW;
  public static final int _IQ_PFEFFER    = las3.las3Constants._IQ_PFEFFER;
  public static final int _IQ_GEO_REPORT = las3.las3Constants._IQ_GEO_REPORT;
  public static final int _IQ_SHALE      = las3.las3Constants._IQ_SHALE;
  public static final int _IQ_IMAGES     = las3.las3Constants._IQ_IMAGES;
  public static final int _IQ_BRINE      = las3.las3Constants._IQ_BRINE;
  public static final int _IQ_BIO        = las3.las3Constants._IQ_BIO;
  public static final int _IQ_ENV        = las3.las3Constants._IQ_ENV;

  public static final int _LAS_TYPE      = las3.gui.las3ToolsTable._LAS_TYPE;
  public static final int _CORE_TYPE     = las3.gui.las3ToolsTable._CORE_TYPE;
  public static final int _TOPS_TYPE     = las3.gui.las3ToolsTable._TOPS_TYPE;
  public static final int _PERF_TYPE     = las3.gui.las3ToolsTable._PERF_TYPE;
  public static final int _CNTRL_TYPE    = las3.gui.las3ToolsTable._CNTRL_TYPE;
  public static final int _SEQ_TYPE      = las3.gui.las3ToolsTable._SEQ_TYPE;
  public static final int _FLOW_TYPE     = las3.gui.las3ToolsTable._FLOW_TYPE;
  public static final int _GEO_TYPE      = las3.gui.las3ToolsTable._GEO_TYPE;
  public static final int _SHALE_TYPE    = las3.gui.las3ToolsTable._SHALE_TYPE;
  public static final int _IMAGES_TYPE   = las3.gui.las3ToolsTable._IMAGES_TYPE;
  public static final int _BRINE_TYPE    = las3.gui.las3ToolsTable._BRINE_TYPE;
  public static final int _BIO_TYPE      = las3.gui.las3ToolsTable._BIO_TYPE;
  public static final int _ENV_TYPE      = las3.gui.las3ToolsTable._ENV_TYPE;
  public static final int _DST_TYPE      = las3.gui.las3ToolsTable._DST_TYPE;

  // Global Frame Widgets

  private JButton btnContinue = new JButton();

  /** Constructor las3CurvesFrame()
   *  <p> This is the Constructor for this class.
   *  @param pNotifier = Observable
   *  @param stStruct  = Global IQSTRAT Data Structure
   *  @param stLAS     = LAS File Data Structure
   *  @param stLAS3    = LAS 3 File Data List Structure
   *  @param bLog      = Indicator to identify which Data Sections to
   *                     make available ( required data sections )
   */

  public las3CurvesFrame( Observable        pNotifier,
                          iqstratStruct     stStruct,
                          lasFileDataStruct stLAS,
                          las3ListStruct    stLAS3,
                          boolean           bLog[] )
  {
    try
    {
      this.pNotifier = pNotifier;
      this.stStruct  = stStruct;
      this.stCMN     = stStruct.stCMN;
      this.stLAS     = stLAS;
      this.stLAS3    = stLAS3;
      this.bLog      = bLog;

      if (stLAS != null)
      {
        sReadMethod = new String( stLAS.sReadMethod );  // File Read Method
        sDirectory  = new String( stLAS.sDirectory );   // Directory Path
        sFilename   = new String( stLAS.sFilename );    // File name of LAS File
      }

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int i=0;
    int j=0;
    int iCount = 0;

    JPanel      pnlBase    = new JPanel();
    JTabbedPane tabbedPane = new JTabbedPane();

    // Create an Observable

    notifier = new las3CurvesFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Build Frame

    this.setTitle("LAS File Curve Sections");
    this.getContentPane().setLayout(new BorderLayout());

    // Base Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnContinue.setFont(new java.awt.Font("Dialog", 1, 11));
    btnContinue.setText("Continue");
    btnContinue.addActionListener(this);

    // Center Panel

    tabbedPane.setFont(new java.awt.Font("Dialog", 1, 11));

    // Add Log Panel

    if (stLAS3 != null)
    {
      for (i=0; i<stLAS3.iCount; i++)
      {
        switch (stLAS3.stItem[i].iType)
        {
          case _NONE:
          case _VERSION:       // Version Informaiton
          case _WELL:          // Well Information
            break;
          case _LOG:           // Log Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iLogs++;
              iTotal++;
		    }
            break;
          case _CORE:          // Core Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iRock++;
              iTotal++;
			}
            break;
          case _TOPS:          // Tops Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              if (stLAS3.stItem[i].iqstrat == 1) { iIQTops = 1; }
              iTops++;
              iTotal++;
			}
            break;
          case _PERFORATION:   // Perforation Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iPerf++;
              iTotal++;
		    }
            break;
          case _DRILLING:      // Drilling Data Section
          case _INCLINOMETRY:  // Inclinometry Data Section
            break;
          case _TEST:          // Test Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iDST++;
              iTotal++;
		    }
            break;
          case _IQ_CONTROL:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iControl++;
              iTotal++;
		    }
            break;
          case _IQ_BIO:        // Bio-Stratigraphy Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iBio++;
              iTotal++;
		    }
            break;
          case _IQ_ENV:        // Depositional Environment Data Section
            if (bLog[stLAS3.stItem[i].iType])
            {
              iEnv++;
              iTotal++;
		    }
            break;
          case _IQ_SEQ_STRAT:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iSeq++;
              iTotal++;
		    }
            break;
          case _IQ_FLOW:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iFlow++;
              iTotal++;
		    }
            break;
          case _IQ_PFEFFER:
            break;
          case _IQ_LOG:
          case _IQ_CORE:
            break;
          case _IQ_GEO_REPORT:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iRemarks++;
              iTotal++;
		    }
            break;
          case _IQ_SHALE:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iShale++;
              iTotal++;
		    }
            break;
          case _IQ_IMAGES:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iImages++;
              iTotal++;
		    }
            break;
          case _IQ_BRINE:
            if (bLog[stLAS3.stItem[i].iType])
            {
              iBrine++;
              iTotal++;
		    }
            break;
        }
      }

      if (iLogs > 0)
      {
        stLogs = new lasFileDataStruct[iLogs];
        for (i=0; i<iLogs; i++)
          stLogs[i] = new lasFileDataStruct();
      }

      if (iRock > 0)
      {
        stRock = new rockDataListStruct[iRock];
        for (i=0; i<iRock; i++)
          stRock[i] = new rockDataListStruct();
      }

      if (iTops > 0)
      {
        stTops = new stratListStruct[iTops];
        for (i=0; i<iTops; i++)
          stTops[i] = new stratListStruct();
      }

      if (iPerf > 0)
      {
        stPerf = new regionsListStruct[iPerf];
        for (i=0; i<iPerf; i++)
          stPerf[i] = new regionsListStruct();
      }

      if (iDST > 0)
      {
        stDST = new dstStruct[iDST];
        for (i=0; i<iDST; i++)
          stDST[i] = new dstStruct();
      }

      if (iControl > 0)
      {
        stControl = new iqstratControlStruct[iControl];
        for (i=0; i<iControl; i++)
          stControl[i] = new iqstratControlStruct();
      }

      if (iBio > 0)
      {
        stBio = new bioStratListStruct[iBio];
        for (i=0; i<iBio; i++)
          stBio[i] = new bioStratListStruct();
      }

      if (iEnv > 0)
      {
        stEnv = new envListStruct[iEnv];
        for (i=0; i<iEnv; i++)
          stEnv[i] = new envListStruct();
      }

      if (iSeq > 0)
      {
        stSeq = new seqListStruct[iSeq];
        for (i=0; i<iSeq; i++)
          stSeq[i] = new seqListStruct();
      }

      if (iFlow > 0)
      {
        stPfeffer = new pfefferDataListStruct[iFlow];
        for (i=0; i<iFlow; i++)
          stPfeffer[i] = new pfefferDataListStruct();
      }
      if (iRemarks > 0)
      {
        stRemarks = new iqstratRemarkListStruct[iRemarks];
        for (i=0; i<iRemarks; i++)
          stRemarks[i] = new iqstratRemarkListStruct();
      }

      if (iShale > 0)
      {
        stShale = new iqstratShaleListStruct[iShale];
        for (i=0; i<iShale; i++)
          stShale[i] = new iqstratShaleListStruct();
      }

      if (iImages > 0)
      {
        stImages = new rockImagesListStruct[iImages];
        for (i=0; i<iImages; i++)
          stImages[i] = new rockImagesListStruct();
      }

      if (iBrine > 0)
      {
        stBrine = new brineListStruct[iBrine];
        for (i=0; i<iBrine; i++)
          stBrine[i] = new brineListStruct();
      }

      if (iTotal > 0)
      {
        pCurves = new las3CurvesLogPanel[iTotal];
        sCurves = new String[iTotal];
        iData   = new int[iTotal][3];   // LAS 3 Order, Type, Array

        for (i=0; i<iTotal; i++)
        {
          sCurves[i] = new String("");
          for (j=0; j<3; j++)
            iData[i][j] = -1;
        }

        for (i=0; i<stLAS3.iCount; i++)
        {
          iqstrato = -1;
          switch (stLAS3.stItem[i].iType)
          {
            case _LOG:         // Log Data Section
            case _CORE:        // Core Data Section
            case _PERFORATION: // Perforation Data Section
              break;
//            case _IQ_CONTROL:
//            case _IQ_FLOW:
            case _TOPS:        // Tops Data Section
              if (iIQTops == 1) iqstrato = 1;
              break;
          }

          switch (stLAS3.stItem[i].iType)
          {
            case _LOG:           // Log Data Section
            case _CORE:          // Core Data Section
            case _TOPS:          // Tops Data Section
            case _TEST:          // TEST Data Section
            case _PERFORATION:   // Perforation Data Section
            case _IQ_CONTROL:    // PROFILE - Control Plot Data
            case _IQ_SEQ_STRAT:  // PROFILE - Sequence Stratigraphy
            case _IQ_FLOW:       // PROFILE - PfEFFER Data
            case _IQ_GEO_REPORT: // PROFILE - Remarks/Comments/Notes
            case _IQ_SHALE:      // PROFILE - Shale Gamma Ray Limits
            case _IQ_IMAGES:     // PROFILE - Rock Images
            case _IQ_BRINE:      // PROFILE - Brine Data
            case _IQ_BIO:        // PROFILE - Bio Stratigraphy Data
            case _IQ_ENV:        // PROFILE - Depositional Environment Data
              if (bLog[stLAS3.stItem[i].iType])
              {
                if (iCount < iTotal)
                {
                  pCurves[iCount] = new las3CurvesLogPanel(
                      notifier, stStruct, stLAS,
                      stLAS3.dVersion, iqstrato, stLAS3.stItem[i] );

                  if (pCurves[iCount] != null)
                  {
                    sCurves[iCount] = new String( pCurves[iCount].getLabel() );
                    iData[iCount][0] = i;
                    iData[iCount][1] = pCurves[iCount].getTypeID();
                    iData[iCount][2] = pCurves[iCount].getArray();
                  }

                  tabbedPane.add(pCurves[iCount],  sCurves[iCount]);
                  iCount++;
				}
              }
              break;
          }
        }
      }
    }

    // Attach Widgets to Frame

    this.getContentPane().add(pnlBase, BorderLayout.SOUTH);
    pnlBase.add(btnContinue, null);

    this.getContentPane().add(tabbedPane, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(750, 550));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(true);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLAS3()
   * <p> This method will retrieve the LAS & Core Curve Selection and modify
   *     the LAS 3 File Data List Sturcture to retain the users selection and
   *     redefinition of the curve mnemonics to the KGS mnemonics
   * @return stList = modified LAS 3 File Data List Sturcture
   */

  public las3ListStruct getLAS3()
  {
    las3ListStruct stList      = null;
    las3Struct     st          = null;
    int            iSelected[] = null;

    stList = las3.las3Utility.copyList( stLAS3 );

    for (int i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _LAS_TYPE)
        {
          st     = las3.las3Utility.copy( pCurves[i].getLAS3() );
          stList = las3.las3Utility.modify( st, stList );
        }

        if (pCurves[i].getDataType() == _CORE_TYPE)
        {
          st     = las3.las3Utility.copy( pCurves[i].getLAS3() );
          stList = las3.las3Utility.modify( st, stList );
        }

        st.delete();
      }
    }

    return (stList);
  }

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @return stLAS = the LAS File Data Structure
   */

  public lasFileDataStruct getLAS()
  {
    int               i      = 0;
    int               iCount = 0;
    int               iRows  = 0;
    lasFileDataStruct st     = null;
    double            dNull  = 0.0;
    int               iTemp  = 0;
    double            dStep  = 0.0;
    double            dLogMin = 0.0;
    double            dLogMax = 0.0;

/* JRV 10/13/2017
    iTemp            = (int) stLAS.depthStart;
    stLAS.depthStart = (double) iTemp;

    iTemp            = 1 + (int) stLAS.depthEnd;
    stLAS.depthEnd   = (double) iTemp;

    dStep = 0.5;
    if (stLAS.depthStep < 0.5)
      dStep = 0.25;
    if (stLAS.depthStep < 0.25)
      dStep = 0.1;

    stLAS.depthStep  = dStep;
*/

    iRows = 1+(int)(Math.abs(stLAS.depthEnd-stLAS.depthStart)/stLAS.depthStep);

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (iCount < iLogs)
        {
          if ((pCurves[i].getDataType() == _LAS_TYPE) &&
              (pCurves[i].getAction() == cmn.cmnStruct.ON))
          {
            if (stLAS.iLogs == 0)
            {
              stLAS.sReadMethod = new String( sReadMethod ); // File Read Method
              stLAS.sDirectory  = new String( sDirectory );  // Directory Path
              stLAS.sFilename   = new String( sFilename );   // File name of LAS File

              stLAS.stLAS3      = new las3Struct[iLogs];
              stLAS.iLogs       = iLogs;

              stLAS = las3.las3LoadLogData.initData( iRows, stLAS );
            }

            stLAS = pCurves[i].getLAS( stLAS3.dVersion, iCount, stLAS );

            stLogs[iCount] = las.lasFileDataUtility.copyHeader( stLAS, stLogs[iCount] );
            stLogs[iCount] = pCurves[i].getLASData(
                 stLAS3.sDelim, stLAS3.sWrap, stLAS3.dVersion, iCount, stLogs[iCount] );
            stLAS          = las3.las3LoadLogData.mergeLogData( stLogs[iCount], stLAS );

            iCount++;
          }
        }
      }
    }

    if (iCount > 0)
    {
      if (stLogs != null)
      {
        for (i=0; i<iCount; i++)
        {
          if (stLogs[i] != null)
          {
            stLogs[i].delete();
            stLogs[i] = null;
          }
        }
        stLogs = null;
      }
    }

    if (stLAS != null)
    {
	  if (stLAS.iSP > -1)
	  {
        double dRange[] = math.mathLAS.computeCurveMinMax(stLAS.dSP, stLAS.dNull);
        stLAS.dSPMin = dRange[0];
        stLAS.dSPMax = dRange[1];
	  }

	  if (stLAS.iLIN_1 > -1)
	  {
        double dRange[] = math.mathLAS.computeCurveMinMax(stLAS.dLIN_1, stLAS.dNull);
        stLAS.dLIN_1Min = dRange[0];
        stLAS.dLIN_1Max = dRange[1];
	  }

	  if (stLAS.iLIN_2 > -1)
	  {
        double dRange[] = math.mathLAS.computeCurveMinMax(stLAS.dLIN_2, stLAS.dNull);
        stLAS.dLIN_2Min = dRange[0];
        stLAS.dLIN_2Max = dRange[1];
	  }

	  if (stLAS.iLIN_3 > -1)
	  {
        double dRange[] = math.mathLAS.computeCurveMinMax(stLAS.dLIN_3, stLAS.dNull);
        stLAS.dLIN_3Min = dRange[0];
        stLAS.dLIN_3Max = dRange[1];
	  }

	  if (stLAS.iLIN_4 > -1)
	  {
        double dRange[] = math.mathLAS.computeCurveMinMax(stLAS.dLIN_4, stLAS.dNull);
        stLAS.dLIN_4Min = dRange[0];
        stLAS.dLIN_4Max = dRange[1];
	  }

	  if (stLAS.iLOG_1 > -1)
	  {
        double dRange[] = math.mathLAS.computeLogMinMax(stLAS.dLOG_1, stLAS.dNull);
        stLAS.dLOG_1Min = dRange[0];
        stLAS.dLOG_1Max = dRange[1];
	  }

	  if (stLAS.iLOG_2 > -1)
	  {
        double dRange[] = math.mathLAS.computeLogMinMax(stLAS.dLOG_2, stLAS.dNull);
        stLAS.dLOG_2Min = dRange[0];
        stLAS.dLOG_2Max = dRange[1];
	  }

	  if (stLAS.iLOG_3 > -1)
	  {
        double dRange[] = math.mathLAS.computeLogMinMax(stLAS.dLOG_3, stLAS.dNull);
        stLAS.dLOG_3Min = dRange[0];
        stLAS.dLOG_3Max = dRange[1];
	  }

	  if (stLAS.iLOG_4 > -1)
	  {
        double dRange[] = math.mathLAS.computeLogMinMax(stLAS.dLOG_4, stLAS.dNull);
        stLAS.dLOG_4Min = dRange[0];
        stLAS.dLOG_4Max = dRange[1];
	  }

	  if ((stLAS.iLOG_1 > -1) || (stLAS.iLOG_2 > -1) ||
	      (stLAS.iLOG_3 > -1) || (stLAS.iLOG_4 > -1))
	  {
		dLogMin = 0.1;
		dLogMax = 1000.0;

		if (stLAS.iLOG_1 > -1)
		{
		  if (stLAS.dLOG_1Min < dLogMin)
		  {
		    dLogMin = stLAS.dLOG_1Min;
		  }

		  if (stLAS.dLOG_1Max > dLogMax)
		  {
		    dLogMax = stLAS.dLOG_1Max;
		  }
		}

		if (stLAS.iLOG_2 > -1)
		{
		  if (stLAS.dLOG_2Min < dLogMin)
		  {
		    dLogMin = stLAS.dLOG_2Min;
		  }

		  if (stLAS.dLOG_2Max > dLogMax)
		  {
		    dLogMax = stLAS.dLOG_2Max;
		  }
		}

		if (stLAS.iLOG_3 > -1)
		{
		  if (stLAS.dLOG_3Min < dLogMin)
		  {
		    dLogMin = stLAS.dLOG_3Min;
		  }

		  if (stLAS.dLOG_3Max > dLogMax)
		  {
		    dLogMax = stLAS.dLOG_3Max;
		  }
		}

		if (stLAS.iLOG_4 > -1)
		{
		  if (stLAS.dLOG_4Min < dLogMin)
		  {
		    dLogMin = stLAS.dLOG_4Min;
		  }

		  if (stLAS.dLOG_4Max > dLogMax)
		  {
		    dLogMax = stLAS.dLOG_4Max;
		  }
		}

        stLAS.dLOG_1Min = dLogMin;
        stLAS.dLOG_1Max = dLogMax;


        stLAS.dLOG_2Min = dLogMin;
        stLAS.dLOG_2Max = dLogMax;

        stLAS.dLOG_3Min = dLogMin;
        stLAS.dLOG_3Max = dLogMax;

        stLAS.dLOG_4Min = dLogMin;
        stLAS.dLOG_4Max = dLogMax;
	  }
	}

    return (stLAS);
  }

  /** Method getRockData()
   * <p> This method will retrieve the Rock Data List Structure
   * @return stRock = the Rock Data List Structure
   */

  public rockDataListStruct getRockData()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    rockDataListStruct stR = null;
    double             dLogMin = 0.0;
    double             dLogMax = 0.0;
    double             data[]  = null;
    double             dRange[] = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _CORE_TYPE)

        {
          if (iCount < iRock)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stRock[iCount] =
                rock.rockDataUtility.copyList( pCurves[i].getRockData() );
		    }
            iCount++;
          }
        }
      }
    }

    if (iCount == 1)
    {
      stR = rock.rockDataUtility.copyList( stRock[0] );
    }
    else if (iCount > 1)
    {
      for (i=0; i<iCount; i++)
      {
        iSamples = iSamples + stRock[i].iCount;
      }

      if (iSamples > 0)
      {
        stR        = new rockDataListStruct();
        stR.stItem = new rockDataStruct[iSamples];
        stR.iCount = iSamples;

        for (i=0; i<iSamples; i++)
        {
          stR.stItem[i] = new rockDataStruct();
        }
      }

      for (i=0; i<iCount; i++)
      {
        stR = las3.las3LoadCoreData.mergeCoreData( stRock[i], stR );
      }
    }

    if (stR != null)
    {
	  if (stR._LIN_1 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LIN_1 );
        dRange = math.mathLAS.computeCurveMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLIN_1Min = dRange[0];
        stR.dLIN_1Max = dRange[1];
//System.out.println("LINEAR 1 "+ stR.dLIN_1Min +" "+ stR.dLIN_1Max);
	  }

	  if (stR._LIN_2 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LIN_2 );
        dRange = math.mathLAS.computeCurveMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLIN_2Min = dRange[0];
        stR.dLIN_2Max = dRange[1];
//System.out.println("LINEAR 2 "+ stR.dLIN_2Min +" "+ stR.dLIN_2Max);
	  }

	  if (stR._LIN_3 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LIN_3 );
        dRange = math.mathLAS.computeCurveMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLIN_3Min = dRange[0];
        stR.dLIN_3Max = dRange[1];
	  }

	  if (stR._LIN_4 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LIN_4 );
        dRange = math.mathLAS.computeCurveMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLIN_4Min = dRange[0];
        stR.dLIN_4Max = dRange[1];
	  }

	  if (stR._LOG_1 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LOG_1 );
        dRange = math.mathLAS.computeLogMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLOG_1Min = dRange[0];
        stR.dLOG_1Max = dRange[1];
//System.out.println("LOGa 1 "+ stR.dLOG_1Min +" "+ stR.dLOG_1Max);
	  }

	  if (stR._LOG_2 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LOG_2 );
        dRange = math.mathLAS.computeLogMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLOG_2Min = dRange[0];
        stR.dLOG_2Max = dRange[1];
	  }

	  if (stR._LOG_3 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LOG_3 );
        dRange = math.mathLAS.computeLogMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLOG_3Min = dRange[0];
        stR.dLOG_3Max = dRange[1];
	  }

	  if (stR._LOG_4 > -1)
	  {
		data   = stR.getData( rock.rockStandardTools._LOG_4 );
        dRange = math.mathLAS.computeLogMinMax(data, rock.rockDataStruct.dNULL);
        stR.dLOG_4Min = dRange[0];
        stR.dLOG_4Max = dRange[1];
	  }

	  if ((stR._LOG_1 > -1) || (stR._LOG_2 > -1) ||
	      (stR._LOG_3 > -1) || (stR._LOG_4 > -1))
	  {
		dLogMin = 0.1;
		dLogMax = 1000.0;

		if (stR._LOG_1 > -1)
		{
		  if (stR.dLOG_1Min < dLogMin)
		  {
		    dLogMin = stR.dLOG_1Min;
		  }

		  if (stR.dLOG_1Max > dLogMax)
		  {
		    dLogMax = stR.dLOG_1Max;
		  }
		}

		if (stR._LOG_2 > -1)
		{
		  if (stR.dLOG_2Min < dLogMin)
		  {
		    dLogMin = stR.dLOG_2Min;
		  }

		  if (stR.dLOG_2Max > dLogMax)
		  {
		    dLogMax = stR.dLOG_2Max;
		  }
		}

		if (stR._LOG_3 > -1)
		{
		  if (stR.dLOG_3Min < dLogMin)
		  {
		    dLogMin = stR.dLOG_3Min;
		  }

		  if (stR.dLOG_3Max > dLogMax)
		  {
		    dLogMax = stR.dLOG_3Max;
		  }
		}

		if (stR._LOG_4 > -1)
		{
		  if (stR.dLOG_4Min < dLogMin)
		  {
		    dLogMin = stR.dLOG_4Min;
		  }

		  if (stR.dLOG_4Max > dLogMax)
		  {
		    dLogMax = stR.dLOG_4Max;
		  }
		}

        stR.dLOG_1Min = dLogMin;
        stR.dLOG_1Max = dLogMax;


        stR.dLOG_2Min = dLogMin;
        stR.dLOG_2Max = dLogMax;

        stR.dLOG_3Min = dLogMin;
        stR.dLOG_3Max = dLogMax;

        stR.dLOG_4Min = dLogMin;
        stR.dLOG_4Max = dLogMax;
	  }
    }

    if (iCount > 0)
    {
      if (stRock != null)
      {
        for (i=0; i<iCount; i++)
        {
          if (stRock[i] != null)
          {
            stRock[i].delete();
            stRock[i] = null;
          }
        }
        stRock = null;
      }
    }

    return (stR);
  }

  /** Method getBrine()
   * <p> This method will retrieve the Brine Data List Structure
   * @return st = the Brine Data List Structure
   */

  public brineListStruct getBrine()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    brineListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _BRINE_TYPE)

        {
          if (iCount < iBrine)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stBrine[iCount] = brine.brineUtility.copyList( pCurves[i].getBrine() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stBrine != null)
    {
      if (stBrine[0] != null)
      {
        stR = brine.brineUtility.copyList( stBrine[0] );
        stBrine[0].delete();
        stBrine[0] = null;
      }
      stBrine = null;
    }

    return (stR);
  }

  /** Method getImages()
   * <p> This method will retrieve the Rock Images List Structure
   * @return st = the Rock Images List Structure
   */

  public rockImagesListStruct getImages()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    rockImagesListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _IMAGES_TYPE)

        {
          if (iCount < iImages)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stImages[iCount] = rock.rockImagesUtility.copyList( pCurves[i].getImages() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stImages != null)
    {
      if (stImages[0] != null)
      {
        stR = rock.rockImagesUtility.copyList( stImages[0] );
        stImages[0].delete();
        stImages[0] = null;
      }
      stImages = null;
    }

    return (stR);
  }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Units Data Structure
   * @return st = the Stratigraphic Units Data List Structure
   */

  public stratListStruct getTops()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    stratListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _TOPS_TYPE)

        {
          if (iCount < iTops)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stTops[iCount] =
                horizon.strat.stratUtility.copyList( pCurves[i].getTops() );
			}
            iCount++;
          }
        }
      }
    }

    if (iCount == 1)
    {
      stR = horizon.strat.stratUtility.copyList( stTops[0] );
    }
    else if (iCount > 1)
    {
      for (i=0; i<iCount; i++)
      {
        if (stTops[i] != null)
          iSamples = iSamples + stTops[i].iCount;
      }

      if (iSamples > 0)
      {
        stR        = new stratListStruct();
        stR.stItem = new stratStruct[iSamples];
        stR.iCount = iSamples;

        for (i=0; i<iSamples; i++)
        {
          stR.stItem[i] = new stratStruct();
          stR.stItem[i].depthStart = stLAS.dNull;
        }
      }

      for (i=0; i<iCount; i++)
      {
        if (stTops[i] != null)
        {
          stR = las3.las3LoadTopsData.mergeTopsData( stLAS.dNull, stTops[i], stR );
	    }
      }
    }

    if (iCount > 0)
    {
      if (stTops != null)
      {
        for (i=0; i<iCount; i++)
        {
          if (stTops[i] != null)
          {
            stTops[i].delete();
            stTops[i] = null;
          }
        }
        stTops = null;
      }
    }

    return (stR);
  }

  /** Method getPerf()
   * <p> This method will retrieve the Perforations Data Structure
   * @return st = the Perforations Data List Structure
   */

  public regionsListStruct getPerf()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    regionsListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _PERF_TYPE)

        {
          if (iCount < iPerf)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stPerf[iCount] =
                horizon.regions.regionsUtility.copyList( pCurves[i].getPerf() );
		    }
            iCount++;
          }
        }
      }
    }

    if (iCount == 1)
    {
      stR = horizon.regions.regionsUtility.copyList( stPerf[0] );
    }
    else if (iCount > 1)
    {
      for (i=0; i<iCount; i++)
      {
        iSamples = iSamples + stPerf[i].iCount;
      }

      if (iSamples > 0)
      {
        stR        = new regionsListStruct();
        stR.stItem = new regionsStruct[iSamples];
        stR.iCount = iSamples;

        for (i=0; i<iSamples; i++)
        {
          stR.stItem[i] = new regionsStruct();
          stR.stItem[i].depth_top = stLAS.dNull;
        }
      }

      for (i=0; i<iCount; i++)
      {
        stR = las3.las3LoadPerfData.mergePerfData( stLAS.dNull, stPerf[i], stR );
      }
    }

    if (iCount > 0)
    {
      if (stPerf != null)
      {
        for (i=0; i<iCount; i++)
        {
          if (stPerf[i] != null)
          {
            stPerf[i].delete();
            stPerf[i] = null;
          }
        }
        stPerf = null;
      }
    }

    return (stR);
  }

  /** Method getDST()
   * <p> This method will retrieve the DST Data List Structure
   * @return st = the DST Data List Structure
   */

  public dstListStruct getDST()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    dstListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _DST_TYPE)

        {
          if (iCount < iDST)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stDST[iCount] = dst.dstUtility.copy( pCurves[i].getDST() );
              stR = dst.dstUtility.add( stDST[iCount], stR );
		    }
            iCount++;
          }
        }
      }
    }

    if (iCount > 0)
    {
      if (stDST != null)
      {
        for (i=0; i<iCount; i++)
        {
          if (stDST[i] != null)
          {
            stDST[i].delete();
            stDST[i] = null;
          }
        }
        stDST = null;
      }
    }

    return (stR);
  }

  /** Method getBio()
   * <p> This method will retrieve the Bio Stratigraphy List Structure
   * @return st = the Bio Stratigraphy List Structure
   */

  public bioStratListStruct getBio()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    bioStratListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _BIO_TYPE)

        {
          if (iCount < iBio)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stBio[iCount] = horizon.bio.bioStratUtility.copyList( pCurves[i].getBio() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stBio != null)
    {
      if (stBio[0] != null)
      {
        stR = horizon.bio.bioStratUtility.copyList( stBio[0] );
        stBio[0].delete();
        stBio[0] = null;
      }
      stBio = null;
    }

    return (stR);
  }

  /** Method getControl()
   * <p> This method will retrieve the Control Data Structure
   * @return st = the Control Data Structure
   */

  public iqstratControlStruct getControl()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    iqstratControlStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _CNTRL_TYPE)

        {
          if (iCount < iControl)
          {
		    if (pCurves[i].getAction() == cmn.cmnStruct.ON)
		    {
              stControl[iCount] =
                iqstrat.iqstratControlUtility.copy( pCurves[i].getControl() );
			}
            iCount++;
          }
        }
      }
    }

    if (stControl != null)
    {
      if (stControl[0] != null)
      {
        stR = iqstrat.iqstratControlUtility.copy( stControl[0] );
        stControl[0].delete();
        stControl[0] = null;
      }
      stControl = null;
    }

    return (stR);
  }

  /** Method getSeq()
   * <p> This method will retrieve the Sequence Stratigraphy List Structure
   * @return st = the Sequence Stratigraphy List Structure
   */

  public seqListStruct getSeq()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    seqListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _SEQ_TYPE)

        {
          if (iCount < iSeq)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stSeq[iCount] = horizon.seq.seqUtility.copyList( pCurves[i].getSeq() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stSeq != null)
    {
      if (stSeq[0] != null)
      {
        stR = horizon.seq.seqUtility.copyList( stSeq[0] );
        stSeq[0].delete();
        stSeq[0] = null;
      }
      stSeq = null;
    }

    return (stR);
  }

  /** Method getEnv()
   * <p> This method will retrieve the Depositional Environment List Structure
   * @return st = the Depositional Environment List Structure
   */

  public envListStruct getEnv()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    envListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _ENV_TYPE)

        {
          if (iCount < iEnv)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stEnv[iCount] = horizon.env.envUtility.copyList( pCurves[i].getEnv() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stEnv != null)
    {
      if (stEnv[0] != null)
      {
        stR = horizon.env.envUtility.copyList( stEnv[0] );
        stEnv[0].delete();
        stEnv[0] = null;
      }
      stEnv = null;
    }

    return (stR);
  }

  /** Method getPfeffer()
   * <p> This method will retrieve the PfEFFER Data List Structure
   * @return st = the PfEFFER Data List Structure
   */

  public pfefferDataListStruct getPfeffer()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    pfefferDataListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _FLOW_TYPE)

        {
          if (iCount < iControl)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stPfeffer[iCount] =
                pfeffer.pfefferDataUtility.copyList( pCurves[i].getPfeffer() );
		    }
            iCount++;
          }
        }
      }
    }

    if (stPfeffer != null)
    {
      if (stPfeffer[0] != null)
      {
        stR = pfeffer.pfefferDataUtility.copyList( stPfeffer[0] );
        stPfeffer[0].delete();
        stPfeffer[0] = null;
      }
      stPfeffer = null;
    }

    return (stR);
  }

  /** Method getRemarks()
   * <p> This method will retrieve the Remarks Data List Structure
   * @return st = the Remarks Data List Structure
   */

  public iqstratRemarkListStruct getRemarks()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    iqstratRemarkListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _GEO_TYPE)

        {
          if (iCount < iRemarks)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stRemarks[iCount] =
                iqstrat.iqstratRemarkUtility.copyList( pCurves[i].getRemarks() );
			}
            iCount++;
          }
        }
      }
    }

    if (stRemarks != null)
    {
      if (stRemarks[0] != null)
      {
        stR = iqstrat.iqstratRemarkUtility.copyList( stRemarks[0] );
        stRemarks[0].delete();
        stRemarks[0] = null;
      }
      stRemarks = null;
    }

    return (stR);
  }

  /** Method getShale()
   * <p> This method will retrieve the Shale Data List Structure
   * @return st = the Shale Data List Structure
   */

  public iqstratShaleListStruct getShale()
  {
    int i        = 0;
    int iCount   = 0;
    int iSamples = 0;
    iqstratShaleListStruct stR = null;

    for (i=0; i<iTotal; i++)
    {
      if (pCurves[i] != null) // Curve Selection Panel
      {
        if (pCurves[i].getDataType() == _SHALE_TYPE)

        {
          if (iCount < iShale)
          {
			if (pCurves[i].getAction() == cmn.cmnStruct.ON)
			{
              stShale[iCount] =
                iqstrat.iqstratShaleUtility.copyList( pCurves[i].getShale() );
			}
            iCount++;
          }
        }
      }
    }

    if (stShale != null)
    {
      if (stShale[0] != null)
      {
        stR = iqstrat.iqstratShaleUtility.copyList( stShale[0] );
        stShale[0].delete();
        stShale[0] = null;
      }
      stShale = null;
    }

    return (stR);
  }

  /* ===================================================================== *
   * ---------------------------- COMPUTE -------------------------------- *
   * ===================================================================== */

  /** Method computeDepths()
   * <p> This method will compute the initial depth information for the
   *     new LAS File
   * @param  iCount = total number of LAS File Data Structures
   * @return st     = LAS File Data Structure
   */

  private lasFileDataStruct computeDepths( int iCount )
  {
    lasFileDataStruct st = new lasFileDataStruct();
    int     i,j;
    double  dTemp      = 0.0;
    int     iSteps     = 0;
    int     iStart[]   = null;
    double  dStart[]   = null;
    double  dStep[]    = null;
    int     iStepVal[] = null;
    // Allowed Step Size  1.0, 0.5, 0.1, other
    int     iValue[]   = { 0,   0,   0,  0 };

    int     iDepth     = 0;
    double  diff       = 0.0;

    if (iCount > 0)
    {
      iStart   = new int[iCount];
      dStart   = new double[iCount];
      dStep    = new double[iCount];
      iStepVal = new int[iCount];

      for (i=0; i<iCount; i++)
      {
        iStart[i]   = 0;
        dStart[i]   = 0.0;
        dStep[i]    = 0.0;
        iStepVal[i] = _SAME;

        if (stLogs[i].depthStart > stLogs[i].depthEnd)
        {
          dTemp                = stLogs[i].depthStart;
          stLogs[i].depthStart = stLogs[i].depthEnd;
          stLogs[i].depthEnd   = dTemp;

          stLogs[i].depthStep  = Math.abs(stLogs[i].depthStep);
        }

        dStart[i] = stLogs[i].depthStart;
        iStart[i] = (int) dStart[i];
        dStart[i] = dStart[i] - (double) iStart[i];
        dStep[i]  = stLogs[i].depthStep;

        if (dStep[i] == 1.0)      iValue[0]++;
        else if (dStep[i] == 0.5) iValue[1]++;
        else if (dStep[i] == 0.1) iValue[2]++;
        else                      iValue[3]++;

        if (i==0)
        {
          st.depthStart = stLogs[0].depthStart;
          st.depthEnd   = stLogs[0].depthEnd;
        }

        dTemp  = stLogs[i].depthStart;
        if (dTemp < st.depthStart) st.depthStart = dTemp;

        dTemp  = stLogs[i].depthEnd;
        if (dTemp > st.depthEnd) st.depthEnd = dTemp;
      }
    }

    st.depthStep = dStep[0];

    if (iValue[3] > 0)
    {
      for (i=0; i<iCount; i++)
      {
        if (dStep[i] < 1.0) st.depthStep = 1.0;
        if (dStep[i] < 0.5) st.depthStep = 0.5;
        if (dStep[i] < 0.1) st.depthStep = 0.1;
      }
    }

    if ((iValue[0] > 0) && (iValue[1] > 0))
      st.depthStep = 1.0;

    if ((iValue[0] > 0) && (iValue[2] > 0))
      st.depthStep = 1.0;

    if ((iValue[1] > 0) && (iValue[2] > 0))
      st.depthStep = 0.5;

    iDepth = (int) st.depthStart * 10;
    diff   = (st.depthStart * 10.0 - (double) iDepth) / 10.0;

    if (st.depthStart < 0.0) st.depthStart = 0.0;
    else if (diff != 0.0)    st.depthStart = (double) iDepth;

    st.iRows = 1 + (int)( Math.abs(st.depthEnd-st.depthStart) / st.depthStep );

    return (st);
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    int i=0;

    // Input Variables

    pNotifier = null; // Top Observables Object
    stStruct  = null; // Global IQSTRAT Data Structure
    stCMN     = null; // Common Data Structure
    stLAS3    = null; // LAS 3 File Data List Structure
    stLAS     = null; // LAS File Data Structure

    if (stRock != null)
    {
      for (i=0; i<iRock; i++)
      {
        if (stRock[i] != null)
          stRock[i].delete();
        stRock[i] = null; // Rock Data List Structure
      }
    }
    stRock = null;
    iRock  = 0;

    if (stLogs != null)
    {
      for (i=0; i<iLogs; i++)
      {
        if (stLogs[i] != null)
          stLogs[i].delete();
        stLogs[i] = null;
      }
    }
    stLogs = null;
    iLogs  = 0;

    // Global Variables

    sReadMethod = null;  // File Read Method
    sDirectory  = null;  // Directory Path
    sFilename   = null;  // File name of LAS File

    if (stBrine != null)
    {
      for (i=0; i<iBrine; i++)
      {
        if (stBrine[i] != null)
          stBrine[i].delete();
        stBrine[i] = null;
      }
    }
    stBrine = null;
    iBrine  = 0;

    if (stTops != null)
    {
      for (i=0; i<iTops; i++)
      {
        if (stTops[i] != null)
          stTops[i].delete();
        stTops[i] = null;
      }
    }
    stTops = null;
    iTops  = 0;

    if (stSeq != null)
    {
      for (i=0; i<iSeq; i++)
      {
        if (stSeq[i] != null)
          stSeq[i].delete();
        stSeq[i] = null;
      }
    }
    stSeq = null;
    iSeq  = 0;

    if (stPerf != null)
    {
      for (i=0; i<iPerf; i++)
      {
        if (stPerf[i] != null)
          stPerf[i].delete();
        stPerf[i] = null;
      }
    }
    stPerf = null;
    iPerf  = 0;

    if (stDST != null)
    {
      for (i=0; i<iDST; i++)
      {
        if (stDST[i] != null)
          stDST[i].delete();
        stDST[i] = null;
      }
    }
    stDST = null;
    iDST  = 0;

    if (stBio != null)
    {
      for (i=0; i<iBio; i++)
      {
        if (stBio[i] != null)
          stBio[i].delete();
        stBio[i] = null;
      }
    }
    stBio = null;
    iBio  = 0;

    if (stControl != null)
    {
      for (i=0; i<iControl; i++)
      {
        if (stControl[i] != null)
          stControl[i].delete();
        stControl[i] = null;
      }
    }
    stControl = null;
    iControl  = 0;

    if (stPfeffer != null)
    {
      for (i=0; i<iFlow; i++)
      {
        if (stPfeffer[i] != null)
          stPfeffer[i].delete();
        stPfeffer[i] = null;
      }
    }
    stPfeffer = null;
    iFlow     = 0;

    if (stRemarks != null)
    {
      for (i=0; i<iRemarks; i++)
      {
        if (stRemarks[i] != null)
          stRemarks[i].delete();
        stRemarks[i] = null;
      }
    }
    stRemarks = null;
    iRemarks  = 0;

    if (stShale != null)
    {
      for (i=0; i<iShale; i++)
      {
        if (stShale[i] != null)
          stShale[i].delete();
        stShale[i] = null;
      }
    }
    stShale = null;
    iShale  = 0;

    notifier = null;   // Observables Object

    for (i=0; i<iTotal; i++)
      pCurves[i] = null;   // Curve Selection Panel

    sCurves = null;
    iData   = null;   // LAS 3 Order, Type, Array
    iTotal  = 0;      // Total Number of Panels

    // Global Frame Widgets

    btnContinue = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnContinue)
    {
      if (pNotifier != null)
        pNotifier.notifyObservers(new String("LAS 3 - Curves Selected"));
    }
  }

  /* ===================================================================== *
   * ------------------------ OBSERVER ACTIONS --------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("Get LAS Standard Tool"))
    {
      for (int i=0; i<iTotal; i++)
      {
        if (pCurves[i] != null) pCurves[i].getCurve();
      }
    }
  }
}

/** CLASS las3CurvesFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class las3CurvesFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 01/27/2010
 *  @author  John Victorine
 */
