/*
 * @lasPlotTrack.java Version 1.1 09/04/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.plot;

import java.awt.*;
import java.awt.geom.*;

import las.lasFileDataStruct;

/** CLASS lasPlotTrack
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot LAS Tracks.
 *
 *  @version 1.1 09/04/2007
 *  @author  John R. Victorine
 */

public class lasPlotTrack extends Canvas
{
  // Input Variables

  private boolean bSonic     = false; // Indicator to plot by two-way time
  private int     iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double  depthStart = 0.0;   // Starting Depth
  private double  depthEnd   = 0.0;   // Ending Depth
  private int     iScale     = -1;    // Plot Scale

  // Global Variables

  public static final Color cRED     = new Color( 180,   0,   0 );
  public static final Color cGREEN   = new Color( 0,   180,   0 );
  public static final Color cBLUE    = new Color( 0,     0, 180 );
  public static final Color cCYAN    = new Color(  0,  180, 180 );
  public static final Color cMAGENTA = new Color( 180,   0, 180 );
  public static final Color cYELLOW  = new Color( 180, 180,   0 );
  public static final Color cORANGE  = new Color( 204, 102,   0 );
  public static final Color cBROWN   = new Color( 102, 102,   0 );
//  public static final Color cBROWN   = new Color(  51,  51,   0 );

  public static final Color COLORS[] =  // Color for Log Curves
      { cRED, cGREEN, cBLUE, cBROWN, cMAGENTA, cORANGE, cYELLOW, cCYAN };
//  public static final Color COLORS[] =  // Color for Log Curves
//            { Color.red,     Color.green,  Color.blue,   Color.cyan,
//              Color.magenta, Color.yellow, Color.orange, Color.pink };

  private lasFileDataStruct stLASFileData  = null; // LAS File Data Structure

  private int iLASSize = iqstrat.iqstratTracksStruct._DOUBLE;

  private double dLIMITS[][] = null; // Minimum & Maximum Limits to LAS Plot Track

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR lasPlotTrack( int iDataType,
   *                            double depthStart,
   *                            double depthEnd,
   *                            int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lasPlotTrack( int iDataType,
                       double depthStart,
                       double depthEnd,
                       int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR lasPlotTrack( boolean bSonic,
   *                            int iDataType,
   *                            double depthStart,
   *                            double depthEnd,
   *                            int iScale )
   *  <p> This is the constructor for this class.
   *  @param bSonic      = indicator to plot by depth or two-way time
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lasPlotTrack( boolean bSonic,
                       int iDataType,
                       double depthStart,
                       double depthEnd,
                       int iScale )
  {
	this.bSonic    = bSonic;
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
	stLASFileData = null;
	dLIMITS       = null; // Minimum & Maximum Limits to Profile Plot
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setLASFileData( lasFileDataStruct stLASFileData )
   * <p> This method will set the LAS File Data Structure
   * @param stLASFileData - LAS File Data Structure
   */

  public void setLASFileData(lasFileDataStruct stLASFileData)
  {
    this.stLASFileData = stLASFileData;
  }

  /** Method setLASTrackSize(int iLASSize)
   * <p> This method will set the LAS File Track Width
   * @param iLASSize = LAS File Track Width
   */

  public void setLASTrackSize(int iLASSize) { this.iLASSize = iLASSize; }

  /** Method setPlotLimits()
   *  <p> This method will override the default Plot Limits
   *  @param dLIMITS = Array of Minimum & Maximum Plot Limits
   */

  public void setPlotLimits( double dLIMITS[][] ) {	this.dLIMITS = dLIMITS; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iLAS        = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/4;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.LAS_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.LAS_TRACK[iTrack][0];
    int    iValue      = 0;
    int    iY1         = 0;
    int    iY2         = 0;

    double dMinimum = cmn.cmnString.stringToDouble(
                        las.lasStandardTools.LAS_TOOLS[iCurve][4]);
    double dMaximum = cmn.cmnString.stringToDouble(
                        las.lasStandardTools.LAS_TOOLS[iCurve][5]);

    if (dLIMITS != null)
    {
	  dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	  dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	}

    if (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("OHM-M"))
    {
       iLog   = 1;
       dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
       dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
       iCycle = (int) dMaximum - (int) dMinimum;
    }

    if (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("MD"))
    {
      iLog   = 1;
      dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
      dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
      iCycle = (int) dMaximum - (int) dMinimum;
    }

    if (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("LOG_RATIO"))
    {
      iLog   = 1;
      dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
      dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
      iCycle = (int) dMaximum - (int) dMinimum;
    }

    if (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("UNL"))
    {
      if (iCurve == las.lasStandardTools._LOG_1)
      {
        dMinimum = stLASFileData.dLOG_1Min;
        dMaximum = stLASFileData.dLOG_1Max;

        if (dLIMITS != null)
        {
   	      dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	      dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	    }
      }

      if (iCurve == las.lasStandardTools._LOG_2)
      {
        dMinimum = stLASFileData.dLOG_2Min;
        dMaximum = stLASFileData.dLOG_2Max;

        if (dLIMITS != null)
        {
   	      dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	      dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	    }
      }

      if (iCurve == las.lasStandardTools._LOG_3)
      {
        dMinimum = stLASFileData.dLOG_3Min;
        dMaximum = stLASFileData.dLOG_3Max;

        if (dLIMITS != null)
        {
   	      dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	      dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	    }
      }

      if (iCurve == las.lasStandardTools._LOG_4)
      {
        dMinimum = stLASFileData.dLOG_4Min;
        dMaximum = stLASFileData.dLOG_4Max;

        if (dLIMITS != null)
        {
   	      dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	      dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	    }
      }

      iLog   = 1;
      dMinimum = Math.round(Math.log(dMinimum) / Math.log(10));
      dMaximum = Math.round(Math.log(dMaximum) / Math.log(10));
      iCycle = (int) dMaximum - (int) dMinimum;
    }

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.gray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.gray);
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

//        g.setColor(Color.lightGray);
        g.setColor(new Color(230, 230, 230));
        for (j=1; j<5; j++)
        {
          dTemp = (double) iIncrement / 5.0;
          jXinc = i + iStartTrack + j * (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }
    else
    {
      iIncrement = iWidthTrack / iCycle;

      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.gray);
        if (i==0)
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

        g.setColor(Color.gray);
//        g.drawLine(i+iStartTrack, HEIGHT_START,
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);

        int    jStart = 1;
        int    jEnd   = 5;
        double dV     = 2.0;

        if (iLASSize == iqstrat.iqstratTracksStruct._DOUBLE)
        {
          jStart = 2;
          jEnd   = 10;
          dV     = 1.0;
		}

        g.setColor(Color.lightGray);
//        for (j=1; j<5; j++)
        for (j=jStart; j<jEnd; j++)
        {
//          dXinc = (double) j * 2.0;
          dXinc = (double) j * dV;
          dTemp = (double) iIncrement * Math.log(dXinc) / Math.log(10.0);
          jXinc = i + iStartTrack + (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawTrackLabels( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackLabels( Graphics g,
                               int iLAS,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    i=0;
    int    iCurve      = -1;
    int    iContinue   = 0;
    int    iLine       = -1;
    int    length      = 0;
    String sLabel      = "";
    String sTemp       = "";
    String sUnits      = "";
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.LAS_START;

    for (i=0; i<5; i++)
    {
      iCurve = iqstrat.iqstratTracksStruct.LAS_TRACK[iTrack][i];

      iContinue = 1;
      if (stLASFileData != null)
      {
        if (!stLASFileData.checkData(iCurve))
        {
          iContinue = 0;
        }
        else
        {
          iLine++;
        }
      }

      if ((iCurve > -1) && (iContinue == 1))
      {
        sLabel = new String(las.lasStandardTools.LAS_TOOLS[iCurve][1]);
        length = sLabel.length();

        sUnits = new String(las.lasStandardTools.LAS_TOOLS[iCurve][3]);
        if ((sUnits.equals("PU")) || (sUnits.equals("USEC/FT")))
        {
          dMaximum = cmn.cmnString.stringToDouble(
              las.lasStandardTools.LAS_TOOLS[iCurve][4]);
          dMinimum = cmn.cmnString.stringToDouble(
              las.lasStandardTools.LAS_TOOLS[iCurve][5]);

          if (dLIMITS != null)
          {
   	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          if ((iCurve == las.lasStandardTools._NPHI) ||
              (iCurve == las.lasStandardTools._DPHI))
          {
            if (stLASFileData!= null)
            {
              dMinimum = stLASFileData.dPHIMax;
			}

            if (dLIMITS != null)
            {
 	          dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	          dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	        }
          }

          if (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS)
          {
            if (iCurve == las.lasStandardTools._NPHI)
            {
              dMaximum = stLASFileData.dNPHIMin;
              dMinimum = stLASFileData.dNPHIMax;

              if (dLIMITS != null)
              {
   	            dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	            dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	          }
            }
          }
        }
        else
        {
          dMinimum = cmn.cmnString.stringToDouble(
              las.lasStandardTools.LAS_TOOLS[iCurve][4]);
          dMaximum = cmn.cmnString.stringToDouble(
              las.lasStandardTools.LAS_TOOLS[iCurve][5]);

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          if (stLASFileData != null)
          {
            if (iCurve == las.lasStandardTools._SP)
            {
              dMinimum = stLASFileData.dSPMin;
              dMaximum = stLASFileData.dSPMax;

              if (dLIMITS != null)
              {
     	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
    	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
     	      }
            }
          }
        }

        if (iCurve == las.lasStandardTools._RHOB)
        {
          dMinimum = stLASFileData.dRhoMin;
          dMaximum = stLASFileData.dRhoMax;

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._TEMP)
        {
          dMinimum = stLASFileData.dTEMPMin;
          dMaximum = stLASFileData.dTEMPMax;

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._TEMPDIFF)
        {
          dMinimum = stLASFileData.dTEMPDIFFMin;
          dMaximum = stLASFileData.dTEMPDIFFMax;

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LIN_1)
        {
          dMinimum = stLASFileData.dLIN_1Min;
          dMaximum = stLASFileData.dLIN_1Max;
          sLabel = new String(stLASFileData.sUnknown[0][0]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LIN_2)
        {
          dMinimum = stLASFileData.dLIN_2Min;
          dMaximum = stLASFileData.dLIN_2Max;
          sLabel = new String(stLASFileData.sUnknown[1][0]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LIN_3)
        {
          dMinimum = stLASFileData.dLIN_3Min;
          dMaximum = stLASFileData.dLIN_3Max;
          sLabel = new String(stLASFileData.sUnknown[2][0]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LIN_4)
        {
          dMinimum = stLASFileData.dLIN_4Min;
          dMaximum = stLASFileData.dLIN_4Max;
          sLabel = new String(stLASFileData.sUnknown[3][0]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LOG_1)
        {
          dMinimum = stLASFileData.dLOG_1Min;
          dMaximum = stLASFileData.dLOG_1Max;
          sLabel = new String(stLASFileData.sUnknown[0][1]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LOG_2)
        {
          dMinimum = stLASFileData.dLOG_2Min;
          dMaximum = stLASFileData.dLOG_2Max;
          sLabel = new String(stLASFileData.sUnknown[1][1]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LOG_3)
        {
          dMinimum = stLASFileData.dLOG_3Min;
          dMaximum = stLASFileData.dLOG_3Max;
          sLabel = new String(stLASFileData.sUnknown[2][1]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iCurve == las.lasStandardTools._LOG_4)
        {
          dMinimum = stLASFileData.dLOG_4Min;
          dMaximum = stLASFileData.dLOG_4Max;
          sLabel = new String(stLASFileData.sUnknown[3][1]);
          length = sLabel.length();

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }
        }

        if (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS)
        {
          if (iCurve == las.lasStandardTools._GRN)
          {
            dMinimum = stLASFileData.dGRNMinP;
            dMaximum = stLASFileData.dGRNMaxP;

            if (dLIMITS != null)
            {
   	          dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	        }
          }
        }

        if (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS)
        {
          if (iCurve == las.lasStandardTools._NEUT)
          {
            dMinimum = stLASFileData.dNEUTMinP;
            dMaximum = stLASFileData.dNEUTMaxP;

            if (dLIMITS != null)
            {
   	          dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	          dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	        }
          }
        }

        Font fsb = new Font("Serif", Font.BOLD, 10);
        g.setFont( fsb );

        if (iLine > -1)
        {
          g.setColor(COLORS[i]);
          g.drawString(sLabel,
                       (iStartTrack+iWidthTrack/2 - length*4),
                       LABELSTART + (iLine+2)*14);

          g.drawString("" + dMinimum,
                         iStartTrack+1,
                         LABELSTART + (iLine+2)*14);

          sTemp = new String("" + dMaximum);
          length = sTemp.length();
          g.drawString("" + dMaximum, iEndTrack-length*5, LABELSTART + (iLine+2)*14);
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);

//    sLabel = new String("LAS File Log Curves");
    sLabel = new String( iqstrat.iqstratTracksStruct.LAS_TRACK_DESC[iTrack] );
    length = sLabel.length();
//    g.drawString(sLabel, iStartTrack+7, LABELSTART+14);
    g.drawString(sLabel, iStartTrack+iWidthTrack/2-(9*length)/4, LABELSTART+14);
  }

  /** Method drawLimits( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the Limits on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawLimits( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
  {
    int    iX1         = 0;
    int    iX2         = 0;
    int    iY1         = PLOT_TITLES;
    int    iY2         = iHeight;
    double dRatio      = 0.0;
    double dRange[]    = { 0.0, 0.0 };
    double dMin        = 0.0;
    double dMax        = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    double dLog        = 0.0;
    double dLog1       = 0.0;
    double dPercMin    = 0.0;
    double dPercMax    = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.LAS_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.LAS_TRACK[iTrack][1];
    String sLabel      = "";
    int    iLine       = 0;
    int    length      = 0;

    Font fsb = new Font("Serif", Font.BOLD, 10);
    g.setFont( fsb );

    if (stLASFileData != null)
    {
      if ((iLAS == iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS) ||
          (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS))
      {
        if (iCurve == las.lasStandardTools._GRN)
        {
          dMinimum = stLASFileData.dGRNMinP;
          dMaximum = stLASFileData.dGRNMaxP;

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          dPercMin = stLASFileData.dGRNMin;
          dPercMax = stLASFileData.dGRNMax;

          if (dLIMITS != null)
          {
   	        dPercMin = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dPercMax = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          dRange = math.mathLAS.computeCurveMinMax(
                     stLASFileData.dGRN, stLASFileData.dNull );
          dMin   = dRange[0];
          dMax   = dRange[1];

          dLog   = stLASFileData.dGRCMin;
          dLog1  = stLASFileData.dGRCMax;
        }
        else if (iCurve == las.lasStandardTools._NEUT)
        {
          dMinimum = stLASFileData.dNEUTMinP;
          dMaximum = stLASFileData.dNEUTMaxP;

          if (dLIMITS != null)
          {
   	        dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          dPercMin = stLASFileData.dNEUTMin;
          dPercMax = stLASFileData.dNEUTMax;

          if (dLIMITS != null)
          {
   	        dPercMin = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	        dPercMax = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	      }

          dRange = math.mathLAS.computeCurveMinMax(
                     stLASFileData.dNEUT, stLASFileData.dNull );
          dMin   = dRange[0];
          dMax   = dRange[1];

          dLog   = stLASFileData.dNPHICMin;
          dLog1  = stLASFileData.dNPHICMax;
        }

        data  = dMin + dMax * dPercMin / 100.0;
        data1 = dMax * dPercMax / 100.0;

        dRatio = (double) iWidthTrack *
                   (data - dMinimum) / (dMaximum - dMinimum);
        iX1 = iStartTrack + (int) dRatio;

        dRatio = (double) iWidthTrack *
                  (data1 - dMinimum) / (dMaximum - dMinimum);
        iX2 = iStartTrack + (int) dRatio;

        if ((iX1 >= iStartTrack) && (iX2 >= iStartTrack) &&
            (iX1 <= iEndTrack)    && (iX2 <= iEndTrack))
        {
          sLabel = new String("" + dPercMin + " %");
          length = sLabel.length();
          iLine  = 2;
          g.setColor(cBLUE);
          g.drawLine(iX1, iY1, iX1, iY2);
          g.drawString(sLabel,
                       iStartTrack+1, //(iStartTrack+iWidthTrack/2 - length*4),
                       LABELSTART + (iLine+2)*14);

          g.setColor(COLORS[0]);
          sLabel = new String("" + dLog);
          length = sLabel.length();
          g.drawString(sLabel, iEndTrack-length*5, LABELSTART + (iLine+2)*14);

          sLabel = new String("" + dPercMax + " %");
          length = sLabel.length();
          iLine  = 3;
          g.setColor(cMAGENTA);
          g.drawLine(iX2, iY1, iX2, iY2);
          g.drawString(sLabel,
                       iStartTrack+1, //(iStartTrack+iWidthTrack/2 - length*4),
                       LABELSTART + (iLine+2)*14);

          g.setColor(COLORS[0]);
          sLabel = new String("" + dLog1);
          length = sLabel.length();
          g.drawString(sLabel, iEndTrack-length*5, LABELSTART + (iLine+2)*14);
        }
      }
    }
  }

  /** Method drawTrackCurves( Graphics g,
   *                          int iLAS,
   *                          int iStartTrack,
   *                          int iWidthTrack )
   *  <p> This method will draw the curves on the LAS Plot.
   *   @param g           = Graphics Parameter.
   *   @param iLAS        = The LAS Track to draw
   *   @param iStartTrack = The starting pixel for the LAS Track
   *   @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawTrackCurves( Graphics g,
                               int iLAS,
                               int iStartTrack,
                               int iWidthTrack )
  {
    int    iColumns    = 0;
    int    iPlot       = 0;
    int    iX1         = 0;
    int    iX2         = 0;
    int    iY1         = 0;
    int    iY2         = 0;
    int    iLog        = 0;
    int    iColor      = 0;
    double dRatio      = 0.0;
    double dMinimum    = 0.0;
    double dMaximum    = 0.0;
    String sUnits      = "";
    double dLog[]      = null;
    double dStart      = 0.0;
    double dEnd        = 0.0;
    double data        = 0.0;
    double data1       = 0.0;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iTrack      = iLAS - iqstrat.iqstratTracksStruct.LAS_START;
    int    iCurve      = iqstrat.iqstratTracksStruct.LAS_TRACK[iTrack][0];

    if (iCurve == las.lasStandardTools._SP)
    {
      dMinimum = stLASFileData.dSPMin;
      dMaximum = stLASFileData.dSPMax;
    }

    if ((las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("OHM-M")) ||
        (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("MD")) ||
        (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("LOG_RATIO")) ||
        (las.lasStandardTools.LAS_TOOLS[iCurve][3].equals("UNL")))
       iLog   = 1;

    if (stLASFileData != null)
    {
      for (int i=0; i<5; i++)
      {
        dLog   = null;
        iCurve = iqstrat.iqstratTracksStruct.LAS_TRACK[iTrack][i];

        if (iCurve > -1)
        {
          iColor = i;
          iColumns = stLASFileData.iRows-1;

          dLog = stLASFileData.getData(iCurve);

          if (dLog != null)
          {
            for (int j=0; j<iColumns-1; j++)
            {
              if (stLASFileData != null)
              {
				if (bSonic)
				{
                  dStart = stLASFileData.dTime[j];
                  dEnd   = stLASFileData.dTime[j+1];
				}
				else
                {
                  dStart = stLASFileData.depths[j];
                  dEnd   = stLASFileData.depths[j+1];
				}

                data       = dLog[j];
                data1      = dLog[j+1];

                dMinimum = cmn.cmnString.stringToDouble(
                    las.lasStandardTools.LAS_TOOLS[iCurve][4]);
                dMaximum = cmn.cmnString.stringToDouble(
                    las.lasStandardTools.LAS_TOOLS[iCurve][5]);
                sUnits = new String(las.lasStandardTools.LAS_TOOLS[iCurve][3]);

                if (iCurve == las.lasStandardTools._SP)
                {
                  dMinimum = stLASFileData.dSPMin;
                  dMaximum = stLASFileData.dSPMax;
                }

                if (iCurve == las.lasStandardTools._RHOB)
                {
                  dMinimum = stLASFileData.dRhoMin;
                  dMaximum = stLASFileData.dRhoMax;
                }

                if (iCurve == las.lasStandardTools._TEMP)
                {
                  dMinimum = stLASFileData.dTEMPMin;
                  dMaximum = stLASFileData.dTEMPMax;
                }

                if (iCurve == las.lasStandardTools._LIN_1)
                {
                  dMinimum = stLASFileData.dLIN_1Min;
                  dMaximum = stLASFileData.dLIN_1Max;
                }

                if (iCurve == las.lasStandardTools._LIN_2)
                {
                  dMinimum = stLASFileData.dLIN_2Min;
                  dMaximum = stLASFileData.dLIN_2Max;
                }

                if (iCurve == las.lasStandardTools._LIN_3)
                {
                  dMinimum = stLASFileData.dLIN_3Min;
                  dMaximum = stLASFileData.dLIN_3Max;
                }

                if (iCurve == las.lasStandardTools._LIN_4)
                {
                  dMinimum = stLASFileData.dLIN_4Min;
                  dMaximum = stLASFileData.dLIN_4Max;
                }

                if (iCurve == las.lasStandardTools._LOG_1)
                {
                  dMinimum = stLASFileData.dLOG_1Min;
                  dMaximum = stLASFileData.dLOG_1Max;
                }

                if (iCurve == las.lasStandardTools._LOG_2)
                {
                  dMinimum = stLASFileData.dLOG_2Min;
                  dMaximum = stLASFileData.dLOG_2Max;
                }

                if (iCurve == las.lasStandardTools._LOG_3)
                {
                  dMinimum = stLASFileData.dLOG_3Min;
                  dMaximum = stLASFileData.dLOG_3Max;
                }

                if (iCurve == las.lasStandardTools._LOG_4)
                {
                  dMinimum = stLASFileData.dLOG_4Min;
                  dMaximum = stLASFileData.dLOG_4Max;
                }

                if (iCurve == las.lasStandardTools._TEMPDIFF)
                {
                  dMinimum = stLASFileData.dTEMPDIFFMin;
                  dMaximum = stLASFileData.dTEMPDIFFMax;
                }

                if (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS)
                {
                  if (iCurve == las.lasStandardTools._GRN)
                  {
                    dMinimum = stLASFileData.dGRNMinP;
                    dMaximum = stLASFileData.dGRNMaxP;
                  }
                }

                if (iLAS == iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS)
                {
                  if (iCurve == las.lasStandardTools._NPHI)
                  {
                    dMinimum = stLASFileData.dNPHIMin;
                    dMaximum = stLASFileData.dNPHIMax;
                  }

                  if (iCurve == las.lasStandardTools._NEUT)
                  {
                    dMinimum = stLASFileData.dNEUTMinP;
                    dMaximum = stLASFileData.dNEUTMaxP;
                  }
                }

                if (dLIMITS != null)
                {
   	              dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	              dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	            }

                if ((iCurve == las.lasStandardTools._NPHI) ||
                    (iCurve == las.lasStandardTools._DPHI))
                {
//                  dMinimum = stLASFileData.dPHIMax;
                  dMaximum = stLASFileData.dPHIMax;

                  if (dLIMITS != null)
                  {
   	                dMinimum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MINIMUM];
	                dMaximum = dLIMITS[iCurve][las.lasPlotLimitsUtility.MAXIMUM];
	              }
                }

                if (iLog == 1)
                {
                  if (data <= 0.0)  data  = .00001;
                  if (data1 <= 0.0) data1 = .00001;

                  data     = Math.log(data) / Math.log(10);
                  data1    = Math.log(data1) / Math.log(10);
                  dMinimum = Math.log(dMinimum) / Math.log(10);
                  dMaximum = Math.log(dMaximum) / Math.log(10);
                }

                dRatio = (double) iLogHeight * (dStart - depthStart) /
                    (depthEnd - depthStart);
                iY1 = PLOT_TITLES + (int) dRatio;
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iY1 = iHeight - (int) dRatio;

                dRatio = (double) iLogHeight * (dEnd - depthStart) /
                    (depthEnd - depthStart);
                iY2 = PLOT_TITLES + (int) dRatio;
                if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                  iY2 = iHeight - (int) dRatio;

                iPlot = 0;
                if ( (iY1 > PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                    (iY1 < iHeight) && (iY2 < iHeight))
                {
                  iPlot = 1;
                }

                if ( (sUnits.equals("PU")) || (sUnits.equals("USEC/FT")))
                {
                  dRatio = (double) iWidthTrack *
                      (dMaximum - data) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;

                  dRatio = (double) iWidthTrack *
                      (dMaximum - data1) / (dMaximum - dMinimum);
                  iX2 = iStartTrack + (int) dRatio;

                  if ( (iX1 < iStartTrack) && (iX2 < iStartTrack))
                  {
                    while (iX1 < iStartTrack) { iX1 = iX1 + iWidthTrack; }
                    while (iX2 < iStartTrack) { iX2 = iX2 + iWidthTrack; }
                  }
                  else if ( (iX1 >= iStartTrack) && (iX2 < iStartTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX2 < iStartTrack) { iX2 = iX2 + iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iEndTrack, iY1, iX2, iY2);
                    iX2 = iStartTrack;
                  }
                  else if ( (iX1 < iStartTrack) && (iX2 >= iStartTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX1 < iStartTrack) { iX1 = iX1 + iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iX1, iY1, iEndTrack, iY2);
                    iX1 = iStartTrack;
                  }
                }
                else
                {
                  dRatio = (double) iWidthTrack *
                      (data - dMinimum) / (dMaximum - dMinimum);
                  iX1 = iStartTrack + (int) dRatio;

                  dRatio = (double) iWidthTrack *
                      (data1 - dMinimum) / (dMaximum - dMinimum);
                  iX2 = iStartTrack + (int) dRatio;

                  if ((iX1 > iEndTrack) && (iX2 > iEndTrack))
                  {
                    while (iX1 > iEndTrack) { iX1 = iX1 - iWidthTrack; }
                    while (iX2 > iEndTrack) { iX2 = iX2 - iWidthTrack; }
                  }
                  else if ((iX1 <= iEndTrack) && (iX2 > iEndTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX2 > iEndTrack) { iX2 = iX2 - iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iStartTrack, iY1, iX2, iY2);
                    iX2 = iEndTrack;
                  }
                  else if ((iX1 > iEndTrack) && (iX2 <= iEndTrack))
                  {
                    g.setColor(COLORS[iColor]);
                    while (iX1 > iEndTrack) { iX1 = iX1 - iWidthTrack; }
                    if (iPlot == 1)
                      g.drawLine(iX1, iY1, iStartTrack, iY2);
                    iX1 = iEndTrack;
                  }
                }

                if ((iX1 >= iStartTrack) && (iX2 >= iStartTrack) &&
                    (iX1 <= iEndTrack)    && (iX2 <= iEndTrack))
                {
                  g.setColor(COLORS[iColor]);
                  if (iPlot == 1)
                    g.drawLine(iX1, iY1, iX2, iY2);
                }
              }
            }
          }
        }
      }
    }
  }

  /** Method draw( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iLAS        = The LAS Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iLAS, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_LAS;

    g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
    g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

    drawGrid( g, iLAS, iStartTrack, iWidthTrack );
    drawTrackLabels( g, iLAS, iStartTrack, iWidthTrack );
    drawTrackCurves( g, iLAS, iStartTrack, iWidthTrack );
    drawLimits( g, iLAS, iStartTrack, iWidthTrack );
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}
