/*
 * @lasPlotThinPhiTrack.java Version 1.1 09/07/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.plot;

import java.awt.*;
import java.awt.geom.*;

import las.lasFileDataStruct;

/** CLASS lasPlotThinPhiTrack
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot Thin Phi Tracks.
 *
 *  @version 1.1 09/07/2007
 *  @author  John R. Victorine
 */

public class lasPlotThinPhiTrack extends Canvas
{
  // Input Variables

  private boolean bSonic     = false; // Indicator to plot by two-way time
  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables
  // LAS File Data Structure

  private lasFileDataStruct stLAS = null; // LAS File Data Struct

  // Porosity Data

  private int iPHI  = iqstrat.iqstratTracksStruct._THIN_NPHI;

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR lasPlotThinPhiTrack( int iDataType,
   *                                   double depthStart,
   *                                   double depthEnd,
   *                                   int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lasPlotThinPhiTrack( int iDataType,
                              double depthStart,
                              double depthEnd,
                              int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR lasPlotThinPhiTrack( boolean bSonic,
   *                                   int iDataType,
   *                                   double depthStart,
   *                                   double depthEnd,
   *                                   int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public lasPlotThinPhiTrack( boolean bSonic,
                              int iDataType,
                              double depthStart,
                              double depthEnd,
                              int iScale )
  {
	this.bSonic    = bSonic;
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close() { stLAS = null; }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setLASFileData( lasFileDataStruct stLAS )
   * <p> This method will set the LAS File Data Structure
   * @param stLAS - LAS File Data Structure
   */

  public void setLASFileData( lasFileDataStruct stLAS ) { this.stLAS = stLAS; }

  /** Method setPHI( int iPHI )
   * <p> This method will set the Porosity to use in creating the track
   * @param iPHI - identifier for porosity track
   */

  public void setPHI( int iPHI ) { this.iPHI = iPHI; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/5;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.lightGray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    if (iLog == 0)
    {
      g.setColor(Color.lightGray);
      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<=iWidthTrack; i+=iIncrement)
      {
        if ((i==0) || (i==iWidthTrack))
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
        else
        {
          g.drawLine(i+iStartTrack, PLOT_TITLES,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }
      }
    }
    else
    {
      iIncrement = iWidthTrack / iCycle;

      g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

      for (i=0; i<iWidthTrack; i+=iIncrement)
      {
        g.setColor(Color.lightGray);
        if (i==0)
        {
          g.drawLine(i+iStartTrack, LABELSTART,
                     i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
        }

        g.setColor(Color.lightGray);
        g.drawLine(i+iStartTrack, PLOT_TITLES,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);

        g.setColor(Color.lightGray);
        for (j=1; j<5; j++)
        {
          dXinc = (double) j * 2.0;
          dTemp = (double) iIncrement * Math.log(dXinc) / Math.log(10.0);
          jXinc = i + iStartTrack + (int) dTemp;

          if (jXinc <= iEndTrack)
          {
            g.drawLine(jXinc, PLOT_TITLES,
                       jXinc, PLOT_TITLES+iLogHeight); //iHeight);
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawPorosityTrack( Graphics g, int iStartTrack, int iWidthTrack )
   *   <p> This track will plot the porosity in black color fill
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawPorosityTrack( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    double dRatio   = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;
    double dTemp    = 0.0;
    double dPHI     = 0.0;
    int    iStart   = 0;
    int    iEnd     = 0;
    int    iX1      = 0;
    int    iX2      = 0;
    int    iWide    = 0;
    int    length   = 0;
    String sTemp    = "";
    String sPHI     = "";

    g.setColor(Color.black);

    iStart = iStartTrack;
    iEnd   = iStartTrack+iWidthTrack;

    if (stLAS != null)
    {
      for (int i=0; i<stLAS.iRows-1; i++)
      {
		if (bSonic)
		{
          depth1 = stLAS.dTime[i];
          depth2 = stLAS.dTime[i+1];
		}
        else
        {
          depth1 = stLAS.depths[i];
          depth2 = stLAS.depths[i+1];
		}

        dRatio = (double) iLogHeight * (depthStart - depth1) /
                                       (depthStart - depthEnd);
        iDepth1 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth1 = iHeight - (int) dRatio;

        dRatio = (double) iLogHeight * (depthStart - depth2) /
                                       (depthStart - depthEnd);
        iDepth2 = PLOT_TITLES + (int) dRatio;
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iDepth2 = iHeight - (int) dRatio;

        iDepth = Math.abs(iDepth2 - iDepth1);

        if ((iDepth1 > PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
            (iDepth1 < iHeight)     && (iDepth2 < iHeight))
        {
          switch (iPHI)
          {
            case iqstrat.iqstratTracksStruct._THIN_NPHI:
              dPHI = stLAS.dNPHI[i];
              break;
            case iqstrat.iqstratTracksStruct._THIN_DPHI:
              dPHI = stLAS.dDPHI[i];
              break;
            case iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI:
              dPHI = (stLAS.dNPHI[i] + stLAS.dDPHI[i]) / 2.0;
              break;
            case iqstrat.iqstratTracksStruct._THIN_SPHI:
              dPHI = stLAS.dSPHI[i];
              break;
          }

          iWide = 0;
          if (dPHI > stLAS.dNull)
            iWide = (int) ( iWidthTrack * dPHI * 2.0 );

          if (iWide < 0) iWide = 0;
          if (iWide > iWidthTrack) iWide = iWidthTrack;

          g.fillRect(iStart, iDepth1, iWide, iDepth);
        }
      }
    }

    // Draw Gamma Ray Grid & Title

    switch (iPHI)
    {
      case iqstrat.iqstratTracksStruct._THIN_NPHI:
        sPHI = "Neutron";
        break;
      case iqstrat.iqstratTracksStruct._THIN_DPHI:
        sPHI = "Density";
        break;
      case iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI:
        sPHI = "Average";
        break;
      case iqstrat.iqstratTracksStruct._THIN_SPHI:
        sPHI = "Sonic";
        break;
    }

    g.setColor(Color.black);

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Porosity", LABELSTART+14, -1*(iStart+2*iWidthTrack/3));
    g2.drawString(sPHI, LABELSTART+14, -1*(iStart+5));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    fsb  = new Font("Serif", Font.BOLD, 10);
    fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.drawString("0", iStart+1, LABELSTART+74);

    sTemp = new String("50");
    length = sTemp.length();
    g.drawString("50", iEnd-length*5, LABELSTART+74);

    g.drawLine(iStart, LABELSTART,  iEnd,   LABELSTART);
    g.drawLine(iStart, PLOT_TITLES, iEnd,   PLOT_TITLES);
    g.drawLine(iStart, LABELSTART,  iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART,  iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a LAS Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Porosity Track to draw
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_LAS;

    if (iSelected == iqstrat.iqstratTracksStruct._THIN_PHI)
    {
      g.setColor( new Color( iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                             iqstrat.iqstratTracksStruct.COLORS[iColor][2] ) );
      g.fillRect( iStartTrack, LABELSTART, iWidthTrack, 20 );

      drawGrid( g, iStartTrack, iWidthTrack );
      drawPorosityTrack( g, iStartTrack, iWidthTrack );
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}