/*
 * @lasPlotLimitsFrame.java Version 1.1 04/15/2016
 *
 * Copyright (c) 2016 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.plot;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

/** Class lasPlotLimitsFrame
 *  <p> This Class will allow the user to change the Scale on the LAS & PfEFFER Plot
 *      track
 *
 *  @version 1.1 04/15/2016
 *  @author  John R. Victorine
 */

public class lasPlotLimitsFrame extends JFrame implements ActionListener
{
  private static final int   _C_LAS     = iqstrat.iqstratControlStruct._C_LAS;

  private static final int   MINIMUM    = las.lasPlotLimitsUtility.MINIMUM;
  private static final int   MAXIMUM    = las.lasPlotLimitsUtility.MAXIMUM;
  private static final int   GROUP      = las.lasPlotLimitsUtility.GROUP;
  private static final int   DATA       = las.lasPlotLimitsUtility.LAS;
  private static final int   COLUMNS    = las.lasPlotLimitsUtility.COLUMNS;

  private static final Color cDEPTH     = new Color( 165,  42,  42 ); // brown
  private static final Color cOHM       = new Color( 255, 165,   0 ); // orange
  private static final Color cVSH       = new Color( 150, 150, 200 ); // slate blue
  private static final Color cPHI       = new Color(   0, 255, 255 ); // cyan
  private static final Color cGR        = new Color( 255,   0,   0 ); // red
  private static final Color cRHOMAA    = new Color(   0, 255,   0 ); // green
  private static final Color cDTMAA     = new Color(  34, 139,  34 ); // forest green
  private static final Color cUMAA      = new Color( 173, 255,  47 ); // green yellow
  private static final Color cMD        = new Color(   0,   0, 255 ); // blue
  private static final Color cDEGF      = new Color( 255, 105, 180 ); // hot pink
  private static final Color cFRAC      = new Color( 255, 255, 128 ); // middle yellow
  private static final Color cUNI       = new Color( 148,   0, 211 ); // dark violet
  private static final Color cUNL       = new Color( 148, 130, 238 ); // violet
  private static final Color cMISC      = new Color( 255, 255,   0 ); // yellow

  // Input Variables

  private Observable notifier = null; // Observable

  private double     orig_las[][]  = null; // Original Plot Limits for LAS
  private double     data_las[][]  = null; // Plot Limits for LAS

  // Global Frame Variables

  // Global Frame Widgets

  private JButton    btnClose     = null; // Close Dialog

  private JTextField txtLAS[][]   = null; // Array of LAS Curves Minimum & Maximum

  private JButton    btnLAS[]     = null; // Array of Reset Buttons
  private JButton    btnXLAS[]    = null; // Array of change in Limits Buttons

  /** Constructor lasPlotLimitsFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param iData    = Which Plot Track Type to be modified
   *  @param orig     = Original Plot Track Limits
   *  @param data     = Modified Plot Track Limits
   */

  public lasPlotLimitsFrame( Observable notifier,
                             double orig[][],  double data[][] )
  {
    try
    {
      this.notifier = notifier;
      this.orig_las  = orig;
      this.data_las  = data;

      if (data != null)
        jbInit();

      setColor(_C_LAS);

      addWindowListener( new lasPlotLimitsFrame_WindowListener() );
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop       = new JPanel();
    JPanel pnlCenter    = new JPanel();

    // Build Frame

    this.setTitle("Change Plot Limits");
    this.getContentPane().setLayout(new BorderLayout());

    // Top Panel

    pnlTop.setLayout(new BorderLayout());

    btnClose = new JButton(createImageIcon(cmn.cmnStruct.CLOSE));
    btnClose.setPreferredSize(new Dimension(27, 27));
    btnClose.setToolTipText("Close");
    btnClose.addActionListener(this);

    // Center Panel

    pnlCenter = buildLASPanel();

    // Attach Widgets to frame

    this.getContentPane().add(pnlTop,        BorderLayout.NORTH);
    pnlTop.add(btnClose,                     BorderLayout.WEST);

    this.getContentPane().add(pnlCenter,     BorderLayout.CENTER);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(600, 750));
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(true);
    this.setVisible(true);
  }

  /** Method buildLASPanel()
   *  <p> This method will build the change LAS Curves Limits Panel
   *  @return panel = The LAS Curves Limits Panel
   */

  private JPanel buildLASPanel()
  {
	int i,j;
	int    cnt   = 0;
	int    ittl  = 0;
    String sUnit = "";

	JPanel panel        = new JPanel();

    JPanel panelLAS     = new JPanel();

    JPanel pnlLAS       = new JPanel();
    JPanel pnlLASlbl    = new JPanel();
    JPanel pnlLASlbl1   = new JPanel();
    JPanel pnlLASlbl2   = new JPanel();
    JPanel pnlLASEast   = new JPanel();
    JPanel pnlLAStxt    = new JPanel();
    JPanel pnlLAStxt1   = new JPanel();
    JPanel pnlLAStxt2   = new JPanel();
    JPanel pnlLASReset  = new JPanel();

    JLabel lblLASMnem   = new JLabel();
    JLabel lblLASDesc   = new JLabel();
    JLabel lblLASMin    = new JLabel();
    JLabel lblLASMax    = new JLabel();
    JLabel lblLASReset  = new JLabel();

    JPanel pnlL[]       = null;
    JPanel pnlLLabel[]  = null;
    JPanel pnlLLabel1[] = null;
    JPanel pnlLLabel2[] = null;
    JPanel pnlLEast[]   = null;
    JPanel pnlLText[]   = null;
    JPanel pnlLMin[]    = null;
    JPanel pnlLMax[]    = null;

    JLabel lblLAS[][]   = null;

    JScrollPane scrollL = new JScrollPane();

    TitledBorder titledBorderLAS = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "LAS Log Plot Limits:");
    titledBorderLAS.setTitleFont(new java.awt.Font("Dialog", 1, 13));

    ittl = 0;
    for (i=0; i<las.lasStandardTools.TOTAL; i++)
    {
	  if (data_las[i][DATA] != -1.0)
	  {
		ittl++;
	  }
	}

    // Build LAS Panel

    panel.setLayout(new BorderLayout());
    panel.setBorder(titledBorderLAS);

    // -- LAS Panel

    panelLAS.setLayout(new GridLayout(ittl, 1));

    // .. LAS Header Panel

    pnlLAS.setLayout(new BorderLayout());

    // .. .. LAS Header Labels Panel

    pnlLASlbl.setLayout(new BorderLayout());

    lblLASMnem.setFont(new java.awt.Font("Dialog", 1, 13));
    lblLASMnem.setText("Mnem");

    lblLASDesc.setFont(new java.awt.Font("Dialog", 1, 13));
    lblLASDesc.setText("Curve Description");

    // .. .. LAS Header Texts Panel

    pnlLAStxt.setLayout(new GridLayout());

    lblLASMin.setFont(new java.awt.Font("Dialog", 1, 13));
    lblLASMin.setText("Minimum");

    lblLASMax.setFont(new java.awt.Font("Dialog", 1, 13));
    lblLASMax.setText("Maximum");

    // .. .. LAS Header Reset Panel

    pnlLASReset.setLayout(new BorderLayout());

    lblLASReset.setFont(new java.awt.Font("Dialog", 1, 13));
    lblLASReset.setText("Reset");

    // .. Attach LAS Header Widgets

    pnlLAS.add(pnlLASlbl,       BorderLayout.CENTER);

    pnlLASlbl.add(pnlLASlbl1,   BorderLayout.WEST);
    pnlLASlbl1.add(lblLASMnem,  null);

    pnlLASlbl.add(pnlLASlbl2,   BorderLayout.CENTER);
    pnlLASlbl2.add(lblLASDesc,  null);

    pnlLAS.add(pnlLASEast,      BorderLayout.EAST);
    pnlLASEast.add(pnlLAStxt,   BorderLayout.CENTER);

    pnlLAStxt.add(pnlLAStxt1,   null);
    pnlLAStxt1.add(lblLASMin,   null);

    pnlLAStxt.add(pnlLAStxt2,   null);
    pnlLAStxt2.add(lblLASMax,   null);

    pnlLASEast.add(lblLASReset, BorderLayout.EAST);

    // .. LAS List Panel

    pnlL       = new JPanel[las.lasStandardTools.TOTAL];
	pnlLLabel  = new JPanel[las.lasStandardTools.TOTAL];
	pnlLLabel1 = new JPanel[las.lasStandardTools.TOTAL];
	pnlLLabel2 = new JPanel[las.lasStandardTools.TOTAL];
	pnlLEast   = new JPanel[las.lasStandardTools.TOTAL];
	pnlLText   = new JPanel[las.lasStandardTools.TOTAL];
	pnlLMin    = new JPanel[las.lasStandardTools.TOTAL];
	pnlLMax    = new JPanel[las.lasStandardTools.TOTAL];

    lblLAS     = new JLabel[las.lasStandardTools.TOTAL][2];
    txtLAS     = new JTextField[las.lasStandardTools.TOTAL][2];

    btnLAS     = new JButton[las.lasStandardTools.TOTAL];
    btnXLAS    = new JButton[las.lasStandardTools.TOTAL];

    for (i=0; i<las.lasStandardTools.TOTAL; i++)
    {
      pnlL[i]      = new JPanel();
      pnlL[i].setLayout(new BorderLayout());

  	  pnlLLabel[i] = new JPanel();
  	  pnlLLabel[i].setLayout(new GridLayout());

      pnlLLabel1[i] = new JPanel();
      pnlLLabel1[i].setLayout(new BorderLayout());

      lblLAS[i][0] = new JLabel();
      lblLAS[i][0].setFont(new java.awt.Font("Dialog", 1, 12));
      lblLAS[i][0].setText(las.lasStandardTools.LAS_TOOLS[i][1]);

      pnlLLabel2[i] = new JPanel();
      pnlLLabel2[i].setLayout(new BorderLayout());

      lblLAS[i][1] = new JLabel();
      lblLAS[i][1].setFont(new java.awt.Font("Dialog", 1, 12));
      lblLAS[i][1].setText(las.lasStandardTools.LAS_TOOLS[i][2]);

      pnlLEast[i]  = new JPanel();
	  pnlLEast[i].setLayout(new BorderLayout());

	  pnlLText[i]  = new JPanel();
	  pnlLText[i].setLayout(new GridLayout());

	  pnlLMin[i]   = new JPanel();
	  pnlLMin[i].setLayout(new BorderLayout());

      txtLAS[i][0] = new JTextField();
      txtLAS[i][0].setColumns(6);
      txtLAS[i][0].setFont(new java.awt.Font("Dialog", 1, 12));
      txtLAS[i][0].setHorizontalAlignment(SwingConstants.TRAILING);
      txtLAS[i][0].setText(""+data_las[i][MINIMUM]);
      txtLAS[i][0].addFocusListener(new lasPlotLimitsFrameFocusAdapter(this));

	  pnlLMax[i]   = new JPanel();
	  pnlLMax[i].setLayout(new BorderLayout());

      txtLAS[i][1] = new JTextField();
      txtLAS[i][1].setColumns(6);
      txtLAS[i][1].setFont(new java.awt.Font("Dialog", 1, 12));
      txtLAS[i][1].setHorizontalAlignment(SwingConstants.TRAILING);
      txtLAS[i][1].setText(""+data_las[i][MAXIMUM]);
      txtLAS[i][1].addFocusListener(new lasPlotLimitsFrameFocusAdapter(this));

      btnLAS[i] = new JButton();
      btnLAS[i].setPreferredSize(new Dimension(27, 27));
      btnLAS[i].addActionListener(this);

      sUnit = las.lasStandardTools.LAS_TOOLS[i][3];

	  // Resistivity, Spontaneous Potential
	  if ((sUnit.equals("OHM-M")) || (sUnit.equals("OHMM")) || (sUnit.equals("MV")))
        btnLAS[i].setBackground( cOHM );   // orange
      // Porosity
	  else if (sUnit.equals("PU"))
        btnLAS[i].setBackground( cPHI ); // cyan
      // Photoelectric
	  else if (sUnit.equals("BARNS/E"))
        btnLAS[i].setBackground( cUMAA ); // green yellow
      // Grain Density
	  else if ((sUnit.equals("GM/CC"))  || (sUnit.equals("GMCC")))
        btnLAS[i].setBackground( cRHOMAA ); // green
      // Acoustic transit time
	  else if (sUnit.equals("USEC/FT"))
        btnLAS[i].setBackground( cDTMAA ); // forest green
      // Radioactive
	  else if ((sUnit.equals("PPM")) || (sUnit.equals("%")) || (sUnit.equals("API")))
        btnLAS[i].setBackground( cGR );     // red
      // Permeability
	  else if (sUnit.equals("MD"))
        btnLAS[i].setBackground( cMD ); // blue
      // Feet Inches
	  else if ((sUnit.equals("F")) || (sUnit.equals("FT")) || (sUnit.equals("IN")))
        btnLAS[i].setBackground( cDEPTH ); // brown
      // Fraction
	  else if (sUnit.equals("FRAC"))
        btnLAS[i].setBackground( cFRAC );
      // Unknown Linear Curve
	  else if (sUnit.equals("UNI"))
        btnLAS[i].setBackground( cUNI );
      // Unknown Log Curve
	  else if (sUnit.equals("UNL"))
        btnLAS[i].setBackground( cUNL );
      else if (sUnit.equals("DEGF"))
        btnLAS[i].setBackground( cDEGF );
      // Default
      else
        btnLAS[i].setBackground( cMISC ); // yellow

      btnXLAS[i] = new JButton();
      btnXLAS[i].setPreferredSize(new Dimension(10, 27));
      btnXLAS[i].setBackground(Color.lightGray);
      btnXLAS[i].addActionListener(this);

      // .. .. Attach Row

  	  if (data_las[i][DATA] != -1.0)
	  {
        panelLAS.add(pnlL[i],               null);
	  }

      pnlL[i].add(btnXLAS[i],            BorderLayout.WEST);

      pnlL[i].add(pnlLLabel[i],          BorderLayout.CENTER);

      pnlLLabel[i].add(pnlLLabel1[i],    null); //BorderLayout.WEST);
      pnlLLabel1[i].add(lblLAS[i][0],    BorderLayout.WEST);

      pnlLLabel[i].add(pnlLLabel2[i],    null); //BorderLayout.CENTER);
      pnlLLabel2[i].add(lblLAS[i][1],    BorderLayout.WEST);

      pnlL[i].add(pnlLEast[i],           BorderLayout.EAST);
      pnlLEast[i].add(pnlLText[i],       BorderLayout.CENTER);

      pnlLText[i].add(pnlLMin[i],        null);
      pnlLMin[i].add(txtLAS[i][0],       BorderLayout.CENTER);

      pnlLText[i].add(pnlLMax[i],        null);
      pnlLMax[i].add(txtLAS[i][1],       BorderLayout.CENTER);

      pnlLEast[i].add(btnLAS[i],         BorderLayout.EAST);
	}

    // Add Widgets to panel

    panel.add(pnlLAS,                    BorderLayout.NORTH);
    panel.add(scrollL,                   BorderLayout.CENTER);
    scrollL.getViewport().add( panelLAS, null);

	return (panel);
  }

  /** METHOD createImageIcon()
   *  <p> This Method will create the Frame for this class.
   *  @param path = the path to the image
   *  @return image icon
   */

  protected ImageIcon createImageIcon(String path)
  {
    java.net.URL imgURL = getClass().getResource(path);
    ImageIcon    image  = null;

    if (imgURL != null)
    {
      image = new ImageIcon(imgURL);
    }
    else
    {
       System.err.println("Couldn't find file: " + path);
    }

    return (image);
  }

  /* ===================================================================== *
   * -------------------------- CLOSE DIALOG ----------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier  = null; // Observable
    orig_las  = null; // Original Plot Limits for LAS
    data_las  = null; // Plot Limits for LAS

    // Global Frame Widgets

    btnClose = null;

    if (txtLAS != null)
    {
      for (int i=0; i<las.lasStandardTools.TOTAL; i++)
      {
	    for (int j=0; j<2; j++)
	    {
	  	  txtLAS[i][j] = null;
	    }
	    btnXLAS[i] = null;
  	    btnLAS[i]  = null;
	  }
	}
    txtLAS  = null; // LAS Textfields
    btnLAS  = null;
    btnXLAS = null;

    dispose();
  }

  /** METHOD lasPlotLimitsFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class lasPlotLimitsFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
//      close();
      if (notifier != null)
        notifier.notifyObservers(new String("Close LAS Plot Limits"));
    }
  }

  /* ====================================================================== *
   * ---------------------------- GET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method getLASLimits()
   *  <p> this method will retrieve the LAS Limits data array
   *  @return data_las = array of the LAS Limits data
   */

  public double[][] getLASLimits() { return(data_las); } // Plot Limits for LAS

  /** Method getLASOriginal()
   *  <p> this method will retrieve the Standard LAS Limits data array
   *  @return orig_las = array of the standard LAS Limits data
   */

  public double[][] getLASOriginal() { return(orig_las); } // Plot Limits for LAS

  /* ====================================================================== *
   * ---------------------------- SET METHODS ----------------------------- *
   * ====================================================================== */

  /** Method setColor()
   *  <p> This method will set the Button Change Colors
   *  @param iAction = the List selected
   */

  private void setColor(int iAction)
  {
	int i,j;
	String sUnit = "";

    if ((iAction == _C_LAS) && (data_las != null))
    {
      for (i=0; i<las.lasStandardTools.TOTAL; i++)
      {
	    for (j=0; j<2; j++)
	    {
          txtLAS[i][j].setText(""+ data_las[i][j]);

	  	  if (data_las[i][j] != orig_las[i][j])
		  {
            sUnit = las.lasStandardTools.LAS_TOOLS[i][3];

	        // Resistivity, Spontaneous Potential
	        if ((sUnit.equals("OHM-M")) || (sUnit.equals("OHMM")) || (sUnit.equals("MV")))
              btnXLAS[i].setBackground( cOHM );   // orange
            // Porosity
	        else if (sUnit.equals("PU"))
              btnXLAS[i].setBackground( cPHI ); // cyan
            // Photoelectric
	        else if (sUnit.equals("BARNS/E"))
              btnXLAS[i].setBackground( cUMAA ); // green yellow
            // Grain Density
	        else if ((sUnit.equals("GM/CC"))  || (sUnit.equals("GMCC")))
              btnXLAS[i].setBackground( cRHOMAA ); // green
            // Acoustic transit time
	        else if (sUnit.equals("USEC/FT"))
              btnXLAS[i].setBackground( cDTMAA ); // forest green
            // Radioactive
	        else if ((sUnit.equals("PPM")) || (sUnit.equals("%")) || (sUnit.equals("API")))
              btnXLAS[i].setBackground( cGR );     // red
            // Permeability
	        else if (sUnit.equals("MD"))
              btnXLAS[i].setBackground( cMD ); // blue
            // Feet Inches
	        else if ((sUnit.equals("F")) || (sUnit.equals("FT")) || (sUnit.equals("IN")))
              btnXLAS[i].setBackground( cDEPTH ); // brown
            // Fraction
	        else if (sUnit.equals("FRAC"))
              btnXLAS[i].setBackground( cFRAC );
            // Unknown Linear Curve
	        else if (sUnit.equals("UNI"))
              btnXLAS[i].setBackground( cUNI );
            // Unknown Log Curve
	        else if (sUnit.equals("UNL"))
              btnXLAS[i].setBackground( cUNL );
            else if (sUnit.equals("DEGF"))
              btnXLAS[i].setBackground( cDEGF );
            // Default
            else
              btnXLAS[i].setBackground( cMISC ); // yellow
		  }
		}
	  }
	}
  }

  /* ====================================================================== *
   * ---------------------------- ACTIONS --------------------------------- *
   * ====================================================================== */

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
	int i,j,k;
    double dGroup    = -1;
    double dMin      = 0.0;
    double dMax      = 0.0;

    for (i=0; i<las.lasStandardTools.TOTAL; i++)
    {
	  if (event.getSource() == btnLAS[i])
	  {
		dGroup = data_las[i][GROUP];
		dMin   = orig_las[i][MINIMUM];
		dMax   = orig_las[i][MAXIMUM];

        data_las[i][MINIMUM] = dMin;
        data_las[i][MAXIMUM] = dMax;

        if (dGroup != -1.0)
          data_las = las.lasPlotLimitsUtility.setByGroup(dGroup, dMin, dMax, data_las );

        for (k=0; k<las.lasStandardTools.TOTAL; k++)
        {
		  for (j=0; j<2; j++)
		  {
            txtLAS[k][j].setText(""+ data_las[k][j]);
		  }

          if ((data_las[k][MINIMUM] == orig_las[k][MINIMUM]) &&
              (data_las[k][MAXIMUM] == orig_las[k][MAXIMUM]))
          {
	        btnXLAS[k].setBackground(Color.lightGray);
		  }
		}

        if (notifier != null)
          notifier.notifyObservers(new String("LAS Plot Limits Changed"));
	  }
	}

    if (event.getSource() == btnClose)   // Close Dialog
    {
      if (notifier != null)
        notifier.notifyObservers(new String("Close LAS Plot Limits"));
	}
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
	int    i,j;
    int    iNumeric  = 0;
    int    k_row     = -1;
    int    k_column  = -1;
    int    iAction   = -1;
    String sMnemonic = "";
    String sMessage  = "";
    String sTemp     = "";
    double dTemp     = 0.0;
    int    iCycle    = 0;
    double dGroup    = -1;
    double dMin      = 0.0;
    double dMax      = 0.0;

    for (i=0; i<las.lasStandardTools.TOTAL; i++)
    {
	  for (j=0; j<2; j++)
	  {
		if (txtLAS !=  null)
		{
          if (e.getSource() == txtLAS[i][j])
          {
		    iAction   = _C_LAS;
            iNumeric  = 1;
            sTemp     = txtLAS[i][j].getText();
            k_row     = i;
            k_column  = j;
            sMnemonic = new String(las.lasStandardTools.LAS_TOOLS[i][1]);
            dGroup    = data_las[k_row][GROUP];

	        switch (j)
	        {
              case MINIMUM: // Minimum Data Value
                sMessage = new String("The Minimum " + sMnemonic + " is a Numeric Field");
                break;
              case MAXIMUM: // Maximum Data Value
                sMessage = new String("The Maximum " + sMnemonic + " is a Numeric Field");
                break;
			}
		  }
		}
	  }
	}

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        txtLAS[k_row][k_column].setText(""+data_las[k_row][k_column]);

        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
		    dTemp = cmn.cmnString.stringToDouble( sTemp );

            if (((k_row >= las.lasStandardTools._RES) && (k_row <= las.lasStandardTools._AHT90)) ||
                (k_row == las.lasStandardTools._PERM) ||
                ((k_row >= las.lasStandardTools._LOG_1) && (k_row <= las.lasStandardTools._LOG_4)))
            {

	          switch (k_column)
	          {
                case MINIMUM: // Minimum Data Value
      	          iCycle  = (int) Math.floor(Math.log(dTemp) / Math.log(10.0));
                  break;
                case MAXIMUM: // Maximum Data Value
      	          iCycle  = (int) Math.ceil(Math.log(dTemp) / Math.log(10.0));
                  break;
		      }

              dTemp   = Math.pow(10.0, iCycle);
              if (dTemp > 100000) { dTemp = 100000; }

              txtLAS[k_row][k_column].setText(""+ dTemp);
			}

			data_las[k_row][k_column] = dTemp;
			dMin = data_las[k_row][MINIMUM];
			dMax = data_las[k_row][MAXIMUM];

            data_las = las.lasPlotLimitsUtility.setByGroup(dGroup, dMin, dMax, data_las );

		    setColor(iAction);

            if (notifier != null)
              notifier.notifyObservers(new String("LAS Plot Limits Changed"));

        iAction  = -1;
        k_row    = -1;
        k_column = -1;

//        if (notifier != null)
//          notifier.notifyObservers(new String("Profile Plot Limits Changed"));
      }
    }
  }
}

/** CLASS lasPlotLimitsFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class lasPlotLimitsFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  lasPlotLimitsFrame adaptee;

  lasPlotLimitsFrameFocusAdapter(lasPlotLimitsFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 04/15/2016
 *  @author  John Victorine
 */
