/*
 * @lasPlotLimitsUtility.java Version 1.1 12/10/2013
 *
 * Copyright (c) 2013 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las;

import las.lasFileDataStruct;

/** Class lasPlotLimitsUtility
 *  <p> This Class will provide basic utilities to build the
 *      LAS Plot Limits Data Array. This array allows the
 *      users to override the Plot Limits on the LAS Plot Track
 *
 *  @version 1.1 12/10/2013
 *  @author  John R. Victorine
 */

public class lasPlotLimitsUtility
{
  public static final int _GR       = las.lasStandardTools._GR;
  public static final int _CGR      = las.lasStandardTools._CGR;
  public static final int _SP       = las.lasStandardTools._SP;
  public static final int _CAL      = las.lasStandardTools._CAL;

  public static final int _PE       = las.lasStandardTools._PE;
  public static final int _RHOB     = las.lasStandardTools._RHOB;
  public static final int _DRHO     = las.lasStandardTools._DRHO;
  public static final int _DPHI     = las.lasStandardTools._DPHI;
  public static final int _NPHI     = las.lasStandardTools._NPHI;

  public static final int _SPHI     = las.lasStandardTools._SPHI;
  public static final int _DT       = las.lasStandardTools._DT;

  public static final int _COND     = las.lasStandardTools._COND;
  public static final int _CILD     = las.lasStandardTools._CILD;
  public static final int _CILM     = las.lasStandardTools._CILM;

  public static final int _RES      = las.lasStandardTools._RES;
  public static final int _RDEP     = las.lasStandardTools._RDEP;
  public static final int _RMED     = las.lasStandardTools._RMED;
  public static final int _RSHAL    = las.lasStandardTools._RSHAL;

  public static final int _ILD      = las.lasStandardTools._ILD;
  public static final int _ILM      = las.lasStandardTools._ILM;
  public static final int _SFLU     = las.lasStandardTools._SFLU;

  public static final int _LL       = las.lasStandardTools._LL;
  public static final int _MLL      = las.lasStandardTools._MLL;
  public static final int _LL8      = las.lasStandardTools._LL8;

  public static final int _LN       = las.lasStandardTools._LN;
  public static final int _SN       = las.lasStandardTools._SN;

  public static final int _MNOR     = las.lasStandardTools._MNOR;
  public static final int _MSFL     = las.lasStandardTools._MSFL;
  public static final int _MINV     = las.lasStandardTools._MINV;

  public static final int _AHT10    = las.lasStandardTools._AHT10;
  public static final int _AHT20    = las.lasStandardTools._AHT20;
  public static final int _AHT30    = las.lasStandardTools._AHT30;
  public static final int _AHT60    = las.lasStandardTools._AHT60;
  public static final int _AHT90    = las.lasStandardTools._AHT90;

  public static final int _TH       = las.lasStandardTools._TH;
  public static final int _U        = las.lasStandardTools._U;
  public static final int _K        = las.lasStandardTools._K;

  public static final int _GRN      = las.lasStandardTools._GRN;
  public static final int _NEUT     = las.lasStandardTools._NEUT;

  public static final int _PERM     = las.lasStandardTools._PERM;

  public static final int _TEMP     = las.lasStandardTools._TEMP;

  public static final int _LIN_1    = las.lasStandardTools._LIN_1;
  public static final int _LIN_2    = las.lasStandardTools._LIN_2;
  public static final int _LIN_3    = las.lasStandardTools._LIN_3;
  public static final int _LIN_4    = las.lasStandardTools._LIN_4;

  public static final int _LOG_1    = las.lasStandardTools._LOG_1;
  public static final int _LOG_2    = las.lasStandardTools._LOG_2;
  public static final int _LOG_3    = las.lasStandardTools._LOG_3;
  public static final int _LOG_4    = las.lasStandardTools._LOG_4;

  public static final int _PHI      = las.lasStandardTools._PHI;
  public static final int _GMCC     = las.lasStandardTools._GMCC;

  public static final int _RTo      = las.lasStandardTools._RTo;
  public static final int _RWA      = las.lasStandardTools._RWA;
  public static final int _RO       = las.lasStandardTools._RO;
  public static final int _RXRT     = las.lasStandardTools._RXRT;
  public static final int _MA       = las.lasStandardTools._MA;
  public static final int _PHITo    = las.lasStandardTools._PHITo;
  public static final int _PHIE     = las.lasStandardTools._PHIE;
  public static final int _SW       = las.lasStandardTools._SW;
  public static final int _BVW      = las.lasStandardTools._BVW;

  public static final int _ATT1     = las.lasStandardTools._ATT1;
  public static final int _ATT2     = las.lasStandardTools._ATT2;
  public static final int _ATT3     = las.lasStandardTools._ATT3;
  public static final int _ATT4     = las.lasStandardTools._ATT4;
  public static final int _ATT5     = las.lasStandardTools._ATT5;
  public static final int _ATT6     = las.lasStandardTools._ATT6;

  public static final int _VCAL     = las.lasStandardTools._VCAL;
  public static final int _VQTZ     = las.lasStandardTools._VQTZ;
  public static final int _VDOL     = las.lasStandardTools._VDOL;
  public static final int _VSH      = las.lasStandardTools._VSH;
  public static final int _VANH     = las.lasStandardTools._VANH;

  public static final int _RHOMAA   = las.lasStandardTools._RHOMAA;
  public static final int _UMAA     = las.lasStandardTools._UMAA;
  public static final int _DTMAA    = las.lasStandardTools._DTMAA;
  public static final int _PHIDIFF  = las.lasStandardTools._PHIDIFF;

  public static final int _THU      = las.lasStandardTools._THU;
  public static final int _THK      = las.lasStandardTools._THK;

  public static final int _TEMPDIFF = las.lasStandardTools._TEMPDIFF;

  public static final int ROWS      = las.lasStandardTools.TOTAL;

  public static final int MINIMUM   = 0;
  public static final int MAXIMUM   = 1;
  public static final int GROUP     = 2;
  public static final int LAS       = 3;
  public static final int COLUMNS   = 4;

  /** Method initialize()
   *  <p> This method will initialize the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   st   = LAS File Data Structure
   * @return  dOUT = Array of data limits
   */

  public static double[][] initialize(lasFileDataStruct st)
  {
    double dOUT[][] = null;
    double dMinimum = 0.0;
    double dMaximum = 0.0;

    dOUT = new double[ROWS][COLUMNS];

    for (int i=0; i<ROWS; i++)
    {
      dOUT[i][MINIMUM] = cmn.cmnString.stringToDouble(
                           las.lasStandardTools.LAS_TOOLS[i][4]);
      dOUT[i][MAXIMUM] = cmn.cmnString.stringToDouble(
                           las.lasStandardTools.LAS_TOOLS[i][5]);
      dOUT[i][GROUP]   = -1.0;
      dOUT[i][LAS]     = -1.0;

      // Set Curves with Data that will allow change of Limits
      // Add Minimum & Maximums that deviate from standard values

  	  if (st != null)
  	  {
        if (st.checkData(i))
        {
	      dOUT[i][LAS] = 1.0;

          switch (i)
          {
            case _SP:
              dOUT[i][MINIMUM] = st.dSPMin;
              dOUT[i][MAXIMUM] = st.dSPMax;
              break;

            case _RHOB:
              dOUT[i][MINIMUM] = st.dRhoMin;
              dOUT[i][MAXIMUM] = st.dRhoMax;
              break;

            case _DPHI:
            case _NPHI:
            case _SPHI:
              dOUT[i][MINIMUM] = st.dPHIMin;
              dOUT[i][MAXIMUM] = st.dPHIMax;
              break;

            case _TEMP:
              dOUT[i][MINIMUM] = st.dTEMPMin;
              dOUT[i][MAXIMUM] = st.dTEMPMax;
              break;

            case _TEMPDIFF:
              dOUT[i][MINIMUM] = st.dTEMPDIFFMin;
              dOUT[i][MAXIMUM] = st.dTEMPDIFFMax;
              break;

            case _LIN_1:
              dOUT[i][MINIMUM] = st.dLIN_1Min;
              dOUT[i][MAXIMUM] = st.dLIN_1Max;
              break;

            case _LIN_2:
              dOUT[i][MINIMUM] = st.dLIN_2Min;
              dOUT[i][MAXIMUM] = st.dLIN_2Max;
              break;

            case _LIN_3:
              dOUT[i][MINIMUM] = st.dLIN_3Min;
              dOUT[i][MAXIMUM] = st.dLIN_3Max;
              break;

            case _LIN_4:
              dOUT[i][MINIMUM] = st.dLIN_4Min;
              dOUT[i][MAXIMUM] = st.dLIN_4Max;
              break;

            case _LOG_1:
              dOUT[i][MINIMUM] = st.dLOG_1Min;
              dOUT[i][MAXIMUM] = st.dLOG_1Max;
              break;

            case _LOG_2:
              dOUT[i][MINIMUM] = st.dLOG_2Min;
              dOUT[i][MAXIMUM] = st.dLOG_2Max;
              break;

            case _LOG_3:
              dOUT[i][MINIMUM] = st.dLOG_3Min;
              dOUT[i][MAXIMUM] = st.dLOG_3Max;
              break;

            case _LOG_4:
              dOUT[i][MINIMUM] = st.dLOG_4Min;
              dOUT[i][MAXIMUM] = st.dLOG_4Max;
              break;

            case _GRN:
              dOUT[i][MINIMUM] = st.dGRNMinP;
              dOUT[i][MAXIMUM] = st.dGRNMaxP;
              break;

            case _NEUT:
              dOUT[i][MINIMUM] = st.dNEUTMinP;
              dOUT[i][MAXIMUM] = st.dNEUTMaxP;
              break;

            case _RXRT:
              dOUT[i][MINIMUM] = st.dRXRTMin;
              dOUT[i][MAXIMUM] = st.dRXRTMax;
              break;
   		  }
	    }
	  }

      // Identify Groups for Curves

      switch (i)
      {
        case _GR:
        case _CGR:
          dOUT[i][GROUP] = (double) _GR;
          break;
        case _SP:
        case _CAL:
          break;

        case _PE:
          break;
        case _RHOB:
          break;
        case _DRHO:
          break;

        case _DPHI:
        case _NPHI:
        case _SPHI:
          dOUT[i][GROUP] = (double) _NPHI;
          break;

        case _DT:
          break;

        case _COND:
        case _CILD:
        case _CILM:
          dOUT[i][GROUP] = (double) _CILM;
          break;

        case _RES:
        case _RDEP:
        case _RMED:
        case _RSHAL:
          dOUT[i][GROUP] = (double) _RSHAL;
          break;

        case _ILD:
        case _ILM:
        case _SFLU:

        case _LL:
        case _MLL:
        case _LL8:
          dOUT[i][GROUP] = (double) _LL8;
          break;

        case _LN:
        case _SN:
          dOUT[i][GROUP] = (double) _LN;
          break;

        case _MNOR:
        case _MSFL:
        case _MINV:
          dOUT[i][GROUP] = (double) _MNOR;
          break;

        case _AHT10:
        case _AHT20:
        case _AHT30:
        case _AHT60:
        case _AHT90:
          dOUT[i][GROUP] = (double) _AHT10;
          break;

        case _TH:
        case _U:
        case _K:
          break;

        case _GRN:
        case _NEUT:
          break;

        case _PERM:
          break;

        case _TEMP:
        case _TEMPDIFF:
          break;

        case _LIN_1:
        case _LIN_2:
        case _LIN_3:
        case _LIN_4:
          break;

        case _LOG_1:
        case _LOG_2:
        case _LOG_3:
        case _LOG_4:
          dOUT[i][GROUP] = (double) _LOG_1;
          break;

        case _RHOMAA:
        case _UMAA:
        case _DTMAA:
        case _PHIDIFF:
          break;

        case _THU:
        case _THK:
          break;

        case _PHI:
        case _GMCC:
          break;

        case _RTo:
        case _RWA:
        case _RO:
        case _RXRT:
        case _MA:
          break;

        case _PHITo:
        case _PHIE:
          break;

        case _SW:
        case _BVW:
          break;

        case _ATT1:
        case _ATT2:
        case _ATT3:
        case _ATT4:
        case _ATT5:
        case _ATT6:
          break;

        case _VCAL:
        case _VQTZ:
        case _VDOL:
        case _VSH:
        case _VANH:
          break;
	  }
    }

    return (dOUT);
  }

  /** Method copy()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dIN  = Array of data limits
   * @return  dOUT = Array of data limits copied from dIN
   */

  public static double[][] copy( double dIN[][] )
  {
    double dOUT[][] = null;
    double iColumns = 3;

    if (dIN != null)
    {
      dOUT = new double[ROWS][COLUMNS];

      for (int i=0; i<ROWS; i++)
      {
		for (int j=0; j<COLUMNS; j++)
		{
		  dOUT[i][j] = dIN[i][j];
		}
      }
    }

    return (dOUT);
  }

  /** Method setByGroup()
   * <p>  This method will copy the LAS Plot Limits data Array
   *      There are 3 columns Minimum, Maximum, Group, Log Curve Present.
   *      The Group is used to force other curves that reside on the same
   *      plot track, i.e. resistivity, permeability, porosity and
   *      the Log Curve Present is an indicator to include this curve.
   *
   * @param   dGroup = Group ID
   * @param   dMin   = Minimum Value for Group
   * @param   dMax   = Maximum Value for Group
   * @param   data   = Array of data limits
   * @return  data   = Array of data limits normalized by group
   */

  public static double[][] setByGroup( double dGroup, double dMin, double dMax, double data[][] )
  {
    if (data != null)
    {
      for (int i=0; i<ROWS; i++)
      {
		if ((dGroup == data[i][GROUP]) && (data[i][GROUP] > -1))
		{
          data[i][MINIMUM] = dMin;
          data[i][MAXIMUM] = dMax;
		}
      }
    }

    return (data);
  }
}

/*
 *  @version 1.1 12/10/2013
 *  @author  John Victorine
 */
