/*
 * @lasFileDataUtility.java Version 1.1 09/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las;

import las.lasFileDataStruct;
import las.lasDataListStruct;
import las3.las3Struct;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratControlStruct;

import cmn.cmnLASFileStruct;
import cmn.cmnLASCurveListStruct;

/** Class lasFileDataUtility
 *  <p> This Class will provide basic utilities for the LAS Curves
 *      data structures.
 *
 *  @version 1.1 09/05/2007
 *  @author  John R. Victorine
 */

public class lasFileDataUtility
{
  // Resistivity/Porosity Track Macros

  public static final int _RT   = iqstrat.iqstratTracksStruct._RT;
  public static final int _PHIT = iqstrat.iqstratTracksStruct._PHIT;
  public static final int _GRT  = iqstrat.iqstratTracksStruct._GRT;

  // Resistivity Lithology Track

  public static final int _MMHOS_M    = iqstrat.iqstratTracksStruct._MMHOS_M;
  public static final int MMHOS_M[][] = iqstrat.iqstratTracksStruct.MMHOS_M;

  // Porosity Lithology Track

  public static final int _PU    = iqstrat.iqstratTracksStruct._PU;
  public static final int PU[][] = iqstrat.iqstratTracksStruct.PU;

  /* ==================================================================== *
   * --------------- LAS File Well Data Structure Methods --------------- *
   * ==================================================================== */

  /** Method addConversionData()
   * <p> This method will load the Conversion Parameters to the LAS File Data
   *     Structure from the Control Plot Data Structure
   * @param  st  = Control Plot Data Structure
   * @param  stL = LAS File Data Structure
   * @return stL = LAS File Data Structure
   */

  public static lasFileDataStruct addConversionData( iqstratControlStruct st,
                                                     lasFileDataStruct stL )
  {
    double dRange[] = { 0.0, 0.0 };
    int    iTemp    = 0;
    double dTemp    = 0.0;
    double dMin     = 0.0;
    double dMax     = 0.0;
    double dPercMin = 0.0;
    double dPercMax = 100.0;
    double dMinNew  = 0.0;
    double dMaxNew  = 0.0;

    if ((st != null) && (stL != null))
    {
      // Data Conversion Variables

      if (stL.iGRN > -1)
      {
        // -- Gamma Ray Counts to Gamma Ray API

        stL.dGRNMin   = st.dGRNMin;   // Minimum Gamma Ray Counts Value
        stL.dGRNMax   = st.dGRNMax;   // Maximum Gamma Ray Counts Value

        stL.dGRCMin   = st.dGRCMin;   // Minimum Gamma Ray Value
        stL.dGRCMax   = st.dGRCMax;   // Maximum Gamma Ray Value

        dRange = math.mathLAS.computeCurveMinMax( stL.dGRN, stL.dNull );
        dMin = dRange[0];
        dMax = dRange[1];

        dPercMin = stL.dGRNMin;
        dPercMax = stL.dGRNMax;

        iTemp = (int) ( (dMin + dMax * dPercMin/100.0) * 1000.0 );
        dTemp = (double) iTemp / 1000.0;
        dMinNew = dTemp;

        iTemp = (int) ( (dMax * dPercMax / 100.0) * 1000.0 );
        dTemp = (double) iTemp / 1000.0;
        dMaxNew = dTemp;

        // Compute GR from GR Counts

        stL.dGR = math.mathLAS.computeGR( stL.dGRN,    stL.dNull,
                                          dMaxNew,     dMinNew,
                                          stL.dGRCMax, stL.dGRCMin );
      }

      if (stL.iNEUT > -1)
      {
        // -- Neutron Counts to Neutron Porosity

        stL.dNEUTMin  = st.dNEUTMin;  // Minimum Neutron Counts Value
        stL.dNEUTMax  = st.dNEUTMax;  // Maximum Neutron Counts Value

        stL.dNPHICMin = st.dNPHICMin; // Minimum NPHI Value
        stL.dNPHICMax = st.dNPHICMax; // Maximum NPHI Value

        dRange = math.mathLAS.computeCurveMinMax( stL.dNEUT, stL.dNull );
        dMin = dRange[0];
        dMax = dRange[1];

        dPercMin = stL.dNEUTMin;
        dPercMax = stL.dNEUTMax;

        iTemp = (int) ( (dMin + dMax * dPercMin/100.0) * 1000.0 );
        dTemp = (double) iTemp / 1000.0;
        dMinNew = dTemp;

        iTemp = (int) ( (dMax * dPercMax / 100.0) * 1000.0 );
        dTemp = (double) iTemp / 1000.0;
        dMaxNew = dTemp;

        // Compute NPHI from Neutron Counts

        stL.dNPHI = math.mathLAS.computeNPHI( stL.dNEUT,     stL.dNull,
                                              dMaxNew,       dMinNew,
                                              stL.dNPHICMax, stL.dNPHICMin );
      }
    }

    return (stL);
  }

  /** Method copy()
   * <p> This method will copy only the LAS3 Variables from the LAS File Data
   *     Structure to the New LAS File Data Structure
   * @param   stOld = LAS File Data Structure
   * @param   stNew = LAS File Data Structure
   * @return  stNew = LAS File Data Structure
   */

  public static lasFileDataStruct copy( lasFileDataStruct stOld,
                                        lasFileDataStruct stNew )
  {
    int i,j;

    if ((stOld != null) && (stNew != null))
    {
      stNew.iSource    = stOld.iSource;

      // -- Required Parameters

      stNew.sPDAT      = new String( stOld.sPDAT ); // Permanent Data
      stNew.dAPD       = stOld.dAPD;                // Above Permanent Data
      stNew.sDREF      = new String (stOld.sDREF ); // Depth Reference (KB,DF,CB)
      stNew.dElevation = stOld.dElevation;     // "Elevation of Depth Reference"

      // -- Required parameters for AEUB compliance

      stNew.dTDD       = stOld.dTDD;  // Drillers Depth
      stNew.dTLD       = stOld.dTLD;  // Loggers Depth

      stNew.dCSGL      = stOld.dCSGL; // Casing Bottom Logger
      stNew.dCSGD      = stOld.dCSGD; // Casing Bottom Driller
      stNew.dCSGS      = stOld.dCSGS; // Casing Size
      stNew.dCSGW      = stOld.dCSGW; // Casing Weight
      stNew.dBS        = stOld.dBS;   // Bit Size

      stNew.sMUD       = new String(stOld.sMUD);  // Mud type
      stNew.sMUDS      = new String(stOld.sMUDS); // Mud Source
      stNew.dMUDD      = stOld.dMUDD; // Mud Density
      stNew.dMUDV      = stOld.dMUDV; // Mud Viscosity (Funnel)

      stNew.dFL        = stOld.dFL;   // Fluid Loss
      stNew.dPH        = stOld.dPH;   // PH

      stNew.dRM        = stOld.dRM;   // Resistivity of Mud
      stNew.dRMT       = stOld.dRMT;  // Temperature of Mud
      stNew.dRMF       = stOld.dRMF;  // Resistivity of Mud Filtrate
      stNew.dRMFT      = stOld.dRMFT; // Temperature of Mud Filtrate
      stNew.dRMC       = stOld.dRMC;  // Resistivity of Mud Cake
      stNew.dRMCT      = stOld.dRMCT; // Temperature of Mud Cake
      stNew.dTMAX      = stOld.dTMAX; // Maximum Recorded Temp.

      stNew.sTIMC      = new String(stOld.sTIMC); // Date/Time Circulation Stopped
      stNew.sTIML      = new String(stOld.sTIML); // Date/Time Logger Tagged Bottom
      stNew.sUNIT      = new String(stOld.sUNIT); // Logging Unit Number
      stNew.sBASE      = new String(stOld.sBASE); // Home Base of Logging Unit
      stNew.sENG       = new String(stOld.sENG);  // Recording Engineer
      stNew.sWIT       = new String(stOld.sWIT);  // Witnessed By

      // .. Units for Required parameters for AEUB compliance

      stNew.sCSGL      = new String(stOld.sCSGL); // Casing Bottom Logger
      stNew.sCSGD      = new String(stOld.sCSGD); // Casing Bottom Driller
      stNew.sCSGS      = new String(stOld.sCSGS); // Casing Size
      stNew.sCSGW      = new String(stOld.sCSGW); // Casing Weight
      stNew.sBS        = new String(stOld.sBS); // Bit Size
      stNew.sMUDD      = new String(stOld.sMUDD); // Mud Density
      stNew.sMUDV      = new String(stOld.sMUDV); // Mud Viscosity (Funnel)
      stNew.sFL        = new String(stOld.sFL); // Fluid Loss
      stNew.sRM        = new String(stOld.sRM); // Resistivity of Mud
      stNew.sRMT       = new String(stOld.sRMT); // Temperature of Mud
      stNew.sRMF       = new String(stOld.sRMF); //  Rest. of Mud Filtrate
      stNew.sRMFT      = new String(stOld.sRMFT); // Temp. of Mud Filtrate
      stNew.sRMC       = new String(stOld.sRMC); // Rest. of Mud Cake
      stNew.sRMCT      = new String(stOld.sRMCT); // Temp. of Mud Cake
      stNew.sTMAX      = new String(stOld.sTMAX); // Max. Recorded Temp.

      // Unknown Curve Mnemonics

      stNew.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		}
	  }

      // LAS 3 File Parameters

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns
      stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

      for (i=0; i<stNew.iParamRows; i++)
      {
        for (j=0; j<stNew.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
        }
      }

      // Unknown Curve Mnemonics

      stNew.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		}
	  }
    }

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy the LAS Data from one file to another.
   * @param   stOld = LAS File Data Structure
   * @return  stNew = LAS File Data Structure
   */

  public static lasFileDataStruct copy(lasFileDataStruct stOld)
  {
    int i,j;
    lasFileDataStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new lasFileDataStruct();

      stNew.iSource    = stOld.iSource;

      stNew.sKID       = new String(stOld.sKID); // Primary Key for the KGS Well
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key created by user
      stNew.iType      = stOld.iType;            // Type of Data

      stNew.sAPI       = new String(stOld.sAPI);  // API-Number of Well
      stNew.sName      = new String(stOld.sName); // Well Name

      // Other Fields not really displayed

      stNew.state      = new String(stOld.state);    // State Name
      stNew.sCounty    = new String(stOld.sCounty); // County Name

      // Well Location Information

      stNew.dLatitude  = stOld.dLatitude;  // Latitude of Well
      stNew.dLongitude = stOld.dLongitude; // Longitude of Well

      // Well Parameter Information

      stNew.dKB        = stOld.dKB; // Kelly Bushing of Well
      stNew.dGL        = stOld.dGL; // Ground Level of Well
      stNew.dTD        = stOld.dTD; // Total Depth of Well

      // Well Status

      stNew.status     = new String(stOld.status); // Status of Well

      // LAS File Information

      stNew.sReadMethod = new String(stOld.sReadMethod);  // File Read Method
      stNew.sDirectory  = new String(stOld.sDirectory); // Directory Path
      stNew.sFilename   = new String(stOld.sFilename);  // File name of LAS File
      stNew.sVersion    = new String(stOld.sVersion);   // LAS Version Number

      stNew.dNull       = stOld.dNull;      // Null Value of Data
      stNew.depthStart  = stOld.depthStart; // Start Depth of LAS Data
      stNew.depthEnd    = stOld.depthEnd;   // Ending Depth of LAS Data
      stNew.depthStep   = stOld.depthStep;  // Depth Step of LAS DAta

      stNew.dTimeStart  = stOld.dTimeStart; // Start Time of LAS Data
      stNew.dTimeEnd    = stOld.dTimeEnd;   // Ending Time of LAS Data

      stNew.iProprietary = stOld.iProprietary; // Proprietary Data Identfier

      // -- Required Parameters

      stNew.sPDAT      = new String( stOld.sPDAT ); // Permanent Data
      stNew.dAPD       = stOld.dAPD;                // Above Permanent Data
      stNew.sDREF      = new String (stOld.sDREF ); // Depth Reference (KB,DF,CB)
      stNew.dElevation = stOld.dElevation;     // "Elevation of Depth Reference"

      // -- Required parameters for AEUB compliance

      stNew.dTDD       = stOld.dTDD;  // Drillers Depth
      stNew.dTLD       = stOld.dTLD;  // Loggers Depth

      stNew.dCSGL      = stOld.dCSGL; // Casing Bottom Logger
      stNew.dCSGD      = stOld.dCSGD; // Casing Bottom Driller
      stNew.dCSGS      = stOld.dCSGS; // Casing Size
      stNew.dCSGW      = stOld.dCSGW; // Casing Weight
      stNew.dBS        = stOld.dBS;   // Bit Size

      stNew.sMUD       = new String(stOld.sMUD);  // Mud type
      stNew.sMUDS      = new String(stOld.sMUDS); // Mud Source
      stNew.dMUDD      = stOld.dMUDD; // Mud Density
      stNew.dMUDV      = stOld.dMUDV; // Mud Viscosity (Funnel)

      stNew.dFL        = stOld.dFL;   // Fluid Loss
      stNew.dPH        = stOld.dPH;   // PH

      stNew.dRM        = stOld.dRM;   // Resistivity of Mud
      stNew.dRMT       = stOld.dRMT;  // Temperature of Mud
      stNew.dRMF       = stOld.dRMF;  // Resistivity of Mud Filtrate
      stNew.dRMFT      = stOld.dRMFT; // Temperature of Mud Filtrate
      stNew.dRMC       = stOld.dRMC;  // Resistivity of Mud Cake
      stNew.dRMCT      = stOld.dRMCT; // Temperature of Mud Cake
      stNew.dTMAX      = stOld.dTMAX; // Maximum Recorded Temp.

      stNew.sTIMC      = new String(stOld.sTIMC); // Date/Time Circulation Stopped
      stNew.sTIML      = new String(stOld.sTIML); // Date/Time Logger Tagged Bottom
      stNew.sUNIT      = new String(stOld.sUNIT); // Logging Unit Number
      stNew.sBASE      = new String(stOld.sBASE); // Home Base of Logging Unit
      stNew.sENG       = new String(stOld.sENG);  // Recording Engineer
      stNew.sWIT       = new String(stOld.sWIT);  // Witnessed By

      // .. Units for Required parameters for AEUB compliance

      stNew.sCSGL      = new String(stOld.sCSGL); // Casing Bottom Logger
      stNew.sCSGD      = new String(stOld.sCSGD); // Casing Bottom Driller
      stNew.sCSGS      = new String(stOld.sCSGS); // Casing Size
      stNew.sCSGW      = new String(stOld.sCSGW); // Casing Weight
      stNew.sBS        = new String(stOld.sBS); // Bit Size
      stNew.sMUDD      = new String(stOld.sMUDD); // Mud Density
      stNew.sMUDV      = new String(stOld.sMUDV); // Mud Viscosity (Funnel)
      stNew.sFL        = new String(stOld.sFL); // Fluid Loss
      stNew.sRM        = new String(stOld.sRM); // Resistivity of Mud
      stNew.sRMT       = new String(stOld.sRMT); // Temperature of Mud
      stNew.sRMF       = new String(stOld.sRMF); //  Rest. of Mud Filtrate
      stNew.sRMFT      = new String(stOld.sRMFT); // Temp. of Mud Filtrate
      stNew.sRMC       = new String(stOld.sRMC); // Rest. of Mud Cake
      stNew.sRMCT      = new String(stOld.sRMCT); // Temp. of Mud Cake
      stNew.sTMAX      = new String(stOld.sTMAX); // Max. Recorded Temp.

      // LAS 3 File Parameters

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns
      stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

      for (i=0; i<stNew.iParamRows; i++)
      {
        for (j=0; j<stNew.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
        }
      }

      stNew.iLogs = stOld.iLogs;  // Total number of LAS3 Data Structures
      if (stOld.iLogs > 0)
      {
        stNew.stLAS3 = new las3Struct[stOld.iLogs];
        for (i=0; i<stOld.iLogs; i++)
        {
          stNew.stLAS3[i] = las3.las3Utility.copyWOData( stOld.stLAS3[i] );
        }
      }

      // Unknown Curve Mnemonics

      stNew.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		}
	  }

      // LAS File Curve Information

      stNew.iColorlithDefault = stOld.iColorlithDefault; // Default colorlith

      stNew.iGR       = stOld.iGR;       // Gamma Ray Curve Number in LAS File
      stNew.iCGR      = stOld.iCGR;      // Gamma Ray Minus Uranium Curve Number
      stNew.iSP       = stOld.iSP;       // Spontaneous Potential Log Values
      stNew.iCAL      = stOld.iCAL;      // Caliper Log Values

      stNew.iNPHI     = stOld.iNPHI;     // Neutron Porosity Curve Number in LAS File
      stNew.iPE       = stOld.iPE;       // Photoelectric Curve Number in LAS File
      stNew.iRHOB     = stOld.iRHOB;     // Bulk Density Curve Number in LAS File
      stNew.iDPHI     = stOld.iDPHI;     // Density Porosity Curve Number in LAS File
      stNew.iRHO      = stOld.iRHO;      // Density Correction Curve Number in LAS File

      stNew.iDT       = stOld.iDT;       // Sonic Curve Number in LAS File
      stNew.iSPHI     = stOld.iSPHI;     // Sonic Porisity Curve Number in LAS File

      stNew.iCOND     = stOld.iCOND;     // Conductivity
      stNew.iCILD     = stOld.iCILD;     // Deep Induction Conductivity
      stNew.iCILM     = stOld.iCILM;     // Medium Induction Conductivity

      stNew.iRES      = stOld.iRES;      // Resistivity
      stNew.iRMED     = stOld.iRMED;     // Deep Resistivity
      stNew.iRDEP     = stOld.iRDEP;     // Medium Resistivity
      stNew.iRSHAL    = stOld.iRSHAL;    // Shallow Resistivity,

      stNew.iILD      = stOld.iILD;      // Deep Induction Resistivity
      stNew.iILM      = stOld.iILM;      // Medium Induction Resistivity
      stNew.iSFLU     = stOld.iSFLU;     // Spherically Focused Resistivity

      stNew.iLL       = stOld.iLL;       // Deep Laterolog Resistivity
      stNew.iMLL      = stOld.iMLL;      // Micro Laterolog Resistivity
      stNew.iLL8      = stOld.iLL8;      // Shallow Laterolog Resistivity

      stNew.iLN       = stOld.iLN;       // Long Normal Resistivity
      stNew.iSN       = stOld.iSN;       // Shallow Normal Resistivity

      stNew.iMNOR     = stOld.iMNOR;     // Micro Normal Resistivity
      stNew.iMSFL     = stOld.iMSFL;     // Micro Spherically Focused Resistivity
      stNew.iMINV     = stOld.iMINV;     // Micro Inverse Resistivity

      stNew.iAHT10    = stOld.iAHT10;    // Induction Resistivity-10
      stNew.iAHT20    = stOld.iAHT20;    // Induction Resistivity-20
      stNew.iAHT30    = stOld.iAHT30;    // Induction Resistivity-30
      stNew.iAHT60    = stOld.iAHT60;    // Induction Resistivity-60
      stNew.iAHT90    = stOld.iAHT90;    // Induction Resistivity-90

      stNew.iTh       = stOld.iTh;       // Thorium Curve Number in LAS File
      stNew.iU        = stOld.iU;        // Uranium Curve Number in LAS File
      stNew.iK        = stOld.iK;        // Potasium Curve Number in LAS File

      stNew.iGRN      = stOld.iGRN;      // Gamma Ray Count
      stNew.iNEUT     = stOld.iNEUT;     // Neutron Counts
      stNew.iPERM     = stOld.iPERM;     // Permeability

      stNew.iTEMP     = stOld.iTEMP;     // Temperature
      stNew.iTEMPDIFF = stOld.iTEMPDIFF; // Temperature Difference

      stNew.iLIN_1    = stOld.iLIN_1;    // Linear Curve 1
      stNew.iLIN_2    = stOld.iLIN_2;    // Linear Curve 2
      stNew.iLIN_3    = stOld.iLIN_3;    // Linear Curve 3
      stNew.iLIN_4    = stOld.iLIN_4;    // Linear Curve 4

      stNew.iLOG_1    = stOld.iLOG_1;    // Log Curve 1
      stNew.iLOG_2    = stOld.iLOG_2;    // Log Curve 2
      stNew.iLOG_3    = stOld.iLOG_3;    // Log Curve 3
      stNew.iLOG_4    = stOld.iLOG_4;    // Log Curve 4

      stNew.iUmaa     = stOld.iUmaa;     // Apparent Volumetric Photoelectric factor
      stNew.iRhomaa   = stOld.iRhomaa;   // Apparent Matrix Grain Density
      stNew.iDTmaa    = stOld.iDTmaa;    // Apparent Matrix Grain Acoustic transit time

        // Alternate Lithology Curves

      stNew.iNeutron  = stOld.iNeutron;  // Neutron Porosity
      stNew.iDelta    = stOld.iDelta;    // Neutron - Density Porosity

      // Spectral Gamma Ray Lithology Curves

      stNew.iThU      = stOld.iThU;      // Th/U
      stNew.iThK      = stOld.iThK;      // Th/K

      // Zonation Data

      stNew.iPHI      = stOld.iPHI;      // Measured Core Porosity
      stNew.iGMCC     = stOld.iGMCC;     // Measured Core Grain Density

      stNew.iRT       = stOld.iRT;       // Total Resistivity
      stNew.iRWA      = stOld.iRWA;      // Apparent Water Resistivity
      stNew.iRO       = stOld.iRO;       // Water Saturated Rock Resistivity
      stNew.iRXRT     = stOld.iRXRT;     // Quick Look Ratio
      stNew.iMA       = stOld.iMA;       // Archie Cementation
      stNew.iPHIT     = stOld.iPHIT;     // Total Porosity
      stNew.iPHIE     = stOld.iPHIE;     // Effective Porosity
      stNew.iSW       = stOld.iSW;       // Water Saturation
      stNew.iBVW      = stOld.iBVW;      // Bulk Volume Water

      stNew.iATT1     = stOld.iATT1;     // Att 1
      stNew.iATT2     = stOld.iATT2;     // Att 2
      stNew.iATT3     = stOld.iATT3;     // Att 3
      stNew.iATT4     = stOld.iATT4;     // Att 4
      stNew.iATT5     = stOld.iATT5;     // Att 5
      stNew.iATT6     = stOld.iATT6;     // Att 6

      stNew.iVCAL     = stOld.iVCAL;     // Calcite Volume Fraction
      stNew.iVQTZ     = stOld.iVQTZ;     // Quartz Volume Fraction
      stNew.iVDOL     = stOld.iVDOL;     // Dolomite Volume Fraction
      stNew.iVSH      = stOld.iVSH;      // Shale Volume Fraction
      stNew.iVANH     = stOld.iVANH;     // Anhydrite Volume Fraction

      // Lithofacies Log Curves

      stNew.iRows     = stOld.iRows;    // Number of Rows

      stNew.depths    = new double[stNew.iRows]; // Array of Depth Data Values
      stNew.dTime     = new double[stNew.iRows]; // Array of Time Data Values

      stNew.dGR       = new double[stNew.iRows]; // Array of Gamma Ray Log Values
      stNew.dCGR      = new double[stNew.iRows]; // Array of Gamma Ray minus Uranium Log Values
      stNew.dSP       = new double[stNew.iRows]; // Array of Spontaneous Potential Log Values
      stNew.dCAL      = new double[stNew.iRows]; // Array of Caliper Log Values

      stNew.dNPHI     = new double[stNew.iRows]; // Array of Neutron Porosity Log Values
      stNew.dPE       = new double[stNew.iRows]; // Array of PhotoElectric  Values
      stNew.dRHOB     = new double[stNew.iRows]; // Array of Bulk Density  Values
      stNew.dDPHI     = new double[stNew.iRows]; // Array of Density Porosity Log Values
      stNew.dRHO      = new double[stNew.iRows]; // Array of Density Corrrection Log Values

      stNew.dDT       = new double[stNew.iRows]; // Array of Sonic Log Values
      stNew.dSPHI     = new double[stNew.iRows]; // Array of Sonic Porosity Log Values

      stNew.dCOND     = new double[stNew.iRows]; // Array of Conductivity
      stNew.dCILD     = new double[stNew.iRows]; // Array of Deep Induction Conductivity
      stNew.dCILM     = new double[stNew.iRows]; // Array of Medium Induction Conductivity

      stNew.dRES      = new double[stNew.iRows]; // Array of Resistivity
      stNew.dRMED     = new double[stNew.iRows]; // Array of Deep Resistivity
      stNew.dRDEP     = new double[stNew.iRows]; // Array of Medium Resistivity
      stNew.dRSHAL    = new double[stNew.iRows]; // Array of Shallow Resistivity,

      stNew.dILD      = new double[stNew.iRows]; // Array of Deep Induction Resistivity
      stNew.dILM      = new double[stNew.iRows]; // Array of Medium Induction Resistivity
      stNew.dSFLU     = new double[stNew.iRows]; // Array of Spherically Focused Resistivity

      stNew.dLL       = new double[stNew.iRows]; // Array of Deep Laterolog Resistivity
      stNew.dMLL      = new double[stNew.iRows]; // Array of Micro Laterolog Resistivity
      stNew.dLL8      = new double[stNew.iRows]; // Array of Shallow Laterolog Resistivity

      stNew.dLN       = new double[stNew.iRows]; // Array of Long Normal Resistivity
      stNew.dSN       = new double[stNew.iRows]; // Array of Shallow Normal Resistivity

      stNew.dMNOR     = new double[stNew.iRows]; // Array of Micro Normal Resistivity
      stNew.dMSFL     = new double[stNew.iRows]; // Array of Micro Spherically Focused Resistivity
      stNew.dMINV     = new double[stNew.iRows]; // Array of Micro Inverse Resistivity

      stNew.dAHT10    = new double[stNew.iRows]; // Array Induction Resistivity-10
      stNew.dAHT20    = new double[stNew.iRows]; // Array Induction Resistivity-20
      stNew.dAHT30    = new double[stNew.iRows]; // Array Induction Resistivity-30
      stNew.dAHT60    = new double[stNew.iRows]; // Array Induction Resistivity-60
      stNew.dAHT90    = new double[stNew.iRows]; // Array Induction Resistivity-90

      stNew.dTh       = new double[stNew.iRows]; // Array of Thorium Log Values
      stNew.dU        = new double[stNew.iRows]; // Array of Uranium Log Values
      stNew.dK        = new double[stNew.iRows]; // Array of Potassium Log Values

      stNew.dGRN      = new double[stNew.iRows]; // Gamma Ray Count
      stNew.dNEUT     = new double[stNew.iRows]; // Neutron Counts
      stNew.dPERM     = new double[stNew.iRows]; // Permeability

      stNew.dTEMP     = new double[stNew.iRows]; // Temperature
      stNew.dTEMPDIFF = new double[stNew.iRows]; // Temperature Difference

      stNew.dLIN_1    = new double[stNew.iRows]; // Linear Curve 1
      stNew.dLIN_2    = new double[stNew.iRows]; // Linear Curve 2
      stNew.dLIN_3    = new double[stNew.iRows]; // Linear Curve 3
      stNew.dLIN_4    = new double[stNew.iRows]; // Linear Curve 4

      stNew.dLOG_1    = new double[stNew.iRows]; // Log Curve 1
      stNew.dLOG_2    = new double[stNew.iRows]; // Log Curve 2
      stNew.dLOG_3    = new double[stNew.iRows]; // Log Curve 3
      stNew.dLOG_4    = new double[stNew.iRows]; // Log Curve 4

      stNew.dUmaa     = new double[stNew.iRows]; // Apparent Volumetric Photoelectric
      stNew.dRhomaa   = new double[stNew.iRows]; // Apparent Matrix Grain Density
      stNew.dDTmaa    = new double[stNew.iRows]; // Apparent Matrix Grain Acoustic transit time

      stNew.dNeutron  = new double[stNew.iRows]; // Neutron Porosity
      stNew.delta     = new double[stNew.iRows]; // Neutron - Density Porosity

      stNew.dThU      = new double[stNew.iRows]; // Th/U
      stNew.dThK      = new double[stNew.iRows]; // Th/K

      stNew.dPHI      = new double[stNew.iRows]; // Measured Core Porosity
      stNew.dGMCC     = new double[stNew.iRows]; // Measured Core Grain Density

      stNew.dRT       = new double[stNew.iRows]; // Total Resistivity
      stNew.dRWA      = new double[stNew.iRows]; // Apparent Water Resistivity
      stNew.dRO       = new double[stNew.iRows]; // Water Saturated Rock Resistivity
      stNew.dRXRT     = new double[stNew.iRows]; // Quick Look Ratio
      stNew.dMA       = new double[stNew.iRows]; // Archie Cementation
      stNew.dPHIT     = new double[stNew.iRows]; // Total Porosity
      stNew.dPHIE     = new double[stNew.iRows]; // Effective Porosity
      stNew.dSW       = new double[stNew.iRows]; // Water Saturation
      stNew.dBVW      = new double[stNew.iRows]; // Bulk Volume Water

      stNew.dATT1     = new double[stNew.iRows]; // Att 1
      stNew.dATT2     = new double[stNew.iRows]; // Att 2
      stNew.dATT3     = new double[stNew.iRows]; // Att 3
      stNew.dATT4     = new double[stNew.iRows]; // Att 4
      stNew.dATT5     = new double[stNew.iRows]; // Att 5
      stNew.dATT6     = new double[stNew.iRows]; // Att 6

      stNew.dVCAL     = new double[stNew.iRows]; // Calcite Volume Fraction
      stNew.dVQTZ     = new double[stNew.iRows]; // Quartz Volume Fraction
      stNew.dVDOL     = new double[stNew.iRows]; // Dolomite Volume Fraction
      stNew.dVSH      = new double[stNew.iRows]; // Shale Volume Fraction
      stNew.dVANH     = new double[stNew.iRows]; // Anhydrite Volume Fraction

      for (i=0; i<stOld.iRows; i++)
      {
        stNew.depths[i]    = stOld.depths[i];    // Array of Depth Data Values
        stNew.dTime[i]     = stOld.dTime[i];     // Array of Time Data Values

        stNew.dGR[i]       = stOld.dGR[i];       // Array of Gamma Ray Log Values
        stNew.dCGR[i]      = stOld.dCGR[i];      // Array of Gamma Ray minus Uranium Log Values
        stNew.dSP[i]       = stOld.dSP[i];       // Array of Spontaneous Potential Log Values
        stNew.dCAL[i]      = stOld.dCAL[i];      // Array of Caliper Log Values

        stNew.dPE[i]       = stOld.dPE[i];       // Array of PhotoElectric Log Values
        stNew.dNPHI[i]     = stOld.dNPHI[i];     // Array of Neutron Porosity Log Values
        stNew.dRHOB[i]     = stOld.dRHOB[i];     // Array of Bulk Density Log Values
        stNew.dDPHI[i]     = stOld.dDPHI[i];     // Array of Density Porosity Log Values
        stNew.dRHO[i]      = stOld.dRHO[i];      // Array of Density Corrrection Log Values

        stNew.dDT[i]       = stOld.dDT[i];       // Array of Sonic Log Values
        stNew.dSPHI[i]     = stOld.dSPHI[i];     // Array of Sonic Porosity Log Values

        stNew.dCOND[i]     = stOld.dCOND[i];     // Array of Conductivity
        stNew.dCILD[i]     = stOld.dCILD[i];     // Array of Deep Induction Conductivity
        stNew.dCILM[i]     = stOld.dCILM[i];     // Array of Medium Induction Conductivity

        stNew.dRES[i]      = stOld.dRES[i];      // Array of Resistivity
        stNew.dRMED[i]     = stOld.dRMED[i];     // Array of Deep Resistivity
        stNew.dRDEP[i]     = stOld.dRDEP[i];     // Array of Medium Resistivity
        stNew.dRSHAL[i]    = stOld.dRSHAL[i];    // Array of Shallow Resistivity,

        stNew.dILD[i]      = stOld.dILD[i];      // Array of Deep Induction Resistivity
        stNew.dILM[i]      = stOld.dILM[i];      // Array of Medium Induction Resistivity
        stNew.dSFLU[i]     = stOld.dSFLU[i];     // Array of Spherically Focused Resistivity

        stNew.dLL[i]       = stOld.dLL[i];       // Array of Deep Laterolog Resistivity
        stNew.dMLL[i]      = stOld.dMLL[i];      // Array of Micro Laterolog Resistivity
        stNew.dLL8[i]      = stOld.dLL8[i];      // Array of Shallow Laterolog Resistivity

        stNew.dLN[i]       = stOld.dLN[i];       // Array of Long Normal Resistivity
        stNew.dSN[i]       = stOld.dSN[i];       // Array of Shallow Normal Resistivity

        stNew.dMNOR[i]     = stOld.dMNOR[i];     // Array of Micro Normal Resistivity
        stNew.dMSFL[i]     = stOld.dMSFL[i];     // Array of Micro Spherically Focused Resistivity
        stNew.dMINV[i]     = stOld.dMINV[i];     // Array of Micro Inverse Resistivity

        stNew.dAHT10[i]    = stOld.dAHT10[i];    // Array Induction Resistivity-10
        stNew.dAHT20[i]    = stOld.dAHT20[i];    // Array Induction Resistivity-20
        stNew.dAHT30[i]    = stOld.dAHT30[i];    // Array Induction Resistivity-30
        stNew.dAHT60[i]    = stOld.dAHT60[i];    // Array Induction Resistivity-60
        stNew.dAHT90[i]    = stOld.dAHT90[i];    // Array Induction Resistivity-90

        stNew.dTh[i]       = stOld.dTh[i];       // Array of Thorium Log Values
        stNew.dU[i]        = stOld.dU[i];        // Array of Uranium Log Values
        stNew.dK[i]        = stOld.dK[i];        // Array of Potassium Log Values

        stNew.dGRN[i]      = stOld.dGRN[i];      // Gamma Ray Count
        stNew.dNEUT[i]     = stOld.dNEUT[i];     // Neutron Counts
        stNew.dPERM[i]     = stOld.dPERM[i];     // Permeability

        stNew.dTEMP[i]     = stOld.dTEMP[i];     // Temperature
        stNew.dTEMPDIFF[i] = stOld.dTEMPDIFF[i]; // Temperature Difference

        stNew.dLIN_1[i]    = stOld.dLIN_1[i];    // Linear Curve 1
        stNew.dLIN_2[i]    = stOld.dLIN_2[i];    // Linear Curve 2
        stNew.dLIN_3[i]    = stOld.dLIN_3[i];    // Linear Curve 3
        stNew.dLIN_4[i]    = stOld.dLIN_4[i];    // Linear Curve 4

        stNew.dLOG_1[i]    = stOld.dLOG_1[i];    // Log Curve 1
        stNew.dLOG_2[i]    = stOld.dLOG_2[i];    // Log Curve 2
        stNew.dLOG_3[i]    = stOld.dLOG_3[i];    // Log Curve 3
        stNew.dLOG_4[i]    = stOld.dLOG_4[i];    // Log Curve 4

        // Rhomaa-Umaa Log Curves

        stNew.dUmaa[i]     = stOld.dUmaa[i];     // Apparent Volumetric Photoelectric
        stNew.dRhomaa[i]   = stOld.dRhomaa[i];   // Apparent Matrix Grain Density
        stNew.dDTmaa[i]    = stOld.dDTmaa[i];    // Apparent Matrix Grain Acoustic transit time

        // Alternate Lithology Curves

        stNew.dNeutron[i]  = stOld.dNeutron[i];  // Neutron Porosity
        stNew.delta[i]     = stOld.delta[i];     // Neutron - Density Porosity

        // Spectral Gamma Ray Lithology Curves

        stNew.dThU[i]      = stOld.dThU[i];      // Th/U
        stNew.dThK[i]      = stOld.dThK[i];      // Th/K

        // Zonation

        stNew.dPHI[i]      = stOld.dPHI[i];      // Array of Measured Core Porosity
        stNew.dGMCC[i]     = stOld.dGMCC[i];     // Array of Measured Core Grain Density

        stNew.dRT[i]       = stOld.dRT[i];       // Total Resistivity
        stNew.dRWA[i]      = stOld.dRWA[i];      // Apparent Water Resistivity
        stNew.dRO[i]       = stOld.dRO[i];       // Water Saturated Rock Resistivity
        stNew.dRXRT[i]     = stOld.dRXRT[i];     // Quick Look Ratio
        stNew.dMA[i]       = stOld.dMA[i];       // Archie Cementation
        stNew.dPHIT[i]     = stOld.dPHIT[i];     // Total Porosity
        stNew.dPHIE[i]     = stOld.dPHIE[i];     // Effective Porosity
        stNew.dSW[i]       = stOld.dSW[i];       // Water Saturation
        stNew.dBVW[i]      = stOld.dBVW[i];      // Bulk Volume Water

        stNew.dATT1[i]     = stOld.dATT1[i];     // Att 1
        stNew.dATT2[i]     = stOld.dATT2[i];     // Att 2
        stNew.dATT3[i]     = stOld.dATT3[i];     // Att 3
        stNew.dATT4[i]     = stOld.dATT4[i];     // Att 4
        stNew.dATT5[i]     = stOld.dATT5[i];     // Att 5
        stNew.dATT6[i]     = stOld.dATT6[i];     // Att 6

        stNew.dVCAL[i]     = stOld.dVCAL[i];     // Calcite Volume Fraction
        stNew.dVQTZ[i]     = stOld.dVQTZ[i];     // Quartz Volume Fraction
        stNew.dVDOL[i]     = stOld.dVDOL[i];     // Dolomite Volume Fraction
        stNew.dVSH[i]      = stOld.dVSH[i];      // Shale Volume Fraction
        stNew.dVANH[i]     = stOld.dVANH[i];     // Anhydrite Volume Fraction
      }

      // Misc

      if (stOld.dSPMin != 0.0) stNew.dSPMin   = stOld.dSPMin;
      if (stOld.dSPMax != 0.0) stNew.dSPMax   = stOld.dSPMax;

      stNew.dPHIMin      = stOld.dPHIMin;      // Minimum PHI Value
      stNew.dPHIMax      = stOld.dPHIMax;      // Maximum PHI Value

      stNew.dRhoMin      = stOld.dRhoMin;
      stNew.dRhoMax      = stOld.dRhoMax;

      stNew.dGRNMin      = stOld.dGRNMin;      // Minimum Gamma Ray Counts Value
      stNew.dGRNMax      = stOld.dGRNMax;      // Maximum Gamma Ray Counts Value

      stNew.dGRMin       = stOld.dGRMin;       // Minimum Gamma Ray Value
      stNew.dGRMax       = stOld.dGRMax;       // Maximum Gamma Ray Value

      stNew.dGRCMin      = stOld.dGRCMin;      // Minimum Gamma Ray Value
      stNew.dGRCMax      = stOld.dGRCMax;      // Maximum Gamma Ray Value

      stNew.dGRNMinP     = stOld.dGRNMinP;     // Minimum Gamma Ray Counts Value
      stNew.dGRNMaxP     = stOld.dGRNMaxP;     // Maximum Gamma Ray Counts Value

      stNew.dNEUTMin     = stOld.dNEUTMin;     // Minimum Neutron Counts Value
      stNew.dNEUTMax     = stOld.dNEUTMax;     // Maximum Neutron Counts Value

      stNew.dNPHIMin     = stOld.dNPHIMin;     // Minimum NPHI Value
      stNew.dNPHIMax     = stOld.dNPHIMax;     // Maximum NPHI Value

      stNew.dNPHICMin    = stOld.dNPHICMin;    // Minimum NPHI Value
      stNew.dNPHICMax    = stOld.dNPHICMax;    // Maximum NPHI Value

      stNew.dNEUTMinP    = stOld.dNEUTMinP;    // Minimum Neutron Counts Value
      stNew.dNEUTMaxP    = stOld.dNEUTMaxP;    // Maximum Neutron Counts Value

      stNew.dTEMPMin     = stOld.dTEMPMin;     // Minimum TEMP Value
      stNew.dTEMPMax     = stOld.dTEMPMax;     // Maximum TEMP Value

      stNew.dTEMPDIFFMin = stOld.dTEMPDIFFMin; // Minimum TEMP Difference Value
      stNew.dTEMPDIFFMax = stOld.dTEMPDIFFMax; // Maximum TEMP Differnece Value

      stNew.dRXRTMin     = stOld.dRXRTMin;
      stNew.dRXRTMax     = stOld.dRXRTMax;

      // .. Linear Curves

      stNew.dLIN_1Min    = stOld.dLIN_1Min; // Linear Curve 1 Minimum
      stNew.dLIN_2Min    = stOld.dLIN_2Min; // Linear Curve 2 Minimum
      stNew.dLIN_3Min    = stOld.dLIN_3Min; // Linear Curve 3 Minimum
      stNew.dLIN_4Min    = stOld.dLIN_4Min; // Linear Curve 4 Minimum

      stNew.dLIN_1Max    = stOld.dLIN_1Max; // Linear Curve 1 Maximum
      stNew.dLIN_2Max    = stOld.dLIN_2Max; // Linear Curve 2 Maximum
      stNew.dLIN_3Max    = stOld.dLIN_3Max; // Linear Curve 3 Maximum
      stNew.dLIN_4Max    = stOld.dLIN_4Max; // Linear Curve 4 Maximum

      // .. Semilog Curves

      stNew.dLOG_1Min    = stOld.dLOG_1Min; // Log Curve 1 Minimum
      stNew.dLOG_2Min    = stOld.dLOG_2Min; // Log Curve 2 Minimum
      stNew.dLOG_3Min    = stOld.dLOG_3Min; // Log Curve 3 Minimum
      stNew.dLOG_4Min    = stOld.dLOG_4Min; // Log Curve 4 Minimum

      stNew.dLOG_1Max    = stOld.dLOG_1Max; // Log Curve 1 Maximum
      stNew.dLOG_2Max    = stOld.dLOG_2Max; // Log Curve 2 Maximum
      stNew.dLOG_3Max    = stOld.dLOG_3Max; // Log Curve 3 Maximum
      stNew.dLOG_4Max    = stOld.dLOG_4Max; // Log Curve 4 Maximum

      stNew.stList   = iqstrat.iqstratShaleUtility.copyList(stOld.stList);
    }

    return (stNew);
  }

  /** Method transfer()
   *  <p> This method will transfer the contents of the data structure to the new and
   *      delete the old structure.
   * @param   stOld = LAS File Data Structure
   * @return  stNew = LAS File Data Structure
   */

  public static lasFileDataStruct transfer(lasFileDataStruct stOld)
  {
    lasFileDataStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copy( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method computeMath()
   * <p> This method will compute the lithology curves
   * @param  stC = Color Lithology Well Structure
   * @return stC = Color Lithology Well Structure
   */

  public static lasFileDataStruct computeMath(lasFileDataStruct stC)
  {
    int    i        = 0;
    int    idata    = 0;

    if (stC != null)
    {
      // Compute the Rhmaa Umaa Lithology Curves

      if ( (stC.iPE > -1) && (stC.iRHOB > -1) && (stC.iNPHI > -1))
      {
        double dOut[][] = null;
        stC.iColorlithDefault = iqstrat.iqstratTracksStruct._COLORLITH_RU;

        dOut = math.mathLAS.computeRhomaaUmaa( math.mathLAS._FRESH,
                                               stC.iRows,
                                               stC.dNPHI,
                                               stC.dRHOB,
                                               stC.dPE,
                                               stC.dNull );

        if (dOut != null)
        {
          for (i=0; i<stC.iRows; i++)
          {
            stC.dRhomaa[i] = dOut[0][i]; // Apparent Matrix Grain Density
            stC.dUmaa[i]   = dOut[1][i]; // Apparent Volumetric Photoelectric
          }
        }

        stC.iUmaa = 1; // Apparent Volumetric Photoelectric factor
        stC.iRhomaa = 1; // Apparent Matrix Grain Density
      }
      else if ( (stC.iPE > -1) && (stC.iRHOB > -1) && (stC.iDT > -1))
      {
        double dOut[][] = null;
        stC.iColorlithDefault = iqstrat.iqstratTracksStruct._COLORLITH_RU;

	    if (stC.iSPHI == -1)
	    {
		  stC.dSPHI = math.mathLAS.computePHI( stC.iRows,
                                               stC.dDT,
                                               math.mathLAS._SONIC,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH,
                                               stC.dNull );
          stC.iSPHI = 1;
		}

        dOut = math.mathLAS.computeRhomaaUmaa( math.mathLAS._FRESH,
                                               stC.iRows,
                                               stC.dSPHI,
                                               stC.dRHOB,
                                               stC.dPE,
                                               stC.dNull );

        if (dOut != null)
        {
          for (i=0; i<stC.iRows; i++)
          {
            stC.dRhomaa[i] = dOut[0][i]; // Apparent Matrix Grain Density
            stC.dUmaa[i]   = dOut[1][i]; // Apparent Volumetric Photoelectric
          }
        }

        stC.iUmaa = 1; // Apparent Volumetric Photoelectric factor
        stC.iRhomaa = 1; // Apparent Matrix Grain Density
      }
      else if ((stC.iRHOB > -1) && (stC.iNPHI > -1))
      {
        stC.iColorlithDefault = iqstrat.iqstratTracksStruct._COLORLITH_RN;

        stC.dRhomaa = math.mathLAS.computeRhomaa( math.mathLAS._FRESH,
                                                  stC.iRows,
                                                  stC.dNPHI,
                                                  stC.dRHOB,
                                                  stC.dNull );

        stC.iRhomaa = 1; // Apparent Matrix Grain Density
      }
      else if ((stC.iRHOB > -1) && (stC.iDT > -1))
      {
	    if (stC.iSPHI == -1)
	    {
		  stC.dSPHI = math.mathLAS.computePHI( stC.iRows,
                                               stC.dDT,
                                               math.mathLAS._SONIC,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH,
                                               stC.dNull );
          stC.iSPHI = 1;
		}

        stC.dRhomaa = math.mathLAS.computeRhomaa( math.mathLAS._FRESH,
                                                  stC.iRows,
                                                  stC.dSPHI,
                                                  stC.dRHOB,
                                                  stC.dNull );

        stC.iRhomaa = 1; // Apparent Matrix Grain Density
      }
      else if (stC.iRHOB > -1)
      {
	    if (stC.iDPHI == -1)
	    {
		  stC.dDPHI = math.mathLAS.computePHI( stC.iRows,
                                               stC.dDT,
                                               math.mathLAS._SONIC,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH,
                                               stC.dNull );
          stC.iDPHI = 1;
		}

        stC.dRhomaa = math.mathLAS.computeRhomaa( math.mathLAS._FRESH,
                                                  stC.iRows,
                                                  stC.dDPHI,
                                                  stC.dRHOB,
                                                  stC.dNull );

        stC.iRhomaa = 1; // Apparent Matrix Grain Density
	  }

      if ((stC.iRHOB > -1) && (stC.iNPHI > -1) && (stC.iDT > -1))
      {
        stC.dDTmaa = math.mathLAS.computeDTmaa( math.mathLAS._FRESH,
                                                stC.iRows,
                                                stC.dNPHI,
                                                stC.dRHOB,
                                                stC.dDT,
                                                stC.dNull );

        stC.iDTmaa = 1; // Apparent Matrix Grain Acoustic transit time
      }
      else if ((stC.iRHOB > -1) && (stC.iDT > -1))
      {
	    if (stC.iSPHI == -1)
	    {
		  stC.dSPHI = math.mathLAS.computePHI( stC.iRows,
                                               stC.dDT,
                                               math.mathLAS._SONIC,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH,
                                               stC.dNull );
          stC.iSPHI = 1;
		}

        stC.dDTmaa = math.mathLAS.computeDTmaa( math.mathLAS._FRESH,
                                                stC.iRows,
                                                stC.dSPHI,
                                                stC.dRHOB,
                                                stC.dDT,
                                                stC.dNull );

        stC.iDTmaa = 1; // Apparent Matrix Grain Acoustic transit time
	  }
      else if (stC.iDT > -1)
      {
	    if (stC.iSPHI == -1)
	    {
		  stC.dSPHI = math.mathLAS.computePHI( stC.iRows,
                                               stC.dDT,
                                               math.mathLAS._SONIC,
                                               math.mathLAS._CALCITE,
                                               math.mathLAS._FRESH,
                                               stC.dNull );
          stC.iSPHI = 1;
		}

        stC.dDTmaa = math.mathLAS.computeDTmaa( math.mathLAS._FRESH,
                                                stC.iRows,
                                                stC.dSPHI,
                                                stC.dDT,
                                                stC.dNull );

        stC.iDTmaa = 1; // Apparent Matrix Grain Acoustic transit time
	  }

      // Compute the Porosity Difference Lithology Curves

      if ( (stC.iRHOB > -1) && (stC.iNPHI > -1))
      {
        double dOut[][] = null;

        dOut = math.mathLAS.computePorosityDifference( math.mathLAS._FRESH,
                                                       stC.iRows,
                                                       stC.dNPHI,
                                                       stC.dRHOB,
                                                       stC.dNull );

        if (dOut != null)
        {
          for (i=0; i<stC.iRows; i++)
          {
            stC.dNeutron[i] = dOut[0][i]; // Neutron Porosity
            stC.delta[i] = dOut[1][i]; // Neutron - Density Porosity
          }
        }

        stC.iNeutron = 1; // Neutron Porosity
        stC.iDelta = 1; // Neutron - Density Porosity
      }

      // Compute the Spectral Gamma Ray Lithology Curves

      if ( (stC.iTh > -1) && (stC.iU > -1) && (stC.iK > -1))
      {
        double dOut[][] = null;

        if (stC.iColorlithDefault == -1)
          stC.iColorlithDefault = iqstrat.iqstratTracksStruct._COLORLITH_SG;

        dOut = math.mathLAS.computeSpectralGR( stC.iRows,
                                               stC.dTh,
                                               stC.dU,
                                               stC.dK,
                                               stC.dNull );

        if (dOut != null)
        {
          for (i=0; i<stC.iRows; i++)
          {
            stC.dThU[i] = dOut[0][i]; // Thorium-Uranium Ratio
            stC.dThK[i] = dOut[1][i]; // Thorium-Potassium Ratio
          }
        }

        stC.iThU = 1; // Th/U
        stC.iThK = 1; // Th/K
      }
    }
     return (stC);
  }

  /** Method findCurves()
   * <p> This method will identify the color litholgy curves in the LAS File.
   * @param  stCurves = The LAS Curve Data
   * @param  stC = The LAS File Data Structure
   * @return stC = The LAS File Data Structure
   */

  public static lasFileDataStruct findCurves( cmnLASCurveListStruct stCurves,
                                              lasFileDataStruct stC )
  {
    if (stCurves != null)
    {
      for (int i=0; i<stCurves.iCount; i++)
      {
		if (stCurves.stItem[i] != null)
		{

          // Get Gamma Ray Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._GR][0]))
          {
            if (stC.iGR == -1)
              stC.iGR = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._CGR][0]))
          {
            if (stC.iCGR == -1)
              stC.iCGR = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._SP][0]))
          {
            if (stC.iSP == -1)
              stC.iSP = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._CAL][0]))
          {
            if (stC.iCAL == -1)
              stC.iCAL = stCurves.stItem[i].iCurveNumber;
          }

          // Get Photoelectric Factor Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._PE][0]))
          {
            if (stC.iPE == -1)
              stC.iPE = stCurves.stItem[i].iCurveNumber;
          }

          // Get Neutron Porosity Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._NPHI][0]))
          {
            if (stC.iNPHI == -1)
              stC.iNPHI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Density Porosity Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._DPHI][0]))
          {
            if (stC.iDPHI == -1)
              stC.iDPHI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Bulk Density Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RHOB][0]))
          {
            if (stC.iRHOB == -1)
              stC.iRHOB = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._DRHO][0]))
          {
            if (stC.iRHO == -1)
              stC.iRHO = stCurves.stItem[i].iCurveNumber;
          }

          // Get Resistivity Log Curve Numbers

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._COND][0]))
          {
            if (stC.iCOND == -1)
              stC.iCOND = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._CILD][0]))
          {
            if (stC.iCILD == -1)
              stC.iCILD = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._CILM][0]))
          {
            if (stC.iCILM == -1)
              stC.iCILM = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RES][0]))
          {
            if (stC.iRES == -1)
              stC.iRES = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RDEP][0]))
          {
            if (stC.iRDEP == -1)
              stC.iRDEP = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RMED][0]))
          {
            if (stC.iRMED == -1)
              stC.iRMED = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RSHAL][0]))
          {
            if (stC.iRSHAL == -1)
              stC.iRSHAL = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ILD][0]))
          {
            if (stC.iILD == -1)
              stC.iILD = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ILM][0]))
          {
            if (stC.iILM == -1)
             stC.iILM = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._SFLU][0]))
          {
            if (stC.iSFLU == -1)
               stC.iSFLU = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LL][0]))
          {
            if (stC.iLL == -1)
              stC.iLL = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._MLL][0]))
          {
            if (stC.iMLL == -1)
              stC.iMLL = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LL8][0]))
          {
            if (stC.iLL8 == -1)
              stC.iLL8 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LN][0]))
          {
            if (stC.iLN == -1)
              stC.iLN = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._SN][0]))
          {
            if (stC.iSN == -1)
              stC.iSN = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._MNOR][0]))
          {
            if (stC.iMNOR == -1)
              stC.iMNOR = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._MSFL][0]))
          {
            if (stC.iMSFL == -1)
              stC.iMSFL = stCurves.stItem[i].iCurveNumber;
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._MINV][0]))
          {
            if (stC.iMINV == -1)
              stC.iMINV = stCurves.stItem[i].iCurveNumber;
          }

          if ((stCurves.stItem[i].sToolKID.equals(
               las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._AHT10][0])) )
          {
            if (stC.iAHT10 == -1)
              stC.iAHT10 = stCurves.stItem[i].iCurveNumber;
          }
          if ((stCurves.stItem[i].sToolKID.equals(
               las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._AHT20][0])) )
          {
            if (stC.iAHT20 == -1)
              stC.iAHT20 = stCurves.stItem[i].iCurveNumber;
          }

          if ((stCurves.stItem[i].sToolKID.equals(
               las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._AHT30][0])) )
          {
            if (stC.iAHT30 == -1)
              stC.iAHT30 = stCurves.stItem[i].iCurveNumber;
          }

          if ((stCurves.stItem[i].sToolKID.equals(
               las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._AHT60][0])) )
          {
            if (stC.iAHT60 == -1)
              stC.iAHT60 = stCurves.stItem[i].iCurveNumber;
          }

          if ((stCurves.stItem[i].sToolKID.equals(
               las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._AHT90][0])) )
          {
            if (stC.iAHT90 == -1)
              stC.iAHT90 = stCurves.stItem[i].iCurveNumber;
          }

          // Get Sonic Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._DT][0]))
          {
            if (stC.iDT == -1)
              stC.iDT = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._SPHI][0]))
          {
            if (stC.iSPHI == -1)
              stC.iSPHI = stCurves.stItem[i].iCurveNumber;
          }

          // Get Potassium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._K][0]))
          {
            if (stC.iK == -1)
              stC.iK = stCurves.stItem[i].iCurveNumber;
          }

          // Get Uranium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._U][0]))
          {
            if (stC.iU== -1)
              stC.iU = stCurves.stItem[i].iCurveNumber;
          }

          // Get Thorium Log Curve Number

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._TH][0]))
          {
            if (stC.iTh == -1)
              stC.iTh = stCurves.stItem[i].iCurveNumber;
          }

          // Gamma Ray Counts

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._GRN][0]))
          {
            if (stC.iGRN == -1)
              stC.iGRN = stCurves.stItem[i].iCurveNumber;
          }

          // Neutron Counts

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._NEUT][0]))
          {
            if (stC.iNEUT == -1)
              stC.iNEUT = stCurves.stItem[i].iCurveNumber;
          }

          // Permeability

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._PERM][0]))
          {
            if (stC.iPERM == -1)
              stC.iPERM = stCurves.stItem[i].iCurveNumber;
          }

          // Temperature

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._TEMP][0]))
          {
            if (stC.iTEMP == -1)
              stC.iTEMP = stCurves.stItem[i].iCurveNumber;
          }

          // Linear Curves

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LIN_1][0]))
          {
            if (stC.iLIN_1 == -1)
            {
              stC.iLIN_1 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[0][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LIN_2][0]))
          {
            if (stC.iLIN_2 == -1)
            {
              stC.iLIN_2 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[1][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LIN_3][0]))
          {
            if (stC.iLIN_3 == -1)
            {
              stC.iLIN_3 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[2][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LIN_4][0]))
          {
            if (stC.iLIN_4 == -1)
            {
              stC.iLIN_4 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[3][0] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }

          // Log Curves

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LOG_1][0]))
          {
            if (stC.iLOG_1 == -1)
            {
              stC.iLOG_1 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[0][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LOG_2][0]))
          {
            if (stC.iLOG_2 == -1)
            {
              stC.iLOG_2 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[1][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LOG_3][0]))
          {
            if (stC.iLOG_3 == -1)
            {
              stC.iLOG_3 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[2][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }
          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._LOG_4][0]))
          {
            if (stC.iLOG_4 == -1)
            {
              stC.iLOG_4 = stCurves.stItem[i].iCurveNumber;
              stC.sUnknown[3][1] = new String(stCurves.stItem[i].sMnemonic);
		    }
          }

          // Core Data

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._PHI][0]))
          {
            if (stC.iPHI == -1)
              stC.iPHI = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._GMCC][0]))
          {
            if (stC.iGMCC == -1)
              stC.iGMCC = stCurves.stItem[i].iCurveNumber;
          }

          // Other LAS Curves

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RTo][0]))
          {
            if (stC.iRT == -1)
              stC.iRT = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RWA][0]))
          {
            if (stC.iRWA == -1)
              stC.iRWA = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RO][0]))
          {
            if (stC.iRO == -1)
              stC.iRO = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._RXRT][0]))
          {
            if (stC.iRXRT == -1)
              stC.iRXRT = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._MA][0]))
          {
            if (stC.iMA == -1)
              stC.iMA = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._PHITo][0]))
          {
            if (stC.iPHIT == -1)
              stC.iPHIT = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._PHIE][0]))
          {
            if (stC.iPHIE == -1)
              stC.iPHIE = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._SW][0]))
          {
            if (stC.iSW == -1)
              stC.iSW = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._BVW][0]))
          {
            if (stC.iBVW == -1)
              stC.iBVW = stCurves.stItem[i].iCurveNumber;
          }

          // Unknown Curves

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT1][0]))
          {
            if (stC.iATT1 == -1)
              stC.iATT1 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT2][0]))
          {
            if (stC.iATT2 == -1)
              stC.iATT2 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT3][0]))
          {
            if (stC.iATT3 == -1)
              stC.iATT3 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT4][0]))
          {
            if (stC.iATT4 == -1)
              stC.iATT4 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT5][0]))
          {
            if (stC.iATT5 == -1)
              stC.iATT5 = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._ATT6][0]))
          {
            if (stC.iATT6 == -1)
              stC.iATT6 = stCurves.stItem[i].iCurveNumber;
          }

          // Lithology Curves

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._VCAL][0]))
          {
            if (stC.iVCAL == -1)
              stC.iVCAL = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._VQTZ][0]))
          {
            if (stC.iVQTZ == -1)
              stC.iVQTZ = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._VDOL][0]))
          {
            if (stC.iVDOL == -1)
              stC.iVDOL = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._VSH][0]))
          {
            if (stC.iVSH == -1)
              stC.iVSH = stCurves.stItem[i].iCurveNumber;
          }

          if (stCurves.stItem[i].sToolKID.equals(
            las.lasStandardTools.LAS_TOOLS[las.lasStandardTools._VANH][0]))
          {
            if (stC.iVANH == -1)
              stC.iVANH = stCurves.stItem[i].iCurveNumber;
          }
	    }
      }
    }

    return (stC);
  }

  /** Method copyHeader()
   * <p> This method will copy the Header & Parameters Data to the data structure
   * @param  stOld = LAS File Data Structure
   * @param  stNew = LAS File Data Structure
   * @return stNew = LAS File Data Structure
   */

  public static lasFileDataStruct copyHeader( lasFileDataStruct stOld,
                                              lasFileDataStruct stNew )
  {
	int i,j;

	if ((stOld != null) && (stNew != null))
	{
      stNew.sKID       = new String(stOld.sKID); // Primary Key for the KGS Well
      stNew.sKEY       = new String(stOld.sKEY); // Primary Key created by user
      stNew.iType      = stOld.iType;            // Type of Data

      stNew.sAPI       = new String(stOld.sAPI);  // API-Number of Well
      stNew.sName      = new String(stOld.sName); // Well Name

      // Other Fields not really displayed

      stNew.state      = new String(stOld.state);    // State Name
      stNew.sCounty    = new String(stOld.sCounty); // County Name

      // Well Location Information

      stNew.dLatitude  = stOld.dLatitude;  // Latitude of Well
      stNew.dLongitude = stOld.dLongitude; // Longitude of Well

      // Well Parameter Information

      stNew.dKB        = stOld.dKB; // Kelly Bushing of Well
      stNew.dGL        = stOld.dGL; // Ground Level of Well
      stNew.dTD        = stOld.dTD; // Total Depth of Well

      // Well Status

      stNew.status     = new String(stOld.status); // Status of Well

      // LAS File Information

      stNew.sReadMethod = new String(stOld.sReadMethod);  // File Read Method
      stNew.sDirectory  = new String(stOld.sDirectory); // Directory Path
      stNew.sFilename   = new String(stOld.sFilename);  // File name of LAS File
      stNew.sVersion    = new String(stOld.sVersion);   // LAS Version Number

      stNew.dNull       = stOld.dNull;      // Null Value of Data
      stNew.depthStart  = stOld.depthStart; // Start Depth of LAS Data
      stNew.depthEnd    = stOld.depthEnd;   // Ending Depth of LAS Data
      stNew.depthStep   = stOld.depthStep;  // Depth Step of LAS DAta

      stNew.iProprietary = stOld.iProprietary; // Proprietary Data Identfier

      // -- Required Parameters

      stNew.sPDAT      = new String( stOld.sPDAT ); // Permanent Data
      stNew.dAPD       = stOld.dAPD;                // Above Permanent Data
      stNew.sDREF      = new String (stOld.sDREF ); // Depth Reference (KB,DF,CB)
      stNew.dElevation = stOld.dElevation;     // "Elevation of Depth Reference"

      // -- Required parameters for AEUB compliance

      stNew.dTDD       = stOld.dTDD;  // Drillers Depth
      stNew.dTLD       = stOld.dTLD;  // Loggers Depth

      stNew.dCSGL      = stOld.dCSGL; // Casing Bottom Logger
      stNew.dCSGD      = stOld.dCSGD; // Casing Bottom Driller
      stNew.dCSGS      = stOld.dCSGS; // Casing Size
      stNew.dCSGW      = stOld.dCSGW; // Casing Weight
      stNew.dBS        = stOld.dBS;   // Bit Size

      stNew.sMUD       = new String(stOld.sMUD);  // Mud type
      stNew.sMUDS      = new String(stOld.sMUDS); // Mud Source
      stNew.dMUDD      = stOld.dMUDD; // Mud Density
      stNew.dMUDV      = stOld.dMUDV; // Mud Viscosity (Funnel)

      stNew.dFL        = stOld.dFL;   // Fluid Loss
      stNew.dPH        = stOld.dPH;   // PH

      stNew.dRM        = stOld.dRM;   // Resistivity of Mud
      stNew.dRMT       = stOld.dRMT;  // Temperature of Mud
      stNew.dRMF       = stOld.dRMF;  // Resistivity of Mud Filtrate
      stNew.dRMFT      = stOld.dRMFT; // Temperature of Mud Filtrate
      stNew.dRMC       = stOld.dRMC;  // Resistivity of Mud Cake
      stNew.dRMCT      = stOld.dRMCT; // Temperature of Mud Cake
      stNew.dTMAX      = stOld.dTMAX; // Maximum Recorded Temp.

      stNew.sTIMC      = new String(stOld.sTIMC); // Date/Time Circulation Stopped
      stNew.sTIML      = new String(stOld.sTIML); // Date/Time Logger Tagged Bottom
      stNew.sUNIT      = new String(stOld.sUNIT); // Logging Unit Number
      stNew.sBASE      = new String(stOld.sBASE); // Home Base of Logging Unit
      stNew.sENG       = new String(stOld.sENG);  // Recording Engineer
      stNew.sWIT       = new String(stOld.sWIT);  // Witnessed By

      // .. Units for Required parameters for AEUB compliance

      stNew.sCSGL      = new String(stOld.sCSGL); // Casing Bottom Logger
      stNew.sCSGD      = new String(stOld.sCSGD); // Casing Bottom Driller
      stNew.sCSGS      = new String(stOld.sCSGS); // Casing Size
      stNew.sCSGW      = new String(stOld.sCSGW); // Casing Weight
      stNew.sBS        = new String(stOld.sBS); // Bit Size
      stNew.sMUDD      = new String(stOld.sMUDD); // Mud Density
      stNew.sMUDV      = new String(stOld.sMUDV); // Mud Viscosity (Funnel)
      stNew.sFL        = new String(stOld.sFL); // Fluid Loss
      stNew.sRM        = new String(stOld.sRM); // Resistivity of Mud
      stNew.sRMT       = new String(stOld.sRMT); // Temperature of Mud
      stNew.sRMF       = new String(stOld.sRMF); //  Rest. of Mud Filtrate
      stNew.sRMFT      = new String(stOld.sRMFT); // Temp. of Mud Filtrate
      stNew.sRMC       = new String(stOld.sRMC); // Rest. of Mud Cake
      stNew.sRMCT      = new String(stOld.sRMCT); // Temp. of Mud Cake
      stNew.sTMAX      = new String(stOld.sTMAX); // Max. Recorded Temp.

      // LAS 3 File Parameters

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns
      stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

      for (i=0; i<stNew.iParamRows; i++)
      {
        for (j=0; j<stNew.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
        }
      }

      stNew.iLogs = stOld.iLogs;  // Total number of LAS3 Data Structures
      if (stOld.iLogs > 0)
      {
        stNew.stLAS3 = new las3Struct[stOld.iLogs];
        for (i=0; i<stOld.iLogs; i++)
        {
          stNew.stLAS3[i] = las3.las3Utility.copyWOData( stOld.stLAS3[i] );
        }
      }

      // Unknown Curve Mnemonics

      stNew.sUnknown = new String[4][2];
      for (i=0; i<4; i++)
      {
		for (j=0; j<2; j++)
		{
		  stNew.sUnknown[i][j] = new String( stOld.sUnknown[i][j] );
		}
	  }
	}

    return (stNew);
  }

  /** Method setData()
   * <p> This method will fill the Data for the Non-Linear Colorlith
   * @param  iSelected = track selected
   * @param  iTool  = the tool used to make calculations
   * @param  dStart = Starting Depth
   * @param  dEnd   = Ending Depth
   * @param  st     = the LAS File Data Structure
   * @return data   = the array of data with color
   */

  public static double[][] setData( int    iSelected,
                                    int    iTool,
                                    double dStart,
                                    double dEnd,
                                    lasFileDataStruct st)
  {
    int    i = 0;
    int    length   = 0;
    double data[][] = null;
    double dCurve[] = null;

    if (iTool > -1)
    {
      if (st != null)
      {
        if (st.checkData(iTool))
        {
          dCurve = st.getData(iTool);
          data   = new double[st.iRows][3];

          for (i=0; i<st.iRows; i++)
          {
            data[i][0] = st.depths[i];
            data[i][1] = dCurve[i];
            data[i][2] = 0.0;
          }

          data   = trim( st.iRows, 3, 1, st.dNull, dStart, dEnd, data);
          length = data.length;

          for (i=0; i<length; i++)
          {
            switch (iSelected)
            {
              case _RT:
                if (data[i][1] > st.dNull)
                  data[i][1] = 1000.0 / data[i][1];
                else
                  data[i][1] = st.dNull;
                break;
            }
          }

          data = bubbleSort(1, length, data);

          for (i=0; i<length; i++)
          {
            switch (iSelected)
            {
              case _RT:
                data[i][2] = 0.0;
                if (data[i][1] > st.dNull)
                  data[i][2] = ((_MMHOS_M) * i) / length; // st.iRows;
                if (data[i][2] > _MMHOS_M-1) data[i][2] = _MMHOS_M-1;
                break;
              case _PHIT:
                data[i][2] = ((_PU) * i) / length; // st.iRows;
                if (data[i][2] > _PU-1) data[i][2] = _PU-1;
                break;
            }
          }

          data = bubbleSort(0, length, data);
        }
      }
    }

    return (data);
  }

  /** Method setData()
   * <p> This method will fill the Data for the Non-Linear Colorlith
   *  @param bSonic    = indicator to plot by depth or two-way time
   * @param  iSelected = track selected
   * @param  iTool     = the tool used to make calculations
   * @param  dStart    = Starting Depth
   * @param  dEnd      = Ending Depth
   * @param  st        = the LAS File Data Structure
   * @return data      = the array of data with color
   */

  public static double[][] setData( boolean bSonic,
	                                int    iSelected,
                                    int    iTool,
                                    double dStart,
                                    double dEnd,
                                    lasFileDataStruct st)
  {
    int    i = 0;
    int    length   = 0;
    double data[][] = null;
    double dCurve[] = null;

    if (iTool > -1)
    {
      if (st != null)
      {
        if (st.checkData(iTool))
        {
          dCurve = st.getData(iTool);
          data   = new double[st.iRows][3];

          for (i=0; i<st.iRows; i++)
          {
			if (bSonic)
              data[i][0] = st.dTime[i];
            else
              data[i][0] = st.depths[i];
            data[i][1] = dCurve[i];
            data[i][2] = 0.0;
          }

          data   = trim( st.iRows, 3, 1, st.dNull, dStart, dEnd, data);
          length = data.length;

          for (i=0; i<length; i++)
          {
            switch (iSelected)
            {
              case _RT:
                if (data[i][1] > st.dNull)
                  data[i][1] = 1000.0 / data[i][1];
                else
                  data[i][1] = st.dNull;
                break;
            }
          }

          data = bubbleSort(1, length, data);

          for (i=0; i<length; i++)
          {
            switch (iSelected)
            {
              case _RT:
                data[i][2] = 0.0;
                if (data[i][1] > st.dNull)
                  data[i][2] = ((_MMHOS_M) * i) / length; // st.iRows;
                if (data[i][2] > _MMHOS_M-1) data[i][2] = _MMHOS_M-1;
                break;
              case _PHIT:
                data[i][2] = ((_PU) * i) / length; // st.iRows;
                if (data[i][2] > _PU-1) data[i][2] = _PU-1;
                break;
            }
          }

          data = bubbleSort(0, length, data);
        }
      }
    }

    return (data);
  }

  /* --------------------------------------------------------------------------------------------- *
   * --------------------------------- KIMELEON SPECIFIC METHODS --------------------------------- *
   * --------------------------------------------------------------------------------------------- */

  /** Method initData()
   * <p> This method will initialize the LAS File data
   * @param  iRows = total number of rows
   * @param  dNull = Null Value for empty data
   * @return stC   = Color Lithology Well Structure
   */

  public static lasFileDataStruct initData(int iRows, double dNull)
  {
    lasFileDataStruct stC = new lasFileDataStruct();

    if (iRows > 0)
    {
      stC.iRows  = iRows;                 // Number of Rows
      stC.dNull  = dNull;                 // Null Value

      stC.depths = new double[stC.iRows]; // Array of Depth Data Values
      stC.dTime  = new double[stC.iRows]; // Array of Time Data Values

      stC.dGR    = new double[stC.iRows]; // Array of Gamma Ray Log Values
      stC.dCGR   = new double[stC.iRows]; // Array of Gamma Ray minus Uranium Log Values
      stC.dSP    = new double[stC.iRows]; // Array of Spontaneous Potential Log Values
      stC.dCAL   = new double[stC.iRows]; // Array of Caliper Log Values

      stC.dNPHI  = new double[stC.iRows]; // Array of Neutron Porosity Log Values
      stC.dPE    = new double[stC.iRows]; // Array of PhotoElectric  Values
      stC.dRHOB  = new double[stC.iRows]; // Array of Bulk Density  Values
      stC.dDPHI  = new double[stC.iRows]; // Array of Density Porosity Log Values
      stC.dRHO   = new double[stC.iRows]; // Array of Density Corrrection Log Values

      stC.dDT    = new double[stC.iRows]; // Array of Sonic Log Values
      stC.dSPHI  = new double[stC.iRows]; // Array of Sonic Porosity Log Values

      stC.dCOND  = new double[stC.iRows]; // Array of Conductivity
      stC.dCILD  = new double[stC.iRows]; // Array of Deep Induction Conductivity
      stC.dCILM  = new double[stC.iRows]; // Array of Medium Induction Conductivity

      stC.dRES   = new double[stC.iRows]; // Array of Resistivity
      stC.dRMED  = new double[stC.iRows]; // Array of Deep Resistivity
      stC.dRDEP  = new double[stC.iRows]; // Array of Medium Resistivity
      stC.dRSHAL = new double[stC.iRows]; // Array of Shallow Resistivity,

      stC.dILD   = new double[stC.iRows]; // Array of Deep Induction Resistivity
      stC.dILM   = new double[stC.iRows]; // Array of Medium Induction Resistivity
      stC.dSFLU  = new double[stC.iRows]; // Array of Spherically Focused Resistivity

      stC.dLL    = new double[stC.iRows]; // Array of Deep Laterolog Resistivity
      stC.dMLL   = new double[stC.iRows]; // Array of Micro Laterolog Resistivity
      stC.dLL8   = new double[stC.iRows]; // Array of Shallow Laterolog Resistivity

      stC.dLN    = new double[stC.iRows]; // Array of Long Normal Resistivity
      stC.dSN    = new double[stC.iRows]; // Array of Shallow Normal Resistivity

      stC.dMNOR  = new double[stC.iRows]; // Array of Micro Normal Resistivity
      stC.dMSFL  = new double[stC.iRows]; // Array of Micro Spherically Focused Resistivity
      stC.dMINV  = new double[stC.iRows]; // Array of Micro Inverse Resistivity

      stC.dAHT10 = new double[stC.iRows]; // Array Induction Resistivity-10
      stC.dAHT20 = new double[stC.iRows]; // Array Induction Resistivity-20
      stC.dAHT30 = new double[stC.iRows]; // Array Induction Resistivity-30
      stC.dAHT60 = new double[stC.iRows]; // Array Induction Resistivity-60
      stC.dAHT90 = new double[stC.iRows]; // Array Induction Resistivity-90

      stC.dTh    = new double[stC.iRows]; // Array of Thorium Log Values
      stC.dU     = new double[stC.iRows]; // Array of Uranium Log Values
      stC.dK     = new double[stC.iRows]; // Array of Potassium Log Values

      stC.dUmaa     = new double[stC.iRows]; // Apparent Volumetric Photoelectric
      stC.dRhomaa   = new double[stC.iRows]; // Apparent Matrix Grain Density

      stC.dNeutron  = new double[stC.iRows]; // Neutron Porosity
      stC.delta     = new double[stC.iRows]; // Neutron - Density Porosity

      stC.dThU      = new double[stC.iRows]; // Th/U
      stC.dThK      = new double[stC.iRows]; // Th/K

      stC.dPHI      = new double[stC.iRows]; // Measured Core Porosity
      stC.dGMCC     = new double[stC.iRows]; // Measured Core Grain Density

      stC.dGRN      = new double[stC.iRows]; // Gamma Ray Count
      stC.dNEUT     = new double[stC.iRows]; // Neutron Counts
      stC.dPERM     = new double[stC.iRows]; // Permeability

      stC.dTEMP     = new double[stC.iRows];  // Temperature
      stC.dTEMPDIFF = new double[stC.iRows];  // Temperature Difference

      stC.dUmaa     = new double[stC.iRows]; // Apparent Volumetric Photoelectric
      stC.dRhomaa   = new double[stC.iRows]; // Apparent Matrix Grain Density
      stC.dDTmaa    = new double[stC.iRows]; // Apparent Matrix Grain Acoustic transit time

      stC.dNeutron  = new double[stC.iRows]; // Neutron Porosity
      stC.delta     = new double[stC.iRows]; // Neutron - Density Porosity

      stC.dThU      = new double[stC.iRows]; // Th/U
      stC.dThK      = new double[stC.iRows]; // Th/K

      stC.dLIN_1    = new double[stC.iRows]; // Linear Curve 1
      stC.dLIN_2    = new double[stC.iRows]; // Linear Curve 2
      stC.dLIN_3    = new double[stC.iRows]; // Linear Curve 3
      stC.dLIN_4    = new double[stC.iRows]; // Linear Curve 4

      stC.dLOG_1    = new double[stC.iRows]; // Log Curve 1
      stC.dLOG_2    = new double[stC.iRows]; // Log Curve 2
      stC.dLOG_3    = new double[stC.iRows]; // Log Curve 3
      stC.dLOG_4    = new double[stC.iRows]; // Log Curve 4

      // Zonation Curves

      stC.dRT       = new double[stC.iRows]; // Total Resistivity
      stC.dRWA      = new double[stC.iRows]; // Apparent Water Resistivity
      stC.dRO       = new double[stC.iRows]; // Water Saturated Rock Resistivity
      stC.dRXRT     = new double[stC.iRows]; // Quick Look Ratio
      stC.dMA       = new double[stC.iRows]; // Archie Cementation
      stC.dPHIT     = new double[stC.iRows]; // Total Porosity
      stC.dPHIE     = new double[stC.iRows]; // Effective Porosity
      stC.dSW       = new double[stC.iRows]; // Water Saturation
      stC.dBVW      = new double[stC.iRows]; // Bulk Volume Water

      stC.dATT1     = new double[stC.iRows]; // Att 1
      stC.dATT2     = new double[stC.iRows]; // Att 2
      stC.dATT3     = new double[stC.iRows]; // Att 3
      stC.dATT4     = new double[stC.iRows]; // Att 4
      stC.dATT5     = new double[stC.iRows]; // Att 5
      stC.dATT6     = new double[stC.iRows]; // Att 6

      stC.dVCAL     = new double[stC.iRows]; // Calcite Volume Fraction
      stC.dVQTZ     = new double[stC.iRows]; // Quartz Volume Fraction
      stC.dVDOL     = new double[stC.iRows]; // Dolomite Volume Fraction
      stC.dVSH      = new double[stC.iRows]; // Shale Volume Fraction
      stC.dVANH     = new double[stC.iRows]; // Anhydrite Volume Fraction

      for (int i=0; i<iRows; i++)
      {
        stC.depths[i] = stC.dNull; // Array of Depth Data Values
        stC.dTime[i]  = stC.dNull; // Array of Time Data Values

        stC.dGR[i]    = stC.dNull; // Array of Gamma Ray Log Values
        stC.dCGR[i]   = stC.dNull; // Array of Gamma Ray minus Uranium Log Values
        stC.dSP[i]    = stC.dNull; // Array of Spontaneous Potential Log Values
        stC.dCAL[i]   = stC.dNull; // Array of Caliper Log Values

        stC.dNPHI[i]  = stC.dNull; // Array of Neutron Porosity Log Values
        stC.dPE[i]    = stC.dNull; // Array of PhotoElectric  Values
        stC.dRHOB[i]  = stC.dNull; // Array of Bulk Density  Values
        stC.dDPHI[i]  = stC.dNull; // Array of Density Porosity Log Values
        stC.dRHO[i]   = stC.dNull; // Array of Density Corrrection Log Values

        stC.dDT[i]    = stC.dNull; // Array of Sonic Log Values
        stC.dSPHI[i]  = stC.dNull; // Array of Sonic Porosity Log Values

        stC.dCOND[i]  = stC.dNull; // Array of Conductivity
        stC.dCILD[i]  = stC.dNull; // Array of Deep Induction Conductivity
        stC.dCILM[i]  = stC.dNull; // Array of Medium Induction Conductivity

        stC.dRES[i]   = stC.dNull; // Array of Resistivity
        stC.dRMED[i]  = stC.dNull; // Array of Deep Resistivity
        stC.dRDEP[i]  = stC.dNull; // Array of Medium Resistivity
        stC.dRSHAL[i] = stC.dNull; // Array of Shallow Resistivity,

        stC.dILD[i]   = stC.dNull; // Array of Deep Induction Resistivity
        stC.dILM[i]   = stC.dNull; // Array of Medium Induction Resistivity
        stC.dSFLU[i]  = stC.dNull; // Array of Spherically Focused Resistivity

        stC.dLL[i]    = stC.dNull; // Array of Deep Laterolog Resistivity
        stC.dMLL[i]   = stC.dNull; // Array of Micro Laterolog Resistivity
        stC.dLL8[i]   = stC.dNull; // Array of Shallow Laterolog Resistivity

        stC.dAHT10[i] = stC.dNull; // Array Induction Resistivity-10
        stC.dAHT20[i] = stC.dNull; // Array Induction Resistivity-20
        stC.dAHT30[i] = stC.dNull; // Array Induction Resistivity-30
        stC.dAHT60[i] = stC.dNull; // Array Induction Resistivity-60
        stC.dAHT90[i] = stC.dNull; // Array Induction Resistivity-90

        stC.dLN[i]    = stC.dNull; // Array of Long Normal Resistivity
        stC.dSN[i]    = stC.dNull; // Array of Shallow Normal Resistivity

        stC.dMNOR[i]  = stC.dNull; // Array of Micro Normal Resistivity
        stC.dMSFL[i]  = stC.dNull; // Array of Micro Spherically Focused Resistivity
        stC.dMINV[i]  = stC.dNull; // Array of Micro Inverse Resistivity

        stC.dTh[i]    = stC.dNull; // Array of Thorium Log Values
        stC.dU[i]     = stC.dNull; // Array of Uranium Log Values
        stC.dK[i]     = stC.dNull; // Array of Potassium Log Values

        stC.dPHI[i]   = stC.dNull; // Array of Measured Core Porosity
        stC.dGMCC[i]  = stC.dNull; // Array of Measured Core Grain Density

        stC.dGRN[i]      = stC.dNull; // Gamma Ray Count
        stC.dNEUT[i]     = stC.dNull; // Neutron Counts

        stC.dPERM[i]     = stC.dNull; // Permeability

        stC.dTEMP[i]     = stC.dNull; // Temperature
        stC.dTEMPDIFF[i] = stC.dNull; // Temperature Difference

        stC.dUmaa[i]     = stC.dNull; // Apparent Volumetric Photoelectric
        stC.dRhomaa[i]   = stC.dNull; // Apparent Matrix Grain Density
        stC.dDTmaa[i]    = stC.dNull; // Apparent Matrix Grain Acoustic transit time

        stC.dNeutron[i]  = stC.dNull; // Neutron Porosity
        stC.delta[i]     = stC.dNull; // Neutron - Density Porosity

        stC.dLIN_1[i]    = stC.dNull; // Linear Curve 1
        stC.dLIN_2[i]    = stC.dNull; // Linear Curve 2
        stC.dLIN_3[i]    = stC.dNull; // Linear Curve 3
        stC.dLIN_4[i]    = stC.dNull; // Linear Curve 4

        stC.dLOG_1[i]    = stC.dNull; // Log Curve 1
        stC.dLOG_2[i]    = stC.dNull; // Log Curve 2
        stC.dLOG_3[i]    = stC.dNull; // Log Curve 3
        stC.dLOG_4[i]    = stC.dNull; // Log Curve 4

        // Zonation Curves

        stC.dRT[i]       = stC.dNull; // Total Resistivity
        stC.dRWA[i]      = stC.dNull; // Apparent Water Resistivity
        stC.dRO[i]       = stC.dNull; // Water Saturated Rock Resistivity
        stC.dRXRT[i]     = stC.dNull; // Quick Look Ratio
        stC.dMA[i]       = stC.dNull; // Archie Cementation
        stC.dPHIT[i]     = stC.dNull; // Total Porosity
        stC.dPHIE[i]     = stC.dNull; // Effective Porosity
        stC.dSW [i]      = stC.dNull; // Water Saturation
        stC.dBVW[i]      = stC.dNull; // Bulk Volume Water

        stC.dATT1[i]     = stC.dNull; // Att 1
        stC.dATT2[i]     = stC.dNull; // Att 2
        stC.dATT3[i]     = stC.dNull; // Att 3
        stC.dATT4[i]     = stC.dNull; // Att 4
        stC.dATT5[i]     = stC.dNull; // Att 5
        stC.dATT6[i]     = stC.dNull; // Att 6

        stC.dVCAL[i]     = stC.dNull; // Calcite Volume Fraction
        stC.dVQTZ[i]     = stC.dNull; // Quartz Volume Fraction
        stC.dVDOL[i]     = stC.dNull; // Dolomite Volume Fraction
        stC.dVSH[i]      = stC.dNull; // Shale Volume Fraction
        stC.dVANH[i]     = stC.dNull; // Anhydrite Volume Fraction
      }
    }

    return (stC);
  }

  /** Method addData()
   * <p> This method will add the LOG data to the Color Lithology Well Structure
   * @param  iCurve = The Curve Data ID
   * @param  iRow   = The Row to add the data
   * @param  data   = the data value
   * @param  stC    = Color Lithology Well Structure
   * @return stC    = Color Lithology Well Structure
   */

  public static lasFileDataStruct addData( int    iCurve,
                                           int    iRow,
                                           double data,
                                           lasFileDataStruct stC )
  {
    stC.setRow(iCurve, iRow, data);

    return (stC);
  }

  /** Method findUniqueCurve()
   * <p> This method will retrieve a unique curve for KAMELEON Resistivity and
   *     porosity.
   * @param  iCurve = The type of curve
   * @param  st     = LAS File Data Structure
   * @return iValue = The unique curve
   */

  public static int findUniqueCurve(int iCurve, lasFileDataStruct st)
  {
    int iValue = -1;

    if (st != null)
    {
      switch (iCurve)
      {
        case _RT:
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._ILD )))
            iValue = las.lasStandardTools._ILD;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._ILM )))
            iValue = las.lasStandardTools._ILM;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._SFLU )))
            iValue = las.lasStandardTools._SFLU;

          if ((iValue == -1) && (st.checkData( las.lasStandardTools._MNOR )))
            iValue = las.lasStandardTools._MNOR;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._MSFL )))
            iValue = las.lasStandardTools._MSFL;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._MINV )))
            iValue = las.lasStandardTools._MINV;

          if ((iValue == -1) && (st.checkData( las.lasStandardTools._AHT90 )))
            iValue = las.lasStandardTools._AHT90;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._AHT60 )))
            iValue = las.lasStandardTools._AHT60;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._AHT30 )))
            iValue = las.lasStandardTools._AHT30;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._AHT20 )))
            iValue = las.lasStandardTools._AHT20;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._AHT10 )))
            iValue = las.lasStandardTools._AHT10;

          if ((iValue == -1) && (st.checkData( las.lasStandardTools._RDEP )))
            iValue = las.lasStandardTools._RDEP;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._RMED )))
            iValue = las.lasStandardTools._RMED;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._RSHAL )))
            iValue = las.lasStandardTools._RSHAL;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._RES )))
            iValue = las.lasStandardTools._RES;

          if ((iValue == -1) && (st.checkData( las.lasStandardTools._LL )))
            iValue = las.lasStandardTools._LL;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._MLL )))
            iValue = las.lasStandardTools._MLL;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._LL8 )))
            iValue = las.lasStandardTools._LL8;

          if ((iValue == -1) && (st.checkData( las.lasStandardTools._LN )))
            iValue = las.lasStandardTools._LN;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._SN )))
            iValue = las.lasStandardTools._SN;
          break;

        case _PHIT:
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._NPHI )))
            iValue = las.lasStandardTools._NPHI;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._DPHI )))
            iValue = las.lasStandardTools._DPHI;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._SPHI )))
            iValue = las.lasStandardTools._SPHI;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._PHI )))
            iValue = las.lasStandardTools._PHI;
          break;

        case _GRT:
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._GR )))
            iValue = las.lasStandardTools._GR;
          if ((iValue == -1) && (st.checkData( las.lasStandardTools._CGR )))
            iValue = las.lasStandardTools._CGR;
          break;
      }
    }

    return (iValue);
  }

  /* --------------------------------------------------------------------------------------------- *
   * ------------------------------- END KIMELEON SPECIFIC METHODS ------------------------------- *
   * --------------------------------------------------------------------------------------------- */

  /** Method trim()
   * <p> This method will sort in ascending data order (lowest to highest)
   * @param  iRows = number of data sets
   * @param  iColumns = total number of columns
   * @param  iCheck = check the data for
   * @param  dNull  = Null Value
   * @param  dStart = The starting depth
   * @param  dEnd   = The ending depth
   * @param  data = array of data ==> [depth][log curve]
   * @return data = array of data ==> [depth][log curve]
   */

  public static double[][] trim(int iRows,
                                int iColumns,
                                int iCheck,
                                double dNull,
                                double dStart,
                                double dEnd,
                                double data[][])
  {
    int    i = 0;
    int    j = 0;
    int    iCount   = 0;
    int    iTotal   = 0;
    double dNew[][] = null;

    for (i=0; i<iRows; i++)
      if (data[i][iCheck] > dNull)
        if ((data[i][0] >= dStart) && (data[i][0] <= dEnd))
          iTotal++;

    dNew = new double[iTotal][iColumns];

    for (i=0; i<iRows; i++)
    {
      if (data[i][iCheck] > dNull)
      {
        if ((data[i][0] >= dStart) && (data[i][0] <= dEnd))
        {
          if (iCount < iTotal)
          {
            for (j=0; j<iColumns; j++)
            {
              dNew[iCount][j] = data[i][j];
            }

            iCount++;
          }
        }
      }
    }

    return (dNew);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending data order (lowest to highest)
   * @param  iSort  = The column to sort by
   * @param  iCount = number of data sets
   * @param  data   = array of data ==> [depth][log curve]
   * @return data   = array of data ==> [depth][log curve]
   */

  public static double[][] bubbleSort(int iSort, int iCount, double data[][])
  {
    boolean swappedOnPrevRun = true;
    double  dTemp[]          = {0.0, 0.0, 0.0};

    if (iCount > 1)
    {
      while(swappedOnPrevRun)
      {
        // this variable keeps track of whether to continue sorting or exit

        swappedOnPrevRun = false;

        // loop through every element in the array, except for the last one

        for(int i=0; i<iCount-1; i++)
        {
          // if current element is greater than the next swap the two elements

          if(data[i][iSort] > data[i+1][iSort])
          {
            // we don't want the loop to end just yet, we're not done

            swappedOnPrevRun = true;

            // store element i in a temporary variable

            dTemp[0] = data[i][0];
            dTemp[1] = data[i][1];
            dTemp[2] = data[i][2];

            // set element i+1 to where i used to be

            data[i][0] = data[i+1][0];
            data[i][1] = data[i+1][1];
            data[i][2] = data[i+1][2];

            // release the old i from temp into i+1 slot

            data[i+1][0] = dTemp[0];
            data[i+1][1] = dTemp[1];
            data[i+1][2] = dTemp[2];
          }
        }
      }
    }

    return (data);
  }

  /** Method print()
   * <p> This method will print the required LAS Data File Fields
   * @param st = the LAS File Data Structure
   */

  public static void print( lasFileDataStruct st )
  {
    if (st != null)
    {
      System.out.println(
        "Start Depth: " + st.depthStart + "\n" +
        "End Depth:   " + st.depthEnd   + "\n" +
        "Step By:     " + st.depthStep  + "\n" +
        "NULL Value:  " + st.dNull      + "\n\n");

      for (int i=0; i<st.iRows; i++)
      {
        System.out.print("<"+st.depths[i]+"> "); // Array of Depth Data Values

        if (st.iGR > -1) // Gamma Ray Curve Number in LAS File
        {
          System.out.print("<GR="+st.dGR[i]+"> ");
        }
        if (st.iCGR > -1) // Gamma Ray Minus Uranium Curve Number in LAS File
        {
          System.out.print("<CGR="+st.dCGR[i]+"> ");
        }
        if (st.iSP > -1) // Array of Spontaneous Potential Log Values
        {
          System.out.print("<SP="+st.dSP[i]+"> ");
        }
        if (st.iCAL > -1) // Array of Caliper Log Values
        {
          System.out.print("<CAL="+st.dCAL[i]+"> ");
        }

        if (st.iNPHI > -1) // Neutron Porosity Curve Number in LAS File
        {
          System.out.print("<NPHI="+st.dNPHI[i]+"> ");
        }
        if (st.iPE > -1) // Photoelectric Curve Number in LAS File
        {
          System.out.print("<PE="+st.dPE[i]+"> ");
        }
        if (st.iRHOB > -1) // Bulk Density Curve Number in LAS File
        {
          System.out.print("<RHOB="+st.dRHOB[i]+"> ");
        }
        if (st.iDPHI > -1) // Density Porosity Curve Number in LAS File
        {
          System.out.print("<DPHI="+st.dDPHI[i]+"> ");
        }
        if (st.iRHO > -1) // Density Correction Curve Number in LAS File
        {
          System.out.print("<RHO="+st.dRHO[i]+"> ");
        }

        if (st.iDT > -1) // Sonic Curve Number in LAS File
        {
          System.out.print("<DT="+st.dDT[i]+"> ");
        }
        if (st.iSPHI > -1) // Sonic Porisity Curve Number in LAS File
        {
          System.out.print("<SPHI="+st.dSPHI[i]+"> ");
        }

        if (st.iCOND > -1) // Array of Conductivity
        {
          System.out.print("<COND="+st.dCOND[i]+"> ");
        }
        if (st.iCILD > -1) // Array of Deep Induction Conductivity
        {
          System.out.print("<CILD="+st.dCILD[i]+"> ");
        }
        if (st.iCILM > -1) // Array of Medium Induction Conductivity
        {
          System.out.print("<CILM="+st.dCILM[i]+"> ");
        }

        if (st.iRES > -1) // Array of Resistivity
        {
          System.out.print("<RES="+st.dRES[i]+"> ");
        }
        if (st.iRMED > -1) // Array of Deep Resistivity
        {
          System.out.print("<RMED="+st.dRMED[i]+"> ");
        }
        if (st.iRDEP > -1) // Array of Medium Resistivity
        {
          System.out.print("<RDEP="+st.dRDEP[i]+"> ");
        }
        if (st.iRSHAL > -1) // Array of Shallow Resistivity,
        {
          System.out.print("<RSHAL="+st.dRSHAL[i]+"> ");
        }

        if (st.iILD > -1) // Array of Deep Induction Resistivity
        {
          System.out.print("<ILD="+st.dILD[i]+"> ");
        }
        if (st.iILM > -1) // Array of Medium Induction Resistivity
        {
          System.out.print("<ILM="+st.dILM[i]+"> ");
        }
        if (st.iSFLU > -1) // Array of Spherically Focused Resistivity
        {
          System.out.print("<SFLU="+st.dSFLU[i]+"> ");
        }

        if (st.iLL > -1) // Array of Deep Laterolog Resistivity
        {
          System.out.print("<LL="+st.dLL[i]+"> ");
        }
        if (st.iMLL > -1) // Array of Micro Laterolog Resistivity
        {
          System.out.print("<MLL="+st.dMLL[i]+"> ");
        }
        if (st.iLL8 > -1) // Array of Shallow Laterolog Resistivity
        {
          System.out.print("<LL8="+st.dLL8[i]+"> ");
        }

        if (st.iLN > -1) // Array of Long Normal Resistivity
        {
          System.out.print("<LN="+st.dLN[i]+"> ");
        }
        if (st.iSN > -1) // Array of Shallow Normal Resistivity
        {
          System.out.print("<SN="+st.dSN[i]+"> ");
        }

        if (st.iMNOR > -1) // Array of Micro Normal Resistivity
        {
          System.out.print("<MNOR="+st.dMNOR[i]+"> ");
        }
        if (st.iMSFL > -1) // Array of Micro Spherically Focused Resistivity
        {
          System.out.print("<MSFL="+st.dMSFL[i]+"> ");
        }
        if (st.iMINV > -1) // Array of Micro Inverse Resistivity
        {
          System.out.print("<MINV="+st.dMINV[i]+"> ");
        }

        if (st.iAHT10 > -1) // Array Induction Resistivity-10
        {
          System.out.print("<AHT10="+st.dAHT10[i]+"> ");
        }
        if (st.iAHT20 > -1) // Array Induction Resistivity-20
        {
          System.out.print("<AHT20="+st.dAHT20[i]+"> ");
        }
        if (st.iAHT30 > -1) // Array Induction Resistivity-30
        {
          System.out.print("<AHT30="+st.dAHT30[i]+"> ");
        }
        if (st.iAHT60 > -1) // Array Induction Resistivity-60
        {
          System.out.print("<AHT60="+st.dAHT60[i]+"> ");
        }
        if (st.iAHT90 > -1) // Array Induction Resistivity-90
        {
          System.out.print("<AHT90="+st.dAHT90[i]+"> ");
        }

        if (st.iTh > -1) // Thorium Curve Number in LAS File
        {
          System.out.print("<Th="+st.dTh[i]+"> ");
        }
        if (st.iU > -1) // Uranium Curve Number in LAS File
        {
          System.out.print("<U="+st.dU[i]+"> ");
        }
        if (st.iK > -1) // Potasium Curve Number in LAS File
        {
          System.out.print("<K="+st.dK[i]+"> ");
        }

        if (st.iLIN_1 > -1) // Unknown Linear Curve 1 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_1[i]+"> ");
        }
        if (st.iLIN_2 > -1) // Unknown Linear Curve 2 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_2[i]+"> ");
        }
        if (st.iLIN_3 > -1) // Unknown Linear Curve 3 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_3[i]+"> ");
        }
        if (st.iLIN_4 > -1) // Unknown Linear Curve 4 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_4[i]+"> ");
        }

        if (st.iLOG_1 > -1) // Unknown Log Curve 1 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_1[i]+"> ");
        }
        if (st.iLOG_2 > -1) // Unknown Log Curve 2 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_2[i]+"> ");
        }
        if (st.iLOG_3 > -1) // Unknown Log Curve 3 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_3[i]+"> ");
        }
        if (st.iLOG_4 > -1) // Unknown Log Curve 4 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_4[i]+"> ");
        }

        System.out.println("");
      }
    }
  }

  /** Method print()
   * <p> This method will print the required LAS Data File Fields
   * @param iRow = The row number
   * @param st   = the LAS File Data Structure
   */

  public static void print( int iRow, lasFileDataStruct st )
  {
	int i=0;
    if (st != null)
    {
//      System.out.println(
//        "Start Depth: " + st.depthStart + "\n" +
//        "End Depth:   " + st.depthEnd   + "\n" +
//        "Step By:     " + st.depthStep  + "\n" +
//        "NULL Value:  " + st.dNull      + "\n\n");

//      for (int i=0; i<st.iRows; i++)
      if ((iRow < st.iRows) && (iRow > -1))
      {
		i = iRow;

        System.out.print("<"+st.depths[i]+"> "); // Array of Depth Data Values

        if (st.iGR > -1) // Gamma Ray Curve Number in LAS File
        {
          System.out.print("<GR="+st.dGR[i]+"> ");
        }
        if (st.iCGR > -1) // Gamma Ray Minus Uranium Curve Number in LAS File
        {
          System.out.print("<CGR="+st.dCGR[i]+"> ");
        }
        if (st.iSP > -1) // Array of Spontaneous Potential Log Values
        {
          System.out.print("<SP="+st.dSP[i]+"> ");
        }
        if (st.iCAL > -1) // Array of Caliper Log Values
        {
          System.out.print("<CAL="+st.dCAL[i]+"> ");
        }

        if (st.iNPHI > -1) // Neutron Porosity Curve Number in LAS File
        {
          System.out.print("<NPHI="+st.dNPHI[i]+"> ");
        }
        if (st.iPE > -1) // Photoelectric Curve Number in LAS File
        {
          System.out.print("<PE="+st.dPE[i]+"> ");
        }
        if (st.iRHOB > -1) // Bulk Density Curve Number in LAS File
        {
          System.out.print("<RHOB="+st.dRHOB[i]+"> ");
        }
        if (st.iDPHI > -1) // Density Porosity Curve Number in LAS File
        {
          System.out.print("<DPHI="+st.dDPHI[i]+"> ");
        }
        if (st.iRHO > -1) // Density Correction Curve Number in LAS File
        {
          System.out.print("<RHO="+st.dRHO[i]+"> ");
        }

        if (st.iDT > -1) // Sonic Curve Number in LAS File
        {
          System.out.print("<DT="+st.dDT[i]+"> ");
        }
        if (st.iSPHI > -1) // Sonic Porisity Curve Number in LAS File
        {
          System.out.print("<SPHI="+st.dSPHI[i]+"> ");
        }

        if (st.iCOND > -1) // Array of Conductivity
        {
          System.out.print("<COND="+st.dCOND[i]+"> ");
        }
        if (st.iCILD > -1) // Array of Deep Induction Conductivity
        {
          System.out.print("<CILD="+st.dCILD[i]+"> ");
        }
        if (st.iCILM > -1) // Array of Medium Induction Conductivity
        {
          System.out.print("<CILM="+st.dCILM[i]+"> ");
        }

        if (st.iRES > -1) // Array of Resistivity
        {
          System.out.print("<RES="+st.dRES[i]+"> ");
        }
        if (st.iRMED > -1) // Array of Deep Resistivity
        {
          System.out.print("<RMED="+st.dRMED[i]+"> ");
        }
        if (st.iRDEP > -1) // Array of Medium Resistivity
        {
          System.out.print("<RDEP="+st.dRDEP[i]+"> ");
        }
        if (st.iRSHAL > -1) // Array of Shallow Resistivity,
        {
          System.out.print("<RSHAL="+st.dRSHAL[i]+"> ");
        }

        if (st.iILD > -1) // Array of Deep Induction Resistivity
        {
          System.out.print("<ILD="+st.dILD[i]+"> ");
        }
        if (st.iILM > -1) // Array of Medium Induction Resistivity
        {
          System.out.print("<ILM="+st.dILM[i]+"> ");
        }
        if (st.iSFLU > -1) // Array of Spherically Focused Resistivity
        {
          System.out.print("<SFLU="+st.dSFLU[i]+"> ");
        }

        if (st.iLL > -1) // Array of Deep Laterolog Resistivity
        {
          System.out.print("<LL="+st.dLL[i]+"> ");
        }
        if (st.iMLL > -1) // Array of Micro Laterolog Resistivity
        {
          System.out.print("<MLL="+st.dMLL[i]+"> ");
        }
        if (st.iLL8 > -1) // Array of Shallow Laterolog Resistivity
        {
          System.out.print("<LL8="+st.dLL8[i]+"> ");
        }

        if (st.iLN > -1) // Array of Long Normal Resistivity
        {
          System.out.print("<LN="+st.dLN[i]+"> ");
        }
        if (st.iSN > -1) // Array of Shallow Normal Resistivity
        {
          System.out.print("<SN="+st.dSN[i]+"> ");
        }

        if (st.iMNOR > -1) // Array of Micro Normal Resistivity
        {
          System.out.print("<MNOR="+st.dMNOR[i]+"> ");
        }
        if (st.iMSFL > -1) // Array of Micro Spherically Focused Resistivity
        {
          System.out.print("<MSFL="+st.dMSFL[i]+"> ");
        }
        if (st.iMINV > -1) // Array of Micro Inverse Resistivity
        {
          System.out.print("<MINV="+st.dMINV[i]+"> ");
        }

        if (st.iAHT10 > -1) // Array Induction Resistivity-10
        {
          System.out.print("<AHT10="+st.dAHT10[i]+"> ");
        }
        if (st.iAHT20 > -1) // Array Induction Resistivity-20
        {
          System.out.print("<AHT20="+st.dAHT20[i]+"> ");
        }
        if (st.iAHT30 > -1) // Array Induction Resistivity-30
        {
          System.out.print("<AHT30="+st.dAHT30[i]+"> ");
        }
        if (st.iAHT60 > -1) // Array Induction Resistivity-60
        {
          System.out.print("<AHT60="+st.dAHT60[i]+"> ");
        }
        if (st.iAHT90 > -1) // Array Induction Resistivity-90
        {
          System.out.print("<AHT90="+st.dAHT90[i]+"> ");
        }

        if (st.iTh > -1) // Thorium Curve Number in LAS File
        {
          System.out.print("<Th="+st.dTh[i]+"> ");
        }
        if (st.iU > -1) // Uranium Curve Number in LAS File
        {
          System.out.print("<U="+st.dU[i]+"> ");
        }
        if (st.iK > -1) // Potasium Curve Number in LAS File
        {
          System.out.print("<K="+st.dK[i]+"> ");
        }

        if (st.iLIN_1 > -1) // Unknown Linear Curve 1 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_1[i]+"> ");
        }
        if (st.iLIN_2 > -1) // Unknown Linear Curve 2 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_2[i]+"> ");
        }
        if (st.iLIN_3 > -1) // Unknown Linear Curve 3 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_3[i]+"> ");
        }
        if (st.iLIN_4 > -1) // Unknown Linear Curve 4 Number in LAS File
        {
          System.out.print("<Linear 1="+st.dLIN_4[i]+"> ");
        }

        if (st.iLOG_1 > -1) // Unknown Log Curve 1 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_1[i]+"> ");
        }
        if (st.iLOG_2 > -1) // Unknown Log Curve 2 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_2[i]+"> ");
        }
        if (st.iLOG_3 > -1) // Unknown Log Curve 3 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_3[i]+"> ");
        }
        if (st.iLOG_4 > -1) // Unknown Log Curve 4 Number in LAS File
        {
          System.out.print("<Log 1="+st.dLOG_4[i]+"> ");
        }

        System.out.println("");
      }
    }
  }

  /** Method printHeader()
   * <p> This method will print only the header information
   * @param st = the LAS File Data Structure
   */

  public static void printHeader( lasFileDataStruct st )
  {
    if (st != null)
    {
      if (st.sAPI.length() > 0) System.out.println( st.sAPI );  // API-Number of Well
      if (st.sName.length() > 0) System.out.println( st.sName ); // Well Name

      // Other Fields not really displayed

      if (st.state.length() > 0) System.out.println( st.state );    // State Name
      if (st.sCounty.length() > 0) System.out.println( st.sCounty ); // County Name

      // Well Location Information

      System.out.println( "Lat: "  + st.dLatitude + " " +   // Latitude of Well
                          "Long: " + st.dLongitude); // Longitude of Well

      // Well Parameter Information

      System.out.println( "KB: " + st.dKB + " " + // Kelly Bushing of Well
                          "GL: " + st.dGL + " " + // Ground Level of Well
                          "TD: " + st.dTD ); // Total Depth of Well

      // Well Status

      System.out.println( "Status: " + st.status ); // Status of Well

      // LAS File Information

      if (st.sReadMethod.length() > 0) System.out.println( "RD " + st.sReadMethod );  // File Read Method
      if (st.sDirectory.length() > 0) System.out.println( "Dir " + st.sDirectory ); // Directory Path
      if (st.sFilename.length() > 0) System.out.println( "File " + st.sFilename );  // File name of LAS File
      if (st.sVersion.length() > 0) System.out.println( "VER " + st.sVersion );   // LAS Version Number

      System.out.println(
        "Start Depth: " + st.depthStart + "\n" +
        "End Depth:   " + st.depthEnd   + "\n" +
        "Step By:     " + st.depthStep  + "\n" +
        "NULL Value:  " + st.dNull      + "\n\n");

      // -- Required Parameters

      if (st.sPDAT.length() > 0) System.out.println( "PERM " +st.sPDAT ); // Permanent Data
      System.out.println( "Above: " + st.dAPD );                 // Above Permanent Data
      if (st.sDREF.length() > 0) System.out.println( "REF " + st.sDREF ); // Depth Reference (KB,DF,CB)
      System.out.println( "Elev: " + st.dElevation );           // "Elevation of Depth Reference"

      // -- Required parameters for AEUB compliance

      if (st.dTDD  > 0.0) System.out.println( "TDD " + st.dTDD  );  // Drillers Depth
      if (st.dTLD  > 0.0) System.out.println( "TDL " + st.dTLD  );  // Loggers Depth

      if (st.dCSGL > 0.0) System.out.println( "CSGL " + st.dCSGL  ); // Casing Bottom Logger
      if (st.dCSGD > 0.0) System.out.println( "CSGD " + st.dCSGD  ); // Casing Bottom Driller
      if (st.dCSGS > 0.0) System.out.println( "CSGS " + st.dCSGS  ); // Casing Size
      if (st.dCSGW > 0.0) System.out.println( "CSGW " + st.dCSGW  ); // Casing Weight
      if (st.dBS   > 0.0) System.out.println( "BS "   + st.dBS    );   // Bit Size

      if (st.sMUD.length()  > 0) System.out.println( "MUD " + st.sMUD   );  // Mud type
      if (st.sMUDS.length() > 0) System.out.println( "MUDS " + st.sMUDS  ); // Mud Source
      if (st.dMUDD > 0.0) System.out.println( "MUDD " + st.dMUDD  ); // Mud Density
      if (st.dMUDV > 0.0) System.out.println( "MUDV " + st.dMUDV  ); // Mud Viscosity (Funnel)

      if (st.dFL   > 0.0) System.out.println( "FL " + st.dFL    );   // Fluid Loss
      if (st.dPH   > 0.0) System.out.println( "PH " + st.dPH    );   // PH

      if (st.dRM   > 0.0) System.out.println( "RM " + st.dRM    );   // Resistivity of Mud
      if (st.dRMT  > 0.0) System.out.println( "RMT " + st.dRMT   );  // Temperature of Mud
      if (st.dRMF  > 0.0) System.out.println( "RMF " + st.dRMF   );  // Resistivity of Mud Filtrate
      if (st.dRMFT > 0.0) System.out.println( "RMFT " + st.dRMFT  ); // Temperature of Mud Filtrate
      if (st.dRMC  > 0.0) System.out.println( "RMC " + st.dRMC   );  // Resistivity of Mud Cake
      if (st.dRMCT > 0.0) System.out.println( "RMCT " + st.dRMCT  ); // Temperature of Mud Cake
      if (st.dTMAX > 0.0) System.out.println( "TMAX " + st.dTMAX  ); // Maximum Recorded Temp.

      if (st.sTIMC.length() > 0) System.out.println( "TIMC " + st.sTIMC  ); // Date/Time Circulation Stopped
      if (st.sTIML.length() > 0) System.out.println( "TIML " + st.sTIML  ); // Date/Time Logger Tagged Bottom
      if (st.sUNIT.length() > 0) System.out.println( "UNIT " + st.sUNIT  ); // Logging Unit Number
      if (st.sBASE.length() > 0) System.out.println( "BASE " + st.sBASE  ); // Home Base of Logging Unit
      if (st.sENG.length()  > 0) System.out.println( "ENG " + st.sENG   );  // Recording Engineer
      if (st.sWIT.length()  > 0) System.out.println( "WIT " + st.sWIT   );  // Witnessed By
    }
  }

  // Columns

  public static final int _MNEMONIC    = las3.las3Struct._MNEMONIC;
  public static final int _UNIT        = las3.las3Struct._UNIT;
  public static final int _VALUE       = las3.las3Struct._VALUE;
  public static final int _DESCRIPTION = las3.las3Struct._DESCRIPTION;
  public static final int _FORMAT      = las3.las3Struct._FORMAT;
  public static final int _ASSOCIATION = las3.las3Struct._ASSOCIATION;

  /** Method printParameters()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  iParamRows  = Total number of rows
   *  @param  iParamCols  = Total number of columns
   *  @param  sParams     = Array holding the Parameter Definitions
   */

  public static void printParameters( int iParamRows, int iParamCols, String sParams[][] )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    int    SPACE  = 15;

    if (iParamRows > 0)
    {
      str = new String("~Log_Parameter A\n");

      SPACE = 10;

      if (iParamRows > 0)
      {
        for (i=0; i<iParamRows; i++)
        {
          len = sParams[i][_MNEMONIC].length();
          if (iMnem < len)   iMnem = len;

          len = sParams[i][_UNIT].length();
          if (iUnit < len)   iUnit = len;

          len = sParams[i][_VALUE].length();
          if (iValue < len)  iValue = len;
        }
      }

      // Write a Comment Identifying the Columns

      sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
      sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
      sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
      sDES   = String.format(" : %-30s ",           "DESCRIPTION");
      sFORM  = String.format("{%s}",                "FORMAT");
      sASSOC = String.format(" | %s",               "ASSOCIATION");

      str = new String(
          str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n");

      for (i=0; i<iParamRows; i++)
      {
        sMNEM  = String.format(" %-" + iMnem  + "s ", sParams[i][_MNEMONIC]);
        sUNIT  = String.format(".%-" + iUnit  + "s ", sParams[i][_UNIT]);
        sVAL   = String.format(" %"  + iValue + "s ", sParams[i][_VALUE]);
        sDES   = String.format(" : %-30s ", sParams[i][_DESCRIPTION]);

        sFORM = new String("");
        if (sParams[i][_FORMAT].length() > 0)
          sFORM  = String.format("{%s}",    sParams[i][_FORMAT]);

        sASSOC = new String("");
        if (sParams[i][_ASSOCIATION].length() > 0)
          sASSOC = String.format(" | %s",   sParams[i][_ASSOCIATION]);

        str = new String(
                str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n" );
      }

      str = new String(str + "\n");

      System.out.println( str );
    }
  }

  /** Method printParameters()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  st    = LAS Version 3.0 Data Structure
   */

  public static void printParameters( las3Struct st )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    int    SPACE  = 15;

    if (st != null)
    {
      if (st.iParamRows > 0)
      {
        if (st.iType == las3.las3Constants._LOG)
          str = new String("~Parameter\n");
        else
          str = new String("~" + st.sType + "_Parameter\n");

        iArray = st.iArray + 1;
        str = new String("~" + st.sType + "_Parameter[" + iArray + "]\n");

        SPACE = 10;

        if (st.iParamRows > 0)
        {
          for (i=0; i<st.iParamRows; i++)
          {
            len = st.sParams[i][_MNEMONIC].length();
            if (iMnem < len)   iMnem = len;

            len = st.sParams[i][_UNIT].length();
            if (iUnit < len)   iUnit = len;

            len = st.sParams[i][_VALUE].length();
            if (iValue < len)  iValue = len;
          }
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n");

        for (i=0; i<st.iParamRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.sParams[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.sParams[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.sParams[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.sParams[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.sParams[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.sParams[i][_FORMAT]);

          sASSOC = new String("");
          if (st.sParams[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.sParams[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n" );
        }

        str = new String(str + "\n");

        System.out.println( str );
      }
    }
  }

  /** Method printCurvesIQ()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  st    = LAS Version 3.0 Data Structure
   */

  public static void printCurvesIQ( las3Struct st )
  {
    int i=0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    String sType  = "";
    int    SPACE  = 15;

    if (st != null)
    {
      if (st.iSelectRows > 0)
      {
        if (st.sType.equals("Log"))
          sType = "Las";
        else if (st.sType.equals("Core"))
          sType = "Rock";

        iArray = st.iArray + 1;
        str = new String( "~IQ_" + sType + "_Parameter[" + iArray + "]\n" );

        SPACE = 10;

        if (st.iSelectRows > 0)
        {
          for (i=0; i<st.iSelectRows; i++)
          {
            len = st.select[i][_MNEMONIC].length();
            if (iMnem < len)   iMnem = len;

            len = st.select[i][_UNIT].length();
            if (iUnit < len)   iUnit = len;

            len = st.select[i][_VALUE].length();
            if (iValue < len)  iValue = len;
          }
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n");

        for (i=0; i<st.iSelectRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.select[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.select[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.select[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.select[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.select[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.select[i][_FORMAT]);

          sASSOC = new String("");
          if (st.select[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.select[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n" );
        }

        str = new String(str + "\n");

        System.out.println( str );
      }
    }
  }

  /** Method printCurves()
   * <p> This method will build the LAS 3 File Output string for printing.
   *  @param  st    = LAS Version 3.0 Data Structure
   */

  public static void printCurves( las3Struct st )
  {
    int    i      = 0;
    String str    = "";
    String sMNEM  = "";
    String sUNIT  = "";
    String sVAL   = "";
    String sDES   = "";
    String sFORM  = "";
    String sASSOC = "";
    int    iArray = 0;
    int    iMnem  = 5;
    int    iUnit  = 8;
    int    iValue = 10;
    int    len    = 0;
    int    SPACE  = 15;

    if (st != null)
    {
      if (st.iCurveRows > 0)
      {
        iArray = st.iArray + 1;
        str = new String("~" + st.sType + "_Definition[" + iArray + "]\n");

        SPACE = 10;

        for (i=0; i<st.iCurveRows; i++)
        {
          len = st.sCurves[i][_MNEMONIC].length();
          if (iMnem < len)   iMnem = len;

          len = st.sCurves[i][_UNIT].length();
          if (iUnit < len)   iUnit = len;

          len = st.sCurves[i][_VALUE].length();
          if (iValue < len)  iValue = len;
        }

        // Write a Comment Identifying the Columns

        sMNEM  = String.format("%-"  + iMnem  + "s ", "#MNEM");
        sUNIT  = String.format(".%-" + iUnit  + "s ", "UNIT");
        sVAL   = String.format(" %"  + iValue + "s ", "VALUE");
        sDES   = String.format(" : %-30s ",           "DESCRIPTION");
        sFORM  = String.format("{%s}",                "FORMAT");
        sASSOC = String.format(" | %s",               "ASSOCIATION");

        str = new String(
            str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n");

        for (i=0; i<st.iCurveRows; i++)
        {
          sMNEM  = String.format(" %-" + iMnem  + "s ", st.sCurves[i][_MNEMONIC]);
          sUNIT  = String.format(".%-" + iUnit  + "s ", st.sCurves[i][_UNIT]);
          sVAL   = String.format(" %"  + iValue + "s ", st.sCurves[i][_VALUE]);
          sDES   = String.format(" : %-30s ", st.sCurves[i][_DESCRIPTION]);

          sFORM = new String("");
          if (st.sCurves[i][_FORMAT].length() > 0)
            sFORM  = String.format("{%s}",    st.sCurves[i][_FORMAT]);

          sASSOC = new String("");
          if (st.sCurves[i][_ASSOCIATION].length() > 0)
            sASSOC = String.format(" | %s",   st.sCurves[i][_ASSOCIATION]);

          str = new String(
                  str + sMNEM + sUNIT + sVAL + sDES + sFORM + sASSOC + "\n" );
        }

        str = new String(str + "\n");

        System.out.println( str );
      }
    }
  }
}