/*
 * @lasFileDataStruct.java Version 1.1 09/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las;

import iqstrat.iqstratShaleListStruct;
import las3.las3Struct;

/** Class lasFileDataStruct
 *  <p> This Class holds the LAS File Well Data.
 *
 *  @version 1.1 09/05/2007
 *  @author  John R. Victorine
 */

public class lasFileDataStruct
{
  public int    iSource    = las3.las3Constants._SOURCE_NONE;

  // Well Information Data

  public String sKID       = "";  // Primary Key for the KGS Well
  public String sKEY       = "";  // Primary Key created by user
  public int    iType      = iqstrat.iqstratHeadersStruct.WELL; // Type of Data

  public String sAPI       = "";  // API-Number of Well
  public String sName      = "";  // Well Name
  public String state      = "";  // State Name
  public String sCounty    = "";  // County Name
  public String status     = "";  // Status of Well, i.e., OIL, GAS, etc.

  public double dLatitude  = 0.0; // Latitude of Well
  public double dLongitude = 0.0; // Longitude of Well

  public double dKB        = 0.0; // Kelly Bushing of Well
  public double dGL        = 0.0; // Ground Level of Well
  public double dTD        = 0.0; // Total Depth of Well

  // LAS File Information

  public String sReadMethod  = "";  // File Read Method
  public String sDirectory   = "";  // Directory Path to the LAS File
  public String sFilename    = "";  // File name of LAS File
  public String sVersion     = "";  // LAS Version Number

  public double dNull        = -999.95; // Default Null Value of Data
  public double depthStart   = 0.0;     // Start Depth of LAS Data
  public double depthEnd     = 0.0;     // Ending Depth of LAS Data
  public double depthStep    = 0.0;     // Depth Step of LAS DAta

  public double dTimeStart   = 0.0;     // Start Time of LAS Data
  public double dTimeEnd     = 0.0;     // Ending Time of LAS Data

  public int    iProprietary = 0; // Proprietary Data Identfier

  // LAS 3 File Parameters

  // -- Required Parameters

  public String sPDAT      = "";    // Permanent Data
  public double dAPD       = 0.0;   // Above Permanent Data
  public String sDREF      = "";    // Depth Reference (KB,DF,CB)
  public double dElevation = 0.0;   // "Elevation of Depth Reference"

  // -- Required parameters for AEUB compliance

  public double dTDD       = 0.0;   // Drillers Depth
  public double dTLD       = 0.0;   // Loggers Depth
  public double dCSGL      = 0.0;   // Casing Bottom Logger
  public double dCSGD      = 0.0;   // Casing Bottom Driller
  public double dCSGS      = 0.0;   // Casing Size
  public double dCSGW      = 0.0;   // Casing Weight
  public double dBS        = 0.0;   // Bit Size
  public String sMUD       = "";    // Mud type
  public String sMUDS      = "";    // Mud Source
  public double dMUDD      = 0.0;   // Mud Density
  public double dMUDV      = 0.0;   // Mud Viscosity (Funnel)
  public double dFL        = 0.0;   // Fluid Loss
  public double dPH        = 0.0;   // PH
  public double dRM        = 0.0;   // Resistivity of Mud
  public double dRMT       = 0.0;   // Temperature of Mud
  public double dRMF       = 0.0;   //  Rest. of Mud Filtrate
  public double dRMFT      = 0.0;   // Temp. of Mud Filtrate
  public double dRMC       = 0.0;   // Rest. of Mud Cake
  public double dRMCT      = 0.0;   // Temp. of Mud Cake
  public double dTMAX      = 0.0;   // Max. Recorded Temp.
  public String sTIMC      = "";    // Date/Time Circulation Stopped
  public String sTIML      = "";    // Date/Time Logger Tagged Bottom
  public String sUNIT      = "";    // Logging Unit Number
  public String sBASE      = "";    // Home Base of Logging Unit
  public String sENG       = "";    // Recording Engineer
  public String sWIT       = "";    // Witnessed By

  // .. Units for Required parameters for AEUB compliance

  public String sCSGL      = "";    // Casing Bottom Logger
  public String sCSGD      = "";    // Casing Bottom Driller
  public String sCSGS      = "";    // Casing Size
  public String sCSGW      = "";    // Casing Weight
  public String sBS        = "";    // Bit Size
  public String sMUDD      = "";    // Mud Density
  public String sMUDV      = "";    // Mud Viscosity (Funnel)
  public String sFL        = "";    // Fluid Loss
  public String sRM        = "";    // Resistivity of Mud
  public String sRMT       = "";    // Temperature of Mud
  public String sRMF       = "";    //  Rest. of Mud Filtrate
  public String sRMFT      = "";    // Temp. of Mud Filtrate
  public String sRMC       = "";    // Rest. of Mud Cake
  public String sRMCT      = "";    // Temp. of Mud Cake
  public String sTMAX      = "";    // Max. Recorded Temp.

  // LAS 3 Data Structure without the data section

  public int        iLogs    = 0;
  public las3Struct stLAS3[] = null;

  public int    iParamRows  = 0;      // Total number of rows
  public int    iParamCols  = las3.las3Struct._TOTAL; // Total number of columns
  public String sParams[][] = null;   // Array holding the Parameter Definitions

  // Unknown Curve Mnemonics

  public String sUnknown[][] = {
    {"", ""},
    {"", ""},
    {"", ""},
    {"", ""},
  };

  // LAS File Curve Information

  public int    iColorlithDefault = -1;

  public int    iGR   = -1;     // Gamma Ray Curve Number in LAS File
  public int    iCGR  = -1;     // Gamma Ray Minus Uranium Curve Number in LAS File
  public int    iSP   = -1;     // Array of Spontaneous Potential Log Values
  public int    iCAL  = -1;     // Array of Caliper Log Values

  public int    iNPHI = -1;     // Neutron Porosity Curve Number in LAS File
  public int    iPE   = -1;     // Photoelectric Curve Number in LAS File
  public int    iRHOB = -1;     // Bulk Density Curve Number in LAS File
  public int    iDPHI = -1;     // Density Porosity Curve Number in LAS File
  public int    iRHO  = -1;     // Density Correction Curve Number in LAS File

  public int    iDT   = -1;     // Sonic Curve Number in LAS File
  public int    iSPHI = -1;     // Sonic Porisity Curve Number in LAS File

  public int    iCOND  = -1;    // Array of Conductivity
  public int    iCILD  = -1;    // Array of Deep Induction Conductivity
  public int    iCILM  = -1;    // Array of Medium Induction Conductivity

  public int    iRES   = -1;    // Array of Resistivity
  public int    iRMED  = -1;    // Array of Deep Resistivity
  public int    iRDEP  = -1;    // Array of Medium Resistivity
  public int    iRSHAL = -1;    // Array of Shallow Resistivity,

  public int    iILD   = -1;    // Array of Deep Induction Resistivity
  public int    iILM   = -1;    // Array of Medium Induction Resistivity
  public int    iSFLU  = -1;    // Array of Spherically Focused Resistivity

  public int    iLL    = -1;    // Array of Deep Laterolog Resistivity
  public int    iMLL   = -1;    // Array of Micro Laterolog Resistivity
  public int    iLL8   = -1;    // Array of Shallow Laterolog Resistivity

  public int    iLN    = -1;    // Array of Long Normal Resistivity
  public int    iSN    = -1;    // Array of Shallow Normal Resistivity

  public int    iMNOR  = -1;    // Array of Micro Normal Resistivity
  public int    iMSFL  = -1;    // Array of Micro Spherically Focused Resistivity
  public int    iMINV  = -1;    // Array of Micro Inverse Resistivity

  public int    iAHT10 = -1;    // Array Induction Resistivity-10
  public int    iAHT20 = -1;    // Array Induction Resistivity-20
  public int    iAHT30 = -1;    // Array Induction Resistivity-30
  public int    iAHT60 = -1;    // Array Induction Resistivity-60
  public int    iAHT90 = -1;    // Array Induction Resistivity-90

  public int    iTh    = -1;    // Thorium Curve Number in LAS File
  public int    iU     = -1;    // Uranium Curve Number in LAS File
  public int    iK     = -1;    // Potasium Curve Number in LAS File

  public int    iGRN   = -1;    // Gamma Ray Count
  public int    iNEUT  = -1;    // Neutron Counts

  public int    iPERM  = -1;    // Permeability

  public int    iTEMP  = -1;    // Temperature
  public int    iTEMPDIFF = -1; // Temperature Difference

  public int    iLIN_1  = -1;   // Linear Curve 1
  public int    iLIN_2  = -1;   // Linear Curve 2
  public int    iLIN_3  = -1;   // Linear Curve 3
  public int    iLIN_4  = -1;   // Linear Curve 4

  public int    iLOG_1  = -1;   // Log Curve 1
  public int    iLOG_2  = -1;   // Log Curve 2
  public int    iLOG_3  = -1;   // Log Curve 3
  public int    iLOG_4  = -1;   // Log Curve 4

  public int    iUmaa   = -1;   // Apparent Volumetric Photoelectric factor
  public int    iRhomaa = -1;   // Apparent Matrix Grain Density
  public int    iDTmaa  = -1;   // Apparent Matrix Grain Acoustic transit time

    // Alternate Lithology Curves

  public int    iNeutron = -1;  // Neutron Porosity
  public int    iDelta   = -1;  // Neutron - Density Porosity

  // Spectral Gamma Ray Lithology Curves

  public int    iThU     = -1;  // Th/U
  public int    iThK     = -1;  // Th/K

  // Zonation Curves
  // -- Core Data Curves

  public int    iPHI     = -1;  // Porosity
  public int    iGMCC    = -1;  // Grain Density

  // -- Pfeffer Like Curves

  public int    iRT    = -1;    // Total Resistivity
  public int    iRWA   = -1;    // Apparent Water Resistivity
  public int    iRO    = -1;    // Water Saturated Rock Resistivity
  public int    iRXRT  = -1;    // Quick Look Ratio
  public int    iMA    = -1;    // Archie Cementation
  public int    iPHIT  = -1;    // Total Porosity
  public int    iPHIE  = -1;    // Effective Porosity
  public int    iSW    = -1;    // Water Saturation
  public int    iBVW   = -1;    // Bulk Volume Water

  // -- Unknown Curves

  public int    iATT1  = -1;    // Att 1
  public int    iATT2  = -1;    // Att 2
  public int    iATT3  = -1;    // Att 3
  public int    iATT4  = -1;    // Att 4
  public int    iATT5  = -1;    // Att 5
  public int    iATT6  = -1;    // Att 6

  // -- Lithology Curves

  public int    iVCAL   = -1;   // Calcite Volume Fraction
  public int    iVQTZ   = -1;   // Quartz Volume Fraction
  public int    iVDOL   = -1;   // Dolomite Volume Fraction
  public int    iVSH    = -1;   // Shale Volume Fraction
  public int    iVANH   = -1;   // Anhydrite Volume Fraction

  // Lithofacies Log Curves

  public int    iRows       = 0;    // Number of Rows
  public double depths[]    = null; // Array of Depth Data Values
  public double dTime[]     = null; // Array of Time  Data Values

  public double dGR[]       = null; // Array of Gamma Ray Log Values
  public double dCGR[]      = null; // Array of Gamma Ray minus Uranium Log Values
  public double dSP[]       = null; // Array of Spontaneous Potential Log Values
  public double dCAL[]      = null; // Array of Caliper Log Values

  public double dNPHI[]     = null; // Array of Neutron Porosity Log Values
  public double dPE[]       = null; // Array of PhotoElectric Log Values
  public double dRHOB[]     = null; // Array of Bulk Density Log Values
  public double dDPHI[]     = null; // Array of Density Porosity Log Values
  public double dRHO[]      = null; // Array of Density Corrrection Log Values

  public double dDT[]       = null; // Array of Sonic Log Values
  public double dSPHI[]     = null; // Array of Sonic Porosity Log Values

  public double dCOND[]     = null; // Array of Conductivity
  public double dCILD[]     = null; // Array of Deep Induction Conductivity
  public double dCILM[]     = null; // Array of Medium Induction Conductivity

  public double dRES[]      = null; // Array of Resistivity
  public double dRMED[]     = null; // Array of Deep Resistivity
  public double dRDEP[]     = null; // Array of Medium Resistivity
  public double dRSHAL[]    = null; // Array of Shallow Resistivity,

  public double dILD[]      = null; // Array of Deep Induction Resistivity
  public double dILM[]      = null; // Array of Medium Induction Resistivity
  public double dSFLU[]     = null; // Array of Spherically Focused Resistivity

  public double dLL[]       = null; // Array of Deep Laterolog Resistivity
  public double dMLL[]      = null; // Array of Micro Laterolog Resistivity
  public double dLL8[]      = null; // Array of Shallow Laterolog Resistivity

  public double dLN[]       = null; // Array of Long Normal Resistivity
  public double dSN[]       = null; // Array of Shallow Normal Resistivity

  public double dMNOR[]     = null; // Array of Micro Normal Resistivity
  public double dMSFL[]     = null; // Array of Micro Spherically Focused Resistivity
  public double dMINV[]     = null; // Array of Micro Inverse Resistivity

  public double dAHT10[]    = null; // Array Induction Resistivity-10
  public double dAHT20[]    = null; // Array Induction Resistivity-20
  public double dAHT30[]    = null; // Array Induction Resistivity-30
  public double dAHT60[]    = null; // Array Induction Resistivity-60
  public double dAHT90[]    = null; // Array Induction Resistivity-90

  public double dTh[]       = null; // Array of Thorium Log Values
  public double dU[]        = null; // Array of Uranium Log Values
  public double dK[]        = null; // Array of Potassium Log Values

  public double dGRN[]      = null; // Gamma Ray Count
  public double dNEUT[]     = null; // Neutron Counts

  public double dPERM[]     = null; // Permeability

  public double dTEMP[]     = null; // Temperature
  public double dTEMPDIFF[] = null; // Temperature Difference

  public double dLIN_1[]    = null; // Linear Curve 1
  public double dLIN_2[]    = null; // Linear Curve 2
  public double dLIN_3[]    = null; // Linear Curve 3
  public double dLIN_4[]    = null; // Linear Curve 4

  public double dLOG_1[]    = null; // Log Curve 1
  public double dLOG_2[]    = null; // Log Curve 2
  public double dLOG_3[]    = null; // Log Curve 3
  public double dLOG_4[]    = null; // Log Curve 4

  // Rhomaa-Umaa Log Curves

  public double dUmaa[]     = null; // Apparent Volumetric Photoelectric factor
  public double dRhomaa[]   = null; // Apparent Matrix Grain Density
  public double dDTmaa[]    = null; // Apparent Matrix Grain Acoustic transit time

  // Alternate Lithology Curves

  public double dNeutron[]  = null; // Neutron Porosity
  public double delta[]     = null; // Neutron - Density Porosity

  // Spectral Gamma Ray Lithology Curves

  public double dThU[]      = null; // Th/U
  public double dThK[]      = null; // Th/K

  // Zonation Curves
  // -- Core Data Curves

  public double dPHI[]      = null; // Porosity
  public double dGMCC[]     = null; // Grain Density

  // -- Pfeffer Like Curves

  public double dRT[]     = null; // Total Resistivity
  public double dRWA[]    = null; // Apparent Water Resistivity
  public double dRO[]     = null; // Water Saturated Rock Resistivity
  public double dRXRT[]   = null; // Quick Look Ratio
  public double dMA[]     = null; // Archie Cementation
  public double dPHIT[]   = null; // Total Porosity
  public double dPHIE[]   = null; // Effective Porosity
  public double dSW[]     = null; // Water Saturation
  public double dBVW[]    = null; // Bulk Volume Water

  // -- Unknown Curves

  public double dATT1[]   = null; // Att 1
  public double dATT2[]   = null; // Att 2
  public double dATT3[]   = null; // Att 3
  public double dATT4[]   = null; // Att 4
  public double dATT5[]   = null; // Att 5
  public double dATT6[]   = null; // Att 6

  // -- Lithology Curves

  public double dVCAL[]   = null; // Calcite Volume Fraction
  public double dVQTZ[]   = null; // Quartz Volume Fraction
  public double dVDOL[]   = null; // Dolomite Volume Fraction
  public double dVSH[]    = null; // Shale Volume Fraction
  public double dVANH[]   = null; // Anhydrite Volume Fraction

  // Misc

  public double dSPMin      = 0.0;  // Minimum SP Value
  public double dSPMax      = 0.0;  // Maximum SP Value

  public double dPHIMin     = -.1;  // Minimum SP Value
  public double dPHIMax     = .3;  // Maximum SP Value

  public double dRhoMin     = 2.0;
  public double dRhoMax     = 3.0;

  // Counts to Log Conversion Variables

  // -- Gamma Ray Counts to Gamma Ray API

  public double dGRNMin     = 0.0;   // Minimum Gamma Ray Counts Value
  public double dGRNMax     = 100.0; // Maximum Gamma Ray Counts Value

  public double dGRCMin     = 0.0;   // Minimum Computation Value
  public double dGRCMax     = 150.0; // Maximum Computation Value

  // .. Gamma Ray Plot Limits

  public double dGRMin      = 0.0;   // Minimum Gamma Ray Value
  public double dGRMax      = 150.0; // Maximum Gamma Ray Value

  public double dGRNMinP    = 0.0;   // Minimum Gamma Ray Counts Value
  public double dGRNMaxP    = 10.0;  // Maximum Gamma Ray Counts Value

  // -- Neutron Counts to Neutron Porosity

  public double dNEUTMin    = 0.0;   // Minimum Neutron Counts Value
  public double dNEUTMax    = 100.0; // Maximum Neutron Counts Value

  public double dNPHICMin   = 0.01;  // Minimum Computation Value
  public double dNPHICMax   = 0.4;   // Maximum Computation Value

  // .. Neutron Plot Limits

  public double dNPHIMin    = 0.0;   // Minimum NPHI Plot Limit
  public double dNPHIMax     = 0.5;   // Maximum NPHI Plot Limit

  public double dNEUTMinP   = 0.0;   // Minimum Neutron Counts Value
  public double dNEUTMaxP   = 10.0;  // Maximum Neutron Counts Value

  // .. Temperature

  public double dTEMPMin    = 0.0;   // Minimum TEMP Plot Limit
  public double dTEMPMax    = 1.0;   // Maximum TEMP Plot Limit

  public double dTEMPDIFFMin = 0.0;   // Minimum TEMPDIFF Plot Limit
  public double dTEMPDIFFMax = 1.0;   // Maximum TEMPDIFF Plot Limit

  // .. Linear Curves

  public double dLIN_1Min = 0.0; // Linear Curve 1 Minimum
  public double dLIN_2Min = 0.0; // Linear Curve 2 Minimum
  public double dLIN_3Min = 0.0; // Linear Curve 3 Minimum
  public double dLIN_4Min = 0.0; // Linear Curve 4 Minimum

  public double dLIN_1Max = 1.0; // Linear Curve 1 Maximum
  public double dLIN_2Max = 1.0; // Linear Curve 2 Maximum
  public double dLIN_3Max = 1.0; // Linear Curve 3 Maximum
  public double dLIN_4Max = 1.0; // Linear Curve 4 Maximum

  // .. Semilog Curves

  public double dLOG_1Min = 0.1;    // Log Curve 1 Minimum
  public double dLOG_2Min = 0.1;    // Log Curve 2 Minimum
  public double dLOG_3Min = 0.1;    // Log Curve 3 Minimum
  public double dLOG_4Min = 0.1;    // Log Curve 4 Minimum

  public double dLOG_1Max = 1000.0; // Log Curve 1 Maximum
  public double dLOG_2Max = 1000.0; // Log Curve 2 Maximum
  public double dLOG_3Max = 1000.0; // Log Curve 3 Maximum
  public double dLOG_4Max = 1000.0; // Log Curve 4 Maximum

  // RX/RT Curve Limits

  public double dRXRTMin  = 0.0;    // Minimum Rx/Rt Value
  public double dRXRTMax  = 0.0;    // Maximum Rx/Rt Value

  // Gamma Ray Levels

  public iqstratShaleListStruct  stList = null;

  /** Method delete()
   * <p> This method will set the arrays and strings to null to force Java to
   *     deallocate memory
   */

  public void delete()
  {
    sKID       = null; // primary key by KGS db
    sKEY       = null; // Primary Key by user
    sAPI       = null; // API-Number of Well
    sName      = null; // Well Name

      // Other Fields not really displayed

    sCounty    = null; // County Name
    state      = null; // State Name

      // Well Location Information

    dLatitude  = 0.0; // Latitude of Well
    dLongitude = 0.0; // Longitude of Well

      // Well Parameter Information

    dKB        = 0.0; // Kelly Bushing of Well
    dGL        = 0.0; // Ground Level of Well
    dTD        = 0.0; // Total Depth of Well

      // Well Status

    status     = null; // Status of Well

    // LAS File Information

    sReadMethod  = null;  // File Read Method
    sDirectory   = null;  // Directory Path to the LAS File
    sFilename    = null;  // File name of LAS File
    sVersion     = null;  // LAS Version Number

    dNull        = 0.0; // Null Value of Data
    depthStart   = 0.0; // Start Depth of LAS Data
    depthEnd     = 0.0; // Ending Depth of LAS Data
    depthStep    = 0.0; // Depth Step of LAS DAta

    iProprietary = 0; // Proprietary Data Identfier

    // -- Required Parameters

    sPDAT      = null; // Permanent Data
    dAPD       = 0.0;  // Above Permanent Data
    sDREF      = null; // Depth Reference (KB,DF,CB)
    dElevation = 0.0;  // "Elevation of Depth Reference"

    // -- Required parameters for AEUB compliance

    dTDD    = 0.0;  // Drillers Depth
    dTLD    = 0.0;  // Loggers Depth

    dCSGL   = 0.0;  // Casing Bottom Logger
    dCSGD   = 0.0;  // Casing Bottom Driller
    dCSGS   = 0.0;  // Casing Size
    dCSGW   = 0.0;  // Casing Weight
    dBS     = 0.0;  // Bit Size

    sMUD    = null; // Mud type
    sMUDS   = null; // Mud Source
    dMUDD   = 0.0;  // Mud Density
    dMUDV   = 0.0;  // Mud Viscosity (Funnel)

    dFL     = 0.0;  // Fluid Loss
    dPH     = 0.0;  // PH

    dRM     = 0.0;  // Resistivity of Mud
    dRMT    = 0.0;  // Temperature of Mud
    dRMF    = 0.0;  // Resistivity of Mud Filtrate
    dRMFT   = 0.0;  // Temperature of Mud Filtrate
    dRMC    = 0.0;  // Resistivity of Mud Cake
    dRMCT   = 0.0;  // Temperature of Mud Cake
    dTMAX   = 0.0;  // Maximum Recorded Temp.

    sTIMC   = null; // Date/Time Circulation Stopped
    sTIML   = null; // Date/Time Logger Tagged Bottom
    sUNIT   = null; // Logging Unit Number
    sBASE   = null; // Home Base of Logging Unit
    sENG    = null; // Recording Engineer
    sWIT    = null; // Witnessed By

    // .. Units for Required parameters for AEUB compliance

    sCSGL   = null; // Casing Bottom Logger
    sCSGD   = null; // Casing Bottom Driller
    sCSGS   = null; // Casing Size
    sCSGW   = null; // Casing Weight
    sBS     = null; // Bit Size
    sMUDD   = null; // Mud Density
    sMUDV   = null; // Mud Viscosity (Funnel)
    sFL     = null; // Fluid Loss
    sRM     = null; // Resistivity of Mud
    sRMT    = null; // Temperature of Mud
    sRMF    = null; //  Rest. of Mud Filtrate
    sRMFT   = null; // Temp. of Mud Filtrate
    sRMC    = null; // Rest. of Mud Cake
    sRMCT   = null; // Temp. of Mud Cake
    sTMAX   = null; // Max. Recorded Temp.

    // LAS 3 File Parameters

    if (iLogs > 0)
    {
      for (int i=0; i<iLogs; i++)
      {
        if (stLAS3[i] != null)
          stLAS3[i].delete();
        stLAS3[i] = null;
      }
    }

    iParamRows = 0;    // Total number of rows
    iParamCols = 0;    // Total number of columns
    sParams    = null; // Array holding the Parameter Definitions

    // LAS File Curve Information

    iColorlithDefault = -1; // Default colorlith setting

    iGR       = -1;   // Gamma Ray Curve Number in LAS File
    iCGR      = -1;   // Gamma Ray Minus Uranium Curve Number in LAS File
    iSP       = -1;   // Array of Spontaneous Potential Log Values
    iCAL      = -1;   // Array of Caliper Log Values

    iNPHI     = -1;   // Neutron Porosity Curve Number in LAS File
    iPE       = -1;   // Photoelectric Curve Number in LAS File
    iRHOB     = -1;   // Bulk Density Curve Number in LAS File
    iDPHI     = -1;   // Density Porosity Curve Number in LAS File
    iRHO      = -1;   // Density Correction Curve Number in LAS File

    iDT       = -1;   // Sonic Curve Number in LAS File
    iSPHI     = -1;   // Sonic Porisity Curve Number in LAS File

    iCOND     = -1;   // Array of Conductivity
    iCILD     = -1;   // Array of Deep Induction Conductivity
    iCILM     = -1;   // Array of Medium Induction Conductivity

    iRES      = -1;   // Array of Resistivity
    iRMED     = -1;   // Array of Deep Resistivity
    iRDEP     = -1;   // Array of Medium Resistivity
    iRSHAL    = -1;   // Array of Shallow Resistivity,

    iILD      = -1;   // Array of Deep Induction Resistivity
    iILM      = -1;   // Array of Medium Induction Resistivity
    iSFLU     = -1;   // Array of Spherically Focused Resistivity

    iLL       = -1;   // Array of Deep Laterolog Resistivity
    iMLL      = -1;   // Array of Micro Laterolog Resistivity
    iLL8      = -1;   // Array of Shallow Laterolog Resistivity

    iLN       = -1;   // Array of Long Normal Resistivity
    iSN       = -1;   // Array of Shallow Normal Resistivity

    iMNOR     = -1;   // Array of Micro Normal Resistivity
    iMSFL     = -1;   // Array of Micro Spherically Focused Resistivity
    iMINV     = -1;   // Array of Micro Inverse Resistivity

    iAHT10    = -1;   // Array Induction Resistivity-10
    iAHT20    = -1;   // Array Induction Resistivity-20
    iAHT30    = -1;   // Array Induction Resistivity-30
    iAHT60    = -1;   // Array Induction Resistivity-60
    iAHT90    = -1;   // Array Induction Resistivity-90

    iTh       = -1;   // Thorium Curve Number in LAS File
    iU        = -1;   // Uranium Curve Number in LAS File
    iK        = -1;   // Potasium Curve Number in LAS File

    iGRN      = -1;   // Gamma Ray Count
    iNEUT     = -1;   // Neutron Counts
    iPERM     = -1;   // Permeability

    iTEMP     = -1;   // Temperature
    iTEMPDIFF = -1;   // Temperature Difference

    // Zonation Curves

    iPHI      = -1;   // Porosity
    iGMCC     = -1;   // Grain Density

    iRT       = -1;   // Total Resistivity
    iRWA      = -1;   // Apparent Water Resistivity
    iRO       = -1;   // Water Saturated Rock Resistivity
    iRXRT     = -1;   // Quick Look Ratio
    iMA       = -1;   // Archie Cementation
    iPHIT     = -1;   // Total Porosity
    iPHIE     = -1;   // Effective Porosity
    iSW       = -1;   // Water Saturation
    iBVW      = -1;   // Bulk Volume Water

    iATT1     = -1;   // Att 1
    iATT2     = -1;   // Att 2
    iATT3     = -1;   // Att 3
    iATT4     = -1;   // Att 4
    iATT5     = -1;   // Att 5
    iATT6     = -1;   // Att 6

    iVCAL     = -1;   // Calcite Volume Fraction
    iVQTZ     = -1;   // Quartz Volume Fraction
    iVDOL     = -1;   // Dolomite Volume Fraction
    iVSH      = -1;   // Shale Volume Fraction
    iVANH     = -1;   // Anhydrite Volume Fraction

    // Lithofacies Log Curves

    iRows     = 0;    // Number of Rows
    depths    = null; // Array of Depth Data Values
    dTime     = null; // Array of Time Data Values

    dGR       = null; // Array of Gamma Ray Log Values
    dCGR      = null; // Array of Gamma Ray minus Uranium Log Values
    dSP       = null; // Array of Spontaneous Potential Log Values
    dCAL      = null; // Array of Caliper Log Values

    dPE       = null; // Array of PhotoElectric Log Values
    dNPHI     = null; // Array of Neutron Porosity Log Values
    dRHOB     = null; // Array of Bulk Density Log Values
    dDPHI     = null; // Array of Density Porosity Log Values
    dRHO      = null; // Array of Density Corrrection Log Values

    dDT       = null; // Array of Sonic Log Values
    dSPHI     = null; // Array of Sonic Porosity Log Values

    dCOND     = null; // Array of Conductivity
    dCILD     = null; // Array of Deep Induction Conductivity
    dCILM     = null; // Array of Medium Induction Conductivity

    dRES      = null; // Array of Resistivity
    dRMED     = null; // Array of Deep Resistivity
    dRDEP     = null; // Array of Medium Resistivity
    dRSHAL    = null; // Array of Shallow Resistivity,

    dILD      = null; // Array of Deep Induction Resistivity
    dILM      = null; // Array of Medium Induction Resistivity
    dSFLU     = null; // Array of Spherically Focused Resistivity

    dLL       = null; // Array of Deep Laterolog Resistivity
    dMLL      = null; // Array of Micro Laterolog Resistivity
    dLL8      = null; // Array of Shallow Laterolog Resistivity

    dLN       = null; // Array of Long Normal Resistivity
    dSN       = null; // Array of Shallow Normal Resistivity

    dMNOR     = null; // Array of Micro Normal Resistivity
    dMSFL     = null; // Array of Micro Spherically Focused Resistivity
    dMINV     = null; // Array of Micro Inverse Resistivity

    dAHT10    = null; // Array Induction Resistivity-10
    dAHT20    = null; // Array Induction Resistivity-20
    dAHT30    = null; // Array Induction Resistivity-30
    dAHT60    = null; // Array Induction Resistivity-60
    dAHT90    = null; // Array Induction Resistivity-90

    dTh       = null; // Array of Thorium Log Values
    dU        = null; // Array of Uranium Log Values
    dK        = null; // Array of Potassium Log Values

    dGRN      = null; // Gamma Ray Count
    dNEUT     = null; // Neutron Counts
    dPERM     = null; // Permeability

    dTEMP     = null; // Temperature
    dTEMPDIFF = null; // Temperature Difference

      // Rhomaa-Umaa Log Curves

    dUmaa     = null; // Apparent Volumetric Photoelectric factor
    dRhomaa   = null; // Apparent Matrix Grain Density
    dDTmaa    = null; // Apparent Matrix Grain Acoustic transit time

      // Alternate Lithology Curves

    dNeutron  = null; // Neutron Porosity
    delta     = null; // Neutron - Density Porosity

    // Spectral Gamma Ray Lithology Curves

    dThU      = null; // Th/U
    dThK      = null; // Th/K

    // Zonation Curves

    dPHI      = null; // Porosity
    dGMCC     = null; // Grain Density

    dRT       = null; // Total Resistivity
    dRWA      = null; // Apparent Water Resistivity
    dRO       = null; // Water Saturated Rock Resistivity
    dRXRT     = null; // Quick Look Ratio
    dMA       = null; // Archie Cementation
    dPHIT     = null; // Total Porosity
    dPHIE     = null; // Effective Porosity
    dSW       = null; // Water Saturation
    dBVW      = null; // Bulk Volume Water

    dATT1     = null; // Att 1
    dATT2     = null; // Att 2
    dATT3     = null; // Att 3
    dATT4     = null; // Att 4
    dATT5     = null; // Att 5
    dATT6     = null; // Att 6

    dVCAL     = null; // Calcite Volume Fraction
    dVQTZ     = null; // Quartz Volume Fraction
    dVDOL     = null; // Dolomite Volume Fraction
    dVSH      = null; // Shale Volume Fraction
    dVANH     = null; // Anhydrite Volume Fraction

    // Misc

    dSPMin       = 0.0;
    dSPMax       = 0.0;

    // .. Temperature

    dTEMPMin     = 0.0;   // Minimum TEMP Plot Limit
    dTEMPMax     = 0.0;   // Maximum TEMP Plot Limit

    dTEMPDIFFMin = 0.0;   // Minimum TEMPDIFF Plot Limit
    dTEMPDIFFMax = 0.0;   // Maximum TEMPDIFF Plot Limit

    if (stList != null)
      stList.delete();
    stList = null;
  }


  /** Method checkData(int iCurve)
   * <p> This method will check to see if the curve data is present
   * @param  iCurve = the requested curve type
   * @return bvalue = true data exist, false data does not exist
   */

  public boolean checkData(int iCurve)
  {
    boolean bValue = false;
    int     data   = -1;

    switch (iCurve)
    {
      case las.lasStandardTools._GR:
        data = iGR;
        break;
      case las.lasStandardTools._CGR:
        data = iCGR;
        break;
      case las.lasStandardTools._SP:
        data = iSP;
        break;
      case las.lasStandardTools._CAL:
        data = iCAL;
        break;

      case las.lasStandardTools._PE:
        data = iPE;
        break;
      case las.lasStandardTools._RHOB:
        data = iRHOB;
        break;
      case las.lasStandardTools._DPHI:
        data = iDPHI;
        break;
      case las.lasStandardTools._DRHO:
        data = iRHO;
        break;
      case las.lasStandardTools._NPHI:
        data = iNPHI;
        break;

      case las.lasStandardTools._DT:
        data = iDT;
        break;
      case las.lasStandardTools._SPHI:
        data = iSPHI;
        break;

      case las.lasStandardTools._COND:
        data = iCOND;
        break;
      case las.lasStandardTools._CILD:
        data = iCILD;
        break;
      case las.lasStandardTools._CILM:
        data = iCILM;
        break;

      case las.lasStandardTools._RES:
        data = iRES;
        break;
      case las.lasStandardTools._RDEP:
        data = iRDEP;
        break;
      case las.lasStandardTools._RMED:
        data = iRMED;
        break;
      case las.lasStandardTools._RSHAL:
        data = iRSHAL;
        break;

      case las.lasStandardTools._ILD:
        data = iILD;
        break;
      case las.lasStandardTools._ILM:
        data = iILM;
        break;
      case las.lasStandardTools._SFLU:
        data = iSFLU;
        break;

      case las.lasStandardTools._LL:
        data = iLL;
        break;
      case las.lasStandardTools._MLL:
        data = iMLL;
        break;
      case las.lasStandardTools._LL8:
        data = iLL8;
        break;

      case las.lasStandardTools._LN:
        data = iLN;
        break;
      case las.lasStandardTools._SN:
        data = iSN;
        break;

      case las.lasStandardTools._MNOR:
        data = iMNOR;
        break;
      case las.lasStandardTools._MSFL:
        data = iMSFL;
        break;
      case las.lasStandardTools._MINV:
        data = iMINV;
        break;

      case las.lasStandardTools._AHT10:
        data = iAHT10;
        break;
      case las.lasStandardTools._AHT20:
        data = iAHT20;
        break;
      case las.lasStandardTools._AHT30:
        data = iAHT30;
        break;
      case las.lasStandardTools._AHT60:
        data = iAHT60;
        break;
      case las.lasStandardTools._AHT90:
        data = iAHT90;
        break;

      case las.lasStandardTools._TH:
        data = iTh;
        break;
      case las.lasStandardTools._U:
        data = iU;
        break;
      case las.lasStandardTools._K:
        data = iK;
        break;

      case las.lasStandardTools._GRN:
        data = iGRN;
        break;
      case las.lasStandardTools._NEUT:
        data = iNEUT;
        break;
      case las.lasStandardTools._PERM:
        data = iPERM;
        break;

      case las.lasStandardTools._TEMP:
        data = iTEMP;
        break;
      case las.lasStandardTools._TEMPDIFF:
        data = iTEMPDIFF;
        break;

      case las.lasStandardTools._LIN_1:
        data = iLIN_1;
        break;
      case las.lasStandardTools._LIN_2:
        data = iLIN_2;
        break;
      case las.lasStandardTools._LIN_3:
        data = iLIN_3;
        break;
      case las.lasStandardTools._LIN_4:
        data = iLIN_4;
        break;

      case las.lasStandardTools._LOG_1:
        data = iLOG_1;
        break;
      case las.lasStandardTools._LOG_2:
        data = iLOG_2;
        break;
      case las.lasStandardTools._LOG_3:
        data = iLOG_3;
        break;
      case las.lasStandardTools._LOG_4:
        data = iLOG_4;
        break;

      case las.lasStandardTools._RHOMAA:
        data = iRhomaa;
        break;
      case las.lasStandardTools._UMAA:
        data = iUmaa;
        break;
      case las.lasStandardTools._DTMAA:
        data = iDTmaa;
        break;
      case las.lasStandardTools._PHIDIFF:
        data = iDelta;
        break;

      case las.lasStandardTools._THU:
        data = iThU;
        break;
      case las.lasStandardTools._THK:
        data = iThK;
        break;

      case las.lasStandardTools._PHI:
        data = iPHI;
        break;
      case las.lasStandardTools._GMCC:
        data = iGMCC;
        break;

      case las.lasStandardTools._RTo:
        data = iRT;
        break;
      case las.lasStandardTools._RWA:
        data = iRWA;
        break;
      case las.lasStandardTools._RO:
        data = iRO;
        break;
      case las.lasStandardTools._RXRT:
        data = iRXRT;
        break;
      case las.lasStandardTools._MA:
        data = iMA;
        break;

      case las.lasStandardTools._PHITo:
        data = iPHIT;
        break;
      case las.lasStandardTools._PHIE:
        data = iPHIE;
        break;

      case las.lasStandardTools._SW:
        data = iSW;
        break;
      case las.lasStandardTools._BVW:
        data = iBVW;
        break;

      case las.lasStandardTools._ATT1:
        data = iATT1;
        break;
      case las.lasStandardTools._ATT2:
        data = iATT2;
        break;
      case las.lasStandardTools._ATT3:
        data = iATT3;
        break;
      case las.lasStandardTools._ATT4:
        data = iATT4;
        break;
      case las.lasStandardTools._ATT5:
        data = iATT5;
        break;
      case las.lasStandardTools._ATT6:
        data = iATT6;
        break;

      case las.lasStandardTools._VCAL:
        data = iVCAL;    // Calcite Volume Fraction
        break;
      case las.lasStandardTools._VQTZ:
        data = iVQTZ;    // Quartz Volume Fraction
        break;
      case las.lasStandardTools._VDOL:
        data = iVDOL;    // Dolomite Volume Fraction
        break;
      case las.lasStandardTools._VSH:
        data = iVSH;    // Shale Volume Fraction
        break;
      case las.lasStandardTools._VANH:
        data = iVANH;    // Anhydrite Volume Fraction
        break;
    }

    if (data > -1)
      bValue = true;

    return (bValue);
  }

  /** Method getData(int iCurve)
   * <p> This method will retrive the curve data requested
   * @param  iCurve = the requested curve type
   * @return data   = the data requested.
   */

  public double[] getData(int iCurve)
  {
    double data[] = null;

    if (iRows > 0)
    {
      data = new double[iRows];

      switch (iCurve)
      {
        case las.lasStandardTools._GR:
          data = dGR;
          break;
        case las.lasStandardTools._CGR:
          data = dCGR;
          break;
        case las.lasStandardTools._SP:
          data = dSP;
          break;
        case las.lasStandardTools._CAL:
          data = dCAL;
          break;

        case las.lasStandardTools._PE:
          data = dPE;
          break;
        case las.lasStandardTools._RHOB:
          data = dRHOB;
          break;
        case las.lasStandardTools._DPHI:
          data = dDPHI;
          break;
        case las.lasStandardTools._DRHO:
          data = dRHO;
          break;
        case las.lasStandardTools._NPHI:
          data = dNPHI;
          break;

        case las.lasStandardTools._DT:
          data = dDT;
          break;
        case las.lasStandardTools._SPHI:
          data = dSPHI;
          break;

        case las.lasStandardTools._COND:
          data = dCOND;
          break;
        case las.lasStandardTools._CILD:
          data = dCILD;
          break;
        case las.lasStandardTools._CILM:
          data = dCILM;
          break;

        case las.lasStandardTools._RES:
          data = dRES;
          break;
        case las.lasStandardTools._RDEP:
          data = dRDEP;
          break;
        case las.lasStandardTools._RMED:
          data = dRMED;
          break;
        case las.lasStandardTools._RSHAL:
          data = dRSHAL;
          break;

        case las.lasStandardTools._ILD:
          data = dILD;
          break;
        case las.lasStandardTools._ILM:
          data = dILM;
          break;
        case las.lasStandardTools._SFLU:
          data = dSFLU;
          break;

        case las.lasStandardTools._LL:
          data = dLL;
          break;
        case las.lasStandardTools._MLL:
          data = dMLL;
          break;
        case las.lasStandardTools._LL8:
          data = dLL8;
          break;

        case las.lasStandardTools._LN:
          data = dLN;
          break;
        case las.lasStandardTools._SN:
          data = dSN;
          break;

        case las.lasStandardTools._MNOR:
          data = dMNOR;
          break;
        case las.lasStandardTools._MSFL:
          data = dMSFL;
          break;
        case las.lasStandardTools._MINV:
          data = dMINV;
          break;

        case las.lasStandardTools._AHT10:
          data = dAHT10;
          break;
        case las.lasStandardTools._AHT20:
          data = dAHT20;
          break;
        case las.lasStandardTools._AHT30:
          data = dAHT30;
          break;
        case las.lasStandardTools._AHT60:
          data = dAHT60;
          break;
        case las.lasStandardTools._AHT90:
          data = dAHT90;
          break;

        case las.lasStandardTools._TH:
          data = dTh;
          break;
        case las.lasStandardTools._U:
          data = dU;
          break;
        case las.lasStandardTools._K:
          data = dK;
          break;

        case las.lasStandardTools._GRN:
          data = dGRN;
          break;
        case las.lasStandardTools._NEUT:
          data = dNEUT;
          break;
        case las.lasStandardTools._PERM:
          data = dPERM;
          break;

        case las.lasStandardTools._TEMP:
          data = dTEMP;
          break;
        case las.lasStandardTools._TEMPDIFF:
          data = dTEMPDIFF;
          break;

        case las.lasStandardTools._LIN_1:
          data = dLIN_1;
          break;
        case las.lasStandardTools._LIN_2:
          data = dLIN_2;
          break;
        case las.lasStandardTools._LIN_3:
          data = dLIN_3;
          break;
        case las.lasStandardTools._LIN_4:
          data = dLIN_4;
          break;

        case las.lasStandardTools._LOG_1:
          data = dLOG_1;
          break;
        case las.lasStandardTools._LOG_2:
          data = dLOG_2;
          break;
        case las.lasStandardTools._LOG_3:
          data = dLOG_3;
          break;
        case las.lasStandardTools._LOG_4:
          data = dLOG_4;
          break;

        case las.lasStandardTools._RHOMAA:
          data = dRhomaa;
          break;
        case las.lasStandardTools._UMAA:
          data = dUmaa;
          break;
        case las.lasStandardTools._DTMAA:
          data = dDTmaa;
          break;
        case las.lasStandardTools._PHIDIFF:
          data = delta;
          break;

        case las.lasStandardTools._THU:
          data = dThU;
          break;
        case las.lasStandardTools._THK:
          data = dThK;
          break;

        case las.lasStandardTools._PHI:
          data = dPHI;
          break;
        case las.lasStandardTools._GMCC:
          data = dGMCC;
          break;

        case las.lasStandardTools._RTo:
          data = dRT;
          break;
        case las.lasStandardTools._RWA:
          data = dRWA;
          break;
        case las.lasStandardTools._RO:
          data = dRO;
          break;
        case las.lasStandardTools._RXRT:
          data = dRXRT;
          break;
        case las.lasStandardTools._MA:
          data = dMA;
          break;

        case las.lasStandardTools._PHITo:
          data = dPHIT;
          break;
        case las.lasStandardTools._PHIE:
          data = dPHIE;
          break;

        case las.lasStandardTools._SW:
          data = dSW;
          break;
        case las.lasStandardTools._BVW:
          data = dBVW;
          break;

        case las.lasStandardTools._ATT1:
          data = dATT1;
          break;
        case las.lasStandardTools._ATT2:
          data = dATT2;
          break;
        case las.lasStandardTools._ATT3:
          data = dATT3;
          break;
        case las.lasStandardTools._ATT4:
          data = dATT4;
          break;
        case las.lasStandardTools._ATT5:
          data = dATT5;
          break;
        case las.lasStandardTools._ATT6:
          data = dATT6;
          break;

        case las.lasStandardTools._VCAL:
          data = dVCAL;    // Calcite Volume Fraction
          break;
        case las.lasStandardTools._VQTZ:
          data = dVQTZ;    // Quartz Volume Fraction
          break;
        case las.lasStandardTools._VDOL:
          data = dVDOL;    // Dolomite Volume Fraction
          break;
        case las.lasStandardTools._VSH:
          data = dVSH;    // Shale Volume Fraction
          break;
        case las.lasStandardTools._VANH:
          data = dVANH;    // Anhydrite Volume Fraction
          break;
      }
    }

    return (data);
  }

  /** Method getRow(int iCurve, int iRow)
   * <p> This method will retrive the curve data requested
   * @param  iCurve  = the requested curve type
   * @param  iRow    = The LAS Row
   * @return data    = the data requested.
   */

  public double getRow(int iCurve, int iRow)
  {
    double data = dNull;

    if (iRows > 0)
    {
      switch (iCurve)
      {
        case las.lasStandardTools._GR:
          data = dGR[iRow];
          break;
        case las.lasStandardTools._CGR:
          data = dCGR[iRow];
          break;
        case las.lasStandardTools._SP:
          data = dSP[iRow];
          break;
        case las.lasStandardTools._CAL:
          data = dCAL[iRow];
          break;

        case las.lasStandardTools._PE:
          data = dPE[iRow];
          break;
        case las.lasStandardTools._RHOB:
          data = dRHOB[iRow];
          break;
        case las.lasStandardTools._DPHI:
          data = dDPHI[iRow];
          break;
        case las.lasStandardTools._DRHO:
          data = dRHO[iRow];
          break;
        case las.lasStandardTools._NPHI:
          data = dNPHI[iRow];
          break;

        case las.lasStandardTools._DT:
          data = dDT[iRow];
          break;
        case las.lasStandardTools._SPHI:
          data = dSPHI[iRow];
          break;

        case las.lasStandardTools._COND:
          data = dCOND[iRow];
          break;
        case las.lasStandardTools._CILD:
          data = dCILD[iRow];
          break;
        case las.lasStandardTools._CILM:
          data = dCILM[iRow];
          break;

        case las.lasStandardTools._RES:
          data = dRES[iRow];
          break;
        case las.lasStandardTools._RDEP:
          data = dRDEP[iRow];
          break;
        case las.lasStandardTools._RMED:
          data = dRMED[iRow];
          break;
        case las.lasStandardTools._RSHAL:
          data = dRSHAL[iRow];
          break;

        case las.lasStandardTools._ILD:
          data = dILD[iRow];
          break;
        case las.lasStandardTools._ILM:
          data = dILM[iRow];
          break;
        case las.lasStandardTools._SFLU:
          data = dSFLU[iRow];
          break;

        case las.lasStandardTools._LL:
          data = dLL[iRow];
          break;
        case las.lasStandardTools._MLL:
          data = dMLL[iRow];
          break;
        case las.lasStandardTools._LL8:
          data = dLL8[iRow];
          break;

        case las.lasStandardTools._LN:
          data = dLN[iRow];
          break;
        case las.lasStandardTools._SN:
          data = dSN[iRow];
          break;

        case las.lasStandardTools._MNOR:
          data = dMNOR[iRow];
          break;
        case las.lasStandardTools._MSFL:
          data = dMSFL[iRow];
          break;
        case las.lasStandardTools._MINV:
          data = dMINV[iRow];
          break;

        case las.lasStandardTools._AHT10:
          data = dAHT10[iRow];
          break;
        case las.lasStandardTools._AHT20:
          data = dAHT20[iRow];
          break;
        case las.lasStandardTools._AHT30:
          data = dAHT30[iRow];
          break;
        case las.lasStandardTools._AHT60:
          data = dAHT60[iRow];
          break;
        case las.lasStandardTools._AHT90:
          data = dAHT90[iRow];
          break;

        case las.lasStandardTools._TH:
          data = dTh[iRow];
          break;
        case las.lasStandardTools._U:
          data = dU[iRow];
          break;
        case las.lasStandardTools._K:
          data = dK[iRow];
          break;

        case las.lasStandardTools._GRN:
          data = dGRN[iRow];
          break;
        case las.lasStandardTools._NEUT:
          data = dNEUT[iRow];
          break;
        case las.lasStandardTools._PERM:
          data = dPERM[iRow];
          break;

        case las.lasStandardTools._TEMP:
          data = dTEMP[iRow];
          break;
        case las.lasStandardTools._TEMPDIFF:
          data = dTEMPDIFF[iRow];
          break;

        case las.lasStandardTools._LIN_1:
          data = dLIN_1[iRow];
          break;
        case las.lasStandardTools._LIN_2:
          data = dLIN_2[iRow];
          break;
        case las.lasStandardTools._LIN_3:
          data = dLIN_3[iRow];
          break;
        case las.lasStandardTools._LIN_4:
          data = dLIN_4[iRow];
          break;

        case las.lasStandardTools._LOG_1:
          data = dLOG_1[iRow];
          break;
        case las.lasStandardTools._LOG_2:
          data = dLOG_2[iRow];
          break;
        case las.lasStandardTools._LOG_3:
          data = dLOG_3[iRow];
          break;
        case las.lasStandardTools._LOG_4:
          data = dLOG_4[iRow];
          break;

        case las.lasStandardTools._RHOMAA:
          data = dRhomaa[iRow];
          break;
        case las.lasStandardTools._UMAA:
          data = dUmaa[iRow];
          break;
        case las.lasStandardTools._DTMAA:
          data = dDTmaa[iRow];
          break;
        case las.lasStandardTools._PHIDIFF:
          data = delta[iRow];
          break;

        case las.lasStandardTools._THU:
          data = dThU[iRow];
          break;
        case las.lasStandardTools._THK:
          data = dThK[iRow];
          break;

        case las.lasStandardTools._PHI:
          data = dPHI[iRow];
          break;
        case las.lasStandardTools._GMCC:
          data = dGMCC[iRow];
          break;

        case las.lasStandardTools._RTo:
          data = dRT[iRow];
          break;
        case las.lasStandardTools._RWA:
          data = dRWA[iRow];
          break;
        case las.lasStandardTools._RO:
          data = dRO[iRow];
          break;
        case las.lasStandardTools._RXRT:
          data = dRXRT[iRow];
          break;
        case las.lasStandardTools._MA:
          data = dMA[iRow];
          break;

        case las.lasStandardTools._PHITo:
          data = dPHIT[iRow];
          break;
        case las.lasStandardTools._PHIE:
          data = dPHIE[iRow];
          break;

        case las.lasStandardTools._SW:
          data = dSW[iRow];
          break;
        case las.lasStandardTools._BVW:
          data = dBVW[iRow];
          break;

        case las.lasStandardTools._ATT1:
          data = dATT1[iRow];
          break;
        case las.lasStandardTools._ATT2:
          data = dATT2[iRow];
          break;
        case las.lasStandardTools._ATT3:
          data = dATT3[iRow];
          break;
        case las.lasStandardTools._ATT4:
          data = dATT4[iRow];
          break;
        case las.lasStandardTools._ATT5:
          data = dATT5[iRow];
          break;
        case las.lasStandardTools._ATT6:
          data = dATT6[iRow];
          break;

        case las.lasStandardTools._VCAL:
          data = dVCAL[iRow];    // Calcite Volume Fraction
          break;
        case las.lasStandardTools._VQTZ:
          data = dVQTZ[iRow];    // Quartz Volume Fraction
          break;
        case las.lasStandardTools._VDOL:
          data = dVDOL[iRow];    // Dolomite Volume Fraction
          break;
        case las.lasStandardTools._VSH:
          data = dVSH[iRow];    // Shale Volume Fraction
          break;
        case las.lasStandardTools._VANH:
          data = dVANH[iRow];    // Anhydrite Volume Fraction
          break;

      }
    }

    return (data);
  }

  /** Method checkData(int iCurve)
   * <p> This method will check to see if the curve data is present
   * @param  iCurve  = the requested curve type
   * @param  iColumn = the column number for rock data
   */

  public void setData(int iCurve, int iColumn)
  {
    switch (iCurve)
    {
      case las.lasStandardTools._GR:
        iGR = iColumn;
        break;
      case las.lasStandardTools._CGR:
        iCGR = iColumn;
        break;
      case las.lasStandardTools._SP:
        iSP = iColumn;
        break;
      case las.lasStandardTools._CAL:
        iCAL = iColumn;
        break;

      case las.lasStandardTools._PE:
        iPE = iColumn;
        break;
      case las.lasStandardTools._RHOB:
        iRHOB = iColumn;
        break;
      case las.lasStandardTools._DPHI:
        iDPHI = iColumn;
        break;
      case las.lasStandardTools._DRHO:
        iRHO = iColumn;
        break;
      case las.lasStandardTools._NPHI:
        iNPHI = iColumn;
        break;

      case las.lasStandardTools._DT:
        iDT = iColumn;
        break;
      case las.lasStandardTools._SPHI:
        iSPHI = iColumn;
        break;

      case las.lasStandardTools._COND:
        iCOND = iColumn;
        break;
      case las.lasStandardTools._CILD:
        iCILD = iColumn;
        break;
      case las.lasStandardTools._CILM:
        iCILM = iColumn;
        break;

      case las.lasStandardTools._RES:
        iRES = iColumn;
        break;
      case las.lasStandardTools._RDEP:
        iRDEP = iColumn;
        break;
      case las.lasStandardTools._RMED:
        iRMED = iColumn;
        break;
      case las.lasStandardTools._RSHAL:
        iRSHAL = iColumn;
        break;

      case las.lasStandardTools._ILD:
        iILD = iColumn;
        break;
      case las.lasStandardTools._ILM:
        iILM = iColumn;
        break;
      case las.lasStandardTools._SFLU:
        iSFLU = iColumn;
        break;

      case las.lasStandardTools._LL:
        iLL = iColumn;
        break;
      case las.lasStandardTools._MLL:
        iMLL = iColumn;
        break;
      case las.lasStandardTools._LL8:
        iLL8 = iColumn;
        break;

      case las.lasStandardTools._LN:
        iLN = iColumn;
        break;
      case las.lasStandardTools._SN:
        iSN = iColumn;
        break;

      case las.lasStandardTools._MNOR:
        iMNOR = iColumn;
        break;
      case las.lasStandardTools._MSFL:
        iMSFL = iColumn;
        break;
      case las.lasStandardTools._MINV:
        iMINV = iColumn;
        break;

      case las.lasStandardTools._AHT10:
        iAHT10 = iColumn;
        break;
      case las.lasStandardTools._AHT20:
        iAHT20 = iColumn;
        break;
      case las.lasStandardTools._AHT30:
        iAHT30 = iColumn;
        break;
      case las.lasStandardTools._AHT60:
        iAHT60 = iColumn;
        break;
      case las.lasStandardTools._AHT90:
        iAHT90 = iColumn;
        break;

      case las.lasStandardTools._TH:
        iTh = iColumn;
        break;
      case las.lasStandardTools._U:
        iU = iColumn;
        break;
      case las.lasStandardTools._K:
        iK = iColumn;
        break;

      case las.lasStandardTools._RHOMAA:
        iRhomaa = iColumn;
        break;
      case las.lasStandardTools._UMAA:
        iUmaa = iColumn;
        break;
      case las.lasStandardTools._PHIDIFF:
        iDelta = iColumn;
        break;

      case las.lasStandardTools._THU:
        iThU = iColumn;
        break;
      case las.lasStandardTools._THK:
        iThK = iColumn;
        break;

      case las.lasStandardTools._PHI:
        iPHI = iColumn;
        break;

      case las.lasStandardTools._GMCC:
        iGMCC = iColumn;
        break;
    }
  }

  /** Method setRow(int iCurve, int iRow, double data)
   * <p> This method will set the curve data requested
   * @param  iCurve  = the requested curve type
   * @param  iRow    = The LAS Row
   * @param  data    = the data being modified.
   */

  public void setRow(int iCurve, int iRow, double data)
  {
    if (iRows > 0)
    {
      switch (iCurve)
      {
        case las.lasStandardTools._GR:
          dGR[iRow] = data;
          break;
        case las.lasStandardTools._CGR:
          dCGR[iRow] = data;
          break;
        case las.lasStandardTools._SP:
          dSP[iRow] = data;
          break;
        case las.lasStandardTools._CAL:
          dCAL[iRow] = data;
          break;

        case las.lasStandardTools._PE:
          dPE[iRow] = data;
          break;
        case las.lasStandardTools._RHOB:
          dRHOB[iRow] = data;
          break;
        case las.lasStandardTools._DPHI:
          dDPHI[iRow] = data;
          break;
        case las.lasStandardTools._DRHO:
          dRHO[iRow] = data;
          break;
        case las.lasStandardTools._NPHI:
          dNPHI[iRow] = data;
          break;

        case las.lasStandardTools._DT:
          dDT[iRow] = data;
          break;
        case las.lasStandardTools._SPHI:
          dSPHI[iRow] = data;
          break;

        case las.lasStandardTools._COND:
          dCOND[iRow] = data;
          break;
        case las.lasStandardTools._CILD:
          dCILD[iRow] = data;
          break;
        case las.lasStandardTools._CILM:
          dCILM[iRow] = data;
          break;

        case las.lasStandardTools._RES:
          dRES[iRow] = data;
          break;
        case las.lasStandardTools._RDEP:
          dRDEP[iRow] = data;
          break;
        case las.lasStandardTools._RMED:
          dRMED[iRow] = data;
          break;
        case las.lasStandardTools._RSHAL:
          dRSHAL[iRow] = data;
          break;

        case las.lasStandardTools._ILD:
          dILD[iRow] = data;
          break;
        case las.lasStandardTools._ILM:
          dILM[iRow] = data;
          break;
        case las.lasStandardTools._SFLU:
          dSFLU[iRow] = data;
          break;

        case las.lasStandardTools._LL:
          dLL[iRow] = data;
          break;
        case las.lasStandardTools._MLL:
          dMLL[iRow] = data;
          break;
        case las.lasStandardTools._LL8:
          dLL8[iRow] = data;
          break;

        case las.lasStandardTools._LN:
          dLN[iRow] = data;
          break;
        case las.lasStandardTools._SN:
          dSN[iRow] = data;
          break;

        case las.lasStandardTools._MNOR:
          dMNOR[iRow] = data;
          break;
        case las.lasStandardTools._MSFL:
          dMSFL[iRow] = data;
          break;
        case las.lasStandardTools._MINV:
          dMINV[iRow] = data;
          break;

        case las.lasStandardTools._AHT10:
          dAHT10[iRow] = data;
          break;
        case las.lasStandardTools._AHT20:
          dAHT20[iRow] = data;
          break;
        case las.lasStandardTools._AHT30:
          dAHT30[iRow] = data;
          break;
        case las.lasStandardTools._AHT60:
          dAHT60[iRow] = data;
          break;
        case las.lasStandardTools._AHT90:
          dAHT90[iRow] = data;
          break;

        case las.lasStandardTools._TH:
          dTh[iRow] = data;
          break;
        case las.lasStandardTools._U:
          dU[iRow] = data;
          break;
        case las.lasStandardTools._K:
          dK[iRow] = data;
          break;

        case las.lasStandardTools._GRN:
          dGRN[iRow] = data;
          break;
        case las.lasStandardTools._NEUT:
          dNEUT[iRow] = data;
          break;
        case las.lasStandardTools._PERM:
          dPERM[iRow] = data;
          break;

        case las.lasStandardTools._TEMP:
          dTEMP[iRow] = data;
          break;
        case las.lasStandardTools._TEMPDIFF:
          dTEMPDIFF[iRow] = data;
          break;

        case las.lasStandardTools._LIN_1:
          dLIN_1[iRow] = data;
          break;
        case las.lasStandardTools._LIN_2:
          dLIN_2[iRow] = data;
          break;
        case las.lasStandardTools._LIN_3:
          dLIN_3[iRow] = data;
          break;
        case las.lasStandardTools._LIN_4:
          dLIN_4[iRow] = data;
          break;

        case las.lasStandardTools._LOG_1:
          dLOG_1[iRow] = data;
          break;
        case las.lasStandardTools._LOG_2:
          dLOG_2[iRow] = data;
          break;
        case las.lasStandardTools._LOG_3:
          dLOG_3[iRow] = data;
          break;
        case las.lasStandardTools._LOG_4:
          dLOG_4[iRow] = data;
          break;

        case las.lasStandardTools._RHOMAA:
          dRhomaa[iRow] = data;
          break;
        case las.lasStandardTools._UMAA:
          dUmaa[iRow] = data;
          break;
        case las.lasStandardTools._DTMAA:
          dDTmaa[iRow] = data;
          break;
        case las.lasStandardTools._PHIDIFF:
          delta[iRow] = data;
          break;

        case las.lasStandardTools._THU:
          dThU[iRow] = data;
          break;
        case las.lasStandardTools._THK:
          dThK[iRow] = data;
          break;

        case las.lasStandardTools._PHI:
          dPHI[iRow] = data;
          break;
        case las.lasStandardTools._GMCC:
          dGMCC[iRow] = data;
          break;


        case las.lasStandardTools._RTo:
          dRT[iRow] = data;
          break;
        case las.lasStandardTools._RWA:
          dRWA[iRow] = data;
          break;
        case las.lasStandardTools._RO:
          dRO[iRow] = data;
          break;
        case las.lasStandardTools._RXRT:
          dRXRT[iRow] = data;
          break;
        case las.lasStandardTools._MA:
          dMA[iRow] = data;
          break;

        case las.lasStandardTools._PHITo:
          dPHIT[iRow] = data;
          break;
        case las.lasStandardTools._PHIE:
          dPHIE[iRow] = data;
          break;

        case las.lasStandardTools._SW:
          dSW[iRow] = data;
          break;
        case las.lasStandardTools._BVW:
          dBVW[iRow] = data;
          break;

        case las.lasStandardTools._ATT1:
          dATT1[iRow] = data;
          break;
        case las.lasStandardTools._ATT2:
          dATT2[iRow] = data;
          break;
        case las.lasStandardTools._ATT3:
          dATT3[iRow] = data;
          break;
        case las.lasStandardTools._ATT4:
          dATT4[iRow] = data;
          break;
        case las.lasStandardTools._ATT5:
          dATT5[iRow] = data;
          break;
        case las.lasStandardTools._ATT6:
          dATT6[iRow] = data;
          break;

        case las.lasStandardTools._VCAL:
          dVCAL[iRow] = data;    // Calcite Volume Fraction
          break;
        case las.lasStandardTools._VQTZ:
          dVQTZ[iRow] = data;    // Quartz Volume Fraction
          break;
        case las.lasStandardTools._VDOL:
          dVDOL[iRow] = data;    // Dolomite Volume Fraction
          break;
        case las.lasStandardTools._VSH:
          dVSH[iRow] = data;    // Shale Volume Fraction
          break;
        case las.lasStandardTools._VANH:
          dVANH[iRow] = data;    // Anhydrite Volume Fraction
          break;

      }
    }
  }
}