/*
 * @lasThinTrackFrame.java Version 1.1 02/05/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;

import las.lasFileDataStruct;

/** Class lasThinTrackFrame
 *  <p> This Class will allow the user to change the Porosity used on the
 *      LAS Porosity Thin Track
 *
 *  @version 1.1 02/05/2009
 *  @author  John R. Victorine
 */

public class lasThinTrackFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable        notifier = null;
  private lasFileDataStruct stLAS    = null;

  // Global Frame Variables

  private int               iPHI     = iqstrat.iqstratTracksStruct._THIN_NPHI;

  // Global Frame Widgets

  private JRadioButton rbNPHI  = new JRadioButton();
  private JRadioButton rbDPHI  = new JRadioButton();
  private JRadioButton rbTotal = new JRadioButton();
  private JRadioButton rbSPHI  = new JRadioButton();

  private JButton btnClose = new JButton();

  /** Constructor lasThinTrackFrame( Observable notifier, lasFileDataStruct stLAS )
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stLAS    = LAS File Data Structure
   */

  public lasThinTrackFrame( Observable notifier, lasFileDataStruct stLAS )
  {
    try
    {
      this.notifier = notifier;
      this.stLAS    = stLAS;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlPHI  = new JPanel();
    JPanel pnlBase = new JPanel();
    ButtonGroup group = new ButtonGroup();

    this.getContentPane().setLayout(new BorderLayout());

    // Porosity Panel

    pnlPHI.setLayout(new GridLayout(4,1));

    rbNPHI.setFont(new java.awt.Font("Dialog", 1, 11));
    rbNPHI.setSelected(true);
    rbNPHI.setText("NPHI - Neutron Porosity");
    rbNPHI.addActionListener(this);

    rbDPHI.setFont(new java.awt.Font("Dialog", 1, 11));
    rbDPHI.setText("DPHI - Density Porosity");
    rbDPHI.addActionListener(this);

    rbTotal.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTotal.setText("( NPHI + DPHI ) / 2");
    rbTotal.addActionListener(this);

    rbSPHI.setFont(new java.awt.Font("Dialog", 1, 11));
    rbSPHI.setText("SPHI - Sonic Porosity");
    rbSPHI.addActionListener(this);

    group.add(rbNPHI);
    group.add(rbDPHI);
    group.add(rbTotal);
    group.add(rbSPHI);

    // Base Panel

    pnlBase.setLayout(new BorderLayout());

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // Attach Widgets to Panel

    this.getContentPane().add(pnlPHI, null);
    if (stLAS != null)
    {
      if (stLAS.iNPHI > -1)   pnlPHI.add(rbNPHI,  null);
      if (stLAS.iDPHI > -1)   pnlPHI.add(rbDPHI,  null);
      if ((stLAS.iNPHI > -1) &&
          (stLAS.iDPHI > -1)) pnlPHI.add(rbTotal, null);
      if (stLAS.iSPHI > -1)   pnlPHI.add(rbSPHI,  null);
    }

    this.getContentPane().add(pnlBase,  BorderLayout.SOUTH);
    pnlBase.add(btnClose, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(200, 150));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(false);
  }

  /* ===================================================================== *
   * --------------------------- GET METHODS ----------------------------- *
   * ===================================================================== */

  /** Method getPHI()
   * <p> This method will set the porosity type to display
   * @return iPHI = Thin Track Porosity Identifier
   */

  public int getPHI() { return (iPHI); }

  /* ===================================================================== *
   * --------------------------- SET METHODS ----------------------------- *
   * ===================================================================== */

  /** Method setPHI( int iPHI )
   * <p> This method will set the porosity type to display
   * @param iPHI = Thin Track Porosity Identifier
   */

  public void setPHI( int iPHI )
  {
    this.iPHI = iPHI;

    rbNPHI.setSelected( false );
    rbDPHI.setSelected( false );
    rbTotal.setSelected( false );
    rbSPHI.setSelected( false );

    switch (iPHI)
    {
      case iqstrat.iqstratTracksStruct._THIN_NPHI:
        rbNPHI.setSelected( true );
        break;
      case iqstrat.iqstratTracksStruct._THIN_DPHI:
        rbDPHI.setSelected( true );
        break;
      case iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI:
        rbTotal.setSelected( true );
        break;
      case iqstrat.iqstratTracksStruct._THIN_SPHI:
        rbSPHI.setSelected( true );
        break;
    }
  }

  /** Method setPorosity
   * <p> This method will set the porosity type to display
   * @param iPHI = Thin Track Porosity Identifier
   */

  private void setPorosity( int iPHI )
  {
    this.iPHI = iPHI;
    notifier.notifyObservers(new String("Thin Porosity Changed"));
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    notifier = null;
    stLAS    = null;

    rbNPHI   = null;
    rbDPHI   = null;
    rbTotal  = null;
    rbSPHI   = null;

    btnClose = null;

    dispose();
  }

  /** METHOD actionPerformed( ActionEvent event )
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event = Action Event
   */

  public void actionPerformed( ActionEvent event )
  {
    if (event.getSource() == rbNPHI)
    {
      setPorosity(iqstrat.iqstratTracksStruct._THIN_NPHI);
    }
    if (event.getSource() == rbDPHI)
    {
      setPorosity(iqstrat.iqstratTracksStruct._THIN_DPHI);
    }
    if (event.getSource() == rbTotal)
    {
      setPorosity(iqstrat.iqstratTracksStruct._THIN_NPHI_DPHI);
    }
    if (event.getSource() == rbSPHI)
    {
      setPorosity(iqstrat.iqstratTracksStruct._THIN_SPHI);
    }

    if (event.getSource() == btnClose) { this.setVisible(false); }
  }
}

/*
 *  @version 1.1 02/05/2009
 *  @author  John Victorine
 */
