/*
 * @lasFilesFrame.java Version 1.1 10/22/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observer;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;

import iqstrat.iqstratStruct;

import cmn.cmnStruct;
import cmn.cmnLASFileListStruct;
import cmn.cmnLASFileStruct;
import cmn.cmnLASCurveListStruct;

import las.lasStruct;
import las3.las3ListStruct;
import las.lasFileDataStruct;
import las3.io.las3Read;
import las.gui.lasFilesTable;

/** Class lasFilesFrame
 *  <p> This Class will allow the user to load data files from their PC or the
 *      KGS Database
 *
 *  @version 1.1 10/22/2007
 *  @author  John R. Victorine
 *
 *  modified 02/09/2010 John R. Victorine
 *  Replace the LAS Read with a LAS 3 Read, this version will allow the user
 *  to override the default curve selections.
 */

public class lasFilesFrame extends JFrame implements ActionListener, Observer
{
  // Input Variables

  private Observable           pNotifier = null; // Calling Object Observables
  private iqstratStruct        stStruct  = null; // Global IQSTRAT Data Struct
  private cmnStruct            stCMN     = null; // Global Common Data Structure
  private cmnLASFileListStruct stList    = null; // Common LAS File List Struct

  // Global variables

  private Observable        notifier = null; // This Frame Observables Object
  private lasFilesTable     pTable   = null; // List of Available LAS Files

  private las3Read          pRead         = null;   // Read LAS File
  private las3ListStruct    stLAS3        = null;   // LAS 3 File List Structure
  private lasFileDataStruct stLASFileData = null;   // LAS File Data Struct
  private double            dStart        = 0.0;    // Starting LAS File Depth
  private double            dEnd          = 2000.0; // Ending LAS File Depth
  private int               iColorlith    = 0;      // Colorlith setting

  // Global Frame Widgets

  private JButton btnSelect = new JButton();
  private JButton btnCancel = new JButton();

  /** Constructor lasFilesFrame()
   *  <p> This is the Constructor for this class.
   *  @param stStruct  = IQSTRAT Global Data Structure
   *  @param pNotifier = Observable
   *  @param stList    = LAS File List Data Structure
   */

  public lasFilesFrame( iqstratStruct stStruct,
                        Observable pNotifier,
                        cmnLASFileListStruct stList )
  {
    try
    {
      this.stStruct  = stStruct;
      this.stCMN     = stStruct.stCMN;
      this.pNotifier = pNotifier;
      this.stList    = stList;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlButtons = new JPanel();
    JPanel pnlCenter  = new JPanel();
    JScrollPane scroll = new JScrollPane();

    // Create an Observable

    notifier = new lasFilesFrameObservable();

    // Add this object to the notifier as observer

    notifier.addObserver(this);

    // Build Frame

    this.setTitle("Select LAS File");
    this.getContentPane().setLayout(new BorderLayout());

    // Global Frame Buttons Panel

    pnlButtons.setBorder(BorderFactory.createEtchedBorder());

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setText("Select");
    btnSelect.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // List of Available LAS Files

    pnlCenter.setLayout(new BorderLayout());

    pTable = new lasFilesTable(stList);
    scroll = pTable.getScrollPane();

    // Attach widgets to Frame

    this.getContentPane().add(pnlButtons, BorderLayout.SOUTH);
    pnlButtons.add(btnSelect, null);
    pnlButtons.add(btnCancel, null);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(scroll, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(600, 200));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getLAS3()
   * <p> This method will return the LAS 3 Data List Structure
   * @return the LAS 3 Data List Structure
   */

  public las3ListStruct getLAS3() { return (stLAS3); }

  /** Method getLASDataStructure()
   * <p> This method will return the LAS Data Structure
   * @return stLASFileData = the LAS Data Structure
   */

  public lasFileDataStruct getLASDataStructure() { return (stLASFileData); }

  /** Method getStartingDepth()
   * <p> This method will return the starting depth of the LAS File
   * @return dStart = the starting depth of the LAS File
   */

  public double getStartingDepth() { return (dStart); }

  /** Method getEndingDepth()
   * <p> This method will return the ending depth of the LAS File
   * @return dStart = the ending depth of the LAS File
   */

  public double getEndingDepth() { return (dEnd); }

  /** Method getColorlith()
   * <p> This method will return the colorlith identifier from the LAS File
   * @return  iColorlith = Colorlith Identifier
   */

  public int getColorlith() { return (iColorlith); }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method select()
   * <p> This method will select and plot the LAS File
   */

  private void select()
  {
    cmnLASFileStruct st    = pTable.getRow();

    pRead = new las3Read( notifier, stStruct );
    pRead.setPath( "URL", st.sDirectory, st.sFilename );
  }

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    pNotifier  = null;
    stStruct   = null;
    stCMN      = null;

    notifier = null;

    stList     = null;

    if (stLAS3 != null)
      stLAS3.delete();
    stLAS3 = null;

    if (stLASFileData != null)
      stLASFileData.delete();
    stLASFileData = null;   // LAS File Data Struct

    if (pRead != null)
      pRead.delete();
    pRead = null;

    if (pTable != null)
      pTable.close();
    pTable = null;

    btnSelect  = null;
    btnCancel  = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnSelect) { select(); }
    if (event.getSource() == btnCancel) { close(); }
  }

  /* ===================================================================== *
   * ------------------------ OBSERVER ACTIONS --------------------------- *
   * ===================================================================== */

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg = new String( (String) arg);

    if (sArg.equals("LAS 3 - Curves Selected"))
    {
      if (pRead != null)
      {
//        stLAS3        = las3.las3Utility.copyList( pRead.getLAS3Data() );
        stLASFileData = las.lasFileDataUtility.copy( pRead.getLASData() );
        dStart        = stLASFileData.depthStart;
        dEnd          = stLASFileData.depthEnd;
        iColorlith    = stLASFileData.iColorlithDefault;

        pRead.delete();
        pRead = null;
      }

      if (pNotifier != null)
        pNotifier.notifyObservers(new String("KGS LAS File Loaded"));
    }
  }
}

/** CLASS lasFilesFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class lasFilesFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 10/22/2007
 *  @author  John Victorine
 */
