/*
 * @lasConvertFrame.java Version 1.1 03/29/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import las.lasFileDataStruct;

/** Class lasConvertFrame
 *  <p> This Class will display the Curves sections of the LAS 3 File and allow
 *      the user to select the curves they wish to display on the KGS web apps
 *      and to identify unknown curves with KGS Standard Curves.
 *
 *  @version 1.1 03/29/2010
 *  @author  John R. Victorine
 */

public class lasConvertFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable         notifier  = null; // Observables Object
  private lasFileDataStruct  stLAS     = null; // LAS File Data Structure

  // Global Variables

  private int iCurve = iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS;

  private double dMin     = 0.0;
  private double dMax     = 0.0;

  private double dPercMin = 0.0;
  private double dPercMax = 100.0;

  private double dMinNew  = 0.0;
  private double dMaxNew  = 0.0;

  private double dLogMin  = 0.0;
  private double dLogMax  = 0.0;

  // Global Frame Widgets

  // Frame Buttons

  private JButton btnCompute = new JButton();
  private JButton btnCancel  = new JButton();

  // -- From Widgets

  private JTextField txtDataMin  = new JTextField();
  private JTextField txtNewMin   = new JTextField();
  private JSpinner txtSpinnerMin = null;

  private JTextField txtDataMax  = new JTextField();
  private JTextField txtNewMax   = new JTextField();
  private JSpinner txtSpinnerMax = null;

  // -- To Log Widgets

  private JTextField txtLogMin   = new JTextField();
  private JTextField txtLogMax   = new JTextField();

  // Title Borders

  private JPanel pnlFrom       = new JPanel();
  private JPanel pnlTo         = new JPanel();

  private TitledBorder titledBorderCounts = null;
  private TitledBorder titledBorderLog    = null;

  /** Constructor lasConvertFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public lasConvertFrame( Observable notifier )
  {
    try
    {
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlBase       = new JPanel();
    JPanel pnlCenter     = new JPanel();

    JPanel pnlMin        = new JPanel();
    JPanel pnlDataMin    = new JPanel();
    JPanel pnlSpinnerMin = new JPanel();
    JPanel pnlNewMin     = new JPanel();

    JPanel pnlMax        = new JPanel();
    JPanel pnlDataMax    = new JPanel();
    JPanel pnlSpinnerMax = new JPanel();
    JPanel pnlNewMax     = new JPanel();

    JPanel pnlLogMin     = new JPanel();
    JPanel pnlLogMax     = new JPanel();

    titledBorderCounts = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "Neutron Counts");

    titledBorderLog = new TitledBorder(
        new EtchedBorder( EtchedBorder.RAISED,
                          Color.white,
                          new Color(165, 163, 151) ),
        "Neutron Porosity");

    TitledBorder titledBorderLogMin = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Minimum:");

    TitledBorder titledBorderLogMax = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Maximum:");

    titledBorderCounts.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderLog.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderLogMin.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderLogMax.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Convert");

    // Spinner Action

    ChangeListener listener = new ChangeListener()
    {
      public void stateChanged(ChangeEvent e)
      {
        SpinnerModel source = (SpinnerModel)e.getSource();
        double       dTemp  = 0.0;
        String       sTemp  = "";
        int          iTemp  = 0;

        if (cmn.cmnString.stringToDouble(source.getValue().toString()) < 51.0)
        {
          sTemp    = txtSpinnerMin.getValue().toString();
          if (cmn.cmnString.isNumeric( sTemp ))
            dPercMin = cmn.cmnString.stringToDouble( sTemp );
          else
            txtSpinnerMin.setValue(new Double(dPercMin));

          iTemp = (int) ( (dMin + dMax * dPercMin/100.0) * 1000.0 );
          dTemp = (double) iTemp / 1000.0;
          dMinNew = dTemp;
          txtNewMin.setText( "" + dTemp );
        }

        if (cmn.cmnString.stringToDouble(source.getValue().toString()) > 50.0)
        {
          sTemp    = txtSpinnerMax.getValue().toString();
          if (cmn.cmnString.isNumeric( sTemp ))
            dPercMax = cmn.cmnString.stringToDouble( sTemp );
          else
            txtSpinnerMax.setValue(new Double(dPercMax));

          iTemp = (int) ( (dMax * dPercMax / 100.0) * 1000.0 );
          dTemp = (double) iTemp / 1000.0;
          dMaxNew = dTemp;
          txtNewMax.setText( "" + dTemp );
        }
      }
    };

    // Frame Buttons Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnCompute.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCompute.setText("Compute");
    btnCompute.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setPreferredSize(new Dimension(85, 25));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // Center Panel

    pnlCenter.setLayout(new BorderLayout());

    // -- From Panel

    pnlFrom.setBorder(titledBorderCounts);
    pnlFrom.setLayout(new GridLayout());

    // .. Minimum Panel

    pnlMin.setLayout(new GridLayout(3,1));
    pnlMin.setBorder(titledBorderLogMin);

    pnlDataMin.setLayout(new BorderLayout());

    txtDataMin.setEditable(false);
    txtDataMin.setText("0.0");
    txtDataMin.setHorizontalAlignment(SwingConstants.TRAILING);

    pnlSpinnerMin.setLayout(new BorderLayout());

    SpinnerNumberModel modelRMin =
        new SpinnerNumberModel(dPercMin, 0.0, 50.0, 1.0);
    txtSpinnerMin = new JSpinner(modelRMin);
    modelRMin.addChangeListener(listener);

    pnlNewMin.setLayout(new BorderLayout());

    txtNewMin.setEditable(false);
    txtNewMin.setText("0.0");
    txtNewMin.setHorizontalAlignment(SwingConstants.TRAILING);

    // .. Maximum Panel

    pnlMax.setLayout(new GridLayout(3,1));
    pnlMax.setBorder(titledBorderLogMax);

    pnlDataMax.setLayout(new BorderLayout());

    txtDataMax.setEditable(false);
    txtDataMax.setText("0.0");
    txtDataMax.setHorizontalAlignment(SwingConstants.TRAILING);

    pnlSpinnerMax.setLayout(new BorderLayout());

    SpinnerNumberModel modelRMax =
        new SpinnerNumberModel(dPercMax, 51.0, 100.0, 1.0);
    txtSpinnerMax = new JSpinner(modelRMax);
    modelRMax.addChangeListener(listener);

    pnlNewMax.setLayout(new BorderLayout());

    txtNewMax.setEditable(false);
    txtNewMax.setText("0.0");
    txtNewMax.setHorizontalAlignment(SwingConstants.TRAILING);

    // -- To Panel

    pnlTo.setBorder(titledBorderLog);
    pnlTo.setPreferredSize(new Dimension(150, 27));
    pnlTo.setLayout(new GridLayout(2,1));

    // .. Log Minimum Panel

    pnlLogMin.setBorder(titledBorderLogMin);
    pnlLogMin.setLayout(new BorderLayout());

    txtLogMin.setText("0.0");
    txtLogMin.setHorizontalAlignment(SwingConstants.TRAILING);
    txtLogMin.addFocusListener(new lasConvertFrameFocusAdapter(this));

    // .. Log Maximum Panel

    pnlLogMax.setBorder(titledBorderLogMax);
    pnlLogMax.setLayout(new BorderLayout());

    txtLogMax.setText("0.0");
    txtLogMax.setHorizontalAlignment(SwingConstants.TRAILING);
    txtLogMax.addFocusListener(new lasConvertFrameFocusAdapter(this));

    this.getContentPane().add(pnlBase, BorderLayout.SOUTH);
    pnlBase.add(btnCompute,            null);
    pnlBase.add(btnCancel,             null);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(pnlFrom,  BorderLayout.CENTER);
    pnlFrom.add(pnlMin, null);
    pnlMin.add(pnlDataMin, null);
    pnlDataMin.add(txtDataMin, BorderLayout.CENTER);

    pnlMin.add(pnlSpinnerMin, null);
    pnlSpinnerMin.add(txtSpinnerMin, BorderLayout.CENTER);

    pnlMin.add(pnlNewMin, null);
    pnlNewMin.add(txtNewMin, BorderLayout.CENTER);

    pnlFrom.add(pnlMax, null);
    pnlMax.add(pnlDataMax, null);
    pnlDataMax.add(txtDataMax, BorderLayout.CENTER);

    pnlMax.add(pnlSpinnerMax, null);
    pnlSpinnerMax.add(txtSpinnerMax, BorderLayout.CENTER);

    pnlMax.add(pnlNewMax, null);
    pnlNewMax.add(txtNewMax, BorderLayout.CENTER);

    pnlCenter.add(pnlTo,  BorderLayout.EAST);
    pnlTo.add(pnlLogMin, null);
    pnlLogMin.add(txtLogMin, BorderLayout.CENTER);

    pnlTo.add(pnlLogMax, null);
    pnlLogMax.add(txtLogMax, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(450, 200));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
//    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @return stLAS = the LAS File Data Structure
   */

  public lasFileDataStruct getLAS()
  {
    if (stLAS != null)
    {
      switch (iCurve)
      {
        case iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS:
          // GR Counts Computed Minimum & Maximum

          stLAS.dGRNMin = dPercMin;
          stLAS.dGRNMax = dPercMax;

          // GR Compute Minimum & Maximum

          stLAS.dGRCMin = cmn.cmnString.stringToDouble( txtLogMin.getText() );
          stLAS.dGRCMax = cmn.cmnString.stringToDouble( txtLogMax.getText() );

          // Compute GR from GR Counts

          stLAS.dGR = math.mathLAS.computeGR( stLAS.dGRN,    stLAS.dNull,
                                              dMaxNew,       dMinNew,
                                              stLAS.dGRCMax, stLAS.dGRCMin );
          break;
        case iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS:
          // Neutron Counts Computed Minimum & Maximum

          stLAS.dNEUTMin = dPercMin;
          stLAS.dNEUTMax = dPercMax;

          // NPHI Compute Minimum & Maximum

          stLAS.dNPHICMin = cmn.cmnString.stringToDouble( txtLogMin.getText() );
          stLAS.dNPHICMax = cmn.cmnString.stringToDouble( txtLogMax.getText() );

          // Compute NPHI from Neutron Counts

          stLAS.dNPHI = math.mathLAS.computeNPHI(
                          stLAS.dNEUT,     stLAS.dNull,
                          dMaxNew, dMinNew,
                          stLAS.dNPHICMax, stLAS.dNPHICMin );
          break;
      }
    }

    dPercMin = 0.0;
    txtSpinnerMin.setValue(new Double(dPercMin));

    dPercMax = 100.0;
    txtSpinnerMin.setValue(new Double(dPercMax));

    return (stLAS);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setLAS()
   * <p> This method will set the LAS File Data Structure
   * @param st = the LAS File Data Structure
   */

  public void setLAS( lasFileDataStruct st ) { this.stLAS = st; }

  /** Method setCurve()
   * <p> This method will load the selected Track data
   * @param iCurve = the selected track data
   */

  public void setCurve( int iCurve )
  {
    double dRange[] = { 0.0, 0.0 };
    int    iTemp    = 0;
    double dTemp    = 0.0;

    this.iCurve = iCurve;

    switch (iCurve)
    {
      case iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS:
        dRange = math.mathLAS.computeCurveMinMax( stLAS.dGRN, stLAS.dNull );
        dMin = dRange[0];
        dMax = dRange[1];

        dPercMin = stLAS.dGRNMin;
        dPercMax = stLAS.dGRNMax;

        dLogMin = stLAS.dGRCMin;
        dLogMax = stLAS.dGRCMax;

        titledBorderCounts.setTitle("Gamma Ray Counts");
        titledBorderLog.setTitle("Gamma Ray");
        break;
      case iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS:
        dRange = math.mathLAS.computeCurveMinMax( stLAS.dNEUT, stLAS.dNull );
        dMin = dRange[0];
        dMax = dRange[1];

        dPercMin = stLAS.dNEUTMin;
        dPercMax = stLAS.dNEUTMax;

        dLogMin = stLAS.dNPHICMin;
        dLogMax = stLAS.dNPHICMax;

        titledBorderCounts.setTitle("Neutron Counts");
        titledBorderLog.setTitle("Neutron Porosity");
        break;
      default:
        dMin = 0.0;
        dMax = 10.0;

        txtLogMin.setText( "" + 0.0 );
        txtLogMax.setText( "" + 10.0 );

        titledBorderCounts.setTitle("");
        titledBorderLog.setTitle("");
        break;
    }

    pnlFrom.updateUI();
    pnlTo.updateUI();

    txtDataMin.setText( "" + dMin );
    txtSpinnerMin.setValue(new Double(dPercMin));

    iTemp = (int) ( (dMin + dMax * dPercMin/100.0) * 1000.0 );
    dTemp = (double) iTemp / 1000.0;
    dMinNew = dTemp;
    txtNewMin.setText( "" + dTemp );

    txtDataMax.setText( "" + dMax );
    txtSpinnerMax.setValue(new Double(dPercMax));

    iTemp = (int) ( (dMax * dPercMax / 100.0) * 1000.0 );
    dTemp = (double) iTemp / 1000.0;
    dMaxNew = dTemp;
    txtNewMax.setText( "" + dTemp );

    txtLogMin.setText( "" + dLogMin );
    txtLogMax.setText( "" + dLogMax );
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    // Input Variables

    notifier  = null; // Observables Object

    if (stLAS != null)
      stLAS.delete();
    stLAS     = null; // LAS File Data Structure

    // Global Frame Widgets

    // Frame Buttons

    btnCompute    = null;
    btnCancel     = null;

    // -- From Widgets

    txtDataMin    = null;
    txtNewMin     = null;
    txtSpinnerMin = null;

    txtDataMax    = null;
    txtNewMax     = null;
    txtSpinnerMax = null;

    // -- To Log Widgets

    txtLogMin     = null;
    txtLogMax     = null;

    // Title Borders

    pnlFrom            = null;
    pnlTo              = null;

    titledBorderCounts = null;
    titledBorderLog    = null;

    this.dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() ==  btnCompute)
    {
      if (notifier != null)
        notifier.notifyObservers(new String("Log Values Converted"));
    }
    if (event.getSource() ==  btnCancel) { this.setVisible(false);}
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    // -- KGS - Township Range Section Search Text Fields

    if (e.getSource() == txtLogMin)
    {
      iNumeric = 1;
      sTemp = txtLogMin.getText();
      sMessage = new String("Minimum Log Value is a Numeric Field");
    }

    if (e.getSource() == txtLogMax)
    {
      iNumeric = 1;
      sTemp = txtLogMax.getText();
      sMessage = new String("Maximum Log Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtLogMin)
          dLogMin = cmn.cmnString.stringToDouble(txtLogMin.getText());
        if (e.getSource() == txtLogMax)
          dLogMax = cmn.cmnString.stringToDouble(txtLogMax.getText());
      }
    }
  }
}

/** CLASS lasConvertFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class lasConvertFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  lasConvertFrame adaptee;

  lasConvertFrameFocusAdapter(lasConvertFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 03/29/2010
 *  @author  John Victorine
 */
