/*
 * @lasControlPlotFrame.java Version 1.1 06/06/2014
 *
 * Copyright (c) 2014 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package las.gui;

import java.net.*;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;
import java.util.Observer;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import cmn.cmnLASFileListStruct;             // LAS File Information List Structure

import util.utilWaitAnimation;               // Wait Animation Dialog
import util.SwingWorker;                     // Swing Worker

import gui.guiSearchDirectoryFrame;          // File Search Dialog
import gui.guiMemoryFrame;                   // Available Memory Dialog

import iqstrat.iqstratStruct;                // Global IQSTRAT Data Structure
import iqstrat.iqstratControlListStruct;     // IQSTRAT Control List Data Structure
import iqstrat.iqstratControlStruct;         // IQSTRAT Control Data Structure
import iqstrat.iqstratHeadersStruct;         // IQSTRAT Headers Data Structure
import iqstrat.iqstratShaleListStruct;       // IQSTRAT Shale List Data Structure
import iqstrat.iqstratRemarkListStruct;      // IQSTRAT Georeport List Data Structure
import iqstrat.gui.iqstratHeadersFrame;      // Edit Headers Data Dialog
import iqstrat.gui.iqstratTracksTable;       // User Define Tracks Table
import iqstrat.gui.iqstratPlotFrame;         // Profile Plot Frame

import las.lasStruct;                        // Global LAS Data Structure
import las.lasFileDataStruct;                // LAS File Data Structure
import las.gui.lasThinTrackFrame;            // Thin Track Curve Selection Dialog
import las.gui.lasConvertFrame;              // Convert from Counts to API Dialog
import las.gui.lasAvailablePanel;            // Available Plot Tracks Dialog
import las.plot.lasPlotLimitsFrame;          // LAS Plot Limits Frame

import las3.las3ListStruct;                  // LAS 3.0 List Data Structure
import las3.gui.las3FileFrame;               // LAS 3.0 Save

import horizon.regions.regionsListStruct;    // Perforations List Data Structure
import horizon.strat.stratListStruct;        // Stratigraphic Units List Data Structure
import horizon.seq.seqListStruct;            // Sequence Stratigrapy List Data Structure
import horizon.gui.horizonEnterDataFrame;    // Horizon Data Entry Dialog

import horizon.env.envListStruct;            // Depositional Environment List Structure
import horizon.bio.bioStratListStruct;       // Biostratigraphy List Data Structure

import brine.brineListStruct;                // Brine Data List Structure

import rock.rockFileDataStruct;              // Rock File Data Structure
import rock.rockDataListStruct;              // Measured Rock Data List Data Strucuture
import rock.rockImagesListStruct;            // Rock Images List Data Structure
import rock.fossil.fossilListStruct;         // General Fossil List Data Structure
import rock.sedimentary.sedimentaryListStruct;  // Sedimentary Structure List Structure
import rock.phi.phiListStruct;               // Porosity List Data Structure
import rock.color.rockColorListStruct;       // Rock Color List Data Structure
import rock.plot.rockPlotLimitsFrame;        // Rock Plot Limits Frame

import lith.lithology.lithologyListStruct;   // Lithology List Data Structure
import lith.texture.textureListStruct;       // Texture List Data Structure
import lith.gui.lithImageTrackFrame;         // Image Track Curve Selection Dialog
import lith.rock.rockColumnListStruct;       // Rock Lithology Column List Structure

import mask.maskSymbolsStruct;               // Plot Symbol Mask Data Structure

import dst.dstListStruct;                    // DST Data List Structure
import dst.dstStruct;                        // DST Data Structure

import pfeffer.pfefferDataListStruct;        // PfEFFER List Data Structure
import pfeffer.pfefferDataStruct;            // PfEFFER Data Structure

/** Class lasControlPlotFrame
 *  <p> This Class will create the Lithology Control Plot Frame.
 *
 *  @version 1.1 06/06/2014
 *  @author  John R. Victorine
 */

public class lasControlPlotFrame extends JFrame implements ActionListener, Observer
{
  private Observable          notifier = null; // Observalbles Object
  private int                 iAction  = -1;

  // Variables passed to this panel

  private JFrame               parent   = null; // Frame calling this frame
  private iqstratStruct        stStruct = null; // Global PROFILE Data Structure
  private int                  iSource  = iqstrat.io.iqstratIOHeaders.USER_PC;
  private iqstratHeadersStruct stHeader = null; // Headers data structure

  // Type of data

  public static final int _HEADERS     = 0;  // Headers Data

  // -- LAS File & Geological Data

  public static final int _LAS3        = 1;  // LAS File Data
  public static final int _LAS         = 2;  // LAS File Data
  public static final int _ROCK        = 3;  // Rock Data
  public static final int _ROCK_IMAGES = 4;  // Rock Data

  // -- Horizon Data

  public static final int _TOPS        = 5;  // Tops Data
  public static final int _SEQ_STRAT   = 6;  // Sequence Stratigraphy
  public static final int _PERFORATION = 7;  // Perforations

  public static final int _DEP_ENV     = 19; // Depositional Environment
  public static final int _BIO_STRAT   = 18; // Bio Stratigraphy

  // -- Brine Data

  public static final int _BRINE       = 17; // Brine Data

  // -- Rock Observational & Computed Data

  public static final int _ROCK_COLUMN = 20; // Rock Column Data

  public static final int _LITHOLOGY   = 8;  // Rock Lithology
  public static final int _TEXTURE     = 9;  // Rock Texture
  public static final int _ROCK_PHI    = 10; // Rock Porosity
  public static final int _SED_STRUCT  = 11; // Sedimentary Structure
  public static final int _FOSSILS     = 12; // Fossils
  public static final int _ROCK_COLOR  = 13; // Rock Color
  public static final int _REMARKS     = 14; // Remarks/Comments

  // -- Misc Computation Type Tracks

  public static final int _PFEFFER     = 15;  // Perforations
  public static final int _SHALE       = 16;  // Shale Gamma Ray Levels

  // Track Panel Displayed

  private static final int _DEFAULT = 0;
  private static final int _USER    = 1;

  private int iPanel = _DEFAULT;

  private int iConfig = iqstrat.iqstratControlStruct._C_ALL;

  // Depth Range of Data (initial value)

  private double dStart     = 0.0;
  private double dEnd       = 2000.0;

  private double dStartOrig = 0.0;
  private double dEndOrig   = 2000.0;

  // LAS Track Plot Size

  private int iLASSize = las.gui.lasAvailablePanel._DOUBLE;

  // Load Panel Identifiers

  private static final int _PC           = 0;
  private static final int _KGS          = 1;
  private static final int _NOTES        = 2;
  private static final int _STRAT_UNIT   = 3;
  private static final int _LAS_3_FILE   = 4;
  private static final int _ROCK_ENTRY   = 5;
  private static final int _TOPS_ENTRY   = 6;
  private static final int _CONVERT      = 7;

  private static final int _MEAS_SECT    = 8;
  private static final int _GEO_REPORT   = 9;

  private static final int _GR_SHALE     = 10;
  private static final int _XSECTION     = 11;
  private static final int _ROCK_DATA    = 12;
  private static final int _PERF         = 13;
  private static final int _BRINE_DATA   = 14;
  private static final int _DEP_ENV_DATA = 15;
  private static final int _BIO_ENTRY    = 16;

  // Display Profile Plot Identifier

  private static final int _HTML = 0;
  private static final int _PDF  = 1;

  private int iDocument = _HTML;

  // Load Profile Data Frames

  private las3FileFrame p3File = null; // LAS 3 File Data Frame

  // Depth Scale Buttons

  private int iScale = iqstrat.iqstratTracksStruct.SCALE_100;

  private int  iStatus[]   = null; // Status of each button
  private int  iSelected[] = null; // Selected Tracks

  // Profile Plot Limits

  private static final int _C_LAS     = iqstrat.iqstratControlStruct._C_LAS;
  private static final int _C_CORE    = iqstrat.iqstratControlStruct._C_CORE;
  private static final int _C_PFEFFER = iqstrat.iqstratControlStruct._C_PFEFFER;

  private int    iData         = _C_LAS;

  private double orig_las[][]  = null; // Original Plot Limits for LAS
  private double orig_pf[][]   = null; // Original Plot Limits for PfEFFER
  private double orig_core[][] = null; // Original Plot Limits for CORE

  private double data_las[][]  = null; // Plot Limits for LAS
  private double data_pf[][]   = null; // Plot Limits for PfEFFER
  private double data_core[][] = null; // Plot Limits for CORE

  // Plot Screen Location & Size

  private int iFrameX      = -1;    // Plot Frame X-Coordintate
  private int iFrameY      = -1;    // Plot Frame Y-Coordintate
  private int iFrameWidth  = -1;    // Plot Frame Width
  private int iFrameHeight = -1;    // Plot Frame Height

  // Screen Width

  private int iControlWidth = 325; //275;  // Control Panel Width
  private int iScreenWidth  = 0;    // Screen Width
  private int iScreenHigh   = 0;    // Scree Hieght

  // Available Tracks Panels

  private lasAvailablePanel       pnlDefaultList = null; // Available Tracks
  private iqstratTracksTable      pTracks        = null; // Modify Order

  // Profile Plot Frame

  private iqstratPlotFrame        plotframe      = null; // Plot Frame

  // Data Entry Frames

  private guiSearchDirectoryFrame pSearch        = null; // Search for Directory
  private guiMemoryFrame          pMemory        = null; // Available Memory Dialog

  private iqstratHeadersFrame     pHeaders       = null; // Headers Info data Frame

  private horizonEnterDataFrame   pTopsEntry     = null; // Horizon Data Entry

  private lasThinTrackFrame       pnlThinTrack   = null; // Thin Track Data Entry
  private lithImageTrackFrame     pnlImageTrack  = null; // Image Track Data Entry

  private lasPlotLimitsFrame      pLimitsLAS     = null; // LAS Plot Track Limits Dialog
  private rockPlotLimitsFrame     pLimitsCORE    = null; // CORE Plot Track Limits Dialog

  // -- Location of Data Track

  public static final int _EDIT_PANEL = 0;
  public static final int _AVAILABLE  = 1;

  // -- Porosity Thin Track

  public int    iThin    = cmn.cmnStruct._NO;
  public int    iTHINPHI = iqstrat.iqstratTracksStruct._THIN_NPHI;

  // Image Track Identifier

  private static final int _PHI_IMAGE = 0;
  private static final int _OHM_IMAGE = 1;

  // -- Porosity Image Track

  public int    iPHI     = cmn.cmnStruct._NO;
  public double dPHIMin  = 0.0;
  public double dPHIMax  = 0.3;
  public int    iPHITool = las.lasStandardTools._NPHI;

  // -- Resistivity Image Track

  public int    iOHM     = cmn.cmnStruct._NO;
  public double dOHMMin  = 0.0;
  public double dOHMMax  = 500.0;
  public int    iOHMTool = las.lasStandardTools._ILD;

  // Non-Linear Image Track Data

  private double dRt[][]   = null;
  private double dPhit[][] = null;
  private double dPhi[][]  = null;

  // Change from counts to log value

  private lasConvertFrame pConvert = null;

  // LAS File Data

  private iqstratControlStruct    stControl       = new iqstratControlStruct();

  private cmnLASFileListStruct    stFiles         = null; // LAS Files Information

  private las3ListStruct          stLAS3Data      = null; // LAS 3 List Structure
  private lasFileDataStruct       stLASFileData   = null; // LAS File Data Struct
  private lithologyListStruct     stLASLithology  = null; // LAS Rock Column
  private lithologyListStruct     stRTLithology   = null; // LAS Rock Column by Density-Sonic Logs
  private textureListStruct       stLASTexture    = null; // Rock Texture
  private int                     iColorlith      = 0;    // Colorlith setting

  private iqstratShaleListStruct  stShale         = null; // Shale List Structure

  // Horizons Data

  private regionsListStruct       stTops          = null; // Stratigraphic Tops
  private stratListStruct         stStrat         = null; // Stratigraphic Units List
  private seqListStruct           stSequence      = null; // Sequence List Data Struct
  private regionsListStruct       stOther         = null; // DST & Perforations

  private envListStruct           stDepEnv        = null; // Depositional Environment
  private bioStratListStruct      stBio           = null; // Bio Stratigraphy Data

  // .. DST Data

  private dstListStruct           stDST           = null; // DST List Structure

  // Brine Data

  private brineListStruct         stBrine         = null; // Brine Data

  // Rock Data

  private rockDataListStruct      stCore          = null; // Measured Rock Data
  private rockImagesListStruct    stImages        = null; // Rock Image Data
  private lithologyListStruct     stRockLithology = null; // Rock Column
  private lithologyListStruct     stPlotLithology = null; // Rock Column (Plot)
  private textureListStruct       stTexture       = null; // Rock Texture
  private textureListStruct       stPlotTexture   = null; // Rock Texture (Plot)
  private phiListStruct           stPHI           = null; // Porosity type struct
  private fossilListStruct        stFossil        = null; // Fossil List struct
  private sedimentaryListStruct   stSedimentary   = null; // Sed Struct Data
  private rockColorListStruct     stRockColor     = null; // Rock Color Data
  private iqstratRemarkListStruct stRemarks       = null; // Remarks Data

  private rockColumnListStruct    stColumn        = null; // Rock Column Data List

  private rockFileDataStruct      stRockData      = null; // Rock Data Structure

  private pfefferDataStruct       stPfeffer       = null; // General Pfeffer Data
  private pfefferDataListStruct   stFlowUnits     = null; // Flow Units

  // Frame Widgets

  // Mouse Cursor Depth Indicator

  private JTextField txtCursorDepth = new JTextField();

  // Depth Scale Buttons

  private JLabel lblScale = new JLabel();

  // Depth Range Textfields

  private JTextField txtStart = new JTextField();
  private JTextField txtEnd   = new JTextField();

  // Depth Range Buttons

  private static final String BTN_RESET  = "Reset Depth";
  private static final String BTN_MODIFY = "Modify Depth";

  private JButton btnReset  = new JButton();
  private JButton btnModify = new JButton();

  // Well Information Plot Data

  private JLabel lblName   = new JLabel();
  private JLabel lblAPI    = new JLabel();
  private JLabel lblStatus = new JLabel();
  private JLabel lblLat    = new JLabel();
  private JLabel lblLong   = new JLabel();
  private JLabel lblTD     = new JLabel();
  private JLabel lblElev   = new JLabel();

  private JButton btnHeader = new JButton();

  // Plot Track Buttons

  private static final String BTN_DEFAULT = "Default";
  private static final String BTN_USER    = "User";
  private static final String BTN_CREATE  = "Create User Defined";

  private JButton btnDefault = new JButton();
  private JButton btnUser    = new JButton();

  private JButton btnUp      = new JButton();
  private JButton btnDown    = new JButton();

  // Change Plot Limits Button

//  private JButton btnLimits  = new JButton();

  private JButton btnLimitsLAS  = new JButton();
  private JButton btnLimitsCORE = new JButton();

  // Labels

  private static final String LBL_START_DEPTH  = "Start Depth:";
  private static final String LBL_CURSOR_DEPTH = "Cursor:";
  private static final String LBL_TRACKS       = "Stratigraphic Plot Tracks:";
  private static final String LBL_SCALE_RANGE  = "Depth Scale & Range:";
  private static final String LBL_END_DEPTH    = "End Depth:";
  private static final String LBL_TITLE        = "Control";
  private static final String LBL_NAME         = "Name:";
  private static final String LBL_TRACK_BTNS   =
      "Show Available Tracks by Type of Data:";

  // Available Track Widgets

  private JPanel     pnlPlotTracks = new JPanel();
  private CardLayout cardLayout    = new CardLayout();

  // Menus

  // File Button Menu

  private JMenuItem mSave    = null;  // Save Profile Data
  private JMenuItem mPDF     = null;  // Create a PDF of Profile Plot
  private JMenuItem memory   = null;  // Display Available Memory Dialog
  private JMenuItem mExit    = null;  // Exit Profile Plot

  // Help Button Menu

  private JMenuItem mHelp  = null;

  // Depth Scale Menu

  private static final String scale[] =
  {
      "1 ft/in",   "2 ft/in",  "5 ft/in",
     "10 ft/in",  "20 ft/in", "50 ft/in",
    "100 ft/in", "200 ft/in"
  };

  private JRadioButtonMenuItem  m1   = null;  // Display Plot at   1 ft / in
  private JRadioButtonMenuItem  m2   = null;  // Display Plot at   2 ft / in
  private JRadioButtonMenuItem  m5   = null;  // Display Plot at   5 ft / in
  private JRadioButtonMenuItem  m10  = null;  // Display Plot at  10 ft / in
  private JRadioButtonMenuItem  m20  = null;  // Display Plot at  20 ft / in
  private JRadioButtonMenuItem  m50  = null;  // Display Plot at  50 ft / in
  private JRadioButtonMenuItem  m100 = null;  // Display Plot at 100 ft / in
  private JRadioButtonMenuItem  m200 = null;  // Display Plot at 200 ft / in

  /** Constructor lasControlPlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param parent     = the frame calling this panel
   *  @param stStruct   = Global Color Lithology Data Structure
   */

  public lasControlPlotFrame(JFrame parent, iqstratStruct stStruct)
  {
    try
    {
      this.parent   = parent;
      this.stStruct = stStruct;

      dStart     = stStruct.depthStart;
      dEnd       = stStruct.depthEnd;

      dStartOrig = stStruct.depthStart;
      dEndOrig   = stStruct.depthEnd;

      jbInit();

      addWindowListener( new lasControlPlotFrame_WindowListener() );

      setInitialConfiguration( iConfig );

      plot();  // Display the plot panel
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor lasControlPlotFrame()
   *  <p> This is the Constructor for this class.
   *  @param stStruct   = Global Color Lithology Data Structure
   *  @param iAction    = Identifier for the type of data to display
   */

  public lasControlPlotFrame( iqstratStruct stStruct, int iAction )
  {
    try
    {
      this.stStruct = stStruct;
      this.iAction  = iAction;

      dStart     = stStruct.depthStart;
      dEnd       = stStruct.depthEnd;

      dStartOrig = stStruct.depthStart;
      dEndOrig   = stStruct.depthEnd;

      jbInit();

      addWindowListener( new lasControlPlotFrame_WindowListener() );

      setInitialConfiguration( iConfig );

      plot();  // Display the plot panel
    }
    catch (Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlTop          = new JPanel();
    JPanel pnlButtons      = new JPanel();
    JPanel pnlCenter       = new JPanel();
    JPanel pnlDepthControl = new JPanel();
    JPanel pnlScale        = new JPanel();
    JPanel pnlDepthRange   = new JPanel();
    JPanel pnlDepthButtons = new JPanel();
    JPanel pnlDepths       = new JPanel();
    JPanel pnlStart        = new JPanel();
    JPanel pnlEnd          = new JPanel();
    JPanel pnlTracks       = new JPanel();
    JPanel pnlTracksBTNS   = new JPanel();
    JPanel pnlTrackButtons = new JPanel();
    JPanel pnlTrackPref    = new JPanel();
    JPanel pnlSelect       = new JPanel();
    JPanel pnlMouseDepth   = new JPanel();
    JPanel pnlDefault      = new JPanel();
    JPanel pnlUser         = new JPanel();
    JPanel pnlIcons        = new JPanel();

    JPanel pnlModifyOrder  = new JPanel();
    JPanel pnlTrackButton  = new JPanel();
    JPanel pnlTrackList    = new JPanel();

    JPanel pnlHeader        = new JPanel();
    JPanel pnlInfo          = new JPanel();
    JPanel pnlInfoPosition  = new JPanel();
    JPanel pnlInfoXY        = new JPanel();
    JPanel pnlInfoZ         = new JPanel();

    JPanel pnlID            = new JPanel();
    JPanel pnlName          = new JPanel();
    JPanel pnlStatus        = new JPanel();

    JScrollPane scrollPref = new JScrollPane();
    JScrollPane scrollTrackOrder = new JScrollPane();

    JMenuBar menuBar    = new JMenuBar();
    JMenu    menuFile   = new JMenu("File");
    JMenu    menuScale  = new JMenu("Depth Scale");
    JMenu    menuMisc   = new JMenu("Data Entry Dialog");
    JMenu    menuHelp   = new JMenu("Help");
    ButtonGroup groupDepth = new ButtonGroup();

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white, new Color(165, 163, 151)),
        LBL_START_DEPTH);
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderCursor = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        LBL_CURSOR_DEPTH);
    titledBorderCursor.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderTracks = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        LBL_TRACKS);
    titledBorderTracks.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderDepth = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        LBL_SCALE_RANGE);
    titledBorderDepth.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white, new Color(165, 163, 151)),
        LBL_END_DEPTH);
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderName = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(148, 145, 140)),
        LBL_NAME);
    titledBorderName.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderAvailTracks = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(148, 145, 140)),
        LBL_TRACK_BTNS);
    titledBorderAvailTracks.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderOrder = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Modify Track Order");
    titledBorderOrder.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderWell = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Header Information:");
    titledBorderWell.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderLimits = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white, new Color(165, 163, 151)),
        "Change Log & Core Plot Limits");
    titledBorderLimits.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Menus

    // Menu Bar

    mSave = new JMenuItem("Save Data");
    mSave.addActionListener(this);

    mPDF = new JMenuItem("Create PDF Document of Profile Plot");
    mPDF.addActionListener(this);

    memory = new JMenuItem("Display Available Memory Dialog");
    memory.addActionListener(this);

    mExit = new JMenuItem("Exit");
    mExit.addActionListener(this);

    menuBar.setFont(new java.awt.Font("Dialog", 1, 11));
    menuFile.setFont(new java.awt.Font("Dialog", 1, 12));
    menuScale.setFont(new java.awt.Font("Dialog", 1, 12));
    menuMisc.setFont(new java.awt.Font("Dialog", 1, 12));
    menuHelp.setFont(new java.awt.Font("Dialog", 1, 12));

    menuFile.add( mSave );
    menuFile.addSeparator();
    menuFile.add( mPDF );
    menuFile.addSeparator();
    menuFile.add( memory );
    menuFile.addSeparator();
    menuFile.addSeparator();
    menuFile.add( mExit );

    // Help & Legends Button Menu

    mHelp = new JMenuItem("Help");
    mHelp.addActionListener(this);
    menuHelp.add(mHelp);

    // Depth Scale Menu Items

    m1   = new JRadioButtonMenuItem("1 ft / in");
    m1.setHorizontalTextPosition(JMenuItem.RIGHT);
    m1.addActionListener(this);
    groupDepth.add(m1);

    m2   = new JRadioButtonMenuItem("2 ft / in");
    m2.setHorizontalTextPosition(JMenuItem.RIGHT);
    m2.addActionListener(this);
    groupDepth.add(m2);

    m5   = new JRadioButtonMenuItem("5 ft / in");
    m5.setHorizontalTextPosition(JMenuItem.RIGHT);
    m5.addActionListener(this);
    groupDepth.add(m5);

    m10  = new JRadioButtonMenuItem("10 ft / in");
    m10.setHorizontalTextPosition(JMenuItem.RIGHT);
    m10.addActionListener(this);
    groupDepth.add(m10);

    m20  = new JRadioButtonMenuItem("20 ft / in");
    m20.setHorizontalTextPosition(JMenuItem.RIGHT);
    m20.addActionListener(this);
    groupDepth.add(m20);

    m50  = new JRadioButtonMenuItem("50 ft / in");
    m50.setHorizontalTextPosition(JMenuItem.RIGHT);
    m50.addActionListener(this);
    groupDepth.add(m50);

    m100 = new JRadioButtonMenuItem("100 ft / in");
    m100.setSelected(true);
    m100.setHorizontalTextPosition(JMenuItem.RIGHT);
    m100.addActionListener(this);
    groupDepth.add(m100);

    m200 = new JRadioButtonMenuItem("200 ft / in");
    m200.setHorizontalTextPosition(JMenuItem.RIGHT);
    m200.addActionListener(this);
    groupDepth.add(m200);

    menuScale.add(m1);
    menuScale.add(m2);
    menuScale.add(m5);
    menuScale.add(m10);
    menuScale.add(m20);
    menuScale.add(m50);
    menuScale.add(m100);
    menuScale.add(m200);

    menuBar.add(menuFile);
    menuBar.add(menuScale);
    menuBar.add(menuHelp);

    this.setJMenuBar(menuBar);

    // Build Other Widgets

    this.setTitle(LBL_TITLE);
    this.getContentPane().setLayout(new BorderLayout());

    // Create an Observable

    notifier = new lasControlPlotFrameObservable();

    // Add this Dialog to the notifier as observer

    notifier.addObserver(this);

    // Panel Top

    pnlTop.setLayout(new BorderLayout());
    pnlTop.setPreferredSize(new Dimension(10, 120));

    // -- Header Information Plot Labels, & Cross Section Depth Range Panels

    pnlHeader.setLayout(new BorderLayout());

    // ---- Header Information Plot Labels Panel

    pnlInfo.setBorder(titledBorderWell);
    pnlInfo.setLayout(new BorderLayout());

    // .. Well Information Panel

    pnlID.setLayout(new GridLayout(2,1));

    // .... Well Name Panel

    pnlName.setLayout(new BorderLayout());

    lblName.setText("Name: ");

    // .... Well API-Number & Well Status Panel

    pnlStatus.setLayout(new GridLayout());

    lblAPI.setText("API: ");
    lblStatus.setText("Status:");

    // .. Well Position Panel

    pnlInfoPosition.setLayout(new BorderLayout());

    // .... XY Well Position Panel

    pnlInfoXY.setLayout(new GridLayout());

    lblLat.setText("Lat: ");
    lblLong.setText("Long: ");

    // .... Z Well Position Panel

    pnlInfoZ.setLayout(new GridLayout());

    lblTD.setText("Depth:");
    lblElev.setText("Elev (GL):");

    // Edit Header Information

    btnHeader.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHeader.setText("Edit Header Information");
    btnHeader.addActionListener(this);

    // Convert Old Neutron Gamma Ray Log Counts to Neutron Porosity Gamma Ray

    pConvert = new lasConvertFrame(notifier);

    // Create the Available Plot Tracks List and an Observer

    pnlDefaultList = new lasAvailablePanel(notifier);
    getSelected(); //iSelected = pnlDefaultList.getSelected();

    // .. Modify Track Order Panel

    pnlModifyOrder.setLayout(new BorderLayout());
    pnlModifyOrder.setBorder(titledBorderOrder);

    // .... Track List Panel

    pnlTrackList.setLayout(new BorderLayout());

    pTracks          = new iqstratTracksTable( iSelected );
    scrollTrackOrder = pTracks.getScrollPane();

    // .... Track List Button Panel

    pnlTrackButton.setLayout(new GridLayout());

    btnUp.setFont(new java.awt.Font("Dialog", 1, 11));
    btnUp.setText("Move Up");
    btnUp.addActionListener(this);

    btnDown.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDown.setText("Move Down");
    btnDown.addActionListener(this);

    // Create the Colorlith selector frame

//    pColorlith = new lithColorlithFrame(notifier);
//    iColorlith = pColorlith.getColorlith();

    // -- Mouse Cursor Depth

    pnlMouseDepth.setBorder(titledBorderCursor);
    pnlMouseDepth.setLayout(new BorderLayout());

    txtCursorDepth.setEditable(false);
    txtCursorDepth.setText("0.0");
    txtCursorDepth.setHorizontalAlignment(SwingConstants.TRAILING);

    // Panel Global Buttons

    pnlButtons.setLayout(new GridLayout(4, 2));

    // Panel Center

    pnlCenter.setLayout(new BorderLayout());

    // -- Panel Depth Control

    pnlDepthControl.setBorder(titledBorderDepth);
    pnlDepthControl.setPreferredSize(new Dimension(10, 120));
    pnlDepthControl.setLayout(new BorderLayout());

    // ---- Panel Depth Scale

    pnlScale.setLayout(new BorderLayout());

    lblScale.setFont(new java.awt.Font("Dialog", 1, 14));
    lblScale.setHorizontalAlignment(SwingConstants.CENTER);
    lblScale.setText("Depth Scale: "+ scale[iScale]);

    // ---- Panel Depth

    pnlDepthRange.setLayout(new BorderLayout());
    pnlDepths.setLayout(new GridLayout());

    // .... Panel Depth Ranges

    pnlStart.setBorder(titledBorderStart);
    pnlStart.setLayout(new BorderLayout());

    txtStart.setText("" + dStart);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new lasControlPlotFrameFocusAdapter(this));

    pnlEnd.setBorder(titledBorderEnd);
    pnlEnd.setLayout(new BorderLayout());

    txtEnd.setText("" + dEnd);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new lasControlPlotFrameFocusAdapter(this));

    // .... Panel Depth Buttons

    pnlDepthButtons.setLayout(new GridLayout());

    btnReset.setFont(new java.awt.Font("Dialog", 1, 11));
    btnReset.setPreferredSize(new Dimension(67, 20));
    btnReset.setText(BTN_RESET);
    btnReset.addActionListener(this);

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setPreferredSize(new Dimension(71, 20));
    btnModify.setText(BTN_MODIFY);
    btnModify.addActionListener(this);

    // -- Panel Tracks

    pnlTracks.setLayout(new BorderLayout());
    pnlTracks.setBorder(titledBorderTracks);

    // ---- Panel Track Preference

    pnlTrackPref.setLayout(new BorderLayout());
    pnlPlotTracks.setLayout(cardLayout);

    // .. Track Buttons Panel

    pnlTracksBTNS.setLayout(new GridLayout());
    pnlTracksBTNS.setBorder(titledBorderLimits);

    btnLimitsLAS.setFont(new java.awt.Font("Dialog", 1, 11));
    btnLimitsLAS.setText("Log Data");
    btnLimitsLAS.setBackground(new Color(204, 255, 153));
    btnLimitsLAS.addActionListener(this);

    btnLimitsCORE.setEnabled(false);
    btnLimitsCORE.setFont(new java.awt.Font("Dialog", 1, 11));
    btnLimitsCORE.setText("Core Data");
    btnLimitsCORE.setBackground(new Color(255, 204, 102));
    btnLimitsCORE.addActionListener(this);

    // .... Panel Track Buttons

    pnlTrackButtons.setLayout(new BorderLayout());

    pnlSelect.setLayout(new GridLayout());
    pnlDefault.setLayout(new BorderLayout());
    pnlUser.setLayout(new BorderLayout());

    btnDefault.setFont(new java.awt.Font("Dialog", 1, 11));
    btnDefault.setText(BTN_DEFAULT);
    btnDefault.addActionListener(this);

    btnUser.setFont(new java.awt.Font("Dialog", 1, 11));
    btnUser.setText(BTN_USER);
    btnUser.addActionListener(this);

    // .... Available Track Button Panel

    setTrackPanelColors(); // Set Track Button Colors
    setButtons();          // Set Frame Buttons

    // Add Widgets to Frame

    this.getContentPane().add(pnlTop,     BorderLayout.NORTH);
    pnlTop.add(pnlHeader,                 BorderLayout.CENTER);
    pnlHeader.add(pnlInfo,                BorderLayout.CENTER);

    pnlInfo.add(pnlID,                    BorderLayout.NORTH);
    pnlID.add(pnlName,                    null);
    pnlName.add(lblName,                  BorderLayout.WEST);

    pnlID.add(pnlStatus,                  null);
    pnlStatus.add(lblAPI,                 BorderLayout.WEST);
    pnlStatus.add(lblStatus,              BorderLayout.CENTER);

    pnlInfo.add(pnlInfoPosition,          BorderLayout.CENTER);
    pnlInfoPosition.add(pnlInfoXY,        BorderLayout.NORTH);
    pnlInfoXY.add(lblLat,                 null);
    pnlInfoXY.add(lblLong,                null);

    pnlInfoPosition.add(pnlInfoZ,         BorderLayout.CENTER);
    pnlInfoZ.add(lblTD,                   null);
    pnlInfoZ.add(lblElev,                 null);

    pnlInfo.add(btnHeader,                BorderLayout.SOUTH);

    this.getContentPane().add(pnlCenter,  BorderLayout.CENTER);
    pnlCenter.add(pnlDepthControl,        BorderLayout.NORTH);

    pnlDepthControl.add(pnlScale,         BorderLayout.NORTH);
    pnlDepthRange.add(lblScale,           BorderLayout.NORTH);

    pnlDepthControl.add(pnlDepthRange,    BorderLayout.CENTER);
    pnlDepthRange.add(pnlDepths,          BorderLayout.CENTER);

    pnlDepths.add(pnlMouseDepth,          null);
    pnlMouseDepth.add(txtCursorDepth,     BorderLayout.CENTER);

    pnlDepths.add(pnlStart,               null);
    pnlStart.add(txtStart,                BorderLayout.CENTER);

    pnlDepths.add(pnlEnd,                 null);
    pnlEnd.add(txtEnd,                    BorderLayout.CENTER);

    pnlDepthRange.add(pnlDepthButtons,    BorderLayout.SOUTH);
    pnlDepthButtons.add(btnReset,         null);
    pnlDepthButtons.add(btnModify,        null);

    pnlCenter.add(pnlTracks,              BorderLayout.CENTER);
    pnlTracks.add(pnlTracksBTNS,          BorderLayout.NORTH);
	pnlTracksBTNS.add(btnLimitsLAS,       null);
	pnlTracksBTNS.add(btnLimitsCORE,      null);

    pnlTracks.add(pnlTrackButtons,        BorderLayout.SOUTH);
    pnlTrackButtons.add(pnlSelect,        BorderLayout.CENTER);
    pnlSelect.add(btnDefault,             null);
    pnlSelect.add(btnUser,                null);

    pnlTracks.add(pnlTrackPref,           BorderLayout.CENTER);
    pnlTrackPref.add(pnlPlotTracks,       BorderLayout.CENTER);

    pnlPlotTracks.add(pnlDefault,         "DEFAULT");
    pnlDefault.add(pnlDefaultList,        BorderLayout.CENTER);

    pnlPlotTracks.add(pnlModifyOrder,     "MODIFY");
    pnlModifyOrder.add(pnlTrackButton,   BorderLayout.SOUTH);
    pnlTrackButton.add(btnUp,            null);
    pnlTrackButton.add(btnDown,          null);

    pnlModifyOrder.add(pnlTrackList,      BorderLayout.CENTER);
    pnlTrackList.add(scrollTrackOrder,    BorderLayout.CENTER);

    // Display the Frame

    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    iScreenWidth = d.width;
    iScreenHigh  = d.height-50;

    this.setSize(new Dimension(iControlWidth, d.height-50)); //600
//    this.setSize(new Dimension(iControlWidth, 600));
    this.setLocation(1,1);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /** METHOD createImageIcon()
   *  <p> This Method will create the Frame for this class.
   *  @param path = the path to the image
   *  @return image icon
   */

  protected ImageIcon createImageIcon(String path)
  {
    java.net.URL imgURL = getClass().getResource(path);
    ImageIcon    image  = null;

    if (imgURL != null)
    {
      image = new ImageIcon(imgURL);
    }
    else
    {
       System.err.println("Couldn't find file: " + path);
    }

    return (image);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve data
   * @param iData     = Data Type being retrieved
   */

  private void getData( int iData )
  {
    double diff   = 0.0;
    double depth1 = 0.0;
    double depth2 = 0.0;

    switch (iData)
    {
      case _HEADERS:    // Headers Data
        setWellHeaderLabels();             // Set the Well Header Labels

        plotframe.setHeaders( stHeader );  // Send Data to Plot
        break;

      case _LAS:        // LAS File Data
        if (stLASFileData != null)
        {
          stLASFileData = las3.las3LoadLogData.normalizeDepth(
                              stHeader, stLASFileData );

          ImageTrackFrame();  // New Image Track dialog

          // Set the Image Track Default Values

          setLimits( _PHI_IMAGE );
          setLimits( _OHM_IMAGE );

          // Set the Depth Range

          dStart     = stLASFileData.depthStart;
          dEnd       = stLASFileData.depthEnd;
          iColorlith = stLASFileData.iColorlithDefault;

          setScale( iScale, dEnd, dStart );

          dStartOrig = dStart;
          dEndOrig   = dEnd;

          txtStart.setText( "" + dStart );
          txtEnd.setText( ""   + dEnd );

          diff = dEnd - dStart;

          if (diff < 10)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_1);
          else if (diff < 20)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_2);
          else if (diff < 50)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_5);
          else if (diff < 100)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_10);
          else if (diff < 200)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_20);
          else if (diff < 500)
            changeScale(iqstrat.iqstratTracksStruct.SCALE_50);
          else
            changeScale(iqstrat.iqstratTracksStruct.SCALE_100);

          // Set LAS File Data Structure int Covert Log Data

          pConvert.setLAS( las.lasFileDataUtility.copy( stLASFileData ));

          // Compute the Lithology from the LAS Log Curves

          stLASLithology =
              iqstrat.iqstratControlPlotData.computeLithologyFromLAS(
                iqstrat.iqstratTracksStruct._COLORLITH_RU,
                stLASFileData, stStruct.stSymbols );

          stRTLithology =
              iqstrat.iqstratControlPlotData.computeLithologyFromLAS(
                iqstrat.iqstratTracksStruct._COLORLITH_RT,
                stLASFileData, stStruct.stSymbols );

          // Send Data to Plot

          plotframe.setLASFileData( stLASFileData );        // LAS File Data
          plotframe.setLASRockColumn( stLASLithology );     // Rock Lithology
          plotframe.setLASRockColumnByRT( stRTLithology );  // Rock Lithology from LAS by Rhomaa-DT
          plotframe.setColorlith( iColorlith );             // Colorlith identifier

          stLASTexture = plotframe.getTexture();

          // -- Thin Track Entry Dialog

//          plotframe.setThinTrackFrame( pnlThinTrack );
          plotframe.setPHI( iTHINPHI );

          // -- Resistivity & Porosity Image Track Entry Dialog

          plotframe.setImageTrackFrame( pnlImageTrack );
          plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
          plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );

          // Compute the Non-Linear component of the Image Track

          WaitCompute();
        }
        break;

      case _SHALE:  // Shale gamma ray limits
        if (stLASFileData != null)
          stLASFileData.stList = iqstrat.iqstratShaleUtility.copyList( stShale );

        plot();
        break;

      case _BRINE:
        if (stBrine != null)
          plotframe.setBrine( stBrine );
        break;

      case _ROCK:       // Rock Data
        if (stCore != null)
        {
          // Compute the Depth Range if the LAS File Data Structure does not Exist

          if (stLASFileData == null)
          {
            dStart     = stCore.depthStart;
            dEnd       = stCore.depthEnd;

            dStartOrig = dStart;
            dEndOrig   = dEnd;

            txtStart.setText("" + dStart);
            txtEnd.setText(""   + dEnd);
          }

          // Compute the Rock Texture & Lithology from the Measured Core Data
          // if the Lithofacies or Depositional Environment is not present

//          getTexture();

          // Send Data to Plot

          plotframe.setRockData( stCore );  // Rock Data list structure
        }
        break;

      case _ROCK_IMAGES: // Rock Images
        plotframe.setRockImages(stImages); // Pass the Rock Image List Structure
        break;

      case _TOPS:       // Tops Data
        plotframe.setTops( stStrat );
        break;

      case _SEQ_STRAT:  // Sequence Stratigraphy
        plotframe.setSequence( stSequence );
        break;

      case _DEP_ENV:  // Depositional Environment
        plotframe.setDepEnvrionment( stDepEnv );
        break;

      case _PERFORATION: // Perforations
        plotframe.setTops( stOther );
        break;

      case _ROCK_COLUMN:  // Rock Column Litholody
        plotframe.setRockColumn( stColumn );
        break;

      case _LITHOLOGY:  // Rock Lithology
        break;

      case _TEXTURE:  // Rock Texture
        break;

      case _ROCK_PHI:  // Rock Porosity
        plotframe.setPHI(stPHI);
        break;

      case _SED_STRUCT: // Sedimentary Structure
        plotframe.setStructures(stSedimentary);

        // Modify the plot width based on the fossils and sturctures

        modifyWidths();
        break;

      case _FOSSILS: // Fossils
        plotframe.setFossils(stFossil);

        // Modify the plot width based on the fossils and sturctures

        modifyWidths();
        break;

      case _BIO_STRAT: // Bio Stratigraphy
        plotframe.setBioStrat(stBio);

        // Modify the plot width based on the fossils and sturctures

        modifyWidths();
        break;

      case _ROCK_COLOR: // Rock Color
        plotframe.setRockColorData(stRockColor);
        break;

      case _REMARKS: // Remarks/Comments
        plotframe.setRemarks( stRemarks );
        break;
    }

    setConfiguration( iConfig ); // Force the Control Plot Track Buttons
  }

  /** Method getData()
   * <p> This method will retrieve data
   * @param iLocation = Location of Data being retrieved
   * @param iData     = Data Type being retrieved
   */

  private void getData( int iLocation, int iData)
  {
    switch (iData)
    {
      case _LAS:        // LAS File Data
        stLASFileData = las.lasFileDataUtility.copy( getLAS( iLocation, pConvert ) );

        if (stLASFileData != null)
        {
          // Set LAS File Data Structure int Covert Log Data

          pConvert.setLAS( las.lasFileDataUtility.copy( stLASFileData ));
	    }
        break;

      case _SHALE:  // Shale gamma ray limits
        break;

      case _TOPS:       // Tops Data
        stStrat = horizon.strat.stratUtility.copyList( getTops( pTopsEntry ) );

        plotframe.setTops( stStrat );
        break;

      case _SEQ_STRAT:  // Sequence Stratigraphy
      case _DEP_ENV:  // Depositional Environment
      case _PERFORATION: // Perforations
      case _BRINE:
      case _LITHOLOGY:  // Rock Lithology
      case _TEXTURE:  // Rock Texture
      case _ROCK_COLUMN:  // Rock Column Lithology
      case _ROCK_PHI:  // Rock Porosity
      case _SED_STRUCT: // Sedimentary Structure
      case _FOSSILS: // Fossils
      case _BIO_STRAT: // Bio stratigraphy
      case _ROCK_COLOR: // Rock Color
      case _REMARKS: // Remarks/Comments
        break;
	}
  }

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @param  iLocation  = Location identifier
   * @param  pConvert   = Convert Old Neutron Gamma Ray Counts to Log Data
   * @return st         = LAS File Data Structure
   */

  public static lasFileDataStruct getLAS(
      int                  iLocation,  // Location identifier
      lasConvertFrame      pConvert )  // Convert Counts to Log Data
  {
    lasFileDataStruct st = null;

    switch (iLocation)
    {
      case _CONVERT:
        st = pConvert.getLAS();
        pConvert.setVisible(false);
        break;
    }

    return (st);
  }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Units Data List Structure
   * @param  pTopsEntry = Tops Entry Data Frame
   * @return st         = the Stratigraphic Units Data List Structure
   */

  public static stratListStruct getTops( horizonEnterDataFrame pTopsEntry )
  {
    stratListStruct st = null;

    if (pTopsEntry != null)
      st = pTopsEntry.getTopsData();

    return ( st );
  }

  /** Method getGeoReportData()
   * <p> This method will compute the Geologist Report Data tracks if the
   *     tracks are empty
   */

  private void getGeoReportData()
  {
    double diff = 0.0;

    if (stRemarks != null)
    {
      // Find the Starting & Ending Depth from the Texture Data

      if ((stPlotTexture != null) && (stLASFileData == null))
      {
        dStart = stPlotTexture.depthStart;
        dEnd   = stPlotTexture.depthEnd;

        stStruct.depthStart = stPlotTexture.depthStart;
        stStruct.depthEnd   = stPlotTexture.depthEnd;
        stStruct.depthStep  =
            iqstrat.iqstratTracksStruct._SCALE[stPlotTexture.iUnit];
      }

      if ((stColumn != null) && (stLASFileData == null))
      {
        stColumn = lith.rock.rockColumnUtility.computeDepthRange( stColumn );

        dStart = stColumn.depthStart;
        dEnd   = stColumn.depthEnd;

        stStruct.depthStart = stColumn.depthStart;
        stStruct.depthEnd   = stColumn.depthEnd;
      }

      if (dEnd > dStart)
      {
        dStartOrig = dStart;
        dEndOrig = dEnd;

        // Set the Starting & Ending Depth in the Control Panel

        txtStart.setText("" + dStart);
        txtEnd.setText("" + dEnd);

        diff = dEnd - dStart;
        if (stHeader == null) stHeader = new iqstratHeadersStruct();
        stHeader.depth = diff;

        // Set Profile to Rock Data

        setConfiguration( iqstrat.iqstratControlStruct._C_CORE );
        getSelected();
        plotframe.setSelected(iSelected);

        // Set Scale factor depending on depth range of Measured Section

        if (diff < 10)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_1);
        else if (diff < 20)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_2);
        else if (diff < 50)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_5);
        else if (diff < 100)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_10);
        else if (diff < 200)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_20);
        else if (diff < 500)
          changeScale(iqstrat.iqstratTracksStruct.SCALE_50);
        else
          changeScale(iqstrat.iqstratTracksStruct.SCALE_100);

        setWellHeaderLabels(); // Set the Well Header Labels
        plotframe.setHeaders( stHeader );
      }

      if ((stCore == null) && (stRockData == null) && (stPlotTexture != null))
      {
        // Get Lithology data from Measured Section texture

        stRockData = rock.rockFileDataUtility.buildNew( 0, dStart, dEnd );
        stRockData = rock.rockFileDataUtility.addData( stPlotTexture, stRockData );
        stRockData = rock.rockFileDataUtility.lithologyCurves( stRockData );

        plotframe.setRockFileData( stRockData );

        setButtons(); // Set Frame Buttons
      }
    }
  }

  /** Method ThinTrackFrame()
   * <p> This method will open a new Thin Track Input Dialog
   */

  private void ThinTrackFrame()
  {
    if (pnlThinTrack != null)
      pnlThinTrack.close();
    pnlThinTrack = null;

    pnlThinTrack  = new lasThinTrackFrame( notifier, stLASFileData );
  }

  /** Method ImageTrackFrame()
   * <p> This method will open a new Image Track Input Dialog
   */

  private void ImageTrackFrame()
  {
    if (pnlImageTrack != null)
      pnlImageTrack.close();
    pnlImageTrack = null;

    pnlImageTrack = new lithImageTrackFrame( notifier, stLASFileData );
  }

  /** Method getSelected()
   * <p> This method will retrieve the selected tracks
   */

  private void getSelected() { iSelected = pnlDefaultList.getSelected(); }

  /** Method getLASTrackSize()
   * <p> This method will retrieve the LAS Track Size
   */

  private void getLASTrackSize()
  {
    iLASSize = pnlDefaultList.getLASTrackSize();
    if (plotframe != null) { plotframe.setLASTrackSize(iLASSize); }
    if (plotframe != null) { plotframe.setSelected( iSelected ); }
  }

  /** Method getThinPorosity()
   * <p> This method will get the thin track porosity data
   */

  private void getThinPorosity( int iSource )
  {
    iThin           = cmn.cmnStruct._NO;
    stControl.iThin = cmn.cmnStruct._NO;

    for (int i=0; i<iSelected.length; i++)
    {
      if (iSelected[i] == iqstrat.iqstratTracksStruct._THIN_PHI)
      {
        iThin           = cmn.cmnStruct._YES;
        stControl.iThin = cmn.cmnStruct._YES;
      }
    }

    switch (iSource)
    {
      case _EDIT_PANEL:
        if (pnlThinTrack != null)
        {
          iTHINPHI           = pnlThinTrack.getPHI();
          stControl.iTHINPHI = pnlThinTrack.getPHI();
          pnlThinTrack.setVisible( false );
        }
        break;

      case _AVAILABLE:
        if (pnlDefaultList != null)
        {
          iTHINPHI           = pnlDefaultList.getThinCurve();
          stControl.iTHINPHI = pnlDefaultList.getThinCurve();
		}
        break;
    }

    if (plotframe != null) { plotframe.setPHI( iTHINPHI ); }
  }

  /** Method getImageTrack()
   * <p> This method will get the Resistivity & Porosity Image track data
   */

  private void getImageTrack()
  {
    int iTrack = pnlImageTrack.getTrack();

    switch (iTrack)
    {
      case lith.gui.lithImageTrackFrame._RT:
        iOHM           = cmn.cmnStruct._NO;
        stControl.iOHM = cmn.cmnStruct._NO;

        for (int i=0; i<iSelected.length; i++)
        {
          if (iSelected[i] == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER)
          {
            iOHM           = cmn.cmnStruct._YES;
            stControl.iOHM = cmn.cmnStruct._YES;
          }
        }

        iOHMTool           = pnlImageTrack.getTool();
        dOHMMax            = pnlImageTrack.getMaximum();
        dOHMMin            = pnlImageTrack.getMinimum();

        stControl.iOHMTool = pnlImageTrack.getTool();
        stControl.dOHMMax  = pnlImageTrack.getMaximum();
        stControl.dOHMMin  = pnlImageTrack.getMinimum();

        if (plotframe != null)
        {
          plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
        }
        break;

      case lith.gui.lithImageTrackFrame._PHIT:
        iPHI           = cmn.cmnStruct._NO;
        stControl.iPHI = cmn.cmnStruct._NO;

        for (int i=0; i<iSelected.length; i++)
        {
          if (iSelected[i] == iqstrat.iqstratTracksStruct._POROSITY_IMAGER)
          {
            iPHI           = cmn.cmnStruct._YES;
            stControl.iPHI = cmn.cmnStruct._YES;
          }
        }

        iPHITool           = pnlImageTrack.getTool();
        dPHIMax            = pnlImageTrack.getMaximum();
        dPHIMin            = pnlImageTrack.getMinimum();

        stControl.iPHITool = pnlImageTrack.getTool();
        stControl.dPHIMax  = pnlImageTrack.getMaximum();
        stControl.dPHIMin  = pnlImageTrack.getMinimum();

        if (plotframe != null)
        {
          plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );
        }
        break;
    }

    pnlImageTrack.setVisible( false );
    WaitCompute();
  }

  /** Method getImageTrack()
   * <p> This method will get the Resistivity & Porosity Image track data
   */

  private void getImageTrack( int iSource, int iTrack )
  {
	int iRefresh = 0;

    switch (iTrack)
    {
      case lith.gui.lithImageTrackFrame._RT:
        iOHM           = cmn.cmnStruct._NO;
        stControl.iOHM = cmn.cmnStruct._NO;

        for (int i=0; i<iSelected.length; i++)
        {
          if (iSelected[i] == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER)
          {
            iOHM           = cmn.cmnStruct._YES;
            stControl.iOHM = cmn.cmnStruct._YES;
            iRefresh = 1;
          }
        }

        iOHMTool           = pnlDefaultList.getOHMCurve();
        stControl.iOHMTool = pnlDefaultList.getOHMCurve();

        if (plotframe != null)
        {
          plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
        }
        break;

      case lith.gui.lithImageTrackFrame._PHIT:
        iPHI           = cmn.cmnStruct._NO;
        stControl.iPHI = cmn.cmnStruct._NO;

        for (int i=0; i<iSelected.length; i++)
        {
          if (iSelected[i] == iqstrat.iqstratTracksStruct._POROSITY_IMAGER)
          {
            iPHI           = cmn.cmnStruct._YES;
            stControl.iPHI = cmn.cmnStruct._YES;
            iRefresh = 1;
          }
        }

        iPHITool           = pnlDefaultList.getPHICurve();
        stControl.iPHITool = pnlDefaultList.getPHICurve();

        if (plotframe != null)
        {
          plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );
        }
        break;
    }

    if (iRefresh == 1) WaitCompute();
  }

  /** Method modifyWidths()
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   */

  private void modifyWidths()
  {
    int iBio  = 0;
    int iSed  = 0;
    int iBioS = 0;

    iSed  = rock.sedimentary.sedimentaryUtility.computeWidth( stSedimentary );
    iBio  = rock.fossil.fossilUtility.computeWidth( stFossil );
    iBioS = horizon.bio.bioStratUtility.computeWidth( stBio );

    if (plotframe != null) plotframe.modifyWidths( iBio, iSed, iBioS );
  }

  /** Method getControl()
   * <p> This method will get the Control Data data to the Profile Plot
   * @return st = Control Data Structure
   */

  private iqstratControlStruct getControl()
  {
    int i=0;
    iqstratControlStruct st = new iqstratControlStruct();

    st.depthStart = dStart;
    st.depthEnd   = dEnd;
    st.iScale     = iScale;

    st.iPanel   = iPanel;
    st.iConfig  = iConfig;

    if (iSelected != null)
    {
      st.iTracks = iSelected;

      for (i=0; i<iSelected.length; i++)
      {
        if (iSelected[i] == iqstrat.iqstratTracksStruct._THIN_PHI)
        {
          st.iThin = cmn.cmnStruct._YES;
        }
      }

      for (i=0; i<iSelected.length; i++)
      {
        if ((iSelected[i] == iqstrat.iqstratTracksStruct._POROSITY_IMAGER) ||
            (iSelected[i] == iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N))
        {
          st.iPHI = cmn.cmnStruct._YES;
        }
      }

      for (i=0; i<iSelected.length; i++)
      {
        if ((iSelected[i] == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER) ||
            (iSelected[i] == iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N))
        {
          st.iOHM = cmn.cmnStruct._YES;
        }
      }
    }

    st.iTHINPHI = iTHINPHI;

    // -- Porosity Image Track

    st.dPHIMin  = dPHIMin;
    st.dPHIMax  = dPHIMax;
    st.iPHITool = iPHITool;

    // -- Resistivity Image Track

    st.dOHMMin  = dOHMMin;
    st.dOHMMax  = dOHMMax;
    st.iOHMTool = iOHMTool;

    st = iqstrat.iqstratControlUtility.addConversionData(stLASFileData, st);

    return ( st );
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setData()
   * <p> This method will set the extra data structures
   * @param stControl = Plot Control Data Structure
   * @param stHeader  = Header Information Data Structure
   * @param stFiles   = LAS File Information List Strcuture
   * @param stLAS     = LAS File Data Structure
   * @param stShale   = Shale Gamma Ray Limits Data List Structure
   * @param stFlow    = PfEFFER Flow Units Data List Structure
   * @param stStrat   = Stratigraphic Units Data List Structure
   * @param stSeq     = Sequence Stratigraphy Data List Structure
   * @param stOther   = Perforations Data List Structure
   * @param stDepEnv  = Depositional Environment List Structure
   * @param stBrine   = Brine Data List Structure
   * @param stCore    = Measured Rock Data List Structure
   * @param stImages  = Rock Images
   * @param stColumn  = Rock Column Lithology
   * @param stLith    = Rock Lithology Data List Structure
   * @param stTexture = Rock Texture Data List Structure
   * @param stPHI     = Porosity Type Data List Structure
   * @param stFossil  = Fossils Data List Structure
   * @param stSed     = Sedimentary Structure Data List Structure
   * @param stColor   = Rock Color Data List Structure
   * @param stRemarks = Remarks/Notes/Comments Data List Structure
   */

  public void setData( iqstratControlStruct    stControl,
                       iqstratHeadersStruct    stHeader,
                       // LAS Data
                       cmnLASFileListStruct    stFiles,
                       lasFileDataStruct       stLAS,
                       iqstratShaleListStruct  stShale,
                       // Horizon Data
                       stratListStruct         stStrat,
                       seqListStruct           stSeq,
                       regionsListStruct       stOther,
                       envListStruct           stDepEnv,
                       bioStratListStruct      stBio,
                       // Brine Data
                       brineListStruct         stBrine,
                       // Rock Data
                       rockDataListStruct      stCore,
                       rockImagesListStruct    stImages,
                       rockColumnListStruct    stColumn,
                       phiListStruct           stPHI,
                       fossilListStruct        stFossil,
                       sedimentaryListStruct   stSed,
                       rockColorListStruct     stColor,
                       iqstratRemarkListStruct stRemarks )
  {
	double dVer = 0.0;
	int    itmp = -1;

    this.stControl       = stControl;
    this.stHeader        = stHeader;
    this.stFiles         = stFiles;
    this.stLASFileData   = stLAS;
    this.stShale         = stShale;
    this.stStrat         = stStrat;
    this.stSequence      = stSeq;
    this.stOther         = stOther;
    this.stDepEnv        = stDepEnv;
    this.stBio           = stBio;
    this.stBrine         = stBrine;
    this.stCore          = stCore;
    this.stImages        = stImages;
    this.stColumn        = stColumn;
    this.stPHI           = stPHI;
    this.stFossil        = stFossil;
    this.stSedimentary   = stSed;
    this.stRockColor     = stColor;
    this.stRemarks       = iqstrat.iqstratRemarkUtility.copyList( stRemarks );

    if (stCore != null)
      btnLimitsCORE.setEnabled(true);

    getData( _HEADERS );      // Header Information
    getData( _LAS );          // Log Data
    getData( _TOPS );         // Formation Tops
    getData( _SEQ_STRAT );    // Sequence Stratigraphy
    getData( _DEP_ENV );      // Depositional Environment
    getData( _PERFORATION );  // Perforation
    getData( _SHALE );        // Shale Limits

    // Brine Data

    getData( _BRINE );        // Brine Data

    // Rock Measured & Observational Data

    getData( _ROCK );         // Measured Rock Data
    getData( _REMARKS );      // Remarks/Comments/Notes

    // Other Rock Data

    getData( _ROCK_COLUMN );  // Rock Lithology Column
    getData( _ROCK_PHI );     // Rock Porosity Type
    getData( _FOSSILS );      // Fossils
    getData( _SED_STRUCT );   // Sedimentary Structures
    getData( _ROCK_COLOR );   // Rock Color

    getData( _BIO_STRAT );    // Bio stratigraphy

    // Compute Rock Image Tracks from Remarks Data

    getGeoReportData();

    setControl( stControl );

    switch (iAction)
	{
	  case cmn.cmnStruct._LAS:     // LAS File Data Entry & Viewer
	    iConfig = iqstrat.iqstratControlStruct._C_LAS;
	    break;
	  case cmn.cmnStruct._TOPS:    // Horizons Data Entry & Viewer
	    iConfig = iqstrat.iqstratControlStruct._C_HORIZON;
	    break;
	  case cmn.cmnStruct._CORE:    // Core Data Data Entry & Viewer
	  case cmn.cmnStruct._ROCK:    // Geologist Report Data & Viewer
	    iConfig = iqstrat.iqstratControlStruct._C_CORE;
	    break;
	  case cmn.cmnStruct._LAS3:    // LAS 3.0 File Viewer
	  case cmn.cmnStruct._KGS:     // KGS Data Viewer
        iConfig = iqstrat.iqstratControlStruct._C_ALL;
	    break;
	}

    setConfiguration( iConfig );

    if (stLAS != null)
    {
	  if (cmn.cmnString.isNumeric(stLAS.sVersion))
	    dVer = cmn.cmnString.stringToDouble(stLAS.sVersion);
	}

	if (dVer != 3.0)
	{
	  itmp = pnlDefaultList.getThinCurve();
	  if (itmp != -1)
	  {
        iThin    = cmn.cmnStruct._YES;
        iTHINPHI = itmp;
        plotframe.setPHI( iTHINPHI );
	  }

	  itmp = pnlDefaultList.getOHMCurve();
	  if (itmp != -1)
	  {
        iOHM     = cmn.cmnStruct._YES;
        iOHMTool = itmp;
        plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
      }

	  itmp = pnlDefaultList.getPHICurve();
	  if (itmp != -1)
	  {
        iPHI     = cmn.cmnStruct._YES;
        iPHITool = itmp;
        plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );
      }
	}
  }

  /** Method setWellHeaderLabels()
   *  <p> This method will set the Well Header Label Fields
   */

  private void setWellHeaderLabels()
  {
    if (stHeader != null)
    {
      lblName.setText( "Name: "     + stHeader.sName );
      lblAPI.setText(stHeader.sAPI  + " ");
      lblStatus.setText( "Status: " + stHeader.status );
      lblLat.setText( "Lat: "       + stHeader.dLatitude );
      lblLong.setText( "Long: "     + stHeader.dLongitude );
      lblTD.setText( "Depth: "      + stHeader.depth );

      lblElev.setText("Elev: ");
      if (stHeader.dGL > 0.0)
        lblElev.setText("Elev (GL): " + stHeader.dGL);
      else if (stHeader.dKB > 0.0)
        lblElev.setText("Elev (KB): " + stHeader.dKB);
      else if (stHeader.dDF > 0.0)
        lblElev.setText("Elev (DF): " + stHeader.dDF);
    }
    else
    {
      lblName.setText( "Name: " );
      lblAPI.setText( " " );
      lblStatus.setText( "Status: " );
      lblLat.setText( "Lat: " );
      lblLong.setText( "Long: " );
      lblTD.setText( "Depth: " );

      lblElev.setText("Elev: ");
    }
  }

  /** Method setButtons()
   * <p> This method will set the buttons depending on the data present.
   */

  private void setButtons()
  {
    m1.setSelected(false);    // Display Plot at   1 ft / in
    m2.setSelected(false);    // Display Plot at   2 ft / in
    m5.setSelected(false);    // Display Plot at   5 ft / in
    m10.setSelected(false);   // Display Plot at  10 ft / in
    m20.setSelected(false);   // Display Plot at  20 ft / in
    m50.setSelected(false);   // Display Plot at  50 ft / in
    m100.setSelected(false);  // Display Plot at 100 ft / in
    m200.setSelected(false);  // Display Plot at 200 ft / in

    switch (iScale)
    {
      case iqstrat.iqstratTracksStruct.SCALE_1:   //   1 Feet / Inch
        m1.setSelected(true);    // Display Plot at   1 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_2:   //   2 Feet / Inch
        m2.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_5:   //   5 Feet / Inch
        m5.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_10:   //  10 Feet / Inch
        m10.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_20:   //  20 Feet / Inch
        m20.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_50:   //  50 Feet / Inch
        m50.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_100:   // 100 Feet / Inch
        m100.setSelected(true);    // Display Plot at   2 ft / in
        break;
      case iqstrat.iqstratTracksStruct.SCALE_200:   // 200 Feet / Inch
        m200.setSelected(true);    // Display Plot at   2 ft / in
        break;
    }

    btnUp.setEnabled(false);
    btnDown.setEnabled(false);

    if (pTracks.getTotalRows() > 0)
    {
      btnUp.setEnabled(true);
      btnDown.setEnabled(true);
    }
  }

  /** Method setLimits()
   * <p> This method will set the color limits
   * @param iTrack = The Image Track
   */

  private void setLimits(int iTrack)
  {
    int    i        = 0;
    int    iChng    = 0;
    int    iCont    = 0;
    int    iTool    = -1;
    double dValue   = 0.0;
    double dMinimum = 0.0;
    double dMaximum = 0.0;
    double data[]   = null;

    if (stLASFileData != null)
    {
      switch (iTrack)
      {
        case _PHI_IMAGE:
          iTool    = iPHITool;
          dMinimum = dPHIMin;
          dMaximum = dPHIMax;

          if ((iCont == 0) &&
              (stLASFileData.checkData(las.lasStandardTools._NPHI)))
          {
			iTool = las.lasStandardTools._NPHI;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._DPHI)))
          {
            iTool = las.lasStandardTools._DPHI;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._SPHI)))
          {
            iTool = las.lasStandardTools._SPHI;
            iCont = 1;
		  }
          break;

        case _OHM_IMAGE:
          iTool    = iOHMTool;
          dMinimum = dOHMMin;
          dMaximum = dOHMMax;

          // Induction

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._ILD)))
          {
            iTool = las.lasStandardTools._ILD;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._ILM)))
          {
            iTool = las.lasStandardTools._ILM;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._SFLU)))
          {
            iTool = las.lasStandardTools._SFLU;
            iCont = 1;
		  }

          // Induction Array

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._AHT90)))
          {
            iTool = las.lasStandardTools._AHT90;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._AHT60)))
          {
            iTool = las.lasStandardTools._AHT60;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._AHT30)))
          {
            iTool = las.lasStandardTools._AHT30;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._AHT20)))
          {
            iTool = las.lasStandardTools._AHT20;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._AHT10)))
          {
            iTool = las.lasStandardTools._AHT10;
            iCont = 1;
		  }

          // Micro resistivity

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._MINV)))
          {
            iTool = las.lasStandardTools._MINV;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._MNOR)))
          {
            iTool = las.lasStandardTools._MNOR;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._MSFL)))
          {
            iTool = las.lasStandardTools._MSFL;
            iCont = 1;
		  }

          // resistivity

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._RDEP)))
          {
            iTool = las.lasStandardTools._RDEP;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._RES)))
          {
            iTool = las.lasStandardTools._RES;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._RMED)))
          {
            iTool = las.lasStandardTools._RMED;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._RSHAL)))
          {
            iTool = las.lasStandardTools._RSHAL;
            iCont = 1;
		  }

          // Laterlog

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._LL)))
          {
            iTool = las.lasStandardTools._LL;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._MLL)))
          {
            iTool = las.lasStandardTools._MLL;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._LL8)))
          {
            iTool = las.lasStandardTools._LL8;
            iCont = 1;
		  }

          // Normal

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._LN)))
          {
            iTool = las.lasStandardTools._LN;
            iCont = 1;
		  }

		  if ((iCont == 0) &&
		      (stLASFileData.checkData(las.lasStandardTools._SN)))
          {
            iTool = las.lasStandardTools._SN;
            iCont = 1;
		  }
          break;
      }

      data = stLASFileData.getData(iTool);
      if (data != null)
      {
        for (int j=0; j<data.length; j++)
        {
          dValue = stLASFileData.dNull;

          if ((iTool >= las.lasStandardTools._DPHI) &&
              (iTool <= las.lasStandardTools._SPHI))
            dValue = data[j];
          else if ((iTool >= las.lasStandardTools._ILD) &&
                   (iTool <= las.lasStandardTools._AHT90))
            dValue = 1000.0 / data[j];

          if (iChng == 0)
          {
           if (dValue > stLASFileData.dNull)
            {
              dMinimum = dValue;
              dMaximum = dValue;
              iChng = 1;
            }
          }
          else
          {
            if (dValue > stLASFileData.dNull)
            {
              if (dMinimum > dValue) dMinimum = dValue;
              if (dMaximum < dValue) dMaximum = dValue;
            }
          }
        }
      }

      switch (iTrack)
      {
        case _PHI_IMAGE:
          dPHIMin  = dMinimum;
          dPHIMax  = dMaximum;
          iPHITool = iTool;
          break;
        case _OHM_IMAGE:
          dOHMMin  = dMinimum;
          dOHMMax  = dMaximum;
          iOHMTool = iTool;
          break;
      }
    }
  }

  /** Method setOrderPanel( int iPanel )
   *  <p> This method will set the default panel or the modify order panel
   * @param iPanel = panel id
   */

  public void setOrderPanel( int iPanel )
  {
    this.iPanel      = iPanel;
    stControl.iPanel = iPanel;

    switch (iPanel)
    {
      case _DEFAULT:
        cardLayout.show(pnlPlotTracks, "DEFAULT");
        pnlDefaultList.setByAction( iAction,
                                    stLASFileData,
                                    stSequence,
                                    stStrat,
                                    stDepEnv,
                                    stBrine,
                                    stCore,
                                    stImages,
                                    stColumn,
                                    stRockLithology,
                                    stTexture,
                                    stPHI,
                                    stFossil,
                                    stBio,
                                    stSedimentary,
                                    stRockColor,
                                    stRemarks );

        pnlDefaultList.setNotify();
        break;

      case _USER:
        pTracks.setData( iSelected );
        cardLayout.show(pnlPlotTracks, "MODIFY");
        break;
    }
  }

  /** Method setInitialConfiguration( int iConfig )
   * <p> This method will set the Available Track Configuration
   * @param iConfig = identifier for the Available Track Configuration
   */

  private void setInitialConfiguration( int iConfig )
  {
    this.iConfig      = iConfig;
    stControl.iConfig = iConfig;

    setConfiguration( iConfig ); // NEW

    pnlDefaultList.setNotify();

    pTracks.setData( iSelected );
  }

  /** Method setConfiguration( int iConfig )
   * <p> This method will set the Available Track Configuration
   * @param iConfig = identifier for the Available Track Configuration
   */

  private void setConfiguration( int iConfig )
  {

    pnlDefaultList.setByAction( iAction,
                                stLASFileData,
                                stSequence,
                                stStrat,
                                stDepEnv,
                                stBrine,
                                stCore,
                                stImages,
                                stColumn,
                                stRockLithology,
                                stTexture,
                                stPHI,
                                stFossil,
                                stBio,
                                stSedimentary,
                                stRockColor,
                                stRemarks );

    this.iConfig      = iConfig;
    stControl.iConfig = iConfig;

    pnlDefaultList.setNotify();

    pTracks.setData( iSelected );
  }

  /** Method setFront()
   * <p> This method will set both the Control and Plot Frames to the front
   */

  public void setFront()
  {
    this.toFront();
    if (plotframe != null) plotframe.toFront();
  }

  /** Method setScale(int iScale, double dMaximum, double dMinimum)
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale(int iScale, double dMaximum, double dMinimum)
  {
    if (plotframe != null)
    {
      plotframe.setScale(iScale, dMaximum, dMinimum);
    }
  }

  /** Method setDepth()
   * <p> This method will set the Depth Range
   */

  private void setDepth()
  {
    double dTemp = 0.0;

    dStart = cmn.cmnString.stringToDouble(txtStart.getText());
    dEnd   = cmn.cmnString.stringToDouble(txtEnd.getText());

    if (dEnd < dStart)
    {
      dTemp  = dEnd;
      dEnd   = dStart;
      dStart = dTemp;
    }

    // Set the Starting & Ending Depth in the Control Panel

    txtStart.setText("" + dStart);
    txtEnd.setText(""   + dEnd);

    stControl.depthStart = dStart;
    stControl.depthEnd   = dEnd;

    setScale(iScale, dEnd, dStart);

    WaitCompute();
  }

  /** Method setDefaultDepth()
   * <p> This method will reset the depth to the original LAS Depth
   */

  private void setDefaultDepth()
  {
    txtStart.setText("" + dStartOrig);
    txtEnd.setText(""   + dEndOrig);

    dStart = dStartOrig;
    dEnd   = dEndOrig;

    setScale(iScale, dEnd, dStart);

    WaitCompute();
  }

  /** Method setTrackPanelColors()
   * <p> This method will set the Track Panel colors depending on what is
   *     showing.
   */

  private void setTrackPanelColors()
  {
    btnDefault.setBackground(new Color(100, 50, 0));
    btnDefault.setForeground(new Color(240, 240, 0));
    btnUser.setBackground(new Color(100, 50, 0));
    btnUser.setForeground(new Color(240, 240, 0));

    switch (iPanel)
    {
      case _DEFAULT:
        btnDefault.setBackground(new Color(240, 240, 0));
        btnDefault.setForeground(new Color(100, 50, 0));
        break;
      case _USER:
        btnUser.setBackground(new Color(240, 240, 0));
        btnUser.setForeground(new Color(100, 50, 0));
        break;
    }
  }

  /** Method setControl()
   * <p> This method will set the Control Data data to the Profile Plot
   * @param st = Control Data Structure
   */

  private void setControl( iqstratControlStruct st )
  {
    if (st != null)
    {
      this.stControl = iqstrat.iqstratControlUtility.copy( st );

      // -- Porosity Thin Track

      iThin    = st.iThin;
      iTHINPHI = st.iTHINPHI;

      // -- Porosity Image Track

      iPHI     = st.iPHI;
      dPHIMin  = st.dPHIMin;
      dPHIMax  = st.dPHIMax;
      iPHITool = st.iPHITool;

      // -- Resistivity Image Track

      iOHM     = st.iOHM;
      dOHMMin  = st.dOHMMin;
      dOHMMax  = st.dOHMMax;
      iOHMTool = st.iOHMTool;

      // Set the Starting & Ending Depth in the Control Panel

      if (st.depthStart != st.depthEnd)
      {
        dStart   = st.depthStart;
        dEnd     = st.depthEnd;
        iScale   = st.iScale;

        txtStart.setText( "" + dStart );
        txtEnd.setText( ""   + dEnd );
      }

      setConfiguration( st.iConfig );

      if (plotframe != null)
      {
        plotframe.setSelected( iSelected );
        plotframe.setPHI( iTHINPHI );
        plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
        plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );
      }

      if (stLASFileData != null)
      {
        // Add Conversion data to LAS File data Structure

        stLASFileData =
            las.lasFileDataUtility.addConversionData( st, stLASFileData );

        // Set LAS File Data Structure int Covert Log Data

        if (pConvert != null)
          pConvert.setLAS( las.lasFileDataUtility.copy( stLASFileData ));
      }

      setScale( iScale, dEnd, dStart );
    }
  }

  /** Method setLimits()
   * <p> This method will modify the LAS Plot Limits
   * @param data_las = array LAS Plot Track Limits
   * @param data_pf  = array PfEFFER Plot Track Limits
   */

  public void setLimits(double data_las[][], double data_core[][], double data_pf[][])
  {
	this.data_las  = data_las;
	this.data_core = data_core;
	this.data_pf   = data_pf;

    if (plotframe != null)
    {
	  plotframe.setPlotLimits( data_las, data_core, data_pf );
	}
  }

  /* ===================================================================== *
   * ---------------- Compute Non-Linear Image Track Data ---------------- *
   * ===================================================================== */

  /** Method Compute()
   * <p> This method will compute the non-linear component of the image tracks
   */

  private void Compute()
  {
    dRt   = null;
    dPhit = null;
    dPhi  = null;

    if (stLASFileData != null)
    {
      dRt = las.lasFileDataUtility.setData( iqstrat.iqstratTracksStruct._RT,
                     iOHMTool, dStart, dEnd, stLASFileData );
      dPhit = las.lasFileDataUtility.setData( iqstrat.iqstratTracksStruct._PHIT,
                     iPHITool, dStart, dEnd, stLASFileData );

      if (plotframe != null)
      {
        plotframe.setImageData( dRt, dPhit, dPhi );
        plotframe.refresh();
      }
    }
  }

  /** Method WaitCompute()
   * <p> This method will display the Wait Animation for computing the non
   *     linear component
   */

  private void WaitCompute()
  {
    java.net.URL url = null;

    // Get the URL Path to display the Wait Animation

    try
    {
      url = new java.net.URL(cmn.cmnStruct.IQSTRAT_BASE + "/");
    }
    catch(Exception ex)
    {
      System.out.println("Wrong Image Directory" + ex);
    }

    // Start the Wait Animation Dialog

    final utilWaitAnimation pDialog = new utilWaitAnimation(url);
    pDialog.startAnimation();

    final SwingWorker worker = new SwingWorker()
    {
      public Object construct()
      {
        Compute();

        return "OK";
      }

      //Runs on the event-dispatching thread.

      public void finished() { pDialog.stopAnimation(); }
    };

    // Start the Wait Animation Thread

    worker.start();  //required for SwingWorker
  }

  /* ===================================================================== *
   * ------------------------------ plot --------------------------------- *
   * ===================================================================== */

  /** Method plot()
   * <p> This method will create the stratigraphic plot
   */

  private void plot()
  {
    Dimension d       = null;
    String    sTitle  = "Plot Well: ";
    String    sTitle1 = "";
    String    sTitle2 = "";

    stStruct.depthStart = dStart;
    stStruct.depthEnd   = dEnd;
    stStruct.iScale     = iScale;

    if (stStruct.iDataType == iqstrat.iqstratStruct._OUTCROP)
      sTitle = "Plot Outcrop: ";

    if (plotframe != null)
    {
      iFrameX      = plotframe.getLocation().x;
      iFrameY      = plotframe.getLocation().y;
      iFrameWidth  = plotframe.getSize().width;
      iFrameHeight = plotframe.getSize().height;
      plotframe.close();
      plotframe = null;
    }

    plotframe = new iqstratPlotFrame( this,
                                      stStruct,
                                      iSelected,
                                      iControlWidth,
                                      iFrameX,
                                      iFrameY,
                                      iFrameWidth,
                                      iFrameHeight );

    plotframe.setTextfield(txtCursorDepth);
    plotframe.setHeaders( stHeader );

    // LAS File Data

    plotframe.setLASFileData(stLASFileData);       // LAS File Data structure
    plotframe.setLASRockColumn( stLASLithology );  // Rock Lithology from LAS
    plotframe.setLASRockColumnByRT( stRTLithology );  // Rock Lithology from LAS by Rhomaa-DT
    plotframe.setColorlith(iColorlith);            // Colorlith identifier

    // Pfeffer Data

    plotframe.setPfeffer( stPfeffer );             // Default Pfeffer Data
    plotframe.setPfefferList( stFlowUnits );       // User Defined Flow Units

    // Horizons Data

    plotframe.setTops( stStrat );                  // Strat Units list structure
    plotframe.setSequence( stSequence );           // Sequence List Structure
    plotframe.setTops( stOther );                  // Perforations & DST Data

    plotframe.setDepEnvrionment( stDepEnv );       // Depositional Environment Data
    plotframe.setBioStrat( stBio );                // Bio Stratigraphy Data

    // Brine Data

    plotframe.setBrine( stBrine );                 // Brine Data List Structure

    // Measured Rock Data

    plotframe.setRockData( stCore );               // Rock Data list structure
    plotframe.setRockImages( stImages );           // Rock Image List Structure
    plotframe.setRockColumn( stColumn );           // Rock Column List Structure
//    plotframe.setRockLithology( stPlotLithology ); // Rock Lithology structure
//    plotframe.setTexture( stPlotTexture );         // Rock Texture structure
    plotframe.setPHI( stPHI );                     // Rock Porosity Type structure
    plotframe.setFossils( stFossil );              // Fossils list structure
    plotframe.setStructures( stSedimentary );      // Sed Structures structure
    plotframe.setRockColorData(stRockColor);       // Rock color list structure
    plotframe.setRemarks( stRemarks );             // Remarks list structure

    plotframe.setRockFileData( stRockData );

    modifyWidths(); // Modify the plot width based on the fossils and sturctures

    // -- Thin Track Entry Dialog

    plotframe.setPHI( iTHINPHI );

    // -- Resistivity & Porosity Image Track Entry Dialog

    plotframe.setImageTrackFrame( pnlImageTrack );
    plotframe.setOHM( iOHMTool, dOHMMax, dOHMMin );
    plotframe.setPHI( iPHITool, dPHIMax, dPHIMin );

    // -- Add Convert Counts to Log Dialog

    plotframe.setConvertFrame( pConvert );

    // Set Observable Object

    plotframe.setObservable(notifier);             // Observable Notifier
    plotframe.setCommon(stStruct.stCMN);           // Global Common Data Struct

    plotframe.toFront();
  }

  /* ===================================================================== *
   * ------------ Profile Plot Control Plot Limits DIALOG ---------------- *
   * ===================================================================== */

  /** Method Find_Plot_Limits()
   *  <p> This method will allow the user to change the plot limits for
   *      the Profile Plots
   *  @param iData = Plot Limits Identifier
   */

  private void Find_Plot_Limits(int iData)
  {
	this.iData = iData;

    // LAS & PfEFFER Plot Track Limits

    if (orig_las == null)
    {
	  orig_las = las.lasPlotLimitsUtility.initialize(stLASFileData);
	  data_las = las.lasPlotLimitsUtility.copy(orig_las);
	}

    if (orig_core == null)
    {
	  orig_core = rock.rockPlotLimitsUtility.initialize(stCore);
	  data_core = rock.rockPlotLimitsUtility.copy(orig_core);
	}

    if (orig_pf == null)
    {
	  orig_pf = pfeffer.pfefferPlotLimitsUtility.initialize();
	  data_pf = pfeffer.pfefferPlotLimitsUtility.copy(orig_pf);
	}

    switch (iData)
    {
	  case _C_LAS:
        if (pLimitsLAS != null)
          pLimitsLAS.close();
        pLimitsLAS  = null; // LAS Plot Track Limits Dialog

        pLimitsLAS = new lasPlotLimitsFrame( notifier, orig_las, data_las );
	    break;
	  case _C_CORE:
        if (pLimitsCORE != null)
          pLimitsCORE.close();
        pLimitsCORE  = null; // CORE Plot Track Limits Dialog

        pLimitsCORE = new rockPlotLimitsFrame( notifier, orig_core, data_core );
	    break;
	}
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method moveUp()
   * <p> Move the selected track up one row
   */

  private void moveUp()
  {
    int iRow = pTracks.getSelectedRow();
    int iUp  = iRow-1;
    int iTrack = -1;

    if (iUp > -1)
    {
      iTrack          = iSelected[iRow];
      iSelected[iRow] = iSelected[iUp];
      iSelected[iUp]  = iTrack;
    }

    stControl.iTracks = iSelected;
    pTracks.setData( iSelected );
    if (iUp > -1) pTracks.setSelectedRow( iUp );
    if (plotframe != null) { plotframe.setSelected( iSelected ); }
  }

  /** Method moveDown()
   * <p> Move the selected track down one row
   */

  private void moveDown()
  {
    int iRow   = pTracks.getSelectedRow();
    int iDown  = iRow+1;
    int iTrack = -1;

    if (iDown < iSelected.length)
    {
      iTrack           = iSelected[iRow];
      iSelected[iRow]  = iSelected[iDown];
      iSelected[iDown] = iTrack;
    }

    stControl.iTracks = iSelected;
    pTracks.setData( iSelected );
    if (iDown < iSelected.length) pTracks.setSelectedRow( iDown );
    if (plotframe != null) { plotframe.setSelected( iSelected ); }
  }

  /** Method save()
   * <p> This method will save the data to the PC and/or KGS database.
   */

  private void save()
  {
    if (p3File != null)
      p3File.close();
    p3File = null;

    p3File = new las3FileFrame( notifier,        // Observables Object
                                stStruct,        // Global IQSTRAT Data Struct
                                stFiles,         // LAS File Information
                                stHeader,        // Header Information
                                stLASFileData,   // LAS File Data
                                stCore,          // Measured Rock Data
                                stBrine,         // Brine Data
                                stImages,        // Rock Images List
                                stStrat,         // Stratigraphic Units List
                                stSequence,      // Sequence Stratigraphy List
                                stDepEnv,        // Depositional Environment List
                                stOther,         // Perforation Data List
                                stBio,           // Bio-Stratigraphy List
                                stDST,           // DST Data
                                getControl(),    // Plot Control Data
                                stShale,         // Shale Data List Structure
                                stFlowUnits,     // Pfeffer Data List Structure
                                stRemarks );     // Remarks Data List Structure
  }

  /** Method checkData()
   * <p> This method will check if core or remarks data is present and allows
   *     the user to ignore certain data types in the LAS 3 Save
   * @return true if core and or remarks data types are present false otherwise
   */

  private boolean checkData()
  {
    boolean bval = false;

    if (stRemarks != null)
    {
      if (stRemarks.iCount > 0)
        bval = true;
    }

    if (stCore != null)
    {
      if (stCore.iCount > 0)
        bval = true;
    }

    return (bval);
  }

  /** Method changeScale()
   * <p> This method will change the depth Scale.
   * @param iScale = The Depth Scale
   */

  private void changeScale(int iScale)
  {
    this.iScale = iScale;

    lblScale.setText("Depth Scale: "+ scale[iScale]);

    setScale(iScale, dEnd, dStart);
  }

  /** Method createPDF()
   *  <p> This method will create a PDF Image of Profile Plot
   */

  private void createPDF()
  {
    iqstrat.io.iqstratIO_Files.createImage( stStruct, plotframe.getImage() );
    iqstrat.io.iqstratIO_Files.createHTML( stStruct );
//    iqstrat.io.iqstratIO_PDF.createPDF( stStruct );

    if (plotframe != null)
    {
      iqstrat.io.iqstratIO_Files.createDRA( stStruct,
                                            stHeader,
                                            plotframe.getPlotWidth(),
                                            plotframe.getPlotHeight(),
                                            dStart,
                                            dEnd );
    }

    stStruct.sDirectory = new String( "" );
    stStruct.sName      = new String( "" );
  }

  /** Method showMemory()
   * <p> This method will display the available memory Dialog
   */

  private void showMemory()
  {
    if (pMemory != null)
      pMemory.close();
    pMemory = null;

    pMemory = new guiMemoryFrame();
    pMemory.setMemory();
  }


  /** Method closeOtherOBSData()
   * <p> This method will release the well or outcrop data to load a new data
   *     set.
   */

  public void closeOtherOBSData()
  {
	if (stColumn != null)
	  stColumn.delete();
	stColumn = null;

    if (stRockLithology != null)
      stRockLithology.delete();
    stRockLithology = null; // Rock Column

    if (stPlotLithology != null)
      stPlotLithology.delete();
    stPlotLithology = null; // Rock Column (plot)

    if (stTexture != null)
      stTexture.delete();
    stTexture    = null;    // Rock Texture

    if (stPlotTexture != null)
      stPlotTexture.delete();
    stPlotTexture = null;   // Rock Texture (plot)

    if (stPHI != null)
      stPHI.delete();
    stPHI = null;           // Porosity Track data

    if (stFossil != null)
      stFossil.delete();
    stFossil = null;        // Fossils

    if (stBio != null)
      stBio.delete();
    stBio = null;           // Bio Stratigraphy Data

    if (stSedimentary != null)
      stSedimentary.delete();
    stSedimentary = null;   // Sedimentary Structure

    if (stRockColor != null)
      stRockColor.delete();
    stRockColor = null; // Rock Color Data
  }

  /** Method closeOBSData()
   * <p> This method will release the well or outcrop data to load a new data
   *     set.
   */

  public void closeOBSData()
  {
    if (stRockLithology != null)
      stRockLithology.delete();
    stRockLithology = null; // Rock Column

    if (stPlotLithology != null)
      stPlotLithology.delete();
    stPlotLithology = null; // Rock Column (plot)

    if (stTexture != null)
      stTexture.delete();
    stTexture    = null;    // Rock Texture

    if (stPlotTexture != null)
      stPlotTexture.delete();
    stPlotTexture = null;   // Rock Texture (plot)

    if (stPHI != null)
      stPHI.delete();
    stPHI = null;           // Porosity Track data

    if (stFossil != null)
      stFossil.delete();
    stFossil = null;        // Fossils

    if (stBio != null)
      stBio.delete();
    stBio = null;           // Bio Stratigraphy Data

    if (stSedimentary != null)
      stSedimentary.delete();
    stSedimentary = null;   // Sedimentary Structure

    if (stRockColor != null)
      stRockColor.delete();
    stRockColor = null;     // Rock Color Data

    if (stRemarks != null)
      stRemarks.delete();
    stRemarks = null;       // Remarks
  }

  /** Method closeData()
   * <p> This method will release the well or outcrop data to load a new data
   *     set.
   */

  public void closeData()
  {
    if (stHeader != null)
      stHeader.delete();
    stHeader = null;

    if (stFiles != null)
      stFiles.delete();
    stFiles = null;

    // LAS Data

    if (stLAS3Data != null)
      stLAS3Data.delete();
    stLAS3Data = null;  // LAS 3 File Data List Structure

    if (stLASFileData != null)
      stLASFileData.delete();
    stLASFileData  = null; // LAS File Data Structure

    if (stLASLithology != null)
      stLASLithology.delete();
    stLASLithology  = null; // LAS Rock Column

    if (stRTLithology != null)
      stRTLithology.delete();
    stRTLithology = null;

    if (stLASTexture != null)
     stLASTexture.delete();
    stLASTexture   = null; // Rock Texture

    if (stShale != null)
      stShale.delete();
    stShale = null;

    if (stPfeffer != null)
      stPfeffer.delete();
    stPfeffer = null;

    if (stFlowUnits != null)
      stFlowUnits.delete();
    stFlowUnits = null;

    // Horizons Data

    if (stTops != null)
      stTops.delete();
    stTops    = null;    // Formation Tops

    if (stStrat != null)
      stStrat.delete();
    stStrat = null;      // Stratigraphic Units

    if (stSequence != null)
      stSequence.delete();
    stSequence = null;  // Sequence Stratigraphy

    if (stOther != null)
      stOther.delete();
    stOther = null; // DST & Perforations

    if (stDepEnv != null)
      stDepEnv.delete();
    stDepEnv = null;  // Depositonal Environment

    if (stBio != null)
      stBio.delete();
    stBio = null; // Bio Stratigraphy Data

    // Brine Data

    if (stBrine != null)
      stBrine.delete();
    stBrine = null; // Brine Data

    // .. DST Data

    if (stDST != null)
      stDST.delete();
    stDST      = null; // DST List Structure

    // Rock Data

    if (stCore != null)
      stCore.delete();
    stCore = null;      // Measured Rock Data List Structure

    if (stImages != null)
      stImages.delete();
    stImages = null;    // Rock Images List Structure

    if (stRockColor != null)
      stRockColor.delete();
    stRockColor = null; // Rock Color Data

    if (stRockLithology != null)
      stRockLithology.delete();
    stRockLithology = null; // Rock Column

    if (stPlotLithology != null)
      stPlotLithology.delete();
    stPlotLithology = null; // Rock Column (plot)

    if (stTexture != null)
      stTexture.delete();
    stTexture    = null;    // Rock Texture

    if (stPlotTexture != null)
      stPlotTexture.delete();
    stPlotTexture = null;   // Rock Texture (plot)

    if (stPHI != null)
      stPHI.delete();
    stPHI = null;           // Porosity Track data

    if (stFossil != null)
      stFossil.delete();
    stFossil = null;        // Fossils

    if (stSedimentary != null)
      stSedimentary.delete();
    stSedimentary = null;   // Sedimentary Structure

    if (stRemarks != null)
      stRemarks.delete();
    stRemarks = null;       // Remarks

    if (stColumn != null)
      stColumn.delete();
    stColumn        = null; // Rock Column Data List

    // Image Track Variables

    dRt   = null;
    dPhit = null;
    dPhi  = null;
  }

  /** Method closeTopsDataEntry()
   * <p> This method will close the tops data entry dialog
   */

  public void closeTopsDataEntry()
  {
    if (pTopsEntry != null)
      pTopsEntry.close();
    pTopsEntry = null;     // Horizon Data Entry Panel
  }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier = null;

    // Variables passed to this panel

    iqstratStruct stStruct = null;

    closeData(); // Delete All Data structures

    // Profile Plot Limits

    orig_las  = null; // Original Plot Limits for LAS
    orig_pf   = null; // Original Plot Limits for PfEFFER
    orig_core = null; // Original Plot Limits for CORE

    data_las  = null; // Plot Limits for LAS
    data_pf   = null; // Plot Limits for PfEFFER
    data_core = null; // Plot Limits for CORE

    if (pSearch != null)
      pSearch.close();
    pSearch = null;

    // Available Data Frames

    if (pnlDefaultList != null)
      pnlDefaultList.close();
    pnlDefaultList = null; // Available Tracks List

    if (pTracks != null)
      pTracks.close();
    pTracks      = null;

    iStatus        = null; // Status of each button
    iSelected      = null; // Selected Tracks

    if (pMemory != null)
      pMemory.close();
    pMemory = null;

    // Data Entry Frames

    if (pHeaders != null)
      pHeaders.close();
    pHeaders = null;    // Headers Data

    closeTopsDataEntry();

    if (p3File != null)
      p3File.close();
    p3File = null;

    if (pnlThinTrack != null)
      pnlThinTrack.close();
    pnlThinTrack = null;

    if (pnlImageTrack != null)
      pnlImageTrack.close();
    pnlImageTrack = null;

    if (pConvert != null)
      pConvert.close();
    pConvert = null;

    if (pLimitsLAS != null)
      pLimitsLAS.close();
    pLimitsLAS  = null; // LAS Plot Track Limits Dialog

    if (pLimitsCORE != null)
      pLimitsCORE.close();
    pLimitsCORE  = null; // CORE Plot Track Limits Dialog

    // Menus

    mSave         = null; // Save Data
    mPDF          = null; // PDF of Profile Plot
    memory        = null; // Show Memory Usage
    mExit         = null; // Exit
    mHelp         = null; // Help

    // -- Depth Scale Buttons

    lblScale   = null;

    m1         = null;         //   1 foot / inch
    m2         = null;         //   2 feet / inch
    m5         = null;         //   5 feet / inch
    m10        = null;         //  10 feet / inch
    m20        = null;         //  20 feet / inch
    m50        = null;         //  50 feet / inch
    m100       = null;         // 200 feet / inch
    m200       = null;         // 200 feet / inch

    // Text Fields

    txtCursorDepth = null;  // Mouse Cursor Depth Indicator
    txtStart   = null;      // Starting Depth
    txtEnd     = null;      // Ending Depth

    // Depth Range Buttons

    btnReset   = null;
    btnModify  = null;

    // Well Information Plot Data

    lblName   = null;
    lblAPI    = null;
    lblStatus = null;
    lblLat    = null;
    lblLong   = null;
    lblTD     = null;
    lblElev   = null;

    btnHeader = null;

    btnLimitsLAS  = null;
    btnLimitsCORE = null;

    // Plot Track Buttons

    btnDefault = null;
    btnUser    = null;

    btnUp      = null;
    btnDown    = null;

    // Available Track Widgets

    pnlPlotTracks = null;
    cardLayout = null;

    // Destroy the plot

    if (plotframe != null)
      plotframe.close();
    plotframe = null;   // Plot Frame

    dispose();
  }

  /** METHOD lasControlPlotFrame_WindowListener()
   *  <p> This Class will handle Frame Close Action using the X Button on Frame
   */

  public class lasControlPlotFrame_WindowListener extends WindowAdapter
  {
    public void windowClosing(WindowEvent e)
    {
      close();
    }
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    String sEventArgument = event.getActionCommand();
    String sPath = "";

    if (pMemory != null) pMemory.setMemory();

    // Memory Usage

    if (event.getSource() == memory) { showMemory(); }

    // Display Header Information Frame

    if (event.getSource() == btnHeader)
    {
      if (pHeaders != null)
        pHeaders.close();
      pHeaders = null;

      pHeaders = new iqstratHeadersFrame(stStruct.stCMN, notifier, stHeader);
    }

    // Set Depth Scale

    if (event.getSource() == m200)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_200);
    }
    if (event.getSource() == m100)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_100);
    }
    if (event.getSource() == m50)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_50);
    }
    if (event.getSource() == m20)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_20);
    }
    if (event.getSource() == m10)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_10);
    }
    if (event.getSource() == m5)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_5);
    }
    if (event.getSource() == m2)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_2);
    }
    if (event.getSource() == m1)
    {
      changeScale(iqstrat.iqstratTracksStruct.SCALE_1);
    }

    if (event.getSource() == btnLimitsLAS)  { Find_Plot_Limits(_C_LAS); }
    if (event.getSource() == btnLimitsCORE) { Find_Plot_Limits(_C_CORE); }

    // Set Depths

    if (event.getSource() == btnReset)  { setDefaultDepth(); }
    if (event.getSource() == btnModify) { setDepth(); }

    // Change Panels

    if (event.getSource() == btnDefault) { setOrderPanel( _DEFAULT ); }
    if (event.getSource() == btnUser)    { setOrderPanel( _USER ); }

    setTrackPanelColors(); // Set Track Button Colors

    if (event.getSource() == btnUp)   { moveUp(); }
    if (event.getSource() == btnDown) { moveDown(); }

    setButtons();          // Set Frame Buttons

    // File Menu

    if (event.getSource() == mSave)   { save(); }

    if (event.getSource() == mPDF)
    {
      iDocument = _PDF;
      sPath               = kgs.io.kgsIOUser.getHomeDir();
      stStruct.sDirectory = new String( sPath );
      stStruct.sName      = new String( "Profile" );

      pSearch = new guiSearchDirectoryFrame(notifier, sPath, "Profile", ".png");
    }

    // Help

    if (event.getSource() == mHelp)
    {
      util.BrowserControl.displayURL( cmn.cmnStruct.HELP );
    }

    util.utilMemory.free();
    if (pMemory != null) pMemory.setMemory();
    if (pMemory  != null) pMemory.toFront();

    // Exit

    if (event.getSource() == mExit)  { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());
      }
    }

    util.utilMemory.free();
    if (pMemory != null) pMemory.setMemory();
    if (pMemory != null) pMemory.toFront();
  }

  /** Method update()
   * <p> This method will handle the Observable Actions
   * @param obj = The Observable making the call
   * @param arg = The returned object
   */

  public void update(Observable obj, Object arg)
  {
    String sArg      = new String((String) arg);
    int    i         = 0;
    String sKID      = new String("0");
    String sKEY      = new String("0");
    String sAPI      = new String("");
    String sName     = new String("");
    int    iNoData   = 0;

    if (pMemory != null) pMemory.setMemory();

    if (sArg.equals("Change Thin Porosity Data")) { ThinTrackFrame(); }
    if (sArg.equals("Thin Porosity Changed"))
    {
	  getThinPorosity( _EDIT_PANEL );
	  plotframe.refresh();
	}

    if (sArg.equals("Image Track Changed"))   { getImageTrack(); }

    /* ------------------------------------------------------------------ *
     * ----------------- Header Data Modified Action -------------------- *
     * ------------------------------------------------------------------ */

    if (sArg.equals("Well Header Information Data Selected"))
    {
      if (pHeaders != null) pHeaders.getData();
    }

    if (sArg.equals("Status Selected"))
    {
      if (pHeaders  != null) pHeaders.getStatus();
    }

    if (sArg.equals("Change Header Information"))
    {
      stHeader = iqstrat.iqstratHeadersUtility.copy(pHeaders.getWellHeader());
      setWellHeaderLabels(); // Set the Well Header Labels
      plot();

      pHeaders.close();
      pHeaders = null;
    }

    /* ------------------------------------------------------------------ *
     * -------------- Available Plot Track Panel Actions ---------------- *
     * ------------------------------------------------------------------ */

    if (sArg.equals("Get LAS Track Size")) { getLASTrackSize(); }
    if (sArg.equals("Available Track Panel Changed"))
    {
      getSelected();
      if (plotframe != null)
      {
        plotframe.setSelected(iSelected);
      }
    }

    if (sArg.equals("Available Track Panel - Group Changed"))
    {
      pnlDefaultList.setByGroup( stLASFileData,
                                 stStrat,
                                 stBrine,
                                 stCore,
                                 stImages,
                                 stBio,
                                 stColumn,
                                 stPHI,
                                 stFossil,
                                 stSedimentary,
                                 stRockColor,
                                 stRemarks );

      getThinPorosity( _AVAILABLE );
      getImageTrack( _AVAILABLE, lith.gui.lithImageTrackFrame._RT );
      getImageTrack( _AVAILABLE, lith.gui.lithImageTrackFrame._PHIT );

      if (notifier != null)
        notifier.notifyObservers(new String("Available Track Panel Changed"));
	}

    if (sArg.equals("Available Track Panel - Data Changed"))
    {
      pnlDefaultList.setByGroup( stLASFileData,
                                 stStrat,
                                 stBrine,
                                 stCore,
                                 stImages,
                                 stBio,
                                 stColumn,
                                 stPHI,
                                 stFossil,
                                 stSedimentary,
                                 stRockColor,
                                 stRemarks );

      if (notifier != null)
        notifier.notifyObservers(new String("Available Track Panel Changed"));
	}

    /* ------------------------------------------------------------------ *
     * ----------------------- TOPS Track Actions ----------------------- *
     * ------------------------------------------------------------------ */

    if (sArg.equals("Select Well Stratigraphic Units"))
    {
      getData( _STRAT_UNIT, _TOPS );

      if (plotframe != null) { plotframe.refresh(); }
    }

    if (sArg.equals("Stratigraphic Units modified"))
    {
      getData( _TOPS_ENTRY, _TOPS );

      if (plotframe != null) { plotframe.refresh(); }
    }

    /* ================================================================== *
     * -------------------- CONVERT Log Count Actions ------------------- *
     * ================================================================== */

    if (sArg.equals("Log Values Converted"))
    {
      getData( _CONVERT, _LAS );
    }

    setButtons(); // Set Frame Buttons

    /* ------------------------------------------------------------------ *
     * ----------------------- Directory Actions ------------------------ *
     * ------------------------------------------------------------------ */

    if (sArg.equals("Directory Path Selected"))
    {
      switch (iDocument)
      {
        case _HTML:
          break;
        case _PDF:
          if (pSearch != null)
          {
            stStruct.sDirectory = new String( pSearch.getPath() );
            stStruct.sName      = new String( pSearch.getFile() );
          }

          createPDF();
          break;
      }

      if (pSearch != null)
        pSearch.close();
      pSearch = null;
    }

    if (sArg.equals("Directory Path Cancel"))
    {
      switch (iDocument)
      {
        case _HTML:
          break;
        case _PDF:
          createPDF();
          break;
      }

      if (pSearch != null)
        pSearch.close();
      pSearch = null;
    }

    if (sArg.equals("Display Horizon Data Entry"))
    {
      closeTopsDataEntry();

      pTopsEntry = new horizonEnterDataFrame( notifier, stStruct );
      pTopsEntry.setData( stHeader );
      pTopsEntry.setTopsData( stStrat );
      pTopsEntry.setSeqList( stSequence );
      pTopsEntry.setEnvList( stDepEnv );

      if (plotframe != null)
      {
        pTopsEntry.setDataType( plotframe.getDataEntryPanel() );
        pTopsEntry.setDepthRange( plotframe.getStartingDepth(),
                                  plotframe.getEndingDepth() );
      }
    }

    if (sArg.equals("LAS Plot Limits Changed"))
    {
	  if (pLimitsLAS != null)
	  {
        data_las = las.lasPlotLimitsUtility.copy(pLimitsLAS.getLASLimits());
        orig_las = las.lasPlotLimitsUtility.copy(pLimitsLAS.getLASOriginal());

        plotframe.setPlotLimits(data_las, data_core, data_pf);
	  }
	}

    if (sArg.equals("CORE Plot Limits Changed"))
    {
	  if (pLimitsCORE != null)
	  {
        data_core = rock.rockPlotLimitsUtility.copy(pLimitsCORE.getCORELimits());
        orig_core = rock.rockPlotLimitsUtility.copy(pLimitsCORE.getCOREOriginal());

        plotframe.setPlotLimits(data_las, data_core, data_pf);
	  }
	}

    if (sArg.equals("Close LAS Plot Limits"))
    {
      if (pLimitsLAS != null)
      {
        pLimitsLAS.close();
	  }
      pLimitsLAS  = null; // LAS Plot Track Limits Dialog
	}

    if (sArg.equals("Close CORE Plot Limits"))
    {
      if (pLimitsCORE != null)
      {
        pLimitsCORE.close();
	  }
      pLimitsCORE  = null; // CORE Plot Track Limits Dialog
	}

    if (sArg.equals("MEMORY CHANGE")) {}

    util.utilMemory.free();
    if (pMemory != null) pMemory.setMemory();
    if (pMemory != null) pMemory.toFront();
  }
}

/** CLASS lasControlPlotFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class lasControlPlotFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  lasControlPlotFrame adaptee;

  lasControlPlotFrameFocusAdapter(lasControlPlotFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/** CLASS lasControlPlotFrameObservable()
  *  <p> This Class will handle Observable Actions when action occurs on a
  *      frame or panel
  */

class lasControlPlotFrameObservable extends Observable
{
  public void notifyObservers(Object b)
  {
    setChanged();
    super.notifyObservers(b);
  }
}

/*
 *  @version 1.1 06/06/2014
 *  @author  John Victorine
 */
