/*
 * @kgsIORead.java Version 1.1 10/19/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.io;

import java.io.*;
import java.awt.*;
import javax.swing.*;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratHeadersListStruct;
import iqstrat.io.ReadHeadersXMLFile;
import iqstrat.io.WriteHeadersXMLFile;

import xml.ReadLASFileXMLFile;
import cmn.cmnLASFileListStruct;

import horizon.regions.io.ReadRegionsXMLFile;
import horizon.regions.regionsListStruct;

import horizon.strat.io.ReadStratUnitsXMLFile;
import horizon.strat.stratListStruct;

import rock.io.ReadRockImageXMLFile;
import rock.rockImagesListStruct;

import rock.io.ReadRockDataXMLFile;
import rock.rockDataListStruct;

/** Class kgsIORead
 *  <p> This Class will retrieve the Oil & Gas Well Data from the KGS Database.
 *      This class will retrieve the following information,
 *
 *      - LAS File Information by API-Number
 *      - Formation Regions, Tops, Perforation zones, DST Measurements
 *      - Measured Core Data
 *      - Core Images
 *
 *  @version 1.1 10/19/2007
 *  @author  John R. Victorine
 */

public class kgsIORead
{
  /** Constructor kgsIORead()
   *  <p> This is the Constructor for this class.
   */

  public kgsIORead() { }

  /* ===================================================================== *
   * ---------------------------- XML Methods ---------------------------- *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will read the User information xml data
   * @param  iMethod = The Method used in retrieving the XML
   * @param  sPath   = The url or File Path to use
   * @return stUser  = The User Data Structure
   */

  public static iqstratHeadersListStruct getData(int iMethod, String sPath)
  {
    iqstratHeadersListStruct st = null; // headers Information Data Structure

    ReadHeadersXMLFile pLocInfo = new ReadHeadersXMLFile(iMethod);
    st = pLocInfo.Process(sPath);

    String sError = pLocInfo.GetError();
    if (sError.length() > 0)
      JOptionPane.showMessageDialog( (Component)null,
                                     sError,
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);

    return (st);
  }

  /** Method ReadLASFileInformation()
   * <p> This method will retrieve the LAS File Information
   *     for a specific well from the database
   * @param  sAPI    = API-Number of the Well.
   * @param  iMethod = The Method used in retrieving the XML
   * @return st  = The LAS File Information Data
   */

  public static cmnLASFileListStruct ReadLASFileInformation(String sAPI,
                                                            int    iMethod)
  {
    cmnLASFileListStruct st = new cmnLASFileListStruct();

    ReadLASFileXMLFile pXML = new ReadLASFileXMLFile(iMethod);
    st = pXML.Process(cmn.cmnStruct.KGS_LAS_FILES + "sAPI=" + sAPI);

    return (st);
  }

  /** Method ReadWellRegions()
   * <p> This method will retrieve the Well Regions for a specific well
   * @param  sAPI    = API-Number of the Well.
   * @param  iMethod = The Method used in retrieving the XML
   * @return st      = The Regions List Data Structure
   */

  public static regionsListStruct ReadWellRegions(String sAPI, int iMethod)
  {
    regionsListStruct st = null;

    ReadRegionsXMLFile pXML = new ReadRegionsXMLFile(iMethod);
    st = pXML.Process(cmn.cmnStruct.KGS_REGIONS + "sAPI="+sAPI);

    return (st);
  }

  /** Method ReadWellRegions()
   * <p> This method will retrieve the Well Stratigraphic Units for a
   *     specific well
   * @param  sAPI    = API-Number of the Well.
   * @param  sKID    = Primary KEY of the Well
   * @param  iMethod = The Method used in retrieving the XML
   * @return st      = The Stratigraphic Units List Data Structure
   */

  public static stratListStruct ReadWellRegions( String sAPI,
                                                 String sKID,
                                                 int iMethod )
  {
    stratListStruct st    = null;
    stratListStruct stKGS = null;

    ReadStratUnitsXMLFile pXML = new ReadStratUnitsXMLFile(iMethod);
    st = pXML.Process( cmn.cmnStruct.KGS_WELL_STRAT_UNITS +
                       "sAPI="+sAPI+"&sKID="+sKID);

    stKGS = ReadKGSStratUnits( iMethod );
    st    = horizon.strat.stratUtility.fillBaseDepth(st, stKGS);

    return (st);
  }

  /** Method ReadKGSStratUnits()
   * <p> This method will retrieve the Stratigraphic Units for Kansas
   * @param  iMethod = The Method used in retrieving the XML
   * @return st      = The Stratigraphic Units List Data Structure
   */

  public static stratListStruct ReadKGSStratUnits( int iMethod )
  {
    stratListStruct st = null;

    ReadStratUnitsXMLFile pXML = new ReadStratUnitsXMLFile(iMethod);
    st = pXML.Process(cmn.cmnStruct.KGS_STRAT_UNITS);

    return (st);
  }

  /** Method ReadCoreData()
   * <p> This method will retrieve the Core Data
   *     for a specific Well from the database
   * @param sAPI  = API-Number of the Well.
   * @param iMethod = The Method used in retrieving the XML
   * @return st     = the Rock Catalog Data List Data Structure
   */

  public static rockDataListStruct ReadCoreData(String sAPI, int iMethod)
  {
    rockDataListStruct st = null;

    ReadRockDataXMLFile pXML = new ReadRockDataXMLFile(iMethod);
    st = pXML.Process(cmn.cmnStruct.KGS_CORE_DATA + "sAPI=" + sAPI);

    return (st);
  }

  /** Method ReadCoreImages()
   * <p> This method will retrieve the Core Image File Information
   *     for a specific well from the database
   * @param sAPI  = API-Number of the Well.
   * @param  iMethod = The Method used in retrieving the XML
   * @return st      = the Core Image File Information list data structure
   */

  public static rockImagesListStruct ReadCoreImages(String sAPI, int iMethod)
  {
    rockImagesListStruct st = null;

    ReadRockImageXMLFile pXML = new ReadRockImageXMLFile(iMethod);
    st = pXML.Process(cmn.cmnStruct.KGS_CORE_IMAGES + "sAPI=" + sAPI);

    return (st);
  }
}