/*
 * @ReadMeasSectXMLFile.java Version 1.0 09/27/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import kgs.kgsMeasSectListStruct;
import kgs.kgsMeasSectStruct;
import kgs.kgsMeasSectDataStruct;

import cmn.cmnString;

/** Class ReadMeasSectXMLFile()
 * <p> This Class will parse the Measured Section Information XML Data Stream
 *     directly to the Measured Section Information List Data Structure.
 *
 *  @version 1.1 09/27/2009
 *  @author  John R. Victorine
 */

public class ReadMeasSectXMLFile extends DefaultHandler
{
  public static final int FILE     = 0;
  public static final int URL      = 1;
  public static final int SERVER   = 2;
  public static final int SERVER_ZIP = 3;

  public static final int NOERROR    =  0;
  public static final int ERROR      = -1;

  public static final String ROOTNAME     = "outcrop";   // rootname
  public static final String RECORDS      = "records";   // Total Wells
  public static final String LOCATION     = "location";  // ELEMENT info

  // Identification Information

  public static final String INFO         = "info";      // ELEMENT info

  public static final String KID          = "mid";       // Primary key
  public static final String NAME         = "name";      // outcrop name
  public static final String ERROR_VAL    = "error";     // DB Error

  // Location Information

  public static final String STATE        = "state";     // State Name
  public static final String STATE_CODE   = "state_cd";  // State Name
  public static final String COUNTY       = "county";    // County Name
  public static final String COUNTY_CODE  = "county_cd"; // County Name

  public static final String TOWNSHIP     = "town";      // township
  public static final String TOWNSHIP_DIR = "town_dir";  // township dir
  public static final String RANGE        = "range";     // range
  public static final String RANGE_DIR    = "range_dir"; // range direction
  public static final String SECTION      = "section";   // section

  // XY Position Information

  public static final String LATITUDE     = "latitude";  // Latitude
  public static final String LONGITUDE    = "longitude"; // Longitude

  // Z Position Information

  public static final String GROUND_LEVEL = "gl";        // Ground Level

  // Misc Information

  public static final String COMMENTS     = "comments";  // Comments
  public static final String MEASURED     = "meas_by";   // Measured By

  public static final String DATA         = "data";      // ELEMENT data

  public static final String KEY          = "key";       // Primary Key of Unit
  public static final String UNIT_ORDER   = "order";     // Unit Order
  public static final String UNIT_NO      = "no";        // Unit Number
  public static final String DESCRIPTION  = "desc";      // Bed Description
  public static final String THICKNESS    = "thickness"; // Thickness
  public static final String UNITS        = "units";     // Thickness Unit

  // Global Variables;

  private int    iError = NOERROR;
  private String sError = new String("");

  private static final int _BEGIN    = 0;
  private static final int _LOCATION = 1;
  private static final int _INFO     = 2;
  private static final int _COMMENTS = 3;
  private static final int _DATA     = 4;

  private int                   iData     = _BEGIN;
  private int                   iType     = FILE;
  private int                   iSections = 0;
  private int                   iBeds     = 0;
  private int                   iCount    = -1;
  private int                   iRows     = -1;
  private int                   iComments = 0;
  private kgsMeasSectListStruct stList    = new kgsMeasSectListStruct();
  private boolean               bContinue = false;

  /** Method ReadMeasSectXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadMeasSectXMLFile() { iType = FILE; }

  /** Method ReadMeasSectXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadMeasSectXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public kgsMeasSectListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);

    if (iRows > 0) iRows++;
    stList.iRows = iRows;

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      iError = ERROR;
      sError = new String("\n** Parsing error"            +
                          ", line " + spe.getLineNumber() +
                          ", uri "  + spe.getSystemId()   + "\n" +
                          spe.getMessage());
      System.out.println(sError);

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(LOCATION))
    {
      iData     = _LOCATION;
      iComments = 0;
    }

    if (eName.equals(INFO))
    {
      iData     = _INFO;
      iComments = 0;

      iCount++;
      if (iCount < iSections)
      {
        stList.stItem[iCount] = new kgsMeasSectStruct();
      }
    }

    if (eName.equals(COMMENTS))
    {
      iData     = _COMMENTS;
      iComments = 1;
    }

    if (eName.equals(DATA))
    {
      iData     = _DATA;
      iComments = 0;

      if (!bContinue)
      {
        iRows++;
        if (iRows < iBeds)
        {
          stList.stData[iRows] = new kgsMeasSectDataStruct();
        }
      }
    }

    // Add Attributes to the Measured Section Data List Structure

    if (attrs != null)
    {
      for (int i = 0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          switch (iData)
          {
            case _BEGIN:
              iSections = cmn.cmnString.stringToInt(sOut.trim());

              if (iSections > 0)
              {
                stList.stItem = new kgsMeasSectStruct[iSections];
                stList.iCount = iSections;
              }
              break;

            case _LOCATION:
              iBeds = cmn.cmnString.stringToInt(sOut.trim());

              if (iBeds > 0)
              {
                stList.stData = new kgsMeasSectDataStruct[iBeds];
                stList.iRows  = iBeds;
              }
              break;
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          switch (iData)
          {
            case _INFO:
              if (iCount < iSections)
                stList.stItem[iCount] = ParseLoc( aName,
                                                  sOut.trim(),
                                                  stList.stItem[iCount] );
              break;

            case _DATA:
              if (iRows < iBeds)
                stList.stData[iRows] = ParseData( aName,
                                                  sOut.trim(),
                                                  stList.stData[iRows] );
              break;
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    String s = new String(buf, offset, len);
    if (iComments == 1)
    {
      stList.stItem[iCount].sComments = new String(
          stList.stItem[iCount].sComments + " " + s.trim());
    }
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException
  {
    iError    = ERROR;
    sError    = new String(e.toString());
    throw e;
  }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    iError = ERROR;
    sError = new String("** Warning" +
                        ", line "    + err.getLineNumber() +
                        ", uri "     + err.getSystemId()   + "\n" +
                        err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseLoc()
   * <p> This method will parse the Data Stream for the Individual Measured
   *     Sections Information
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Measured Section Information String
   * @param  st          = The Measured Section Information Data Structure
   * @return st          = The Measured Section Information Data Structure
   */

  private kgsMeasSectStruct ParseLoc( String sIdentifier,
                                      String sData,
                                      kgsMeasSectStruct st)
  {
    // Identification Information

    if (sIdentifier.equals(KID))  { st.sKID = new String(sData); }
    if (sIdentifier.equals(NAME)) { st.sName = new String(sData); }

    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = ERROR;
      sError    = new String(sData);
      st.sError = new String(sError);
    }

    // Location Information

    if (sIdentifier.equals(STATE)) { st.state = new String(sData); }
    if (sIdentifier.equals(STATE_CODE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iState = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(COUNTY)) { st.sCounty = new String(sData); }
    if (sIdentifier.equals(COUNTY_CODE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iCounty = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(TOWNSHIP))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iTownship = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(TOWNSHIP_DIR)) { st.sTownship = new String(sData); }
    if (sIdentifier.equals(RANGE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iRange = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(RANGE_DIR)) { st.sRange = new String(sData); }
    if (sIdentifier.equals(SECTION))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.iSection = cmn.cmnString.stringToInt(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dLongitude = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dGL = cmn.cmnString.stringToDouble(sData);
    }

    // Misc Information

    if (sIdentifier.equals(MEASURED)) { st.sMeasuredBy = new String(sData); }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Measured
   *     Sections Data
   * @param  sIdentifier = The Attribute Identifier
   * @param  sData       = Measured Section Data String
   * @param  st          = The Measured Section Data Structure
   * @return st          = The Measured Section Data Structure
   */

  private kgsMeasSectDataStruct ParseData( String sIdentifier,
                                           String sData,
                                           kgsMeasSectDataStruct st)
  {
    String sTemp = "";

    if (!bContinue)
    {
    if (sIdentifier.equals(KEY))         {st.sKEY = new String(sData);}
    if (sIdentifier.equals(UNIT_ORDER))  {st.sOrder = new String(sData);}
    if (sIdentifier.equals(UNIT_NO))
    {
      st.sUnitNo = new String(sData);

//      if (st.sUnitNo.length() > 0)
//        st.dThickness = 0.1;
    }

    if (sIdentifier.equals(THICKNESS))
    {
      if (cmn.cmnString.isNumeric(sData))
        st.dThickness = cmn.cmnString.stringToDouble(sData);
      else
        st.dThickness = getThickness(sData);
    }

    if (sIdentifier.equals(UNITS))
    {
      st.sUnits = new String(sData);
      if ((st.sUnits.equals("m.")) || (st.sUnits.equals("m")))
      {
        st.dThickness = 3.281 * st.dThickness;
        st.sUnits = new String("ft.");
      }

      if (st.sUnits.equals("in."))
      {
        st.dThickness = st.dThickness / 12.0;
        st.sUnits = new String("ft.");
      }
    }
    }

    if (sIdentifier.equals(DESCRIPTION))
    {
      sTemp           = new String( removePAREN(sData) );

      if (!bContinue) st.sDescription = new String( sTemp );
      else st.sDescription = new String( st.sDescription + " " + sTemp );

      bContinue       = checkCont( sTemp );
    }

    return (st);
  }

  /** Method getThickness()
   * <p> This method will assume that the thickness must be present and
   *     the text is a range or has a +/- descriptor
   * @param  str  = thickness string
   * @return data = thickness data
   */

  private double getThickness(String str)
  {
    double data       = 0.0;
    String tokens[]   = null;
    String sDelimiter = new String("[- ]+");
    int    iCount     = 0;
    int    iMax       = 2;
    int    iTotal     = 0;
    double dThick[]   = { 0.0, 0.0, 0.0 };

    tokens = str.split(sDelimiter);
    iCount = tokens.length;

    for (int i=0; i<iCount; i++)
    {
      if (iTotal < iMax)
      {
        if (cmn.cmnString.isNumeric(tokens[i]))
        {
          dThick[iTotal] = cmn.cmnString.stringToDouble(tokens[i]);
          iTotal++;
        }
      }
    }

    if (iTotal == 1)      { data = dThick[0]; }
    else if (iTotal == 2) { data = (dThick[0] + dThick[1]) / 2.0; }

    return (data);
  }

  /** Method removePAREN()
   * <p> This method will remove the Quotes around the String
   * @param  str  = The string to be converted
   * @return sNew = The text converted
   */

  public static String removePAREN( String str )
  {
    String sNew = new String("");
    char   c[]  = null;

    if (str != null)
    {
      c = str.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if ((c[i] != '(') && (c[i] != ')'))
          sNew = new String(sNew + c[i]);
      }
    }

    return (sNew);
  }

  /** Method checkCont()
   * <p> This method will check for the phrase "CON'T-->" in the String and
   *     return a true if it does and false if it does not.
   * @param  str    = The string to be checked
   * @return bValue = true - phrase is present; false - pharse is not present.
   */

  public static boolean checkCont( String str)
  {
    boolean bValue = false;
    String  s      = new String(str.toUpperCase());

    if (s.contains("CON'T-->"))
      bValue = true;

    return (bValue);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
