/*
 * @kgsSearchKGSHeadersFrame.java Version 1.1 11/25/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import cmn.cmnStruct;

import iqstrat.iqstratHeadersListStruct;
import iqstrat.iqstratHeadersStruct;

import kgs.gui.kgsWellListTable;

/** Class kgsSearchKGSHeadersFrame
 *  <p> This Class will allow the user to load data files from their PC or the
 *      KGS Database
 *
 *  @version 1.1 11/25/2008
 *  @author  John R. Victorine
 */

public class kgsSearchKGSHeadersFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable notifier;      // Observable Object
  private cmnStruct  stCMN = null;  // Global Common Data Structure

  // Global Frame Variables

  private int    iClose    = 0;

  private int    iTownship = 0;     // Township
  private String sTownship = "S";   // Township Direction
  private int    iRange    = 0;     // Range
  private String sRange    = "W";   // Range Direction
  private int    iSection  = 0;     // Section

  private String sPath = "";        // URL to retrieve Well List

  private iqstratHeadersListStruct stList     = null; // Headers List Data Struct
  private iqstratHeadersStruct     stHeader   = null; // Headers Structure

  private kgsWellListTable         pTable     = null; // Well List Table

  // Search By

  private static final int _API   = 0;  // Search by API Identifier
  private static final int _LEASE = 1;  // Search by Partial Lease Identifier
  private static final int _TRS   = 2;  // Search by TRS Identifier

  private int iSearch = _API;           // Search ID

  // KGS Search Widgets

  // -- Type of text search panel

  private JPanel     pnlKGSTextSearch = new JPanel();     // Search By Panel
  private CardLayout cardLayout2      = new CardLayout(); // Search By Layout

  // -- KGS - Type of Text Search

  private JRadioButton rbAPI   = new JRadioButton();  // Search by API button
  private JRadioButton rbLease = new JRadioButton();  // Search by Lease button
  private JRadioButton rbTRS   = new JRadioButton();  // Search by TRS button

  private TitledBorder titledBorderText = null;
  private JPanel       pnlKGSText       = new JPanel();

  // -- KGS - API-Number or Lease Name Search Text Fields

  private JTextField txtText     = new JTextField();

  // -- KGS - Township Range Section Search Text Fields

  // .. Township Widgets

  private JTextField txtTownship = new JTextField();
  private JRadioButton rbNorth   = new JRadioButton();
  private JRadioButton rbSouth   = new JRadioButton();

  // .. Range Widgets

  private JTextField txtRange    = new JTextField();
  private JRadioButton rbEast    = new JRadioButton();
  private JRadioButton rbWest    = new JRadioButton();

  // .. Section Widgets

  private JTextField txtSection  = new JTextField();

  // Global Frame Buttons

  private JButton btnClose  = new JButton();  // Close Dialog
  private JButton btnSearch = new JButton();  // Search KGS by input
  private JButton btnSelect = new JButton();  // Select Well

  /** Constructor kgsSearchKGSHeadersFrame()
   *  <p> This is the Constructor for this class.
   *  @param stCMN    = Common Global Data Structure
   *  @param notifier = Observable
   */

  public kgsSearchKGSHeadersFrame(cmnStruct stCMN, Observable notifier)
  {
    try
    {
      this.stCMN    = stCMN;
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlKGS         = new JPanel();
    JPanel pnlKGSTexts    = new JPanel();
    JPanel pnlKGSSearchBy = new JPanel();
    JPanel pnlKGSTRS      = new JPanel();
    JPanel pnlTownship    = new JPanel();
    JPanel pnlRange       = new JPanel();
    JPanel pnlSection     = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlBase        = new JPanel();
    JPanel pnlDataButtons = new JPanel();
    JPanel pnlSearch      = new JPanel();

    JScrollPane scroll    = new JScrollPane();

    ButtonGroup groupSearchText = new ButtonGroup();
    ButtonGroup groupTownship   = new ButtonGroup();
    ButtonGroup groupRange      = new ButtonGroup();

    TitledBorder titledBorderKGS = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Search for Data in Kansas Geological Survey Database:");
    titledBorderKGS.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderList = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "List of Oil & Gas Wells:");
    titledBorderList.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderKGSSearchBy = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Search By:");
    titledBorderKGSSearchBy.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    titledBorderText = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Enter API-Number:");
    titledBorderText.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderTownship = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Township:");
    titledBorderTownship.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderRange = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Range:");
    titledBorderRange.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSection = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Section:");
    titledBorderSection.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderLoaded = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Data Loaded:");
    titledBorderLoaded.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Search for Data on KGS Server");

    // KGS Search Panel

    pnlKGS.setBorder(titledBorderKGS);
    pnlKGS.setLayout(new BorderLayout());

    // Search for Wells

    pnlSearch.setLayout(new BorderLayout());

    btnSearch.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSearch.setActionCommand("btnSearch");
    btnSearch.setText("Search");
    btnSearch.addActionListener(this);

    // -- KGS Search Texts

    pnlKGSTexts.setBorder(titledBorderKGSSearchBy);
    pnlKGSTexts.setLayout(new BorderLayout());

    // -- Search by Text

    pnlKGSSearchBy.setLayout(new BorderLayout());

    rbAPI.setFont(new java.awt.Font("Dialog", 1, 11));
    rbAPI.setSelected(true);
    rbAPI.setText("API-Number");
    rbAPI.addActionListener(this);

    rbLease.setFont(new java.awt.Font("Dialog", 1, 11));
    rbLease.setHorizontalAlignment(SwingConstants.CENTER);
    rbLease.setText("Lease name");
    rbLease.addActionListener(this);

    rbTRS.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTRS.setText("Township Range Section");
    rbTRS.addActionListener(this);

    // Text Search Panel

    pnlKGSTextSearch.setLayout(cardLayout2);

    // -- Text Panel

    pnlKGSText.setBorder(titledBorderText);
    pnlKGSText.setLayout(new BorderLayout());

    txtText.setText("");
    txtText.setColumns(50);
    txtText.addFocusListener(new kgsSearchKGSHeadersFrameFocusAdapter(this));

    // -- TRS Panel

    pnlKGSTRS.setLayout(new GridLayout());

    // ---- Township Panel

    pnlTownship.setBorder(titledBorderTownship);
    pnlTownship.setLayout(new BorderLayout());

    txtTownship.setText("0");
    txtTownship.setColumns(4);
    txtTownship.setHorizontalAlignment(SwingConstants.TRAILING);
    txtTownship.addFocusListener(new kgsSearchKGSHeadersFrameFocusAdapter(this));

    rbNorth.setText("N");
    rbNorth.addActionListener(this);

    rbSouth.setSelected(true);
    rbSouth.setText("S");
    rbSouth.addActionListener(this);

    // ---- Range Panel

    pnlRange.setBorder(titledBorderRange);
    pnlRange.setLayout(new BorderLayout());

    txtRange.setText("0");
    txtRange.setColumns(4);
    txtRange.setHorizontalAlignment(SwingConstants.TRAILING);
    txtRange.addFocusListener(new kgsSearchKGSHeadersFrameFocusAdapter(this));

    rbEast.setSelected(true);
    rbEast.setText("E");
    rbEast.addActionListener(this);

    rbWest.setText("W");
    rbWest.addActionListener(this);

    // ---- Section Panel

    pnlSection.setBorder(titledBorderSection);
    pnlSection.setLayout(new BorderLayout());

    txtSection.setText("0");
    txtSection.setColumns(4);
    txtSection.setHorizontalAlignment(SwingConstants.TRAILING);
    txtSection.addFocusListener(new kgsSearchKGSHeadersFrameFocusAdapter(this));

    // Well List Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBorder(titledBorderList);

    // .. Well List Table

    pTable = new kgsWellListTable();
    scroll = pTable.getScrollPane();

    // Global Frame Button Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setPreferredSize(new Dimension(75, 25));
    btnSelect.setText("Select");
    btnSelect.addActionListener(this);

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setPreferredSize(new Dimension(75, 25));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // Enable & Disable Widgets depending on the Data

    setWidgets();

    // Attach Widgets to Frame

    this.getContentPane().add(pnlKGS, BorderLayout.NORTH);
    pnlKGS.add(pnlKGSTexts,           BorderLayout.CENTER);
    pnlKGSTexts.add(pnlKGSSearchBy, BorderLayout.NORTH);
    pnlKGSSearchBy.add(rbAPI,         BorderLayout.WEST);
    pnlKGSSearchBy.add(rbLease,       BorderLayout.CENTER);
    pnlKGSSearchBy.add(rbTRS,         BorderLayout.EAST);

    pnlKGSTexts.add(pnlKGSTextSearch,  BorderLayout.CENTER);
    pnlKGSTextSearch.add(pnlKGSText,  "TEXT");
    pnlKGSText.add(txtText,           BorderLayout.CENTER);

    pnlKGSTextSearch.add(pnlKGSTRS,   "TRS");
    pnlKGSTRS.add(pnlSection,         null);
    pnlSection.add(txtSection,        BorderLayout.EAST);

    pnlKGSTRS.add(pnlTownship,        null);
    pnlTownship.add(txtTownship,      BorderLayout.WEST);
    pnlTownship.add(rbNorth,          BorderLayout.CENTER);
    pnlTownship.add(rbSouth,          BorderLayout.EAST);

    pnlKGSTRS.add(pnlRange,           null);
    pnlRange.add(txtRange,            BorderLayout.WEST);
    pnlRange.add(rbEast,              BorderLayout.EAST);
    pnlRange.add(rbWest,              BorderLayout.CENTER);

    pnlKGSTexts.add(pnlSearch,        BorderLayout.SOUTH);
    pnlSearch.add(btnSearch,          BorderLayout.CENTER);

    groupSearchText.add(rbAPI);
    groupSearchText.add(rbLease);
    groupSearchText.add(rbTRS);

    groupTownship.add(rbNorth);
    groupTownship.add(rbSouth);

    groupRange.add(rbEast);
    groupRange.add(rbWest);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(scroll,                BorderLayout.CENTER);

    this.getContentPane().add(pnlBase,   BorderLayout.SOUTH);
    pnlBase.add(btnSelect,               null);
    pnlBase.add(btnClose,                null);

    // Display the Frame

    this.setSize(new Dimension(500, 500));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getWellHeader()
   * <p> This method will return the Well Header Data
   * @return st     = Well Header Data Structure
   */

  public iqstratHeadersStruct getWellHeader() { return (stHeader); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setWidgets()
   *  <p> This method will enable or disable buttons depending on the data.
   */

  private void setWidgets()
  {
    btnSelect.setEnabled(false);

    if (pTable.getTotalRows() > 0)
      btnSelect.setEnabled(true);

    if (rbAPI.isSelected())
    {
      titledBorderText.setTitle("Enter API-Number :");
      pnlKGSText.updateUI();
      pnlKGSTextSearch.updateUI();
    }
    else if (rbLease.isSelected())
    {
      titledBorderText.setTitle(
          "Enter Lease (Drop Well Number, Not Case Sensitive):");
      pnlKGSText.updateUI();
      pnlKGSTextSearch.updateUI();
    }
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method search()
   * <p> This methodw will search the KGS Database for wells that fit the
   *     constraint passed.
   */

  private void search()
  {
    stList = iqstrat.io.iqstratIOHeaders.getData(
               iqstrat.io.ReadHeadersXMLFile.URL, sPath);

    pTable.repopulateList(stList);
  }

  /** Method selectHeader()
   * <p> This method will select the well Header data
   */

  private void selectHeader()
  {
    stHeader = iqstrat.iqstratHeadersUtility.copy( pTable.getRowData() );

    if (notifier != null)
      notifier.notifyObservers(
          new String("Well Header Information Data Selected"));
  }

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier = null;
    stCMN    = null;

    sTownship = null;
    sRange    = null;
    sPath     = null;

    stList   = null;
    stHeader = null;

    if (stList != null)
      stList.delete();
    stList     = null; // Headers List Data Struct

    if (stHeader != null)
      stHeader.delete();
    stHeader   = null; // Headers Structure

    if (pTable != null)
      pTable.close();
    pTable = null;

    // KGS Search Widgets

    // -- Type of text search panel

    pnlKGSTextSearch = null;
    cardLayout2      = null;

    // -- KGS - Type of Text Search

    rbAPI   = null;
    rbLease = null;
    rbTRS   = null;

    titledBorderText = null;
    pnlKGSText       = null;

    // -- KGS - API-Number or Lease Name Search Text Fields

    txtText = null;

    // -- KGS - Township Range Section Search Text Fields

    txtTownship = null;
    rbNorth     = null;
    rbSouth     = null;

    txtRange    = null;
    rbEast      = null;
    rbWest      = null;

    txtSection  = null;

    // Global Frame Buttons

    btnSelect   = null;
    btnClose    = null;

    iClose      = 1;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // KGS Search Widgets

    if (event.getSource() == btnSearch) { search(); }

    // -- KGS - Type of Text Search

    if (event.getSource() == rbAPI)
    {
      iSearch = _API;
      cardLayout2.show(pnlKGSTextSearch, "TEXT");
    }
    if (event.getSource() == rbLease)
    {
      iSearch = _LEASE;
      cardLayout2.show(pnlKGSTextSearch,"TEXT");
    }
    if (event.getSource() == rbTRS)
    {
      iSearch = _TRS;
      cardLayout2.show(pnlKGSTextSearch, "TRS");
    }

    // -- KGS - Township Range Section Search Text Fields

    if (event.getSource() == rbNorth) { sTownship = "N"; }
    if (event.getSource() == rbSouth) { sTownship = "S"; }

    if (event.getSource() == rbEast) { sRange = "E"; }
    if (event.getSource() == rbWest) { sRange = "W"; }

    // Select Well from list for Headers data

    if (event.getSource() == btnSelect) { selectHeader(); }

    // Enable & Disable Widgets depending on the Data

    if (iClose == 0) setWidgets();

    // Close Dialog Action

    if (event.getSource() == btnClose) { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtText)
    {
      if (txtText.getText().length() > 0)
      {
        if (rbAPI.isSelected())
        {
          sTemp = cmn.cmnString.convertToAPI(txtText.getText());
          txtText.setText(sTemp);

          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +"sAPI="+sTemp;
        }
        else if (rbLease.isSelected())
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
                  "sLease="+txtText.getText();
        }
      }
    }

    // -- KGS - Township Range Section Search Text Fields

    if (e.getSource() == txtTownship)
    {
      iNumeric = 1;
      sTemp = txtTownship.getText();
      sMessage = new String("Township Value is a Numeric Field");
    }
    if (e.getSource() == txtRange)
    {
      iNumeric = 1;
      sTemp = txtRange.getText();
      sMessage = new String("Range Value is a Numeric Field");
    }
    if (e.getSource() == txtSection)
    {
      iNumeric = 1;
      sTemp = txtSection.getText();
      sMessage = new String("Section Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtTownship)
          iTownship = cmn.cmnString.stringToInt(txtTownship.getText());
        if (e.getSource() == txtRange)
          iRange = cmn.cmnString.stringToInt(txtRange.getText());
        if (e.getSource() == txtSection)
          iSection = cmn.cmnString.stringToInt(txtSection.getText());

        if ((iTownship > 0) && (iRange > 0) && (iSection > 0))
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange    +
              "&iSection=" + iSection;
        }
        else if ((iTownship > 0) && (iRange > 0))
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange;
        }
      }
    }
  }
}


/** CLASS kgsSearchKGSHeadersFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class kgsSearchKGSHeadersFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  kgsSearchKGSHeadersFrame adaptee;

  kgsSearchKGSHeadersFrameFocusAdapter(kgsSearchKGSHeadersFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 11/25/2008
 *  @author  John Victorine
 */
