/*
 * @kgsSearchKGSFrame.java Version 1.1 10/18/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import cmn.cmnStruct;
import cmn.cmnLASFileListStruct;

import iqstrat.iqstratStruct;
import iqstrat.iqstratHeadersListStruct;
import iqstrat.iqstratHeadersStruct;

import kgs.gui.kgsWellListTable;

import las3.las3ListStruct;
import las.lasFileDataStruct;
import las.gui.lasFilesFrame;

import horizon.regions.regionsListStruct;
import horizon.regions.io.ReadRegionsXMLFile;
//import horizon.regions.gui.topsPanel;

import horizon.strat.stratListStruct;
import horizon.strat.gui.stratKGSFrame;

import rock.rockDataListStruct;

import rock.rockImagesListStruct;
import rock.gui.rockImageFrame;

import util.utilWaitAnimation;
import util.SwingWorker;

/** Class kgsSearchKGSFrame
 *  <p> This Class will allow the user to load data files from their PC or the
 *      KGS Database
 *
 *  @version 1.1 10/18/2007
 *  @author  John R. Victorine
 */

public class kgsSearchKGSFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable    notifier = null;  // Observable Object
  private iqstratStruct stStruct = null;  // Global IQSTRAT Data Structure
  private cmnStruct     stCMN    = null;  // Global Common Data Structure

  // Global Frame Variables

  private int    iTownship = 0;     // Township
  private String sTownship = "S";   // Township Direction
  private int    iRange    = 0;     // Range
  private String sRange    = "E";   // Range Direction
  private int    iSection  = 0;     // Section

  private String sPath = "";        // URL to retrieve Well List

  private iqstratHeadersListStruct stList     = null; // Headers List Data Struct
  private iqstratHeadersStruct     stHeader   = null; // Headers Structure
  private stratListStruct          stTops     = null;  // Formation Tops List
  private regionsListStruct        stPerf     = null;  // Perforation data List
//  private regionsListStruct        stTops     = null;  // Formation Tops List
  private rockDataListStruct       stCore     = null; // Measured Core Data
  private rockImagesListStruct     stImages   = null; // Core Images

  private kgsWellListTable         pTable     = null; // Well List Table
  private lasFilesFrame            pLASFiles  = null; // LAS File List Frame
  private stratKGSFrame            pTopsFrame = null; // Strat Units Frame
//  private JFrame                   pTopsFrame = null;  // Formation Tops Frame
//  private topsPanel                pTops      = null;  // Formation Tops Panel
  private rockImageFrame           pImages    = null;

  // Search By

  private static final int _API   = 0;  // Search by API Identifier
  private static final int _LEASE = 1;  // Search by Partial Lease Identifier
  private static final int _TRS   = 2;  // Search by TRS Identifier

  private int iSearch = _API;           // Search ID

  // KGS Search Widgets

  // -- Type of text search panel

  private JPanel     pnlKGSTextSearch = new JPanel();     // Search By Panel
  private CardLayout cardLayout2      = new CardLayout(); // Search By Layout

  // Search for Well List Button

  private JButton btnSelect = new JButton();  // Select Well

  // -- KGS - Type of Text Search

  private JRadioButton rbAPI   = new JRadioButton();  // Search by API button
  private JRadioButton rbLease = new JRadioButton();  // Search by Lease button
  private JRadioButton rbTRS   = new JRadioButton();  // Search by TRS button

  private TitledBorder titledBorderText = null;
  private JPanel       pnlKGSText       = new JPanel();

  // -- KGS - API-Number or Lease Name Search Text Fields

  private JTextField   txtText     = new JTextField();

  // -- KGS - Township Range Section Search Text Fields

  // .. Township Widgets

  private JTextField   txtTownship = new JTextField();
  private JRadioButton rbNorth     = new JRadioButton();
  private JRadioButton rbSouth     = new JRadioButton();

  // .. Range Widgets

  private JTextField   txtRange    = new JTextField();
  private JRadioButton rbEast      = new JRadioButton();
  private JRadioButton rbWest      = new JRadioButton();

  // .. Section Widgets

  private JTextField   txtSection  = new JTextField();

  // Well List Widgets

  private JButton btnLAS    = new JButton();
  private JButton btnTops   = new JButton();
  private JButton btnCore   = new JButton();
  private JButton btnImage  = new JButton();

  // Global Frame Buttons

  private JButton btnClose  = new JButton();
  private JButton btnSearch = new JButton();

  /** Constructor kgsSearchKGSFrame()
   *  <p> This is the Constructor for this class.
   *  @param stStruct = IQSTRAT Global Data Structure
   *  @param notifier = Observable
   */

  public kgsSearchKGSFrame(iqstratStruct stStruct, Observable notifier)
  {
    try
    {
      this.stStruct = stStruct;
      this.stCMN    = stStruct.stCMN;
      this.notifier = notifier;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlKGS         = new JPanel();
    JPanel pnlKGSTexts    = new JPanel();
    JPanel pnlKGSSearchBy = new JPanel();
    JPanel pnlKGSTRS      = new JPanel();
    JPanel pnlTownship    = new JPanel();
    JPanel pnlRange       = new JPanel();
    JPanel pnlSection     = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlBase        = new JPanel();
    JPanel pnlDataButtons = new JPanel();
    JPanel pnlSearch      = new JPanel();

    JScrollPane scroll    = new JScrollPane();

    ButtonGroup groupSearchText = new ButtonGroup();
    ButtonGroup groupTownship   = new ButtonGroup();
    ButtonGroup groupRange      = new ButtonGroup();

    TitledBorder titledBorderKGS = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Search for Data in Kansas Geological Survey Database:");
    titledBorderKGS.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderList = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "List of Oil & Gas Wells:");
    titledBorderList.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderKGSSearchBy = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Search By:");
    titledBorderKGSSearchBy.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    titledBorderText = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Enter API-Number:");
    titledBorderText.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderTownship = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Township:");
    titledBorderTownship.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderRange = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Range:");
    titledBorderRange.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderSection = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Section:");
    titledBorderSection.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderLoaded = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Data Loaded:");
    titledBorderLoaded.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderLoad = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Load Data:");
    titledBorderLoad.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Search for Data on KGS Server");

    // KGS Search Panel

    pnlKGS.setBorder(titledBorderKGS);
    pnlKGS.setLayout(new BorderLayout());

    // Search for Wells

    pnlSearch.setLayout(new BorderLayout());

    btnSearch.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSearch.setActionCommand("btnSearch");
    btnSearch.setText("Search");
    btnSearch.addActionListener(this);

    // -- KGS Search Texts

    pnlKGSTexts.setBorder(titledBorderKGSSearchBy);
    pnlKGSTexts.setLayout(new BorderLayout());

    // -- Search by Text

    pnlKGSSearchBy.setLayout(new BorderLayout());

    rbAPI.setFont(new java.awt.Font("Dialog", 1, 11));
    rbAPI.setSelected(true);
    rbAPI.setText("API-Number");
    rbAPI.addActionListener(this);

    rbLease.setFont(new java.awt.Font("Dialog", 1, 11));
    rbLease.setHorizontalAlignment(SwingConstants.CENTER);
    rbLease.setText("Lease name");
    rbLease.addActionListener(this);

    rbTRS.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTRS.setText("Township Range Section");
    rbTRS.addActionListener(this);

    // Text Search Panel

    pnlKGSTextSearch.setLayout(cardLayout2);

    // -- Text Panel

    pnlKGSText.setBorder(titledBorderText);
    pnlKGSText.setLayout(new BorderLayout());

    txtText.setText("");
    txtText.setColumns(50);
    txtText.addFocusListener(new kgsSearchKGSFrameFocusAdapter(this));

    // -- TRS Panel

    pnlKGSTRS.setLayout(new GridLayout());

    // ---- Township Panel

    pnlTownship.setBorder(titledBorderTownship);
    pnlTownship.setLayout(new BorderLayout());

    txtTownship.setText("0");
    txtTownship.setColumns(4);
    txtTownship.setHorizontalAlignment(SwingConstants.TRAILING);
    txtTownship.addFocusListener(new kgsSearchKGSFrameFocusAdapter(this));

    rbNorth.setText("N");
    rbNorth.addActionListener(this);

    rbSouth.setSelected(true);
    rbSouth.setText("S");
    rbSouth.addActionListener(this);

    // ---- Range Panel

    pnlRange.setBorder(titledBorderRange);
    pnlRange.setLayout(new BorderLayout());

    txtRange.setText("0");
    txtRange.setColumns(4);
    txtRange.setHorizontalAlignment(SwingConstants.TRAILING);
    txtRange.addFocusListener(new kgsSearchKGSFrameFocusAdapter(this));

    rbEast.setSelected(true);
    rbEast.setText("E");
    rbEast.addActionListener(this);

    rbWest.setText("W");
    rbWest.addActionListener(this);

    // ---- Section Panel

    pnlSection.setBorder(titledBorderSection);
    pnlSection.setLayout(new BorderLayout());

    txtSection.setText("0");
    txtSection.setColumns(4);
    txtSection.setHorizontalAlignment(SwingConstants.TRAILING);
    txtSection.addFocusListener(new kgsSearchKGSFrameFocusAdapter(this));

    // Well List Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBorder(titledBorderList);

    // .. Well List Table

    pTable = new kgsWellListTable();
    scroll = pTable.getScrollPane();

    // .. Well List Buttons

    pnlDataButtons.setLayout(new GridLayout());
    pnlDataButtons.setBorder(titledBorderLoad);

    btnLAS.setFont(new java.awt.Font("Dialog", 1, 11));
    btnLAS.setText("LAS File Data");
    btnLAS.addActionListener(this);

    btnTops.setFont(new java.awt.Font("Dialog", 1, 11));
    btnTops.setText("Top Picks");
    btnTops.addActionListener(this);

    btnCore.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCore.setText("Core Data");
    btnCore.addActionListener(this);

    btnImage.setFont(new java.awt.Font("Dialog", 1, 11));
    btnImage.setText("Core Images");
    btnImage.addActionListener(this);

    // Global Frame Button Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // Enable & Disable Widgets depending on the Data

    setWidgets();

    // Attach Widgets to Frame

    this.getContentPane().add(pnlKGS, BorderLayout.NORTH);
    pnlKGS.add(pnlKGSTexts,           BorderLayout.CENTER);
    pnlKGSTexts.add(pnlKGSSearchBy, BorderLayout.NORTH);
    pnlKGSSearchBy.add(rbAPI,         BorderLayout.WEST);
    pnlKGSSearchBy.add(rbLease,       BorderLayout.CENTER);
    pnlKGSSearchBy.add(rbTRS,         BorderLayout.EAST);

    pnlKGSTexts.add(pnlKGSTextSearch,  BorderLayout.CENTER);
    pnlKGSTextSearch.add(pnlKGSText,  "TEXT");
    pnlKGSText.add(txtText,           BorderLayout.CENTER);

    pnlKGSTextSearch.add(pnlKGSTRS,   "TRS");
    pnlKGSTRS.add(pnlSection,         null);
    pnlSection.add(txtSection,        BorderLayout.EAST);

    pnlKGSTRS.add(pnlTownship,        null);
    pnlTownship.add(txtTownship,      BorderLayout.WEST);
    pnlTownship.add(rbNorth,          BorderLayout.CENTER);
    pnlTownship.add(rbSouth,          BorderLayout.EAST);

    pnlKGSTRS.add(pnlRange,           null);
    pnlRange.add(txtRange,            BorderLayout.WEST);
    pnlRange.add(rbEast,              BorderLayout.EAST);
    pnlRange.add(rbWest,              BorderLayout.CENTER);

    pnlKGSTexts.add(pnlSearch,        BorderLayout.SOUTH);
    pnlSearch.add(btnSearch,          BorderLayout.CENTER);

    groupSearchText.add(rbAPI);
    groupSearchText.add(rbLease);
    groupSearchText.add(rbTRS);

    groupTownship.add(rbNorth);
    groupTownship.add(rbSouth);

    groupRange.add(rbEast);
    groupRange.add(rbWest);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(scroll,                BorderLayout.CENTER);
    pnlCenter.add(pnlDataButtons,        BorderLayout.SOUTH);
    if (stStruct.bLAS != null)
    {
	  if (stStruct.bLAS[las3.las3Constants._LOG])
        pnlDataButtons.add(btnLAS,       null);
	  if (stStruct.bLAS[las3.las3Constants._TOPS])
        pnlDataButtons.add(btnTops,      null);
	  if (stStruct.bLAS[las3.las3Constants._CORE])
        pnlDataButtons.add(btnCore,      null);
	  if (stStruct.bLAS[las3.las3Constants._IQ_IMAGES])
        pnlDataButtons.add(btnImage,     null);
    }

    this.getContentPane().add(pnlBase,   BorderLayout.SOUTH);
    pnlBase.add(btnClose,                null);

    // Display the Frame

    this.setSize(new Dimension(500, 500));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getWellHeader()
   * <p> This method will return the Well Header Data
   * @return st     = Well Header Data Structure
   */

  public iqstratHeadersStruct getWellHeader() { return (stHeader); }

  /** Method getLAS3Data()
   * <p> This method will return the LAS 3 File Data List Structure
   * @return = the LAS 3 File Data List Structure
   */

  public las3ListStruct getLAS3Data() { return ( pLASFiles.getLAS3() ); }

  /** Method getLASData()
   * <p> This method will return the LAS Data Structure
   * @return st = the LAS Data Structure
   */

//  public lasFileDataStruct getLASDataStructure()
  public lasFileDataStruct getLASData()
  {
    return ( pLASFiles.getLASDataStructure() );
  }

  /** Method getStartingDepth()
   * <p> This method will return the starting depth of the LAS File
   * @return dStart = the starting depth of the LAS File
   */

  public double getStartingDepth() { return ( pLASFiles.getStartingDepth() ); }

  /** Method getEndingDepth()
   * <p> This method will return the ending depth of the LAS File
   * @return dEnd = the ending depth of the LAS File
   */

  public double getEndingDepth() { return ( pLASFiles.getEndingDepth() ); }

  /** Method getColorlith()
   * <p> This method will return the colorlith identifier from the LAS File
   * @return  iColorlith = Colorlith Identifier
   */

  public int getColorlith() {  return ( pLASFiles.getColorlith() ); }

  /** Method getTops()
   * <p> This method will return the Formation Tops
   * @return stTops = Formation Tops Data List Struct
   */

  public stratListStruct getTops() { return ( pTopsFrame.getDownloadData() ); }
//  public regionsListStruct getTops() { return (pTops.getDownloadData()); }

  /** Method getPerf()
   * <p> This method will retrieve the Well Regions for a specific well
   * @return st      = The Regions List Data Structure
   */

  public regionsListStruct getPerf()
  {
    ReadRegionsXMLFile pXML    = null;
    regionsListStruct  st      = null;
    int                iMethod = horizon.regions.io.ReadRegionsXMLFile.URL;

    stHeader = pTable.getRowData();

    if (stHeader != null)
    {
      pXML = new ReadRegionsXMLFile(iMethod);
      st   = pXML.Process(cmn.cmnStruct.KGS_DST_PERFS + "sAPI="+stHeader.sAPI);
    }

    return (st);
  }

  /** Method getRockData()
   * <p> This method will get the Rock Data for the well
   * @return rockDataListStruct = the Rock Data List Structure
   */

  public rockDataListStruct getRockData() { return (stCore); }

  /** Method getRockImages()
   * <p> This method will get the Rock Images for the well
   * @return stImagesSelected = the Rock Image List Structure
   */

  public rockImagesListStruct getRockImages() { return (pImages.getData()); }

  /** Method closeLASFile()
   * <p> This method will close the LAS File Chooser Dialog
   */

  public void closeLASFile()
  {
    if (pLASFiles != null)
      pLASFiles.close();
    pLASFiles = null;
  }

  /** Method closeTops()
   * <p> This method will close the KGS Stratigraphic Units Frame
   */

  public void closeTops()
  {
    if (pTopsFrame != null)
      pTopsFrame.close();
    pTopsFrame = null;
  }

  /** Method closeCoreImages()
   * <p> This method will close the Core Image File Chooser Dialog
   */

  public void closeCoreImages()
  {
    if (pImages != null)
      pImages.close();
    pImages = null;
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setWidgets()
   *  <p> This method will enable or disable buttons depending on the data.
   */

  private void setWidgets()
  {
    btnLAS.setEnabled(false);
    btnTops.setEnabled(false);
    btnCore.setEnabled(false);
    btnImage.setEnabled(false);

    if (pTable.getTotalRows() > 0)
    {
      btnLAS.setEnabled(true);
      btnTops.setEnabled(true);
      btnCore.setEnabled(true);
      btnImage.setEnabled(true);
    }

    if (rbAPI.isSelected())
    {
      titledBorderText.setTitle("Enter API-Number :");
      pnlKGSText.updateUI();
      pnlKGSTextSearch.updateUI();
    }
    else if (rbLease.isSelected())
    {
      titledBorderText.setTitle(
          "Enter Lease (Drop Well Number, Not Case Sensitive):");
      pnlKGSText.updateUI();
      pnlKGSTextSearch.updateUI();
    }
  }

  /* ===================================================================== *
   * ------------------------ WAIT ANIMATION ----------------------------- *
   * ===================================================================== */

  /** Method WaitWellList()
   * <p> This method will display the Wait Animation for searching KGS db
   */

  private void WaitWellList()
  {
    java.net.URL url = null;

    // Get the URL Path to display the Wait Animation

    try
    {
      url = new java.net.URL(cmn.cmnStruct.IQSTRAT_BASE + "/");
    }
    catch(Exception ex)
    {
      System.out.println("Wrong Image Directory" + ex);
    }

    // Start the Wait Animation Dialog

    final utilWaitAnimation pDialog = new utilWaitAnimation(url);
    pDialog.startAnimation();

    final SwingWorker worker = new SwingWorker()
    {
      public Object construct()
      {
        search();

        return "OK";
      }

      //Runs on the event-dispatching thread.

      public void finished() { pDialog.stopAnimation(); }
    };

    // Start the Wait Animation Thread

    worker.start();  //required for SwingWorker
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method loadLASData()
   * <p> This method will display a dialog with a list of available LAS Files
   */

  private void loadLASData()
  {
    cmnLASFileListStruct stLAS = null;
    String sMessage = "";
    int    iError   = 0;

    closeLASFile();

    stHeader = pTable.getRowData();

    if (stHeader != null)
    {
      if (stHeader.iLAS > 0)
      {
        stLAS = kgs.io.kgsIORead.ReadLASFileInformation(
                  stHeader.sAPI, xml.ReadLASFileXMLFile.URL);

        pLASFiles = new lasFilesFrame(stStruct, notifier, stLAS);
      }
      else
      {
        sMessage = "There are no digital LAS Files for this well "+
                   "in the KGS Server.";
        iError = -1;
      }
    }
    else
    {
      sMessage = "No Well Record Exists";
      iError = -1;
    }

    if (iError == -1)
    {
      JOptionPane.showMessageDialog( (Component)null,
                                     sMessage,
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);
    }
  }

  /** Method loadTOPSData()
   * <p> This method will retrieve the Tops List
   */

  private void loadTOPSData()
  {
    String          sMessage = "";
    int             iError   = 0;
    stratListStruct st       = null;

    // Close Tops Frame

//    if (pTops != null)
//      pTops.close();
//    pTops = null;

    closeTops();

    stHeader = pTable.getRowData();

    if (stHeader != null)
    {
      if (stHeader.iTops > 0)
      {
        stTops = kgs.io.kgsIORead.ReadWellRegions(
                     stHeader.sAPI,
                     stHeader.sKID,
                     horizon.strat.io.ReadStratUnitsXMLFile.URL);
//        stTops = kgs.io.kgsIORead.ReadWellRegions(
//              stHeader.sAPI, horizon.regions.io.ReadRegionsXMLFile.URL);
      }
      else
      {
        sMessage = "There are no well tops for this well "+
                   "in the KGS Server.";
        iError = -1;
      }
    }
    else
    {
      sMessage = "No Well Record Exists";
      iError = -1;
    }

    if (stTops != null)
    {
      pTopsFrame = new stratKGSFrame(notifier, stTops, st);
/*
      pTopsFrame = new JFrame();
      pTopsFrame.setTitle("Select Formation Tops Source Name");

      pTops = new topsPanel(pTopsFrame, notifier, stTops);

      pTopsFrame.getContentPane().add(pTops, null);

      pTopsFrame.setSize(new Dimension(700, 400));
      Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
      pTopsFrame.setLocation((d.width - pTopsFrame.getSize().width) / 2,
                             (d.height - pTopsFrame.getSize().height) / 2);

      pTopsFrame.setResizable(false);
      pTopsFrame.setVisible(true);
*/
    }
  }

  /** Method loadCoreData()
   * <p> This method will load the core data if it is present
   */

  public void loadCoreData()
  {
    String sMessage = "";
    int    iError   = 0;

    stHeader = pTable.getRowData();

    if (stHeader != null)
    {
      if (stHeader.iCore > 0)
      {
        stCore = kgs.io.kgsIORead.ReadCoreData(
                   stHeader.sAPI, rock.io.ReadRockDataXMLFile.URL);
      }
      else
      {
        sMessage = "There are no core data for this well "+
                   "in the KGS Server.";
        iError = -1;
      }
    }
    else
    {
      sMessage = "No Well Record Exists";
      iError = -1;
    }

    if (stCore != null)
      notifier.notifyObservers(new String("KGS Rock Data Loaded"));
  }

  /** Method loadImages()
   * <p> This method will display the images Frame
   */

  public void loadImages()
  {
    String sMessage = "";
    int    iError   = 0;

    closeCoreImages();

    stHeader = pTable.getRowData();

    if (stHeader != null)
    {
      if (stHeader.images > 0)
      {
        stImages = kgs.io.kgsIORead.ReadCoreImages(
                     stHeader.sAPI, rock.io.ReadRockImageXMLFile.URL);
      }
      else
      {
        sMessage = "There are no core images for this well "+
                   "in the KGS Server.";
        iError = -1;
      }
    }
    else
    {
      sMessage = "No Well Record Exists";
      iError = -1;
    }

    if (stImages != null)
      pImages = new rockImageFrame(stCMN, notifier, stImages);
  }

  /** Method search()
   * <p> This methodw will search the KGS Database for wells that fit the
   *     constraint passed.
   */

  private void search()
  {
    stList = iqstrat.io.iqstratIOHeaders.getData(
               iqstrat.io.ReadHeadersXMLFile.URL, sPath);
//System.out.println(sPath);

    pTable.repopulateList(stList);
  }

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier = null;
    stStruct = null;
    stCMN    = null;

    sTownship = null;
    sRange    = null;
    sPath     = null;

    stList   = null;
    stHeader = null;

    if (stList != null)
      stList.delete();
    stList     = null; // Headers List Data Struct

    if (stHeader != null)
      stHeader.delete();
    stHeader   = null; // Headers Structure

    if (stTops != null)
      stTops.delete();
    stTops     = null; // Formation Tops List

    if (stPerf != null)
      stPerf.delete();
    stPerf     = null; // Perforation Data List

    if (stCore != null)
      stCore.delete();
    stCore     = null; // Core Data

    if (stImages != null)
      stImages.delete();
    stImages   = null; // Core Images

    if (pTable != null)
      pTable.close();
    pTable = null;

    // KGS Search Widgets

    // -- Type of text search panel

    pnlKGSTextSearch = null;
    cardLayout2      = null;

    // -- KGS - Type of Text Search

    rbAPI   = null;
    rbLease = null;
    rbTRS   = null;

    titledBorderText = null;
    pnlKGSText       = null;

    // -- KGS - API-Number or Lease Name Search Text Fields

    txtText = null;

    // -- KGS - Township Range Section Search Text Fields

    txtTownship = null;
    rbNorth     = null;
    rbSouth     = null;

    txtRange    = null;
    rbEast      = null;
    rbWest      = null;

    txtSection  = null;

    // Well List Widgets

    btnLAS   = null;
    btnTops  = null;
    btnCore  = null;
    btnImage = null;

    // LAS Files Frame

    closeLASFile();
    closeTops();
    closeCoreImages();

    // Global Frame Buttons

    btnClose = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // KGS Search Widgets

    if (event.getSource() == btnSearch) { search(); }

    // -- KGS - Type of Text Search

    if (event.getSource() == rbAPI)
    {
      iSearch = _API;
      cardLayout2.show(pnlKGSTextSearch, "TEXT");
    }
    if (event.getSource() == rbLease)
    {
      iSearch = _LEASE;
      cardLayout2.show(pnlKGSTextSearch,"TEXT");
    }
    if (event.getSource() == rbTRS)
    {
      iSearch = _TRS;
      cardLayout2.show(pnlKGSTextSearch, "TRS");
    }

    // -- KGS - Township Range Section Search Text Fields

    if ((event.getSource() == rbNorth) || (event.getSource() == rbSouth) ||
        (event.getSource() == rbEast)  || (event.getSource() == rbWest))
    {
      if (event.getSource() == rbNorth) { sTownship = "N"; }
      if (event.getSource() == rbSouth) { sTownship = "S"; }

      if (event.getSource() == rbEast) { sRange = "E"; }
      if (event.getSource() == rbWest) { sRange = "W"; }

      if ((iTownship > 0) && (iRange > 0) && (iSection > 0))
      {
        sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
            "iTownship=" + iTownship + "&sTownship=" + sTownship +
            "&iRange="   + iRange    + "&sRange="    + sRange    +
            "&iSection=" + iSection;
      }
      else if ((iTownship > 0) && (iRange > 0))
      {
        sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
            "iTownship=" + iTownship + "&sTownship=" + sTownship +
            "&iRange="   + iRange    + "&sRange="    + sRange;
      }
    }

    // Load Well Data Actions

    if (event.getSource() == btnLAS)   { loadLASData(); }
    if (event.getSource() == btnTops)  { loadTOPSData(); }
    if (event.getSource() == btnCore)  { loadCoreData(); }
    if (event.getSource() == btnImage) { loadImages(); }

    // Enable & Disable Widgets depending on the Data

    setWidgets();

    // Close Dialog Action

    if (event.getSource() == btnClose) { close(); }

    if (pLASFiles != null)  pLASFiles.toFront();
    if (pTopsFrame != null) pTopsFrame.toFront();
    if (pImages != null)    pImages.toFront();
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtText)
    {
      if (txtText.getText().length() > 0)
      {
        if (rbAPI.isSelected())
        {
          sTemp = cmn.cmnString.convertToAPI(txtText.getText());
          txtText.setText(sTemp);

          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +"sAPI="+sTemp;
        }
        else if (rbLease.isSelected())
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
                  "sLease="+txtText.getText();
        }
      }
    }

    // -- KGS - Township Range Section Search Text Fields

    if (e.getSource() == txtTownship)
    {
      iNumeric = 1;
      sTemp = txtTownship.getText();
      sMessage = new String("Township Value is a Numeric Field");
    }
    if (e.getSource() == txtRange)
    {
      iNumeric = 1;
      sTemp = txtRange.getText();
      sMessage = new String("Range Value is a Numeric Field");
    }
    if (e.getSource() == txtSection)
    {
      iNumeric = 1;
      sTemp = txtSection.getText();
      sMessage = new String("Section Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtTownship)
          iTownship = cmn.cmnString.stringToInt(txtTownship.getText());
        if (e.getSource() == txtRange)
          iRange = cmn.cmnString.stringToInt(txtRange.getText());
        if (e.getSource() == txtSection)
          iSection = cmn.cmnString.stringToInt(txtSection.getText());

        if ((iTownship > 0) && (iRange > 0) && (iSection > 0))
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange    +
              "&iSection=" + iSection;
        }
        else if ((iTownship > 0) && (iRange > 0))
        {
          sPath = cmn.cmnStruct.KGS_WELL_HEADERS_READ +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange;
        }
      }
    }
  }
}


/** CLASS kgsSearchKGSFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class kgsSearchKGSFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  kgsSearchKGSFrame adaptee;

  kgsSearchKGSFrameFocusAdapter(kgsSearchKGSFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 10/18/2007
 *  @author  John Victorine
 */
