/*
 * @kgsBrineTable.java Version 1.1 06/03/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.gui;

import java.awt.*;
import java.lang.*;
import javax.swing.*;
import javax.swing.table.*;
import java.util.Observable;

import gui.guiTable;

import brine.brineListStruct;
import brine.brineStruct;

/** Class kgsBrineTable
 *  <p> This Class will display the Formation Tops.
 *
 *  @version 1.1 06/03/2011
 *  @author  John R. Victorine
 */

public class kgsBrineTable
{
  public static final int _SINGLE_SELECTION = gui.guiTable._SINGLE_SELECTION;
  public static final int _MULTI_SELECTION  = gui.guiTable._MULTI_SELECTION;

  private int      iRows     = 0;
  private int      iColumns  = 21;
  private String   sColumn[] = {"id",
                                "Lease Name",
                                "Location",
                                "Latitude",
                                "Longitude",
                                "Source",
                                "",
                                "Top (ft)",
                                "Base (ft)",
                                "",
                                "Na (mg/l)",
                                "K (mg/l)",
                                "Mg (mg/l)",
                                "Ca (mg/l)",
                                "",
                                "Cl (mg/l)",
                                "I (mg/l)",
                                "Br (mg/l)",
                                "CO3 (mg/l)",
                                "HCO3 (mg/l)",
                                "SO4 (mg/l)" };
  private int     iColLength[] = {16,  // "id"
                                  30,  // Lease Name
                                  16,  // Location
                                   8,  // Latitude
                                  10,  // Longitude
                                  20,  // Source
                                   1,  // Break
                                   8,  // Top
                                   8,  // Base
                                   1,  // Break
                                  12,  // Na
                                  12,  // K
                                  12,  // Mg
                                  12,  // Ca
                                   1,  // Break
                                  12,  // Cl
                                  12,  // I
                                  12,  // Br
                                  12,  // CO3
                                  12,  // HCO3
                                  12 }; // SO4

  private Object          oData[][] = null;
  private guiTable        pTable    = null;

  private Observable      notifier  = null; // Observable Object
  private brineListStruct st        = null; // Brine Data List Structure
  private int             iChange   = 0;

  /** Constructor kgsBrineTable()
   *  <p> This is the Constructor for this class.
   */

  public kgsBrineTable()
  {
    pTable = new guiTable(0, iColumns, sColumn, _MULTI_SELECTION);
    resetColumnWidths();
    pTable.showHorScroll(true);
  }

  /** Constructor kgsBrineTable()
   *  <p> This is the Constructor for this class.
   * @param st     = The Headers List Data Structure
   */

  public kgsBrineTable(brineListStruct st)
  {
    this.st = st;

    populateList();

    if (iRows > 0)
      pTable = new guiTable(0, iRows, iColumns, sColumn, oData, _MULTI_SELECTION);
    else
      pTable = new guiTable(0, iColumns, sColumn, _MULTI_SELECTION);

    resetColumnWidths();
    pTable.showHorScroll(true);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /** Constructor kgsBrineTable()
   *  <p> This is the Constructor for this class.
   * @param st     = The Headers List Data Structure
   */

  public kgsBrineTable(Observable notifier, brineListStruct st)
  {
	this.notifier = notifier;
    this.st       = st;

    populateList();

    if (iRows > 0)
      pTable = new guiTable(0, iRows, iColumns, sColumn, oData,
                            _MULTI_SELECTION, notifier);
    else
      pTable = new guiTable(0, iColumns, sColumn,
                            _MULTI_SELECTION, notifier);

    resetColumnWidths();
    pTable.showHorScroll(true);

    if (iChange == 0)
      iRows = 0;

//    if (iRows > 0)
//      pTable.setSelectedRow(0);
  }

  /** Constructor kgsBrineTable()
   *  <p> This is the Constructor for this class.
   * @param st     = The Headers List Data Structure
   */

  public kgsBrineTable(Observable notifier, brineListStruct st, int iSelection)
  {
	this.notifier = notifier;
    this.st       = st;

    populateList();

    if (iRows > 0)
      pTable = new guiTable(0, iRows, iColumns, sColumn, oData, iSelection, notifier);
    else
      pTable = new guiTable(0, iColumns, sColumn, iSelection, notifier);

    resetColumnWidths();
    pTable.showHorScroll(true);

    if (iChange == 0)
      iRows = 0;

//    if (iRows > 0)
//      pTable.setSelectedRow(0);
  }

  /** Method resetColumnWidths()
   *  <p> This method will reset the column widths to a specific size.
   */

  private void resetColumnWidths()
  {
    for (int i=0; i<iColumns; i++)
    {
      pTable.resizeColumn(i, iColLength[i]);
    }
  }

  /** Method close()
   * <p> This method will set global variables to null in order to force
   *     Java to deallocate memory
   */

  public void close()
  {
    sColumn    = null;
    iColLength = null;
    oData      = null;
    pTable     = null;

    notifier   = null; // Observable Object
    st         = null;
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method populateList()
   *  <p> This method will populate the Region List data.
   */

  private void populateList()
  {
    int i      = 0;
    int iCount = 0;

    iChange = 0;
    iRows   = 1;

    oData = new String[1][iColumns];
    for (i=0; i<iColumns; i++)
    {
      oData[0][i] = new String("");
    }

    if (st != null)
    {
      if (st.iCount > 0)
      {
        iChange = 1;
        iCount  = 0;
        oData   = new Object[st.iCount][iColumns];

        for (i=0; i<st.iCount; i++)
        {
          if (st.stItem[i] != null)
          {
            oData[iCount][0]  = new String(st.stItem[i].sKID);
            oData[iCount][1]  = new String(st.stItem[i].sName);
            oData[iCount][2]  = new String("");
            if (!st.stItem[i].sLocation.equals("T R Sec."))
              oData[iCount][2]  = new String(st.stItem[i].sLocation);
            oData[iCount][3]  = new String(""+st.stItem[i].dLatitude);
            oData[iCount][4]  = new String(""+st.stItem[i].dLongitude);
            oData[iCount][5]  = new String(st.stItem[i].source);

            oData[iCount][6]  = new String("");
            oData[iCount][7]  = new String("");
            if (st.stItem[i].dTOP >= 0.0)
              oData[iCount][7]  = new String(""+st.stItem[i].dTOP);
            oData[iCount][8]  = new String("");
            if (st.stItem[i].dBASE >= 0.0)
              oData[iCount][8]  = new String(""+st.stItem[i].dBASE);

            oData[iCount][9]  = new String("");
            oData[iCount][10] = new String("");
            if (st.stItem[i].dNA > -1.0)
              oData[iCount][10] = new String(""+st.stItem[i].dNA);

            oData[iCount][11] = new String("");
            if (st.stItem[i].dK > -1.0)
              oData[iCount][11] = new String(""+st.stItem[i].dK);

            oData[iCount][12] = new String("");
            if (st.stItem[i].dMG > -1.0)
              oData[iCount][12] = new String(""+st.stItem[i].dMG);

            oData[iCount][13] = new String("");
            if (st.stItem[i].dCA > -1.0)
              oData[iCount][13] = new String(""+st.stItem[i].dCA);

            oData[iCount][14] = new String("");
            oData[iCount][15] = new String("");
            if (st.stItem[i].dCL > -1.0)
              oData[iCount][15] = new String(""+st.stItem[i].dCL);

            oData[iCount][16] = new String("");
            if (st.stItem[i].dI > -1.0)
              oData[iCount][16] = new String(""+st.stItem[i].dI);

            oData[iCount][17] = new String("");
            if (st.stItem[i].dBR > -1.0)
              oData[iCount][17] = new String(""+st.stItem[i].dBR);

            oData[iCount][18] = new String("");
            if (st.stItem[i].dCO3 > -1.0)
              oData[iCount][18] = new String(""+st.stItem[i].dCO3);

            oData[iCount][19] = new String("");
            if (st.stItem[i].dHCO3 > -1.0)
              oData[iCount][19] = new String(""+st.stItem[i].dHCO3);

            oData[iCount][20] = new String("");
            if (st.stItem[i].dSO4 > -1.0)
              oData[iCount][20] = new String(""+st.stItem[i].dSO4);

            iCount++;
          }
        }

        iRows = iCount;
      }
    }
  }

  /** Method repopulateList()
   *  <p> This method will reload the Region List after adding, deleting from
   *      the formation name list.
   * @param st     = The Regions List Data Structure
   */

  public void repopulateList(brineListStruct st)
  {
    this.st = st;

    populateList();

    pTable.reload(iRows, oData);

    if (iChange == 0)
      iRows = 0;

    if (iRows > 0)
      pTable.setSelectedRow(0);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getTotalRows()
   *  <p> This method will return the total rows in the table.
   *  @return Total Number of Rows.
   */

  public int getTotalRows() { return (iRows); }

  /** Method getRow()
   *  <p> This method will return the selected data id in the table.
   *  @return selected data id.
   */

  public int getRow() { return (pTable.getSelectedRow()); }

  /** Method getRowData()
   *  <p> This method will return the Region Data Structure of the
   *      selected row.
   *  @return stData = Region Data Structure.
   */

  public brineStruct getRowData()
  {
    brineStruct stData = null;

    stData = st.stItem[pTable.getSelectedRow()];

    return (stData);
  }

  /** Method getData()
   *  <p> This will return the Brine Data List Structure
   *  @return stData = Brine Data List Structure of Selected Data
   */

  public brineListStruct getData()
  {
    brineListStruct stList      = null;
    int             i           = 0;
    int             j           = 0;
    int             iCount      = 1;
    int             icnt        = 0;
    int             iSelected[] = null;

    iSelected = pTable.getSelectedRows();

    if (iSelected != null)
    {
      for (i=0; i<iSelected.length; i++)
      {
        if (iSelected[i] > 0)
          iCount++;
      }
    }

    if (iCount > 0)
    {
      stList        = new brineListStruct();
      stList.stItem = new brineStruct[iCount];

      stList = brine.brineUtility.copyBrineHeader( st );

      iCount = 0;
      icnt   = 0;
      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<iSelected.length; j++)
        {
          if (icnt == iSelected[j])
          {
            stList.stItem[iCount] =
              brine.brineUtility.copy(st.stItem[i]);
            iCount++;
          }
        }
        icnt++;
      }
      stList.iCount = iCount;
    }

    return (stList);
  }

  /** Method getAllData()
   *  <p> This method will returns the Region List Data Structure.
   *  @return st = Region List Data Structure.
   */

  public brineListStruct getAllData()
  {
    brineListStruct stList      = null;
    int             i           = 0;
    int             j           = 0;
    int             iCount      = 1;

    if (st != null)
    {
      stList        = new brineListStruct();
      stList.stItem = new brineStruct[st.iCount];

      stList = brine.brineUtility.copyBrineHeader( st );

      iCount = 0;
      for (i=0; i<st.iCount; i++)
      {
        stList.stItem[iCount] =
            brine.brineUtility.copy(st.stItem[i]);
        iCount++;
      }
      stList.iCount = iCount;
    }

	return (stList);
  }

  /** Method getScrollPane()
   *  <p> This method will return the scroll pane.
   *  @return jScrollPane - Scroll Pane containing the Table.
   */

  public JScrollPane getScrollPane() { return (pTable.getScrollPane()); }
}

/*
 *  @version 1.1 06/03/2011
 *  @author  John Victorine
 */
