/*
 * @kgsBrineFrame.java Version 1.1 06/03/2011
 *
 * Copyright (c) 2011 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package kgs.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.iqstratHeadersStruct;

import brine.brineListStruct;
import brine.brineStruct;
import brine.io.ReadBrineDataXMLFile;

import kgs.kgsMeasSectListStruct;
import kgs.io.ReadMeasSectXMLFile;
import kgs.gui.kgsBrineTable;

/** Class kgsBrineFrame
 *  <p> This Class will allow the user to load data files from their PC or the
 *      KGS Database
 *
 *  @version 1.1 06/03/2011
 *  @author  John R. Victorine
 */

public class kgsBrineFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable notifier = null; // Observable Object

  // Global Frame Variables

  private int    iTownship = 0;     // Township
  private String sTownship = "S";   // Township Direction
  private int    iRange    = 0;     // Range
  private String sRange    = "E";   // Range Direction
  private int    iSection  = 0;     // Section

  private String sPath     = "";    // URL to retrieve Well List

  // Search By

  private static final int _LEASE  = 1;  // Search by Partial Lease Identifier
  private static final int _TRS    = 2;  // Search by TRS Identifier
  private static final int _COUNTY = 3;  // Search by County Identifier

  private int iSearch = _LEASE;         // Search ID

  private kgsBrineTable   pTable  = null; // Display Measured Sections

  private brineListStruct stList  = null; // Brine List Data Struct
  private brineStruct     stBrine = null; // Brine Structure
  private brineListStruct stData  = null; // Brine List Data Struct - Selected

  private iqstratHeadersStruct stHeader = null; // Header Info Struct

  private int    _COUNTIES = 0;
  private int    _CODES[]  = null;
  private String _NAMES[]  = null;

  // KGS Search Widgets

  // -- Type of text search panel

  private JPanel     pnlKGSTextSearch = new JPanel();     // Search By Panel
  private CardLayout cardLayout       = new CardLayout(); // Search By Layout

  // -- KGS - Type of Text Search

  private JRadioButton rbLease  = new JRadioButton(); // Search by Lease button
  private JRadioButton rbTRS    = new JRadioButton(); // Search by TRS button
  private JRadioButton rbCounty = new JRadioButton(); // Search by County button

  // Search for Well List Button

  private JButton btnSearch = new JButton();

  // -- KGS - Lease Name Search Text Fields

  private JTextField   txtText     = new JTextField();

  // -- KGS - Township Range Section Search Text Fields

  // .. Township Widgets

  private JTextField   txtTownship = new JTextField();
  private JRadioButton rbNorth     = new JRadioButton();
  private JRadioButton rbSouth     = new JRadioButton();

  // .. Range Widgets

  private JTextField   txtRange    = new JTextField();
  private JRadioButton rbEast      = new JRadioButton();
  private JRadioButton rbWest      = new JRadioButton();

  // -- KGS - List of Counties

  private JButton btn[] = null;

  // .. Section Widgets

  private JTextField   txtSection  = new JTextField();

  // Load List of Measure Sections

  private JButton    btnSelect     = new JButton();  // Select Measured Section

  // Exit Dialog

  private JMenuItem  mExit         = null;

  /** Constructor kgsBrineFrame( Observable notifier )
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   */

  public kgsBrineFrame( Observable notifier )
  {
    try
    {
      this.notifier = notifier;

      getCounties();  // Retrieve the List of Counties with Measured Sections

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int    iRows          = 0;

    JPanel pnlKGS         = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlSearch      = new JPanel();
    JPanel pnlKGSTexts    = new JPanel();
    JPanel pnlKGSSearchBy = new JPanel();
    JPanel pnlKGSText     = new JPanel();

    JPanel pnlKGSTRS      = new JPanel();
    JPanel pnlTownship    = new JPanel();
    JPanel pnlRange       = new JPanel();
    JPanel pnlSection     = new JPanel();

    JPanel pnlKGSCounty   = new JPanel();
    JPanel pnlCounties    = new JPanel();

    JScrollPane scrollCounties = new JScrollPane();

    JPanel pnlDataButtons = new JPanel();

    ButtonGroup groupSearchText = new ButtonGroup();
    ButtonGroup groupTownship   = new ButtonGroup();
    ButtonGroup groupRange      = new ButtonGroup();

    JScrollPane scroll    = new JScrollPane();

    TitledBorder titledBorderKGS = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "Search for Data in Kansas Geological Survey Database:");

    TitledBorder titledBorderKGSSearchBy = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Search By:");

    TitledBorder titledBorderTownship = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Township:");

    TitledBorder titledBorderRange = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Range:");

    TitledBorder titledBorderSection = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Section:");

    TitledBorder titledBorderText = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Enter Partial Lease Name:");

    TitledBorder titledBorderList = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(165, 163, 151)),
        "List of Brine Samples:");

    titledBorderKGS.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderKGSSearchBy.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderTownship.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderRange.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderSection.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderText.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderList.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    // Build Menu

    JMenuBar menuBar  = new JMenuBar();
    JMenu    menuFile = new JMenu("Close Dialog");

    mExit = new JMenuItem("Close");
    mExit.addActionListener(this);

    menuFile.add( mExit );

    menuBar.add(menuFile);

    this.setJMenuBar(menuBar);

    // Build Frame

    this.setTitle("Search for KGS Brine Data");
    this.getContentPane().setLayout(new BorderLayout());

    // KGS Search Panel

    pnlKGS.setBorder(titledBorderKGS);
    pnlKGS.setLayout(new BorderLayout());

    // Search for Wells

    pnlSearch.setLayout(new BorderLayout());

    btnSearch.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSearch.setActionCommand("btnSearch");
    btnSearch.setText("Search");
    btnSearch.addActionListener(this);

    // -- KGS Search Texts

    pnlKGSTexts.setBorder(titledBorderKGSSearchBy);
    pnlKGSTexts.setLayout(new BorderLayout());

    // -- Search by Text

    pnlKGSSearchBy.setLayout(new GridLayout(3,1));

    rbLease.setFont(new java.awt.Font("Dialog", 1, 11));
    rbLease.setText("Lease Name");
    rbLease.setHorizontalAlignment(SwingConstants.LEADING);
    rbLease.addActionListener(this);

    rbTRS.setFont(new java.awt.Font("Dialog", 1, 11));
    rbTRS.setSelected(true);
    rbTRS.setText("Sec. Town. Range");
    rbTRS.setHorizontalAlignment(SwingConstants.LEADING);
    rbTRS.addActionListener(this);

    rbCounty.setFont(new java.awt.Font("Dialog", 1, 11));
    rbCounty.setText("Counties");
    rbCounty.setHorizontalAlignment(SwingConstants.LEADING);
    rbCounty.addActionListener(this);

    // Text Search Panel

    pnlKGSTextSearch.setLayout(cardLayout);

    // -- Text Panel

    pnlKGSText.setBorder(titledBorderText);
    pnlKGSText.setLayout(new BorderLayout());

    txtText.setText("");
    txtText.setColumns(50);
    txtText.addFocusListener(new kgsBrineFrameFocusAdapter(this));

    // -- TRS Panel

    pnlKGSTRS.setLayout(new GridLayout());

    // ---- Township Panel

    pnlTownship.setBorder(titledBorderTownship);
    pnlTownship.setLayout(new BorderLayout());

    txtTownship.setText("0");
    txtTownship.setColumns(3);
    txtTownship.setHorizontalAlignment(SwingConstants.TRAILING);
    txtTownship.addFocusListener(new kgsBrineFrameFocusAdapter(this));

    rbNorth.setText("N");
    rbNorth.addActionListener(this);

    rbSouth.setSelected(true);
    rbSouth.setText("S");
    rbSouth.addActionListener(this);

    // ---- Range Panel

    pnlRange.setBorder(titledBorderRange);
    pnlRange.setLayout(new BorderLayout());

    txtRange.setText("0");
    txtRange.setColumns(3);
    txtRange.setHorizontalAlignment(SwingConstants.TRAILING);
    txtRange.addFocusListener(new kgsBrineFrameFocusAdapter(this));

    rbEast.setSelected(true);
    rbEast.setText("E");
    rbEast.addActionListener(this);

    rbWest.setText("W");
    rbWest.addActionListener(this);

    // ---- Section Panel

    pnlSection.setBorder(titledBorderSection);
    pnlSection.setLayout(new BorderLayout());

    txtSection.setText("0");
    txtSection.setColumns(3);
    txtSection.setHorizontalAlignment(SwingConstants.TRAILING);
    txtSection.addFocusListener(new kgsBrineFrameFocusAdapter(this));

    // -- KGS Counties Panel

    pnlKGSCounty.setLayout(new BorderLayout());
    pnlKGSCounty.setPreferredSize(new Dimension(412, 60));

    // -- List of Counties Panel

    if (_COUNTIES > 0)
    {
      iRows = (_COUNTIES + 1) / 2;
      pnlCounties.setLayout(new GridLayout(iRows, 2));

      btn = new JButton[_COUNTIES];

      for (int i=0; i<_COUNTIES; i++)
      {
        btn[i] = new JButton();
        btn[i].setFont(new java.awt.Font("Dialog", 1, 11));
        if (_NAMES[i] != null)
          btn[i].setText(_NAMES[i]);
        btn[i].addActionListener(this);
        pnlCounties.add(btn[i], null);
      }
    }

    // Measured Section List Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlCenter.setBorder(titledBorderList);

    // .. Measured Section List Table

    pTable = new kgsBrineTable();
    scroll = pTable.getScrollPane();

    // .. Measured Section List Buttons

    pnlDataButtons.setLayout(new BorderLayout());

    btnSelect.setFont(new java.awt.Font("Dialog", 1, 11));
    btnSelect.setText("Load Data");
    btnSelect.addActionListener(this);

    // Set Buttons depending on the data

    setButtons();

    // Attach Widgets to Frame

    this.getContentPane().add(pnlKGS,    BorderLayout.NORTH);
    pnlKGS.add(pnlKGSTexts,              BorderLayout.CENTER);
    pnlKGSTexts.add(pnlKGSSearchBy,      BorderLayout.WEST);
    pnlKGSSearchBy.add(rbLease,          null);
    pnlKGSSearchBy.add(rbTRS,            null);
    pnlKGSSearchBy.add(rbCounty,         null);

    groupSearchText.add(rbLease);
    groupSearchText.add(rbTRS);
    groupSearchText.add(rbCounty);

    pnlKGSTexts.add(pnlKGSTextSearch,    BorderLayout.CENTER);
    pnlKGSTextSearch.add(pnlKGSTRS,      "TRS");
    pnlKGSTRS.add(pnlSection,            null);
    pnlSection.add(txtSection,           BorderLayout.EAST);

    pnlKGSTextSearch.add(pnlKGSText,     "TEXT");
    pnlKGSText.add(txtText,              BorderLayout.CENTER);

    pnlKGSTextSearch.add(pnlKGSCounty,   "COUNTY");
    pnlKGSCounty.add(scrollCounties,     BorderLayout.CENTER);
    scrollCounties.getViewport().add(pnlCounties, null);

    pnlKGSTRS.add(pnlTownship,           null);
    pnlTownship.add(txtTownship,         BorderLayout.WEST);
    pnlTownship.add(rbNorth,             BorderLayout.CENTER);
    pnlTownship.add(rbSouth,             BorderLayout.EAST);

    groupTownship.add(rbNorth);
    groupTownship.add(rbSouth);

    pnlKGSTRS.add(pnlRange,              null);
    pnlRange.add(txtRange,               BorderLayout.WEST);
    pnlRange.add(rbEast,                 BorderLayout.EAST);
    pnlRange.add(rbWest,                 BorderLayout.CENTER);

    groupRange.add(rbEast);
    groupRange.add(rbWest);

    pnlKGS.add(pnlSearch,                BorderLayout.SOUTH);
    pnlSearch.add(btnSearch,             BorderLayout.CENTER);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(scroll,                BorderLayout.CENTER);

    pnlCenter.add(pnlDataButtons,        BorderLayout.SOUTH);
    pnlDataButtons.add(btnSelect,        BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(540, 700));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(true);
    this.setVisible(true);
  }

  /*============+==========================================================*
   * -------------------------- GET METHODS -------------------------------*
   *=======================================================================*/

  /** Method getData()
   * <p> This method will retrieve the Brine Data.
   * @return stBrine = Brine Data
   */

  public brineListStruct getData() { return (stData); }

  /** Method getHeader()
   *  <p> This method will retrieve the Header Data
   *  @return stHeader = Header data structure
   */

  public iqstratHeadersStruct getHeader() { return (stHeader); }

  /** Method getList()
   * <p> This method will get the KGS Database for measured sections that fit
   *     the constraint passed in.
   */

  private void getList()
  {
	ReadBrineDataXMLFile pRead =
	  new ReadBrineDataXMLFile(brine.io.ReadBrineDataXMLFile.URL);
    stList = pRead.Process( sPath );

    pTable.repopulateList(stList);
  }

  /** Method getCounties()
   * <p> This methodw will get the KGS Database for measured sections that fit
   *     the constraint passed in.
   */

  private void getCounties()
  {
    kgsMeasSectListStruct st = readCounties();

    if (st != null)
    {
      _COUNTIES = st.iCount;
      _CODES    = new int[_COUNTIES];
      _NAMES    = new String[_COUNTIES];

      for (int i=0; i<_COUNTIES; i++)
      {
        _CODES[i] = st.stItem[i].iCounty;
        _NAMES[i] = new String(st.stItem[i].sCounty);
      }
    }
  }

  /** Method readCounties()
   * <p> This method will retrieve the list of counties that have Measured
   *     sections in the KGS Database.
   */

  public static kgsMeasSectListStruct readCounties()
  {
    kgsMeasSectListStruct st = null;

    ReadMeasSectXMLFile pRead =
        new ReadMeasSectXMLFile(kgs.io.ReadMeasSectXMLFile.URL);

    st = pRead.Process( cmn.cmnStruct.KGS_BRINE_COUNTIES );

    String sError = pRead.GetError();
    if (sError.length() > 0)
      JOptionPane.showMessageDialog( (Component)null,
                                     sError,
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);
    return (st);
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setButtons()
   * <p> This method will set the buttons depending on the data present.
   */

  private void setButtons()
  {
    if (btnSearch != null)
    {
      btnSearch.setEnabled(true);

      if (rbCounty.isSelected())
        btnSearch.setEnabled(false);
    }

    if (btnSelect != null)
    {
      btnSelect.setEnabled(false);

      if (pTable.getTotalRows() > 0)
        btnSelect.setEnabled(true);
    }
  }

  /** Method setHeader()
   *  <p> This method will load the header information from the
   *      Brine Data
   *  @param  st = brine data list structure
   *  @return st = brine data list structure
   */

  private brineListStruct setHeader(brineListStruct st)
  {
	int i,j;
	double dStart = 0.0;
	double dEnd   = 0.0;
    String sForm[] = null;
    int    iForm[] = null;
    int    ifrm    = 0;
    int    iFound  = 0;
    int    iCount  = 0;

    if (st != null)
    {
      stHeader = new iqstratHeadersStruct();
      sForm    = new String[st.iCount];
      iForm    = new int[st.iCount];

      for (i=0; i<st.iCount; i++)
      {
	    sForm[i] = new String("");
	    iForm[i] = 0;
      }

      for (i=0; i<st.iCount; i++)
      {
	    if (st.stItem[i].sAPI.length() > 0)
	    {
          stHeader.sAPI = new String(st.stItem[i].sAPI);
          st.sAPI       = new String(st.stItem[i].sAPI);
		}

		if (st.stItem[i].sName.length() > 0)
		{
		  stHeader.sName = new String(st.stItem[i].sName);
		  st.sName       = new String(st.stItem[i].sName);
		}

		if (st.stItem[i].sField.length() > 0)
		{
		  stHeader.sField = new String(st.stItem[i].sField);
		  st.sField       = new String(st.stItem[i].sField);
		}

		if (st.stItem[i].sCounty.length() > 0)
		{
		  stHeader.sCounty = new String(st.stItem[i].sCounty);
		  st.sCounty       = new String(st.stItem[i].sCounty);
		}

		if (st.stItem[i].state.length() > 0)
		{
		  stHeader.state = new String(st.stItem[i].state);
		  st.state       = new String(st.stItem[i].state);
		}

		if (st.stItem[i].sLocation.length() > 0)
		{
		  stHeader.sLocation = new String(st.stItem[i].sLocation);
          stHeader = las3.las3LoadWellHeader.parseSTR(
			  stHeader.sLocation, stHeader );
		  st.sLocation       = new String(st.stItem[i].sLocation);
		}

		if (st.stItem[i].dLatitude != 0.0)
		{
		  stHeader.dLatitude = st.stItem[i].dLatitude;
		  st.dLatitude       = st.stItem[i].dLatitude;
		}

		if (st.stItem[i].dLongitude != 0.0)
		{
		  stHeader.dLongitude = st.stItem[i].dLongitude;
		  st.dLongitude       = st.stItem[i].dLongitude;
		}

		if (st.stItem[i].dKB > 0.0)
		{
		  stHeader.dKB = st.stItem[i].dKB;
		  st.dKB       = st.stItem[i].dKB;
		}

		if (st.stItem[i].dGL > 0.0)
		{
		  stHeader.dGL = st.stItem[i].dGL;
		  st.dGL       = st.stItem[i].dGL;
		}

		if (st.stItem[i].dDF > 0.0)
		{
		  stHeader.dDF = st.stItem[i].dDF;
		  st.dDF       = st.stItem[i].dDF;
		}

        if (st.stItem[i].dTOP > dStart)
          dStart = st.stItem[i].dTOP;

        if (st.stItem[i].dBASE > dEnd)
          dEnd = st.stItem[i].dBASE;

		if (st.stItem[i].source.length() > 0)
		{
		  st.source = new String(st.stItem[i].source);
		}

		if (st.stItem[i].sRecovery.length() > 0)
		{
		  st.sRecovery = new String(st.stItem[i].sRecovery);
		}

		if (st.stItem[i].sFORM.length() > 0)
		{
		  iFound = 0;
		  for (j=0; j<iCount; j++)
		  {
		    if (sForm[j].equals(st.stItem[i].sFORM))
		    {
			  iForm[j]++;
		      iFound = 1;
		    }
		  }

		  if (iFound == 0)
		  {
		    sForm[iCount] = new String(st.stItem[i].sFORM);
		    iForm[iCount] = 1;
		    iCount++;
		  }
		}
	  }

	  if ((dStart > 0.0) && (dEnd > 0.0))
	    stHeader.depth = (dEnd - dStart);

	  if (dStart > 0.0)
        st.depthStart = dStart;     // Brine Top Depth {F}
	  if (dEnd > 0.0)
        st.depthEnd   = dEnd;   // Brine Base Depth {F}

	  if (iCount > 0)
	  {
		iFound = -1;

	    for(i=0; i<iCount; i++)
	    {
		  if (iForm[i] > ifrm)
		  {
			ifrm   = iForm[i];
		    iFound = i;
		  }
		}

		st.sFormation = new String( sForm[iFound] );
	  }
	}

    return (st);
  }

  /** Method loadData()
   * <p> This method will retrieve the measured section information
   */

  private void loadData()
  {
    stData = pTable.getData();

    stData = setHeader(stData);

//brine.brineUtility.print(stData);
    if (stData != null)
      notifier.notifyObservers(new String("KGS Brine Loaded"));
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    // Input Variables

    notifier = null; // Observable Object

    // Global Frame Variables

    sTownship = "S";   // Township Direction
    sRange    = "W";   // Range Direction

    sPath     = "";    // URL to retrieve Well List

    _CODES    = null;  // Array of County Codes
    _NAMES    = null;  // Array of County Names

    // Search By

    if (pTable != null)
      pTable.close();
    pTable   = null; // Display Measured Sections

    if (stList != null)
      stList.delete();
    stList   = null; // Brine List Data Structure
    stBrine  = null; // Brine Structure
    stData   = null; // Brine List Data Structure Selected

    // KGS Search Widgets

    // -- Type of text search panel

    pnlKGSTextSearch = null;     // Search By Panel
    cardLayout       = null; // Search By Layout

    // -- KGS - Type of Text Search

    rbLease     = null;  // Search by Lease button
    rbTRS       = null;  // Search by TRS button
    rbCounty    = null;  // Search by County button

    // Search for Well List Button

    btnSearch   = null;

    // -- KGS - Lease Name Search Text Fields

    txtText     = null;

    // -- KGS - Township Range Section Search Text Fields

    // .. Township Widgets

    txtTownship = null;
    rbNorth     = null;
    rbSouth     = null;

    // .. Range Widgets

    txtRange    = null;
    rbEast      = null;
    rbWest      = null;

    // .. Section Widgets

    txtSection  = null;

    // -- KGS - List of Counties Widgets

    for (int i=0; i<_COUNTIES; i++)
      btn[i] = null;

    btnSelect   = null;  // Select Measured Section

    mExit       = null; // Exit Profile Plot

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // KGS Search Widgets

    if (event.getSource() == btnSearch) { getList(); }

    // -- KGS - Type of Text Search

    if (event.getSource() == rbLease)
    {
      iSearch = _LEASE;
      cardLayout.show(pnlKGSTextSearch, "TEXT");
    }
    if (event.getSource() == rbTRS)
    {
      iSearch = _TRS;
      cardLayout.show(pnlKGSTextSearch, "TRS");
    }
    if (event.getSource() == rbCounty)
    {
      iSearch = _COUNTY;
      cardLayout.show(pnlKGSTextSearch, "COUNTY");
    }

    // -- KGS - Township Range Section Search Text Fields

    if (event.getSource() == rbNorth) { sTownship = "N"; }
    if (event.getSource() == rbSouth) { sTownship = "S"; }

    if (event.getSource() == rbEast) { sRange = "E"; }
    if (event.getSource() == rbWest) { sRange = "W"; }

    // -- KGS - List of Counties Buttons

    for (int i=0; i<_COUNTIES; i++)
    {
      if (event.getSource() == btn[i])
      {
        sPath = cmn.cmnStruct.KGS_BRINE + "county=" + _NAMES[i];
//                "code=" + _CODES[i] + "&county=" + _NAMES[i];
        getList();
      }
    }

    // Load Data

    if (event.getSource() == btnSelect) { loadData(); }

    setButtons();

    if (event.getSource() == mExit) { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtText)
    {
      if (txtText.getText().length() > 0)
      {
          sPath = cmn.cmnStruct.KGS_BRINE +
                  "sLease="+txtText.getText();
      }
    }

    // -- KGS - Township Range Section Search Text Fields

    if (e.getSource() == txtTownship)
    {
      iNumeric = 1;
      sTemp = txtTownship.getText();
      sMessage = new String("Township Value is a Numeric Field");
    }
    if (e.getSource() == txtRange)
    {
      iNumeric = 1;
      sTemp = txtRange.getText();
      sMessage = new String("Range Value is a Numeric Field");
    }
    if (e.getSource() == txtSection)
    {
      iNumeric = 1;
      sTemp = txtSection.getText();
      sMessage = new String("Section Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtTownship)
          iTownship = cmn.cmnString.stringToInt(txtTownship.getText());
        if (e.getSource() == txtRange)
          iRange = cmn.cmnString.stringToInt(txtRange.getText());
        if (e.getSource() == txtSection)
          iSection = cmn.cmnString.stringToInt(txtSection.getText());

        if ((iTownship > 0) && (iRange > 0) && (iSection > 0))
        {
          sPath = cmn.cmnStruct.KGS_BRINE +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange    +
              "&iSection=" + iSection;
        }
        else if ((iTownship > 0) && (iRange > 0))
        {
          sPath = cmn.cmnStruct.KGS_BRINE +
              "iTownship=" + iTownship + "&sTownship=" + sTownship +
              "&iRange="   + iRange    + "&sRange="    + sRange;
        }
      }
    }
  }
}

/** CLASS kgsBrineFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class kgsBrineFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  kgsBrineFrame adaptee;

  kgsBrineFrameFocusAdapter(kgsBrineFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 06/03/2011
 *  @author  John Victorine
 */
