/*
 * @iqstratPlotRemarks.java Version 1.1 09/25/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.plot;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import iqstrat.iqstratRemarkListStruct;

/** CLASS iqstratPlotRemarks
 *  <p> This Class will create the Remarks Track.
 *
 *  @version 1.1 09/25/2007
 *  @author  John R. Victorine
 */

public class iqstratPlotRemarks extends Canvas
{
  // Input Variables

  private int    iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart = 0.0;   // Starting Depth
  private double depthEnd   = 0.0;   // Ending Depth
  private int    iScale     = -1;    // Plot Scale

  // Global Variables

  private static final int CHARS  = 32; //30;   // Number of Character Widths
  private static final int CHAR_H =  7;//8;   // Character Pixel Height Size

  private iqstratRemarkListStruct stRemarks = null; // Remarks Data List Struct

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR iqstratPlotRemarks( int iDataType,
   *                                  double depthStart,
   *                                  double depthEnd,
   *                                  int iScale )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   */

  public iqstratPlotRemarks( int iDataType,
                             double depthStart,
                             double depthEnd,
                             int iScale )
  {
    this.iDataType = iDataType;
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close() { stRemarks = null; }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getFossilData()
   * <p> This method will get an existing Fossil List Data Structure
   * @return stFossil = Fossil List Data Structure
   */

  public iqstratRemarkListStruct getRemarksData() { return (stRemarks); }

  /** Method getNextDepth( int iRow, double diff )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow   = the present row
   * @param  iDepth = the last pixel depth
   * @return iNext  = The next Row
   */

  private int getNextDepth( int iRow, int iDepth )
  {
    int    iNext   = -1;
    int    iDepth1 = 0;
    double depth1  = 0.0;
    double dRatio  = 0.0;

    if (iRow > -1)
    {
      if (stRemarks != null)
      {
        for (int i=iRow; i<stRemarks.iCount; i++)
        {
          depth1 = stRemarks.stItem[i].depthStart;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;

          if (iNext == -1)
          {
            if (iDepth1 > iDepth)
            {
              iNext = i;
              break;
            }
          }
        }
      }
    }

    return (iNext);
  }

  /** Method getNextDepth( int iRow, double diff )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow  = the present row
   * @param  diff  = the difference between text
   * @return iNext = The next Row
   */

  private int getNextDepth( int iRow, double diff )
  {
    int iNext = -1;
    int iValue = (int) diff;

    diff = (double) iValue;

    if (iRow > -1)
    {
      if (stRemarks != null)
      {
        for (int i=iRow; i<stRemarks.iCount; i++)
        {
          if ((iNext == -1) &&
              (stRemarks.stItem[i].depthStart >=
                 stRemarks.stItem[iRow].depthEnd + diff))
          {
            iNext = i;
            break;
          }
        }
      }
    }

    return (iNext);
  }

  /** Method getNextDepth( int iRow )
   * <p> This method will retrieve the next depth data in the list in such
   *     a way that the text does not overwrite
   * @param  iRow  = the present row
   * @return iNext = The next Row
   */

  private int getNextDepth( int iRow )
  {
    int    iNext = -1;
    int    iDepth1  = 0;
    int    iDepth2  = 0;
    int    iDepth   = 0;
    double dRatio   = 0.0;
    double depth1   = 0.0;
    double depth2   = 0.0;

    if (iRow > -1)
    {
      if (stRemarks != null)
      {
        for (int i=iRow; i<stRemarks.iCount; i++)
        {
          depth1 = stRemarks.stItem[i].depthStart;
          depth2 = stRemarks.stItem[i].depthEnd;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;

          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio = (double) iLogHeight * (depthStart - depth2) /
                                         (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;

          iDepth = iDepth2-iDepth1;

          if ((iNext == -1) &&
              (depth1 >= stRemarks.stItem[iRow].depthEnd) &&
              (iDepth >= CHAR_H))
          {
            iNext = i;
            break;
          }
        }
      }
    }

    return (iNext);
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setRemarksData( iqstratRemarkListStruct stRemarks )
   * <p> This method will set an existing Remarks List Data Structure
   * @param stRemarks = Remarks List Data Structure
   */

  public void setRemarksData( iqstratRemarkListStruct stRemarks )
  {
    this.stRemarks = stRemarks;
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   * @param iStartTrack = The starting pixel for the LAS Track
   * @param iWidthTrack = The width of LAS Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/5;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;
/*
      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if (iY2 >= PLOT_TITLES)
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }
*/
//      g.setColor(Color.gray);
      g.setColor(Color.lightGray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawString("Remarks", iStartTrack+2*iIncrement, LABELSTART+14);

    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawRemarks( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the remarks images that were selected
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawRemarks( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int             iStart   = iStartTrack;
    int             iEnd     = iStartTrack+iWidthTrack;
    int             iDepth1  = 0;
    int             iDepth2  = 0;
    int             iDepth   = 0;
    int             iCheck   = 0;
    int             iTemp    = 0;
    double          dRatio   = 0.0;
    double          depth1   = 0.0;
    double          depth2   = 0.0;
    double          diff     = 0.0;
    int             iCount   = 0;
    int             len1     = 0;
    int             len2     = 0;
//    StringTokenizer stToken  = null;
    String          sText    = "";
    String          sTemp    = "";
    String          sRemarks = "";
    Font            fsb      = new Font("Monospaced", Font.BOLD, 10); // new Font("Arial", Font.BOLD, 10);
    FontMetrics     fsbm     = g.getFontMetrics(fsb);
    String          sDelimiter = new String("[ ,;]+");
    String          tokens[]   = null;

    g.setFont( fsb );
    g.setColor(Color.black);

    diff = iqstrat.iqstratTracksStruct.SCALE[iScale] / 8.0;

    if (stRemarks != null)
    {
      for (int i=0; i<stRemarks.iCount; i++)
      {
        if (stRemarks.stItem[i] != null)
        {
          depth1 = stRemarks.stItem[i].depthStart;
          depth2 = stRemarks.stItem[i].depthEnd;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;

          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          dRatio = (double) iLogHeight * (depthStart - depth2) /
                                         (depthStart - depthEnd);
          iDepth2 = PLOT_TITLES + (int) dRatio;

          iDepth = iDepth1;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth = iDepth2;

          if ((iDepth2-iDepth1) < CHAR_H) iDepth2 = iDepth1 + CHAR_H+2;

          if ((iDepth1 >= PLOT_TITLES) && (iDepth2 > PLOT_TITLES) &&
              (iDepth1 < iHeight)     && (iDepth2 <= iHeight))
          {
            iDepth  = iDepth + CHAR_H;
            iCheck  = iDepth;

            sText   = stRemarks.stItem[i].sText;
            sText   = new String(sText.replace('\t', ' '));
            sText   = new String(sText.replace('\n', ' '));
//            stToken = new StringTokenizer(sText, " ");

//            iCount  = stToken.countTokens();

            tokens = sText.split( sDelimiter );
            iCount = tokens.length;

            sRemarks = new String("");
            for (int j=0; j<iCount; j++)
            {
//              sTemp  = stToken.nextToken();
              sTemp  = new String(tokens[j]);

              len1   = sRemarks.length();
              len2   = sTemp.length()+1;

              if (len1+len2 < CHARS)
              {
                sRemarks = new String(sRemarks + " " + sTemp);
              }
              else
              {
                if (iDepth < iDepth2)
                  g.drawString(sRemarks, iStartTrack+5, iDepth);
                else iCheck = iDepth2-3; // Backing off just a little to allow next depth

                sRemarks = new String(sTemp);
                iDepth   = iDepth + CHAR_H;
              }
            }

            iTemp  = getNextDepth( i, iCheck );

            if (iTemp > 0)
            {
              if (iTemp-1 > i) i = iTemp-1;
            }

            if (sRemarks.length() != 0)
            {
              if (iDepth < iDepth2)
                g.drawString(sRemarks, iStartTrack+5, iDepth);
            }
          }
        }
      }
    }

    g.setColor(Color.black);
    g.drawLine(iStart, LABELSTART, iStart, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEnd,   LABELSTART, iEnd,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a Remarks Plot Track
   * @param g - Graphics Parameter.
   * @param iSelected   = The Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void draw( Graphics g,
                    int iSelected,
                    int iStartTrack,
                    int iWidthTrack )
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_ROCK;

    if (iSelected == iqstrat.iqstratTracksStruct._REMARKS)
    {
      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawGrid(g, iStartTrack, iWidthTrack);
      drawRemarks(g, iStartTrack, iWidthTrack);
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g - Graphics Parameter.
   */

  public void paint( Graphics g )
  {
   g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}