/*
 * @iqstratPlot.java Version 1.1 09/04/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.plot;

import java.awt.*;
import java.awt.geom.*;
import java.awt.event.*;
import java.awt.image.*;
import java.util.Observable;
import javax.swing.*;

import cmn.cmnStruct;

import iqstrat.iqstratStruct;

import iqstrat.plot.iqstratPlotRemarks;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;

import las.lasFileDataStruct;
import las.plot.lasPlotTrack;
import las.plot.lasPlotThinPhiTrack;
//import las.gui.lasThinTrackFrame;
import las.gui.lasConvertFrame;

//import lith.gui.lithColorlithFrame;
import lith.gui.lithImageTrackFrame;
import lith.plot.lithPlotColorlith;

import lith.lithology.plot.lithologyPlotSymbolLegend;
import lith.lithology.plot.lithologyPlotTrack;
import lith.lithology.lithologyListStruct;
import lith.lithology.lithologySymbolsListStruct;

import lith.texture.plot.texturePlotTrack;
import lith.texture.textureListStruct;

import lith.rock.plot.rockColumnPlotLegend;
import lith.rock.plot.rockColumnPlotTrack;
import lith.rock.rockColumnListStruct;

import horizon.regions.regionsListStruct;
import horizon.regions.regionsStruct;

import horizon.strat.plot.stratPlotTrack;
import horizon.strat.stratListStruct;

import horizon.seq.plot.seqPlotTrack;
import horizon.seq.seqListStruct;

import horizon.env.plot.envPlotTrack;
import horizon.env.envListStruct;

import horizon.bio.plot.bioStratPlotTrack;
import horizon.bio.bioStratListStruct;

import brine.brineListStruct;
import brine.plot.brinePlotTrack;
import brine.plot.brinePlotImageTrack;

import rock.color.plot.rockColorPlotTrack;
import rock.color.rockColorListStruct;
import rock.color.rockColorStruct;

import rock.fossil.plot.fossilPlotTrack;
import rock.fossil.fossilListStruct;
import rock.fossil.fossilStruct;

import rock.sedimentary.plot.sedimentaryPlotTrack;
import rock.sedimentary.sedimentaryListStruct;
import rock.sedimentary.sedimentaryStruct;

import rock.rockFileDataStruct;
import rock.rockDataListStruct;
import rock.plot.rockPlotTrack;

import rock.rockImagesListStruct;
import rock.plot.rockPlotImageTrack;
import rock.gui.rockImageViewFrame;

import rock.phi.phiListStruct;
import rock.phi.plot.phiPlotTrack;

import pfeffer.pfefferDataStruct;
import pfeffer.pfefferDataListStruct;
import pfeffer.plot.pfefferPlotTrack;

/** CLASS iqstratPlot
 *  <p> This Class will create the IQSTRAT - Stratigraphic Plot.
 *
 *  @version 1.1 09/04/2007
 *  @author  John R. Victorine
 */

public class iqstratPlot extends Canvas
    implements MouseListener, MouseMotionListener
{
  public static final int _SOURCE      = iqstrat.iqstratTracksStruct._SOURCE;
  public static final int _TYPE        = iqstrat.iqstratTracksStruct._TYPE;
  public static final int _DESCRIPTION = iqstrat.iqstratTracksStruct._DESCRIPTION;
  public static final int _STATUS      = iqstrat.iqstratTracksStruct._STATUS;

  // Input Variables

  private int    iUnit       = 0;     // Lithology/Texture Data Plot Sampling
  private int    iDataType   = -1;    // Format of Data, LAS File or Outcrop
  private double depthStart  = 0.0;   // Starting Depth
  private double depthEnd    = 0.0;   // Ending Depth
  private int    iScale      = -1;    // Plot Scale
  private int    iSelected[] = null;  // Selected Tracks

  private int    iLASSize    = iqstrat.gui.iqstratAvailablePanel._DOUBLE;

  private Cursor normalCursor = null; // Normal Cursor
  private Cursor mouseCursor  = null; // Edit Track Cursor

  private Observable notifier = null; // Observable

  private cmnStruct         stCMN    = null; // Global Common Data Structure
  private iqstratStruct     stStruct = null; // Global IQSTRAT Data Structure
  private regionsListStruct stOther  = null; // DST & Perforations

  private int    iStart[]    = null;  // Array of Track Starting X-Position
  private int    iWide[]     = null;  // Array of Track Widths
  private int    iButtons    = 0;     // Total Number of Tracks

  private int    iBio        = 10;    // Total number of fossils/depth/well
  private int    iSed        = 20;    // Total number of structurs/depth/well
  private int    iBioS       = 10;    // Total number of fossils/depth/well

//  private int    iFossils[]  = null;  // Fossil selected indicator
  private BufferedImage imgFOSSIL = null;
  private BufferedImage imgSED    = null;
  private BufferedImage imgPHI    = null;

  private JTextField txtDepth = null; // Depth Textfield

  private lasPlotTrack          plotLAS       = null; // LAS Plot Tracks
  private lasPlotThinPhiTrack   plotThinPhi   = null; // Thin Phi Track
  private rockColorPlotTrack    plotRockColor = null; // Rock Color Track

  private lithologyPlotSymbolLegend pLegend   = null; // Plot Legend

  private lithPlotColorlith     plotColorlith = null; // Colorlith Track
  private lithologyPlotTrack    plotLithology = null; // Rock Column Track
  private lithologyPlotTrack    plotRTLithology = null; // Rock Column Track by Rhomaa-DTmaa
  private texturePlotTrack      plotTexture   = null; // Rock Texture Track

//  private lithColorlithFrame    pColorlith    = null; // Colorlith Frame

  private rockColumnPlotTrack   pRockColumn   = null; // Rock Column Plot Track
  private rockColumnPlotLegend  pRockLegend   = null; // Rock Column Plot Legend

  private fossilPlotTrack       plotFossils   = null; // Fossils Plot Track
  private bioStratPlotTrack     plotBioStrat  = null; // Bio stratigraphy Track

  private iqstratPlotRemarks    plotRemarks   = null; // Remarks Track

  private stratPlotTrack        plotStrat     = null; // Stratigraphic Unit Plot
  private seqPlotTrack          plotSequence  = null; // Sequence Plot Track
  private envPlotTrack          plotEnviron   = null; // Depositional Environment

  private sedimentaryPlotTrack  plotSedimentary = null; // Sedimentary Track

  private rockPlotTrack         plotLASRock   = null; // Rock Data LAS Plot
  private phiPlotTrack          plotPhi       = null; // Porosity Type Track
  private rockPlotImageTrack    plotImages    = null; // Rock Image Track
  private rockImagesListStruct  stImages      = null; // Rock Images Structure
  private rockImageViewFrame    pViewImage    = null; // View Rock Image

  private pfefferPlotTrack      plotPfeffer   = null; // PfEFFER Plot Track

  private brinePlotTrack        plotBrine     = null; // Brine Plot Track
  private brinePlotImageTrack   plotBrineImg  = null; // Brine Plot Image Track

  // Data Entry Dialogs

//  private lasThinTrackFrame     pnlThin    = null; // Thin Track Selection Frame
  private lithImageTrackFrame   pnlImage   = null; // OHM & PHI Image Track Frame
  private lasConvertFrame       pnlConvert = null; // Convert Counts to Log Frame

  // Plot Variables

  private int    iWidth           = 650;  // Width of Plot
  private int    iHeight          = 100;  // Height of Plot with the Titles
  private int    iLogHeight       = 100;  // Height of Plot without the Titles
  private int    iIncrementY      = 100;  // Increment Height
  private int    iLegend          = 125;  // Height of LAS Lithology Legend
  private int    jLegend          = 200;  // Width of Rock Legend

  private double dIncrementDepth  = 0.0;  // Increment Depth
  private int    iStartDepthLabel = 10;   // Start Depth Label at

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int DEPTH_TRACK = iqstrat.iqstratTracksStruct.DEPTH_TRACK;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  // Plot Title

  private String sTitle1 = "";
  private String sTitle2 = "";

  // Mouse Action Variables

  private int iAction      = -1;
  private int iAddBox      = 0;
  private int iAddStart    = 0;
  private int iAddWidth    = 0;

  private int    iXBegin   = 0;
  private int    iXEnd     = 0;
  private int    iYBegin   = 0;
  private int    iYEnd     = 0;

  private int    iPanel    = -1;
  private double dStart    = 0.0;
  private double dEnd      = 0.0;

  /** CONSTRUCTOR iqstratPlot()
   *  <p> This is the constructor for this class.
   *  @param stStruct   = Global IQSTRAT Data Structure
   *  @param iDataType  = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param iSelected  = The selected tracks for plotting
   */

  public iqstratPlot( iqstratStruct stStruct,
                      int iDataType,
                      double depthStart,
                      double depthEnd,
                      int iScale,
                      int iSelected[] )
  {
    this.iDataType = iDataType;
    iUnit          = stStruct.iUnit;

    setPlotWidth(iSelected);
    setPlotHeight(iScale, depthEnd, depthStart);

    plotLAS       = new lasPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotThinPhi   = new lasPlotThinPhiTrack(
        iDataType, depthStart, depthEnd, iScale);

    plotColorlith = new lithPlotColorlith(
        iDataType, depthStart, depthEnd, iScale);
    plotLithology = new lithologyPlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotRTLithology = new lithologyPlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotTexture   = new texturePlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);

    pLegend       = new lithologyPlotSymbolLegend( stStruct.stSymbols );
    pRockLegend   = new rockColumnPlotLegend( stStruct.stParseRock );

    pRockColumn   = new rockColumnPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stParseRock);

    plotPfeffer   = new pfefferPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stSymbols);

    plotLASRock   = new rockPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotPhi       = new phiPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stPorosity);
    plotFossils   = new fossilPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stFossils);
    plotBioStrat  = new bioStratPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stFossils);
    plotSedimentary = new sedimentaryPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stSedStruct);
    plotRockColor = new rockColorPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotRemarks   = new iqstratPlotRemarks(iDataType, depthStart, depthEnd, iScale);

    plotBrine     = new brinePlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotBrineImg  = new brinePlotImageTrack(iDataType, depthStart, depthEnd, iScale);

    plotStrat   = new stratPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stICS);
    plotSequence = new seqPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotEnviron  = new envPlotTrack(iDataType, depthStart, depthEnd, iScale);

    plotImages  = new rockPlotImageTrack(
        iDataType, depthStart, depthEnd, iScale);
    pViewImage  = new rockImageViewFrame();

    addMouseListener(this);
    addMouseMotionListener(this);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR iqstratPlot()
   *  <p> This is the constructor for this class.
   *  @param stStruct   = The Global IQSTRAT Data Structure
   *  @param iSelected  = The selected tracks for plotting
   *  @param normalCursor = Normal Cursor
   *  @param mouseCursor  = Hand Cursor
   */

  public iqstratPlot( iqstratStruct stStruct, int iSelected[],
                      Cursor normalCursor,  Cursor mouseCursor)
  {
    this.stStruct = stStruct;

    this.normalCursor = normalCursor;
    this.mouseCursor  = mouseCursor;

    iDataType     = stStruct.iDataType;
    depthStart    = stStruct.depthStart;
    depthEnd      = stStruct.depthEnd;
    iScale        = stStruct.iScale;
    iUnit         = stStruct.iUnit;

    setPlotWidth(iSelected);
    setPlotHeight(iScale, depthEnd, depthStart);

    plotLAS       = new lasPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotThinPhi   = new lasPlotThinPhiTrack(
        iDataType, depthStart, depthEnd, iScale);

    plotColorlith = new lithPlotColorlith(
        iDataType, depthStart, depthEnd, iScale);
    plotLithology = new lithologyPlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotRTLithology = new lithologyPlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);
    plotTexture   = new texturePlotTrack(
        iDataType, depthStart, depthEnd, iScale, iUnit, stStruct.stSymbols);

    pLegend       = new lithologyPlotSymbolLegend( stStruct.stSymbols );
    pRockLegend   = new rockColumnPlotLegend( stStruct.stParseRock );

    pRockColumn   = new rockColumnPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stParseRock);

    plotPfeffer   = new pfefferPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stSymbols);

    plotLASRock   = new rockPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotPhi       = new phiPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stPorosity);
    plotRockColor = new rockColorPlotTrack(
        iDataType, depthStart, depthEnd, iScale);
    plotFossils   = new fossilPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stFossils);
    plotBioStrat  = new bioStratPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stFossils);
    plotSedimentary = new sedimentaryPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stSedStruct);
    plotRemarks   = new iqstratPlotRemarks(
        iDataType, depthStart, depthEnd, iScale);

    plotBrine     = new brinePlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotBrineImg  = new brinePlotImageTrack(iDataType, depthStart, depthEnd, iScale);

    plotStrat   = new stratPlotTrack(
        iDataType, depthStart, depthEnd, iScale, stStruct.stICS);
    plotSequence = new seqPlotTrack(iDataType, depthStart, depthEnd, iScale);
    plotEnviron  = new envPlotTrack(iDataType, depthStart, depthEnd, iScale);

    plotImages  = new rockPlotImageTrack(
        iDataType, depthStart, depthEnd, iScale);
    pViewImage  = new rockImageViewFrame();

    addMouseListener(this);
    addMouseMotionListener(this);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will set object to null to force JAVA to reallocate memory
   */

  public void close()
  {
    normalCursor = null;
    mouseCursor  = null;

    stOther      = null;

    iSelected = null;
    iStart    = null;
    iWide     = null;

//    iFossils  = null;  // Fossil selected indicator
    imgFOSSIL = null;
    imgSED    = null;
    imgPHI    = null;

    if (plotLAS != null)
      plotLAS.close();
    plotLAS = null; // LAS Plot Tracks

    if (plotLASRock != null)
      plotLASRock.close();
    plotLASRock = null;

    if (plotBrine != null)
      plotBrine.close();
    plotBrine = null;

    if (plotBrineImg != null)
      plotBrineImg.close();
    plotBrineImg = null;

    if (plotPfeffer != null)
      plotPfeffer.close();
    plotPfeffer = null;

    if (plotImages != null)
      plotImages.close();
    plotImages = null;

    if (pViewImage != null)
      pViewImage.close();
    pViewImage = null;

    if (plotThinPhi != null)
      plotThinPhi.close();
    plotThinPhi = null;

    if (plotRockColor != null)
      plotRockColor.close();
    plotRockColor = null; // Rock Color Track

    if (plotColorlith != null)
      plotColorlith.close();
    plotColorlith = null;

    if (pRockColumn != null)
      pRockColumn.close();
    pRockColumn = null;

    if (plotLithology != null)
      plotLithology.close();
    plotLithology = null;

    if (plotRTLithology != null)
      plotRTLithology.close();
    plotRTLithology = null;

    if (plotTexture != null)
      plotTexture.close();
    plotTexture = null; // Rock Texture Track

//    pColorlith     = null; // Colorlith Frame

//    pnlThin    = null; // Thin Track Selection Frame
    pnlImage   = null; // OHM & PHI Image Track Frame
    pnlConvert = null; // Convert Counts to Log Frame

    if (plotPhi != null)
      plotPhi.close();
    plotPhi         = null;

    if (plotFossils != null)
      plotFossils.close();
    plotFossils = null; // Fossils Plot Track

    if (plotBioStrat != null)
      plotBioStrat.close();
    plotBioStrat = null;  // Bio Stratigraphy Plot Track

    if (plotRemarks != null)
      plotRemarks.close();
    plotRemarks = null;  // Remarks Track

    if (plotSequence != null)
      plotSequence.close();
    plotSequence = null;

    if (plotEnviron != null)
      plotEnviron.close();
    plotEnviron = null;

    if (plotStrat != null)
      plotStrat.close();
    plotStrat = null; // Stratigraphic Unit Plot

    if (plotSedimentary != null)
      plotSedimentary.close();
    plotSedimentary = null; // Sedimentary Track
  }

  /** Method refresh()
   *  <p> This method will refresh the plot
   */

  public void refresh() { repaint(); }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth+jLegend); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight+iLegend); }

  /** Method getTexture()
   * <p> This method will return the LAS Texture
   * @return LAS Texture Data List Structure
   */

  public textureListStruct getTexture() { return (plotTexture.getTexture()); }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;
    int w = this.getWidth();
    int h = this.getHeight();

    image = new BufferedImage( w, h, BufferedImage.TYPE_INT_RGB );

    Graphics2D g2 = image.createGraphics();
    g2.setColor( Color.white );
    g2.fillRect( 0, 0, w, h );
    this.paint(g2);
    g2.dispose();

    return (image);
  }

  /** Method getDataEntryPanel()
   * <p> This method will return the Data Entry Panel Identifier
   * @return iPanel = the Data Entry Panel Identifier
   */

  public int getDataEntryPanel()   { return (iPanel); }

  /** Method getStartingDepth()
   * <p> This method will return the starting depth for the Data
   * @return dStart = the starting depth for the Data
   */

  public double getStartingDepth() { return (dStart); }

  /** Method getEndingDepth()
   * <p> This method will return the ending depth for the Data
   * @return dEnd = the ending depth for the Data
   */

  public double getEndingDepth()   { return (dEnd); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setObservable(Observable notifier)
   * <p> This method will set the Observerable Object
   * @param notifier = the observable object
   */

  public void setObservable(Observable notifier) { this.notifier = notifier; }

  /* ------------------------------------------------------------------ *
   * ---------------- Set Global Common Data Structure ---------------- *
   * ------------------------------------------------------------------ */

  /** Method setCommon()
   * <p> This method will set the Global Common Data Structure
   *  @param stCMN  = the Global Common Data Structure
   */

  public void setCommon(cmnStruct stCMN)
  {
    this.stCMN = stCMN;

    if (plotImages      != null) plotImages.setCommon(stCMN);
    if (pViewImage      != null) pViewImage.setCommon(stCMN);
  }

  /** Method setSelected()
   * <p> This method will change the selected tracks that are plotted
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setSelected(int iSelected[])
  {
    setPlotWidth(iSelected);
    repaint();
  }

  /** Method setTextfield(JTextField txtDepth)
   * <p> This method will set the depth textfield
   * @param txtDepth = the depth text field for displaying the cursor location
   */

  public void setTextfield(JTextField txtDepth) { this.txtDepth = txtDepth; }

  /** Method setPlotWidth()
   * <p> This method will set the Plot Width
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setPlotWidth(int iSelected[])
  {
    int iTrackWidth = 0;

    iWidth = DEPTH_TRACK;

    this.iSelected = iSelected;
    if (iSelected != null)
      this.iButtons = iSelected.length;

    if (this.iButtons > 0)
    {
      iStart = new int[this.iButtons];
      iWide  = new int[this.iButtons];

      for (int i=0; i<this.iButtons; i++)
      {
        iTrackWidth = 0;
        for (int j=0; j<iqstrat.iqstratTracksStruct.TOTAL_TYPES; j++)
        {
          if (iSelected[i] != -1)
          {
            if (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                iqstrat.iqstratTracksStruct.TYPE[j][0]))
            {
              if ( iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                   "ICON"))
              {
                if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
                  iTrackWidth = iBio;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
                  iTrackWidth = iSed;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._BIO_STRATIGRAPHY)
                  iTrackWidth = iBioS;
              }
              else
              {
                iTrackWidth = cmn.cmnString.stringToInt(
                    iqstrat.iqstratTracksStruct.TYPE[j][2]);

                if ( iLASSize == iqstrat.gui.iqstratAvailablePanel._DOUBLE )
                {
                  if ((iSelected[i] == iqstrat.iqstratTracksStruct._LAS_INDUCTION) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_MICRO_RESISTIVITY) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_RESISTIVITY) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_LATERLOG) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_AHT) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_SONIC) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_LITHO_DENSITY) ||
                      (iSelected[i] == iqstrat.iqstratTracksStruct._LAS_SPECTRAL_GR))
                  {
                    iTrackWidth = 2 * iTrackWidth;
                  }
                }
              }
            }
          }
        }

        iStart[i] = iWidth;
        iWide[i]  = iTrackWidth;
        iWidth    = iWidth + iTrackWidth;
      }
    }
  }

  /** Method setPlotWidth()
   * <p> This method will set the Plot Width
   */

  public void setPlotWidth()
  {
    int iTrackWidth = 0;

    iWidth = DEPTH_TRACK;

    if (iButtons > 0)
    {
      iStart = new int[iButtons];
      iWide  = new int[iButtons];

      for (int i=0; i<iButtons; i++)
      {
        iTrackWidth = 0;
        for (int j=0; j<iqstrat.iqstratTracksStruct.TOTAL_TYPES; j++)
        {
          if (iSelected[i] != -1)
          {
            if (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                iqstrat.iqstratTracksStruct.TYPE[j][0]))
            {
              if ( iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals(
                   "ICON"))
              {
                if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
                  iTrackWidth = iBio;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
                  iTrackWidth = iSed;

                if (iSelected[i] == iqstrat.iqstratTracksStruct._BIO_STRATIGRAPHY)
                  iTrackWidth = iBioS;
              }
              else
                iTrackWidth = cmn.cmnString.stringToInt(
                    iqstrat.iqstratTracksStruct.TYPE[j][2]);
            }
          }
        }


        iStart[i] = iWidth;
        iWide[i]  = iTrackWidth;
        iWidth    = iWidth + iTrackWidth;
      }
    }
  }

  /** Method setPlotHeight(int iScale, double dMaximum, double dMinimum)
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight(int iScale, double dMaximum, double dMinimum)
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];
  }

  /** Method setScale(int iScale, double dMaximum, double dMinimum)
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale(int iScale, double dMaximum, double dMinimum)
  {
    setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotLAS != null)
      plotLAS.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotBrine != null)
      plotBrine.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotBrineImg != null)
      plotBrineImg.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotPfeffer != null)
      plotPfeffer.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotLASRock != null)
      plotLASRock.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotColorlith != null)
     plotColorlith.setPlotHeight(iScale, dMaximum, dMinimum);

    if (pRockColumn != null)
      pRockColumn.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotLithology != null)
      plotLithology.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotRTLithology != null)
      plotRTLithology.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotThinPhi != null)
      plotThinPhi.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotImages != null)
      plotImages.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotRockColor != null)
      plotRockColor.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotTexture != null)
      plotTexture.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotSedimentary != null)
      plotSedimentary.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotPhi != null)
      plotPhi.setPlotHeight( iScale, dMaximum, dMinimum );

    if (plotFossils != null)
      plotFossils.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotBioStrat != null)
      plotBioStrat.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotRemarks != null)
      plotRemarks.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotStrat != null)
      plotStrat.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotSequence != null)
      plotSequence.setPlotHeight(iScale, dMaximum, dMinimum);

    if (plotEnviron != null)
      plotEnviron.setPlotHeight(iScale, dMaximum, dMinimum);

    repaint();
  }

  /** Method setTitles()
   * <p> This method will set the titles for this plot
   * @param sTitle1 = The first title
   * @param sTitle2 = The second title
   */

  public void setTitles(String sTitle1, String sTitle2)
  {
    this.sTitle1 = new String(sTitle1);
    this.sTitle2 = new String(sTitle2);

    repaint();
  }


  /** Method modifyWidths( int iBio, int iSed, int iBioS )
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   * @param iBio  = The total number of fossils per depth per well
   * @param iSed  = The total number of structures per depth per well
   * @param iBioS = The total number of structures per depth per well
   */

  public void modifyWidths( int iBio, int iSed, int iBioS )
  {
    this.iBio  = iBio;
    this.iSed  = iSed;
    this.iBioS = iBioS;

    setPlotWidth();
//    repaint();
  }

  /** Method setPlotLimits()
   *  <p> This method will override the default Plot Limits
   *  @param data_las  = Array of Minimum & Maximum LAS Plot Limits
   *  @param data_core = Array of Minimum & Maximum CORE Plot Limits
   *  @param data_pf   = Array of Minimum & Maximum PfEFFER Plot Limits
   */

  public void setPlotLimits( double data_las[][], double data_core[][], double data_pf[][] )
  {
	plotLAS.setPlotLimits( data_las );
	plotLASRock.setPlotLimits( data_core );
	plotPfeffer.setPlotLimits( data_pf );
    repaint();
  }

  /* ------------------------------------------------------------------ *
   * -------------------------- LAS File Data ------------------------- *
   * ------------------------------------------------------------------ */

  /** Method setLASFileData(lasFileDataStruct stLASFileData)
   * <p> This method will set the LAS File Data Structure
   * @param stLASFileData - LAS File Data Structure
   */

  public void setLASFileData(lasFileDataStruct stLASFileData)
  {
    if (plotLAS != null)
      plotLAS.setLASFileData( stLASFileData );

    if (plotPfeffer != null)
      plotPfeffer.setLASFileData( stLASFileData );

    if (plotColorlith != null)
      plotColorlith.setLASFileData( stLASFileData );

    if (plotLithology != null)
    {
      if (stLASFileData != null)
      {
        if (stLASFileData.iPE == -1)
          plotLithology.setMethod( iqstrat.iqstratTracksStruct._COLORLITH_PD );
        else
          plotLithology.setMethod( iqstrat.iqstratTracksStruct._COLORLITH_RU );
      }
    }

    if (plotRTLithology != null)
    {
      if (stLASFileData != null)
      {
        plotRTLithology.setMethod( iqstrat.iqstratTracksStruct._COLORLITH_RT );
	  }
	}

    if (plotThinPhi != null)
      plotThinPhi.setLASFileData( stLASFileData );

    if (plotTexture != null)
      plotTexture.setLASFileData( stLASFileData );

//    repaint();
  }

  /** Method setLASTrackSize(int iLASSize)
   * <p> This method will set the LAS File Track Width
   * @param iLASSize = LAS File Track Width
   */

  public void setLASTrackSize(int iLASSize)
  {
    this.iLASSize = iLASSize;
    plotLAS.setLASTrackSize( iLASSize );
    if (iSelected != null) setPlotWidth( iSelected );
  }

  /** Method setRockFileData( rockFileDataStruct stLASFileData )
   * <p> This method will set the Rock File Data Structure
   * @param st - Rock File Data Structure
   */

  public void setRockFileData( rockFileDataStruct st )
  {
    if (plotColorlith != null)
      plotColorlith.setRockFileData( st );
  }

  /** Method setRockData(rockDataListStruct stRock)
   * <p> This method will set the Rock Data List Structure
   * @param stRock - Rock Data List Structure
   */

  public void setRockData(rockDataListStruct stRock)
  {
    if (plotLASRock != null)
      plotLASRock.setRockData(stRock);

//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * --------------------------- Brine Data --------------------------- *
   * ------------------------------------------------------------------ */

  /** Method setBrine()
   *  <p> This method will set the Brine Data.
   *  @param st = The Brine Data Structure
   */

  public void setBrine( brineListStruct st)
  {
    if (plotBrine != null) plotBrine.setBrineData( st );
    if (plotBrineImg != null) plotBrineImg.setBrineData( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ------------------- PfEFFER Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setPfeffer()
   * <p> This method will set the Default PfEFFER Data
   * @param st = The PfEFFER Data Structure
   */

  public void setPfeffer( pfefferDataStruct st )
  {
    if (plotPfeffer != null) plotPfeffer.setData( st );
//    repaint();
  }

  /** Method setPfefferList()
   * <p> This method will set the PfEFFER Data List Structure
   * @param st = The PfEFFER Data List Structure
   */

  public void setPfefferList( pfefferDataListStruct st )
  {
    if (plotPfeffer != null)  plotPfeffer.setData( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ---------------- Thin Porosity Plot Track Actions ---------------- *
   * ------------------------------------------------------------------ */

  /** Method setPHI(int iPHI)
   * <p> This method will set the Porosity to use in creating the track
   * @param iPHI - identifier for porosity track
   */

  public void setPHI(int iPHI)
  {
    plotThinPhi.setPHI(iPHI);
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * -------------- Colorlith Column Plot Track Actions --------------- *
   * ------------------------------------------------------------------ */

  /** Method setColorlith(int iColorlith)
   * <p> This method will set the colorlith that will be shown
   * @param iColorlith = colorlith identifier
   */

  public void setColorlith(int iColorlith)
  {
    plotColorlith.setColorlith(iColorlith);
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * -------------- Rock Color Column Plot Track Actions -------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockColorData()
   * <p> This method will set the Rock Color Data.
   * @param st = the Rock Color List Data Structure
   */

  public void setRockColorData(rockColorListStruct st)
  {
    if (plotRockColor != null) plotRockColor.setRockColorData(st);
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * --------------- OHM & PHI Image Plot Track Actions --------------- *
   * ------------------------------------------------------------------ */

  /** Method setOHM( int iRt, double dRtMax, double dRtMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iRt    = Identifier representing curve to use in plotting
   * @param dRtMax = the maximum conductivity value
   * @param dRtMin = the minimum conductivity value
   */

  public void setOHM( int iRt, double dRtMax, double dRtMin )
  {
    plotColorlith.setOHM( iRt, dRtMax, dRtMin );
    if (pnlImage != null) pnlImage.setOHM( iRt, dRtMax, dRtMin );
//    repaint();
  }

  /** Method setPHI( int iPhit, double dPhitMax, double dPhitMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iPhit    = Identifier representing curve to use in plotting
   * @param dPhitMax = the maximum porosity value
   * @param dPhitMin = the minimum porosity value
   */

  public void setPHI( int iPhit, double dPhitMax, double dPhitMin )
  {
    plotColorlith.setPHI( iPhit, dPhitMax, dPhitMin );
    if (pnlImage != null) pnlImage.setPHI( iPhit, dPhitMax, dPhitMin );
//    repaint();
  }

  /** Method setImageData( double dataRt, double dataPhit, double dataPhi )
   * <p> This method will set the Non-Linear Data for the Non-Linear
   *     Resistivity & Porosity Image Tracks
   * @param dataRt   = The Resistivity Data for the Non-Linear Resistivity Track
   * @param dataPhit = The Porosity Data for the Non-Linear LAS-Porosity Track
   * @param dataPhi  = The Porosity Data for the Non-Linear Rock-Porosity Track
   */

  public void setImageData(
                double dataRt[][], double dataPhit[][], double dataPhi[][] )
  {
    if (plotColorlith != null)
    {
      plotColorlith.setImageData(dataRt, dataPhit, dataPhi);
//      repaint();
    }
  }

  /* ------------------------------------------------------------------ *
   * ------------ Rock Lithology Column Plot Track Actions ------------ *
   * ------------------------------------------------------------------ */

  /** Method setLASRockColumn(lithologyListStruct st)
   * <p> This method will set the las rock column that will be shown
   * @param st = LAS Rock Column determined by user from las file data
   */

  public void setLASRockColumn(lithologyListStruct st)
  {
    plotLithology.setLASRockColumn( st );
//    repaint();
  }

  /** Method setLASRockColumnByRT(lithologyListStruct st)
   * <p> This method will set the las rock column that will be shown
   * @param st = LAS Rock Column determined by user from las file data
   */

  public void setLASRockColumnByRT(lithologyListStruct st)
  {
    plotRTLithology.setLASRockColumn( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ---------------- Rock Column Plot Track Actions ----------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockColumn( rockColumnListStruct st)
   *  <p> This method will set the rock column lithology
   *  @param st = Rock Column List Data Structure
   */

  public void setRockColumn( rockColumnListStruct st )
  {
	pRockColumn.setRockColumn( st );
	pRockLegend.setRockColumn( st );
//	repaint();
  }

  /* ------------------------------------------------------------------ *
   * ------------------- Porosity Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setPHI()
   * <p> This method will set the Porosity Data
   * @param st = The Porosity Data List Structure
   */

  public void setPHI(phiListStruct st)
  {
	setPHILegend( st );
    if (plotPhi != null) plotPhi.setPHIData( st );
//    repaint();
  }

  /** Method setPHILegend()
   *  <p> This method will set the porosity type legend depending on the porosity
   *      types selected.
   */

  public void setPHILegend(phiListStruct st)
  {
    int iPHITypes[] = null;

    if ((st != null) && (stStruct.stPorosity != null))
    {
      if ((stStruct.stPorosity.iCount > 0) && (st.iCount > 0))
      {
        iPHITypes = new int[stStruct.stPorosity.iCount];
        for (int i=0; i<stStruct.stPorosity.iCount; i++)
          iPHITypes[i] = -1;

        iPHITypes = rock.phi.phiUtility.setLegend(
                      iPHITypes, st, stStruct.stPorosity );

        imgPHI = rock.phi.plot.phiPlotLegendBuffered.makePlotImage(
                  iPHITypes, stStruct.stPorosity );
      }
    }
  }

  /* ------------------------------------------------------------------ *
   * ------------------- Fossil Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setFossils()
   * <p> This method will set the Fossil Data
   * @param st = The Fossil Data List Structure
   */

  public void setFossils(fossilListStruct st)
  {
    setFossilLegend( st );
    if (plotFossils != null) plotFossils.setFossilData( st );
//    repaint();
  }

  /** Method setFossilLegend()
   *  <p> This method will set the fossil legend depending on the fossils
   *      selected.
   */

  public void setFossilLegend(fossilListStruct st)
  {
    int    iFossils[] = null;  // Fossil selected indicator

    if ((st != null) && (stStruct.stFossils != null))
    {
      if ((stStruct.stFossils.iCount > 0) && (st.iCount > 0))
      {
        iFossils = new int[stStruct.stFossils.iCount];
        for (int i=0; i<stStruct.stFossils.iCount; i++)
          iFossils[i] = -1;

        iFossils = rock.fossil.fossilUtility.setLegend(
                     iFossils, st, stStruct.stFossils);

        imgFOSSIL = rock.fossil.plot.fossilPlotLegendBuffered.makePlotImage(
                     iFossils, stStruct.stFossils );
      }
    }
  }

  /** Method setBioStrat( bioStratListStruct st )
   * <p> This method will set the Bio Stratigraphy Data
   * @param st = The Bio Stratigraphy Data List Structure
   */

  public void setBioStrat( bioStratListStruct st )
  {
    if (plotBioStrat != null) plotBioStrat.setFossilData( st );
//    repaint();
  }


  /* ------------------------------------------------------------------ *
   * ----------- Sedimentary Structure Plot Track Actions ------------- *
   * ------------------------------------------------------------------ */

  /** Method setStructures()
   * <p> This method will set the sedimentary structures Data
   * @param st = The sedimentary structures Data List Structure
   */

  public void setStructures(sedimentaryListStruct st)
  {
	setSedStructLegend( st );
    if (plotSedimentary != null) plotSedimentary.setData( st );
//    repaint();
  }

  /** Method setSedStructLegend()
   *  <p> This method will set the fossil legend depending on the Sedimentary
   *      Structures selected.
   */

  public void setSedStructLegend(sedimentaryListStruct st)
  {
    int    iSedStructs[] = null;  // Fossil selected indicator

    if ((st != null) && (stStruct.stSedStruct != null))
    {
      if ((stStruct.stSedStruct.iCount > 0) && (st.iCount > 0))
      {
        iSedStructs = new int[stStruct.stSedStruct.iCount];
        for (int i=0; i<stStruct.stSedStruct.iCount; i++)
          iSedStructs[i] = -1;

        iSedStructs = rock.sedimentary.sedimentaryUtility.setLegend(
                        iSedStructs, st, stStruct.stSedStruct);

        imgSED = rock.sedimentary.plot.sedimentaryPlotLegendBuffered.makePlotImage(
                        iSedStructs, stStruct.stSedStruct );
      }
    }
  }

  /* ------------------------------------------------------------------ *
   * ------------------- Remarks Plot Track Actions ------------------- *
   * ------------------------------------------------------------------ */

  /** Method setRemarks()
   * <p> This method will set the Fossil Data
   * @param st = The Remarks Data List Structure
   */

  public void setRemarks(iqstratRemarkListStruct st)
  {
    if (plotRemarks != null) plotRemarks.setRemarksData( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * --------------- Formation Tops Plot Track Actions ---------------- *
   * ------------------------------------------------------------------ */

  /** Method setTops()
   * <p> This method will set the Stratigraphic Units Data
   * @param st = The Stratigraphic Units Data List Structure
   */

  public void setTops(stratListStruct st)
  {
    if (plotStrat != null) plotStrat.setData(st);
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ------------ Sequence Stratigraphy Plot Track Actions ------------- *
   * ------------------------------------------------------------------ */

  /** Method setSequence()
   * <p> This method will set the sequence data list structure.
   * @param st = the Selected sequence data list structure
   */

  public void setSequence(seqListStruct st)
  {
    if (plotSequence != null) plotSequence.setSequenceData( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ---------- Depositional Environment Plot Track Actions ----------- *
   * ------------------------------------------------------------------ */

  /** Method setDepEnvrionment()
   * <p> This method will set the Depositional Environment list structure.
   * @param st = the Selected Depositional Environment data list structure
   */

  public void setDepEnvrionment(envListStruct st)
  {
    if (plotEnviron != null) plotEnviron.setData( st );
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ----------------- Rock Images Plot Track Actions ----------------- *
   * ------------------------------------------------------------------ */

  /** Method setRockImages()
   * <p> This method will set the Rock Image Data
   * @param st = The Rock Image Data List Structure
   */

  public void setRockImages(rockImagesListStruct st)
  {
    plotImages.setData(st);
    pViewImage.setRockImages(st);
//    repaint();
  }

  /** Method setTops( regionsListStruct st )
   * <p> This method will set the Perforations & DST Data
   * @param st = The DST & Perforation Regions Data List Structure
   */

  public void setTops( regionsListStruct st )
  {
    this.stOther = st;
//    repaint();
  }

  /* ------------------------------------------------------------------ *
   * ----------------------- Data Entry Dialogs ----------------------- *
   * ------------------------------------------------------------------ */

  /** Method setThinTrackFrame()
   * <p> This method will set the Thin Track Entry Frame.
   * @param pnl = Thin Track Entry Frame
   *

  public void setThinTrackFrame( lasThinTrackFrame pnl ) { this.pnlThin = pnl; }
*/
  /** Method setImageTrackFrame()
   * <p> This method will set the Log Curve colorlith selector frame
   * @param pnl = the Log Curve colorlith selector frame
   */

  public void setImageTrackFrame( lithImageTrackFrame pnl )
  {
    this.pnlImage = pnl;
  }

  /** Method setConvertFrame()
   * <p> This method will set the Convert Log Curve Frame.
   * @param pnl = the Convert Log Curve Frame
   */

  public void setConvertFrame( lasConvertFrame pnl ) { this.pnlConvert = pnl; }

  /* =============================================================== *
   * --------------------------- COMPUTE --------------------------- *
   * =============================================================== */

  /** Method computeMouseDepth(int iYm)
   * <p> This method will compute the Mouse Depth
   * @param iYm = the cursor y-coordinate
   * @return depth = depth computed from mouse cursor location
   */

  public double computeMouseDepth(int iYm)
  {
    double depth = 0.0;

    if (iYm > PLOT_TITLES)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        depth = depthStart + (depthEnd-depthStart) * (iHeight-iYm)/iLogHeight;
      else
        depth = depthStart + (depthEnd-depthStart) * (iYm-PLOT_TITLES)/iLogHeight;

      if (txtDepth != null)
      {
        depth = (double) (Math.round(depth * 100.0) / 100.0);
        txtDepth.setText(""+depth);
      }
    }

    return (depth);
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  /** Method mouseMoved()
   *  <p> This method will retrieve the location of the mouse pointer and
   *      convert the postion to depth position.
   * @param event - Mouse Moved Event
   */

  public void mouseMoved(MouseEvent event)
  {
    int    iXm  = event.getX();
    int    iYm  = event.getY();
    double depth = 0.0;
    String sKEY  = "0";

    if (iYm > PLOT_TITLES)
    {
      depth = computeMouseDepth(iYm);

      if (txtDepth != null)
      {
        txtDepth.setText(""+depth);
      }
    }

    for (int i=0; i<iButtons; i++)
    {
      switch (iSelected[i])
      {
        case iqstrat.iqstratTracksStruct._LITHOLOGY_GR:
        case iqstrat.iqstratTracksStruct._LAS_REFERENCE:
        case iqstrat.iqstratTracksStruct._LAS_INDUCTION:
        case iqstrat.iqstratTracksStruct._LAS_MICRO_RESISTIVITY:
        case iqstrat.iqstratTracksStruct._LAS_RESISTIVITY:
        case iqstrat.iqstratTracksStruct._LAS_LATERLOG:
        case iqstrat.iqstratTracksStruct._LAS_AHT:
        case iqstrat.iqstratTracksStruct._LAS_LITHO_DENSITY_3:
        case iqstrat.iqstratTracksStruct._LAS_SONIC:
        case iqstrat.iqstratTracksStruct._LAS_LITHO_DENSITY:
        case iqstrat.iqstratTracksStruct._LAS_SPECTRAL_GR:
        case iqstrat.iqstratTracksStruct._LAS_RHOMAA_UMAA:
        case iqstrat.iqstratTracksStruct._LAS_RHOMAA_NPHI:
        case iqstrat.iqstratTracksStruct._LAS_SPECTRAL_RATIO:
        case iqstrat.iqstratTracksStruct._CLRLITH_RU:
        case iqstrat.iqstratTracksStruct._CLRLITH_RN:
        case iqstrat.iqstratTracksStruct._ROCK_COLUMN:
        case iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD:
        case iqstrat.iqstratTracksStruct._CLRLITH_HALL:
        case iqstrat.iqstratTracksStruct._CLRLITH_SPGR:
        case iqstrat.iqstratTracksStruct._LAS_TEXTURE:
        case iqstrat.iqstratTracksStruct._PFEFFER_RESISTIVITY:
        case iqstrat.iqstratTracksStruct._PFEFFER_POROSITY:
        case iqstrat.iqstratTracksStruct._PFEFFER_OTHER:
        case iqstrat.iqstratTracksStruct._PFEFFER_FLOW_UNITS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
            if (normalCursor != null) setCursor(normalCursor);
          break;

        case iqstrat.iqstratTracksStruct._THIN_PHI:
        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER:
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER:
        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N:
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N:
        case iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS:
        case iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
            if (mouseCursor != null) setCursor(mouseCursor);
          break;

        case iqstrat.iqstratTracksStruct._DEP_ENVIRONMENT:
        case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
        case iqstrat.iqstratTracksStruct._STRAT_UNITS_A:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
            if (mouseCursor != null) setCursor(mouseCursor);
          break;

//        case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY_GR:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_CORE_PHI:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_CORE_K:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_PLUG_K:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_SATURATION:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_ARCHIE:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_GR:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_SPGR:
        case iqstrat.iqstratTracksStruct._ROCK_LAS_OUTCROP:
//        case iqstrat.iqstratTracksStruct._ROCK_COLORLITH:
//        case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY_ABREV:
//        case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY_STRK:
//        case iqstrat.iqstratTracksStruct._ROCK_POROSITY_IMAGER:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
            if (normalCursor != null) setCursor(normalCursor);
          break;
//        case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
        case iqstrat.iqstratTracksStruct._ROCK_THIN_PHI:
//        case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
        case iqstrat.iqstratTracksStruct._SED_STRUCTURE:
        case iqstrat.iqstratTracksStruct._ROCK_FOSSILS:
        case iqstrat.iqstratTracksStruct._ROCK_COLOR:
        case iqstrat.iqstratTracksStruct._REMARKS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
            if (mouseCursor != null) setCursor(normalCursor);
          break;

        case iqstrat.iqstratTracksStruct._BRINE_STIFF:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            plotBrineImg.displayDiagram(
		      iXm, iYm, cmn.cmnStruct.STIFF, iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._BRINE_COLLINS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            plotBrineImg.displayDiagram(
			  iXm, iYm, cmn.cmnStruct.COLLINS, iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._BRINE_PIPER:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            plotBrineImg.displayDiagram(
			  iXm, iYm, cmn.cmnStruct.PIPER, iStart[i], iWide[i]);
		  }
          break;

        case iqstrat.iqstratTracksStruct._ROCK_BOX:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            sKEY = plotImages.getKEY(iXm, iYm,
                                     cmn.cmnStruct.CORE_BOX,
                                     iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SLAB:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            sKEY = plotImages.getKEY(iXm, iYm,
                                     cmn.cmnStruct.CORE_SLAB,
                                     iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_THINSECTION:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            sKEY = plotImages.getKEY(iXm, iYm,
                                     cmn.cmnStruct.CORE_THIN,
                                     iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SEM:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (mouseCursor != null) setCursor(mouseCursor);
            sKEY = plotImages.getKEY(iXm, iYm,
                                     cmn.cmnStruct.CORE_SEM,
                                     iStart[i], iWide[i]);
          }
          break;
      }
    }

    if (!sKEY.equals("0"))
      pViewImage.DisplayCoreImage(sKEY);
    else
      pViewImage.cancel();
  }

  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event)  { }

  /** Method mousePressed(MouseEvent)
   *  <p> gets the current coordinates and creates a new rectangle,
   *      updates the rectangle to be drawn and
   *      calls repaint on the parent panel
   *  @param event - MouseEvent info
   */

  public void mousePressed(MouseEvent event)
  {
    iXBegin      = event.getX();
    iXEnd        = 0;
    iYBegin      = event.getY();
    iYEnd        = 0;

    for (int i=0; i<iButtons; i++)
    {
      switch (iSelected[i])
      {
        case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._SEQUENCE_STRAT;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
/*
        case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._ROCK_TEXTURE;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_COLOR:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._ROCK_COLOR;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_THIN_PHI:
        case iqstrat.iqstratTracksStruct._ROCK_POROSITY:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._ROCK_POROSITY;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_FOSSILS:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._ROCK_FOSSILS;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._SED_STRUCTURE:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._SED_STRUCTURE;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
        case iqstrat.iqstratTracksStruct._REMARKS:
          if ((iXBegin > iStart[i]) &&
              (iXBegin < (iStart[i]+iWide[i])))
          {
            iAddBox   = 1;
            iAction   = iqstrat.iqstratTracksStruct._REMARKS;
            iAddStart = iStart[i];
            iAddWidth = iWide[i];
            dStart    = computeMouseDepth(iYBegin);
          }
          break;
*/
      }
    }
  }

  /** Method mouseDragged(MouseEvent)
   *  <p> updates the size of the rectangle as the user drags the mouse
   *  @param event - MouseEvent info
   */

  public void mouseDragged(MouseEvent event)
  {
    switch (iAction)
    {
//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
//      case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
      case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
//      case iqstrat.iqstratTracksStruct._ROCK_COLOR:
//      case iqstrat.iqstratTracksStruct._ROCK_THIN_PHI:
//      case iqstrat.iqstratTracksStruct._ROCK_POROSITY:
//      case iqstrat.iqstratTracksStruct._ROCK_FOSSILS:
//      case iqstrat.iqstratTracksStruct._SED_STRUCTURE:
//      case iqstrat.iqstratTracksStruct._REMARKS:
        iAddBox = 1;
        showRubberBand(event);
        break;

      default:
        iAction = -1;
    }
  }

  /** Method mouseReleased(MouseEvent)
   *  <p> updates the size of the rectangle as the user releases the mouse
   *  @param event - MouseEvent info
   */

  public void mouseReleased(MouseEvent event)
  {
    double dTemp = 0.0;
    String sKEY  = "0";
    fossilStruct        stFossil        = null;
    sedimentaryStruct   stSedimentary   = null;
    iqstratRemarkStruct stRemark        = null;
    lithologyListStruct stRockLithology = null;
    textureListStruct   stRockTexture   = null;
    int iTemp      = 0;
    int iUnitScale = 0;

    iXEnd = event.getX();
    iYEnd = event.getY();

    showRubberBand(event);
    iAddBox = 0;

    dEnd = computeMouseDepth(iYEnd);

    if (dEnd < dStart)
    {
      dTemp  = dStart;
      dStart = dEnd;
      dEnd   = dTemp;
    }

    dStart = Math.round(dStart);
    dEnd   = Math.round(dEnd);

    switch (iAction)
    {
      case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
        repaint();

        iPanel = horizon.gui.horizonEnterDataFrame._SEQ;

        if (notifier != null)
          notifier.notifyObservers(new String("Display Horizon Data Entry"));
        break;

      default:
        break;
    }

    iAction = -1;
  }

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location and decides if the it was clicked in on a Core Button
   *      then it will display a core image in a browser or did it click on
   *      the Well Profile Plot Region then display the Formation Tops &
   *      PfEFFER Region Selection Frame and pass back to the Formation Tops
   *      & PfEFFER Region Selection Frame the Depth of the Mouse Click.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm  = event.getX();
    int    iYm  = event.getY();

    for (int i=0; i<iButtons; i++)
    {
      switch (iSelected[i])
      {
        case iqstrat.iqstratTracksStruct._BRINE_STIFF:
        case iqstrat.iqstratTracksStruct._BRINE_COLLINS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotBrineImg.displaySamplePlot(
		      iXm, iYm, cmn.cmnStruct.STIFF, iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._BRINE_PIPER:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotBrineImg.displayPiperPlot(
		      iXm, iYm, cmn.cmnStruct.PIPER, iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_BOX:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotImages.displayCore(iXm, iYm,
                                   cmn.cmnStruct.CORE_BOX,
                                   iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SLAB:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotImages.displayCore(iXm, iYm,
                                   cmn.cmnStruct.CORE_SLAB,
                                   iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_THINSECTION:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotImages.displayCore(iXm, iYm,
                                   cmn.cmnStruct.CORE_THIN,
                                   iStart[i], iWide[i]);
          }
          break;
        case iqstrat.iqstratTracksStruct._ROCK_SEM:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            plotImages.displayCore(iXm, iYm,
                                   cmn.cmnStruct.CORE_SEM,
                                   iStart[i], iWide[i]);
          }
          break;

        case iqstrat.iqstratTracksStruct._DEP_ENVIRONMENT:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            dStart = computeMouseDepth(iYm);

            iPanel = horizon.gui.horizonEnterDataFrame._ENV;

            if (notifier != null)
              notifier.notifyObservers(new String("Display Horizon Data Entry"));
          }
          break;

        case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            dStart = computeMouseDepth(iYm);

            iPanel = horizon.gui.horizonEnterDataFrame._SEQ;

            if (notifier != null)
              notifier.notifyObservers(new String("Display Horizon Data Entry"));
          }
          break;

        case iqstrat.iqstratTracksStruct._STRAT_UNITS_A:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            dStart = computeMouseDepth(iYm);

            iPanel = horizon.gui.horizonEnterDataFrame._TOPS;

            if (notifier != null)
              notifier.notifyObservers(new String("Display Horizon Data Entry"));
          }
          break;

        case iqstrat.iqstratTracksStruct._THIN_PHI:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
//            if (pnlThin != null) pnlThin.setVisible( true );
            if (notifier != null)
              notifier.notifyObservers(new String("Change Thin Porosity Data"));
          }
          break;

        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER:
        case iqstrat.iqstratTracksStruct._POROSITY_IMAGER_N:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
//            if (notifier != null)
//              notifier.notifyObservers(new String("Display Porosity Imager"));
            if (pnlImage != null)
            {
              pnlImage.setTrack( lith.gui.lithImageTrackFrame._PHIT );
              pnlImage.setVisible(true);
            }
          }
          break;
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER:
        case iqstrat.iqstratTracksStruct._RESISTIVITY_IMAGER_N:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
//            if (notifier != null)
//              notifier.notifyObservers(new String("Display Conductivity Imager"));
            if (pnlImage != null)
            {
              pnlImage.setTrack( lith.gui.lithImageTrackFrame._RT );
              pnlImage.setVisible(true);
            }
          }
          break;
        case iqstrat.iqstratTracksStruct._LAS_SET_GR_COUNTS:
        case iqstrat.iqstratTracksStruct._LAS_SET_NEUT_COUNTS:
          if ((iXm > iStart[i]) && (iXm < (iStart[i]+iWide[i])))
          {
            if (pnlConvert !=  null)
            {
              pnlConvert.setCurve( iSelected[i] );
              pnlConvert.setVisible(true);
            }
          }
          break;
        default:
          iAction = -1;
      }
    }
  }

  /** Method showRubberBand(MouseEvent)
   *  <p> This method is adapted from java.sun.com(MouseMotionEventDemo.java)
   *      updates the size of the current rectangle
   *  @param e - MouseEvent info
   */

  public void showRubberBand(MouseEvent e)
  {
    iXEnd = e.getX();
    iYEnd = e.getY();

    switch (iAction)
    {
//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
//      case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
      case iqstrat.iqstratTracksStruct._SEQUENCE_STRAT:
//      case iqstrat.iqstratTracksStruct._ROCK_COLOR:
//      case iqstrat.iqstratTracksStruct._ROCK_THIN_PHI:
//      case iqstrat.iqstratTracksStruct._ROCK_POROSITY:
//      case iqstrat.iqstratTracksStruct._ROCK_FOSSILS:
//      case iqstrat.iqstratTracksStruct._SED_STRUCTURE:
//      case iqstrat.iqstratTracksStruct._REMARKS:
        this.repaint();
        break;
    }
  }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawRubberBand()
   *  <p> This method will draw a box around the selected area.
   *  @param g - Graphics pointer.
   */

  public void drawRubberBand(Graphics g)
  {
    int iXStartMouse = iXBegin;
    int iYStartMouse = iYBegin;
    int iMouseWidth  = 0;
    int iMouseHeight = 0;

    if (iXEnd < iXBegin)
    {
      iXStartMouse = iXEnd;
      iMouseWidth  = iXBegin - iXEnd;
    }
    else
      iMouseWidth = iXEnd - iXBegin;

    if (iYEnd < iYBegin)
    {
      iYStartMouse = iYEnd;
      iMouseHeight  = iYBegin - iYEnd;
    }
    else
      iMouseHeight = iYEnd - iYBegin;

    if (iAddBox == 1)
    {
      g.setColor(Color.cyan);
      g.drawRect(iXStartMouse, iYStartMouse, iMouseWidth, iMouseHeight);

      g.setColor(Color.gray);
      g.drawRect(iXStartMouse+1, iYStartMouse+1, iMouseWidth, iMouseHeight);
    }
  }

  /** Method drawTitles(Graphics g)
   * <p> This method will draw the Titles for this plot
   * @param g - Graphics Parameter.
   */

  public void drawTitles(Graphics g)
  {
    Font        fsb    = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm   = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawString(sTitle1, 10, 10);
    g.drawString(sTitle2, 10, 24);
  }

  /** Method drawDepth(Graphics g)
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   */

  public void drawDepth(Graphics g)
  {
    int    i=0;
    int    j=0;
    int    iDepth;
    int    jXinc;
    int    iStartTrack = 0;
    int    iWidthTrack = DEPTH_TRACK;
    int    iY1 = 0;
    int    iY2 = 0;

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      iDepth = (int) (depthStart +
                      (double)i * dIncrementDepth / (double) iIncrementY);

      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES)  && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iStartTrack + iWidthTrack, iY2);
        }
      }

      g.setColor(Color.black);
      g.drawLine(iStartTrack, iY1, iStartTrack+iWidthTrack, iY1);

      g.drawString("" + iDepth, iStartDepthLabel, iY1);
    }

    g.drawLine(iStartTrack, PLOT_TITLES, iStartTrack+iWidthTrack, PLOT_TITLES);
  }

  /** Method drawPerfRegions( Graphics g )
   *  <p> This method will draw Circles in the depth column to
   *      show the regions that were Perforated in the well.
   * @param g - Graphics Parameter.
   */

  public void drawPerfRegions( Graphics g )
  {
    int    iY1;
    int    iY2;
    int    iColumns = 0;
    int    iCircles = 0;
    int    iStartTrack = 0;
    int    iWidthTrack = DEPTH_TRACK;
    double dRatio;
    double depth;

    if (stOther != null)
    {
      if (stOther.iCount > 0)
      {
        for (int i=0; i<stOther.iCount; i++)
        {
          if (stOther.stItem[i].id == horizon.regions.regionsStruct._PERFORATION)
          {
            depth = stOther.stItem[i].depth_top;
            dRatio = (double) iLogHeight * (depthStart - depth) /
                                           (depthStart - depthEnd);
            iY1 = PLOT_TITLES + (int) dRatio;

            depth = stOther.stItem[i].depth_base;
            dRatio = (double) iLogHeight * (depthStart - depth) /
                                           (depthStart - depthEnd);
            iY2 = PLOT_TITLES + (int) dRatio;

//            iCircles = (iY2 - iY1) / 6;
            iCircles = (iY2 - iY1) / 9;
            if (iCircles == 0)
              iCircles = 1;

            if ((iY1 >= PLOT_TITLES) && (iY2 > PLOT_TITLES) &&
                (iY1 < iHeight)      && (iY2 < iHeight))
            {
              g.setColor(Color.red);
              for (int j=0; j<iCircles; j++)
              {
                g.drawOval(DEPTH_TRACK-10, iY1+(j*9), 8, 8);
//                g.drawOval(DEPTH_TRACK-9, iY1+(j*9)+1, 7, 7);
//                g.drawOval(DEPTH_TRACK-8, iY1+(j*6), 5, 5);
//                g.drawOval(DEPTH_TRACK-8, iY1+(j*6), 4, 4);

                if (j==0)
                  g.drawString("Perf", iStartTrack+10, iY1+5);
              }
            }
          }
        }
      }
    }
  }

  /** Method drawTrackLabels()
   *  <p> This method will draw the labels on the LAS Plot.
   *   @param g - Graphics Parameter.
   */

  public void drawTrackLabels(Graphics g)
  {
    int length      = 0;
    int iStartTrack = 0;
    int iWidthTrack = DEPTH_TRACK;
    String sTemp    = "";

    g.setColor(Color.black);

    Font fsb = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);
    g.setFont( fsb );

    length = sTemp.length();
    g.drawString(sTemp, iWidth/2-length*4, 14);
    g.drawLine(iStartTrack,
               LABELSTART,
               iWidthTrack, //iWidth,
               LABELSTART);

    g.setColor(Color.black);
    g.drawString("Depth", iStartDepthLabel, LABELSTART + 10);
  }

  /** Method toImage()
   * <p> This method will convert a buffered image into an image
   * @param bImage - Buffered Image.
   * @param Java Image of Buffered Image.
   */

  public static Image toImage(BufferedImage bImage)
  {
    return (Toolkit.getDefaultToolkit().createImage(bImage.getSource()));
  }

  /** Method draw(graphics g)
   * <p> This method will draw the stratigraphic plot
   * @param g - Graphics Parameter.
   */

  public void draw(Graphics g)
  {
    int iStartWidth = DEPTH_TRACK;
    int iTrackWidth = 0;
    int iLength     = 0;
    int iLAS        = 0;
    int iROCK       = 0;
    int iFOSSIL     = 0;
    int iSED        = 0;
    int iPHI        = 0;
    int iPage       = 0;

    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    drawTitles(g);
    drawTrackLabels(g);
    drawDepth(g);
    drawPerfRegions( g );

    for (int i=0; i<iButtons; i++)
    {
      if (iSelected[i] != -1)
      {
        plotColorlith.draw(g, iSelected[i], iStart[i], iWide[i]);
        if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN) ||
            (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD))
        {
          plotLithology.draw(g, iSelected[i], iStart[i], iWide[i]);
	    }
        if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_RT)
        {
          plotRTLithology.draw(g, iSelected[i], iStart[i], iWide[i]);
		}
        plotThinPhi.draw(g, iSelected[i], iStart[i], iWide[i]);

        pRockColumn.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotPfeffer.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotStrat.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotSequence.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotEnviron.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotImages.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotTexture.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotRockColor.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotPhi.draw(g, iSelected[i], iStart[i], iWide[i]); // PHI Type
        plotFossils.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotSedimentary.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotRemarks.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotBrine.draw(g, iSelected[i], iStart[i], iWide[i]);
        plotBrineImg.draw(g, iSelected[i], iStart[i], iWide[i]);

        plotBioStrat.draw(g, iSelected[i], iStart[i], iWide[i]);

        if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN) ||
            (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD ))
        {
          iLAS  = iSelected[i];
	    }

//      	if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION)
       	if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION) ||
       	    (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_TEXTURE))
       	{
		  iROCK = iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION;
		}

        if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
		{
		  iFOSSIL = iSelected[i];
		}

        if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
		{
		  iSED = iSelected[i];
		}

        if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_POROSITY)
		{
		  iPHI = iSelected[i];
		}

        if ( (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals("LAS")) &&
            (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_SOURCE].equals("LAS")))
        {
          plotLAS.draw(g, iSelected[i], iStart[i], iWide[i]);
        }

        if ( (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals("LAS")) &&
            (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_SOURCE].equals("ROCK")))
        {
          plotLASRock.draw(g, iSelected[i], iStart[i], iWide[i]);
        }
      }
    }

    pLegend.draw(g, iLAS , 0, iHeight, iWidth);
    iLength = pRockLegend.draw(g, iROCK, iWidth, jLegend);

    if (imgFOSSIL != null) { iPage = imgFOSSIL.getHeight(null); }
    if ((iFOSSIL == iqstrat.iqstratTracksStruct._ROCK_FOSSILS) && (iLength + iPage < iHeight))
    {
      g.drawImage(imgFOSSIL, iWidth+5, iLength+40, null);
      if (imgFOSSIL != null) { iLength = iLength + imgFOSSIL.getHeight(null); }
    }

    if (imgSED != null) { iPage = imgSED.getHeight(null); }
    if ((iSED == iqstrat.iqstratTracksStruct._SED_STRUCTURE) && (iLength + iPage < iHeight))
	{
      g.drawImage(imgSED, iWidth+5, iLength+20, null);
      if (imgSED != null) { iLength = iLength + imgSED.getHeight(null); }
	}

    if (imgPHI != null) { iPage = imgPHI.getHeight(null); }
    if ((iPHI == iqstrat.iqstratTracksStruct._ROCK_POROSITY) && (iLength + iPage < iHeight))
	{
      g.drawImage(imgPHI, iWidth+5, iLength, null);
	}
  }

  /** Method paint(Graphics g)
   * <p> This method will paint stratigraphic Plot
   * @param g - Graphics Parameter.
   */

  public void paint(Graphics g)
  {
    int iStartWidth = DEPTH_TRACK;
    int iTrackWidth = 0;
    int iLength     = 0;
    int iLAS        = 0;
    int iROCK       = 0;
    int iFOSSIL     = 0;
    int iSED        = 0;
    int iPHI        = 0;
    int iPage       = 0;

    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    drawTitles(g);
    drawTrackLabels(g);
    drawDepth(g);
    drawPerfRegions( g );

    switch (iAction)
    {
//      case iqstrat.iqstratTracksStruct._ROCK_LITHOLOGY:
//      case iqstrat.iqstratTracksStruct._ROCK_TEXTURE:
//      case iqstrat.iqstratTracksStruct._ROCK_THIN_PHI:
//      case iqstrat.iqstratTracksStruct._ROCK_POROSITY:
//      case iqstrat.iqstratTracksStruct._ROCK_FOSSILS:
      case iqstrat.iqstratTracksStruct._SED_STRUCTURE:
//      case iqstrat.iqstratTracksStruct._ROCK_COLOR:
//      case iqstrat.iqstratTracksStruct._REMARKS:
        plotLithology.draw(g, iAction, iAddStart, iAddWidth);
        plotRTLithology.draw(g, iAction, iAddStart, iAddWidth);
        plotTexture.draw(g, iAction, iAddStart, iAddWidth);
        plotRockColor.draw(g, iAction, iAddStart, iAddWidth);
        plotFossils.draw(g, iAction, iAddStart, iAddWidth);
        plotSedimentary.draw(g, iAction, iAddStart, iAddWidth);
        plotRemarks.draw(g, iAction, iAddStart, iAddWidth);
        plotThinPhi.draw(g, iAction, iAddStart, iAddWidth);
        drawRubberBand(this.getGraphics());
        break;

      default:
        for (int i=0; i<iButtons; i++)
        {
          if (iSelected[i] != -1)
          {
            plotColorlith.draw(g, iSelected[i], iStart[i], iWide[i]);
            if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN) ||
                (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD))
            {
              plotLithology.draw(g, iSelected[i], iStart[i], iWide[i]);
  	        }
            if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_RT)
            {
              plotRTLithology.draw(g, iSelected[i], iStart[i], iWide[i]);
		    }
//            plotRU.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotThinPhi.draw(g, iSelected[i], iStart[i], iWide[i]);

            pRockColumn.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotPfeffer.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotStrat.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotSequence.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotEnviron.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotImages.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotTexture.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotRockColor.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotPhi.draw(g, iSelected[i], iStart[i], iWide[i]); // PHI Type
            plotFossils.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotSedimentary.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotRemarks.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotBrine.draw(g, iSelected[i], iStart[i], iWide[i]);
            plotBrineImg.draw(g, iSelected[i], iStart[i], iWide[i]);

            plotBioStrat.draw(g, iSelected[i], iStart[i], iWide[i]);

            if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN) ||
                (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_COLUMN_PD ))
            {
              iLAS  = iSelected[i];
	        }

//          	if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION)
         	if ((iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION) ||
         	    (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_TEXTURE))
          	{
			  iROCK = iqstrat.iqstratTracksStruct._ROCK_MEASURED_SECTION;
			}

            if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_FOSSILS)
		    {
		      iFOSSIL = iSelected[i];
		    }

            if (iSelected[i] == iqstrat.iqstratTracksStruct._SED_STRUCTURE)
		    {
		      iSED = iSelected[i];
		    }

            if (iSelected[i] == iqstrat.iqstratTracksStruct._ROCK_POROSITY)
		    {
		      iPHI = iSelected[i];
		    }

            if ((iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals("LAS")) &&
                (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_SOURCE].equals("LAS")))
            {
              plotLAS.draw(g, iSelected[i], iStart[i], iWide[i]);
            }

            if ((iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_TYPE].equals("LAS")) &&
                (iqstrat.iqstratTracksStruct.TRACK[iSelected[i]][_SOURCE].equals("ROCK")))
            {
              plotLASRock.draw(g, iSelected[i], iStart[i], iWide[i]);
            }
          }
        }
        break;
    }

    pLegend.draw(g, iLAS , 0, iHeight, iWidth);
    iLength = pRockLegend.draw(g, iROCK, iWidth, jLegend);

    if (imgFOSSIL != null) { iPage = imgFOSSIL.getHeight(null); }
    if ((iFOSSIL == iqstrat.iqstratTracksStruct._ROCK_FOSSILS) && (iLength + iPage < iHeight))
    {
      g.drawImage(imgFOSSIL, iWidth+5, iLength+40, null);
      if (imgFOSSIL != null) { iLength = iLength + imgFOSSIL.getHeight(null); }
    }

    if (imgSED != null) { iPage = imgSED.getHeight(null); }
    if ((iSED == iqstrat.iqstratTracksStruct._SED_STRUCTURE) && (iLength + iPage < iHeight))
	{
      g.drawImage(imgSED, iWidth+5, iLength+20, null);
      if (imgSED != null) { iLength = iLength + imgSED.getHeight(null); }
	}

    if (imgPHI != null) { iPage = imgPHI.getHeight(null); }
    if ((iPHI == iqstrat.iqstratTracksStruct._ROCK_POROSITY) && (iLength + iPage < iHeight))
	{
      g.drawImage(imgPHI, iWidth+5, iLength+20, null);
	}
  }
}

/*
 *  @version 1.1 09/04/2007
 *  @author  John Victorine
 */
