/*
 * @iqstratShaleUtility.java Version 1.1 10/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratShaleListStruct;
import iqstrat.iqstratShaleStruct;

/** Class iqstratShaleUtility
 *  <p> This Class will provide basic utility for the data structures
 *
 *  @version 1.1 10/05/2007
 *  @author  John R. Victorine
 */

public class iqstratShaleUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Rhomaa-Umaa data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rhomaa-Umaa List Data Structure
   * @return st       = The Rhomaa-Umaa List Data Structure
   */

  public static iqstratShaleListStruct addHeaderData(iqstratHeadersStruct stHeader,
                                                     iqstratShaleListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method initialize()
   *  <p> This method will initialize the Location Data Structure
   *   @return st = The New Shale Data Structure
   */

  public static iqstratShaleStruct initialize()
  {
    iqstratShaleStruct st = new iqstratShaleStruct();

    if (st != null)
    {
      st.sKEY       = "0";   // Key of Depth Record

      st.depthStart = 0.0;   // Starting Depth
      st.depthEnd   = 0.0;   // Ending Depth

      st.dGammaMin  =   0.0;  // Minimum Gamma Ray Value
      st.dShaly     =  60.0;  // Start of Shaly Formation
      st.dShale     =  70.0;  // Start of Shale Formation
      st.dHotShale  = 100.0;  // Start of Hot Shale Formation
      st.dGammaMax  = 150.0;  // Maximum  Gamma Ray Value
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one Location Info structure to another
   * @param  stOld = The Shale Data Structure
   * @return stNew = The Shale Data Structure
   */

  public static iqstratShaleStruct copy(iqstratShaleStruct stOld)
  {
    iqstratShaleStruct stNew = new iqstratShaleStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY);

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.dGammaMin  = stOld.dGammaMin;  // Minimum Gamma Ray Value
      stNew.dShaly     = stOld.dShaly;     // Start of Shaly Formation
      stNew.dShale     = stOld.dShale;     // Start of Shale Formation
      stNew.dHotShale  = stOld.dHotShale;  // Start of Hot Shale Formation
      stNew.dGammaMax  = stOld.dGammaMax;  // Maximum  Gamma Ray Value
    }

    return (stNew);
  }

  /** Method exists()
   * <p> This method will check to see if the record exists in the list
   * @param  st   =  Shale XML Data List Structure
   * @param  sKEY = Primary Key of data defined by user
   * @return sKEY = The user created primary key
   */

  public static String exists(iqstratShaleListStruct st,  String sKEY)
  {
    int iFound = -1;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(sKEY))  { iFound = i; }
      }
    }

    sKEY = new String("0");
    if (iFound > -1)
      sKEY = new String(st.stItem[iFound].sKEY);

    return (sKEY);
  }

  /** Method copyList( lithologyListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratShaleListStruct copyList( iqstratShaleListStruct stOld )
  {
    iqstratShaleListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new iqstratShaleListStruct();
      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new iqstratShaleStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratShaleListStruct transfer(iqstratShaleListStruct stOld)
  {
    iqstratShaleListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method add()
   * <p> This method will remove a Shale from the Table
   * @param  st    =  Shale XML Data List Structure
   * @param  stNew =  Shale to be added to list
   * @return st    =  Shale XML Data List Structure
   */

  public static iqstratShaleListStruct add( iqstratShaleListStruct st,
                                            iqstratShaleStruct stNew)
  {
    int i=0;
    int iCount=0;
    int iFound=0;

    iqstratShaleListStruct stTemp = new iqstratShaleListStruct();
    stTemp.stItem = new iqstratShaleStruct[1];
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratShaleStruct[st.iCount+1];
      for (i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sKEY.equals(stNew.sKEY))  { iFound = 1; }

        stTemp.stItem[iCount] = new iqstratShaleStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    stTemp.stItem[iCount] = new iqstratShaleStruct();
    stTemp.stItem[iCount] = copy(stNew);
    iCount++;

    st        = new iqstratShaleListStruct();
    st.stItem = new iqstratShaleStruct[iCount];
    st.iCount = iCount;

    for (i=0; i<iCount; i++)
    {
      st.stItem[i] = new iqstratShaleStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    if (stTemp != null)
      stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Shale data in the Table
   * @param  st    =  Shale XML Data List Structure
   * @param  stNew =  Shale to be modified in the list
   * @param  sKEY  = Primary Key of data defined by user
   * @return st    =  Shale XML Data List Structure
   */

  public static iqstratShaleListStruct modify( iqstratShaleListStruct st,
                                               iqstratShaleStruct stNew,
                                               String sKEY)
  {
    int i=0;
    int iCount=0;

    iqstratShaleListStruct stTemp = new iqstratShaleListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratShaleStruct[st.iCount];
      for (i=0; i<st.iCount; i++)
      {
        if (sKEY.equals(st.stItem[i].sKEY))
        {
          stTemp.stItem[iCount] = new iqstratShaleStruct();
          stTemp.stItem[iCount] = copy(stNew);
        }
        else
        {
          stTemp.stItem[iCount] = new iqstratShaleStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
        }
        iCount++;
      }

      st.delete();
      st = null;

      st        = new iqstratShaleListStruct();
      st.stItem = new iqstratShaleStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratShaleStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }
    }

    if (stTemp != null)
      stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Shale from the Table
   * @param  st   =  Shale XML Data List Structure
   * @param  sKEY = Primary Key of data defined by user
   * @return st   =  Shale XML Data List Structure
   */

  public static iqstratShaleListStruct remove( iqstratShaleListStruct st,
                                               String sKEY)
  {
    int i=0;
    int iCount = 0;
    int iFound = 0;

    // If the Shale is to be removed from the Table Selection only
    //   GET the API Number from the highlighted row.

    iqstratShaleListStruct stTemp = new iqstratShaleListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratShaleStruct[st.iCount-1];
      for (i=0; i<st.iCount; i++)
      {
        iFound = 0;
        if (sKEY.equals(st.stItem[i].sKEY))
        {
          iFound = 1;
        }

        if (iFound != 1)
        {
          stTemp.stItem[iCount] = new iqstratShaleStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new iqstratShaleListStruct();
      st.stItem = new iqstratShaleStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratShaleStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }
    }

    if (stTemp != null)
      stTemp.delete();
    stTemp = null;

    return (st);
  }

  public static void print(iqstratShaleListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].sKEY+" "+
                           st.stItem[i].depthStart+" "+
                           st.stItem[i].depthEnd+" "+
                           st.stItem[i].dGammaMin+" "+
                           st.stItem[i].dShaly+" "+
                           st.stItem[i].dShale+" "+
                           st.stItem[i].dHotShale+" "+
                           st.stItem[i].dGammaMax);
      }
    }
  }
}