/*
 * @iqstratRemarkUtility.java Version 1.1 09/25/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;
import kgs.kgsMeasSectListStruct;

/** Class iqstratRemarkUtility
 *  <p> This Class will provide basic utilities for the Remarks
 *      data structures.
 *
 *  @version 1.1 09/25/2007
 *  @author  John R. Victorine
 */

public class iqstratRemarkUtility
{
  /** Method addHeaderData()
   * <p> This method will add the header data to the Rhomaa-Umaa data List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Rhomaa-Umaa List Data Structure
   * @return st       = The Rhomaa-Umaa List Data Structure
   */

  public static iqstratRemarkListStruct addHeaderData(
      iqstratHeadersStruct stHeader, iqstratRemarkListStruct st)
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method add()
   * <p> This method will add the Remarks to an existing list
   * @param  stBlock = The Remarks Block List Data Structure
   * @param  st      = The Old Remarks List Data Structure
   * @return st      = The new Remarks List.
   */

  public static iqstratRemarkListStruct add( iqstratRemarkStruct stBlock,
                                             iqstratRemarkListStruct st )
  {
    int    i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    iqstratRemarkListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    double depthStart = 0.0;   // Starting Depth
    double depthEnd   = 0.0;   // Ending Depth
    String source     = "";    // Source of Data
    String sRef       = "";    // Depth Reference

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new iqstratRemarkListStruct();
    stTemp.stItem = new iqstratRemarkStruct[iRecords];

    if (st != null)
    {
	  iSource    = st.iSource;

      sKGS       = new String(st.sKGS);   // KGS Saved Data Indicator
      depthStart = st.depthStart;         // Starting Depth
      depthEnd   = st.depthEnd;           // Ending Depth
      source     = new String(st.source); // Source of Data
      sRef       = new String(st.sRef);   // Depth Reference

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new iqstratRemarkStruct();
    stTemp.stItem[iCount] = copy(stBlock);
    iCount++;

    stTemp.iCount = iCount;

    st        = new iqstratRemarkListStruct();
    st.stItem = new iqstratRemarkStruct[iCount];
    st.iCount = iCount;

    st.iSource    = iSource;

    st.sKGS       = new String(sKGS);   // KGS Saved Data Indicator
    st.depthStart = depthStart;         // Starting Depth
    st.depthEnd   = depthEnd;           // Ending Depth
    st.source     = new String(source); // Source of Data
    st.sRef       = new String(sRef);   // Depth Reference

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Remarks in an existing list
   * @param  sKEY    = The Primary Key of the data to be modified
   * @param  stBlock = The Block of Remarks that will be modified
   * @param  st      = The Old Remarks List Data Structure
   * @return st      = The new Remarks List.
   */

  public static iqstratRemarkListStruct modify( String sKEY,
                                                iqstratRemarkStruct stBlock,
                                                iqstratRemarkListStruct st )
  {
    int i=0;
    int    iSource    = -1;
    iqstratRemarkListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    double depthStart = 0.0;   // Starting Depth
    double depthEnd   = 0.0;   // Ending Depth
    String source     = "";    // Source of Data
    String sRef       = "";    // Depth Reference

    if (st != null)
    {
	  iSource    = st.iSource;

      sKGS       = new String(st.sKGS);   // KGS Saved Data Indicator
      depthStart = st.depthStart;         // Starting Depth
      depthEnd   = st.depthEnd;           // Ending Depth
      source     = new String(st.source); // Source of Data
      sRef       = new String(st.sRef);   // Depth Reference

      stTemp        = new iqstratRemarkListStruct();
      stTemp.stItem = new iqstratRemarkStruct[st.iCount];
      stTemp.iCount = st.iCount;

      for (i=0; i<st.iCount; i++)
      {
        if (sKEY.equals(st.stItem[i].sKEY))
          stTemp.stItem[i] = copy(stBlock);
        else
          stTemp.stItem[i] = copy(st.stItem[i]);
      }

      st.delete();

      st        = new iqstratRemarkListStruct();
      st.stItem = new iqstratRemarkStruct[stTemp.iCount];
      st.iCount = stTemp.iCount;

      st.iSource    = iSource;

      st.sKGS       = new String(sKGS);   // KGS Saved Data Indicator
      st.depthStart = depthStart;         // Starting Depth
      st.depthEnd   = depthEnd;           // Ending Depth
      st.source     = new String(source); // Source of Data
      st.sRef       = new String(sRef);   // Depth Reference

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Remarks from an existing list
   * @param  sKEY   = The Primary Key of the data to be removed
   * @param  st     = The Old Remarks List Data Structure
   * @return st     = The new Remarks List.
   */

  public static iqstratRemarkListStruct remove(
                  String sKEY, iqstratRemarkListStruct st)
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    int    iSource  = -1;
    iqstratRemarkListStruct stTemp = null;
    String sKGS       = "YES"; // KGS Saved Data Indicator
    double depthStart = 0.0;   // Starting Depth
    double depthEnd   = 0.0;   // Ending Depth
    String source     = "";    // Source of Data
    String sRef       = "";    // Depth Reference

    if (st.iCount == 1)
    {
      st.delete();
      st = null;
    }
    else
    {
      if (st != null)
      {
		iSource    = st.iSource;

        sKGS       = new String(st.sKGS);   // KGS Saved Data Indicator
        depthStart = st.depthStart;         // Starting Depth
        depthEnd   = st.depthEnd;           // Ending Depth
        source     = new String(st.source); // Source of Data
        sRef       = new String(st.sRef);   // Depth Reference

        iRecords = st.iCount-1;

        stTemp        = new iqstratRemarkListStruct();
        stTemp.stItem = new iqstratRemarkStruct[iRecords];

        for (i=0; i<st.iCount; i++)
        {
          if (sKEY.equals(st.stItem[i].sKEY))
          {
            iFound = i;
          }
        }
      }

      if (st != null)
      {
        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();
      }

      stTemp.iCount = iCount;

      st        = new iqstratRemarkListStruct();
      st.stItem = new iqstratRemarkStruct[iCount];
      st.iCount = iCount;

      st.iSource    = iSource;

      st.sKGS       = new String(sKGS);   // KGS Saved Data Indicator
      st.depthStart = depthStart;         // Starting Depth
      st.depthEnd   = depthEnd;           // Ending Depth
      st.source     = new String(source); // Source of Data
      st.sRef       = new String(sRef);   // Depth Reference

      for (i=0; i<stTemp.iCount; i++)
      {
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratRemarkListStruct copyList(iqstratRemarkListStruct stOld)
  {
    iqstratRemarkListStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new iqstratRemarkListStruct();
      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new iqstratRemarkStruct[stOld.iCount];

      stNew.iSource    = stOld.iSource;

      // LAS 3 Control Parameter

      stNew.sKGS       = new String(stOld.sKGS);   // KGS Saved Data Indicator
      stNew.depthStart = stOld.depthStart;         // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;           // Ending Depth
      stNew.source     = new String(stOld.source); // Source of Data
      stNew.sRef       = new String(stOld.sRef);   // Depth Reference

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy(stOld.stItem[i]);
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratRemarkListStruct transfer(iqstratRemarkListStruct stOld)
  {
    iqstratRemarkListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy()
   * <p> This method will copy data from one structure to another
   * @param  stOld = The original Fossil Data Structure
   * @return stNew = The Fossil Data Structure
   */

  public static iqstratRemarkStruct copy(iqstratRemarkStruct stOld)
  {
    iqstratRemarkStruct stNew = new iqstratRemarkStruct();

    if (stOld != null)
    {
      stNew.sKEY       = new String(stOld.sKEY);   // Primary Key for data set
      stNew.depthStart = stOld.depthStart;         // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;           // Ending Depth
      stNew.dThickness = stOld.dThickness;         // Thickness
      stNew.sNote      = new String(stOld.sNote);  // Brief Note
      stNew.sText      = new String(stOld.sText);  // Remark
    }

    return (stNew);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into remarks list
   *     data structure
   * @param  stList     = KGS Measured Section List Data Structure
   * @return stRemarks  = Remarks Data List Structure
   */

  public static iqstratRemarkListStruct parse( kgsMeasSectListStruct stList )
  {
    int i=0;
    iqstratRemarkListStruct  stRemarks  = null;
    int                      iStart     = 0;
    double                   depth      = 0.0;
    double                   depth0     = 0.0;
    String                   tokens[]   = null;
    String                   sDelimiter = new String("[,-; ]+");
    iqstratRemarkStruct      stRemark   = null;

    if (stList != null)
    {
      for (i=0; i<stList.iRows; i++)
      {
        if (stList.stData[i].sUnitNo.length() > 0)
        {
          depth = depth + stList.stData[i].dThickness;

          if (stList.stData[i].sUnitNo.length() > 0)
          {
            if (stList.stData[i].dThickness > 0.0)
            {
              stRemark            = new iqstratRemarkStruct();
              stRemark.sKEY       = new String(cmn.cmnString.UniqueName() +
                                               ((int) depth0));

              stRemark.depthStart = (double) ( (int) (depth0*1000.0)) / 1000.0;
              stRemark.depthEnd   = (double) ( (int) (depth*1000.0)) / 1000.0;
              stRemark.dThickness =
                 Math.abs(stRemark.depthEnd - stRemark.depthStart);
              stRemark.sText = new String("(" + stList.stData[i].sUnitNo +
                                          ") " + stList.stData[i].sDescription +
                                          " ");

              if (stRemark.sText.length() < 41)
                stRemark.sNote = new String(stRemark.sText);
              else
                stRemark.sNote = new String(stRemark.sText.substring(0, 40));

              if (stRemark != null)
              {
                stRemarks = add(stRemark, stRemarks);
                if (iStart == 0)
                {
                  stRemarks.depthStart = stRemark.depthStart; // Starting Depth
                  stRemarks.depthEnd   = stRemark.depthEnd;   // Ending Depth
                  stRemarks.source =
                      new String("Kansas Geological Survey Database"); // Source
                  stRemarks.sRef   = new String("Log Depth"); // Depth Reference

                  iStart = 1;
                }

                if (stRemarks.depthEnd < stRemark.depthEnd)
                  stRemarks.depthEnd = stRemark.depthEnd;   // Ending Depth
              }
            }

            depth0 = depth;
          }
        }
      }

      stList.delete();
      stList = null;
    }

    if (stRemarks != null)
    {
	  stRemarks.iSource = las3.las3Constants._SOURCE_KGS;

      for (i=0; i<stRemarks.iCount; i++)
      {
        stRemarks.stItem[i].sText =
            new String( stRemarks.stItem[i].sText.replace('\t', ' ') );
        stRemarks.stItem[i].sText =
            new String( stRemarks.stItem[i].sText.replace('\n', ' ') );
        stRemarks.stItem[i].sText =
            new String(las3.las3Parse.removeQuotes( stRemarks.stItem[i].sText ));
      }
    }

    return (stRemarks);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = data list structure
   * @return st = sorted data list structure
   */

  public static iqstratRemarkListStruct bubbleSort(iqstratRemarkListStruct st)
  {
    boolean swappedOnPrevRun   = true;
    iqstratRemarkStruct stTemp = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method print()
   * <p> This method will print the contents of the Remarks List
   * @param  st = data list structure
   */

  public static void print(iqstratRemarkListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(st.stItem[i].sKEY+" "+
                           st.stItem[i].depthStart+" "+st.stItem[i].depthEnd);
        System.out.println("  "+st.stItem[i].sNote);
        System.out.println(" ----------------------------------------------- ");
      }
    }
  }
}