/*
 * @iqstratPlotLoad.java Version 1.1 03/02/2010
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat;

import kgs.gui.kgsSearchKGSFrame;

import iqstrat.gui.iqstratNotesCSVFrame;
import iqstrat.gui.iqstratShaleFrame;
import iqstrat.iqstratStruct;
import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratShaleListStruct;

import las3.io.las3Read;
import las3.gui.las3FileFrame;
import las3.las3ListStruct;
import las.lasFileDataStruct;
import las.gui.lasConvertFrame;

import horizon.gui.regionsMapCSVFrame;
import horizon.gui.horizonEnterDataFrame;
import horizon.regions.gui.regionsPerfFrame;
import horizon.strat.stratListStruct;
import horizon.strat.stratProjectStruct;
import horizon.seq.seqListStruct;
import horizon.regions.regionsListStruct;

import horizon.bio.bioStratListStruct;
import horizon.bio.bioStratStruct;
import horizon.bio.gui.bioDataEntryFrame;

import horizon.env.envListStruct;           // Depositional Environment List Structure
//import horizon.env.gui.envMapCSVFrame;

import brine.brineListStruct;               // Brine Data List Structure
import brine.brineStruct;                   // Brine Data Structure
import brine.gui.brineMapCSVFrame;

import rock.gui.rockMapCSVFrame;
import rock.gui.rockObsEntryFrame;
import rock.rockDataListStruct;
import rock.rockImagesListStruct;
import rock.fossil.fossilListStruct;
import rock.sedimentary.sedimentaryListStruct;
import rock.phi.phiListStruct;
import rock.color.rockColorListStruct;

import lith.lithology.lithologyListStruct;
import lith.texture.textureListStruct;
import lith.rock.rockColumnListStruct;

import kgs.kgsMeasSectListStruct;
import kgs.gui.kgsBrineFrame;               // Brine Data KGS Frame

/** Class iqstratPlotLoad
 *  <p> This Class will load the data from input frames
 *
 *  @version 1.1 03/02/2010
 *  @author  John R. Victorine
 */

public class iqstratPlotLoad
{
  // Location of the Data

  public static final int _PC           = 0;
  public static final int _KGS          = 1;
  public static final int _NOTES        = 2;
  public static final int _STRAT_UNIT   = 3;
  public static final int _LAS_3_FILE   = 4;
  public static final int _ROCK_ENTRY   = 5;
  public static final int _TOPS_ENTRY   = 6;
  public static final int _CONVERT      = 7;

  public static final int _MEAS_SECT    = 8;
  public static final int _GEO_REPORT   = 9;
  public static final int _GR_SHALE     = 10;
  public static final int _XSECTION     = 11;
  public static final int _ROCK_DATA    = 12;
  public static final int _PERF         = 13;
  public static final int _BRINE_DATA   = 14;
  public static final int _DEP_ENV_DATA = 15;
  public static final int _BIO_ENTRY    = 16;
  public static final int _ROCK_COLUMN  = 17;

  /** Method getHeader()
   * <p> This method will retrieve the Header Data Structure
   * @param  iLocation  = Location identifier
   * @param  pc         = Search User's PC for Data
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = the Header Data Structure
   */

  public static iqstratHeadersStruct getHeader(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      kgsBrineFrame        pBrine,     // KGS Brine Data Frame
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    iqstratHeadersStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        st = pKGS.getWellHeader();
        break;
      case _NOTES:
        st = pNotesFile.getHeader();
        break;
      case _LAS_3_FILE:
        st = p3.getWellData();
        break;
      case _BRINE_DATA:
        st = pBrine.getHeader();
        break;
    }

    return (st);
  }

  /** Method getLAS3()
   * <p> This method will retrieve the LAS 3 File Data Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = LAS 3 File Data Structure
   */

  public static las3ListStruct getLAS3(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    las3ListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        st = pKGS.getLAS3Data();
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getLAS3Data();
        break;
    }

    return (st);
  }

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = LAS File Data Structure
   */

  public static lasFileDataStruct getLAS(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    lasFileDataStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        st = las.lasFileDataUtility.copy( pKGS.getLASData() );
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_KGS; }
        pKGS.closeLASFile();
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getLASData();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_PC; }
        break;
    }

    return (st);
  }

  /** Method getLAS()
   * <p> This method will retrieve the LAS File Data Structure
   * @param  iLocation  = Location identifier
   * @param  pConvert   = Convert Old Neutron Gamma Ray Counts to Log Data
   * @return st         = LAS File Data Structure
   */

  public static lasFileDataStruct getLAS(
      int                  iLocation,  // Location identifier
      lasConvertFrame      pConvert )  // Convert Counts to Log Data
  {
    lasFileDataStruct st = null;

    switch (iLocation)
    {
      case _CONVERT:
        st = pConvert.getLAS();
        pConvert.setVisible(false);
        break;
    }

    return (st);
  }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Units Data List Structure
   * @param  pTopsEntry = Tops Entry Data Frame
   * @return st         = the Stratigraphic Units Data List Structure
   */

  public static stratListStruct getTops( horizonEnterDataFrame pTopsEntry )
  {
    stratListStruct st = null;

    if (pTopsEntry != null)
      st = pTopsEntry.getTopsData();

    return ( st );
  }

  /** Method getTops()
   * <p> This method will retrieve the Stratigraphic Units Data List Structure
   * @param  iLocation   = Location identifier
   * @param  stStruct    = IQSTRAT Global Data Structure
   * @param  pMapRegions = Search User's PC for Data Tops CSV
   * @param  pKGS        = Search KGS Database for Data
   * @param  pNotesFile  = Read Remarks CSV File
   * @param  p3          = Read LAS 3 File for Data
   * @param  stKGSMeas   = KGS Measured Section Data List Structure
   * @return st          = the Stratigraphic Units Data List Structure
   */

  public static stratListStruct getTops(
      int                   iLocation,   // Location identifier
      iqstratStruct         stStruct,    // IQSTRAT Global Data Structure
      regionsMapCSVFrame    pMapRegions, // PC-Map Tops Comma Delimited File
      kgsSearchKGSFrame     pKGS ,       // Search KGS Database for Data
      iqstratNotesCSVFrame  pNotesFile,  // Read Remarks CSV File
      las3Read              p3,          // Read LAS 3 File for Data
      kgsMeasSectListStruct stKGSMeas )  // KGS Measured Section List Structure
  {
    stratListStruct st = null;

    switch (iLocation)
    {
      case _PC:
        st = horizon.strat.stratUtility.copyList( pMapRegions.getTops() );
        pMapRegions.close();
        pMapRegions = null;

        st = horizon.strat.stratUtility.addICSArray( st, stStruct.stICS );
        st = horizon.strat.stratUtility.addAges( st, stStruct.stICS );
        if (st != null)
        {
          st.sTOPS   = new String("Comma Delimited File"); // Formation Source
          st.sTOPDR  = new String("Log Depth"); // Tops Depth Reference
          st.iSource = las3.las3Constants._SOURCE_CSV;
	    }
        break;
      case _KGS:
        st = horizon.strat.stratUtility.copyList( pKGS.getTops() );
        if (st != null)
        {
          st.sTOPS  = new String("Kansas Geological Survey Database"); // Formation Source
          st.sTOPDR = new String("Log Depth");    // Tops Depth Reference
          st.iSource = las3.las3Constants._SOURCE_KGS;
	    }
        pKGS.closeTops();
        break;
      case _NOTES:
        st = pNotesFile.getTops();
        break;
      case _LAS_3_FILE:
        st = p3.getTops();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
      case _MEAS_SECT:
        st = horizon.strat.stratUtility.parse(
                     kgs.kgsMeasSectUtility.copyList( stKGSMeas ),
                     stStruct.stKGS );
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_KGS; }
        break;
    }

    return (st);
  }

  /** Method getSeqStrat()
   * <p> This method will retrieve the Sequence Stratigraphy Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pc         = Search User's PC for Data
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  pTopsEntry = Tops Entry Frame
   * @return st         = Sequence Stratigraphy Data List Structure
   */

  public static seqListStruct getSeqStrat(
      int                   iLocation,  // Location identifier
      kgsSearchKGSFrame     pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame  pNotesFile, // Read Remarks CSV File
      las3Read              p3 )        // Read LAS 3 File for Data
  {
    seqListStruct st  = null;

    switch (iLocation)
    {
      case _PC:
        break;
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getSeq();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
    }

    return (st);
  }

  /** Method getSeqStrat()
   * <p> This method will retrieve the Sequence Stratigraphy Data List Structure
   * @param  pTopsEntry = Tops Entry Frame
   * @return st         = Sequence Stratigraphy Data List Structure
   */

  public static seqListStruct getSeqStrat(
      horizonEnterDataFrame pTopsEntry ) // Tops Entry Data Frame
  {
    seqListStruct st  = null;

    if (pTopsEntry != null)
    {
      st = pTopsEntry.getSeqList();
      if (st != null) { st.iSource = las3.las3Constants._SOURCE_APP; }
    }

    return (st);
  }

  /** Method getPerf()
   * <p> This method will retrieve the Perforations Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = Perforations Data List Structure
   */

  public static regionsListStruct getPerf(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    regionsListStruct st  = null;

    switch (iLocation)
    {
      case _PC:
        break;
      case _KGS:
        st = pKGS.getPerf();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_KGS; }
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
         st = p3.getPerf();
         if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
    }

    return (st);
  }

  /** Method getPerf()
   * <p> This method will retrieve the Perforations Data List Structure
   * @param  perf  = Perforations Data Entry Dialog
   * @return st    = Perforations Data List Structure
   */

  public static regionsListStruct getPerf( regionsPerfFrame  perf )
  {
    regionsListStruct st  = null;

    if (perf != null)
    {
      st = perf.getData();
      if (st != null) { st.iSource = las3.las3Constants._SOURCE_APP; }
    }

    return (st);
  }

  /** Method getDepEnv()
   * <p> This method will retrieve the Brine Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Depositional Environment List Structure
   */

  public static envListStruct getDepEnv(
      int             iLocation,  // Location identifier
      iqstratStruct   stStruct,   // IQSTRAT Global Data Structure
//      envMapCSVFrame  pc,         // Brine Map CSV Frame
      las3Read        p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    envListStruct st = null;

    switch (iLocation)
    {
      case _PC:
//        st = pc.getData();
//        pc.close();
//        pc = null;
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getEnv();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
      case _GEO_REPORT:
        st = parse.parseDepEnvUtility.parse(
                    iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
                    stStruct );
//        st = parse.parseDepEnvUtility.parse(
//                    iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
//                    stStruct.stParseDepEnv );

        break;
    }

    return (st);
  }

  /** Method getDepEnv()
   * <p> This method will retrieve the
   *     Depositional Environment Data List Structure
   * @param  pTopsEntry = Tops Entry Frame
   * @return st         = Depositional Environment List Structure
   */

  public static envListStruct getDepEnv(
      horizonEnterDataFrame pTopsEntry ) // Tops Entry Data Frame
  {
    envListStruct st  = null;

    if (pTopsEntry != null)
    {
      st = pTopsEntry.getEnvList();
      if (st != null) { st.iSource = las3.las3Constants._SOURCE_APP; }
    }

    return (st);
  }

  /** Method getShale()
   * <p> This method will retrieve the Shale Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = Shale Data List Structure
   */

  public static iqstratShaleListStruct getShale(
      int                   iLocation,  // Location identifier
      kgsSearchKGSFrame     pKGS,       // Search KGS Database for Data
      las3Read              p3 )        // Read LAS 3 File for Data
  {
    iqstratShaleListStruct st  = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _LAS_3_FILE:
        st = p3.getShale();
        break;
    }

    return (st);
  }

  /** Method getShale()
   * <p> This method will retrieve the Shale Data List Structure
   * @param  pShale     = Shale Entry Dialog
   * @return st         = Shale Data List Structure
   */

  public static iqstratShaleListStruct getShale(
      iqstratShaleFrame     pShale )    // Shale Entry Dialog
  {
    iqstratShaleListStruct st  = null;

    if (pShale != null)
    {
       st = iqstrat.iqstratShaleUtility.copyList( pShale.getData() );
       pShale.close();
       pShale = null;
    }

    return (st);
  }

  /** Method getImages()
   * <p> This method will retrieve the Rock Images Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = Rock Images Data List Structure
   */

  public static rockImagesListStruct getImages(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    rockImagesListStruct st  = null;

    switch (iLocation)
    {
      case _KGS:
        if (st != null)
          st.delete();
        st = null;

        st = rock.rockImagesUtility.copyList(pKGS.getRockImages());
        pKGS.closeCoreImages();
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = rock.rockImagesUtility.copyList(p3.getImages());
        break;
    }

    return (st);
  }

  /** Method getBrine()
   * <p> This method will retrieve the Brine Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = Brine Data List Structure
   */

  public static brineListStruct getBrine(
      int                   iLocation,  // Location identifier
      brineMapCSVFrame      pc,         // Brine Map CSV Frame
      kgsBrineFrame         pKGS ,      // Search KGS Database for Data
      las3Read              p3 )        // Read LAS 3 File for Data
  {
    brineListStruct st  = null;

    switch (iLocation)
    {
      case _PC:
        st = pc.getData();
        pc.close();
        pc = null;
        break;
      case _KGS:
        st = pKGS.getData();
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getBrine();
        break;
    }

    return (st);
  }

  /** Method getRock()
   * <p> This method will retrieve the Rock Data List Structure
   * @param  iLocation  = Location identifier
   * @param  pc         = Search User's PC for Rock CSV
   * @param  pKGS       = Search KGS Database for Data
   * @param  p3         = Read LAS 3 File for Data
   * @return st         = Rock Data List Structure
   */

  public static rockDataListStruct getRock(
      int                  iLocation,  // Location identifier
      rockMapCSVFrame      pc,         // Rock CSV File
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      las3Read             p3 )        // Read LAS 3 File for Data
  {
    rockDataListStruct st  = null;
    rockDataListStruct stR = null;
    int    iType[]  = { 0, 0 };
    String source = "";
    String sDate  = "";
    String sUnit  = "";
    double dStart = 0.0;
    double dEnd   = 0.0;
    int    iBegin = 0;

    switch (iLocation)
    {
      case _PC:
        st = rock.rockDataUtility.copyList( pc.getRock() );
        pc.close();
        pc = null;

        if (st != null)
        {
          st.source = new String("Comma Delimited File"); // Core Source {S}
          st.iSource = las3.las3Constants._SOURCE_CSV;
	    }
      case _KGS:
        if (iLocation == _KGS)
          st = pKGS.getRockData();

        if (st != null)
        {
          for (int i=0; i<st.iCount; i++)
          {
            if ((st.stItem[i].dPCORE > 0.0) || (st.stItem[i].dKMAX > 0.0) ||
                (st.stItem[i].dK90 > 0.0) || (st.stItem[i].dKVRT > 0.0) )
              iType[0] = 1;
            if ((st.stItem[i].dPPLUG > 0.0) || (st.stItem[i].dP800 > 0.0) ||
                (st.stItem[i].dPINSI > 0.0) || (st.stItem[i].dKPLG > 0.0) ||
                (st.stItem[i].dKKL > 0.0) || (st.stItem[i].dKINSI > 0.0) ||
                (st.stItem[i].dKKLIN > 0.0) || (st.stItem[i].dKPVRT > 0.0))
              iType[1] = 1;

            if (st.stItem[i].source.length() > 0)
              source = new String(st.stItem[i].source);
            if (st.stItem[i].sDate.length() > 0)
              sDate = new String(st.stItem[i].sDate);
            if (st.stItem[i].sUNIT.length() > 0)
              sUnit = new String(st.stItem[i].sUNIT);

            if ((iBegin==0) && (st.stItem[i].dTOP != rock.rockDataStruct.dNULL))
            {
              iBegin = 1;
              dStart = st.stItem[i].dTOP;
              dEnd   = st.stItem[i].dTOP;
              if (st.stItem[i].dBASE > 0.0)
                dEnd   = st.stItem[i].dBASE;
            }
            else
            {
              if (st.stItem[i].dTOP > 0.0)
              {
                if (dStart > st.stItem[i].dTOP)
                  dStart = st.stItem[i].dTOP;
              }

              if (st.stItem[i].dBASE > 0.0)
              {
                if (dEnd < st.stItem[i].dBASE)
                  dEnd = st.stItem[i].dBASE;
              }
              else if (dEnd < st.stItem[i].dTOP)
                dEnd = st.stItem[i].dTOP;
            }
          }

          if (iLocation == _KGS)
            st.source = new String("Kansas Geological Survey Database"); // Core Source {S}

          st.sType  = "";   // Core Type {S}
          if ((iType[0] > 0) && (iType[1] > 0))
          {
            st.sType = new String("Whole Core & Core Plug");
            st.diameter = 4.0;  // Core Diameter {F}
          }
          else if (iType[0] > 0)
          {
            st.sType = new String( "Whole Core" );
            st.diameter = 4.0;  // Core Diameter {F}
          }
          else if (iType[1] > 0)
          {
            st.sType    = new String( "Core Plug" );
            st.diameter = 1.0;  // Core Diameter {F}
          }

          st.sRecovery  = "";    // Recovery Date (Date Core Cut) {D}
          st.depthStart = dStart;                // Core Top Depth {F}
          st.depthEnd   = dEnd;                  // Core Base Depth {F}
          st.dLength    = Math.abs(dEnd-dStart); // Recovered Amount (Length) {F}
          st.sFormation = new String(sUnit);     // Primary Formation Cored {S}

          st.sCompany   = new String(source); // Analyzing Company {S}
          st.sAnalyzed  = new String(sDate);  // Analysis Date {D}

          st.iSource = las3.las3Constants._SOURCE_KGS;
        }
        break;
      case _LAS_3_FILE:
        stR = p3.getRockData();
        if (stR != null)
        {
          if (stR.iCount > 0)
          {
            st = p3.getRockData();
          }
        }
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
    }

    return (st);
  }

  /** Method getLithology()
   * <p> This method will retrieve the Rock Lithology Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Rock Lithology Data List Structure
   */

  public static lithologyListStruct getLithology(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    lithologyListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getLithology();
        break;
      case _GEO_REPORT:
        st = parse.parseMineralsUtility.parse(
                    iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
                    stStruct.stSymbols,
                    stStruct.stParseTexture,
                    stStruct.stMinerals );

        break;
    }

    return (st);
  }

  /** Method getLithology()
   * <p> This method will retrieve the Rock Lithology Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Rock Lithology Data List Structure
   */

  public static lithologyListStruct getLithology(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    lithologyListStruct st = null;

    st = parse.parseMineralsUtility.parse(
                iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
                stStruct.stSymbols,
                stStruct.stParseTexture,
                stStruct.stMinerals );

    return (st);
  }

  /** Method getTexture()
   * <p> This method will retrieve the Texture Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Texture Data List Structure
   */

  public static textureListStruct getTexture(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    textureListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getTexture();
        break;
      case _GEO_REPORT:
        st = parse.parseTextureUtility.parse(
                iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
                stStruct.stSymbols,
                stStruct.stParseTexture );
        break;
    }

    return (st);
  }

  /** Method getTexture()
   * <p> This method will retrieve the Texture Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Texture Data List Structure
   */

  public static textureListStruct getTexture(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    textureListStruct st = null;

    st = parse.parseTextureUtility.parse(
            iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
            stStruct.stSymbols,
            stStruct.stParseTexture );

    return (st);
  }

  /** Method getRockColumn()
   * <p> This method will retrieve the Rock Column Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Rock Column Data List Structure
   */

  public static rockColumnListStruct getRockColumn(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    rockColumnListStruct st = null;

    st = parse.parseRockColumnSymbolsUtility.parse(
            iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
            stStruct.stParseRock,
            stStruct.stParseGrain );
//    st = parse.parseRockColumnSymbolsUtility.parse(
//            iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
//            stStruct.stParseRock );

    return (st);
  }

  /** Method getPorosity()
   * <p> This method will retrieve the Porosity Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Porosity Data List Structure
   */

  public static phiListStruct getPorosity(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    phiListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getPhi();
        break;
      case _GEO_REPORT:
        st = parse.parsePHIUtility.parse(
               iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
               stStruct.stParsePorosity);
        break;
    }

    return (st);
  }

  /** Method getPorosity()
   * <p> This method will retrieve the Porosity Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Porosity Data List Structure
   */

  public static phiListStruct getPorosity(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    phiListStruct st = null;

    st = parse.parsePHIUtility.parse(
           iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
           stStruct.stParsePorosity);

    return (st);
  }

  /** Method getFossils()
   * <p> This method will retrieve the Fossil Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Fossil Data List Structure
   */

  public static fossilListStruct getFossils(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    fossilListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getFossils();
        break;
      case _GEO_REPORT:
        st = parse.parseFossilUtility.parse(
               iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
               stStruct.stParseFossils,
               stStruct.stBio,
               stStruct.stKSBio );
        break;
    }

    return (st);
  }

  /** Method getFossils()
   * <p> This method will retrieve the Fossil Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Fossil Data List Structure
   */

  public static fossilListStruct getFossils(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    fossilListStruct st = null;

    st = parse.parseFossilUtility.parse(
           iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
           stStruct.stParseFossils,
           stStruct.stBio,
           stStruct.stKSBio );

    return (st);
  }

  /** Method getGeneraList()
   * <p> This method will retrieve the Genera Fossil Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Fossil Data List Structure
   */

  public static bioStratListStruct getGeneraList(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    bioStratListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
        st = p3.getBio();
        break;
      case _GEO_REPORT:
        st = parse.parseGeneraUtility.parse(
               iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
               stStruct.stKSBio );
        break;
    }

    return (st);
  }

  /** Method getGeneraList()
   * <p> This method will retrieve the Genera Fossil Data List Structure
   * @param  pBio       = Bio-Stratigraphy Data Entry Frame
   * @return st         = Fossil Data List Structure
   */

  public static bioStratListStruct getGeneraList( bioDataEntryFrame pBio )
  {
    bioStratListStruct st = null;

    if (pBio != null) { st = pBio.getList(); }

    return (st);
  }

  /** Method getGeneraList()
   * <p> This method will retrieve the Genera Fossil Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Genera Fossil Data List Structure
   */

  public static bioStratListStruct getGeneraList(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    bioStratListStruct st = null;

    st = parse.parseGeneraUtility.parse(
           iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
           stStruct.stKSBio );

    return (st);
  }

  /** Method getSedStruct()
   * <p> This method will retrieve the Sedimentary Struct Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Sedimentary Struct Data List Structure
   */

  public static sedimentaryListStruct getSedStruct(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    sedimentaryListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getSedStruct();
        break;
      case _GEO_REPORT:
        st = parse.parseSedStructUtility.parse(
               iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
               stStruct.stParseSedStruct );
        break;
    }

    return (st);
  }

  /** Method getSedStruct()
   * <p> This method will retrieve the Sedimentary Struct Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Sedimentary Struct Data List Structure
   */

  public static sedimentaryListStruct getSedStruct(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    sedimentaryListStruct st = null;

    st = parse.parseSedStructUtility.parse(
           iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
           stStruct.stParseSedStruct );

    return (st);
  }

  /** Method getColor()
   * <p> This method will retrieve the Rock Color Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Rock Color Data List Structure
   */

  public static rockColorListStruct getColor(
      int                     iLocation,  // Location identifier
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      kgsSearchKGSFrame       pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame    pNotesFile, // Read Remarks CSV File
      las3Read                p3,         // Read LAS 3 File for Data
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    rockColorListStruct st = null;

    switch (iLocation)
    {
      case _KGS:
        break;
      case _NOTES:
        break;
      case _LAS_3_FILE:
//        st = p3.getColors();
        break;
      case _GEO_REPORT:
        st = parse.parseColorUtility.parse(
               iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
               stStruct ); //.stParseColors );
        break;
    }

    return (st);
  }

  /** Method getColor()
   * <p> This method will retrieve the Rock Color Data List Structure
   * @param  stStruct   = IQSTRAT Global Data Structure
   * @param  stRemarks  = Remarks Data list Structure
   * @return st         = Rock Color Data List Structure
   */

  public static rockColorListStruct getColor(
      iqstratStruct           stStruct,   // IQSTRAT Global Data Structure
      iqstratRemarkListStruct stRemarks ) // Remarks Data list Structure
  {
    rockColorListStruct st = null;

    st = parse.parseColorUtility.parse(
           iqstrat.iqstratRemarkUtility.copyList( stRemarks ),
           stStruct ); //.stParseColors );

    return (st);
  }

  /** Method getRemarks()
   * <p> This method will retrieve the Remarks Data List Structure
   * @param  iLocation  = Location identifier
   * @param  stRock     = The Rock Data List Structure
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  p3         = Read LAS 3 File for Data
   * @param  stKGSMeas  = KGS Measured Section Data List Structure
   * @return st         = Remarks Data List Structure
   */

  public static iqstratRemarkListStruct getRemarks(
      int                   iLocation,   // Location identifier
      rockDataListStruct    stRock,      // Rock Data List Structure
      kgsSearchKGSFrame     pKGS ,       // Search KGS Database for Data
      iqstratNotesCSVFrame  pNotesFile,  // Read Remarks CSV File
      las3Read              p3,          // Read LAS 3 File for Data
      kgsMeasSectListStruct stKGSMeas )  // KGS Measured Section List Structure
  {
    iqstratRemarkListStruct st  = null;
    int iStart = 0;

    switch (iLocation)
    {
      case _ROCK_DATA:
        if (stRock != null)
        {
          if ((stRock._LITHO > -1) || (stRock._ENV > -1))
          {
            st         = new iqstratRemarkListStruct();
            st.stItem  = new iqstratRemarkStruct[stRock.iCount];
            st.iCount  = stRock.iCount;

            st.iSource = stRock.iSource;

            for (int i=0; i<stRock.iCount; i++)
            {
              st.stItem[i]            = new iqstratRemarkStruct();
              st.stItem[i].sKEY       = new String( stRock.sKEY );
              st.stItem[i].depthStart = stRock.stItem[i].dTOP;
              st.stItem[i].depthEnd   = stRock.stItem[i].dBASE;
              st.stItem[i].dThickness =
                 Math.abs(st.stItem[i].depthEnd - st.stItem[i].depthStart);
              st.stItem[i].sText      = new String( "" );
              if (stRock._LITHO > -1)
                st.stItem[i].sText    = new String( stRock.stItem[i].sLITHO );
              if (stRock._ENV > -1)
                st.stItem[i].sText    = new String( st.stItem[i].sText + ", " +
                                                    stRock.stItem[i].sENV );

              if (st.stItem[i].sText.length() < 41)
                st.stItem[i].sNote    = new String( st.stItem[i].sText );
              else
                st.stItem[i].sNote    =
                    new String( st.stItem[i].sText.substring(0, 40) );

              if (iStart == 0)
              {
                st.depthStart = st.stItem[i].depthStart; // Starting Depth
                st.depthEnd   = st.stItem[i].depthEnd;   // Ending Depth
                st.source =
                    new String("Lithofacies Rock Data"); // Source
                st.sRef   = new String("Log Depth"); // Depth Reference

                iStart = 1;
              }

              if (st.depthEnd < st.stItem[i].depthEnd)
                st.depthEnd = st.stItem[i].depthEnd;   // Ending Depth
            }
          }
        }
        break;
      case _KGS:
        break;
      case _NOTES:
        st = pNotesFile.getRemarks();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_CSV; }
        break;
      case _LAS_3_FILE:
        st = p3.getRemarks();
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_LAS; }
        break;
      case _MEAS_SECT:
        st = iqstrat.iqstratRemarkUtility.parse(
                 kgs.kgsMeasSectUtility.copyList( stKGSMeas ) );
        if (st != null) { st.iSource = las3.las3Constants._SOURCE_KGS; }
        break;
    }

    return (st);
  }

  /** Method getRemarks()
   * <p> This method will retrieve the Remarks Data List Structure
   * @param  pRockEntry = Rock Observational Entry Frame
   * @return st         = Remarks Data List Structure
   */

  public static iqstratRemarkListStruct getRemarks(
      rockObsEntryFrame  pRockEntry )  // Rock Data Entry Frame
  {
    iqstratRemarkListStruct st  = null;

    if (pRockEntry != null) { st = pRockEntry.getRemarks(); }

    return (st);
  }

  /** Method setOrder()
   * <p> This method will force a specific Frame to the front
   * @param  iLocation  = Location identifier
   * @param  pKGS       = Search KGS Database for Data
   * @param  pNotesFile = Read Remarks CSV File
   * @param  pX         = Cross Section Project Frame
   * @param  p3         = Read LAS 3 File for Data
   */

  public static void setOrder(
      int                  iLocation,  // Location identifier
      kgsSearchKGSFrame    pKGS ,      // Search KGS Database for Data
      iqstratNotesCSVFrame pNotesFile, // Read Remarks CSV File
      las3Read             p3)         // Read LAS 3 File for Data
  {
    switch (iLocation)
    {
      case _KGS:
        pKGS.toFront();
        break;
      case _NOTES:
        break;
      case _ROCK_ENTRY:
        break;
      case _LAS_3_FILE:
        break;
    }
  }
}