/*
 * @iqstratHeadersUtility.java Version 1.1 10/05/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat;

import iqstrat.iqstratHeadersListStruct;
import iqstrat.iqstratHeadersStruct;
import kgs.kgsMeasSectStruct;

/** Class iqstratHeadersUtility
 *  <p> This Class will provide basic utility for the data structures
 *
 *  @version 1.1 10/05/2007
 *  @author  John R. Victorine
 */

public class iqstratHeadersUtility
{
  /** Method exists()
   * <p> This method will check to see if the record exists in the list
   * @param  st    = Location Header Information Data List Structure
   * @param  sKEY  = Primary Key of data defined by user
   * @param  sAPI  = The API-Number
   * @param  sKID  = Well Header KID or KGS Primary Key
   * @param  sName = The name of the well or location
   * @return sKEY  = The user created primary key
   * @return sID   = The user created primary key
   */

  public static String exists( iqstratHeadersListStruct st,
                               String sKEY,
                               String sAPI,
                               String sKID,
                               String sName)
  {
    int    iFound = -1;
    String sID    = "0";

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].sKEY.equals(sKEY)) && (sKEY.length() > 1))
        {
          iFound = i;
          sID    = new String( st.stItem[i].sKEY );
		}

        if ((iFound == -1) &&
            (st.stItem[i].sAPI.equals(sAPI)) && (sAPI.length() > 1))
        {
          iFound = i;
          sID    = new String( st.stItem[i].sKEY );
		}

        if ((iFound == -1) &&
            (st.stItem[i].sKID.equals(sKID)) && (sKID.length() > 1))
        {
          iFound = i;
          sID    = new String( st.stItem[i].sKEY );
		}

        if ((iFound == -1) &&
            (st.stItem[i].sName.toUpperCase().equals(sName.toUpperCase())) &&
            (sName.length() > 1))
        {
          iFound = i;
          sID    = new String( st.stItem[i].sKEY );
		}
      }
    }

    return (sID);
  }

  /** Method getKEY()
   * <p> This method will check to see if the record exists in the list
   * @param  st    = Header Information Data List Structure
   * @return sID   = The user created primary key
   */

  public static String getKEY( iqstratHeadersStruct st )
  {
    int    iFound = -1;
    String sID    = "0";

    if (st != null)
    {
      if ((iFound == -1) && (st.sKEY.length() > 1))
      {
        iFound = 1;
        sID    = new String( st.sKEY );
      }

      if ((iFound == -1) && (st.sAPI.length() > 1))
      {
        iFound = 1;
        sID    = new String( st.sAPI );
      }

      if ((iFound == -1) && (st.sKID.length() > 1))
      {
        iFound = 1;
        sID    = new String( st.sKID );
	  }

      if ((iFound == -1) && (st.sName.length() > 1))
      {
        iFound = 1;
        sID    = new String( st.sName.toUpperCase() );
        sID    = cmn.cmnString.replaceChar(sID, ' ', '_');
        sID    = cmn.cmnString.removeChar(sID, '\'');
        sID    = cmn.cmnString.removeChar(sID, '\"');
        sID    = cmn.cmnString.removeChar(sID, '&');
	  }

      if ((iFound == -1) && (sID.equals("0")))
      {
		sID    = new String( "W" + cmn.cmnString.UniqueName());
      }
    }

    return (sID);
  }

  /** Method initialize()
   *  <p> This method will initialize the Location Data Structure
   *   @return st = The New Location Data Structure
   *

  public static iqstratHeadersStruct initialize()
  {
    iqstratHeadersStruct st = new iqstratHeadersStruct();

    if (st != null)
    {
      // Identification Information

      st.sKID       = new String("0"); // Primary Key
      st.sWELLKID   = new String("0"); // KGS Primary Key of Well
      st.sKEY       = new String(""); // Primary Key created on user's PC
      st.iType      = iqstrat.iqstratHeadersStruct.WELL;
      st.sAPI       = new String(""); // API-Number
      st.sUWI       = new String(""); // UWI
      st.sName      = new String(""); // Well Name or Outcrop Name
      st.status     = iqstrat.iqstratStatusStruct.STATUS[0][0]; // Status
      st.sError     = "";

      // other well information

      st.service       = new String("");  // Service Company
      st.sLicense      = new String("");  // License

      st.sOperator     = new String("");  // Operator
      st.sOperator_kid = new String("0"); // Operator Primary Key
      st.sField        = new String("");  // Field Name
      st.sField_kid    = new String("0"); // Field Name Primary Key

      // Location Information

      st.sCountry   = new String(""); // Country
      st.sProvince  = new String(""); // Province

      st.state      = new String(""); // State Name
      st.iState     = -1;             // State Code
      st.sCounty    = new String(""); // County Name
      st.iCounty    = -1;             // County Code

      st.sLocation  = new String(""); // location of outcrop or stream cut

      st.iTownship = 0;               // Township
      st.sTownship = new String("");  // Township Direction
      st.iRange    = 0;               // Range
      st.sRange    = new String("");  // Range Direction
      st.iSection  = 0;               // Section

      // XY Position Information

      st.dLatitude  = 0.0; // Latitude of Location
      st.dLongitude = 0.0; // Longitude of Location
      st.dZone      = 0.0; // UTM Zone
      st.dUTMx      = 0.0; // UTM X-Position
      st.dUTMy      = 0.0; // UTM Y-Position

      // Z Position - Depth, Elevation

      st.depth      = 0.0; // Total Depth of Well or Height of Outcrop
      st.dGL        = 0.0; // Ground Level
      st.dKB        = 0.0; // Kelly Bushing
      st.dDF        = 0.0; // Derrick Floor
      st.dRT        = 0.0;   // Rotary Table

//      st.dTDD       = 0.0;   // Drillers Depth
//      st.dTLD       = 0.0;   // Loggers Depth

      // Well Depth Information For LAS 3 Format

//      st.sPDAT      = new String("");    // Permanent Data
//      st.dAPD       = 0.0;               // Above Permanent Data
//      st.sDREF      = new String("");    // Depth Reference (KB,DF,CB)
//      st.dElevation = 0.0;               // "Elevation of Depth Reference"

      // Other Fields not really displayed

      st.sUpdate    = new String(""); // update by
      st.source     = new String(""); // source of data
      st.iAccess    = iqstrat.iqstratHeadersStruct.USER_ONLY;
      st.sDate      = new String(""); // update date
      st.sComments  = new String(""); // general comments
    }

    return (st);
  }
*/
  /** Method copy()
   * <p> This method will copy one Location Info structure to another
   * @param  stOld = The Location Information Data Structure
   * @return stNew = The Location Information Data Structure
   */

  public static iqstratHeadersStruct copy(iqstratHeadersStruct stOld)
  {
    iqstratHeadersStruct stNew = new iqstratHeadersStruct();

    if (stOld != null)
    {
      // Identification Information

      stNew.sKID   = new String(stOld.sKID);   // Location Header KID
      stNew.sWELLKID = new String(stOld.sWELLKID); // KGS Primary Key
      stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
      stNew.iType  = stOld.iType;

      stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
      stNew.sUWI   = new String(stOld.sUWI);   // UWI
      stNew.sName  = new String(stOld.sName);  // Well Name or Outcrop Name
      stNew.status = new String(stOld.status); // Status of well or Outcrop Type
      stNew.sError = "";

      // other well information

      stNew.service       = new String(stOld.service);       // Service Company
      stNew.sLicense      = new String(stOld.sLicense);      // License

      stNew.sOperator     = new String(stOld.sOperator);     // Operator
      stNew.sOperator_kid = new String(stOld.sOperator_kid); // Operator Key
      stNew.sField        = new String(stOld.sField);        // Field Name
      stNew.sField_kid    = new String(stOld.sField_kid);    // Field Name Key

      // Location Information

      stNew.sCountry   = new String(stOld.sCountry); // Country
      stNew.sProvince  = new String(stOld.sProvince); // Province

      stNew.state     = new String(stOld.state);     // State Name
      stNew.iState    = stOld.iState;                // State Code
      stNew.sCounty   = new String(stOld.sCounty);   // County Name
      stNew.iCounty   = stOld.iCounty;               // County Code

      stNew.sLocation  = new String(stOld.sLocation);  // location of outcrop
      stNew.sLocation1 = new String(stOld.sLocation1); // location 1
      stNew.sLocation2 = new String(stOld.sLocation2); // location 2

      stNew.iTownship = stOld.iTownship;              // Township
      stNew.sTownship = new String(stOld.sTownship);  // Township Direction
      stNew.iRange    = stOld.iRange;                 // Range
      stNew.sRange    = new String(stOld.sRange);     // Range Direction
      stNew.iSection  = stOld.iSection;               // Section

      // XY Position Information

      stNew.dLatitude  = stOld.dLatitude;   // Latitude
      stNew.dLongitude = stOld.dLongitude;  // Longitude
      stNew.dZone      = stOld.dZone;       // UTM Zone
      stNew.dUTMx      = stOld.dUTMx;       // UTM X-Position
      stNew.dUTMy      = stOld.dUTMy;       // UTM Y-Position

      stNew.sGDAT      = new String(stOld.sGDAT); // Geodetic Datum
      stNew.sHZCS      = new String(stOld.sHZCS); // Hori Co-ordinate System

      // Z Position - Depth, Elevation

      stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
      stNew.dGL   = stOld.dGL;   // Ground Level
      stNew.dKB   = stOld.dKB;   // Kelly Bushing
      stNew.dDF   = stOld.dDF;   // Derrick Floor
      stNew.dRT   = stOld.dRT;   // Rotary Table

//      stNew.dTDD  = stOld.dTDD;  // Drillers Depth
//      stNew.dTLD  = stOld.dTLD;  // Loggers Depth

//      stNew.sPDAT      = new String( stOld.sPDAT ); // Permanent Data
//      stNew.dAPD       = stOld.dAPD;                // Above Permanent Data
//      stNew.sDREF      = new String (stOld.sDREF ); // Depth Reference (KB,DF,CB)
//      stNew.dElevation = stOld.dElevation;     // "Elevation of Depth Reference"

      // Other Fields not really displayed

      stNew.sUpdate   = new String(stOld.sUpdate);   // update by
      stNew.sDate     = new String(stOld.sDate);     // update date
      stNew.source    = new String(stOld.source);    // Source of data
      stNew.iAccess   = stOld.iAccess;               // Access to data
      stNew.sComments = new String(stOld.sComments); // general comments

      stNew.iLAS      = stOld.iLAS;     // Number of LAS Files
      stNew.iTops     = stOld.iTops;    // Number of Tops
      stNew.iCore     = stOld.iCore;    // Number of Core Data
      stNew.images    = stOld.images;   // Number of images
      stNew.iPerf     = stOld.iPerf;    // Number of Perforations
      stNew.iDST      = stOld.iDST;     // Number of DST Records
      stNew.iPfeffer  = stOld.iPfeffer; // Number of Pfeffer Records
      stNew.iGeo      = stOld.iGeo;     // Number of Geo Report Records

      stNew.iOHM      = stOld.iOHM;     // Resistivity Log Data Exists
      stNew.iLitho    = stOld.iLitho;   // Litho-Density Log Data Exists
      stNew.iNPHI     = stOld.iNPHI;    // Neutron Porosity Log Data Exists
      stNew.iRHOB     = stOld.iRHOB;    // Bulk Density Log Data Exists
      stNew.iPE       = stOld.iPE;      // Photoelectric Factor Log Data Exists
      stNew.iSonic    = stOld.iSonic;   // Sonic Log Data Exists
      stNew.iGR       = stOld.iGR;      // Gamma Ray Log Data Exists
      stNew.iSPGR     = stOld.iSPGR;    // Spectral Gamma Ray Log Data Exists
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratHeadersStruct transfer(iqstratHeadersStruct stOld)
  {
    iqstratHeadersStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copy( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method add()
   * <p> This method will remove a Location from the Table
   * @param  st    = Location Header Information XML Data List Structure
   * @param  stNew = Location Header Information to be added to list
   * @return st    = Location Header Information XML Data List Structure
   */

  public static iqstratHeadersListStruct add( iqstratHeadersListStruct st,
                                              iqstratHeadersStruct stNew)
  {
    int i=0;
    int iCount=0;
    int iFound=-1;

    iqstratHeadersListStruct stTemp = new iqstratHeadersListStruct();
    stTemp.stItem = new iqstratHeadersStruct[1];
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratHeadersStruct[st.iCount+1];
      for (i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].sKID.equals(stNew.sKID)) &&
            (stNew.sKID.length() > 1))
          iFound = i;

        if ((st.stItem[i].sKEY.equals(stNew.sKEY)) &&
            (stNew.sKEY.length() > 1))
          iFound = i;

        if ((st.stItem[i].sAPI.equals(stNew.sAPI)) &&
            (stNew.sAPI.length() > 1))
          iFound = i;

        stTemp.stItem[iCount] = new iqstratHeadersStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }
    }

    if (iFound == -1)
    {
      stTemp.stItem[iCount] = new iqstratHeadersStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new iqstratHeadersListStruct();
    st.stItem = new iqstratHeadersStruct[iCount];
    st.iCount = iCount;

    for (i=0; i<iCount; i++)
    {
      st.stItem[i] = new iqstratHeadersStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Location data in the Table
   * @param  st    = Location Header Information XML Data List Structure
   * @param  stNew = Location Header Information to be modified in the list
   * @return st    = Location Header Information XML Data List Structure
   */

  public static iqstratHeadersListStruct modify( iqstratHeadersListStruct st,
                                                 iqstratHeadersStruct stNew)
  {
    int i=0;
    int iCount=0;

    iqstratHeadersListStruct stTemp = new iqstratHeadersListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratHeadersStruct[st.iCount];
      for (i=0; i<st.iCount; i++)
      {
        if ((stNew.sKEY.equals(st.stItem[i].sKEY)) &&
            (stNew.sKEY.length() > 1))
        {
          stTemp.stItem[iCount] = new iqstratHeadersStruct();
          stTemp.stItem[iCount] = copy(stNew);
        }
        else
        {
          stTemp.stItem[iCount] = new iqstratHeadersStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
        }
        iCount++;
      }

      st        = new iqstratHeadersListStruct();
      st.stItem = new iqstratHeadersStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratHeadersStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Location from the Table
   * @param  st   = Location Header Information XML Data List Structure
   * @param  sKEY = Primary Key of data defined by user
   * @return st   = Location Header Information XML Data List Structure
   */

  public static iqstratHeadersListStruct remove( iqstratHeadersListStruct st,
                                                 String sKEY)
  {
    int i=0;
    int iCount = 0;
    int iFound = 0;

    // If the Location is to be removed from the Table Selection only
    //   GET the API Number from the highlighted row.

    iqstratHeadersListStruct stTemp = new iqstratHeadersListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratHeadersStruct[st.iCount-1];
      for (i=0; i<st.iCount; i++)
      {
        iFound = 0;
        if ((sKEY.equals(st.stItem[i].sKEY)) && (sKEY.length() > 1))
        {
          iFound = 1;
        }

        if (iFound != 1)
        {
          stTemp.stItem[iCount] = new iqstratHeadersStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st        = new iqstratHeadersListStruct();
      st.stItem = new iqstratHeadersStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratHeadersStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }
    }

    return (st);
  }

  /** Method setHeaders()
   * <p> This method will set the Header Data Structure data variables
   * @param  st       = KGS Measured Section List Data Structure
   * @return stHeader = Headers Information Data Structure
   */

  public static iqstratHeadersStruct setHeaders( kgsMeasSectStruct st )
  {
    iqstratHeadersStruct stHeader = null;

    if (st != null)
    {
      stHeader = new iqstratHeadersStruct();

      // Identification Information

      stHeader.sKID       = new String(st.sKID);
      stHeader.iType      = iqstrat.iqstratHeadersStruct.WELL; // Type of Data
      stHeader.sLocation1 = new String(st.sName);
      stHeader.status     = iqstrat.iqstratStatusStruct.STATUS[0][0]; // Status

      if (stHeader.sName.length() > 60)
        stHeader.sName = new String(stHeader.sName.substring(0, 61));

      // Location Information

      stHeader.state     = new String(st.state);     // State Name
      stHeader.iState    = st.iState;                // State Code
      stHeader.sCounty   = new String(st.sCounty);   // County Name
      stHeader.iCounty   = st.iCounty;               // County Code

      stHeader.iTownship = st.iTownship;             // Township
      stHeader.sTownship = new String(st.sTownship); // Township Direction
      stHeader.iRange    = st.iRange;                // Range
      stHeader.sRange    = new String(st.sRange);    // Range Direction
      stHeader.iSection  = st.iSection;              // Section
      stHeader.sLocation = new String( st.iSection  + "-" +
                                       st.iTownship + st.sTownship + "-" +
                                       st.iRange    + st.sRange );

      // XY Position Information

      stHeader.dLatitude  = st.dLatitude;    // Latitude of Location
      stHeader.dLongitude = st.dLongitude;   // Longitude of Location

      // Z Position - Depth, Elevation

      stHeader.dGL        = st.dGL;   // Ground Level

      // Miscellaneous Fields

      stHeader.source    = new String( st.sMeasuredBy );
      stHeader.iAccess   = iqstrat.iqstratHeadersStruct.USER_ONLY;
      stHeader.sComments = new String( st.sKID  + " " +
                                       st.sName + " " +
                                       st.sComments );
    }

    return (stHeader);
  }

  /** Method getElevation()
   * <p> This method will retrieve the elevation for a specific well/outcrop
   * @param  st    = The cross section data structure
   * @return dElev = The Elevation Depth
   */

  public static double getElevation( iqstratHeadersStruct st )
  {
    double dElev =0.0;

    if (st != null)
    {
      if (st.dGL > 0.0)
        dElev = st.dGL;
      else if (st.dKB > 0.0)
        dElev = st.dKB;
      else
        dElev = st.dDF;
    }

    return (dElev);
  }

  /** Method print()
   * <p> This method will print the Headers Information Data Structure
   * @param st = Headers Information Data Structure
   */

  public static void print(iqstratHeadersStruct st)
  {
    String sTemp = "";
    if (st != null)
    {
      // Identification Information

      if (st.sKID.length() > 1)
        sTemp = new String(sTemp + "KID:             "+st.sKID+"\n");
      if (st.sWELLKID.length() > 1)
        sTemp = new String(sTemp + "KGS Well KID:    "+st.sWELLKID+"\n");
      if (st.sKEY.length() > 1)
        sTemp = new String(sTemp + "KEY:             "+st.sKEY+"\n");
      sTemp = new String(sTemp + "Type:             "+st.iType+"\n");
      if (st.sAPI.length() > 1)
        sTemp = new String(sTemp + "API-Number:      "+st.sAPI+"\n");
      if (st.sUWI.length() > 1)
        sTemp = new String(sTemp + "Unique Well ID:  "+st.sUWI+"\n");
      if (st.sName.length() > 1)
        sTemp = new String(sTemp + "Name:            "+st.sName+"\n");
      if (st.status.length() > 1)
        sTemp = new String(sTemp + "Status:          "+st.status+"\n");

      System.out.println(sTemp);
      sTemp = new String("");

      // other well information

      if (st.service.length() > 1)
        sTemp = new String(sTemp + "Service Company: "+st.service+"\n");
      if (st.sLicense.length() > 1)
        sTemp = new String(sTemp + "License:         "+st.sLicense+"\n");
      if (st.sDate.length() > 1)
        sTemp = new String(sTemp + "Date:            "+st.sDate+"\n");
      sTemp = new String("\n");

      if (st.sOperator.length() > 1)
        sTemp = new String(sTemp + "Operator:        "+st.sOperator+"\n");
      if (st.sOperator_kid.length() > 1)
        sTemp = new String(sTemp + "Operator KID:    "+st.sOperator_kid+"\n");
      if (st.sField.length() > 1)
        sTemp = new String(sTemp + "Field:           "+st.sField+"\n");
      if (st.sField_kid.length() > 1)
        sTemp = new String(sTemp + "Field KID:       "+st.sField_kid+"\n");

      System.out.println(sTemp);
      sTemp = new String("");

      // Location Information

      if (st.sCountry.length() > 1)
        sTemp = new String(sTemp + "Country:         "+st.sCountry+"\n");
      if (st.sProvince.length() > 1)
        sTemp = new String(sTemp + "Province:        "+st.sProvince+"\n");

      if (st.state.length() > 1)
        sTemp = new String(sTemp + "State:           "+st.state+"\n");
      if (st.iState > -1)
        sTemp = new String(sTemp + "State Code:      "+st.iState+"\n");
      if (st.sCounty.length() > 1)
        sTemp = new String(sTemp + "County:          "+st.sCounty+"\n");
      if (st.iCounty > -1)
        sTemp = new String(sTemp + "County Code:     "+st.iCounty+"\n");

      if (st.sLocation.length() > 1)
        sTemp = new String(sTemp + "Location:        "+st.sLocation+"\n");
      if (st.sLocation1.length() > 1)
        sTemp = new String(sTemp + "Location 1:      "+st.sLocation1+"\n");
      if (st.sLocation2.length() > 1)
        sTemp = new String(sTemp + "Location 2:      "+st.sLocation2+"\n");

      if (st.iTownship > 0)
        sTemp = new String(sTemp + "T: "+st.iTownship+st.sTownship+" ");
      if (st.iRange > 0)
        sTemp = new String(sTemp + "R: "+st.iRange+st.sRange+" ");
      if (st.iSection > 0)
        sTemp = new String(sTemp + "S: "+st.iSection+" ");
      if ((st.iTownship > 0) || (st.iRange > 0) || (st.iSection > 0))
        sTemp = new String(sTemp + "\n");

      System.out.println(sTemp);
      sTemp = new String("");

      // XY Position Information

      if (Math.abs(st.dLatitude) > 0)
        sTemp = new String(sTemp + "Latitude:        "+st.dLatitude+"\n");
      if (Math.abs(st.dLongitude) > 0)
        sTemp = new String(sTemp + "Longitude:       "+st.dLongitude+"\n");

      System.out.println(sTemp);
      sTemp = new String("");

      if ((Math.abs(st.dZone) > 0) ||
          (Math.abs(st.dUTMx) > 0) || (Math.abs(st.dUTMy) > 0))
      {
        System.out.println(
          "Zone:  "+st.dZone+"\n"+
          "UTM-X: "+st.dUTMx+"\n"+
          "UTM-Y: "+st.dUTMy+"\n\n");
      }

      if (st.sGDAT.length() > 1)
        sTemp = new String(sTemp + "Geodetic Datum:  "+st.sGDAT+"\n");
      if (st.sHZCS.length() > 1)
        sTemp = new String(sTemp + "Horizontal Coor: "+st.sHZCS+"\n");

      System.out.println(sTemp);
      sTemp = new String("");

      // Z Position - Depth, Elevation

      if (Math.abs(st.depth) > 0)
        sTemp = new String(sTemp + "Total Depth/Height: "+st.depth+"\n");
      if (Math.abs(st.dGL) > 0)
        sTemp = new String(sTemp + "Ground Level:       "+st.dGL+"\n");
      if (Math.abs(st.dKB) > 0)
        sTemp = new String(sTemp + "Kelly Bushing:      "+st.dKB+"\n");
      if (Math.abs(st.dDF) > 0)
        sTemp = new String(sTemp + "Drilling Rig Floor: "+st.dDF+"\n");

      System.out.println(sTemp);
      sTemp = new String("");
    }
  }
}