/*
 * @iqstratControlUtility.java Version 1.1 02/09/2009
 *
 * Copyright (c) 2009 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat;

import iqstrat.iqstratControlListStruct;
import iqstrat.iqstratControlStruct;
import iqstrat.iqstratHeadersStruct;
import las.lasFileDataStruct;

/** Class iqstratControlUtility
 *  <p> This Class will provide basic utility for the data structures
 *
 *  @version 1.1 02/09/2009
 *  @author  John R. Victorine
 */

public class iqstratControlUtility
{
  public static final int _LATITUDE  = 0;
  public static final int _LONGITUDE = 1;

  /** Method initialize()
   *  <p> This method will initialize the Control Data Structure
   *   @return st = The New Control Data Structure
   */

  public static iqstratControlStruct initialize()
  {
    iqstratControlStruct st = new iqstratControlStruct();

    if (st != null)
    {
      // LAS 3 Control Parameter

      st.sKGS       = new String( "YES" );

      // Depth Range Data

      st.depthStart = 0.0;  // Starting Depth of Plot (at Elevation)
      st.depthEnd   = 0.0;  // Ending Depth of Plot (at Elevation)
      st.iScale     = iqstrat.iqstratTracksStruct.SCALE_100;

      // Well Plot Data Tracks

      st.iPanel  = iqstrat.iqstratControlStruct._DEFAULT_ORDER;
      st.iConfig = iqstrat.iqstratControlStruct._C_ALL;
      st.iTracks = null;  // Selected Tracks

      // Well Information

//      st.stHeader = iqstrat.iqstratHeadersUtility.initialize();
      st.stHeader = new iqstratHeadersStruct();

      // -- Porosity Thin Track

      st.iThin    = iqstrat.iqstratControlStruct._NO;
      st.iTHINPHI = iqstrat.iqstratTracksStruct._THIN_NPHI;

      // -- Porosity Image Track

      st.iPHI     = iqstrat.iqstratControlStruct._NO;
      st.dPHIMin  = 0.0;
      st.dPHIMax  = 0.3;
      st.iPHITool = las.lasStandardTools._NPHI;

      // -- Resistivity Image Track

      st.iOHM     = iqstrat.iqstratControlStruct._NO;
      st.dOHMMin  = 0.0;
      st.dOHMMax  = 500.0;
      st.iOHMTool = las.lasStandardTools._ILD;

      // Data Conversion Variables

      // -- Gamma Ray Counts to Gamma Ray API

      st.dGRNMin   = 0.0;   // Minimum Gamma Ray Counts Value
      st.dGRNMax   = 100.0; // Maximum Gamma Ray Counts Value

      st.dGRCMin   = 0.0;   // Minimum Computation Value
      st.dGRCMax   = 150.0; // Maximum Computation Value

      // -- Neutron Counts to Neutron Porosity

      st.dNEUTMin  = 0.0;   // Minimum Neutron Counts Value
      st.dNEUTMax  = 100.0; // Maximum Neutron Counts Value

      st.dNPHICMin = 0.01;  // Minimum Computation Value
      st.dNPHICMax = 0.4;   // Maximum Computation Value
    }

    return ( st );
  }

  /** Method addHeaders( iqstratHeadersStruct stHeader,
   *                     iqstratControlStruct st )
   * <p> This method will add the Header data structure to
   *     the Control data structure
   * @param  stHeader = the Headers Data Struct
   * @param  st       = The Control Data Structure
   * @return st       = The Control Data Structure
   */

  public static iqstratControlStruct addHeaders( iqstratHeadersStruct stHeader,
                                                  iqstratControlStruct st )
  {
    if (stHeader != null)
    {
      if (st.stHeader != null)
        st.stHeader.delete();
      st.stHeader = null;

      st.stHeader = iqstrat.iqstratHeadersUtility.copy( stHeader );
    }

    return ( st );
  }

  /** Method copyList( iqstratControlListStruct stOld )
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratControlListStruct copyList(
                                           iqstratControlListStruct stOld )
  {
    iqstratControlListStruct stNew = null;

    if (stOld != null)
    {
      stNew      = new iqstratControlListStruct();

      // Project Identifier

      stNew.sKID = new String(stOld.sKID); // Project KID - Created on KGS DB
      stNew.sKEY = new String(stOld.sKEY); // Project KEY - Created from Data & Time

      // Plot titles

      stNew.sTitle1 = new String(stOld.sTitle1); // 1st Plot Title
      stNew.sTitle2 = new String(stOld.sTitle2); // 2nd Plot Title

      // Cross Section Plot Depth Range

      stNew.depthStart = stOld.depthStart; // Starting Depth
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth

      stNew.iCount     = stOld.iCount;
      stNew.stItem     = new iqstratControlStruct[stOld.iCount];

      for (int i=0; i<stOld.iCount; i++)
      {
        stNew.stItem[i] = copy( stOld.stItem[i] );
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratControlListStruct transfer(iqstratControlListStruct stOld)
  {
    iqstratControlListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method copy( iqstratControlStruct stOld )
   * <p> This method will copy one Control data structure to another
   * @param  stOld = The Control Data Structure
   * @return stNew = The Control Data Structure
   */

  public static iqstratControlStruct copy( iqstratControlStruct stOld )
  {
    iqstratControlStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new iqstratControlStruct();

      stNew.sKGS       = new String( stOld.sKGS ); // KGS Saved Data Indicator

      // Depth Range Data

      stNew.depthStart = stOld.depthStart; // Starting Depth of Plot (at Elevation)
      stNew.depthEnd   = stOld.depthEnd;   // Ending Depth of Plot (at Elevation)
      stNew.iScale     = stOld.iScale;     // Scale Identifier

      // Well Plot Data Tracks

      stNew.iPanel  = stOld.iPanel;
      stNew.iConfig = stOld.iConfig;

      if (stOld.iTracks != null)
      {
        stNew.iTracks = new int[stOld.iTracks.length]; // Selected Tracks
        for (int i=0; i<stOld.iTracks.length; i++)
        {
          stNew.iTracks[i] = stOld.iTracks[i];
        }
      }

      // Well Information

      stNew.stHeader = iqstrat.iqstratHeadersUtility.copy( stOld.stHeader );

      // -- Porosity Thin Track

      stNew.iThin    = stOld.iThin;
      stNew.iTHINPHI = stOld.iTHINPHI;

      // -- Porosity Image Track

      stNew.iPHI     = stOld.iPHI;
      stNew.dPHIMin  = stOld.dPHIMin;
      stNew.dPHIMax  = stOld.dPHIMax;
      stNew.iPHITool = stOld.iPHITool;

      // -- Resistivity Image Track

      stNew.iOHM     = stOld.iOHM;
      stNew.dOHMMin  = stOld.dOHMMin;
      stNew.dOHMMax  = stOld.dOHMMax;
      stNew.iOHMTool = stOld.iOHMTool;

      // Data Conversion Variables

      // -- Gamma Ray Counts to Gamma Ray API

      stNew.dGRNMin   = stOld.dGRNMin;   // Minimum Gamma Ray Counts Value
      stNew.dGRNMax   = stOld.dGRNMax;   // Maximum Gamma Ray Counts Value

      stNew.dGRCMin   = stOld.dGRCMin;   // Minimum Gamma Ray Value
      stNew.dGRCMax   = stOld.dGRCMax;   // Maximum Gamma Ray Value

      // -- Neutron Counts to Neutron Porosity

      stNew.dNEUTMin  = stOld.dNEUTMin;  // Minimum Neutron Counts Value
      stNew.dNEUTMax  = stOld.dNEUTMax;  // Maximum Neutron Counts Value

      stNew.dNPHICMin = stOld.dNPHICMin; // Minimum NPHI Value
      stNew.dNPHICMax = stOld.dNPHICMax; // Maximum NPHI Value
    }

    return ( stNew );
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static iqstratControlStruct transfer(iqstratControlStruct stOld)
  {
    iqstratControlStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copy( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }


  /** Method exists( iqstratControlListStruct st, String sKID,  String sKEY )
   * <p> This method will check to see if the record exists in the list
   * @param  st    = Control Data List Structure
   * @param  sKEY  = Primary Key of data defined by user
   * @return sKEY  = The primary key
   */

  public static String exists( iqstratControlListStruct st, String sKEY )
  {
    int iFound = -1;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].stHeader.sKEY.equals(sKEY)) && (sKEY.length() > 1))
        {
          iFound = i;
        }
      }
    }

    sKEY = new String("0");
    if (iFound > -1)
      sKEY = new String(st.stItem[iFound].stHeader.sKEY);

    return ( sKEY );
  }

  /** Method exists_db( iqstratControlListStruct st, String sKID )
   * <p> This method will check to see if the record exists in the list
   * @param  st    = Control Data List Structure
   * @param  sKID  = Well Header KID or KGS Primary Key
   * @return sKID  = The primary key
   */

  public static String exists_db( iqstratControlListStruct st, String sKID )
  {
    int iFound = -1;

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].stHeader.sKID.equals(sKID)) && (sKID.length() > 1))
        {
          iFound = i;
        }
      }
    }

    sKID = new String("0");
    if (iFound > -1)
      sKID = new String(st.stItem[iFound].stHeader.sKID);

    return ( sKID );
  }

  /** Method addConversionData()
   * <p> This method will load the Conversion Parameters from the LAS File Data
   *     Structure to the Control Plot Data Structure
   * @param  stL = LAS File Data Structure
   * @param  st  = Control Plot Data Structure
   * @return st  = Control Plot Data Structure
   */

  public static iqstratControlStruct addConversionData(
                    lasFileDataStruct stL, iqstratControlStruct st )
  {
    if ((st != null) && (stL != null))
    {
      // Data Conversion Variables

      if (stL.iGRN > -1)
      {
        // -- Gamma Ray Counts to Gamma Ray API

        st.dGRNMin   = stL.dGRNMin;   // Minimum Gamma Ray Counts Value
        st.dGRNMax   = stL.dGRNMax;   // Maximum Gamma Ray Counts Value

        st.dGRCMin   = stL.dGRCMin;   // Minimum Gamma Ray Value
        st.dGRCMax   = stL.dGRCMax;   // Maximum Gamma Ray Value
      }

      if (stL.iNEUT > -1)
      {
        // -- Neutron Counts to Neutron Porosity

        st.dNEUTMin  = stL.dNEUTMin;  // Minimum Neutron Counts Value
        st.dNEUTMax  = stL.dNEUTMax;  // Maximum Neutron Counts Value

        st.dNPHICMin = stL.dNPHICMin; // Minimum NPHI Value
        st.dNPHICMax = stL.dNPHICMax; // Maximum NPHI Value
      }
    }

    return (st);
  }

  /** Method add()
   * <p> This method will remove a Location from the Table
   * @param  st    = Control Data List Structure
   * @param  stNew = Control Data Structure to be added to list
   * @return st    = Control Data List Structure
   */

  public static iqstratControlListStruct add( iqstratControlListStruct st,
                                               iqstratControlStruct stNew)
  {
    int i=0;
    int iCount=0;
    int iFound=-1;

    iqstratControlListStruct stTemp = new iqstratControlListStruct();
    stTemp.stItem = new iqstratControlStruct[1];
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratControlStruct[st.iCount+1];
      for (i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].stHeader.sKID.equals(stNew.stHeader.sKID)) &&
            (stNew.stHeader.sKID.length() > 1))
          iFound = i;

        if ((st.stItem[i].stHeader.sKEY.equals(stNew.stHeader.sKEY)) &&
            (stNew.stHeader.sKEY.length() > 1))
          iFound = i;

        if ((st.stItem[i].stHeader.sAPI.equals(stNew.stHeader.sAPI)) &&
            (stNew.stHeader.sAPI.length() > 1))
          iFound = i;

        stTemp.stItem[iCount] = new iqstratControlStruct();
        stTemp.stItem[iCount] = copy(st.stItem[i]);
        iCount++;
      }

      st.delete();
      st = null;
    }

    if (iFound == -1)
    {
      stTemp.stItem[iCount] = new iqstratControlStruct();
      stTemp.stItem[iCount] = copy(stNew);
      iCount++;
    }

    st        = new iqstratControlListStruct();
    st.stItem = new iqstratControlStruct[iCount];
    st.iCount = iCount;

    for (i=0; i<iCount; i++)
    {
      st.stItem[i] = new iqstratControlStruct();
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();
    stTemp = null;

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a Location data in the Table
   * @param  st = Control Data List Structure
   * @param  stNew = Control Data Structure to be modified in the list
   * @return st = Control Data List Structure
   */

  public static iqstratControlListStruct modify( iqstratControlListStruct st,
                                                  iqstratControlStruct stNew)
  {
    int i=0;
    int iCount=0;

    iqstratControlListStruct stTemp = new iqstratControlListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratControlStruct[st.iCount];
      for (i=0; i<st.iCount; i++)
      {
        if (((stNew.stHeader.sAPI.equals(st.stItem[i].stHeader.sAPI)) &&
             (stNew.stHeader.sAPI.length() > 1)) ||
            ((stNew.stHeader.sKID.equals(st.stItem[i].stHeader.sKID)) &&
             (stNew.stHeader.sKID.length() > 1)) ||
            ((stNew.stHeader.sKEY.equals(st.stItem[i].stHeader.sKEY)) &&
             (stNew.stHeader.sKEY.length() > 1)))
        {
          stTemp.stItem[iCount] = new iqstratControlStruct();
          stTemp.stItem[iCount] = copy(stNew);
        }
        else
        {
          stTemp.stItem[iCount] = new iqstratControlStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
        }
        iCount++;
      }

      st.delete();
      st = null;

      st        = new iqstratControlListStruct();
      st.stItem = new iqstratControlStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratControlStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Location from the Table
   * @param  st = Control Data List Structure
   * @param  sAPI = The API-Number of the Location to be removed.
   *                To remove Location from Table List only set sAPI = null.
   * @param  sKID = Well Header KID or KGS Primary Key
   * @param  sKEY = Primary Key of data defined by user
   * @return st   = Control Data List Structure
   */

  public static iqstratControlListStruct remove( iqstratControlListStruct st,
                                                  String sAPI,
                                                  String sKID,
                                                  String sKEY)
  {
    int i=0;
    int iCount = 0;
    int iFound = 0;

    iqstratControlListStruct stTemp = new iqstratControlListStruct();
    stTemp.iCount = 0;

    if (st != null)
    {
      stTemp.stItem = new iqstratControlStruct[st.iCount-1];
      for (i=0; i<st.iCount; i++)
      {
        iFound = 0;
        if (((sAPI.equals(st.stItem[i].stHeader.sAPI)) && (sAPI.length() > 1)) ||
            ((sKID.equals(st.stItem[i].stHeader.sKID)) && (sKID.length() > 1)) ||
            ((sKEY.equals(st.stItem[i].stHeader.sKEY)) && (sKEY.length() > 1)))
        {
          iFound = 1;
        }

        if (iFound != 1)
        {
          stTemp.stItem[iCount] = new iqstratControlStruct();
          stTemp.stItem[iCount] = copy(st.stItem[i]);
          iCount++;
        }
      }

      st.delete();
      st = null;

      st        = new iqstratControlListStruct();
      st.stItem = new iqstratControlStruct[iCount];
      st.iCount = iCount;

      for (i=0; i<iCount; i++)
      {
        st.stItem[i] = new iqstratControlStruct();
        st.stItem[i] = copy(stTemp.stItem[i]);
      }

      stTemp.delete();
      stTemp = null;
    }

    return (st);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = Cross Section Control list structure
   * @return st = sorted Cross Section Control list structure
   */

  public static iqstratControlListStruct bubbleSort(
      iqstratControlListStruct st )
  {
    boolean swappedOnPrevRun     = true;
    iqstratControlStruct stTemp = null;
    double data1    = 0.0;
    double data2    = 0.0;
    int    iOrderBy = _LATITUDE;
    double dLatMax  = 0.0;
    double dLatMin  = 0.0;
    double dLongMax = 0.0;
    double dLongMin = 0.0;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        for(int i=0; i<st.iCount-1; i++)
        {
          if (i==0)
          {
            dLatMax  = st.stItem[i].stHeader.dLatitude;
            dLatMin  = st.stItem[i].stHeader.dLatitude;
            dLongMax = Math.abs( st.stItem[i].stHeader.dLongitude );
            dLongMin = Math.abs( st.stItem[i].stHeader.dLongitude );
          }

          if (dLatMin > st.stItem[i].stHeader.dLatitude)
            dLatMin  = st.stItem[i].stHeader.dLatitude;

          if (dLatMax < st.stItem[i].stHeader.dLatitude)
            dLatMax  = st.stItem[i].stHeader.dLatitude;

          if (dLongMin > Math.abs( st.stItem[i].stHeader.dLongitude ))
            dLongMin  = Math.abs( st.stItem[i].stHeader.dLongitude );

          if (dLongMax < Math.abs( st.stItem[i].stHeader.dLongitude ))
            dLongMax  = Math.abs( st.stItem[i].stHeader.dLongitude );
        }

        if (Math.abs(dLongMax-dLongMin) > (dLatMax-dLatMin))
          iOrderBy = _LONGITUDE;
        else
          iOrderBy = _LATITUDE;
      }
    }

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            switch (iOrderBy)
            {
              case _LATITUDE:
                data1 = st.stItem[i].stHeader.dLatitude;
                data2 = st.stItem[i+1].stHeader.dLatitude;
                break;

              case _LONGITUDE:
                data1 = st.stItem[i].stHeader.dLongitude;
                data2 = st.stItem[i+1].stHeader.dLongitude;
                break;
            }

            // if current element is greater than the next swap the two elements

            if (data1 > data2)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method getElevation()
   * <p> This method will retrieve the elevation for a specific well/outcrop
   * @param  st    = The cross section control data structure
   * @return dElev = The Elevation Depth
   */

  public static double getElevation( iqstratControlStruct st )
  {
    double dElev = 0.0;

    if (st != null)
    {
      if (st.stHeader != null)
      {
        if (st.stHeader.dGL > 0.0)
          dElev = st.stHeader.dGL;
        else if (st.stHeader.dKB > 0.0)
          dElev = st.stHeader.dKB;
        else
          dElev = st.stHeader.dDF;
      }
    }

    return (dElev);
  }

  /** Method print(iqstratHeadersStruct st)
   * <p> this method will print out the headers informaiton data structure
   * @param st = Header Information Data Structure
   */

  public static void print( iqstratControlStruct st )
  {
    if (st != null)
    {
      System.out.println("--------------------------------------------------");

      iqstrat.iqstratHeadersUtility.print( st.stHeader );

      System.out.println(
        "Depth Range: " + st.depthStart + " " + st.depthEnd );

      System.out.println( "PANEL: " + st.iPanel +
                          " CONFIGURATION: " + st.iConfig);

      if (st.iTracks != null)
      {
        if (st.iTracks.length > 0)
        {
          for (int i=0; i<st.iTracks.length; i++)
          {
            if (st.iTracks[i] != -1)
              System.out.println(i+" "+st.iTracks[i]+" "+
                  iqstrat.iqstratTracksStruct.TRACK[st.iTracks[i]][2] + "\n");
          }
        }

        if (st.iThin != iqstrat.iqstratControlStruct._NO)
          System.out.println(st.iTHINPHI);

        // -- Porosity Image Track

        if (st.iPHI != iqstrat.iqstratControlStruct._NO)
          System.out.println(st.dPHIMin + " " + st.dPHIMax + " "+
                             las.lasStandardTools.LAS_TOOLS[st.iPHITool][2]);

        // -- Resistivity Image Track

        if (st.iOHM != iqstrat.iqstratControlStruct._NO)
          System.out.println(st.dOHMMin + " " + st.dOHMMax + " "+
                             las.lasStandardTools.LAS_TOOLS[st.iOHMTool][2]);
      }
    }
  }
}