/*
 * @iqstratNotesCSVFile.java Version 1.1 07/22/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.io;

import java.awt.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import util.utilFileIO;
import cmn.cmnString;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.iqstratRemarkStruct;

/** Class iqstratNotesCSVFile
 *  <p> This Class is designed to open an ascii text CSV file and read
 *      the contents and parse the data.
 *
 *  @version 1.1 07/22/2008
 *  @author  John R. Victorine
 */

public class iqstratNotesCSVFile
{
  private utilFileIO  pFILE    = null; // Utility File IO Object

  private int         iRows    = 0;    // Number of Rows
  private String      sText[]  = null; // File Contents

  private int         iError   = util.utilFileIO.NOERROR;
  private String      sError   = new String("");

  /** Construct iqstratNotesCSVFile()
   *  <p> This is the Constructor for this class.
   */

  public iqstratNotesCSVFile() { }

  /* ====================================================================== *
   * ---------------------------- FILE METHODS ---------------------------- *
   * ====================================================================== */

  /** Method Open()
    * <p> This is the Constructor for this class.
    * @param iFileID    = To access the file as URL or FILE.
    * @param sDirectory = File URL or Directory Path
    * @param sFilename  = File Name
    */

  public void Open(int iFileID, String sDirectory, String sFilename)
  {
    pFILE   = new utilFileIO();
    pFILE.Open(iFileID, util.utilFileIO.READ, sDirectory, sFilename);

    iError = pFILE.getErrorID();
    sError = pFILE.getError();
  }

  /** Method Close()
   *  <p> This method closes the File.
   */

  public void Close() { pFILE.Close(); }

  /** Method delete()
   * <p> This method will set all the structures to null to force java to free
   *     the allocated memory
   */

  public void delete()
  {
    sText  = null;
    sError = null;

    if (pFILE != null)
      pFILE.delete();
    pFILE = null;
  }

  /* ====================================================================== *
   * -------------------------- PARSE FILE METHODS ------------------------ *
   * ====================================================================== */

  /** Method count()
   * <p> This method will count the number of rows in the file
   */

  public void count()
  {
    iRows = 0;

    while (!pFILE.Next().equals(util.utilFileIO.EOF))
    {
      iError = pFILE.getErrorID();
      sError = pFILE.getError();

      if (iError == util.utilFileIO.NOERROR)
        iRows++;
    }
  }

  /** Method parseData()
   * <p> This method will parse the Notes File into Depth & Text
   */

  public void parseData()
  {
    int iCount = 0;

    sText  = null;

    if (iRows > 0)
    {
      sText = new String[iRows];

      while (!pFILE.Next().equals(util.utilFileIO.EOF))
      {
        iError = pFILE.getErrorID();
        sError = pFILE.getError();

        if (iError == util.utilFileIO.NOERROR)
        {
          if (iCount < iRows)
          {
            sText[iCount] = new String(pFILE.getString());
            iCount++;
          }
        }
      }
    }
  }

  /* ====================================================================== *
   * ----------------------------- MISC METHODS --------------------------- *
   * ====================================================================== */

  /** Method isBlankLine()
   * <p> This method will determine if the line is a blank line
   * @param  sCheck = the string to be checked
   * @return bValue = true if it is blank and false if is in not
   */

  public static boolean isBlankLine(String sCheck[])
  {
    boolean bValue = false;
    int     len    = sCheck.length;

    if (len == 1)  if (sCheck[0].length() == 0) bValue = true;

    return (bValue);
  }

  /** Method computeDepth()
   * <p> This method will compute a decimal from a number and fraction pair
   * @param  sData   = the data strings
   * @return decimal = the decimal number
   */

  public static double computeDepth(String sData[])
  {
    double decimal  = 0.0;
    String frac[]   = null;
    double dtemp1   = 0.0;
    double dtemp2   = 0.0;
    int    itemp    = 0;

    if (sData != null)
    {
      for (int j=0; j<sData.length; j++)
      {
        if (cmn.cmnString.isNumeric(sData[j]))
          decimal = cmn.cmnString.stringToDouble(sData[j]);
        else
        {
          frac  = sData[j].split("[/]+");

          if (frac != null)
          {
            if (frac.length > 0)
            {
              if (cmn.cmnString.isNumeric(frac[0]))
              {
                dtemp1 = cmn.cmnString.stringToDouble(frac[0]);
                if (cmn.cmnString.isNumeric(frac[1]))
                {
                  dtemp2 = cmn.cmnString.stringToDouble(frac[1]);
                  decimal  = decimal + dtemp1 / dtemp2;
                }
              }
            }  // END frac.length > 0
          }  // END frac != null
        }
      }
    }

    return (decimal);
  }

  /* ====================================================================== *
   * ----------------------------- GET METHODS ---------------------------- *
   * ====================================================================== */

  /** Method getDepthString()
   * <p> This method will convert any character that is not a number or '\'
   *     to a empty charactoer
   * @param  sOld = The string to be converted
   * @return sNew = The text being converted
   */

  public static String getDepthString(String sOld)
  {
    String sNew = new String("");
    char   c[];

    if (sOld != null)
    {
      c = sOld.toCharArray();

      for (int i=0; i<c.length; i++)
      {
        if (((c[i]>='0') && (c[i]<='9')) ||
            (c[i]=='/') || (c[i]=='.') || (c[i]==' '))
        {
          if ((c[i]=='0') || (c[i]=='1') || (c[i]=='2') || (c[i]=='3') ||
              (c[i]=='4') || (c[i]=='5') || (c[i]=='6') || (c[i]=='7') ||
              (c[i]=='8') || (c[i]=='9') || (c[i]=='/') || (c[i]==' ') ||
              (c[i]=='.') )
          {
            sNew = new String(sNew + c[i]);
          }
        }
        else
          sNew = new String(sNew + ' ');
      }
    }

    sNew = new String(sNew.trim());

    return (sNew);
  }

  /** Method getDepthRange()
   * <p> This method will parse the depth range from the text line.
   *     This assumes that there is up to two number fields and it is easily
   *     separated from the rest of the text string.  Assume that the first
   *     two string of three are depth field strings or the first string of
   *     two is a depth field string.
   * @param  sData      = array of strings that had been separated by
   *                      the delimiter.
   * @return dRange = depth range
   */

  public static double[] getDepthRange(String sData[])
  {
    double dRange[] = { 0.0, 0.0 };
    int    len      = sData.length;
    int    len1     = 0;
    int    len2     = 0;
    int    iNumber  = 0;
    String nos[]    = null;

    // Only the first term is a depth field the second depth does not exist
    if (len == 2)
    {
      iNumber = 1;
    }
    // There are two depth terms detected and the first two string terms
    // are the depth range terms
//    else if (len == 3)
    else if (len > 2)
    {
      iNumber = 2;
    }

    for (int i=0; i<iNumber; i++)
    {
      sData[i] = new String(getDepthString(sData[i]));
      if (cmn.cmnString.isNumeric(sData[i]))
        dRange[i] = cmn.cmnString.stringToDouble(sData[i]);

      nos = sData[i].split("[ ]+");

      if (nos != null)
        dRange[i] = iqstrat.io.iqstratNotesCSVFile.computeDepth(nos);

      nos = null;
    }

    if (iNumber == 1) dRange[1] = dRange[0];

    return (dRange);
  }

  /** Method getBeddingThickness()
   * <p> This method will parse the bedding thickness from the text line.
   *     This assumes that there is only one number field and it is easily
   *     separated from the the rest of the text string if it is embedded in
   *     the text string.  This also assumes that there is a description
   *     with the bedding thickness and it is larger in length than the
   *     bedding thickness string.
   *
   * @param  sData      = array of strings that had been separated by
   *                      the delimiter.
   * @return dThickness = the bedding thickness
   */

  public static double getBeddingThickness(String sData[])
  {
    double dThickness = 0.0;
    int    len        = sData.length;
    int    len1       = 0;
    int    len2       = 0;
    int    iFound     = -1;
    String nos[]      = null;

    /*  Assume that there is a bedding thickness and a description of the
     *  data and the bedding thickness will be smaller than the description.
     */

    if ( len == 2 )
    {
      len1 = sData[0].length();
      len2 = sData[1].length();

      if ( len1 > len2 )
        iFound = 1;
      else
        iFound = 0;

      if (iFound > -1)
      {
        sData[iFound] = new String(getDepthString(sData[iFound]));
        if (cmn.cmnString.isNumeric(sData[iFound]))
          dThickness = cmn.cmnString.stringToDouble(sData[iFound]);

        nos = sData[iFound].split("[ ]+");

        if (nos != null)
          dThickness = iqstrat.io.iqstratNotesCSVFile.computeDepth(nos);
      }
    }

    return (dThickness);
  }

  /** METHOD getCount()
   *  <p> This method will return the number of rows in file.
   *  @return iRows = the number of rows in file.
   */

  public int getCount() { return ( iRows );  }  // Number of Rows

  /** METHOD getContents()
   *  <p> This method will return the contents of file.
   *  @return sText = the contents of file.
   */

  public String[]  getContents() { return ( sText ); } // File Contents

  /* ====================================================================== *
   * -------------------------- GET ERROR METHODS ------------------------- *
   * ====================================================================== */

  /** METHOD getErrorID()
   *  <p> This method will return the error number.
   *  @return iError = Error Indicator
   */

  public int getErrorID() { return (iError); }

  /** METHOD getError()
   *  <p> This method will return the error string.
   *  @return sError = Error String
   */

  public String getError() { return (sError); }
}

/*
 *  @version 1.1 07/22/2008
 *  @author  John Victorine
 */
