/*
 * @iqstratIOHeaders.java Version 1.0 10/07/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.io;

import java.io.*;
import java.awt.*;
import javax.swing.*;

import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratHeadersListStruct;
import iqstrat.io.ReadHeadersXMLFile;
import iqstrat.io.WriteHeadersXMLFile;
import iqstrat.io.ReadErrorXMLFile;

import iqstrat.iqstratFilesStruct;
import iqstrat.iqstratFilesListStruct;
//import iqstrat.io.WriteFilesXMLFile;
import iqstrat.io.ReadFilesXMLFile;

/** Class iqstratIOHeaders
 *  <p> This class will manage the headers Information Data
 *
 *  @version 1.1 10/07/2007
 *  @author  John R. Victorine
 */

public class iqstratIOHeaders
{
  public static final String HEADERS_XML = "headers.xml";
  public static final String KGS_DATA    = "header/";
  public static final String KGS_PRE     = "header_";

  public static final String PLOT        = "Profile.html";
  public static final String PLOT_PNG    = "Profile.png";

  public static final String XML_FILE_TYPE = "XML";
  public static final String CSV_FILE_TYPE = "CSV";
  public static final String LAS_FILE_TYPE = "LAS";

  // Identifier for modifying data in Headers database table

  public static final int NO  = 0;
  public static final int YES = 1;

  // Identifier for access to data

  public static final int USER_ONLY   = 0;
  public static final int GROUP_READ  = 1;
  public static final int GROUP_WRITE = 2;

  // Identifiers for saving IQSTRAT Data

  public static final int IGNORE  = -1;
  public static final int USER_PC = 0;
  public static final int KGS_DB  = 1;
  public static final int BOTH    = 2;

  /*========================================================================*
   * ------------------------- DIRECTORY METHODS ---------------------------*
   *========================================================================*/

  /** Method exists()
   * <p> This method will check to see if the headers Information XML File exists.
   * @return bValue = true if file exists and false if it does not
   */

  public static boolean exists()
  {
    boolean bValue = false;

    try
    {
      File p = new File( iqstrat.io.iqstratIO.read() );

      if (p.exists())
      {
        try
        {
          File pNew = new File(p, HEADERS_XML);
          if (pNew.exists())
          {
            bValue = true;
          }
        }
        catch (Exception e)
        {
          e.printStackTrace();
          String sError = new String("iqstratIOHeaders().exists-2\n " +
                                     e.toString());
          JOptionPane.showMessageDialog(
              (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String("iqstratIOHeaders().exists\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
    }

    return (bValue);
  }

  /** Method mkDir()
   * <p> This method will make the IQSTRAT Data Directory in the user home
   *     directory and create a xml file of the location.
   * @param  sKID  = Well Header KID or KGS Primary Key
   * @param  sKEY  = Primary Key of data defined by user
   * @param  sAPI  = The API-Number of the Location to be removed.
   *                 To remove Location from Table List only set sAPI = null.
   * @param  sName = The Name of well or location
   * @return sKEY  = Primary Key of data defined by user
   */

  public static String mkDir(String sKID, String sKEY, String sAPI, String sName)
  {
    String sVal = "P";
    iqstratHeadersListStruct st = null; // headers Information Data Structure

    try
    {
      File p = new File( iqstrat.io.iqstratIO.read() );

      if (p.exists())
      {
        st = read_pc();
        sKEY = iqstrat.iqstratHeadersUtility.exists(st, sAPI, sKID, sKEY, sName);

        if (sKEY.equals("0"))
        {
          try
          {
            sKEY = new String(sVal+cmn.cmnString.UniqueName());

            File pNew = new File(p, sKEY);
            if (!pNew.exists())
            {
              pNew.mkdir();
            }
          }
          catch (Exception e)
          {
            e.printStackTrace();
            String sError = new String("iqstratIOHeaders().mkDir-2\n " +
                                       e.toString());
            JOptionPane.showMessageDialog(
                (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
          }
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String("iqstratIOHeaders().mkDir\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
    }

    return (sKEY);
  }

  /** Method isPlot( String sKEY )
   * <p> This method will check to see if the Cross Section HTML File exists.
   * @param  sKEY  = The Primary Key of project Data
   * @return bValue = true if file exists and false if it does not
   */

  public static boolean isPlot( String sKEY )
  {
    boolean bValue = false;

    try
    {
      File p = new File( getPath( sKEY ) );

      if (p.exists())
      {
        try
        {
          File pNew = new File( p, PLOT );
          if (pNew.exists())
          {
            bValue = true;
          }
        }
        catch (Exception e)
        {
          e.printStackTrace();
          String sError = new String( "iqstratIOHeaders().isPlot-2\n " +
                                      e.toString());
          JOptionPane.showMessageDialog(
              (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE );
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String( "iqstratIOHeaders().isPlot\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE );
    }

    return ( bValue );
  }

  /*========================================================================*
   * ---------------------------- GET METHODS ------------------------------*
   *========================================================================*/

  /** Method getPath()
   * <p> This method will check to see if the headers Information XML File exists.
   * @param  sKEY  = Primary Key of data defined by user
   * @return sPath = The path to the Header Record
   */

  public static String getPath(String sKEY)
  {
    String sPath = "";

    if (sKEY != null)
    {
      try
      {
        File p = new File(iqstrat.io.iqstratIO.read());

        if (p.exists())
        {
          if (!sKEY.equals("0"))
          {
            try
            {
              File pNew = new File(p, sKEY);
              if (pNew.exists())
              {
                sPath = new String( pNew.getCanonicalPath() );
              }
            }
            catch (Exception e)
            {
              e.printStackTrace();
              String sError = new String("iqstratIOHeaders().getPath-2\n " +
                                         e.toString());
              JOptionPane.showMessageDialog(
                  (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
            }
          }
        }
      }
      catch (Exception e)
      {
        e.printStackTrace();
        String sError = new String("iqstratIOHeaders().getPath\n " +
                                   e.toString());
        JOptionPane.showMessageDialog(
            (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
      }
    }

    return (sPath);
  }

  /** Method getData()
   * <p> This method will read the User information xml data
   * @param  iMethod = The Method used in retrieving the XML
   * @param  sPath   = The url or File Path to use
   * @return stUser  = The User Data Structure
   */

  public static iqstratHeadersListStruct getData(int iMethod, String sPath)
  {
    iqstratHeadersListStruct st = null; // headers Information Data Structure

    ReadHeadersXMLFile pLocInfo = new ReadHeadersXMLFile(iMethod);
    st = pLocInfo.Process(sPath);

    String sError = pLocInfo.GetError();
    if (sError.length() > 0)
      JOptionPane.showMessageDialog( (Component)null,
                                     sError,
                                     "ERROR",
                                     JOptionPane.ERROR_MESSAGE);

    return (st);
  }

  /*========================================================================*
   * --------------------------- READ METHODS ------------------------------*
   *========================================================================*/

  /** Method read()
   * <p> This method will read the Headers Information XML List
   * @param  iSource = the source of the data
   * @param  sUser   = The User Primary Key
   * @return st      = The Headers Information List Data Structure
   */

  public static iqstratHeadersListStruct read(int iSource, String sUser)
  {
    iqstratHeadersListStruct st = null; // User Data Structure

    switch (iSource)
    {
      case USER_PC:
        st = read_pc();
        break;

      case KGS_DB:
        st = read_db(sUser);
        break;
    }

    return (st);
  }

  /** Method read_record()
   * <p> This method will read the Headers Information XML List
   * @param  iSource = the source of the data
   * @param  sKID    = The Primary Key of Data Record in KGS Database
   * @param  sKEY    = The User created Primary Key for reference only
   * @return st      = The Headers Information Data Structure
   */

  public static iqstratHeadersStruct read_record(
      int iSource, String sKID, String sKEY)
  {
    iqstratHeadersListStruct stList = null; // User Data List Structure
    iqstratHeadersStruct     st     = null; // User Data Structure

    switch (iSource)
    {
      case USER_PC:
        stList = read_pcRecord(sKEY);
        break;

      case KGS_DB:
        stList = read_dbRecord(sKID, sKEY);
        break;
    }

    if (stList != null)
    {
      if (stList.iCount > 0)
      {
        st = iqstrat.iqstratHeadersUtility.copy(stList.stItem[0]);
      }
    }

    return (st);
  }

  /** Method read_pc()
   * <p> This method will read the Headers Information XML File that is on
   *     the User's PC.
   * @return st = The Headers Information List Data Structure
   */

  public static iqstratHeadersListStruct read_pc()
  {
    iqstratHeadersListStruct st = null; // User Data Structure

    String sDir  = iqstrat.io.iqstratIO.read();  // Retrieve the Home Directory
    String sPath = sDir+"\\"+HEADERS_XML;        // Build Path to Login XML

    if (exists())
      st = getData(iqstrat.io.ReadHeadersXMLFile.FILE, sPath);

    return (st);
  }

  /** Method read_db()
   * <p> This method will read the headers Information XML File that is on
   *     the Database.
   * @param  sUser = The User Primary Key
   * @return st    = The headers Information List Data Structure
   */

  public static iqstratHeadersListStruct read_db(String sUser)
  {
    iqstratHeadersListStruct st = null; // headers Information List Data Structure

    String sPath = cmn.cmnStruct.KGS_HEADERS_READ_LIST + "sUser="+sUser;
    st = getData(iqstrat.io.ReadHeadersXMLFile.URL, sPath);

    return (st);
  }

  /** Method read_pcRecord()
   * <p> This method will read the single headers record
   * @param  sKEY = The User created Primary Key for reference only
   * @return st   = The headers Information Data Structure
   */

  public static iqstratHeadersListStruct read_pcRecord(String sKEY)
  {
    iqstratHeadersListStruct st = null; // headers Information List Data Structure

    String sDir = getPath(sKEY);  // Retrieve the Data Directory
    if (!sDir.equals(""))
    {
      String sPath = sDir + "\\" + HEADERS_XML; // Build Path to Login XML

      st = getData(iqstrat.io.ReadHeadersXMLFile.FILE, sPath);
    }

    return (st);
  }

  /** Method read_dbRecord()
   * <p> This method will read a single record in the headers record
   * @param  sKID = The Primary Key of Data Record in KGS Database
   * @param  sKEY = The User created Primary Key for reference only
   * @return st   = The headers Information List Data Structure
   */

  public static iqstratHeadersListStruct read_dbRecord(String sKID, String sKEY)
  {
    iqstratHeadersListStruct st = null; // headers Information List Data Structure

    String sPath = cmn.cmnStruct.KGS_HEADERS_READ +
        "sKID="+sKID+"&sKEY="+sKEY+"&sError=";
    st = getData(iqstrat.io.ReadHeadersXMLFile.URL, sPath);

    return (st);
  }

  /*========================================================================*
   * --------------------------- WRITE METHODS -----------------------------*
   *========================================================================*/

  /** Method save()
   * <p> This method will force a creation of the IQSTRAT Directory & XML
   * @param iSave   = identifier to save data to right location.
   * @param iModify = Indicator to modify the Headers Table Information
   * @param iAccess = User Access to data
   *                  0 = User Only - Read and Write Data in KGS Database
   *                  1 = Group     - Read Only Data in KGS Database
   *                  2 = Group     - Read and Write Data in KGS Database
   * @param sUser   = User Primary Key
   * @param st      = The headers Information Data Structure
   * @return st     = The headers Information Data Structure
   */

  public static iqstratHeadersStruct save(int iSave,
                                          int iModify,
                                          int iAccess,
                                          String sUser,
                                          iqstratHeadersStruct st)
  {
    String sKEY = "0";
    iqstratHeadersListStruct stList = read_pc();

    sKEY = iqstrat.iqstratHeadersUtility.exists(stList,
                                                st.sAPI,
                                                st.sKID,
                                                st.sKEY,
                                                st.sName);
    st.sKEY = new String(sKEY);

    iqstrat.io.iqstratIO.save(); // Create IQSTRAT Directory & XML

    // IF the User is to Save to their PC THEN

    if ((iSave == USER_PC) || (iSave == BOTH))
    {
      if (sKEY.equals("0"))
        iModify = YES;

      // First create or modify the single headers record to the PC

      st.sKEY = mkDir(st.sKID, st.sKEY, st.sAPI, st.sName);

      if (iModify == YES)
        if (!st.sKEY.equals("0"))
          write_pc_record(st);

      // Read Headers List XML File

      stList = read_pc();

      // Add Headers Data Struct to Headers List

      stList = iqstrat.iqstratHeadersUtility.add(stList, st);

      if (iModify == YES)
        stList = iqstrat.iqstratHeadersUtility.modify( stList, st);

      // Write the Headers List to the User PC as XML

      if (iModify == YES)
        write_pc(stList);
    }

    // IF the user is to Save to the KGS Database THEN

    if ((iSave == KGS_DB) || (iSave == BOTH))
    {
      // Write Headers Data to the KGS Database

//      if (iModify == YES)
        st = write_db(iModify, iAccess, sUser, st);
    }

    return (st);
  }

  /** Method write_pc()
   * <p> This method will write the headers Info XML File to the User's PC at
   *     the top level IQSTRAT_DATA Directory.
   * @param st = The headers Information List Data Structure
   */

  public static void write_pc(iqstratHeadersListStruct st)
  {
    String sDir = iqstrat.io.iqstratIO.read();  // Retrieve the Data Directory

    if (st != null)
    {
      WriteHeadersXMLFile pXML = new WriteHeadersXMLFile();
      pXML.write(sDir, HEADERS_XML, pXML.buildXML(st));
    }
  }

  /** Method write_pc()
   * <p> This method will write the headers Info XML File to the User's PC
   *     under the IQSTRAT_DATA Directory under the single header data
   *     directory.
   * @param st = The headers Information Data Structure
   */

  public static void write_pc_record(iqstratHeadersStruct st)
  {
    String sDir = getPath(st.sKEY);  // Retrieve the Data Directory

    iqstratHeadersListStruct stList = new iqstratHeadersListStruct();

    if (st != null)
    {
      stList.stItem    = new iqstratHeadersStruct[1];
      stList.stItem[0] = iqstrat.iqstratHeadersUtility.copy(st);
      stList.iCount    = 1;

      WriteHeadersXMLFile pXML = new WriteHeadersXMLFile();
      pXML.write(sDir, HEADERS_XML, pXML.buildXML(stList));
    }
  }

  /** Method write_db()
   * <p> This method will write the headers Info XML File to the KGS DB.
   * @param iModify = Indicator to modify the Headers Table Information
   * @param iAccess = User Access to data
   *                  0 = User Only - Read and Write Data in KGS Database
   *                  1 = Group     - Read Only Data in KGS Database
   *                  2 = Group     - Read and Write Data in KGS Database
   * @param sUser   = User Primary Key
   * @param st      = The headers Information Data Structure
   * @return st     = The headers Information Data Structure
   */

  public static iqstratHeadersStruct write_db(int iModify,
                                              int iAccess,
                                              String sUser,
                                              iqstratHeadersStruct st)
  {
    String sError = "";
    iqstratHeadersListStruct stList = new iqstratHeadersListStruct();

    if (st != null)
    {
      stList.stItem    = new iqstratHeadersStruct[1];
      stList.stItem[0] = iqstrat.iqstratHeadersUtility.copy(st);
      stList.iCount    = 1;

      // Open the Headers Write XML File Class

      WriteHeadersXMLFile pXML = new WriteHeadersXMLFile();

      // Create Path to Save Headers XML Data

      String sPath = cmn.cmnStruct.KGS_HEADERS_WRITE +
          "iModify=" + iModify + "&" +
          "iAccess=" + iAccess + "&" +
          "sUser="   + sUser   + "&" +
//          "v_xml="+pXML.createDB_XML(stList);
      "v_xml="+cmn.cmnString.convertBlankTo_Chars(pXML.createDB_XML(stList));
//          "v_xml="+cmn.cmnString.convertCharsToHTMLChars(pXML.buildXML(stList));
System.out.println(sPath);

      // User Read Error XML File to read errors generated during save

      stList = getData(iqstrat.io.ReadHeadersXMLFile.URL, sPath);

      if (stList != null)
      {
        if (stList.iCount > 0)
        {
          if (stList.stItem[0].sError.length() > 0)
          {
            JOptionPane.showMessageDialog( (Component)null,
                                          sError,
                                          "ERROR",
                                          JOptionPane.ERROR_MESSAGE);
          }
          else
          {
            st.sKID = new String(stList.stItem[0].sKID);
            JOptionPane.showMessageDialog( (Component)null,
                                          "Headers Data "+st.sKID+" Saved",
                                          "SUCCESS",
                                          JOptionPane.INFORMATION_MESSAGE);
          }
        }
        else
        {
          JOptionPane.showMessageDialog( (Component)null,
                                        "Problem saving data? "+
                                        stList.iCount+ " records found",
                                        "ERROR",
                                        JOptionPane.ERROR_MESSAGE);
        }
      }
      else
      {
        JOptionPane.showMessageDialog( (Component)null,
                                      "Problem saving data?",
                                      "ERROR",
                                      JOptionPane.ERROR_MESSAGE);
      }
    }
//iqstrat.iqstratHeadersUtility.print(st);

    return (st);
  }

  /*========================================================================*
   *--------------- READ & WRITE XML FILES On MORDER METHODS ---------------*
   *========================================================================*/

  /** Method read_file()
   * <p> This method will read a User File Table record from the database
   * @param  sUser = The user primary key
   * @param  sKID  = The user headers primary key
   * @param  sType = The type of data to retrieve
   * @return st = the user file list data structure
   */

  public static iqstratFilesListStruct read_file(String sUser,
                                                 String sKID,
                                                 String sType)
  {
    iqstratFilesListStruct st = null; // user files Information Data Structure

    ReadFilesXMLFile pXML = new ReadFilesXMLFile(iqstrat.io.ReadFilesXMLFile.URL);

    String sPath = cmn.cmnStruct.KGS_USER_FILES_READ +
                     "sUser="    + sUser + "&" +
                     "sHeaders=" + sKID  + "&" +
                     "sType="    + sType;

    st = pXML.Process(sPath);

    return (st);
  }

  /** Method write_files()
   * <p> This method will write a record to the Users Files Table so the XML
   *     File can be retrieved.
   * @param sUser     = The User Primary Key
   * @param sHeader   = The Header Primary Key
   * @param iRecords  = Number of Records
   * @param sType     = The Data Type of File
   * @param sFile     = The File Type
   * @param sURL      = URL File Path
   * @param sDir      = Directory Path
   * @param sFilename = Filename
   */

  public static void write_files(String sUser,
                                 String sHeader,
                                 int    iRecords,
                                 String sType,
                                 String sFile,
                                 String sURL,
                                 String sDir,
                                 String sFilename)
  {
    iqstratFilesListStruct st = null; // user files Information Data Structure

    ReadFilesXMLFile pXML = new ReadFilesXMLFile(iqstrat.io.ReadFilesXMLFile.URL);

    String sPath = cmn.cmnStruct.KGS_USER_FILES_WRITE +
                     "o_user="    + sUser   + "&" +
                     "o_headers=" + sHeader + "&" +
                     "o_records=" + iRecords + "&" +
                     "o_type="    + sType   + "&" +
                     "o_file="    + sFile   + "&" +
                     "o_url="     + sURL    + "&" +
                     "o_path="    + sDir    + "&" +
                     "o_name="    + sFilename;
//System.out.println(sPath);

    st = pXML.Process(sPath);

    if (pXML.GetErrorID() == iqstrat.io.ReadFilesXMLFile.ERROR)
    {
      JOptionPane.showMessageDialog( (Component)null,
                                      pXML.GetError(),
                                      "ERROR",
                                      JOptionPane.ERROR_MESSAGE);
    }
  }

  /*========================================================================*
   * --------------------------- DELETE METHODS ----------------------------*
   *========================================================================*/

  /** Method deleteDir()
   * <p> This method will delete the Data Directory in the user home
   *     directory.
   * @param  sKEY  = Primary Key of data defined by user
   */

  public static void deleteDir( String sKEY )
  {
    try
    {
      File p = new File( iqstrat.io.iqstratIO.read() );

      if (p.exists())
      {
        if (!sKEY.equals("0"))
        {
          try
          {
            File pNew = new File(p, sKEY);
            if (pNew.exists())
            {
              pNew.delete();
            }
          }
          catch (Exception e)
          {
            e.printStackTrace();
            String sError = new String("iqstratIOHeaders.deleteDir()-2\n " +
                                       e.toString());
            JOptionPane.showMessageDialog(
                (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
          }
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String("iqstratIOHeaders.deleteDir()\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
    }
  }

  /** Method deletePC()
   * <p> This method will check to see if the headers Information XML File exists.
   *     and delete the file if it does
   */

  public static void deletePC()
  {
    try
    {
      File p = new File( iqstrat.io.iqstratIO.read() );

      if (p.exists())
      {
        try
        {
          File pNew = new File(p, HEADERS_XML);
          if (pNew.exists())
          {
            pNew.delete();
          }
        }
        catch (Exception e)
        {
          e.printStackTrace();
          String sError = new String("iqstratIOHeaders.deletePC()-2\n " +
                                     e.toString());
          JOptionPane.showMessageDialog(
              (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String("iqstratIOHeaders.deletePC()\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
    }
  }

  /** Method deletePC()
   * <p> This method will check to see if the headers Information XML File exists
   *     and remove the file if it does.
   * @param  sKEY  = Primary Key of data defined by user
   */

  public static void deletePC(String sKEY)
  {
    try
    {
      File p = new File( iqstrat.io.iqstratIOHeaders.getPath(sKEY) );

      if (p.exists())
      {
        try
        {
          File pNew = new File(p, HEADERS_XML);
          if (pNew.exists())
          {
            pNew.delete();
          }
        }
        catch (Exception e)
        {
          e.printStackTrace();
          String sError = new String("iqstratIOHeaders.deletePC()-2\n " +
                                     e.toString());
          JOptionPane.showMessageDialog(
              (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
        }
      }
    }
    catch (Exception e)
    {
      e.printStackTrace();
      String sError = new String("iqstratIOHeaders.deletePC()\n " + e.toString());
      JOptionPane.showMessageDialog(
          (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
    }

    if ( isPlot( sKEY ) )
    {
      String sDir = getPath( sKEY );

      String sPath = sDir + "\\" + PLOT;
      util.DeleteFile.delete( sPath );
      sPath = sDir + "\\" + PLOT_PNG;
      util.DeleteFile.delete( sPath );
    }
  }

  /** Method deleteDB()
   * <p> This method will delete a well from the User Headers KGS Database
   * @param  sUser = User Primary Key
   * @param  sKEY  = The user headers Information Data Primary Key
   */

  public static void deleteDB(String sUser, String sKEY)
  {
    iqstratFilesListStruct stFile = null;   // User File List Data Structure
    String sPath  = "";
    String sError = "";

    // Delete File from KGS Database.

    ReadErrorXMLFile pXML =
        new ReadErrorXMLFile(iqstrat.io.ReadErrorXMLFile.URL);

    sPath = cmn.cmnStruct.KGS_USER_HEADERS_DELETE +
            "o_user=" + sUser + "&o_kid=" + sKEY;
    sError = pXML.Process(sPath);

    if (sError.length() > 0)
      JOptionPane.showMessageDialog(
         (Component) null, sError, "ERROR", JOptionPane.ERROR_MESSAGE);
  }
}