/*
 * @WriteHeadersXMLFile.java Version 1.1 11/12/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.io;

import java.io.*;
import java.awt.*;

import util.utilFileIO;
import iqstrat.iqstratHeadersListStruct;
import cmn.cmnString;

/** Class WriteHeadersXMLFile
 *  <p> This class will parse the Well Header Information List Structure and
 *      write the information to the Well Header Information XML File in the
 *      selected User's Project Directory.
 *
 *  @version 1.1 11/12/2007
 *  @author  John R. Victorine
 */

public class WriteHeadersXMLFile
{
  public static final String DTD =
    "<?xml version=\"1.0\"?>\n " +
    "<!DOCTYPE headers [\n " +
    "<!ELEMENT headers (data*)>\n"                   +
    "<!ATTLIST headers records CDATA #IMPLIED>\n"    + // Total # of Records
    "<!ELEMENT data (info*, \n"                       +
    "                other?, \n"                      +
    "                loc?, \n"                        +
    "                xy?, \n"                         +
    "                z?, \n"                          +
    "                comments?, \n"                   +
    "                misc?, \n"                       +
    "                cnt?)>\n"                        +
    "<!ELEMENT info  EMPTY>\n"                        +
    "<!ATTLIST info  kid   CDATA #IMPLIED\n"          + // Primary DB KEY
    "                well_kid CDATA #IMPLIED\n"       + // KGS Primary Key
    "                key   CDATA #IMPLIED\n"          + // Primary KEY Defined
    "                type  CDATA #IMPLIED\n"          + // type of data
    "                api   CDATA #IMPLIED\n"          + // API Number of Well
    "                name   CDATA #IMPLIED\n"         + // Outcrop Name
    "                status CDATA #IMPLIED>\n"        + // Well Status
    "<!ELEMENT other EMPTY>\n"                        +
    "<!ATTLIST other operator  CDATA #IMPLIED\n"      + // Operator Name
    "                oper_kid  CDATA #IMPLIED\n"      + // Operator KID
    "                field     CDATA #IMPLIED\n"      + // Field Name
    "                field_kid CDATA #IMPLIED>\n"     + // Field KID
    "<!ELEMENT loc EMPTY>\n"                          +
    "<!ATTLIST loc state     CDATA #IMPLIED\n"        + // State Name
    "              state_cd  CDATA #IMPLIED\n"        + // State Code
    "              county    CDATA #IMPLIED\n"        + // County Name
    "              county_cd CDATA #IMPLIED\n"        + // County Code
    "              loc       CDATA #IMPLIED\n"        + // Location
    "              town      CDATA #IMPLIED\n"        + // township
    "              town_dir  CDATA #IMPLIED\n"        + // township direction
    "              range     CDATA #IMPLIED\n"        + // range
    "              range_dir CDATA #IMPLIED\n"        + // range direction
    "              section   CDATA #IMPLIED>\n"       + // section
    "<!ELEMENT xy  EMPTY>\n"                          +
    "<!ATTLIST xy  latitude  CDATA #IMPLIED\n"        + // Latitude
    "              longitude CDATA #IMPLIED\n"        + // Longitude
    "              zone      CDATA #IMPLIED\n"        + // UTM X-Position
    "              utm_x     CDATA #IMPLIED\n"        + // UTM X-Position
    "              utm_y     CDATA #IMPLIED>\n"       + // UTM Y-Position
    "<!ELEMENT z   EMPTY>\n"                          +
    "<!ATTLIST z   depth     CDATA #IMPLIED\n"        + // Depth or Height
    "              gl        CDATA #IMPLIED\n"        + // Ground Level
    "              kb        CDATA #IMPLIED\n"        + // Kelly Bushing
    "              df        CDATA #IMPLIED>\n"       + // Derrick Floor
    "<!ELEMENT comments (#PCDATA)>"                   + // Comments
    "<!ELEMENT misc EMPTY>\n"                         +
    "<!ATTLIST misc user      CDATA #IMPLIED\n"       + // user update info
    "               access    CDATA #IMPLIED\n"       + // user access
    "               source    CDATA #IMPLIED\n"       + // user primary code
    "               date      CDATA #IMPLIED>\n"      + // Date modified
    "<!ELEMENT cnt  EMPTY>\n"                         +
    "<!ATTLIST cnt  las       CDATA #IMPLIED\n"       + // LAS Files Exist
    "               tops      CDATA #IMPLIED\n"       + // Tops Exist
    "               core      CDATA #IMPLIED\n"       + // Core Data Exist
    "               images    CDATA #IMPLIED>]>\n";     // Core Images Exist

  private int            iError       = util.utilFileIO.NOERROR;
  private String         sError       = new String("");

  /** Constructor WriteHeadersXMLFile()
   *  <p> This is the constructor for this class
   */

  public WriteHeadersXMLFile() {  }

  /** Method write()
   *  <p> This is the method will write the XML String to the user's PC
   *  @param sPath     = The Directory Path for the XML File
   *  @param sFilename = The Filename of the XML File.
   *  @param sXML      = The Headers XML String
   */

  public void write(String sPath, String sFilename, String sXML)
  {
    utilFileIO pIO  = new utilFileIO();

    // Attach DTD File to XML written to PC

    sXML = new String(DTD+sXML);

    // Open the File for write

    pIO.Open(util.utilFileIO.DIRECTORY,
             util.utilFileIO.WRITE,
             sPath,
             sFilename);

    // Write the XML Block to the File

    pIO.Write(sXML);
    iError = pIO.getErrorID();
    sError = pIO.getError();

    // Close the File

    if (iError == util.utilFileIO.NOERROR)
      pIO.Close();
  }

  /** Method buildXML()
   *  <p> This is the method will build the headers XML string
   *  @param  st      = The Headers Information Data List Structure
   *  @return sXML    = The Headers XML String
   */

  public String buildXML(iqstratHeadersListStruct st)
  {
    // Build the XML Block

    String sXML = new String ("<headers ");

    if (st != null)
    {
      sXML = new String (sXML + "records=\"" + st.iCount + "\">\n");
      for (int i=0; i<st.iCount; i++)
      {
        // New Data Record Start

        sXML = new String(sXML +  " <data>\n");

        // START Headers Information Block

        sXML = new String(sXML +  " <info ");

        // --  KGS Primary KEY

        if (st.stItem[i].sKID.length() > 1)
          sXML = new String(sXML + "kid=\""  + st.stItem[i].sKID  + "\" ");

        if (st.stItem[i].sWELLKID.length() > 1)
          sXML = new String(sXML + "well_kid=\""  + st.stItem[i].sWELLKID  + "\" ");

        // -- User Generated KEY

        sXML = new String(sXML + "key=\""  + st.stItem[i].sKEY  + "\" ");

        // --  Type of Data

        switch (st.stItem[i].iType)
        {
          case iqstrat.iqstratHeadersStruct.WELL:    // Oil & Gas Well Data
            sXML = new String(sXML + "type=\"WELL\" ");
            break;
          case iqstrat.iqstratHeadersStruct.OUTCROP: // Outcrop or Stream cut Data
            sXML = new String(sXML + "type=\"OUTCROP\" ");
            break;
        }

        // --  Well API-Number or UWI

        if (st.stItem[i].sAPI.length() > 0)
          sXML = new String(sXML + "api=\""  + st.stItem[i].sAPI  + "\" ");

        // -- Well Name or Outcrop Name

        if (st.stItem[i].sName.length() > 0)
          sXML = new String(sXML + "name=\"" +
            cmn.cmnString.convertCharsToHTMLChars(st.stItem[i].sName) + "\" ");

        // -- Status of well or Outcrop Type

        if (st.stItem[i].status.length() > 0)
          sXML = new String(sXML + "status=\"" +
            cmn.cmnString.convertCharsToHTMLChars(st.stItem[i].status) + "\"");

        sXML = new String(sXML + "/>\n");

        // END Headers Information Block

        // START Other Well Information Block

        if ((st.stItem[i].sOperator.length() > 0) ||
            (st.stItem[i].sField.length() > 0))
        {
          sXML = new String(sXML + " <other ");

          // -- Operator & Operator Primary KEY

          if (st.stItem[i].sOperator.length() > 0)
            sXML = new String(sXML + "operator=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                                st.stItem[i].sOperator) + "\" ");

          if (st.stItem[i].sOperator_kid.length() > 0)
            sXML = new String(sXML + "oper_kid=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                                st.stItem[i].sOperator_kid) + "\" ");

            // -- Field & Field Primary KEY

          if (st.stItem[i].sField.length() > 0)
            sXML = new String(sXML + "field=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                                st.stItem[i].sField) + "\" ");

          if (st.stItem[i].sField_kid.length() > 0)
            sXML = new String(sXML + "field_kid=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                                st.stItem[i].sField_kid) + "\" ");

          sXML = new String(sXML + "/>\n");
        }

        // END Other Well Information Block

        // START Location Information Block

        if ((st.stItem[i].state.length() > 0) ||
            (st.stItem[i].sCounty.length() > 0) ||
            (st.stItem[i].sLocation.length() > 0) ||
            (st.stItem[i].iTownship > 0) ||
            (st.stItem[i].iRange > 0) ||
            (st.stItem[i].iSection > 0))
        {
          sXML = new String(sXML + " <loc ");

          // -- State Name & Code

          if (st.stItem[i].state.length() > 0)
            sXML = new String(sXML + "state=\"" + st.stItem[i].state + "\" ");

          if (st.stItem[i].iState > -1)
            sXML = new String(sXML +
                              "state_cd=\"" + st.stItem[i].iState + "\" ");

            // -- County Name & Code

          if (st.stItem[i].sCounty.length() > 0)
            sXML = new String(sXML +
                              "county=\"" + st.stItem[i].sCounty + "\" ");

          if (st.stItem[i].iCounty > -1)
            sXML = new String(sXML +
                              "county_cd=\"" + st.stItem[i].iCounty + "\" ");

            // -- location of outcrop or stream cut

          if (st.stItem[i].sLocation.length() > 0)
            sXML = new String(sXML + "loc=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                st.stItem[i].sLocation) + "\" ");

            // -- location - township & direction

          if (st.stItem[i].iTownship > 0) {
            sXML = new String(sXML +
                              "town=\"" + st.stItem[i].iTownship + "\" " +
                              "town_dir=\"" + st.stItem[i].sTownship + "\" ");
          }

          // -- location - range & direction

          if (st.stItem[i].iRange > 0) {
            sXML = new String(sXML +
                              "range=\"" + st.stItem[i].iRange + "\" " +
                              "range_dir=\"" + st.stItem[i].sRange + "\" ");
          }

          // -- location - section

          if (st.stItem[i].iSection > 0) {
            sXML = new String(sXML +
                              "section=\"" + st.stItem[i].iSection + "\" ");

          }

          sXML = new String(sXML + "/>\n");
        }

        // END Location Information Block

        // START xy position Block

        // -- latitude & longitude

        if (((Math.abs(st.stItem[i].dLatitude) > 0.0) &&
             (Math.abs(st.stItem[i].dLongitude) > 0.0)) ||
            ((Math.abs(st.stItem[i].dUTMx) > 0.0) &&
             (Math.abs(st.stItem[i].dUTMy) > 0.0)))
        {
          sXML = new String(sXML + " <xy ");

          if ((Math.abs(st.stItem[i].dLatitude) > 0.0) &&
              (Math.abs(st.stItem[i].dLongitude) > 0.0))
          {
            sXML = new String(sXML +
                              "latitude=\"" + st.stItem[i].dLatitude + "\" " +
                              "longitude=\"" + st.stItem[i].dLongitude + "\" ");
          }

          if ((Math.abs(st.stItem[i].dUTMx) > 0.0) &&
              (Math.abs(st.stItem[i].dUTMy) > 0.0))
          {
            // -- utm coordinates

            sXML = new String(sXML +
                "zone=\""  + st.stItem[i].dZone + "\" " +
                "utm_x=\"" + st.stItem[i].dUTMx    + "\" " +
                "utm_y=\"" + st.stItem[i].dUTMy    + "\"");
          }

          sXML = new String(sXML + "/>\n");
        }

        // END xy position Block

        // START z position Block

        if ((st.stItem[i].depth > 0.0) || (st.stItem[i].dGL > 0.0) ||
            (st.stItem[i].dKB > 0.0) || (st.stItem[i].dKB > 0.0))
        {
          sXML = new String(sXML + " <z ");

          if (st.stItem[i].depth > 0.0)
            sXML = new String(sXML + "depth =\"" + st.stItem[i].depth + "\" ");
          if (st.stItem[i].dGL > 0.0)
            sXML = new String(sXML + "gl=\"" + st.stItem[i].dGL + "\" ");
          if (st.stItem[i].dKB > 0.0)
            sXML = new String(sXML + "kb=\"" + st.stItem[i].dKB + "\" ");
          if (st.stItem[i].dKB > 0.0)
            sXML = new String(sXML + "df=\"" + st.stItem[i].dDF + "\"");

          sXML = new String(sXML + "/>\n");
        }

        // END z position Block

        // START comments Block

        // -- Comments

        if (st.stItem[i].sComments.length() > 0)
        {
          sXML = new String(sXML + " <comments>");
          sXML = new String(sXML +
                 cmn.cmnString.convertCharsToHTMLChars(st.stItem[i].sComments));
          sXML = new String(sXML + "</comments>\n");
        }

        // END comments Block

        // START misc Block

        sXML = new String(sXML + " <misc ");

        // -- User Update Name or initials

        if (st.stItem[i].sUpdate.length() > 0)
            sXML = new String(sXML + "user=\"" +
                              cmn.cmnString.convertCharsToHTMLChars(
                                st.stItem[i].sUpdate) + "\" ");
        // -- Access to the data

        sXML = new String(sXML + "access=\"" + st.stItem[i].iAccess + "\" ");
        sXML = new String(sXML + "source=\"" + st.stItem[i].source + "\" ");

        // -- User Update Date

        if (st.stItem[i].sDate.length() > 0)
            sXML = new String(sXML + "date=\"" + st.stItem[i].sDate + "\" ");

        sXML = new String(sXML + "/>\n");

        // END misc Block

        sXML = new String(sXML + " <cnt ");
        sXML = new String(sXML + "las=\""    + st.stItem[i].iLAS   + "\" ");
        sXML = new String(sXML + "tops=\""   + st.stItem[i].iTops  + "\" ");
        sXML = new String(sXML + "core=\""   + st.stItem[i].iCore  + "\" ");
        sXML = new String(sXML + "images=\"" + st.stItem[i].images + "\"/>\n");

        sXML = new String(sXML +  " </data>\n");

        // END New Data Record Start
      }
    }
    else
      sXML = new String (sXML + "records=\"0\">\n");

    sXML = new String (sXML + "</headers>\n");

    return (sXML);
  }

  /** Method createDB_XML()
   *  <p> This is the method will build the headers XML string
   *  @param  st      = The Headers Information Data List Structure
   *  @return sXML    = The Headers XML String
   */

  public String createDB_XML(iqstratHeadersListStruct st)
  {
    // Build the XML Block

    String sXML = new String ("<headers ");

    if (st != null)
    {
      sXML = new String (sXML + "records=\"" + st.iCount + "\">");
      for (int i=0; i<st.iCount; i++)
      {
        // New Data Record Start

        sXML = new String(sXML +  " <data>");

        // START Headers Information Block

        sXML = new String(sXML +  " <info ");

        // --  KGS Primary KEY

        if (st.stItem[i].sKID.length() > 1)
          sXML = new String(sXML + "kid=\""  + st.stItem[i].sKID  + "\" ");
//        else
//          sXML = new String(sXML + "kid=\"0\" ");

        if (st.stItem[i].sWELLKID.length() > 1)
          sXML = new String(sXML + "well_kid=\""  + st.stItem[i].sWELLKID  + "\" ");
//        else
//          sXML = new String(sXML + "well_kid=\"0\" ");

        // -- User Generated KEY

        sXML = new String(sXML + "key=\""  + st.stItem[i].sKEY  + "\" ");

        // --  Type of Data

        switch (st.stItem[i].iType)
        {
          case iqstrat.iqstratHeadersStruct.WELL:    // Oil & Gas Well Data
            sXML = new String(sXML + "type=\"WELL\" ");
            break;
          case iqstrat.iqstratHeadersStruct.OUTCROP: // Outcrop or Stream cut Data
            sXML = new String(sXML + "type=\"OUTCROP\" ");
            break;
        }

        // --  Well API-Number or UWI

        if (st.stItem[i].sAPI.length() > 0)
          sXML = new String(sXML + "api=\""  + st.stItem[i].sAPI  + "\" ");
//        else
//          sXML = new String(sXML + "api=\"\" ");

        // -- Well Name or Outcrop Name

        if (st.stItem[i].sName.length() > 0)
        {
          sXML = new String(sXML + "name=\"" +
             cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sName) + "\" ");
        }
//        else
//          sXML = new String(sXML + "name=\"\" ");


        // -- Status of well or Outcrop Type

        if (st.stItem[i].status.length() > 0)
        {
          sXML = new String(sXML + "status=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].status) + "\"");
        }
//        else
//          sXML = new String(sXML + "status=\"\" ");

        sXML = new String(sXML + "/>");

        // END Headers Information Block

        // START Other Well Information Block

        sXML = new String(sXML + " <other ");
        if ((st.stItem[i].sOperator.length() > 0) ||
            (st.stItem[i].sField.length() > 0))
        {
          // -- Operator & Operator Primary KEY

          if (st.stItem[i].sOperator.length() > 0)
            sXML = new String(sXML + "operator=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sOperator) + "\" ");
//          else
//            sXML = new String(sXML + "operator=\"\" ");

          if (st.stItem[i].sOperator_kid.length() > 0)
            sXML = new String(sXML + "oper_kid=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sOperator_kid) + "\" ");
//          else
//            sXML = new String(sXML + "oper_kid=\"0\" ");

            // -- Field & Field Primary KEY

          if (st.stItem[i].sField.length() > 0)
            sXML = new String(sXML + "field=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sField) + "\" ");
//          else
//            sXML = new String(sXML + "field=\"\" ");

          if (st.stItem[i].sField_kid.length() > 0)
            sXML = new String(sXML + "field_kid=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sField_kid) + "\" ");
//          else
//            sXML = new String(sXML + "field_kid=\"0\" ");
        }
        sXML = new String(sXML + "/>");

        // END Other Well Information Block

        // START Location Information Block

        sXML = new String(sXML + " <loc ");
        if ((st.stItem[i].state.length() > 0) ||
            (st.stItem[i].sCounty.length() > 0) ||
            (st.stItem[i].sLocation.length() > 0) ||
            (st.stItem[i].iTownship > 0) ||
            (st.stItem[i].iRange > 0) ||
            (st.stItem[i].iSection > 0))
        {
          // -- State Name & Code

          if (st.stItem[i].state.length() > 0)
            sXML = new String(sXML + "state=\"" + st.stItem[i].state + "\" ");
//          else
//            sXML = new String(sXML + "state=\"\" ");

          if (st.stItem[i].iState > -1)
            sXML = new String(sXML +
                              "state_cd=\"" + st.stItem[i].iState + "\" ");
//          else
//            sXML = new String(sXML + "state_cd=\"0\" ");

            // -- County Name & Code

          if (st.stItem[i].sCounty.length() > 0)
            sXML = new String(sXML +
                              "county=\"" + st.stItem[i].sCounty + "\" ");
//          else
//            sXML = new String(sXML + "county=\"\" ");

          if (st.stItem[i].iCounty > -1)
            sXML = new String(sXML +
                              "county_cd=\"" + st.stItem[i].iCounty + "\" ");
//          else
//            sXML = new String(sXML + "county_cd=\"0\" ");

            // -- location of outcrop or stream cut

          if (st.stItem[i].sLocation.length() > 0)
            sXML = new String(sXML + "loc=\"" +
              cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sLocation) + "\" ");
//            else
//              sXML = new String(sXML + "loc=\"\" ");

            // -- location - township & direction

          if (st.stItem[i].iTownship > 0)
          {
            sXML = new String(sXML +
                              "town=\"" + st.stItem[i].iTownship + "\" " +
                              "town_dir=\"" + st.stItem[i].sTownship + "\" ");
          }
//          else
//            sXML = new String(sXML +
//                              "town=\"0\" " +
//                              "town_dir=\"S\" ");

          // -- location - range & direction

          if (st.stItem[i].iRange > 0)
          {
            sXML = new String(sXML +
                              "range=\"" + st.stItem[i].iRange + "\" " +
                              "range_dir=\"" + st.stItem[i].sRange + "\" ");
          }
//          else
//            sXML = new String(sXML +
//                              "range=\"0\" " +
//                              "range_dir=\"W\" ");

          // -- location - section

          if (st.stItem[i].iSection > 0)
          {
            sXML = new String(sXML +
                              "section=\"" + st.stItem[i].iSection + "\" ");
          }
//          else
//            sXML = new String(sXML + "section=\"0\" ");

        }
        sXML = new String(sXML + "/>");

        // END Location Information Block

        // START xy position Block

        // -- latitude & longitude

        sXML = new String(sXML + " <xy ");
        if (((Math.abs(st.stItem[i].dLatitude) > 0.0) &&
             (Math.abs(st.stItem[i].dLongitude) > 0.0)) ||
            ((Math.abs(st.stItem[i].dUTMx) > 0.0) &&
             (Math.abs(st.stItem[i].dUTMy) > 0.0)))
        {
          if ((Math.abs(st.stItem[i].dLatitude) > 0.0) &&
              (Math.abs(st.stItem[i].dLongitude) > 0.0))
          {
            sXML = new String(sXML +
                              "latitude=\"" + st.stItem[i].dLatitude + "\" " +
                              "longitude=\"" + st.stItem[i].dLongitude + "\" ");
          }
//          else
//            sXML = new String(sXML +
//                    "latitude=\"0\" " +
//                    "longitude=\"0\" ");

          if ((Math.abs(st.stItem[i].dUTMx) > 0.0) &&
              (Math.abs(st.stItem[i].dUTMy) > 0.0))
          {
            // -- utm coordinates

            sXML = new String(sXML +
                "zone=\""  + st.stItem[i].dZone + "\" " +
                "utm_x=\"" + st.stItem[i].dUTMx    + "\" " +
                "utm_y=\"" + st.stItem[i].dUTMy    + "\"");
          }
//          else
//            sXML = new String(sXML +
//                "zone=\"0\" " +
//                "utm_x=\"0\" " +
//                "utm_y=\"0\"");
        }
        sXML = new String(sXML + "/>");

        // END xy position Block

        // START z position Block

        sXML = new String(sXML + " <z ");
        if ((st.stItem[i].depth > 0.0) || (st.stItem[i].dGL > 0.0) ||
            (st.stItem[i].dKB > 0.0) || (st.stItem[i].dKB > 0.0))
        {
          if (st.stItem[i].depth > 0.0)
            sXML = new String(sXML + "depth =\"" + st.stItem[i].depth + "\" ");
//          else
//            sXML = new String(sXML + "depth=\"0\" ");

          if (st.stItem[i].dGL > 0.0)
            sXML = new String(sXML + "gl=\"" + st.stItem[i].dGL + "\" ");
//          else
//            sXML = new String(sXML + "gl=\"0\" ");

          if (st.stItem[i].dKB > 0.0)
            sXML = new String(sXML + "kb=\"" + st.stItem[i].dKB + "\" ");
//          else
//            sXML = new String(sXML + "kb=\"0\" ");

          if (st.stItem[i].dKB > 0.0)
            sXML = new String(sXML + "df=\"" + st.stItem[i].dDF + "\"");
//          else
//            sXML = new String(sXML + "df=\"0\" ");
        }
        sXML = new String(sXML + "/>");

        // END z position Block

        // START comments Block

        // -- Comments

        if (st.stItem[i].sComments.length() > 0)
        {
          sXML = new String(sXML + " <comments> ");
//          sXML = new String(sXML +
//          cmn.cmnString.convertCharsToSafeChars(st.stItem[i].sComments) + "\"");
          sXML = new String(sXML +
          cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sComments) + "\" ");
          sXML = new String(sXML + " </comments> ");
        }

        // END comments Block

        // START misc Block

        sXML = new String(sXML + " <misc ");

        // -- User Update Name or initials

        if (st.stItem[i].sUpdate.length() > 0)
          sXML = new String(sXML + "user=\"" +
            cmn.cmnString.convertCharsToDBHTMLChars(st.stItem[i].sUpdate) + "\" ");
//        else
//            sXML = new String(sXML + "user=\"0\" ");

        // -- Access to the data

        sXML = new String(sXML + "access=\"" + st.stItem[i].iAccess + "\" ");
        if (st.stItem[i].source.length() > 0)
          sXML = new String(sXML + "source=\"" + st.stItem[i].source + "\" ");
//        else
//            sXML = new String(sXML + "source=\"\" ");

        // -- User Update Date

        if (st.stItem[i].sDate.length() > 0)
            sXML = new String(sXML + "date=\"" + st.stItem[i].sDate + "\" ");
//        else
//          sXML = new String(sXML + "date=\"" + cmn.cmnString.getDate() + "\" ");

        sXML = new String(sXML + "/>");

        // END misc Block

        sXML = new String(sXML + " <cnt ");
        sXML = new String(sXML + "las=\""    + st.stItem[i].iLAS   + "\" ");
        sXML = new String(sXML + "tops=\""   + st.stItem[i].iTops  + "\" ");
        sXML = new String(sXML + "core=\""   + st.stItem[i].iCore  + "\" ");
        sXML = new String(sXML + "images=\"" + st.stItem[i].images + "\"/>");

        sXML = new String(sXML +  " </data>");

        // END New Data Record Start
      }
    }
    else
      sXML = new String (sXML + "records=\"0\">");

    sXML = new String (sXML + "</headers>");
//System.out.println(sXML);

    return (sXML);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}