/*
 * @ReadHeadersXMLFile.java Version 1.0 10/06/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import iqstrat.iqstratHeadersListStruct;
import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratHeadersUtility;

import cmn.cmnString;

/** Class ReadHeadersXMLFile()
 * <p> This Class will parse the Well Header Information XML Data Stream
 *     directly to the Well Header Information List Data Structure.
 *
 *  @version 1.1 10/06/2007
 *  @author  John R. Victorine
 */

public class ReadHeadersXMLFile extends DefaultHandler
{
  public static final int FILE     = 0;
  public static final int URL      = 1;
  public static final int SERVER   = 2;
  public static final int SERVER_ZIP = 3;

  public static final int NOERROR    =  0;
  public static final int ERROR      = -1;

  public int  iComments = 0;

  public static final String ROOTNAME         = "headers"; // rootname
  public static final String RECORDS          = "records";  // Total Wells
  public static final String DATA             = "data";     // ELEMENT data

  // Identification Information

  public static final String INFO             = "info";      // ELEMENT info

  public static final String KID              = "kid";       // Primary key
  public static final String WELL_KID         = "well_kid";  // Well Primary key
  public static final String KEY              = "key";       // Primary Key
  public static final String TYPE             = "type";      // type of data
  public static final String API_NUMBER       = "api";       // API-Number
  public static final String NAME             = "name";      // outcrop name
  public static final String WELL_STATUS      = "status";    // Well Status
  public static final String ERROR_VAL        = "error";     // DB Error

  // other well information

  public static final String OTHER            = "other";     // ELEMENT info

  public static final String OPERATOR         = "operator";  // Operator Name
  public static final String OPERATOR_KID     = "oper_kid";  // Operator KID
  public static final String FIELD            = "field";     // Field Name
  public static final String FIELD_KID        = "field_kid"; // Field KID

  // Location Information

  public static final String LOCATION         = "loc";       // ELEMENT info

  public static final String STATE            = "state";     // State Name
  public static final String STATE_CODE       = "state_cd";  // State Name
  public static final String COUNTY           = "county";    // County Name
  public static final String COUNTY_CODE      = "county_cd"; // County Name

  public static final String TOWNSHIP         = "town";      // township
  public static final String TOWNSHIP_DIR     = "town_dir";  // township dir
  public static final String RANGE            = "range";     // range
  public static final String RANGE_DIR        = "range_dir"; // range direction
  public static final String SECTION          = "section";   // section

  // XY Position Information

  public static final String XY_POSITION      = "xy";        // ELEMENT position

  public static final String LATITUDE         = "latitude";  // Latitude
  public static final String LONGITUDE        = "longitude"; // Longitude
  public static final String ZONE             = "zone";      // UTM Zone
  public static final String UTM_X            = "utm_x";     // UTM Y-Position
  public static final String UTM_Y            = "utm_y";     // UTM X-Position

  // Z Position Information

  public static final String Z_POSITION       = "z";     // ELEMENT position

  public static final String DEPTH            = "depth"; // Depth or Height
  public static final String GROUND_LEVEL     = "gl";    // Ground Level
  public static final String KELLY_BUSHING    = "kb";    // Kelly Bushing
  public static final String DERRICK_FLOOR    = "df";    // Derrick Floor

  // Comments

  public static final String COMMENTS         = "comments"; // Comments

  // Misc Information

  public static final String MISC             = "misc";     // ELEMENT misc

  public static final String USER_UPDATE      = "user";     // User Update
  public static final String SOURCE           = "source";   // Source of data
  public static final String ACCESS           = "access";   // Access to data
  public static final String DATE             = "date";     // Date

  public static final String CNT              = "cnt";       // ELEMENT cnt

  public static final String LAS              = "las";       // # of LAS Files
  public static final String TOPS             = "tops";      // # of Tops
  public static final String CORE             = "core";      // # of Core
  public static final String IMAGES           = "images";    // # of Core Images
  public static final String GEO              = "geo";       // # of Geologist Report
  public static final String DST              = "dst";       // # of DST Data
  public static final String PERF             = "perf";      // # of Perforation Data
  public static final String PFEFFER          = "pfeffer";   // # of PfEFFER Data

  public static final String OHM_LOG          = "ohm_log";   // Has Resistivity Log
  public static final String LITHO_LOG        = "litho_log"; // Has Litho-Density Log
  public static final String NPHI_LOG         = "nphi_log";  // Has Neutron Porosity Log
  public static final String RHOB_LOG         = "rhob_log";  // Has Bulk Density Log
  public static final String PE_LOG           = "pe_log";    // Has Photoelectric Log
  public static final String SONIC_LOG        = "sonic_log"; // Has Sonic Log
  public static final String GR_LOG           = "gr_log";    // Has Gamma Ray Log
  public static final String SPGR_LOG         = "spgr_log";  // Has Spectral Gamma Ray Log

  // Global Variables;

  private int     iError             = NOERROR;
  private String  sError             = new String("");

  private int                       iType       = FILE;
  private int                       iRows       = 0;
  private int                       iCount      = -1;
  private iqstratHeadersListStruct stList = new iqstratHeadersListStruct();

  /** Method ReadHeadersXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadHeadersXMLFile() { iType = FILE; }

  /** Method ReadHeadersXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadHeadersXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public iqstratHeadersListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          iError    = ERROR;
          sError    = new String(e.toString());
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      iError = ERROR;
      sError = new String("\n** Parsing error"            +
                          ", line " + spe.getLineNumber() +
                          ", uri "  + spe.getSystemId()   + "\n" +
                          spe.getMessage());
      System.out.println(sError);

      // Use the contained exception, if any
System.out.print("spe");
      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)
System.out.print("sxe");
      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built
System.out.print("pce");
      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error
System.out.print("ioe");
      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(DATA))
    {
      iComments = 0;

      iCount++;
      if (iCount < iRows)
      {
//        stList.stItem[iCount] = iqstrat.iqstratHeadersUtility.initialize();
        stList.stItem[iCount] = new iqstratHeadersStruct();
      }
    }

    if (eName.equals(COMMENTS)) { iComments = 1; }
    if (eName.equals(MISC))     { iComments = 0; }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i = 0; i < attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new iqstratHeadersStruct[iRows];
            stList.iCount = iRows;
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            stList.stItem[iCount] = ParseData(aName,
                                              sOut.trim(),
                                              stList.stItem[iCount]);
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    String s = new String(buf, offset, len);
    if (iComments == 1)
    {
      stList.stItem[iCount].sComments = new String(
          stList.stItem[iCount].sComments + " " + s.trim());
    }
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException
  {
    iError    = ERROR;
    sError    = new String(e.toString());
    throw e;
  }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    iError = ERROR;
    sError = new String("** Warning" +
                        ", line "    + err.getLineNumber() +
                        ", uri "     + err.getSystemId()   + "\n" +
                        err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Well Header Information String
   * @param  st         = The Well Header Information Data Structure
   * @return st         = The Well Header Information Data Structure
   */

  private iqstratHeadersStruct ParseData(String sIdentifier,
                                          String sData,
                                          iqstratHeadersStruct st)
  {
    // Identification Information

    if (sIdentifier.equals(KID)) { st.sKID = new String(sData); }
    if (sIdentifier.equals(WELL_KID)) { st.sWELLKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { st.sKEY = new String(sData); }
    if (sIdentifier.equals(TYPE))
    {
      if (sData.equals("WELL"))    st.iType = iqstratHeadersStruct.WELL;
      if (sData.equals("OUTCROP")) st.iType = iqstratHeadersStruct.OUTCROP;
    }
    if (sIdentifier.equals(API_NUMBER))  { st.sAPI = new String(sData); }
    if (sIdentifier.equals(NAME))
    {
//      st.sName = new String(cmn.cmnString.convertHTMLSpace(sData));
      st.sName = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(WELL_STATUS))
    {
      st.status = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = ERROR;
      sError    = new String(sData);
      st.sError = new String(sError);
    }

    // other well information

    if (sIdentifier.equals(OPERATOR))
    {
//      st.sOperator = new String(cmn.cmnString.convertHTMLSpace(sData));
      st.sOperator = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(OPERATOR_KID)) {st.sOperator_kid = new String(sData);}
    if (sIdentifier.equals(FIELD))
    {
//      st.sField = new String(cmn.cmnString.convertHTMLSpace(sData));
      st.sField = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(FIELD_KID)) { st.sField_kid = new String(sData); }

    // Location Information

    if (sIdentifier.equals(STATE)) { st.state = new String(sData); }
    if (sIdentifier.equals(STATE_CODE))
    {
      st.iState = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(COUNTY)) { st.sCounty = new String(sData); }
    if (sIdentifier.equals(COUNTY_CODE))
    {
      st.iCounty = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(LOCATION))
    {
      st.sLocation = new String(sData.replaceAll("%20", " "));
    }

    if (sIdentifier.equals(TOWNSHIP))
    {
      st.iTownship = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(TOWNSHIP_DIR)) { st.sTownship = new String(sData); }
    if (sIdentifier.equals(RANGE))
    {
      st.iRange = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(RANGE_DIR)) { st.sRange = new String(sData); }
    if (sIdentifier.equals(SECTION))
    {
      st.iSection = cmn.cmnString.stringToInt(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      st.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      st.dLongitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(ZONE))
    {
      st.dZone = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(UTM_X))
    {
      st.dUTMx = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(UTM_Y))
    {
      st.dUTMy = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(DEPTH))
    {
      st.depth = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      st.dGL = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(KELLY_BUSHING))
    {
      st.dKB = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(DERRICK_FLOOR))
    {
      st.dDF = cmn.cmnString.stringToDouble(sData);
    }

    // Misc Information

    if (sIdentifier.equals(USER_UPDATE)) { st.sUpdate = new String(sData); }
    if (sIdentifier.equals(SOURCE))
    {
//      st.source = new String(cmn.cmnString.convertHTMLSpace(sData));
      st.source = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(ACCESS))
    {
      st.iAccess = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(DATE)) { st.sDate = new String(sData); }

    if (sIdentifier.equals(LAS))
    {
      st.iLAS = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(TOPS))
    {
      st.iTops = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(CORE))
    {
      st.iCore = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(IMAGES))
    {
      st.images = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(GEO))
    {
      st.iGeo = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(DST))
    {
      st.iDST = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(PERF))
    {
      st.iPerf = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(PFEFFER))
    {
      st.iPfeffer = cmn.cmnString.stringToInt(sData);
    }

    if (sIdentifier.equals(OHM_LOG))
    {
      st.iOHM = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(LITHO_LOG))
    {
      st.iLitho = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(NPHI_LOG))
    {
      st.iNPHI = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(RHOB_LOG))
    {
      st.iRHOB = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(PE_LOG))
    {
      st.iPE = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(SONIC_LOG))
    {
      st.iSonic = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(GR_LOG))
    {
      st.iGR = cmn.cmnString.stringToInt(sData);
    }
    if (sIdentifier.equals(SPGR_LOG))
    {
      st.iSPGR = cmn.cmnString.stringToInt(sData);
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
