/*
 * @ReadFilesXMLFile.java Version 1.0 12/12/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import iqstrat.iqstratFilesListStruct;
import iqstrat.iqstratFilesStruct;

import cmn.cmnString;

/** Class ReadFilesXMLFile()
 * <p> This Class will parse the Files Information XML Data Stream
 *     directly to the Files Information List Data Structure.
 *
 *  @version 1.1 12/12/2007
 *  @author  John R. Victorine
 */

public class ReadFilesXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  public static final int NOERROR    =  0;
  public static final int ERROR      = -1;

  public static final String ROOTNAME   = "files";   // rootname
  public static final String RECORDS    = "records"; // Total Wells
  public static final String DATA       = "data";    // ELEMENT data

  // Identification Information

  public static final String KID        = "kid";     // Primary key
  public static final String ERROR_VAL  = "error";   // DB Error
  public static final String USER_KID   = "user";    // Well Primary key
  public static final String HEADER_KID = "header";  // Primary Key
  public static final String DATA_TYPE  = "type";    // type of data
  public static final String FILE_TYPE  = "file";    // API-Number
  public static final String URL_PATH   = "url";     // outcrop name
  public static final String PATH       = "path";    // Well Status
  public static final String FILENAME   = "name";    // Well Status

  // Global Variables;

  private int     iError = NOERROR;
  private String  sError = new String("");

  private int                    iType  = FILE;
  private int                    iRows  = 0;
  private int                    iCount = -1;
  private iqstratFilesListStruct stList = new iqstratFilesListStruct();

  /** Method ReadFilesXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadFilesXMLFile() { iType = FILE; }

  /** Method ReadFilesXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadFilesXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public iqstratFilesListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      iError = ERROR;
      sError = new String("\n** Parsing error"            +
                          ", line " + spe.getLineNumber() +
                          ", uri "  + spe.getSystemId()   + "\n" +
                          spe.getMessage());
      System.out.println(sError);

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(DATA))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new iqstratFilesStruct();
      }
    }

    // Add Attributes to the Files Data List Structure

    if (attrs != null)
    {
      for (int i=0; i<attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());
          stList.iCount = iRows;

          if (iRows > 0)
          {
            stList.stItem = new iqstratFilesStruct[iRows];
          }
        }

        // Parse each attribute and add it to the structure

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            stList.stItem[iCount] = ParseData(aName,
                                              sOut.trim(),
                                              stList.stItem[iCount]);
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    iError = ERROR;
    sError = new String("** Warning" +
                        ", line "    + err.getLineNumber() +
                        ", uri "     + err.getSystemId()   + "\n" +
                        err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual File
   *     Information
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Files Information String
   * @param  st         = The Files Information Data Structure
   * @return st         = The Files Information Data Structure
   */

  private iqstratFilesStruct ParseData(String sIdentifier,
                                       String sData,
                                       iqstratFilesStruct st)
  {
    if (sIdentifier.equals(KID))        { st.sKEY      = new String(sData); }
    if (sIdentifier.equals(USER_KID))   { st.sUser     = new String(sData); }
    if (sIdentifier.equals(HEADER_KID)) { st.sHeader   = new String(sData); }
    if (sIdentifier.equals(DATA_TYPE))  { st.sType     = new String(sData); }
    if (sIdentifier.equals(FILE_TYPE))  { st.sFile     = new String(sData); }
    if (sIdentifier.equals(URL_PATH))   { st.sURL      = new String(sData); }
    if (sIdentifier.equals(PATH))       { st.sPath     = new String(sData); }
    if (sIdentifier.equals(FILENAME))   { st.sFilename = new String(sData); }

    if (sIdentifier.equals(ERROR_VAL))
    {
      if (sData.length() > 0)
        iError  = ERROR;
      sError    = new String(sData);
    }

    return (st);
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}
