/*
 * @iqstratPanel.java Version 1.1 09/04/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import iqstrat.gui.iqstratShaleListTable;
import iqstrat.iqstratShaleListStruct;
import iqstrat.iqstratShaleStruct;

/** Class iqstratPanel
 *  <p> This Class will build the Available Plot Tracks Panel that will allow
 *      the user to turn on or off any track that can be displayed.
 *
 *  @version 1.1 09/04/2007
 *  @author  John R. Victorine
 */

public class iqstratShaleFrame extends JFrame implements ActionListener
{
  private Observable notifier = null;

  // Global  Variables

  private String sKEY        = "0";

  private double dStart      = 0.0;
  private double dEnd        = 0.0;

  private double dGammaMin   = iqstrat.iqstratShaleStruct.GAMMA_MIN;
  private double dGammaShaly = iqstrat.iqstratShaleStruct.SHALY;
  private double dGammaShale = iqstrat.iqstratShaleStruct.SHALE;
  private double dGammaHot   = iqstrat.iqstratShaleStruct.SHALE_HOT;
  private double dGammaMax   = iqstrat.iqstratShaleStruct.GAMMA_MAX;

  private iqstratShaleListStruct stList = null;
  private iqstratShaleListTable  pTable = null;

  private static final int _ADD    = 0;
  private static final int _MODIFY = 1;

  private int iStatus = _ADD;

  private static final String DESCRIP =
    "The Gamma Ray Levels change the appearance of the Image Tracks: \n" +
    "  -- Lithology Composition Plot Track \n" +
    "  -- Texture by Gamma Ray Plot Track \n"  +
    "  -- Colorlith Image Tracks.";

  // Global Widgets

  private JButton btnOk     = new JButton();
  private JButton btnCancel = new JButton();

  private JTextField txtCleanStart = new JTextField();
  private JTextField txtShalyStart = new JTextField();
  private JTextField txtShaleStart = new JTextField();
  private JTextField txtHotStart   = new JTextField();
  private JTextField txtHotEnd     = new JTextField();

  private JButton btnReset     = new JButton();
  private JButton btnAdd       = new JButton();
  private JButton btnModify    = new JButton();
  private JButton btnRemove    = new JButton();
  private JButton btnRemoveAll = new JButton();

  private JTextField txtStart = new JTextField();
  private JTextField txtEnd   = new JTextField();

  /** Constructor iqstratPanel()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stList   = The Shale List Data Structure
   */

  public iqstratShaleFrame(Observable notifier, iqstratShaleListStruct stList)
  {
    try
    {
      this.notifier = notifier;
      this.stList   = stList;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    JPanel pnlGamma       = new JPanel();
    JPanel pnlTop         = new JPanel();
    JPanel pnlDescription = new JPanel();
    JPanel pnlDepths      = new JPanel();
    JPanel pnlStart       = new JPanel();
    JPanel pnlEnd         = new JPanel();
    JPanel pnlGRInput     = new JPanel();
    JPanel pnlGRLabels    = new JPanel();
    JPanel pnlText        = new JPanel();
    JPanel pnlCleanStart  = new JPanel();
    JPanel pnlShalyStart  = new JPanel();
    JPanel pnlShaleStart  = new JPanel();
    JPanel pnlHotEnd      = new JPanel();
    JPanel pnlHotStart    = new JPanel();
    JPanel pnlButtons     = new JPanel();
    JPanel pnlAdd         = new JPanel();
    JPanel pnlCenter      = new JPanel();
    JPanel pnlListButtons = new JPanel();

    JLabel lblStart       = new JLabel();
    JLabel lblEnd         = new JLabel();
    JLabel lblHot         = new JLabel();
    JLabel lblShale       = new JLabel();
    JLabel lblShaly       = new JLabel();
    JLabel lblClean       = new JLabel();

    JTextArea txtDescription = new JTextArea();
    JScrollPane scroll       = new JScrollPane();

    TitledBorder titledBorderGammRay = new TitledBorder(
        new EtchedBorder(EtchedBorder.RAISED,
                         Color.white,
                         new Color(168, 168, 168)),
        "Gamma Ray Zones (API):");
    titledBorderGammRay.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderDepthRange = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "Effective Depth Range:");
    titledBorderDepthRange.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "Start:");
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderShaly = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "Shaly Start:");
    titledBorderShaly.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderShale = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "Shale Start:");
    titledBorderShale.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderHotShale = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "Hot Shale:");
    titledBorderHotShale.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    TitledBorder titledBorderEnd = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(168, 168, 168)),
        "End:");
    titledBorderEnd.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.getContentPane().setLayout(new BorderLayout());
    this.setTitle("Gamma Ray Levels");

    // Global Frame Buttons

    pnlButtons.setBorder(BorderFactory.createEtchedBorder());

    btnOk.setFont(new java.awt.Font("Dialog", 1, 11));
    btnOk.setPreferredSize(new Dimension(75, 25));
    btnOk.setText("Ok");
    btnOk.addActionListener(this);

    btnCancel.setFont(new java.awt.Font("Dialog", 1, 11));
    btnCancel.setPreferredSize(new Dimension(75, 25));
    btnCancel.setText("Cancel");
    btnCancel.addActionListener(this);

    // Gamma Ray Depth Data Panel

    pnlGamma.setBorder(titledBorderGammRay);
    pnlGamma.setLayout(new BorderLayout());

    // -- Top Panel

    pnlTop.setLayout(new BorderLayout());

    // -- Shale Dialog Description Panel

    pnlDescription.setLayout(new BorderLayout());

    txtDescription.setFont(new java.awt.Font("Dialog", 1, 12));
    txtDescription.setForeground(Color.blue);
    txtDescription.setEditable(false);
    txtDescription.setText( DESCRIP );
    txtDescription.setLineWrap(true);
    txtDescription.setRows(4);
    txtDescription.setWrapStyleWord(true);

    // .. Depth Range Panel

    pnlDepths.setLayout(new GridLayout());
    pnlDepths.setBorder(titledBorderDepthRange);

    // .... Starting Depth

    lblStart.setFont(new java.awt.Font("Dialog", 1, 11));
    lblStart.setText("Starting Depth:");

    txtStart.setText("" + dStart);
    txtStart.setColumns(8);
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // .... Ending Depth

    lblEnd.setFont(new java.awt.Font("Dialog", 1, 11));
    lblEnd.setText("Ending Depth:");

    txtEnd.setText("" + dEnd);
    txtEnd.setColumns(8);
    txtEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtEnd.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // -- Gamma Ray Input Panel

    pnlGRInput.setLayout(new BorderLayout());

    // .. Gamma Ray Limits Panel

    pnlGRLabels.setLayout(new GridLayout());

    // -- Gamma Ray Limits Textfield Panel

    pnlText.setLayout(new GridLayout());

    lblClean.setFont(new java.awt.Font("Monospaced", 1, 12));
    lblClean.setHorizontalAlignment(SwingConstants.CENTER);
    lblClean.setText("<= Clean =>");

    lblShaly.setFont(new java.awt.Font("Monospaced", 1, 12));
    lblShaly.setHorizontalAlignment(SwingConstants.CENTER);
    lblShaly.setText("<= Shaly =>");

    lblShale.setFont(new java.awt.Font("Monospaced", 1, 12));
    lblShale.setHorizontalAlignment(SwingConstants.CENTER);
    lblShale.setText("<= Shale =>");

    lblHot.setFont(new java.awt.Font("Monospaced", 1, 12));
    lblHot.setHorizontalAlignment(SwingConstants.CENTER);
    lblHot.setText("<=  Hot  =>");

    // .. Clean Formation

    pnlCleanStart.setBorder(titledBorderStart);
    pnlCleanStart.setLayout(new BorderLayout());

    txtCleanStart.setText("" + dGammaMin);
    txtCleanStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtCleanStart.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // .. Shaly Formation

    pnlShalyStart.setBorder(titledBorderShaly);
    pnlShalyStart.setLayout(new BorderLayout());

    txtShalyStart.setText("" + dGammaShaly);
    txtShalyStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtShalyStart.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // .. Shale Formation

    pnlShaleStart.setBorder(titledBorderShale);
    pnlShaleStart.setLayout(new BorderLayout());

    txtShaleStart.setText("" + dGammaShale);
    txtShaleStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtShaleStart.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // .. Hot Shale Formation

    pnlHotStart.setBorder(titledBorderHotShale);
    pnlHotStart.setLayout(new BorderLayout());

    txtHotStart.setText("" + dGammaHot);
    txtHotStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtHotStart.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    pnlHotEnd.setBorder(titledBorderEnd);
    pnlHotEnd.setLayout(new BorderLayout());

    txtHotEnd.setText("" + dGammaMax);
    txtHotEnd.setHorizontalAlignment(SwingConstants.TRAILING);
    txtHotEnd.addFocusListener(new iqstratShaleFrameFocusAdapter(this));

    // -- Buttton Add Data

    pnlAdd.setLayout(new BorderLayout());

    btnAdd.setFont(new java.awt.Font("Dialog", 1, 11));
    btnAdd.setText("Add Gamma Ray Ranges to List");
    btnAdd.addActionListener(this);

    btnReset.setFont(new java.awt.Font("Dialog", 1, 11));
    btnReset.setText("Reset Data");
    btnReset.addActionListener(this);

    // Gamma Ray Limits Data List Table

    pnlCenter.setLayout(new BorderLayout());

    pTable = new iqstratShaleListTable(stList);
    scroll = pTable.getScrollPane();

    // Modify Gamma Ray Limits Data List Panel

    pnlListButtons.setLayout(new GridLayout());

    btnModify.setFont(new java.awt.Font("Dialog", 1, 11));
    btnModify.setText("Modify");
    btnModify.addActionListener(this);

    btnRemove.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemove.setText("Remove");
    btnRemove.addActionListener(this);

    btnRemoveAll.setFont(new java.awt.Font("Dialog", 1, 11));
    btnRemoveAll.setText("Remove All");
    btnRemoveAll.addActionListener(this);

    setButtons();

    // Add Widgets to Frame

    this.getContentPane().add(pnlButtons, BorderLayout.SOUTH);
    pnlButtons.add(btnOk,     null);
    pnlButtons.add(btnCancel, null);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(scroll,                BorderLayout.CENTER);

    pnlCenter.add(pnlListButtons,    BorderLayout.SOUTH);
    pnlListButtons.add(btnModify,    null);
    pnlListButtons.add(btnRemove,    null);
    pnlListButtons.add(btnRemoveAll, null);

    this.getContentPane().add(pnlGamma, BorderLayout.NORTH);
    pnlGamma.add(pnlGRInput,         BorderLayout.CENTER);
    pnlGRInput.add(pnlGRLabels,      BorderLayout.NORTH);
    pnlGRLabels.add(lblClean,        null);
    pnlGRLabels.add(lblShaly,        null);
    pnlGRLabels.add(lblShale,        null);
    pnlGRLabels.add(lblHot,          null);

    pnlGRInput.add(pnlText,          BorderLayout.SOUTH);

    pnlText.add(pnlCleanStart,       null);
    pnlCleanStart.add(txtCleanStart, BorderLayout.CENTER);

    pnlText.add(pnlShalyStart,       null);
    pnlShalyStart.add(txtShalyStart, BorderLayout.CENTER);

    pnlText.add(pnlShaleStart,       null);
    pnlShaleStart.add(txtShaleStart, BorderLayout.CENTER);

    pnlText.add(pnlHotStart,         null);
    pnlHotStart.add(txtHotStart,     BorderLayout.CENTER);

    pnlText.add(pnlHotEnd,           null);
    pnlHotEnd.add(txtHotEnd,         BorderLayout.CENTER);

    pnlGamma.add(pnlAdd,             BorderLayout.SOUTH);
    pnlAdd.add(btnAdd,               BorderLayout.CENTER);
    pnlAdd.add(btnReset,             BorderLayout.EAST);

    pnlGamma.add(pnlTop,             BorderLayout.NORTH);
    pnlTop.add(pnlDepths,            BorderLayout.SOUTH);

    pnlDepths.add(pnlStart,          null);
    pnlDepths.add(pnlEnd,            null);

    pnlStart.add(lblStart,           null);
    pnlStart.add(txtStart,           null);

    pnlEnd.add(lblEnd,               null);
    pnlEnd.add(txtEnd,               null);

    pnlTop.add(pnlDescription, BorderLayout.NORTH);
    pnlDescription.add(txtDescription, BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(450, 450));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setLocation((d.width - this.getSize().width) / 2,
                     (d.height - this.getSize().height) / 2);

    this.setResizable(false);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getData()
   * <p> This method will retrieve the Shale List Data Structure
   * @return stList = the Shale List Data Structure
   */

  public iqstratShaleListStruct getData() { return (stList); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setButtons()
   * <p> This method will set the button data
   */

  private void setButtons()
  {
    btnAdd.setEnabled(true);
    btnModify.setEnabled(false);
    btnRemove.setEnabled(false);
    btnRemoveAll.setEnabled(false);

    if (iStatus == _ADD)
    {
      if (pTable.getTotalRows() > 0)
      {
        btnModify.setEnabled(true);
        btnRemove.setEnabled(true);
        btnRemoveAll.setEnabled(true);
      }

      btnAdd.setText("Add Gamma Ray Ranges to List");
    }
    else if (iStatus == _MODIFY)
    {
      btnAdd.setText("Modify the Gamma Ray Ranges Data");
    }
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method reset()
   * <p> This method will reset the text fields to the original data
   */

  private void reset()
  {
    iStatus     = _ADD;
    sKEY        = "0";

    dStart      = 0.0;
    dEnd        = 0.0;

    dGammaMin   = iqstrat.iqstratShaleStruct.GAMMA_MIN;
    dGammaShaly = iqstrat.iqstratShaleStruct.SHALY;
    dGammaShale = iqstrat.iqstratShaleStruct.SHALE;
    dGammaHot   = iqstrat.iqstratShaleStruct.SHALE_HOT;
    dGammaMax   = iqstrat.iqstratShaleStruct.GAMMA_MAX;

    txtCleanStart.setText(""+dGammaMin);
    txtShalyStart.setText(""+dGammaShaly);
    txtShaleStart.setText(""+dGammaShale);
    txtHotStart.setText(""+dGammaHot);
    txtHotEnd.setText(""+dGammaMax);
    txtStart.setText(""+dStart);
    txtEnd.setText(""+dEnd);
  }

  /** Method add()
   * <p> This method will add the shale data to the list
   */

  public void add()
  {
    iqstratShaleStruct st = new iqstratShaleStruct();

    st.depthStart = dStart;     // Starting Depth
    st.depthEnd   = dEnd;       // Ending Depth

    st.dGammaMin  = dGammaMin;    // Minimum Gamma Ray Value
    st.dShaly     = dGammaShaly;  // Start of Shaly Formation
    st.dShale     = dGammaShale;  // Start of Shale Formation
    st.dHotShale  = dGammaHot;    // Start of Hot Shale Formation
    st.dGammaMax  = dGammaMax;    // Maximum  Gamma Ray Value

    if (iStatus == _ADD)
    {
      st.sKEY = cmn.cmnString.UniqueName();
      stList = iqstrat.iqstratShaleUtility.add(stList, st);
    }
    else
    {
      st.sKEY = new String(sKEY);
      stList = iqstrat.iqstratShaleUtility.modify(stList, st, sKEY);
    }

    pTable.repopulateList(stList);

    reset();
  }


  /** Method modify()
   * <p> This method will modify the shale data in the list
   */

  public void modify()
  {
    iqstratShaleStruct st = pTable.getRowData();

    iStatus = _MODIFY;

    if (st != null)
    {
      sKEY        = new String(st.sKEY);
      dStart      = st.depthStart;
      dEnd        = st.depthEnd;

      dGammaMin   = st.dGammaMin;
      dGammaShaly = st.dShaly;
      dGammaShale = st.dShale;
      dGammaHot   = st.dHotShale;
      dGammaMax   = st.dGammaMax;

      txtCleanStart.setText(""+dGammaMin);
      txtShalyStart.setText(""+dGammaShaly);
      txtShaleStart.setText(""+dGammaShale);
      txtHotStart.setText(""+dGammaHot);
      txtHotEnd.setText(""+dGammaMax);
      txtStart.setText(""+dStart);
      txtEnd.setText(""+dEnd);
    }
  }

  /** Method remove()
   * <p> This method will remove the shale data from the list
   */

  public void remove()
  {
    iqstratShaleStruct st = pTable.getRowData();

    stList = iqstrat.iqstratShaleUtility.remove(stList, st.sKEY);
    pTable.repopulateList(stList);
  }

  /** Method removeAll()
   * <p> This method will remove all the shale data from the list
   */

  public void removeAll()
  {
    if (stList != null)
      stList.delete();
    stList = null;

    pTable.repopulateList(stList);
  }

  /** Method close()
   * <p> This method will set the global variables to null in order to force
   *     JAVA to free the memory allocated for this panel.
   */

  public void close()
  {
    notifier = null;

    // Global  Variables

    stList = null;

    if (pTable != null)
      pTable.close();
    pTable = null;

    // Global Widgets

    btnOk         = null;
    btnCancel     = null;

    txtCleanStart = null;
    txtShalyStart = null;
    txtShaleStart = null;
    txtHotStart   = null;
    txtHotEnd     = null;

    btnReset      = null;
    btnAdd        = null;
    btnModify     = null;
    btnRemove     = null;
    btnRemoveAll  = null;

    txtStart      = null;
    txtEnd        = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == btnReset    ) { reset(); }
    if (event.getSource() == btnAdd      ) { add(); }
    if (event.getSource() == btnModify   ) { modify(); }
    if (event.getSource() == btnRemove   ) { remove(); }
    if (event.getSource() == btnRemoveAll) { removeAll(); }

    setButtons();

    if (event.getSource() == btnCancel) { close(); }

    if (event.getSource() == btnOk)
    {
      notifier.notifyObservers(new String("Shale data ready"));
    }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtCleanStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Clean Shale Gamma Ray Start Value is a Numeric Field");
    }

    if (e.getSource() == txtShalyStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Shaly Start Gamma Ray Value is a Numeric Field");
    }

    if (e.getSource() == txtShaleStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Shale Start Gamma Ray Value is a Numeric Field");
    }

    if (e.getSource() == txtHotStart  )
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Hot Shale Start Gamma Ray Value is a Numeric Field");
    }

    if (e.getSource() == txtHotEnd    )
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Hot Shale End Gamma Ray Value is a Numeric Field");
    }

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp = txtStart.getText();
      sMessage = new String("Start Depth Value is a Numeric Field");
    }

    if (e.getSource() == txtEnd)
    {
      iNumeric = 1;
      sTemp = txtEnd.getText();
      sMessage = new String("End Depth Value is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {

        if (e.getSource() == txtCleanStart)
          dGammaMin   = cmn.cmnString.stringToDouble(txtCleanStart.getText());

        if (e.getSource() == txtShalyStart)
          dGammaShaly = cmn.cmnString.stringToDouble(txtShalyStart.getText());

        if (e.getSource() == txtShaleStart)
          dGammaShale = cmn.cmnString.stringToDouble(txtShaleStart.getText());

        if (e.getSource() == txtHotStart  )
          dGammaHot   = cmn.cmnString.stringToDouble(txtHotStart.getText());

        if (e.getSource() == txtHotEnd    )
          dGammaMax   = cmn.cmnString.stringToDouble(txtHotEnd.getText());

        if (e.getSource() == txtStart)
          dStart = cmn.cmnString.stringToDouble(txtStart.getText());

        if (e.getSource() == txtEnd)
          dEnd = cmn.cmnString.stringToDouble(txtEnd.getText());
      }
    }
  }
}

/** CLASS iqstratShaleFrameFocusAdapter()
  *  <p> This Class will handle Actions when focus is lost on a textfield
  */

class iqstratShaleFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  iqstratShaleFrame adaptee;

  iqstratShaleFrameFocusAdapter(iqstratShaleFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

