/*
 * @iqstratPlotPanel.java Version 1.1 09/04/2007
 *
 * Copyright (c) 2007 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.gui;

import java.awt.*;
import java.awt.image.*;
import java.util.Observable;
import javax.swing.*;

import cmn.cmnStruct;

import iqstrat.iqstratStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.plot.iqstratPlot;

import las.lasFileDataStruct;
//import las.gui.lasThinTrackFrame;
import las.gui.lasConvertFrame;

import lith.lithology.lithologyListStruct;
import lith.texture.textureListStruct;
import lith.rock.rockColumnListStruct;
//import lith.gui.lithColorlithFrame;
import lith.gui.lithImageTrackFrame;

import brine.brineListStruct;

import rock.rockFileDataStruct;
import rock.rockDataListStruct;
import rock.rockImagesListStruct;
import rock.phi.phiListStruct;
import rock.fossil.fossilListStruct;
import rock.sedimentary.sedimentaryListStruct;
import rock.color.rockColorListStruct;

import horizon.regions.regionsListStruct;
import horizon.strat.stratListStruct;
import horizon.seq.seqListStruct;

import horizon.env.envListStruct;
import horizon.bio.bioStratListStruct;

import pfeffer.pfefferDataListStruct;
import pfeffer.pfefferDataStruct;

/** Class iqstratPlotPanel
 *  <p> This Class will create the stratigraphic Plot.
 *
 *  @version 1.1 09/04/2007
 *  @author  John R. Victorine
 */

public class iqstratPlotPanel extends JPanel
{
  // Input Variables

  private int    iDataType  = -1;
  private JFrame parent     = null;
  private double depthStart = 0.0;
  private double depthEnd   = 0.0;
  private int    iScale     = -1;
  private int    iUnit      = 0;
  private int    iStatus[]  = null;
  private iqstratStruct stStruct = null;

  // Global Variables

  private int iWidth   = 0;
  private int iHeight  = 0;

  private iqstratPlot plot   = null;

  // Panel Widgets

  private ScrollPane scroll = new ScrollPane();
  private Panel      panel  = new Panel();

  private Cursor normalCursor = null;
  private Cursor mouseCursor  = null;

  /** Constructor iqstratPlotPanel()
   *  <p> This is the Constructor for this class.
   *  @param iDataType  = Data Type; 0=Well Data; 1=Outcrop Data
   *  @param parent     = the frame calling this panel
   *  @param depthStart = The starting depth of plot
   *  @param depthEnd   = The ending depth of plot
   *  @param iScale     = The scale
   *  @param iStatus    = The Track Button Status On or Off
   *  @param normalCursor = Normal Cursor
   *  @param mouseCursor  = Hand Cursor
   */

  public iqstratPlotPanel(int iDataType,
                          JFrame parent,
                          double depthStart,
                          double depthEnd,
                          int iScale,
                          int iStatus[],
                          Cursor normalCursor,
                          Cursor mouseCursor)
  {
    try
    {
      this.iDataType  = iDataType;
      this.parent     = parent;
      this.depthStart = depthStart;
      this.depthEnd   = depthEnd;
      this.iScale     = iScale;
      this.iStatus    = iStatus;
      this.normalCursor = normalCursor;
      this.mouseCursor  = mouseCursor;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** Constructor iqstratPlotPanel()
   *  <p> This is the Constructor for this class.
   *  @param parent     = the frame calling this panel
   *  @param stStruct   = The Global IQSTRAT Data Structure
   *  @param iStatus    = The Track Button Status On or Off
   *  @param normalCursor = Normal Cursor
   *  @param mouseCursor  = Hand Cursor
   */

  public iqstratPlotPanel(JFrame parent,
                          iqstratStruct stStruct,
                          int iStatus[],
                          Cursor normalCursor,
                          Cursor mouseCursor)
  {
    try
    {
      this.parent     = parent;
      this.stStruct   = stStruct;
      this.iDataType  = stStruct.iDataType;
      this.depthStart = stStruct.depthStart;
      this.depthEnd   = stStruct.depthEnd;
      this.iScale     = stStruct.iScale;
      this.iUnit      = stStruct.iUnit;
      this.iStatus    = iStatus;
      this.normalCursor = normalCursor;
      this.mouseCursor  = mouseCursor;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Panel for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    Panel pnlCenter = new Panel();

    this.setLayout(new BorderLayout());
    pnlCenter.setLayout(new BorderLayout());

    plot    = new iqstratPlot(stStruct, iStatus, normalCursor, mouseCursor);
    iWidth  = 50 + plot.getPlotWidth();
    iHeight = plot.getPlotHeight();

    panel.setLayout( new BorderLayout() );
    panel.setPreferredSize( new Dimension( iWidth, iHeight ) );

    this.add(pnlCenter,   BorderLayout.CENTER);
    pnlCenter.add(scroll, BorderLayout.CENTER);
    scroll.add(panel,     null);
    panel.add(plot,       BorderLayout.CENTER);
  }


  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    iStatus = null;
    scroll  = null;
    panel   = null;
    normalCursor = null;
    mouseCursor  = null;

    if (plot != null)
      plot.close();
    plot    = null;
  }

  /** Method refresh()
   *  <p> This method will refresh the plot
   */

  public void refresh() { if (plot != null) { plot.refresh(); } }

  /* ======================================================================= *
   * ----------------------------- GET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int  getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getTexture()
   * <p> This method will return the LAS Texture
   * @return st = LAS Texture Data List Structure
   */

  public textureListStruct getTexture()
  {
    textureListStruct st = null;

    if (plot != null)
      st = plot.getTexture();

    return (st);
  }

  /** Method getImage()
   * <p> This method will convert this plot to a buffered image
   * @return image = profile plot buffered image
   */

  public BufferedImage getImage()
  {
    BufferedImage image = null;

    if (plot != null) image = plot.getImage();

    return (image);
  }

  /** Method getDataEntryPanel()
   * <p> This method will return the Data Entry Panel Identifier
   * @return iPanel = the Data Entry Panel Identifier
   */

  public int getDataEntryPanel()   { return (plot.getDataEntryPanel()); }

  /** Method getStartingDepth()
   * <p> This method will return the starting depth for the Data
   * @return dStart = the starting depth for the Data
   */

  public double getStartingDepth() { return (plot.getStartingDepth()); }

  /** Method getEndingDepth()
   * <p> This method will return the ending depth for the Data
   * @return dEnd = the ending depth for the Data
   */

  public double getEndingDepth()   { return (plot.getEndingDepth()); }

  /* ======================================================================= *
   * ----------------------------- SET METHODS ----------------------------- *
   * ======================================================================= */

  /** Method setObservable( Observable notifier )
   * <p> This method will set the Observerable Object
   * @param notifier = the observable object
   */

  public void setObservable( Observable notifier )
  {
    if (plot != null) plot.setObservable( notifier );
  }

  /** Method setCommon( cmnStruct stCMN )
   * <p> This method will set the Global Common Data Structure
   *  @param stCMN  = the Global Common Data Structure
   */

  public void setCommon( cmnStruct stCMN )
  {
    if (plot != null) plot.setCommon( stCMN );
  }

  /** Method setTitles( String sTitle1, String sTitle2 )
   * <p> This method will set the titles for this plot
   * @param sTitle1 = The first title
   * @param sTitle2 = The second title
   */

  public void setTitles( String sTitle1, String sTitle2 )
  {
    if (plot != null)
    {
      plot.setTitles(sTitle1, sTitle2);
    }
  }

  /** Method setScale( int iScale, double dMaximum, double dMinimum )
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale( int iScale, double dMaximum, double dMinimum )
  {
    if (plot != null)
    {
      plot.setScale( iScale, dMaximum, dMinimum );
      iWidth  = 50 + plot.getPlotWidth();
      iHeight = plot.getPlotHeight();
      panel.setPreferredSize( new Dimension( iWidth, iHeight ) );
    }

    scroll.doLayout();
    panel.doLayout();
  }

  /** Method setPlotLimits()
   *  <p> This method will override the default Plot Limits
   *  @param data_las  = Array of Minimum & Maximum LAS Plot Limits
   *  @param data_core = Array of Minimum & Maximum CORE Plot Limits
   *  @param data_pf   = Array of Minimum & Maximum PfEFFER Plot Limits
   */

  public void setPlotLimits( double data_las[][], double data_core[][], double data_pf[][] )
  {
	if (plot != null)
	{
	  plot.setPlotLimits( data_las, data_core, data_pf );
	}
  }

  /* ----------------------------------------------------------------------- *
   * ------------------- Mouse Depth Location Textfield -------------------- *
   * ----------------------------------------------------------------------- */

  /** Method setTextfield( JTextField txtDepth )
   * <p> This method will set the depth textfield
   * @param txtDepth = The Depth Text Field
   */

  public void setTextfield( JTextField txtDepth )
  {
    if (plot != null) plot.setTextfield (txtDepth );
  }

  /* ----------------------------------------------------------------------- *
   * ----------------- Available Tracks Selected for view ------------------ *
   * ----------------------------------------------------------------------- */

  /** Method setSelected( int iSelected[] )
   * <p> This method will change the selected tracks
   *  @param iSelected  = The Tracks that will be displayed
   */

  public void setSelected( int iSelected[] )
  {
    if (plot != null) plot.setSelected( iSelected );
  }

  /* ----------------------------------------------------------------------- *
   * -------------------------- Data Entry Frames -------------------------- *
   * ----------------------------------------------------------------------- */

  /** Method setColorlithFrame( lithColorlithFrame panel )
   * <p> This method will set the colorlith frame
   * @param panel = colorlith frame
   *

  public void setColorlithFrame( lithColorlithFrame panel )
  {
    if (plot != null) plot.setColorlithFrame( panel );
  }
*/
  /** Method setThinTrackFrame( lasThinTrackFrame pnl )
   * <p> This method will set the Thin Track Entry Frame.
   * @param pnl = Thin Track Entry Frame
   *

  public void setThinTrackFrame( lasThinTrackFrame pnl )
  {
    if (plot != null) plot.setThinTrackFrame( pnl );
  }
*/
  /** Method setImageTrackFrame( lithImageTrackFrame pnl )
   * <p> This method will set the Log Curve colorlith selector frame
   * @param pnl = the Log Curve colorlith selector frame
   */

  public void setImageTrackFrame( lithImageTrackFrame pnl )
  {
    if (plot != null) plot.setImageTrackFrame( pnl );
  }

  /** Method setConvertFrame()
   * <p> This method will set the Convert Log Curve Frame.
   * @param pnl = the Convert Log Curve Frame
   */

  public void setConvertFrame( lasConvertFrame pnl )
  {
    if (plot != null) plot.setConvertFrame( pnl );
  }

  /* ----------------------------------------------------------------------- *
   * ---------------------------- LAS File Data ---------------------------- *
   * ----------------------------------------------------------------------- */

  /** Method setLASFileData( lasFileDataStruct st )
   * <p> This method will set the LAS File Data Structure
   * @param st - LAS File Data Structure
   */

  public void setLASFileData( lasFileDataStruct st )
  {
    if (plot != null)  plot.setLASFileData( st );
  }

  /** Method setLASTrackSize(int iLASSize)
   * <p> This method will set the LAS File Track Width
   * @param iLASSize = LAS File Track Width
   */

  public void setLASTrackSize(int iLASSize)
  {
    if (plot != null)  plot.setLASTrackSize( iLASSize );
  }

  /** Method setRockFileData( rockFileDataStruct stLASFileData )
   * <p> This method will set the Rock File Data Structure
   * @param st - Rock File Data Structure
   */

  public void setRockFileData( rockFileDataStruct st )
  {
    if (plot != null) plot.setRockFileData( st );
  }

  /** Method setLASRockColumn( lithologyListStruct st )
   * <p> This method will set the las rock column that will be shown
   * @param st = LAS Rock Column determined by user from las file data
   */

  public void setLASRockColumn( lithologyListStruct st )
  {
    if (plot != null)  plot.setLASRockColumn( st );
  }

  /** Method setLASRockColumnByRT(lithologyListStruct st)
   * <p> This method will set the las rock column that will be shown
   * @param st = LAS Rock Column determined by user from las file data
   */

  public void setLASRockColumnByRT(lithologyListStruct st)
  {
	if (plot != null) plot.setLASRockColumnByRT( st );
  }

  /** Method setColorlith( int iColorlith )
   * <p> This method will set the colorlith that will be shown
   * @param iColorlith = colorlith identifier
   */

  public void setColorlith( int iColorlith )
  {
    if (plot != null) plot.setColorlith( iColorlith );
  }

  /** Method setPHI( int iPHI )
   * <p> This method will set the Porosity to use in creating the track
   * @param iPHI - identifier for porosity track
   */

  public void setPHI( int iPHI )
  {
    if (plot != null) plot.setPHI (iPHI );
  }

  /* ------------------------------------------------------------------ *
   * ------------------- PfEFFER Plot Track Actions -------------------- *
   * ------------------------------------------------------------------ */

  /** Method setPfeffer()
   * <p> This method will set the Default PfEFFER Data
   * @param st = The PfEFFER Data Structure
   */

  public void setPfeffer( pfefferDataStruct st )
  {
    if (plot != null)  plot.setPfeffer( st );
  }

  /** Method setPfefferList()
   * <p> This method will set the PfEFFER Data List Structure
   * @param st = The PfEFFER Data List Structure
   */

  public void setPfefferList( pfefferDataListStruct st )
  {
    if (plot != null)  plot.setPfefferList( st );
  }

  /* ------------------------------------------------------------------ *
   * --------------- OHM & PHI Image Plot Track Actions --------------- *
   * ------------------------------------------------------------------ */

  /** Method setOHM( int iRt, double dRtMax, double dRtMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iRt    = Identifier representing curve to use in plotting
   * @param dRtMax = the maximum conductivity value
   * @param dRtMin = the minimum conductivity value
   */

  public void setOHM( int iRt, double dRtMax, double dRtMin )
  {
    if (plot != null) plot.setOHM( iRt, dRtMax, dRtMin );
  }

  /** Method setPHI( int iPhit, double dPhitMax, double dPhitMin )
   * <p> This method will set the Resistivity Imaging Track Variables
   * @param iPhit    = Identifier representing curve to use in plotting
   * @param dPhitMax = the maximum porosity value
   * @param dPhitMin = the minimum porosity value
   */

  public void setPHI( int iPhit, double dPhitMax, double dPhitMin )
  {
    if (plot != null) plot.setPHI( iPhit, dPhitMax, dPhitMin );
  }

  /** Method setImageData( double dataRt, double dataPhit, double dataPhi )
   * <p> This method will set the Non-Linear Data for the Non-Linear
   *     Resistivity & Porosity Image Tracks
   * @param dataRt   = The Resistivity Data for the Non-Linear Resistivity Track
   * @param dataPhit = The Porosity Data for the Non-Linear LAS-Porosity Track
   * @param dataPhi  = The Porosity Data for the Non-Linear Rock-Porosity Track
   */

  public void setImageData(
                double dataRt[][], double dataPhit[][], double dataPhi[][] )
  {
    if (plot != null) { plot.setImageData(dataRt, dataPhit, dataPhi); }
  }

  /* ----------------------------------------------------------------------- *
   * ---------------------------- Horizons Data ---------------------------- *
   * ----------------------------------------------------------------------- */

  /** Method setTops( stratListStruct st )
   * <p> This method will set the Stratigraphic Units Data
   * @param st = The Stratigraphic Units Data List Structure
   */

  public void setTops( stratListStruct st )
  {
    if (plot != null) plot.setTops( st );
  }

  /** Method setSequence( seqListStruct st )
   * <p> This method will set the sequence data list structure.
   * @param st = the Selected sequence data list structure
   */

  public void setSequence( seqListStruct st )
  {
    if (plot != null) plot.setSequence( st );
  }

  /** Method setTops( regionsListStruct st )
   * <p> This method will set the Perforations & DST Data
   * @param st = The DST & Perforation Regions Data List Structure
   */

  public void setTops( regionsListStruct st )
  {
    if (plot != null) plot.setTops( st );
  }

  /** Method setDepEnvrionment()
   * <p> This method will set the Depositional Environment list structure.
   * @param st = the Selected Depositional Environment data list structure
   */

  public void setDepEnvrionment(envListStruct st)
  {
    if (plot != null) plot.setDepEnvrionment( st );
  }

  /* ------------------------------------------------------------------ *
   * --------------------------- Brine Data --------------------------- *
   * ------------------------------------------------------------------ */

  /** Method setBrine()
   *  <p> This method will set the Brine Data.
   *  @param st = The Brine Data Structure
   */

  public void setBrine( brineListStruct st)
  {
    if (plot != null) plot.setBrine( st );
  }

  /* ----------------------------------------------------------------------- *
   * ------------------------------ Rock Data ------------------------------ *
   * ----------------------------------------------------------------------- */

  /** Method setRockData( rockDataListStruct st )
   * <p> This method will set the Rock Data
   * @param st = The Rock Data List Structure
   */

  public void setRockData( rockDataListStruct st )
  {
    if (plot != null)  plot.setRockData( st );
  }

  /** Method setRockImages( rockImagesListStruct st )
   * <p> This method will set the Rock Image Data
   * @param st = The Formation Tops Data List Structure
   */

  public void setRockImages( rockImagesListStruct st )
  {
    if (plot != null)  plot.setRockImages( st );
  }

  /** Method setRockLithology( lithologyListStruct st )
   * <p> This method will set the rock column that will be shown
   * @param st = Rock Column determined by user from rock data.
   *

  public void setRockLithology( lithologyListStruct st )
  {
    if (plot != null)  plot.setRockLithology( st );
  }
*/
  /** Method setTexture( textureListStruct st )
   * <p> This method will set the rock texture that will be shown
   * @param st = Rock Texture determined by user from rock data.
   *

  public void setTexture( textureListStruct st )
  {
    if (plot != null) plot.setRockTexture( st );
  }
*/
  /** Method setRockColumn( rockColumnListStruct st)
   *  <p> This method will set the rock column lithology
   *  @param st = Rock Column List Data Structure
   */

  public void setRockColumn( rockColumnListStruct st )
  {
	if (plot != null) plot.setRockColumn( st );
  }

  /** Method setPHI( phiListStruct st )
   * <p> This method will set the Porosity Data
   * @param st = The Porosity Data List Structure
   */

  public void setPHI( phiListStruct st )
  {
    if (plot != null) plot.setPHI( st );
  }

  /** Method setFossils( fossilListStruct st )
   * <p> This method will set the Fossil Data
   * @param st = The Fossil Data List Structure
   */

  public void setFossils( fossilListStruct st )
  {
    if (plot != null) plot.setFossils( st );
  }

  /** Method setBioStrat( bioStratListStruct st )
   * <p> This method will set the Bio Stratigraphy Data
   * @param st = The Bio Stratigraphy Data List Structure
   */

  public void setBioStrat( bioStratListStruct st )
  {
    if (plot != null) plot.setBioStrat( st );
  }

  /** Method setStructures( sedimentaryListStruct st)
   * <p> This method will set the sedimentary structures Data
   * @param st = The sedimentary structures Data List Structure
   */

  public void setStructures( sedimentaryListStruct st )
  {
    if (plot != null)  plot.setStructures( st );
  }

  /** Method setRockColorData( rockColorListStruct st )
   * <p> This method will set the Rock Color Data.
   * @param st = the Rock Color List Data Structure
   */

  public void setRockColorData( rockColorListStruct st )
  {
    if (plot != null) plot.setRockColorData( st );
  }

  /** Method setRemarks( iqstratRemarkListStruct st )
   * <p> This method will set the Fossil Data
   * @param st = The Remarks Data List Structure
   */

  public void setRemarks( iqstratRemarkListStruct st )
  {
    if (plot != null)  plot.setRemarks( st );
  }

  /** Method modifyWidths( int iBio, int iSed, int iBioS  )
   * <p> This method will modify the plot track width depending on the
   *     maximum number of fossils or sedimentary structures per depth.
   * @param iBio  = The total number of fossils per depth per well
   * @param iSed  = The total number of structures per depth per well
   * @param iBioS = The total number of structures per depth per well
   */

  public void modifyWidths( int iBio, int iSed, int iBioS  )
  {
    if (plot != null)
    {
      plot.modifyWidths( iBio, iSed, iBioS );
      iWidth  = 50 + plot.getPlotWidth();
      iHeight = plot.getPlotHeight();
      panel.setPreferredSize( new Dimension( iWidth, iHeight ) );
    }

    scroll.doLayout();
    panel.doLayout();
  }
}

/*
 *  @version 1.1 09/04/2007
 *  @author  John Victorine
 */
