/*
 * @iqstratNotesCSVFrame.java Version 1.1 07/22/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package iqstrat.gui;

import java.awt.*;
import java.awt.event.*;
import java.util.Observable;
import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;
import java.math.BigDecimal;

import iqstrat.iqstratStruct;
import iqstrat.iqstratHeadersStruct;
import iqstrat.iqstratRemarkStruct;
import iqstrat.iqstratRemarkListStruct;
import iqstrat.io.iqstratNotesCSVFile;

import horizon.strat.stratListStruct;

/** Class iqstratNotesCSVFrame
 *  <p> This Class will allow the user to load data files from their PC
 *
 *  @version 1.1 07/22/2008
 *  @author  John R. Victorine
 */

public class iqstratNotesCSVFrame extends JFrame implements ActionListener
{
  // Input Variables

  private Observable    notifier    = null;  // Observable Object
  private String        sReadMethod = null;  // Read Method Should be File only
  private String        sDirectory  = null;  // Directory Path to File
  private String        sFilename   = null;  // Filename of File
  private iqstratStruct stStruct    = null;  // IQSTRAT Data Structure
  private iqstratHeadersStruct stHeader = null; // Headers Data Structure

  // Global Static Variables

  // Parse Depth by Identifier

  public static final int _BY_DEPTH  = 0;
  public static final int _BY_THICK  = 1;

  // Depth units Identifier

  public static final int _FEET      = 0;
  public static final int _INCHES    = 1;
  public static final int _METERS    = 2;
  public static final int _CENTIMETER= 3;

  // Global Frame Variables

  private iqstratNotesCSVFile     pFILE   = new iqstratNotesCSVFile();
  private iqstratRemarkListStruct stList  = null;
  private stratListStruct         stStrat = null;

  private int    iTotal  = 0;         // Total number of lines in File
  private String sText[] = null;      // Contents of File

  private int    iRow    = 3;         // Start Reading at Row 3
  private double depth   = 0.0;       // Initial Bedding Thickness Start Depth
  private String sDelim  = ",";       // Initial Delimiter to parse depth data
  private int    iDepth  = _BY_DEPTH; // Parse By Absolute Depth
  private int    iUnit   = _FEET;     // Depth is in Feet

  // Global Frame Widgets

  // Global Frame Button Widgets

  private JButton btnParse = new JButton();
  private JButton btnClose = new JButton();
  private JButton btnHelp  = new JButton();

  // Display Comments/Remarks/Notes File Contents

  private JTextArea txtArea = new JTextArea();

  // Depth Range Widgets

  private JRadioButton rbByUser      = new JRadioButton();
  private JRadioButton rbByThickness = new JRadioButton();

  // Bedding Thickness Depth Range Widget

  private JTextField txtDepth = new JTextField();

  // Depth Units Widgets

  private JRadioButton rbFt = new JRadioButton();
  private JRadioButton rbIn = new JRadioButton();
  private JRadioButton rbM  = new JRadioButton();
  private JRadioButton rbCm = new JRadioButton();

  // File Widgets

  private JTextField txtStart = new JTextField();
  private JTextField txtDelim = new JTextField();

  /** Constructor iqstratNotesCSVFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier    = Observable
   *  @param sReadMethod = File Read Method
   *  @param sDirectory  = The Directory Path to the File
   *  @param sFilename   = The Filename
   *  @param stStruct    = IQSTRAT Data Structure
   *  @param stHeader    = Headers Data Structure
   */

  public iqstratNotesCSVFrame( Observable    notifier,
                               String        sReadMethod,
                               String        sDirectory,
                               String        sFilename,
                               iqstratStruct stStruct,
                               iqstratHeadersStruct stHeader )
  {
    try
    {
      this.notifier    = notifier;
      this.sReadMethod = sReadMethod;
      this.sDirectory  = sDirectory;
      this.sFilename   = sFilename;
      this.stStruct    = stStruct;
      this.stHeader    = stHeader;

      load();

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int    iCount       = 0;
    String sTemp        = "";
    JPanel pnlTop       = new JPanel();
    JPanel pnlBase      = new JPanel();
    JPanel pnlCenter    = new JPanel();
    JPanel pnlFile      = new JPanel();
    JPanel pnlDepthInfo = new JPanel();
    JPanel pnlByDepth   = new JPanel();
    JPanel pnlDepth     = new JPanel();
    JPanel pnlUnits     = new JPanel();
    JPanel pnlRow       = new JPanel();
    JPanel pnlDelim     = new JPanel();

    JScrollPane scroll  = new JScrollPane();

    ButtonGroup groupByDepth = new ButtonGroup();
    ButtonGroup groupUnits   = new ButtonGroup();

    TitledBorder titledBorderByDepth = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Depth Position By:");

    TitledBorder titledBorderStart = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Start Bedding at:");

    TitledBorder titledBorderUnits = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Depth Data Units (will be converted to feet)");

    TitledBorder titledBorderRow = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Start at Row");

    TitledBorder titledBorderDelim = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Delimiters, i.e. ,;:()");

    titledBorderByDepth.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderStart.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderUnits.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderRow.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderDelim.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setTitle("Parse Comments/Remarks/Notes ASCII Text File");
    this.getContentPane().setLayout(new BorderLayout());

    // Global Frame Buttons Panel

    pnlBase.setBorder(BorderFactory.createEtchedBorder());

    btnParse.setFont(new java.awt.Font("Dialog", 1, 11));
    btnParse.setPreferredSize(new Dimension(95, 25));
    btnParse.setText("Parse Data");
    btnParse.addActionListener(this);

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setPreferredSize(new Dimension(95, 25));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    btnHelp.setFont(new java.awt.Font("Dialog", 1, 11));
    btnHelp.setPreferredSize(new Dimension(95, 25));
    btnHelp.setText("Help");
    btnHelp.addActionListener(this);

    // Display File Contents Panel

    pnlCenter.setLayout(new BorderLayout());

    if (iTotal > 0)
    {
      for (int i=0; i<iTotal; i++)
      {
        if (sText[i].length() > 0)
        {
          iCount++;
          sTemp = new String(sTemp + sText[i] + " \n");
        }
      }
    }
    iTotal = iCount;

    txtArea.setText(sTemp);
    txtArea.setCaretPosition(0);
    txtArea.setLineWrap(true);

    // Top Panel

    pnlTop.setLayout(new BorderLayout());

    // Parse File By Depth Panel

    pnlByDepth.setBorder(titledBorderByDepth);
    pnlByDepth.setLayout(new GridLayout());

    // -- User defines depth ranges for each line

    rbByUser.setFont(new java.awt.Font("Dialog", 1, 11));
    rbByUser.setHorizontalAlignment(SwingConstants.CENTER);
    rbByUser.setSelected(true);
    rbByUser.setText("User Defined");
    rbByUser.addActionListener(this);

    // -- depth range is determined by bedding thickness

    rbByThickness.setFont(new java.awt.Font("Dialog", 1, 11));
    rbByThickness.setHorizontalAlignment(SwingConstants.CENTER);
    rbByThickness.setText("Bedding Thickness");
    rbByThickness.addActionListener(this);

    // Depth Information Panel

    pnlDepthInfo.setLayout(new BorderLayout());

    // -- Starting Depth

    pnlDepth.setLayout(new BorderLayout());
    pnlDepth.setBorder(titledBorderStart);
    pnlDepth.setPreferredSize(new Dimension(110, 48));

    txtDepth.setText("0.0");
    txtDepth.setHorizontalAlignment(SwingConstants.TRAILING);
    txtDepth.addFocusListener(new iqstratNotesCSVFrameFocusAdapter(this));

    // -- Depth Units

    pnlUnits.setBorder(titledBorderUnits);
    pnlUnits.setLayout(new GridLayout());

    rbFt.setFont(new java.awt.Font("Dialog", 1, 11));
    rbFt.setHorizontalAlignment(SwingConstants.CENTER);
    rbFt.setSelected(true);
    rbFt.setText("Feet");
    rbFt.addActionListener(this);

    rbIn.setFont(new java.awt.Font("Dialog", 1, 11));
    rbIn.setHorizontalAlignment(SwingConstants.CENTER);
    rbIn.setText("Inch");
    rbIn.addActionListener(this);

    rbM.setEnabled(false);
    rbM.setFont(new java.awt.Font("Dialog", 1, 11));
    rbM.setHorizontalAlignment(SwingConstants.CENTER);
    rbM.setText("Meter");
    rbM.addActionListener(this);

    rbCm.setEnabled(false);
    rbCm.setFont(new java.awt.Font("Dialog", 1, 11));
    rbCm.setHorizontalAlignment(SwingConstants.CENTER);
    rbCm.setText("Centimeter");
    rbCm.addActionListener(this);

    // File Parsing Parameters Panel

    pnlFile.setLayout(new BorderLayout());

    // Start Parsing File Data at Row Panel

    pnlRow.setLayout(new BorderLayout());
    pnlRow.setBorder(titledBorderRow);
    pnlRow.setPreferredSize(new Dimension(120, 48));

    txtStart.setText("3");
    txtStart.setHorizontalAlignment(SwingConstants.TRAILING);
    txtStart.addFocusListener(new iqstratNotesCSVFrameFocusAdapter(this));

    // Separate Terms by Delimiters Panel

    pnlDelim.setLayout(new BorderLayout());
    pnlDelim.setBorder(titledBorderDelim);
    pnlDelim.setPreferredSize(new Dimension(120, 48));

    txtDelim.setText(",");
    txtDelim.addFocusListener(new iqstratNotesCSVFrameFocusAdapter(this));

    // Attach Widgets to Frame

    this.getContentPane().add( pnlTop,    BorderLayout.NORTH );
    pnlTop.add( pnlFile,        BorderLayout.SOUTH );
    pnlFile.add( pnlRow,        BorderLayout.EAST );
    pnlRow.add( txtStart,       BorderLayout.CENTER );

    pnlFile.add( pnlDelim,      BorderLayout.WEST );
    pnlDelim.add( txtDelim,     BorderLayout.NORTH );

    pnlTop.add( pnlDepthInfo,   BorderLayout.CENTER );

    pnlDepthInfo.add( pnlDepth, BorderLayout.WEST );
    pnlDepth.add( txtDepth,     BorderLayout.CENTER );

    pnlDepthInfo.add( pnlUnits, BorderLayout.CENTER );
    pnlUnits.add(rbFt, null);
    pnlUnits.add(rbIn, null);
    pnlUnits.add(rbM, null);
    pnlUnits.add(rbCm, null);

    groupUnits.add( rbFt );
    groupUnits.add( rbIn );
    groupUnits.add( rbM );
    groupUnits.add( rbCm );

    pnlTop.add(pnlByDepth, BorderLayout.NORTH);
    pnlByDepth.add( rbByUser, null );
    pnlByDepth.add( rbByThickness, null );

    groupByDepth.add( rbByUser );
    groupByDepth.add( rbByThickness );

    this.getContentPane().add( pnlBase,   BorderLayout.SOUTH );
    pnlBase.add( btnParse,  null );
    pnlBase.add( btnClose, null );
    pnlBase.add( btnHelp,  null );

    this.getContentPane().add( pnlCenter, BorderLayout.CENTER );
    pnlCenter.add( scroll, BorderLayout.CENTER );
    scroll.getViewport().add( txtArea, null );

    // Display the Frame

//    this.setSize(new Dimension(500, 650));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();
    this.setSize(new Dimension(600, d.height-50));
    this.setLocation((d.width  - this.getSize().width) / 2, 5);
//                     (d.height - this.getSize().height) / 2);

    this.setResizable(true);
    this.setVisible(true);
  }

  /* ===================================================================== *
   * --------------------------- GET METHODS ----------------------------- *
   * ===================================================================== */

  /** Method getHeader()
   * <p> This method will return the Headers Data Structure
   * @return stHeader = the Headers Data Structure
   */

  public iqstratHeadersStruct getHeader() { return (stHeader); }

  /** Method getRemarks()
   * <p> This method will return the Remarks Data List Structure
   * @return stList = the Remarks Data List Structure
   */

  public iqstratRemarkListStruct getRemarks() { return (stList); }

  /** Method getTops()
   * <p> This method will return the Tops Data List Structure
   * @return stStrat = the Tops Data List Structure
   */

  public stratListStruct getTops() { return (stStrat); }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setBedding()
   * <p> This method will set the Bedding Thickness Variables on or off
   *     depending on the setting of the panel
   * @param iDepth = The Parse Depth File Indicator
   */

  private void setBedding(int iDepth)
  {
    this.iDepth = iDepth;

    switch (iDepth)
    {
      case _BY_DEPTH:
        txtDepth.setEditable(false); // Bedding Starting Depth
        txtDepth.setText("0.0");
        break;

      case _BY_THICK:
        txtDepth.setEditable(true); // Bedding Starting Depth
        txtDepth.setText(""+depth);
        break;
    }
  }

  /* ===================================================================== *
   * -------------------- File Read & Parse Method ----------------------- *
   * ===================================================================== */

  /** Method load()
   * <p> This method will load the read and parse the comments file
   */

  private void load()
  {
	int iMethod = util.utilFileIO.DIRECTORY;

	if (sReadMethod.equals("URL"))
	  iMethod = util.utilFileIO.URL_FILE;

    pFILE.Open(iMethod, sDirectory, sFilename);
    pFILE.count();
    pFILE.Close();

    pFILE.Open(iMethod, sDirectory, sFilename);
    pFILE.parseData();
    pFILE.Close();

    iTotal = pFILE.getCount();     // Total number of lines in File
    sText  = pFILE.getContents();  // Contents of File
  }

  /** Method parse()
   * <p> This method will parse the comments file into the comments data
   *     structure
   */

  private void parse()
  {
    int i=0;
    int j=0;
    int    iStart   = 0;
    String sDesc    = "";
    String tokens[] = null;
    String nos[]    = null;
    String sName    = "";
    String sDelimiter = "";
    int    len      = 0;
    int    len2     = 0;
    int    iFound   = -1;
    int    iNew     = 0;
    double dtemp    = 0.0;
    double depth_s  = 0.0;
    double depth_e  = 0.0;
    double dThick   = 0.0;
    double dRange[] = { 0.0, 0.0 };
    double dStart   = 0.0;
    double dEnd     = 0.0;

    iqstratRemarkStruct st = null;
    sName = cmn.cmnString.UniqueName();

    if (iDepth == _BY_THICK) depth_s = depth;

    if (stList != null)
      stList.delete();
    stList = null;

    stList = new iqstratRemarkListStruct();

    sDelimiter = new String("["+sDelim+"]+");

    sText = getText(); // Get the Text from the Text Area

    if (stHeader == null)
      stHeader = new iqstratHeadersStruct();

    if ((iRow-1) > 0)
    {
      if (stHeader.sName != null)
      {
        if (stHeader.sName.length() == 0)
          stHeader.sName = new String(sText[0]);
      }
    }

    if ((iRow-1) > 1)
    {
      if (stHeader.sLocation != null)
      {
        if (stHeader.sLocation.length() == 0)
          stHeader.sLocation = new String(sText[1]);
      }
    }

    for (i=0; i<iRow-1; i++)
    {
      if (stHeader.sComments != null)
        stHeader.sComments = new String( stHeader.sComments + "\n" + sText[i] );
    }

    if (sText != null)
    {
      for (i=iRow-1; i<iTotal; i++)
      {
        if (sDelim != null)
        {
          tokens = sText[i].split(sDelimiter);
          len    = tokens.length;

          if (!iqstrat.io.iqstratNotesCSVFile.isBlankLine(tokens));
          {
            iNew = 0;
            switch (iDepth)
            {
              case _BY_DEPTH:
                dRange = iqstrat.io.iqstratNotesCSVFile.getDepthRange( tokens );

                if (dRange[1] > 0.0)
                {
                  iNew = 1;
                  sDesc = new String(sText[i]);
                }

                for (j=0; j<2; j++)
                  dRange[j] = convertToFeet( iUnit, dRange[j] );
                break;

              case _BY_THICK:
                dThick = iqstrat.io.iqstratNotesCSVFile.getBeddingThickness( tokens );

                if (dThick > 0.0)
                {
                  iNew = 1;
                  sDesc = new String(sText[i]);
                }

                dThick = convertToFeet( iUnit, dThick );
                break;
            }

            if (iNew == 1)
            {
              if (st != null)
              {
                stList = iqstrat.iqstratRemarkUtility.add(st, stList);

                if (iStart == 0)
                {
                  stList.depthStart = st.depthStart; // Starting Depth
                  stList.depthEnd   = st.depthEnd;   // Ending Depth
                  stList.source =
                      new String("Geologist Report"); // Source
                  stList.sRef   = new String("Log Depth"); // Depth Reference

                  iStart = 1;
                }

                st.delete();
                st = null;
              }

              st       = new iqstratRemarkStruct();
              st.sKEY  = new String(sName+"_"+i);  // Identifier
              st.sText = new String(sDesc); // Remarks at that depth

              if (st.sText.length() < 41)
                st.sNote = new String(st.sText);
              else
                st.sNote = new String(st.sText.substring(0, 40));

              switch (iDepth)
              {
                case _BY_DEPTH:
                  st.depthStart = dRange[0];  // Starting Depth
                  st.depthEnd   = dRange[1];  // Ending Depth
//                  st.dThickness = Math.abs(st.depthEnd - st.depthStart);
                  st.dThickness = subtract_BD( st.depthEnd, st.depthStart );
                  break;

                case _BY_THICK:
                  st.depthStart = depth_s;          // Starting Depth
                  st.depthEnd   = add_BD( depth_s, dThick );
//                  st.depthEnd   = depth_s + dThick; // Ending Depth
                  st.dThickness = dThick;
                  depth_s       = st.depthEnd;
                  break;
              }

              dStart = st.depthEnd;
              dEnd   = 0.0;

              // If the tops get absorbed into the Remarks List

              for (int k=0; k<len; k++)
              {
                if (horizon.strat.stratUtility.isKGS( tokens[k], stStruct.stKGS ))
                {
                  stStrat = horizon.strat.stratUtility.parse( tokens[k],
                                                              dStart,
                                                              dEnd,
                                                              stStrat,
                                                              stStruct.stKGS );
                }
              }
            }
            else if (horizon.strat.stratUtility.isKGS( sText[i], stStruct.stKGS ))
            {
              stStrat = horizon.strat.stratUtility.parse( sText[i],
                                                          dStart,
                                                          dEnd,
                                                          stStrat,
                                                          stStruct.stKGS );
            }
            else
            {
              sDesc = new String(sDesc + " \n" + sText[i]);
              if (st != null)
              {
                st.sText = new String(sDesc); // Remarks at that depth
              }
            }
          }  // END Test for Blank Lines

          tokens = null;
          nos    = null;
        }  // END sDelim != null
      }

      if (st != null)
      {
        stList = iqstrat.iqstratRemarkUtility.add(st, stList);
        if (iStart == 0)
        {
          stList.depthStart = st.depthStart; // Starting Depth
          stList.depthEnd   = st.depthEnd;   // Ending Depth
          stList.source =
              new String("Geologist Report"); // Source
          stList.sRef   = new String("Log Depth"); // Depth Reference

          iStart = 1;
        }

        if (stList.depthEnd < st.depthEnd)
          stList.depthEnd = st.depthEnd;   // Ending Depth

        sDesc = new String("");
        st.delete();
        st = null;
      }
    }

    if (stList != null)
    {
      for (i=0; i<stList.iCount; i++)
      {
        stList.stItem[i].sText =
            new String( stList.stItem[i].sText.replace('\t', ' ') );
        stList.stItem[i].sText =
            new String( stList.stItem[i].sText.replace('\n', ' ') );
        stList.stItem[i].sText =
            new String(las3.las3Parse.removeQuotes( stList.stItem[i].sText ));
      }
    }

    if (stList != null)
    {
      if (notifier != null)
        notifier.notifyObservers(new String("CSV - Comments"));
    }
  }

  /** Method add_BD()
   *  <p> This method will compute a sum using Bid Decimal Math Package
   *  @param  a1 = value a
   *  @param  a2 = value b
   *  @return a  = value added
   */

  public static double add_BD( double a1, double a2 )
  {
	double     a  = 0.0;
	BigDecimal BDa  = new BigDecimal( "0.0" );
	BigDecimal BDa1 = new BigDecimal( new String(""+a1) );
	BigDecimal BDa2 = new BigDecimal( new String(""+a2) );

    BDa = BDa1.add( BDa2 );

	a = BDa.doubleValue();

	return (a);
  }

  /** Method subtract_BD()
   *  <p> This method will compute a sum using Bid Decimal Math Package
   *  @param  a1 = value a
   *  @param  a2 = value b
   *  @return a  = value added
   */

  public static double subtract_BD( double a1, double a2 )
  {
	double     a  = 0.0;
	BigDecimal BDa  = new BigDecimal( "0.0" );
	BigDecimal BDa1 = new BigDecimal( new String(""+a1) );
	BigDecimal BDa2 = new BigDecimal( new String(""+a2) );

    BDa = BDa1.subtract( BDa2 );

	a = BDa.doubleValue();

	return (a);
  }

  /** Method getText()
   * <p> This method will retrieve the Text Area data
   * @return tokens = the text area subdivided into lines
   */

  private String[] getText()
  {
    String sTemp = txtArea.getText();
    String tokens[] = null;
    String sDelimiter = new String("[\n]+");

    tokens = sTemp.split(sDelimiter);

    return (tokens);
  }

  /** Method convertToFeet()
   * <p> This method will convert depth to feet
   * @param  iUnit = the original depth unit identifier
   * @param  data  = the depth that needs to be converted
   * @return data  = the depth converted to feet
   */

  public static double convertToFeet(int iUnit, double data)
  {
    switch (iUnit)
    {
      case _FEET:
        break;
      case _INCHES:
        data = data / 12.0;
        break;
      case _METERS:
        break;
      case _CENTIMETER:
        break;
    }

    return (data);
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method close()
   * <p> This method will set global variables to null in order to force JAVA
   *     to free unwanted memory.
   */

  public void close()
  {
    notifier    = null;
    sReadMethod = null;
    sDirectory  = null;
    sFilename   = null;
    stStruct    = null;  // IQSTRAT Data Structure
    stHeader    = null;

    if (stList != null)
      stList.delete();
    stList  = null;

    if (stStrat != null)
      stStrat.delete();
    stStrat     = null;

    sText       = null;
    sDelim      = null;

    if (pFILE != null)
      pFILE.delete();
    pFILE = null;

    // Global Frame Buttons

    btnParse = null;  // Read & Parse File
    btnClose = null;  // Close Dialog
    btnHelp  = null;  // Help on this dialog

    // Display Comments/Remarks/Notes File Contents

    txtArea = null;

    // Depth Range Widgets

    rbByUser      = null;
    rbByThickness = null;

    // Bedding Thickness Depth Range Widget

    txtDepth = null;

    // Depth Units Widgets

    rbFt = null;
    rbIn = null;
    rbM  = null;
    rbCm = null;

    // File Widgets

    txtStart = null;
    txtDelim = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    // Depth Range Widgets

    if (event.getSource() == rbByUser)       { setBedding(_BY_DEPTH); }
    if (event.getSource() == rbByThickness)  { setBedding(_BY_THICK); }

    // Depth Units Widgets

    if (event.getSource() == rbFt)  { iUnit = _FEET; }
    if (event.getSource() == rbIn)  { iUnit = _INCHES; }
    if (event.getSource() == rbM)   { iUnit = _METERS; }
    if (event.getSource() == rbCm)  { iUnit = _CENTIMETER; }

    // Global Frame Buttons

    if (event.getSource() == btnParse)  { parse(); }
    if (event.getSource() == btnHelp)
    {
      util.BrowserControl.displayURL( cmn.cmnStruct.PC_GEO );
    }
    if (event.getSource() == btnClose) { close(); }
  }

  /** METHOD focusLost()
   *  <p> This Method will handle the lost focus event for the text fields.
   *  @param e      = Focus Lost Event
   */

  protected void focusLost(FocusEvent e)
  {
    int    i        = 0;
    int    iNumeric = 0;
    String sMessage = new String("");
    String sTemp    = new String("");

    if (e.getSource() == txtStart)
    {
      iNumeric = 1;
      sTemp    = txtStart.getText();
      sMessage = new String("Starting Row Number is a Numeric Field");
    }

    if (e.getSource() == txtDepth)
    {
      iNumeric = 1;
      sTemp    = txtDepth.getText();
      sMessage = new String("Starting Depth is a Numeric Field");
    }

    if (iNumeric == 1)
    {
      if (!cmn.cmnString.isNumeric(sTemp))
      {
        JOptionPane.showMessageDialog((Component) null,
                                       sMessage,
                                       "ERROR",
                                       JOptionPane.ERROR_MESSAGE);
      }
      else
      {
        if (e.getSource() == txtStart)
        {
          iRow = cmn.cmnString.stringToInt(txtStart.getText());
        }

        if (e.getSource() == txtDepth)
        {
          depth = cmn.cmnString.stringToDouble(txtDepth.getText());
        }
      }
    }
    else
    {
      if (e.getSource() == txtDelim)
      {
        sDelim = new String( txtDelim.getText() );
      }
    }
  }
}

/** CLASS iqstratNotesCSVFrameFocusAdapter()
  *  <p> This Method will handle Actions when focus is lost on a textfield
  */

class iqstratNotesCSVFrameFocusAdapter extends java.awt.event.FocusAdapter
{
  iqstratNotesCSVFrame adaptee;

  iqstratNotesCSVFrameFocusAdapter(iqstratNotesCSVFrame adaptee)
  {
    this.adaptee = adaptee;
  }

  public void focusLost(FocusEvent e) { adaptee.focusLost(e); }
}

/*
 *  @version 1.1 07/22/2008
 *  @author  John Victorine
 */
