/*
 * @stratUtility.java Version 1.1 10/17/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat;

import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

import horizon.regions.regionsListStruct;

import iqstrat.iqstratHeadersStruct;
import kgs.kgsMeasSectListStruct;

/** Class stratUtility
 *  <p> This Class will provide basic utilities for the Stratigraphic Unit
 *      data structures.
 *
 *  @version 1.1 10/17/2008
 *  @author  John R. Victorine
 */

public class stratUtility
{
  public static final int TOTAL_SOURCES  = 20;

  /** Method addHeaderData()
   * <p> This method will add the header data to the strat List Structure
   * @param  stHeader = The Headers Data Structure
   * @param  st       = The Regions List Data Structure
   * @return st       = The Regions List Data Structure.
   */

  public static stratListStruct addHeaderData( iqstratHeadersStruct stHeader,
                                               stratListStruct st )
  {
    if ((stHeader != null) && (st != null))
    {
      // Identification Information

      st.sKID   = new String(stHeader.sKID);   // Location Header KID
      st.sKEY   = new String(stHeader.sKEY);   // Primary Key created on user
      st.iType  = stHeader.iType;
      st.sAPI   = new String(stHeader.sAPI);   // API-Number of Location
      st.sName  = new String(stHeader.sName);  // Well Name or Outcrop Name
      st.status = new String(stHeader.status); // Status of well or Outcrop Type

      // XY Position Information

      st.dLatitude  = stHeader.dLatitude;   // Latitude
      st.dLongitude = stHeader.dLongitude;  // Longitude

      // Z Position - Depth, Elevation

      st.depth = stHeader.depth; // Total Depth of Well or Height of Outcrop
      st.dGL   = stHeader.dGL;   // Ground Level
    }

    return (st);
  }

  /** Method copyList()
   * <p> This method will copy one stratigraphic unit list structure to another
   * @param  stOld = Old stratigraphic unit list data structure
   * @return stNew = New stratigraphic unit list data structure
   */

  public static stratListStruct copyList( stratListStruct stOld )
  {
    int i,j;
    stratListStruct stNew = null;

    if (stOld != null)
    {
      stNew        = new stratListStruct();

      stNew.iSource = stOld.iSource;

      // Identification Information

      stNew.sKID   = new String(stOld.sKID);   // Location Header KID
      stNew.sKEY   = new String(stOld.sKEY);   // Primary Key created on user
      stNew.iType  = stOld.iType;
      stNew.sAPI   = new String(stOld.sAPI);   // API-Number of Location
      stNew.sName  = new String(stOld.sName);  // Name
      stNew.status = new String(stOld.status); // Status

      // XY Position Information

      stNew.dLatitude  = stOld.dLatitude;  // Latitude
      stNew.dLongitude = stOld.dLongitude; // Longitude

      // Z Position - Depth, Elevation

      stNew.depth = stOld.depth; // Total Depth of Well or Height of Outcrop
      stNew.dGL   = stOld.dGL;   // Ground Level

      stNew.sKGS  = new String( stOld.sKGS );      // KGS Saved Data Indicator

      // LAS 3 Required Fields

      stNew.sTOPS  = new String( stOld.sTOPS );    // Formation Source
      stNew.sTOPDR = new String( stOld.sTOPDR );   // Tops Depth Reference

      // LAS 3 File Parameters

      stNew.iParamRows = stOld.iParamRows;    // Total number of rows
      stNew.iParamCols = stOld.iParamCols;    // Total number of columns
      stNew.sParams    = new String[stNew.iParamRows][stNew.iParamCols];

      for (i=0; i<stNew.iParamRows; i++)
      {
        for (j=0; j<stNew.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stNew.sParams[i][j] = new String( stOld.sParams[i][j] );
        }
      }

      if (stOld.iCount > 0)
      {
        stNew.iCount = stOld.iCount;
        stNew.stItem = new stratStruct[stOld.iCount];

        for (i=0; i<stOld.iCount; i++)
        {
          stNew.stItem[i] = copy(stOld.stItem[i]);
        }
      }
    }

    return (stNew);
  }

  /** Method transfer()
   * <p> This method will copy one structure to another
   * @param  stOld = the Old List Structure
   * @return stNew = the New List structure
   */

  public static stratListStruct transfer(stratListStruct stOld)
  {
    stratListStruct stNew = null;

    if (stOld != null)
    {
	  stNew = copyList( stOld );

	  stOld.delete();
	  stOld = null;
	}

    return (stNew);
  }

  /** Method isUnit()
   * <p> This method will test to see if the stratigraphic unit is present in
   *     list or not
   * @param  stStrat = The stratigraphic unit Data Structure
   * @param  st      = The stratigraphic unit Data List Structure
   * @return bValue  = true the stratigraphic unit exist, false it does not
   */

  public static boolean isUnit( stratStruct stStrat, stratListStruct st )
  {
    boolean bValue = false;

    if (st != null)
    {
      if (stStrat != null)
      {
        for (int i=0; i<st.iCount; i++)
        {
          if (stStrat.sName.equals(st.stItem[i].sName))
            bValue = true;
        }
      }
    }

    return (bValue);
  }

  /** Method add()
   * <p> This method will add the stratigraphic unit to an existing list
   * @param  stStrat = The Region Data Structure
   * @param  st      = The Old stratigraphic unit List Data Structure
   * @return st      = The new stratigraphic unit List Data Structure.
   */

  public static stratListStruct add( stratStruct stStrat,
                                     stratListStruct st )
  {
    int    i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    stratListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new stratListStruct();
    stTemp.stItem = new stratStruct[iRecords];

    if (st != null)
    {
      iSource      = st.iSource;

      stTemp.sKGS  = new String( st.sKGS );     // KGS Saved Data Indicator

      // LAS 3 Required Fields

      stTemp.sTOPS  = new String( st.sTOPS );    // Formation Source
      stTemp.sTOPDR = new String( st.sTOPDR );   // Tops Depth Reference

      // LAS 3 File Parameters

      stTemp.iParamRows = st.iParamRows;    // Total number of rows
      stTemp.iParamCols = st.iParamCols;    // Total number of columns
      stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

      for (i=0; i<stTemp.iParamRows; i++)
      {
        for (j=0; j<stTemp.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stTemp.sParams[i][j] = new String( st.sParams[i][j] );
        }
      }

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new stratStruct();
    stTemp.stItem[iCount] = copy(stStrat);
    iCount++;

    stTemp.iCount = iCount;

    st        = new stratListStruct();
    st.stItem = new stratStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;

    st.sKGS  = new String( stTemp.sKGS );      // KGS Saved Data Indicator

    // LAS 3 Required Fields

    st.sTOPS  = new String( stTemp.sTOPS );    // Formation Source
    st.sTOPDR = new String( stTemp.sTOPDR );   // Tops Depth Reference

    // LAS 3 File Parameters

    st.iParamRows = stTemp.iParamRows;    // Total number of rows
    st.iParamCols = stTemp.iParamCols;    // Total number of columns
    st.sParams    = new String[st.iParamRows][st.iParamCols];

    for (i=0; i<st.iParamRows; i++)
    {
      for (j=0; j<st.iParamCols; j++)
      {
        // Array holding the Parameter Definitions
        st.sParams[i][j] = new String( stTemp.sParams[i][j] );
      }
    }

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method ReplaceWith()
   * <p> This method will Replace an existing stratigraphic unit if it exists
   *     with the new stratigraphic unit in an existing list
   * @param  stStrat = The Region Data Structure
   * @param  st      = The Old stratigraphic unit List Data Structure
   * @return st      = The new stratigraphic unit List Data Structure.
   */

  public static stratListStruct ReplaceWith( stratStruct stStrat,
                                             stratListStruct st )
  {
    int    i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iSource  = -1;
    stratListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new stratListStruct();
    stTemp.stItem = new stratStruct[iRecords];

    if (st != null)
    {
	  iSource      = st.iSource;

      stTemp.sKGS  = new String( st.sKGS );      // KGS Saved Data Indicator

      // LAS 3 Required Fields

      stTemp.sTOPS  = new String( st.sTOPS );    // Formation Source
      stTemp.sTOPDR = new String( st.sTOPDR );   // Tops Depth Reference

      // LAS 3 File Parameters

      stTemp.iParamRows = st.iParamRows;    // Total number of rows
      stTemp.iParamCols = st.iParamCols;    // Total number of columns
      stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

      for (i=0; i<st.iParamRows; i++)
      {
        for (j=0; j<st.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stTemp.sParams[i][j] = new String( st.sParams[i][j] );
        }
      }

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            if ((!st.stItem[i].sName.equals(stStrat.sName)) &&
                (!st.stItem[i].sName2.equals(stStrat.sName2)))
            {
              stTemp.stItem[iCount] = copy(st.stItem[i]);
              iCount++;
            }
          }
        }
      }

      st.delete();
    }

    stTemp.stItem[iCount] = new stratStruct();
    stTemp.stItem[iCount] = copy(stStrat);
    iCount++;

    stTemp.iCount = iCount;

    st        = new stratListStruct();
    st.stItem = new stratStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;

    st.sKGS   = new String( stTemp.sKGS );     // KGS Saved Data Indicator

    // LAS 3 Required Fields

    st.sTOPS  = new String( stTemp.sTOPS );    // Formation Source
    st.sTOPDR = new String( stTemp.sTOPDR );   // Tops Depth Reference

    // LAS 3 File Parameters

    st.iParamRows = stTemp.iParamRows;    // Total number of rows
    st.iParamCols = stTemp.iParamCols;    // Total number of columns
    st.sParams    = new String[st.iParamRows][st.iParamCols];

    for (i=0; i<st.iParamRows; i++)
    {
      for (j=0; j<st.iParamCols; j++)
      {
        // Array holding the Parameter Definitions
        st.sParams[i][j] = new String( stTemp.sParams[i][j] );
      }
    }

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method addNewOnly()
   * <p> This method will add stratigraphic unit only if it does not exist
   *     in the new stratigraphic unit in an existing list
   * @param  stStrat = The Region Data Structure
   * @param  st      = The Old stratigraphic unit List Data Structure
   * @return st      = The new stratigraphic unit List Data Structure.
   */

  public static stratListStruct addNewOnly( stratStruct stStrat,
                                            stratListStruct st )
  {
    int    i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = 0;
    int    iSource  = -1;
    stratListStruct stTemp = null;

    if (st != null)
      iRecords = st.iCount+1;
    else
      iRecords = 1;

    stTemp        = new stratListStruct();
    stTemp.stItem = new stratStruct[iRecords];

    if (st != null)
    {
	  iSource       = st.iSource;

      stTemp.sKGS   = new String( st.sKGS );     // KGS Saved Data Indicator

      // LAS 3 Required Fields

      stTemp.sTOPS  = new String( st.sTOPS );    // Formation Source
      stTemp.sTOPDR = new String( st.sTOPDR );   // Tops Depth Reference

      // LAS 3 File Parameters

      stTemp.iParamRows = st.iParamRows;    // Total number of rows
      stTemp.iParamCols = st.iParamCols;    // Total number of columns
      stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

      for (i=0; i<st.iParamRows; i++)
      {
        for (j=0; j<st.iParamCols; j++)
        {
          // Array holding the Parameter Definitions
          stTemp.sParams[i][j] = new String( st.sParams[i][j] );
        }
      }

      if (st.iCount > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          if (iCount < iRecords)
          {
            if ((st.stItem[i].sName.equals(stStrat.sName)) &&
                (st.stItem[i].sName2.equals(stStrat.sName2)))
            {
              iFound = 1;
            }

            stTemp.stItem[iCount] = copy(st.stItem[i]);
            iCount++;
          }
        }
      }

      st.delete();
    }

    if (iFound == 0)
    {
      stTemp.stItem[iCount] = new stratStruct();
      stTemp.stItem[iCount] = copy(stStrat);
      iCount++;
    }

    stTemp.iCount = iCount;

    st        = new stratListStruct();
    st.stItem = new stratStruct[iCount];
    st.iCount = iCount;

    st.iSource = iSource;

    st.sKGS   = new String( stTemp.sKGS );     // KGS Saved Data Indicator

    // LAS 3 Required Fields

    st.sTOPS  = new String( stTemp.sTOPS );    // Formation Source
    st.sTOPDR = new String( stTemp.sTOPDR );   // Tops Depth Reference

    // LAS 3 File Parameters

    st.iParamRows = stTemp.iParamRows;    // Total number of rows
    st.iParamCols = stTemp.iParamCols;    // Total number of columns
    st.sParams    = new String[st.iParamRows][st.iParamCols];

    for (i=0; i<st.iParamRows; i++)
    {
      for (j=0; j<st.iParamCols; j++)
      {
        // Array holding the Parameter Definitions
        st.sParams[i][j] = new String( stTemp.sParams[i][j] );
      }
    }

    for (i=0; i<stTemp.iCount; i++)
    {
      st.stItem[i] = copy(stTemp.stItem[i]);
    }

    stTemp.delete();

    return (st);
  }

  /** Method modify()
   * <p> This method will modify a stratigraphic unit in an existing list
   * @param  sKID    = The Primary Key from DB
   * @param  sKEY    = The User Created Primary Key of the data to be removed
   * @param  stStrat = stratigraphic unit data structure
   * @param  st      = stratigraphic unit data list structure
   * @return st      = New stratigraphic unit data list structure
   */

  public static stratListStruct modify( String sKID,
                                        String sKEY,
                                        stratStruct stStrat,
                                        stratListStruct st )
  {
    int i=0;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if (((sKID.equals(st.stItem[i].sKID)) && (!sKID.equals("0"))) ||
            ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0"))))
        {
          st.stItem[i] = copy(stStrat);
        }
      }
    }

    return (st);
  }

  /** Method remove()
   * <p> This method will remove a Rock fossil from an existing list
   * @param  sKID = The Primary Key from DB
   * @param  sKEY = The User Created Primary Key of the data to be removed
   * @param  st   = stratigraphic unit data list structure
   * @return st   = New stratigraphic unit data list structure
   */

  public static stratListStruct remove( String sKID,
                                        String sKEY,
                                        stratListStruct st )
  {
    int i,j;
    int    iRecords = 0;
    int    iCount   = 0;
    int    iFound   = -1;
    int    iSource  = -1;
    stratListStruct stTemp = null;

    if (st != null)
    {
      if (st.iCount == 1)
      {
        st.delete();
        st = null;
      }
      else
      {
        iRecords = st.iCount-1;

        stTemp        = new stratListStruct();
        stTemp.stItem = new stratStruct[iRecords];

        iSource       = st.iSource;

        stTemp.sKGS   = new String( st.sKGS );     // KGS Saved Data Indicator

        // LAS 3 Required Fields

        stTemp.sTOPS  = new String( st.sTOPS );    // Formation Source
        stTemp.sTOPDR = new String( st.sTOPDR );   // Tops Depth Reference

        // LAS 3 File Parameters

        stTemp.iParamRows = st.iParamRows;    // Total number of rows
        stTemp.iParamCols = st.iParamCols;    // Total number of columns
        stTemp.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            stTemp.sParams[i][j] = new String( st.sParams[i][j] );
          }
        }

        for (i=0; i<st.iCount; i++)
        {
          if (((sKID.equals(st.stItem[i].sKID)) && (!sKID.equals("0"))) ||
              ((sKEY.equals(st.stItem[i].sKEY)) && (!sKEY.equals("0"))))
          {
            iFound = i;
          }
        }

        if (st.iCount > 0)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (iCount < iRecords)
            {
              if (i != iFound)
              {
                stTemp.stItem[iCount] = copy(st.stItem[i]);
                iCount++;
              }
            }
          }
        }

        st.delete();

        stTemp.iCount = iCount;

        st        = new stratListStruct();
        st.stItem = new stratStruct[iCount];
        st.iCount = iCount;

        st.iSource = iSource;

        st.sKGS   = new String( stTemp.sKGS );     // KGS Saved Data Indicator

        // LAS 3 Required Fields

        st.sTOPS  = new String( stTemp.sTOPS );    // Formation Source
        st.sTOPDR = new String( stTemp.sTOPDR );   // Tops Depth Reference

        // LAS 3 File Parameters

        st.iParamRows = stTemp.iParamRows;    // Total number of rows
        st.iParamCols = stTemp.iParamCols;    // Total number of columns
        st.sParams    = new String[st.iParamRows][st.iParamCols];

        for (i=0; i<st.iParamRows; i++)
        {
          for (j=0; j<st.iParamCols; j++)
          {
            // Array holding the Parameter Definitions
            st.sParams[i][j] = new String( stTemp.sParams[i][j] );
          }
        }

        for (i=0; i<stTemp.iCount; i++)
        {
          st.stItem[i] = copy(stTemp.stItem[i]);
        }

        stTemp.delete();
      }
    }

    return (st);
  }

  /** Method copy()
   * <p> This method will copy one stratigraphic unit structure to another
   * @param  stOld = Old stratigraphic unit data structure
   * @return stNew = New stratigraphic unit data structure
   */

  public static stratStruct copy( stratStruct stOld )
  {
    stratStruct stNew = null;

    if (stOld != null)
    {
      stNew            = new stratStruct();

      // Cross Section Plot Variables

      stNew.sID        = new String(stOld.sID); // Plot ID for strat Unit
      stNew.iStatus    = stOld.iStatus;         // Turn on or off the strat Unit

      // Stratigraphic Units Variables

      stNew.sKID       = new String(stOld.sKID);   // KGS Primary KEY of Record
      stNew.stratKEY   = new String(stOld.stratKEY); // Stratigraphic KEY
      stNew.sKEY       = new String(stOld.sKEY);   // unique KEY of Record
      stNew.sid        = new String(stOld.sid);    // unique order id
//      stNew.id         = stOld.id;                 // unique order id
      stNew.iLevel     = stOld.iLevel;    // Confidence level of selection
      stNew.iRank      = stOld.iRank;              // Rank ID
      stNew.sName      = new String(stOld.sName);  // Name of Stratigraphic Unit
      stNew.sName2     = new String(stOld.sName2); // Informal Name

      stNew.source     = new String(stOld.source); // Source Name of data

      // Age of Stratigraphic Unit

      stNew.dAgeStart  = stOld.dAgeStart;   // Starting Age (Ma).
      stNew.dStartCorr = stOld.dStartCorr;  // +/- of Starting Age
      stNew.dAgeEnd    = stOld.dAgeEnd;     // Ending Age (Ma).
      stNew.dEndCorr   = stOld.dEndCorr;    // +/- of Ending Age

      // Depth of Stratigraphic Unit

      stNew.depthStart = stOld.depthStart;  // Starting Depth.
      stNew.depthEnd   = stOld.depthEnd;    // Ending Depth.

      // Time Unit of Stratigraphic Unit

      stNew.dTimeStart = stOld.dTimeStart;  // Starting Time.
      stNew.dTimeEnd   = stOld.dTimeEnd;    // Ending Time.

      // RGB Background color of stratigraphic unit defined by International
      // Commission on Stratigraphy

      stNew.iRed       = stOld.iRed;
      stNew.iGreen     = stOld.iGreen;
      stNew.iBlue      = stOld.iBlue;

      // Stratigraphic Units Names

      // ISC Stratigraphic Unit Names

      stNew.sEon       = new String(stOld.sEon);       // Eon Names
      stNew.sEra       = new String(stOld.sEra);       // Era Names
      stNew.system     = new String(stOld.system);     // System Names
      stNew.series     = new String(stOld.series);     // Series Names
      stNew.subSystem  = new String(stOld.subSystem);  // Substage Names
      stNew.subSeries  = new String(stOld.subSeries);  // Subseries Names

      // States Stratigraphic Unit Names

      stNew.stage      = new String(stOld.stage);      // Stage Names
      stNew.sprGroup   = new String(stOld.sprGroup);   // Super Group Names
      stNew.sGroup     = new String(stOld.sGroup);     // Group Names
      stNew.subGroup   = new String(stOld.subGroup);   // Subgroup Names
      stNew.sFormation = new String(stOld.sFormation); // Formation Names
    }

    return (stNew);
  }

  /** Method copyRegions()
   * <p> This method will copy the regions list data structure to the
   *     stratigraphic unit data list structure.
   * @param  stRegions = stratigraphic unit data list structure
   * @return st        = stratigraphic unit data list structure
   */

  public static stratListStruct copyRegions( regionsListStruct stRegions )
  {
    stratListStruct st = null;

    if (stRegions != null)
    {
      if (stRegions.iCount > 0)
      {
        st        = new stratListStruct();
        st.iCount = stRegions.iCount;
        st.stItem = new stratStruct[st.iCount];

        st.iSource = stRegions.iSource;

        for (int i=0; i<stRegions.iCount; i++)
        {
          st.stItem[i]      = new stratStruct();

          st.stItem[i].sKID     = new String(stRegions.stItem[i].sKID);
          st.stItem[i].stratKEY = new String(stRegions.stItem[i].sKEY);

          st.stItem[i].iLevel = stRegions.stItem[i].iLevel;

          for (int j=0; j<horizon.strat.stratStruct.RANK.length; j++)
          {
            if (stRegions.stItem[i].sRank.equals(
                horizon.strat.stratStruct.RANK[j]))
            {
              st.stItem[i].iRank = j;
            }
          }

          st.stItem[i].depthStart = stRegions.stItem[i].depth_top;
          st.stItem[i].depthEnd   = stRegions.stItem[i].depth_base;

          st.stItem[i].sName  = new String(stRegions.stItem[i].sName);
          st.stItem[i].sName2 = new String(stRegions.stItem[i].sName2);

          st.stItem[i].source  = new String(stRegions.stItem[i].source);

          st.stItem[i].system = new String(stRegions.stItem[i].systemName);
          st.stItem[i].series = new String(stRegions.stItem[i].seriesName);

          if ((st.stItem[i].system.equals("Pennsylvanian")) ||
              (st.stItem[i].system.equals("Mississippian")))
          {
            st.stItem[i].subSystem = new String(st.stItem[i].system);
            st.stItem[i].system    = new String("Carboniferous");

            if (st.stItem[i].series.length() > 0)
            {
              st.stItem[i].series = new String("");
              st.stItem[i].subSeries = new String(st.stItem[i].series);
            }

            if ((st.stItem[i].sName.equals("Pennsylvanian")) ||
                (st.stItem[i].sName.equals("Mississippian")))
            {
              st.stItem[i].iRank = horizon.strat.stratStruct._SUBSYSTEM;
            }

            if ((st.stItem[i].sName.equals("Upper")) ||
                (st.stItem[i].sName.equals("Middle")) ||
                (st.stItem[i].sName.equals("Lower")))
            {
              st.stItem[i].iRank = horizon.strat.stratStruct._SUBSERIES;
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method addICSArray()
   * <p> This method will load the stratigraphic unit names if Eon is missing
   *     from the data set.
   * @param  st    = stratigraphic unit data list structure
   * @param  stICS = International Commission on Stratigraphy stratigraphic unit
   *                 data list structure
   * @return st    = stratigraphic unit data list structure
   */

  public static stratListStruct addICSArray( stratListStruct st,
                                             stratListStruct stICS )
  {
    int    i    = 0;
    int    k    = 0;
    String s1   = "";
    String sICS = "";

    if ((st != null) && (stICS != null))
    {
      for(i=0; i<st.iCount; i++)
      {
        if (st.stItem[i].sEon.length() == 0)
        {
          s1   = new String(st.stItem[i].system.toLowerCase());

          for (k=0; k<stICS.iCount; k++)
          {
            sICS = new String(stICS.stItem[k].sName.toLowerCase());

            if (s1.equals(sICS))
            {
              st.stItem[i].sEon = new String(stICS.stItem[k].sEon); // Eon Names
              st.stItem[i].sEra = new String(stICS.stItem[k].sEra); // Era Names
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method addAges()
   * <p> This method will check to see if the user has entered an age and it
   *     is either a system, subsystem series or subseries and copy the age
   *     limit to the stratigraphic unit.
   * @param  st    = stratigraphic unit data list structure
   * @param  stICS = International Commission on Stratigraphy stratigraphic unit
   *                 data list structure
   * @return st    = stratigraphic unit data list structure
   */

  public static stratListStruct addAges( stratListStruct st,
                                         stratListStruct stICS )
  {
    String sName     = "";
    String system    = "";
    String series    = "";
    String subSystem = "";
    String subSeries = "";
    String sICS      = "";

    if ((st != null) && (stICS != null))
    {
      for (int i=0; i<st.iCount; i++)
      {
        sName     = new String(st.stItem[i].sName.toLowerCase());
        system    = new String(st.stItem[i].system.toLowerCase());
        series    = new String(st.stItem[i].series.toLowerCase());
        subSystem = new String(st.stItem[i].subSystem.toLowerCase());
        subSeries = new String(st.stItem[i].subSeries.toLowerCase());

        if ((sName.equals(system))    || (sName.equals(series)) ||
            (sName.equals(subSystem)) || (sName.equals(subSeries)))
        {
          for (int j=0; j<stICS.iCount; j++)
          {
            sICS = new String(stICS.stItem[j].sName);
            if (sICS.equals(sName))
            {
              if ((st.stItem[i].dAgeStart == 0.0) &&
                  (stICS.stItem[j].dAgeStart > 0.0))
              {
                st.stItem[i].dAgeStart  = stICS.stItem[j].dAgeStart;
                st.stItem[i].dStartCorr = stICS.stItem[j].dStartCorr;
              }

              if ((st.stItem[i].dAgeEnd == 0.0) &&
                  (stICS.stItem[j].dAgeEnd > 0.0))
              {
                st.stItem[i].dAgeEnd  = stICS.stItem[j].dAgeEnd;
                st.stItem[i].dEndCorr = stICS.stItem[j].dEndCorr;
              }
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method removeDescriptors()
   * <p> This method will remove the descriptors like member, stage, etc. since
   *     the plot routine separates into distinct tracks
   * @param  st    = stratigraphic unit data list structure
   * @return st    = stratigraphic unit data list structure
   */

  public static stratListStruct removeDescriptors( stratListStruct st )
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Stage", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Group", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Formation", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Member", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Bed", ""));
      }
    }

    return (st);
  }

  /** Method fillKGSNames()
   * <p> This method will try to match the KGS Database Names with the Kansas
   *     XML Lookup File Data Structure.
   * @param  st    = stratigraphic unit data list structure
   * @param  stKGS = KGS stratigraphic unit data list structure
   * @return st    = stratigraphic unit data list structure
   */

  public static stratListStruct fillKGSNames( stratListStruct st,
                                              stratListStruct stKGS )
  {
    int    i    = 0;
    int    j    = 0;
    String s1   = "";
    String sKGS = "";

    if ((st != null) && (stKGS != null))
    {
      for (i=0; i<st.iCount; i++)
      {
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Stage", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Group", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Formation", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Member", ""));
        st.stItem[i].sName = new String(st.stItem[i].sName.replaceAll(" Bed", ""));
      }

      for (i=0; i<st.iCount; i++)
      {
        for (j=0; j<stKGS.iCount; j++)
        {
          s1   = new String(st.stItem[i].sName.toLowerCase());
          sKGS = new String(stKGS.stItem[j].sName.toLowerCase());

          if ((s1.equals(sKGS)) ||
              (st.stItem[i].sKID.equals(stKGS.stItem[j].sKID)) ||
              (st.stItem[i].stratKEY.equals(stKGS.stItem[j].sKEY)))
          {
            st.stItem[i].sName      = new String(stKGS.stItem[j].sName);
            st.stItem[i].sid        = new String(stKGS.stItem[j].sid);
//            st.stItem[i].id         = stKGS.stItem[j].id;

            if ((s1.equals(stKGS.stItem[j].system.toLowerCase())) ||
                (s1.equals(stKGS.stItem[j].subSystem.toLowerCase())))
            {
              st.stItem[i].dAgeStart  = stKGS.stItem[j].dAgeStart;  // Starting Age (Ma).
              st.stItem[i].dStartCorr = stKGS.stItem[j].dStartCorr; // +/- of Starting Age
              st.stItem[i].dAgeEnd    = stKGS.stItem[j].dAgeEnd;    // Ending Age (Ma).
              st.stItem[i].dEndCorr   = stKGS.stItem[j].dEndCorr;   // +/- of Ending Age
            }

            // ISC Stratigraphic Unit Names

            st.stItem[i].sEon       = new String(stKGS.stItem[j].sEon);       // Eon Names
            st.stItem[i].sEra       = new String(stKGS.stItem[j].sEra);       // Era Names
            st.stItem[i].system     = new String(stKGS.stItem[j].system);     // System Names
            st.stItem[i].series     = new String(stKGS.stItem[j].series);     // Series Names
            st.stItem[i].subSystem  = new String(stKGS.stItem[j].subSystem);  // Substage Names
            st.stItem[i].subSeries  = new String(stKGS.stItem[j].subSeries);  // Subseries Names

            // States Stratigraphic Unit Names

            st.stItem[i].stage      = new String(stKGS.stItem[j].stage);      // Stage Names
            st.stItem[i].sprGroup   = new String(stKGS.stItem[j].sprGroup);   // Super Group Names
            st.stItem[i].sGroup     = new String(stKGS.stItem[j].sGroup);     // Group Names
            st.stItem[i].subGroup   = new String(stKGS.stItem[j].subGroup);   // Subgroup Names
            st.stItem[i].sFormation = new String(stKGS.stItem[j].sFormation); // Formation Names
          }
        }
      }
    }

    return (st);
  }

  /** Method fillBaseDepth()
   * <p> This method will try to identify the base by the orientation of the
   *     formation with respect to the kansas.xml lookup file. Otherwise the
   *     group, stage, series,etc will be placed at the first occurance with
   *     respect the referenced member or bed.
   *
   *     NOTE: This method is only used for data coming out of the KGS database.
   *
   * @param  st    = stratigraphic unit data list structure
   * @param  stKGS = KGS stratigraphic unit data list structure
   * @return st    = stratigraphic unit data list structure
   */

  public static  stratListStruct fillBaseDepth( stratListStruct st,
                                                stratListStruct stKGS )
  {
    int i = 0;
    int j = 0;
    int k = 0;
    String s1 = "";
    String s2 = "";
    String sKGS = "";
    double d1 = 0.0;
    double d2 = 0.0;
    double dKGS = 0.0;
    double t1 = 0.0;
    stratListStruct stNew = null;

    if ((st != null) && (stKGS != null))
    {
      stNew = copyList( st );

      for (i=0; i<st.iCount; i++)
      {
        d1 = st.stItem[i].depthStart;
        s1 = new String(st.stItem[i].sName.trim().toLowerCase());

        t1 = st.stItem[i].dTimeStart;

        for (j=0; j<stKGS.iCount; j++)
        {
          sKGS = new String(stKGS.stItem[j].sName.trim().toLowerCase());
          if (s1.equals(sKGS))
          {
            dKGS = stKGS.stItem[j].depthStart;
          }
        }

        for (j=0; j<stKGS.iCount; j++)
        {
          if (dKGS == stKGS.stItem[j].depthEnd)
          {
            sKGS = new String(stKGS.stItem[j].sName.trim().toLowerCase());
            for (k=0; k<stNew.iCount; k++)
            {
              s2 = new String(stNew.stItem[k].sName.trim().toLowerCase());
              if (s2.equals(sKGS))
              {
                stNew.stItem[k].depthEnd = d1;
                stNew.stItem[k].dTimeEnd = t1;
                dKGS = -1.0;
              }
            }
          }
        }
      }
    }

    return (stNew);
  }

  /** Method initSystem()
   * <p> This method will initialize the system identifier array to -1
   * @param  st       = stratigraphic unit data list structure
   * @return iSystems = array of system identifiers
   */

  public static int[] initSystem( stratListStruct st )
  {
    int    i      = 0;
    int    icnt   = 0;
    int    iTotal = 0;
    int    iSystems[] = null;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].iRank == horizon.strat.stratStruct._SYSTEM) ||
            (st.stItem[i].iRank == horizon.strat.stratStruct._SUBSYSTEM))
        {
          iTotal++;
        }
      }

      if (iTotal > 0)
      {
        iSystems = new int[iTotal];

        for (i=0; i<iTotal; i++)
          iSystems[i] = -1;
      }
    }

    return (iSystems);
  }

  /** Method setSystem()
   * <p> This method will initialize the system identifier array to -1
   * @param  iSystems = array of system identifiers
   * @param  system   = array of system & subsystem id's & text
   * @param  st       = stratigraphic unit data list structure
   * @return iSystems = array of system identifiers
   */

  public static int[] setSystem( int iSystems[], String system[][], stratListStruct st )
  {
    if ((iSystems != null) && (system != null) && (st != null))
    {
      if (iSystems.length == system.length)
      {
        for (int i=0; i<st.iCount; i++)
        {
          for (int j=0; j<system.length; j++)
          {
            if (system[j][0].equals(st.stItem[i].sKEY))
              iSystems[j] = 1;
          }
        }
      }
    }

    return (iSystems);
  }

  /** Method getSystem()
   * <p> This method will return an array of System & Subsystem id's & text
   * @param  st     = stratigraphic unit data list structure
   * @return system = array of system & subsystem id's & text
   */

  public static String[][] getSystem( stratListStruct st )
  {
    int    i      = 0;
    int    icnt   = 0;
    int    iTotal = 0;
    String system[][] = null;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if ((st.stItem[i].iRank == horizon.strat.stratStruct._SYSTEM) ||
            (st.stItem[i].iRank == horizon.strat.stratStruct._SUBSYSTEM))
        {
          iTotal++;
        }
      }

      if (iTotal > 0)
      {
        system = new String[iTotal][2];

        for (i=0; i<st.iCount; i++)
        {
          if ((st.stItem[i].iRank == horizon.strat.stratStruct._SYSTEM) ||
              (st.stItem[i].iRank == horizon.strat.stratStruct._SUBSYSTEM))
          {
            system[icnt][0] = new String(st.stItem[i].sKEY);
            system[icnt][1] = new String(st.stItem[i].sName);
            icnt++;
          }
        }
      }
    }

    return (system);
  }

  /** Method getSourceNames( stratListStruct st )
   * <p> This method will retrieve the list of source names
   * @param  st      = stratigraphic units List Data Structure
   * @return sources = array of source names
   */

  public static String[] getSourceNames( stratListStruct st )
  {
    String sources[] = null;
    int    iSources  = 0;
    int    iFound    = 0;
    int    i         = 0;
    int    j         = 0;
    int    iCount    = 0;
    String sTemp[]   = null;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        iFound = 0;
        for (j=0; j<iSources; j++)
        {
          if (st.stItem[i].source.equals(sources[j]))
          {
            iFound = 1;
          }
        }

        if (iFound == 0)
        {
          sTemp  = new String[iSources+1];
          iCount = 0;

          for (j=0; j<iSources; j++)
          {
            sTemp[iCount] = new String(sources[j]);
            iCount++;
          }

          sTemp[iCount]  = new String(st.stItem[i].source);
          iCount++;

          iSources = iCount;
          sources  = new String[iSources];

          for (j=0; j<iCount; j++)
          {
            sources[j] = new String(sTemp[j]);
          }
        }
      }
    }

    return (sources);
  }

  /** Method getStratBySource( String src, stratListStruct st )
   * <p> This method will return the filtered stratigraphic units
   * @param  src    = source name for stratigraphic units
   * @param  st     = Complete Stratigraphic Units List for a Well
   * @return stList = Stratigraphic Units List for a single source
   */

  public static stratListStruct getStratBySource(String src, stratListStruct st)
  {
    stratListStruct stList = null;
    int             i      = 0;
    int             iCount = 0;

    if (st != null)
    {
      for (i=0; i<st.iCount; i++)
      {
        if (src.equals(st.stItem[i].source))
        {
          iCount++;
        }
      }

      if (iCount > 0)
      {
        stList        = new stratListStruct();
        stList.stItem = new stratStruct[iCount];

        iCount = 0;
        for (i=0; i<st.iCount; i++)
        {
          if (src.equals(st.stItem[i].source))
          {
            stList.stItem[iCount] =
                horizon.strat.stratUtility.copy( st.stItem[i] );
            iCount++;
          }
        }

        stList.iCount = iCount;
      }
    }

    return (stList);
  }

  /** Method parse()
   * <p> This method will parse the Measured Section into rock color list
   *     data structure
   * @param  stList   = KGS Measured Section List Data Structure
   * @param  stKGS    = KGS Stratigraphic Units List Data Structure
   * @return stStrat  = Stratigraphic Units List Structure
   */

  public static stratListStruct parse( kgsMeasSectListStruct stList,
                                       stratListStruct       stKGS )
  {
    stratListStruct   stStrat    = null;
    int               iStart     = 0;
    double            depth      = 0.0;
    double            depth0     = 0.0;
    stratStruct       stText     = null;

    if (stList != null)
    {
      for (int i=0; i<stList.iRows; i++)
      {
        for (int j=0; j<stKGS.iCount; j++)
        {
          if ((!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SERIES")) &&
              (!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SUBSERIES")))
          {
            if (stKGS.stItem[j].sName.regionMatches( true,
                             0, stList.stData[i].sDescription.trim(),
                             0, stKGS.stItem[j].sName.length()))
            {
              stText = copy( stKGS.stItem[j] );

              stText.sKEY = new String(cmn.cmnString.UniqueName() +
                                       ((int) depth)); // Create a Unique KEY
              stText.depthStart = depth;   // Starting Depth
              stText.depthEnd   = depth;   // Ending Depth
            }
          }
        }

        depth = depth + stList.stData[i].dThickness;

        if (stText != null)
        {
          stStrat = add(stText, stStrat);
          if (iStart == 0)
          {
            stStrat.sTOPS  = new String("Kansas Geological Survey Database");
            stStrat.sTOPDR = new String("Log Depth");
            iStart = 1;
          }
        }

        depth0 = depth;

        if (stText != null)
          stText.delete();
        stText = null;
      }

      stList.delete();
      stList = null;
    }

    return (stStrat);
  }

  /** Method isKGS()
   * <p> This method will check to see if the string belongs to the KGS
   *     stratigraphic Units list
   * @param  str    = String to be checked
   * @param  stKGS  = KGS Stratigraphic Units List Data Structure
   * @return bValue = true it is and false if it is not
   */

  public static boolean isKGS( String str, stratListStruct stKGS )
  {
    boolean bValue = false;

    for (int j=0; j<stKGS.iCount; j++)
    {
      if ((!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SERIES")) &&
          (!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SUBSERIES")))
      {
        if (stKGS.stItem[j].sName.regionMatches( true,
                             0, str.trim(),
                             0, stKGS.stItem[j].sName.length()))
        {
          bValue = true;
        }
      }
    }

    return (bValue);
  }

  /** Method parse()
   * <p> This method will parse the string and determine if it is a Kansas
   *     stratigraphic unit and add it to the list if it is.
   * @param  str      = String to be checked
   * @param  dStart   = Staring Depth
   * @param  dEnd     = Ending Depth
   * @param  stStrat  = Stratigraphic Units List Structure
   * @param  stKGS    = KGS Stratigraphic Units List Data Structure
   * @return stStrat  = Stratigraphic Units List Structure
   */

  public static stratListStruct parse( String          str,
                                       double          dStart,
                                       double          dEnd,
                                       stratListStruct stStrat,
                                       stratListStruct stKGS )
  {
    int         iCount = 0;
    int         iStart = 0;
    stratStruct stText = null;

    if (stStrat != null)
      iCount = stStrat.iCount;

    for (int j=0; j<stKGS.iCount; j++)
    {
      if ((!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SERIES")) &&
          (!horizon.strat.stratStruct.RANK[
                   stKGS.stItem[j].iRank].equals("SUBSERIES")))
      {
        if (stKGS.stItem[j].sName.regionMatches( true,
                             0, str.trim(),
                             0, stKGS.stItem[j].sName.length()))
        {
          stText = copy( stKGS.stItem[j] );

          stText.sKEY = new String(cmn.cmnString.UniqueName() + iCount );
          stText.depthStart = dStart;  // Starting Depth
          stText.depthEnd   = dEnd;    // Ending Depth
        }
      }
    }

    if (stText != null)
    {
      stStrat = add(stText, stStrat);
      if (iStart == 0)
      {
        stStrat.sTOPS  = new String("GeoReport");
        stStrat.sTOPDR = new String("Log Depth");
        iStart = 1;
      }
    }

    if (stText != null)
      stText.delete();
    stText = null;

    return (stStrat);
  }

  /** Method bubbleSort()
   * <p> This method will sort in ascending depth order (lowest to highest)
   * @param  st = stratigraphic unit data list structure
   * @return st = sorted stratigraphic unit data list structure
   */

  public static stratListStruct bubbleSort( stratListStruct st )
  {
    boolean swappedOnPrevRun  = true;
    stratStruct stTemp   = null;

    if (st != null)
    {
      if (st.iCount > 1)
      {
        while(swappedOnPrevRun)
        {
          // this variable keeps track of whether to continue sorting or exit

          swappedOnPrevRun = false;

          // loop through every element in the array, except for the last one

          for(int i=0; i<st.iCount-1; i++)
          {
            // if current element is greater than the next swap the two elements

            if(st.stItem[i].depthStart > st.stItem[i+1].depthStart)
            {
              // we don't want the loop to end just yet, we're not done

              swappedOnPrevRun = true;

              // store element i in a temporary variable

              stTemp = copy(st.stItem[i]);

              // set element i+1 to where i used to be

              st.stItem[i] = copy(st.stItem[i+1]);

              // release the old i from temp into i+1 slot

              st.stItem[i+1] = copy(stTemp);
            }
          }
        }
      }
    }

    return (st);
  }

  /** Method getElevation()
   * <p> This method will retrieve the elevation for a specific well/outcrop
   * @param  st    = The cross section control data structure
   * @param  sName = Stratigraphic Unit Name
   * @return dElev = The Elevation Depth
   */

  public static double getElevation( stratListStruct st, String sName )
  {
    double dElev    = 0.0;
    String sCompare = "";
    String sHorizon = sName.toUpperCase();

    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        sCompare = new String(st.stItem[i].sName.toUpperCase());
        if (sCompare.equals(sHorizon))
          dElev = st.stItem[i].depthStart;
      }
    }

    return (dElev);
  }

  /** Method print()
   * <p> this method will print the Stratigraphic List Data Structure contents
   * @param st = Stratigraphic List Data Structure
   */

  public static void print(stratListStruct st)
  {
    if (st != null)
    {
      for (int i=0; i<st.iCount; i++)
      {
        System.out.println(i +" "+
                           st.stItem[i].sKID +" "+
                           st.stItem[i].sKEY +" "+
                           st.stItem[i].sid);
        System.out.println("----"+st.stItem[i].depthStart + " " +
                                  st.stItem[i].depthEnd );
        System.out.println("----"+
           horizon.strat.stratStruct.RANK[st.stItem[i].iRank]  + " " +
                                        st.stItem[i].sName     + " - " +
                                        st.stItem[i].sEra      + " " +
                                        st.stItem[i].system    + " " +
                                        st.stItem[i].series    + " " +
                                        st.stItem[i].subSystem + " " +
                                        st.stItem[i].subSeries + " - " +
                                        st.stItem[i].stage     + " " +
                                        st.stItem[i].sGroup    + " " +
                                        st.stItem[i].subGroup  + " " +
                                        st.stItem[i].sFormation);
      }
      System.out.println(" ------------------------------------------------- ");
    }
  }
}