/*
 * @stratStandardTools.java Version 1.1 02/12/2010
 *
 * Copyright (c) 2010 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat;

/** Class stratStandardTools
 *  <p> This Class holds the standard tool information to map the Stratigraphic
 *      LAS File curves to standard names.
 */

public class stratStandardTools
{
  // Stratigraphic Unit ID

  public static final int _KID      = 0;  // KGS Primary KEY
  public static final int _KEY      = 1;  // User defined Primary KEY
  public static final int _ID       = 2;  // Kansas Stratigraphic Unit ID

  // Depth

  public static final int _TOPT     = 3;  // Stratigraphic Unit Top Depth
  public static final int _TOPB     = 4;  // Stratigraphic Unit Base Depth

  // Stratigraphic Unit Names, Rank and Level of confidence of selection

  public static final int _RANK     = 5;  // Rank of Stratigraphic Unit
  public static final int _LEVEL    = 6;  // Measure of Goodness of Selection
  public static final int _TOPN     = 7;  // Stratigraphic Unit Name
  public static final int _ALT_NAME = 8;  // Alternate Name

  // Stratigraphic Unit Age

  public static final int _EON      = 9;  // EON Age
  public static final int _ERA      = 10;  // ERA Age
  public static final int _SYS      = 11;  // System
  public static final int _SER      = 12;  // Series
  public static final int _SSYS     = 13;  // Subsystem
  public static final int _SSER     = 14;  // Subseries
  public static final int _STG      = 15;  // Stage
  public static final int _SPGRP    = 16;  // Super Group
  public static final int _GRP      = 17;  // Group
  public static final int _SGRP     = 18;  // Subgroup
  public static final int _FORM     = 19;  // Formation

  public static final int TOTAL     = 20;  // Total

  public static final int _C_ID     = 0;
  public static final int _C_MNEM   = 1;
  public static final int _C_NAME   = 2;
  public static final int _C_UNIT   = 3;

  public static final String CURVES[][] =
  {
    // ID         MNEMONIC  CURVE NAME                               UNIT
    // Stratigraphic Unit ID
    { "1010", "KID",  "Primary Key for the Stratigraphic Unit",        "" },
    { "1012", "KEY",  "User Define Primary Key",                       "" },
    { "1014", "ID",   "Stratigraphic Unit ID",                         "" },
    // Depth
    { "1200", "TOPT", "Top Depth",                                     "F" },
    { "1220", "TOPB", "Base Depth",                                    "F" },
    // Stratigraphic Unit Names, Rank and Level of confidence of selection
    { "1400", "RANK", "Rank of Unit, i.e. (FORMATION, SYSTEM, etc)",   "" },
    { "1420", "LVL",  "Level of Confidence (GOLD, SILVER, COPPER...)", "" },
    { "1440", "TOPN", "Stratigraphic Unit Name",                       "" },
    { "1460", "TOPA", "Alternate Name",                                "" },
    // Stratigraphic Unit Age
    { "2000", "EON",  "EON Age",                                       "" },
    { "2020", "ERA",  "ERA Age",                                       "" },
    { "2200", "SYS",  "System",                                        "" },
    { "2220", "SER",  "Series",                                        "" },
    { "2240", "SSYS", "Subsystem",                                     "" },
    { "2260", "SSER", "Subseries",                                     "" },
    { "2400", "STG",  "Stage",                                         "" },
    { "2600", "SPGRP","Super Group",                                   "" },
    { "2620", "GRP",  "Group",                                         "" },
    { "2640", "SGRP", "Subgroup",                                      "" },
    { "2800", "FORM", "Formation",                                     "" }
  };

  /** Method isCurve()
   * <p> This method will determine if the curve is part of the standard
   *     stratigraphic units curves from the LAS File.
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return bValue  = true it is a valid curve, false it is not
   */

  public static boolean isCurve( int iColumn, String str )
  {
    boolean bValue = false;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        bValue = true;
      }
    }

    return (bValue);
  }

  /** Method getRow()
   * <p> This method will return the row number of the Stratigraphic Unit Curve
   * @param  iColumn = The column number
   * @param  str     = the string being compared.
   * @return iRow = row number
   */

  public static int getRow( int iColumn, String str )
  {
    int iRow = -1;

    if (iColumn==1) str = new String( str.toUpperCase() );

    for (int i=0; i<TOTAL; i++)
    {
      if (str.equals(CURVES[i][iColumn]))
      {
        iRow = i;
      }
    }

    return (iRow);
  }
}