/*
 * @stratPlotTrack.java Version 1.1 09/26/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.plot;

import java.awt.*;
import java.awt.geom.*;
import java.io.*;
import java.lang.*;
import java.util.*;

import horizon.strat.stratListStruct;

/** CLASS stratPlotTrack
 *  <p> This Class will create the Stratigraphic Units Track.
 *
 *  @version 1.1 09/26/2008
 *  @author  John R. Victorine
 */

public class stratPlotTrack extends Canvas
{
  // Input Variables

  private boolean bSonic     = false; // Indicator to plot by two-way time
  private int     iDataType  = -1;    // Format of Data, LAS File or Outcrop
  private double  depthStart = 0.0;   // Starting Depth
  private double  depthEnd   = 0.0;   // Ending Depth
  private int     iScale     = -1;    // Plot Scale

  // Global Variables

  public static final int _TOTAL     = 25;
  public static final int _ERA       = 24;//20;
  public static final int _SYSTEM    = 23;//19;
  public static final int _SUBSYSTEM = 22;//18;
  public static final int _SERIES    = 21;//17;
  public static final int _STAGE     = 20;//16;
  public static final int _GROUP     = 19;//15;
  public static final int _SUBGROUP  = 18;//14;
  public static final int _FORMATION =  9;//7;

  private static final int CHARS  = 45;
  private static final int CHAR_H =  8;

  private stratListStruct stStrat = null; // Stratigraphic Units Data List
  private stratListStruct stICS   = null; // ICS Stratigraphic Units

  // Plot Variables

  private int iWidth      = 100;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES = iqstrat.iqstratTracksStruct.PLOT_TITLES;
  public static final int LABELSTART  = iqstrat.iqstratTracksStruct.LABELSTART;

  /** CONSTRUCTOR stratPlotTrack( int iDataType,
   *                              double depthStart,
   *                              double depthEnd,
   *                              int iScale,
   *                              stratListStruct stICS )
   *  <p> This is the constructor for this class.
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param stICS       = International Committee on Stratigriphy Data
   */

  public stratPlotTrack( int iDataType,
                         double depthStart,
                         double depthEnd,
                         int iScale,
                         stratListStruct stICS )
  {
    this.iDataType = iDataType;
    this.stICS     = stICS;

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR stratPlotTrack( boolean bSonic,
   *                              int iDataType,
   *                              double depthStart,
   *                              double depthEnd,
   *                              int iScale,
   *                              stratListStruct stICS )
   *  <p> This is the constructor for this class.
   *  @param bSonic      = inidicator to plot by two-way time or depth
   *  @param iDataType   = Data type; 0=Well Data, 1=Outcrop Data
   *  @param depthStart  = The starting depth of plot
   *  @param depthEnd    = The ending depth of plot
   *  @param iScale      = The scale
   *  @param stICS       = International Committee on Stratigriphy Data
   */

  public stratPlotTrack( boolean bSonic,
	                     int iDataType,
                         double depthStart,
                         double depthEnd,
                         int iScale,
                         stratListStruct stICS )
  {
	this.bSonic    = bSonic;
    this.iDataType = iDataType;
    this.stICS     = stICS;

    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** Method close()
   * <p> This method will cancel the Rock Color Dialog
   */

  public void close()
  {
    stStrat = null;
    stICS   = null;
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Plot Width
   * @return iWidth = The Plot Width
   */

  public int getPlotWidth()  { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Plot Height
   * @return iHeight = The Plot Height
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getTopsData()
   * <p> This method will get an existing Stratigraphic Units List Data Structure
   * @return stStrat = Stratigraphic Units List Data Structure
   */

  public stratListStruct getTopsData() { return (stStrat); }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight( int iScale, double dMaximum, double dMinimum )
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight( int iScale, double dMaximum, double dMinimum )
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
  }

  /** Method setData( stratListStruct stStrat )
   * <p> This method will set an existing Stratigraphic Units List Data Structure
   * @param stStrat = Formatoin Tops List Data Structure
   */

  public void setData( stratListStruct stStrat ) { this.stStrat = stStrat; }

  /* =============================================================== *
   * ------------------------ DRAW METHODS ------------------------- *
   * =============================================================== */

  /** Method drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the depth scale
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawGrid( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/_TOTAL;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );

    g.setColor(Color.black);

    for (i=0; i<=iLogHeight; i+=iIncrementY)
    {
      if (iDataType == iqstrat.iqstratStruct._OUTCROP)
        iY1 = iHeight-i;
      else
        iY1 = i+PLOT_TITLES;

      jXinc = iIncrementY / 5;
      for (j=0; j<5; j++)
      {
        if (iDataType == iqstrat.iqstratStruct._OUTCROP)
          iY2 = iHeight-(i+(j*jXinc));
        else
          iY2 = i+(j*jXinc)+PLOT_TITLES;

        if ((iY2 >= PLOT_TITLES) && (iY2 < iHeight))
        {
          g.setColor(Color.lightGray);
          g.drawLine(iStartTrack, iY2, iEndTrack, iY2);
        }
      }

      g.setColor(Color.gray);
      g.drawLine(iStartTrack, iY1, iEndTrack, iY1);
    }

    g.setColor(Color.lightGray);
    g.drawLine(iEndTrack, LABELSTART, iEndTrack, PLOT_TITLES+iLogHeight); //iHeight);

    for (i=0; i<=iWidthTrack; i+=iIncrement)
    {
      if ((i==0) || (i==iWidthTrack))
      {
        g.drawLine(i+iStartTrack, LABELSTART,
                   i+iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
      }
    }

    g.setColor(Color.black);
    g.drawString("Members",    iStartTrack+2*iIncrement, LABELSTART+42);
    g.drawString("Formations", iStartTrack+11*iIncrement,   LABELSTART+42);

    g.drawLine(iStartTrack, LABELSTART+20, iEndTrack,   LABELSTART+20);

    g.drawLine(iStartTrack+_FORMATION*iIncrement, LABELSTART+20,
               iStartTrack+_FORMATION*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.drawLine(iStartTrack+_SUBGROUP*iIncrement, LABELSTART+20,
               iStartTrack+_SUBGROUP*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_GROUP*iIncrement, LABELSTART+20,
               iStartTrack+_GROUP*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_STAGE*iIncrement, LABELSTART+20,
               iStartTrack+_STAGE*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_SERIES*iIncrement, LABELSTART+20,
               iStartTrack+_SERIES*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_SUBSYSTEM*iIncrement, LABELSTART+20,
               iStartTrack+_SUBSYSTEM*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_SYSTEM*iIncrement, LABELSTART+20,
               iStartTrack+_SYSTEM*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iStartTrack+_ERA*iIncrement, LABELSTART+20,
               iStartTrack+_ERA*iIncrement, PLOT_TITLES+iLogHeight); //iHeight);

    g.drawString("Stratigraphic Units", iStartTrack+6*iIncrement, LABELSTART+14);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Era",       LABELSTART+40, -1*(iStartTrack + _ERA*iIncrement));
    g2.drawString("System",    LABELSTART+30, -1*(iStartTrack + _SYSTEM*iIncrement));
    g2.drawString("Subsystem", LABELSTART+20, -1*(iStartTrack + _SUBSYSTEM*iIncrement));
    g2.drawString("Series",    LABELSTART+30, -1*(iStartTrack + _SERIES*iIncrement));
    g2.drawString("Stage",     LABELSTART+30, -1*(iStartTrack + _STAGE*iIncrement));
    g2.drawString("Group",     LABELSTART+30, -1*(iStartTrack + _GROUP*iIncrement));
    g2.drawString("Subgroup",  LABELSTART+20, -1*(iStartTrack + _SUBGROUP*iIncrement));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART, iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, LABELSTART, iStartTrack, PLOT_TITLES+iLogHeight); //iHeight);
    g.drawLine(iEndTrack,   LABELSTART, iEndTrack,   PLOT_TITLES+iLogHeight); //iHeight);
  }

  /** Method drawData( Graphics g, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the members & formations track
   * @param g           = Graphics Parameter.
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawData( Graphics g, int iStartTrack, int iWidthTrack )
  {
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iContinue  = 0;
    int    iFormation = 0;
    int    iStart     = iStartTrack;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;
    Font        fsb   = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        switch (stStrat.stItem[i].iRank)
        {
          case horizon.strat.stratStruct._SYSTEM:
          case horizon.strat.stratStruct._SERIES:
          case horizon.strat.stratStruct._SUBSYSTEM:
          case horizon.strat.stratStruct._SUBSERIES:
          case horizon.strat.stratStruct._STAGE:
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._GROUP:
          case horizon.strat.stratStruct._SUBGROUP:
            iContinue  = 0;
            break;
          case horizon.strat.stratStruct._FORMATION:
            iFormation = 1;
            iContinue  = 1;
            iStart     = iStartTrack+_FORMATION*iIncrement;
            break;
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            iFormation = 0;
            iContinue  = 1;
            iStart     = iStartTrack;
            break;
        }

        if (iContinue == 1)
        {
	      if (bSonic)
            depth1 = stStrat.stItem[i].dTimeStart;
          else
            depth1 = stStrat.stItem[i].depthStart;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;
          if (iDataType == iqstrat.iqstratStruct._OUTCROP)
            iDepth1 = iHeight - (int) dRatio;

          iDepth2 = iDepth1;

          iContinue = 0;

          if (bSonic)
          {
            if (stStrat.stItem[i].dTimeEnd > stStrat.stItem[i].dTimeStart)
              iContinue = 1;
		  }
		  else
		  {
            if (stStrat.stItem[i].depthEnd > stStrat.stItem[i].depthStart)
              iContinue = 1;
		  }

//          if (stStrat.stItem[i].depthEnd > stStrat.stItem[i].depthStart)
		  if (iContinue == 1)
          {
			if (bSonic)
              depth2 = stStrat.stItem[i].dTimeEnd;
            else
              depth2 = stStrat.stItem[i].depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                                           (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;
          }

          iDepth = (iDepth1 + iDepth2)/2;

          if ((iDepth1 >= PLOT_TITLES) && (iDepth1 < iHeight))
          {
            if (iFormation == 1)
            {
              g.drawLine(iStartTrack,                 iDepth1,
                         iStartTrack + _SUBGROUP*iIncrement, iDepth1);
              g.drawLine(iStartTrack,                 iDepth2,
                         iStartTrack + _SUBGROUP*iIncrement, iDepth2);

              g.drawString(stStrat.stItem[i].sName, iStart, iDepth1+10);
            }
            else
            {
              g.setColor(Color.gray);
              g.drawLine(iStartTrack,                iDepth1,
                         iStartTrack + iIncrement,   iDepth1);
              g.drawLine(iStartTrack + 2*iIncrement, iDepth1,
                         iStartTrack + 3*iIncrement, iDepth1);
              g.drawLine(iStartTrack + 4*iIncrement, iDepth1,
                         iStartTrack + 5*iIncrement, iDepth1);
              g.drawLine(iStartTrack + 6*iIncrement, iDepth1,
                         iStartTrack + 7*iIncrement, iDepth1);

              g.setColor(Color.black);
              g.drawString(stStrat.stItem[i].sName, iStart, iDepth1+10);
            }
          }
        }
      }
    }
  }

  /** Method drawData( Graphics g, int iRank, int iStartTrack, int iWidthTrack )
   * <p> This method will draw the vertical tracks
   * @param g           = Graphics Parameter.
   * @param iRank       = The Type of Stratigraphic Unit
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawData(Graphics g, int iRank, int iStartTrack, int iWidthTrack)
  {
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iContinue  = 0;
    int    iStart     = iStartTrack;
    int    iBegin     = 0;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    int    iFound     = 0;
    int    len        = 0;
    int    iY         = 0;

    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;

    String sName      = "";
    double dStart     = 0.0;
    double dEnd       = 0.0;

    String sName2     = "";
    double dStart2    = 0.0;
    double dEnd2      = 0.0;

    double dLogStart  = 0.0;
    double dLogEnd    = 0.0;
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        iContinue = 0;
        iFound    = 0;

        switch (iRank)
        {
          case horizon.strat.stratStruct._ERA:
            iStart = iStartTrack + _ERA*iIncrement;
            if (stStrat.stItem[i].sEra.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].sEra);
              sName2 = new String(stStrat.stItem[i].sEra);
            }
            break;
          case horizon.strat.stratStruct._SYSTEM:
            iStart = iStartTrack + _SYSTEM*iIncrement;
            if (stStrat.stItem[i].system.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].system);
              sName2 = new String(stStrat.stItem[i].system);
            }
            break;
          case horizon.strat.stratStruct._SUBSYSTEM:
            iStart = iStartTrack + _SUBSYSTEM*iIncrement;
            if (stStrat.stItem[i].subSystem.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subSystem);
              sName2 = new String(stStrat.stItem[i].subSystem);
            }
            break;
          case horizon.strat.stratStruct._SERIES:
            iStart = iStartTrack + _SERIES*iIncrement;
            if (stStrat.stItem[i].series.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].series);
              sName2 = new String(stStrat.stItem[i].series);
            }
            break;
          case horizon.strat.stratStruct._SUBSERIES:
            iStart = iStartTrack + _SERIES*iIncrement;
            if (stStrat.stItem[i].subSeries.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subSeries);
              sName2 = new String(stStrat.stItem[i].subSeries);
            }
            break;
          case horizon.strat.stratStruct._STAGE:
            iStart = iStartTrack + _STAGE*iIncrement;
            if (stStrat.stItem[i].stage.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].stage);
              sName2 = new String(stStrat.stItem[i].stage);
            }
            break;
          case horizon.strat.stratStruct._GROUP:
            iStart = iStartTrack + _GROUP*iIncrement;
            if (stStrat.stItem[i].sGroup.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].sGroup);
              sName2 = new String(stStrat.stItem[i].sGroup);
            }
            break;
          case horizon.strat.stratStruct._SUBGROUP:
            iStart = iStartTrack + _SUBGROUP*iIncrement;
            if (stStrat.stItem[i].subGroup.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subGroup);
              sName2 = new String(stStrat.stItem[i].subGroup);
            }
            break;
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._FORMATION:
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            break;
        }

        if (bSonic)
        {
          dLogStart = stStrat.stItem[i].dTimeStart;
          dLogEnd   = stStrat.stItem[i].dTimeEnd;
		}
		else
		{
          dLogStart = stStrat.stItem[i].depthStart;
          dLogEnd   = stStrat.stItem[i].depthEnd;
		}

        if (((iFound == 1) || (i == stStrat.iCount-1)) && (sName.length() > 0))
        {
          if ((sName.equals(sName2)) && (iFound == 1))
          {
            if (iBegin == 0)
            {
              iBegin = 1;
              dStart = dLogStart;
              dEnd   = dLogStart;
            }

            if (dLogStart > dEnd)
              dEnd = dLogStart;

            if ((dLogEnd > dLogStart) && (dLogEnd > dEnd))
              dEnd = dLogEnd;
          }

          if ((!sName.equals(sName2)) || (i == stStrat.iCount-1))
          {
            iContinue = 1;

            if (iBegin == 0)
            {
              iBegin = 1;
              dStart = dLogStart;
              dEnd   = dLogStart;

              if (dLogEnd > dLogStart)
                dEnd = dLogEnd;
            }

            dStart2 = dLogStart;
            dEnd2   = dLogStart;
            if (dLogEnd > dLogStart)
              dEnd2   = dLogEnd;
          }
        }

        if (iContinue == 1)
        {
          depth1 = dStart;
          depth2 = dEnd;

          if (((depth1 <= depthStart) && (depth2 < depthStart)) ||
              ((depth1 > depthEnd)   && (depth2 >= depthEnd)))
            iContinue = 0;

          if (iContinue == 1)
          {
            if (depth1 < depthStart) depth1 = depthStart;
            if (depth2 > depthEnd)   depth2 = depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth1) /
                (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDepth = (iDepth1 + iDepth2) / 2;

            if (iRank > horizon.strat.stratStruct._SUBSERIES)
            {
              g.drawLine(iStartTrack,         iDepth1,
                         iStart + iIncrement, iDepth1);
            }

            Graphics2D g2 = (Graphics2D) g;
            g2.rotate( (Math.PI / 2.0));
            len = sName.length();
            iY = iDepth-7*len/2;
            if (iY < iDepth1) iY = iDepth1;
            if (iY >= PLOT_TITLES)
              g.drawString(sName, iY, -1*iStart);
            g2.rotate( -1.0 * (Math.PI / 2.0));
          }

          if ((i == stStrat.iCount-1) && (!sName.equals(sName2)))
          {
            sName  = new String(sName2);
            dStart = dStart2;
            dEnd   = dEnd2;

            depth1 = dStart;
            depth2 = dEnd;

            if (((depth1 <= depthStart) && (depth2 < depthStart)) ||
                ((depth1 > depthEnd)   && (depth2 >= depthEnd)))
              iContinue = 0;

            if (iContinue == 1)
            {
              if (depth1 < depthStart) depth1 = depthStart;
              if (depth2 > depthEnd)   depth2 = depthEnd;

              dRatio = (double) iLogHeight * (depthStart - depth1) /
                  (depthStart - depthEnd);
              iDepth1 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth1 = iHeight - (int) dRatio;

              dRatio = (double) iLogHeight * (depthStart - depth2) /
                  (depthStart - depthEnd);
              iDepth2 = PLOT_TITLES + (int) dRatio;
              if (iDataType == iqstrat.iqstratStruct._OUTCROP)
                iDepth2 = iHeight - (int) dRatio;

              iDepth = (iDepth1 + iDepth2) / 2;

              if (iRank > horizon.strat.stratStruct._SUBSERIES)
              {
                g.drawLine(iStartTrack,         iDepth1,
                           iStart + iIncrement, iDepth1);
              }

              Graphics2D g2 = (Graphics2D) g;
              g2.rotate( (Math.PI / 2.0));
              len = sName.length();
              iY = iDepth-7*len/2;
              if (iY < iDepth1) iY = iDepth1;
              if (iY >= PLOT_TITLES)
                g.drawString(sName, iY, -1*iStart);
              g2.rotate( -1.0 * (Math.PI / 2.0));
            }
          }

          sName  = new String(sName2);
          dStart = dStart2;
          dEnd   = dEnd2;
        }
      }
    }
  }

  /** Method drawBackground( Graphics g,
   *                         int iRank
   *                         int iStartTrack,
   *                         int iWidthTrack )
   * <p> This method will draw the background colors
   * @param g           = Graphics Parameter.
   * @param iRank       = The Type of Stratigraphic Unit
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawBackground( Graphics g,
                              int iRank,
                              int iStartTrack,
                              int iWidthTrack )
  {
    int    i          = 0;
    int    j          = 0;
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iTotal     = 0;
    int    iCount     = -1;
    int    iContinue  = 0;
    String sName      = "";
    String sData[]    = null;
    String sTemp      = "";
    double data[][]   = null;
    int    iDiff      = 0;
    int    iStart     = iStartTrack;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;
    double dLogStart  = 0.0;
    double dLogEnd    = 0.0;

    String sICS       = "";
    int    iRed       = 255;
    int    iGreen     = 255;
    int    iBlue      = 255;

    if (stStrat != null)
    {
      for (i=0; i<stStrat.iCount; i++)
      {
        switch (iRank)
        {
          case horizon.strat.stratStruct._ERA:
            iStart = _ERA*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].sEra)) &&
                 (stStrat.stItem[i].sEra.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].sEra);
            break;
          case horizon.strat.stratStruct._SYSTEM:
            iStart = _SYSTEM*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].system)) &&
                 (stStrat.stItem[i].system.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].system);
            break;
          case horizon.strat.stratStruct._SUBSYSTEM:
            iStart = _SUBSYSTEM*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].subSystem)) &&
                 (stStrat.stItem[i].subSystem.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].subSystem);
            break;
          case horizon.strat.stratStruct._SERIES:

            iStart = _SERIES*iIncrement;
            sTemp = new String(stStrat.stItem[i].system + " " +
                               stStrat.stItem[i].series);

            if ((!sName.equals(sTemp)) &&
                (stStrat.stItem[i].series.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].system + " " +
                               stStrat.stItem[i].series);

            break;
          case horizon.strat.stratStruct._SUBSERIES:

            iStart = _SERIES*iIncrement;
            sTemp = new String(stStrat.stItem[i].subSystem + " " +
                               stStrat.stItem[i].subSeries);

            if ((!sName.equals(sTemp)) &&
                (stStrat.stItem[i].subSeries.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].subSystem + " " +
                               stStrat.stItem[i].subSeries);

            break;
          case horizon.strat.stratStruct._STAGE:
          case horizon.strat.stratStruct._GROUP:
          case horizon.strat.stratStruct._SUBGROUP:
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._FORMATION:
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            break;
        }
      }

      sName = new String("");

      if (iTotal > 0)
      {
        sData  = new String[iTotal];
        data   = new double[iTotal][2];

        for (j=0; j<iTotal; j++)
        {
          sData[j]   = new String("");
          data[j][0] = 0.0;
          data[j][1] = 0.0;
        }

        iCount = -1;
        for (i=0; i<stStrat.iCount; i++)
        {
          iContinue = 0;

          if (bSonic)
          {
            dLogStart = stStrat.stItem[i].dTimeStart;
            dLogEnd   = stStrat.stItem[i].dTimeEnd;
  		  }
		  else
		  {
            dLogStart = stStrat.stItem[i].depthStart;
            dLogEnd   = stStrat.stItem[i].depthEnd;
		  }

          switch (iRank)
          {
            case horizon.strat.stratStruct._ERA:
              if ((!sName.equals(stStrat.stItem[i].sEra)) &&
                   (stStrat.stItem[i].sEra.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].sEra);
                if (iCount < iTotal)
                  data[iCount][0] = dLogStart;
              }

              if (stStrat.stItem[i].sEra.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SYSTEM:
              if ((!sName.equals(stStrat.stItem[i].system)) &&
                   (stStrat.stItem[i].system.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].system);
                if (iCount < iTotal)
                  data[iCount][0] = dLogStart;
              }

              if (stStrat.stItem[i].system.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SUBSYSTEM:
              if ((!sName.equals(stStrat.stItem[i].subSystem)) &&
                   (stStrat.stItem[i].subSystem.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].subSystem);
                if (iCount < iTotal)
                  data[iCount][0] = dLogStart;
              }

              if (stStrat.stItem[i].subSystem.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SERIES:

              sTemp = new String(stStrat.stItem[i].system + " " +
                                 stStrat.stItem[i].series);
              if ((!sName.equals(sTemp)) &&
                  (stStrat.stItem[i].series.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].system + " " +
                                   stStrat.stItem[i].series);
                if (iCount < iTotal)
                  data[iCount][0] = dLogStart;
              }

              if ((sTemp.length() > 1) &&
                  (stStrat.stItem[i].series.length() > 0))
                iContinue = 1;

              break;
            case horizon.strat.stratStruct._SUBSERIES:

              sTemp = new String(stStrat.stItem[i].subSystem + " " +
                                 stStrat.stItem[i].subSeries);

              if ((!sName.equals(sTemp)) &&
                  (stStrat.stItem[i].subSeries.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].subSystem + " " +
                                   stStrat.stItem[i].subSeries);
                if (iCount < iTotal)
                  data[iCount][0] = dLogStart;
              }

              if ((sTemp.length() > 1) &&
                  (stStrat.stItem[i].subSeries.length() > 0))
                iContinue = 1;

              break;
            case horizon.strat.stratStruct._STAGE:
            case horizon.strat.stratStruct._GROUP:
            case horizon.strat.stratStruct._SUBGROUP:
            case horizon.strat.stratStruct._SUPER_GROUP:
            case horizon.strat.stratStruct._FORMATION:
            case horizon.strat.stratStruct._NONE:
            case horizon.strat.stratStruct._MEMBER:
            case horizon.strat.stratStruct._BED:
              break;
          }

          if (iContinue == 1)
          {
            if ((iCount < iTotal) && (iCount > -1))
            {
              sData[iCount]   = new String(sName);

              if ((dLogEnd == 0.0) || (dLogEnd < dLogStart))
                data[iCount][1] = dLogStart;
              else
                data[iCount][1] = dLogEnd;
            }
          }
        }

        for (i=0; i<iTotal; i++)
        {
          depth1 = data[i][0];
          depth2 = data[i][1];

          iContinue  = 1;
          if (((depth1 <= depthStart) && (depth2 < depthStart)) ||
              ((depth1 > depthEnd)   && (depth2 >= depthEnd)))
            iContinue = 0;

          if (iContinue == 1)
          {
            if (depth1 < depthStart)
              depth1 = depthStart;
            if (depth2 > depthEnd)
              depth2 = depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth1) /
                                           (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth1 = iHeight - (int) dRatio;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                                           (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
            if (iDataType == iqstrat.iqstratStruct._OUTCROP)
              iDepth2 = iHeight - (int) dRatio;

            iDiff = iDepth2 - iDepth1;

            if (stICS != null)
            {
              for (j=0; j<stICS.iCount; j++)
              {
                if ((iRank !=  horizon.strat.stratStruct._SERIES) &&
                    (iRank !=  horizon.strat.stratStruct._SUBSERIES))
                {
                  sICS = new String(stICS.stItem[j].sName.toLowerCase());
                }

                else if (iRank ==  horizon.strat.stratStruct._SERIES)
                {
                  sICS = new String(stICS.stItem[j].system.toLowerCase() + " " +
                                    stICS.stItem[j].sName.toLowerCase());

                }
                else if (iRank ==  horizon.strat.stratStruct._SUBSERIES)
                {
                  sICS = new String(stICS.stItem[j].subSystem.toLowerCase() + " " +
                                    stICS.stItem[j].sName.toLowerCase());
                }

                if (sICS.equals(sData[i].toLowerCase()))
                {
                  iRed   = stICS.stItem[j].iRed;
                  iGreen = stICS.stItem[j].iGreen;
                  iBlue  = stICS.stItem[j].iBlue;
                }
              }

              g.setColor(new Color(iRed, iGreen, iBlue));
              g.fillRect(iStartTrack, iDepth1, iStart + iIncrement, iDiff);
            }
          }
        }
      }
    }
  }

  /** Method draw( Graphics g, int iSelected, int iStartTrack, int iWidthTrack )
   * <p> This method will create a Remarks Plot Track
   * @param g           = Graphics Parameter.
   * @param iSelected   = The Track to draw
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void draw(Graphics g, int iSelected, int iStartTrack, int iWidthTrack)
  {
    int iColor = iqstrat.iqstratTracksStruct._SRC_TOPS;

    if (iSelected == iqstrat.iqstratTracksStruct._STRAT_UNITS_A)
    {
      g.setColor(new Color(iqstrat.iqstratTracksStruct.COLORS[iColor][0],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][1],
                           iqstrat.iqstratTracksStruct.COLORS[iColor][2]));
      g.fillRect(iStartTrack, LABELSTART, iWidthTrack, 20);

      drawBackground(g, horizon.strat.stratStruct._ERA, iStartTrack, iWidthTrack);
      drawBackground(g, horizon.strat.stratStruct._SYSTEM, iStartTrack, iWidthTrack);
      drawBackground(g, horizon.strat.stratStruct._SERIES, iStartTrack, iWidthTrack);
      drawBackground(g, horizon.strat.stratStruct._SUBSYSTEM, iStartTrack, iWidthTrack);
      drawBackground(g, horizon.strat.stratStruct._SUBSERIES, iStartTrack, iWidthTrack);
      drawGrid(g, iStartTrack, iWidthTrack);
      drawData(g, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._ERA, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._SYSTEM, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._SERIES, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._SUBSYSTEM, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._SUBSERIES, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._STAGE, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._GROUP, iStartTrack, iWidthTrack);
      drawData(g, horizon.strat.stratStruct._SUBGROUP, iStartTrack, iWidthTrack);
    }
  }

  /** Method paint( Graphics g )
   * <p> This method will paint Porosity & Permeability Plot
   * @param g = Graphics Parameter.
   */

  public void paint( Graphics g )
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);
  }
}