/*
 * @stratPlotState.java Version 1.1 10/28/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.plot;

import java.io.*;
import java.lang.*;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

/** Class stratPlotState
 *  <p> This method will create an interactive Stratigraphic Chart for the user
 *      to select Ages.
 *
 *  @version 1.1 10/28/2008
 *  @author  John R. Victorine
 */

public class stratPlotState extends Canvas
    implements MouseListener, MouseMotionListener
{
  private Observable notifier = null; // Observable Object

  private stratListStruct stStrat = null; // State Stratigraphic Units
  private stratListStruct stICS   = null; // ICS Stratigraphic Units

  private String          sFormation = "";
  private String          sRank      = "";

  private int             iRank      = horizon.strat.stratStruct._NONE;
  private String          system     = "ALL"; // Filter data by
  private stratStruct     stSelected = null; // Selected stratigraphic structure

  private double depthStart = 0.0;  // Starting Depth of Strat data
  private double depthEnd   = 0.0;  // Ending Depth of Strat data
  private int    iScale     = iqstrat.iqstratTracksStruct.SCALE_100; // ft/in

  private int    iStartTrack = 0;
  private int    iWidthTrack = 300;//252;

  private double dIncrementDepth  = 0.0;
  private int    iStartDepthLabel = 10;

  private int iWidth      = 300;  // Width of Plot
  private int iHeight     = 100;  // Height of Plot with the Titles
  private int iLogHeight  = 100;  // Height of Plot without the Titles
  private int iIncrementY = 100;  // Increment Height

  public static final int PLOT_TITLES  = 100; // Plot Titles
  public static final int LABELSTART   = 25;  // Label Start

  public static final int _TOTAL     = 25;
  public static final int _ERA       = 24;//20;
  public static final int _SYSTEM    = 23;//19;
  public static final int _SUBSYSTEM = 22;//18;
  public static final int _SERIES    = 21;//17;
  public static final int _STAGE     = 20;//16;
  public static final int _GROUP     = 19;//15;
  public static final int _SUBGROUP  = 18;//14;
  public static final int _FORMATION =  9;//7;

  private JTextField txtRank = null;
  private JTextField txtName = null;

  /** CONSTRUCTOR stratPlotState()
   *  <p> This is the constructor for this class.
   *  @param stICS = International Committee on Stratigriphy Data List
   *  @param st    = the stratigraphic unit list Structure for the State
   *                 stratigraphic chart
   */

  public stratPlotState( stratListStruct stICS, stratListStruct st )
  {
    this.stICS   = stICS;
    this.stStrat = st;

    initDepthRange();
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR stratPlotState()
   *  <p> This is the constructor for this class.
   *  @param notifier = Observable
   *  @param stICS    = International Committee on Stratigriphy Data List
   *  @param st       = the stratigraphic unit list Structure for the State
   *                    stratigraphic chart
   */

  public stratPlotState( Observable notifier,
                         stratListStruct stICS,
                         stratListStruct st )
  {
    this.notifier = notifier;
    this.stICS    = stICS;
    this.stStrat  = st;

    initDepthRange();
    setPlotHeight(iScale, depthEnd, depthStart);

    this.setBackground(Color.white);

    addMouseListener(this);
    addMouseMotionListener(this);
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier = null;
    stICS    = null;
    stStrat  = null;
    system   = null;

    if (stSelected != null)
      stSelected.delete();
    stSelected = null;

    txtRank  = null;
    txtName  = null;
  }

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will return the ID of the data selected.
   * @return stSelected = the Stratigraphic unit selected.
   */

  public stratStruct getData() { return (stSelected); }

  /** Method initDepthRange()
   * <p> This method will retrieve the initial depth range from the
   *     stratigraphic chart
   */

  private void initDepthRange()
  {
    if (stStrat != null)
    {
      depthStart = stStrat.stItem[0].depthStart;
      depthEnd   = stStrat.stItem[0].depthEnd;

      for (int i=0; i<stStrat.iCount; i++)
      {
        if (depthEnd < stStrat.stItem[i].depthEnd)
          depthEnd = stStrat.stItem[i].depthEnd;
      }
    }
  }

  /* =============================================================== *
   * ------------------------- SET METHODS ------------------------- *
   * =============================================================== */

  /** Method setPlotHeight(int iScale, double dMaximum, double dMinimum)
   *  <p> This method will set the Plot Height.
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setPlotHeight(int iScale, double dMaximum, double dMinimum)
  {
    this.depthStart = dMinimum;
    this.depthEnd   = dMaximum;
    this.iScale     = iScale;

    iLogHeight = (int) ((100 * Math.abs( dMaximum - dMinimum )) /
                        iqstrat.iqstratTracksStruct.SCALE[iScale]);
    iHeight    = PLOT_TITLES + iLogHeight;
    dIncrementDepth = iqstrat.iqstratTracksStruct.SCALE[iScale];
  }

  /** Method setScale(int iScale, double dMaximum, double dMinimum)
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   */

  public void setScale(int iScale) { setScale(iScale, depthEnd, depthStart); }

  /** Method setScale(int iScale, double dMaximum, double dMinimum)
   * <p> This method will set the Plot scale in the
   *     Cross Plot Data List Structure
   * @param iScale   = The Depth Track Scale identifier
   * @param dMaximum = The Maximum Depth Value
   * @param dMinimum = The Minimum Depth Value
   */

  public void setScale(int iScale, double dMaximum, double dMinimum)
  {
    setPlotHeight(iScale, dMaximum, dMinimum);
    repaint();
  }

  /** Method setSystem(int iRank, String sName)
   * <p> This method will set the depth range by system
   * @param iRank = Rank of Strat unit
   * @param sName = name of strat unit
   */

  public void setSystem(int iRank, String sName)
  {
    int iFound = -1;

    this.iRank  = iRank;
    this.system = new String(sName);

    if (stStrat != null)
    {
      depthStart = 0.0;
      depthEnd   = 0.0;

      if (sName.equals("ALL"))
        initDepthRange();
      else
      {
        for (int i=0; i<stStrat.iCount; i++)
        {
          switch (iRank)
          {
            case horizon.strat.stratStruct._SYSTEM:
              if (stStrat.stItem[i].system.equals(sName))
              {
                if (iFound == -1)
                {
                  depthStart = stStrat.stItem[i].depthStart;
                  depthEnd   = stStrat.stItem[i].depthEnd;
                  iFound++;
                }

                if (depthEnd < stStrat.stItem[i].depthEnd)
                  depthEnd = stStrat.stItem[i].depthEnd;
              }
              break;
            case horizon.strat.stratStruct._SUBSYSTEM:
              if (stStrat.stItem[i].subSystem.equals(sName))
              {
                if (iFound == -1)
                {
                  depthStart = stStrat.stItem[i].depthStart;
                  depthEnd   = stStrat.stItem[i].depthEnd;
                  iFound++;
                }

                if (depthEnd < stStrat.stItem[i].depthEnd)
                  depthEnd = stStrat.stItem[i].depthEnd;
              }
              break;
          }
        }
      }

      setScale(iScale, depthEnd, depthStart);
    }
  }

  /** Method setTextFields()
   * <p> This method will set the textfields for the mouse motion listener
   * @param txtRank = the Rank text field
   * @param txtName = the startigraphic name text field
   */

  public void setTextFields( JTextField txtRank, JTextField txtName )
  {
    this.txtRank = txtRank;
    this.txtName = txtName;
  }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  /** Method mouseMoved()
   *  <p> Mouse Moved Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Moved Event
   */

  public void mouseMoved(MouseEvent event)
  {
    int    iXm  = event.getX();
    int    iYm  = event.getY();
    int    iIncrement  = iWidthTrack/_TOTAL;
    int    iRank  = horizon.strat.stratStruct._NONE;
    int    iRank2 = horizon.strat.stratStruct._NONE;
    double depth = 0.0;
    double depth1 = 0.0;

    sRank      = new String("");
    sFormation = new String("");

    if (iXm < iStartTrack+_FORMATION*iIncrement)
    {
      sRank  = new String("Member/Bed");
      iRank  = horizon.strat.stratStruct._MEMBER;
      iRank2 = horizon.strat.stratStruct._BED;
    }
    else if (iXm < iStartTrack+_SUBGROUP*iIncrement)
    {
      sRank = new String("Formation");
      iRank  = horizon.strat.stratStruct._FORMATION;
    }
    else if (iXm < iStartTrack+_GROUP*iIncrement)
    {
      sRank = new String("Subgroup");
      iRank  = horizon.strat.stratStruct._SUBGROUP;
    }
    else if (iXm < iStartTrack+_STAGE*iIncrement)
    {
      sRank = new String("Group");
      iRank  = horizon.strat.stratStruct._GROUP;
    }
    else if (iXm < iStartTrack+_SERIES*iIncrement)
    {
      sRank = new String("Stage");
      iRank  = horizon.strat.stratStruct._STAGE;
    }
    else if (iXm < iStartTrack+_SUBSYSTEM*iIncrement)
    {
      sRank = new String("Series/Subseries");
      iRank  = horizon.strat.stratStruct._SERIES;
      iRank2 = horizon.strat.stratStruct._SUBSERIES;
    }
    else if (iXm < iStartTrack+_SYSTEM*iIncrement)
    {
      sRank = new String("Subsystem");
      iRank  = horizon.strat.stratStruct._SUBSYSTEM;
    }
    else if (iXm < iStartTrack+_ERA*iIncrement)
    {
      sRank = new String("System");
      iRank  = horizon.strat.stratStruct._SYSTEM;
    }
    else if (iXm < iStartTrack+_TOTAL*iIncrement)
    {
      sRank = new String("Era");
      iRank  = horizon.strat.stratStruct._ERA;
    }

    depth = depthStart + (iYm-PLOT_TITLES) * (depthEnd-depthStart) / iLogHeight;

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        if ((iRank  == stStrat.stItem[i].iRank) ||
            (iRank2 == stStrat.stItem[i].iRank))
        {
          depth1 = stStrat.stItem[i].depthEnd;
          if (stStrat.stItem[i].depthStart == stStrat.stItem[i].depthEnd)
            depth1 = stStrat.stItem[i].depthEnd + 5.0;

          if ((depth >= stStrat.stItem[i].depthStart) && (depth <  depth1))
          {
            sFormation = new String(stStrat.stItem[i].sName);
          }
        }
      }
    }

    if (txtRank != null) txtRank.setText(sRank);
    if (txtName != null) txtName.setText(sFormation);
  }

  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event) { }
  public void mousePressed(MouseEvent event) { }
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm  = event.getX();
    int    iYm  = event.getY();
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iRank  = horizon.strat.stratStruct._NONE;
    int    iRank2 = horizon.strat.stratStruct._NONE;
    double depth = 0.0;
    double depth1 = 0.0;

    if (stSelected != null)
      stSelected.delete();
    stSelected = null;

    if (iXm < iStartTrack+_FORMATION*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._MEMBER;
      iRank2 = horizon.strat.stratStruct._BED;
    }
    else if (iXm < iStartTrack+_SUBGROUP*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._FORMATION;
    }
    else if (iXm < iStartTrack+_GROUP*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._SUBGROUP;
    }
    else if (iXm < iStartTrack+_STAGE*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._GROUP;
    }
    else if (iXm < iStartTrack+_SERIES*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._STAGE;
    }
    else if (iXm < iStartTrack+_SUBSYSTEM*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._SERIES;
      iRank2 = horizon.strat.stratStruct._SUBSERIES;
    }
    else if (iXm < iStartTrack+_SYSTEM*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._SUBSYSTEM;
    }
    else if (iXm < iStartTrack+_ERA*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._SYSTEM;
    }
    else if (iXm < iStartTrack+_TOTAL*iIncrement)
    {
      iRank  = horizon.strat.stratStruct._ERA;
    }

    depth = depthStart + (iYm-PLOT_TITLES) * (depthEnd-depthStart) / iLogHeight;

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        if ((iRank  == stStrat.stItem[i].iRank) ||
            (iRank2 == stStrat.stItem[i].iRank))
        {
          depth1 = stStrat.stItem[i].depthEnd;
          if (stStrat.stItem[i].depthStart == stStrat.stItem[i].depthEnd)
            depth1 = stStrat.stItem[i].depthEnd + 5.0;

          if ((depth >= stStrat.stItem[i].depthStart) && (depth <  depth1))
          {
            stSelected = horizon.strat.stratUtility.copy( stStrat.stItem[i] );

            if (notifier != null)
              notifier.notifyObservers(
                  new String("State Stratigraphic Unit Selected"));
          }
        }
      }
    }
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawGrid(Graphics g)
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   * @param iStartTrack = The starting pixel for the Track
   * @param iWidthTrack = The width of Track in pixels
   */

  public void drawGrid(Graphics g, int iStartTrack, int iWidthTrack)
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iIncrement  = iWidthTrack/_TOTAL;
    int    iEndTrack   = iStartTrack+iWidthTrack;
    int    iY1         = 0;
    int    iY2         = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    g.drawString("Members",    iStartTrack+4*iIncrement/2, LABELSTART+42);
    g.drawString("Formations", iStartTrack+11*iIncrement,  LABELSTART+42);

    g.drawLine(iStartTrack, LABELSTART+20, iEndTrack,   LABELSTART+20);

    g.drawLine(iStartTrack+_FORMATION*iIncrement, LABELSTART+20,
               iStartTrack+_FORMATION*iIncrement, PLOT_TITLES+iHeight);

    g.drawLine(iStartTrack+_SUBGROUP*iIncrement,  LABELSTART+20,
               iStartTrack+_SUBGROUP*iIncrement,  PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_GROUP*iIncrement,     LABELSTART+20,
               iStartTrack+_GROUP*iIncrement,     PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_STAGE*iIncrement,     LABELSTART+20,
               iStartTrack+_STAGE*iIncrement,     PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_SERIES*iIncrement,    LABELSTART+20,
               iStartTrack+_SERIES*iIncrement,    PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_SUBSYSTEM*iIncrement, LABELSTART+20,
               iStartTrack+_SUBSYSTEM*iIncrement, PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_SYSTEM*iIncrement,    LABELSTART+20,
               iStartTrack+_SYSTEM*iIncrement,    PLOT_TITLES+iHeight);
    g.drawLine(iStartTrack+_ERA*iIncrement,       LABELSTART+20,
               iStartTrack+_ERA*iIncrement,       PLOT_TITLES+iHeight);

    g.drawString("Stratigraphic Units", iStartTrack+8*iIncrement, LABELSTART+14);

    Graphics2D g2 = (Graphics2D) g;
    g2.rotate( (Math.PI / 2.0));
    g2.drawString("Era",       LABELSTART+40, -1*(iStartTrack + _ERA*iIncrement));
    g2.drawString("System",    LABELSTART+30, -1*(iStartTrack + _SYSTEM*iIncrement));
    g2.drawString("Subsystem", LABELSTART+20, -1*(iStartTrack + _SUBSYSTEM*iIncrement));
    g2.drawString("Series",    LABELSTART+30, -1*(iStartTrack + _SERIES*iIncrement));
    g2.drawString("Stage",     LABELSTART+30, -1*(iStartTrack + _STAGE*iIncrement));
    g2.drawString("Group",     LABELSTART+30, -1*(iStartTrack + _GROUP*iIncrement));
    g2.drawString("Subgroup",  LABELSTART+20, -1*(iStartTrack + _SUBGROUP*iIncrement));
    g2.rotate( -1.0 * (Math.PI / 2.0));

    g.drawLine(iStartTrack, LABELSTART,  iEndTrack,   LABELSTART);
    g.drawLine(iStartTrack, PLOT_TITLES, iEndTrack,   PLOT_TITLES);
    g.drawLine(iStartTrack, LABELSTART,  iStartTrack, PLOT_TITLES+iHeight);
    g.drawLine(iEndTrack,   LABELSTART,  iEndTrack,   PLOT_TITLES+iHeight);
  }

  /** Method drawBackground(Graphics g, int iStartTrack, int iWidthTrack)
   * <p> This method will draw the background colors
   * @param g - Graphics Parameter.
   * @param iRank       = The Type of Stratigraphic Unit
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawBackground(Graphics g, int iRank, int iStartTrack, int iWidthTrack)
  {
    int    i          = 0;
    int    j          = 0;
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iTotal     = 0;
    int    iCount     = -1;
    int    iContinue  = 0;
    String sName      = "";
    String sData[]    = null;
    String sTemp      = "";
    double data[][]   = null;
    int    iDiff      = 0;
    int    iStart     = iStartTrack;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;

    String sICS       = "";
    int    iRed       = 255;
    int    iGreen     = 255;
    int    iBlue      = 255;

    if (stStrat != null)
    {
      for (i=0; i<stStrat.iCount; i++)
      {
        switch (iRank)
        {
          case horizon.strat.stratStruct._ERA:
            iStart = _ERA*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].sEra)) &&
                 (stStrat.stItem[i].sEra.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].sEra);
            break;
          case horizon.strat.stratStruct._SYSTEM:
            iStart = _SYSTEM*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].system)) &&
                 (stStrat.stItem[i].system.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].system);
            break;
          case horizon.strat.stratStruct._SUBSYSTEM:
            iStart = _SUBSYSTEM*iIncrement;
            if ((!sName.equals(stStrat.stItem[i].subSystem)) &&
                 (stStrat.stItem[i].subSystem.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].subSystem);
            break;
          case horizon.strat.stratStruct._SERIES:

            iStart = _SERIES*iIncrement;
            sTemp = new String(stStrat.stItem[i].system + " " +
                               stStrat.stItem[i].series);

            if ((!sName.equals(sTemp)) &&
                (stStrat.stItem[i].series.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].system + " " +
                               stStrat.stItem[i].series);

            break;
          case horizon.strat.stratStruct._SUBSERIES:

            iStart = _SERIES*iIncrement;
            sTemp = new String(stStrat.stItem[i].subSystem + " " +
                               stStrat.stItem[i].subSeries);

            if ((!sName.equals(sTemp)) &&
                (stStrat.stItem[i].subSeries.length() > 0))
              iTotal++;
            sName = new String(stStrat.stItem[i].subSystem + " " +
                               stStrat.stItem[i].subSeries);

            break;
          case horizon.strat.stratStruct._STAGE:
          case horizon.strat.stratStruct._GROUP:
          case horizon.strat.stratStruct._SUBGROUP:
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._FORMATION:
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            break;
        }
      }

      sName = new String("");

      if (iTotal > 0)
      {
        sData  = new String[iTotal];
        data   = new double[iTotal][2];

        for (j=0; j<iTotal; j++)
        {
          sData[j]   = new String("");
          data[j][0] = 0.0;
          data[j][1] = 0.0;
        }

        iCount = -1;
        for (i=0; i<stStrat.iCount; i++)
        {
          iContinue = 0;
          switch (iRank)
          {
            case horizon.strat.stratStruct._ERA:
              if ((!sName.equals(stStrat.stItem[i].sEra)) &&
                   (stStrat.stItem[i].sEra.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].sEra);
                if (iCount < iTotal)
                  data[iCount][0] = stStrat.stItem[i].depthStart;
              }

              if (stStrat.stItem[i].sEra.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SYSTEM:
              if ((!sName.equals(stStrat.stItem[i].system)) &&
                   (stStrat.stItem[i].system.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].system);
                if (iCount < iTotal)
                  data[iCount][0] = stStrat.stItem[i].depthStart;
              }

              if (stStrat.stItem[i].system.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SUBSYSTEM:
              if ((!sName.equals(stStrat.stItem[i].subSystem)) &&
                   (stStrat.stItem[i].subSystem.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].subSystem);
                if (iCount < iTotal)
                  data[iCount][0] = stStrat.stItem[i].depthStart;
              }

              if (stStrat.stItem[i].subSystem.length() > 0)
                iContinue = 1;
              break;
            case horizon.strat.stratStruct._SERIES:

              sTemp = new String(stStrat.stItem[i].system + " " +
                                 stStrat.stItem[i].series);
              if ((!sName.equals(sTemp)) &&
                  (stStrat.stItem[i].series.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].system + " " +
                                   stStrat.stItem[i].series);
                if (iCount < iTotal)
                  data[iCount][0] = stStrat.stItem[i].depthStart;
              }

              if ((sTemp.length() > 1) &&
                  (stStrat.stItem[i].series.length() > 0))
              {
                iContinue = 1;
              }

              break;
            case horizon.strat.stratStruct._SUBSERIES:

              sTemp = new String(stStrat.stItem[i].subSystem + " " +
                                 stStrat.stItem[i].subSeries);

              if ((!sName.equals(sTemp)) &&
                  (stStrat.stItem[i].subSeries.length() > 0))
              {
                iCount++;
                sName = new String(stStrat.stItem[i].subSystem + " " +
                                   stStrat.stItem[i].subSeries);
                if (iCount < iTotal)
                  data[iCount][0] = stStrat.stItem[i].depthStart;
              }

              if ((sTemp.length() > 1) &&
                  (stStrat.stItem[i].subSeries.length() > 0))
              {
                iContinue = 1;
              }

              break;
            case horizon.strat.stratStruct._STAGE:
            case horizon.strat.stratStruct._GROUP:
            case horizon.strat.stratStruct._SUBGROUP:
            case horizon.strat.stratStruct._SUPER_GROUP:
            case horizon.strat.stratStruct._FORMATION:
            case horizon.strat.stratStruct._NONE:
            case horizon.strat.stratStruct._MEMBER:
            case horizon.strat.stratStruct._BED:
              break;
          }

          if (iContinue == 1)
          {
            if ((iCount < iTotal) && (iCount > -1))
            {
              sData[iCount]   = new String(sName);
              data[iCount][1] = stStrat.stItem[i].depthEnd;
            }
          }
        }

        for (i=0; i<iTotal; i++)
        {
          depth1 = data[i][0];
          depth2 = data[i][1];

          iContinue  = 1;
          if (((depth1 < depthStart) && (depth2 < depthStart)) ||
              ((depth1 > depthEnd)   && (depth2 > depthEnd)))
            iContinue = 0;

          if (iContinue == 1)
          {
            if (depth1 < depthStart)
              depth1 = depthStart;
            if (depth2 > depthEnd)
              depth2 = depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth1) /
                                           (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                                           (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;

            iDiff = iDepth2 - iDepth1;

            if (stICS != null)
            {
              for (j=0; j<stICS.iCount; j++)
              {
                if ((iRank !=  horizon.strat.stratStruct._SERIES) &&
                    (iRank !=  horizon.strat.stratStruct._SUBSERIES))
                {
                  sICS = new String(stICS.stItem[j].sName.toLowerCase());
                }

                else if (iRank ==  horizon.strat.stratStruct._SERIES)
                {
                  sICS = new String(stICS.stItem[j].system.toLowerCase() + " " +
                                    stICS.stItem[j].sName.toLowerCase());

                }
                else if (iRank ==  horizon.strat.stratStruct._SUBSERIES)
                {
                  sICS = new String(stICS.stItem[j].subSystem.toLowerCase() + " " +
                                    stICS.stItem[j].sName.toLowerCase());
                }

                if (sICS.equals(sData[i].toLowerCase()))
                {
                  iRed   = stICS.stItem[j].iRed;
                  iGreen = stICS.stItem[j].iGreen;
                  iBlue  = stICS.stItem[j].iBlue;
                }
              }

              g.setColor(new Color(iRed, iGreen, iBlue));
              g.fillRect(iStartTrack, iDepth1, iStart + iIncrement, iDiff);
            }
          }
        }
      }
    }
  }

  /** Method drawData(Graphics g, int iStartTrack, int iWidthTrack)
   * <p> This method will draw the members & formations track
   * @param g - Graphics Parameter.
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawData(Graphics g, int iStartTrack, int iWidthTrack)
  {
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iContinue  = 0;
    int    iFormation = 0;
    int    iStart     = iStartTrack;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;
    Font        fsb   = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        switch (stStrat.stItem[i].iRank)
        {
          case horizon.strat.stratStruct._SYSTEM:
          case horizon.strat.stratStruct._SERIES:
          case horizon.strat.stratStruct._SUBSYSTEM:
          case horizon.strat.stratStruct._SUBSERIES:
          case horizon.strat.stratStruct._STAGE:
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._GROUP:
          case horizon.strat.stratStruct._SUBGROUP:
            iContinue  = 0;
            break;
          case horizon.strat.stratStruct._FORMATION:
            iFormation = 1;
            iContinue  = 1;
            iStart     = iStartTrack+_FORMATION*iIncrement;
            break;
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            iFormation = 0;
            iContinue  = 1;
            iStart     = iStartTrack;
            break;
        }

        if (iContinue == 1)
        {
          depth1 = stStrat.stItem[i].depthStart;

          dRatio = (double) iLogHeight * (depthStart - depth1) /
                                         (depthStart - depthEnd);
          iDepth1 = PLOT_TITLES + (int) dRatio;

          iDepth2 = iDepth1;
          if (stStrat.stItem[i].depthEnd > stStrat.stItem[i].depthStart)
          {
            depth2 = stStrat.stItem[i].depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                                           (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;
          }

          iDepth = (iDepth1 + iDepth2)/2;

          if (((iDepth1 > PLOT_TITLES) && (iDepth1 < iHeight) &&
               (iDepth2 > PLOT_TITLES) && (iDepth2 < iHeight)) ||
               (iDepth  > PLOT_TITLES) && (iDepth  < iHeight))
          {
            if (iFormation == 1)
            {
              g.drawLine(iStartTrack,                        iDepth1,
                         iStartTrack + _SUBGROUP*iIncrement, iDepth1);
              g.drawLine(iStartTrack,                        iDepth2,
                         iStartTrack + _SUBGROUP*iIncrement, iDepth2);

              g.drawString(stStrat.stItem[i].sName, iStart, iDepth1+10);
            }
            else
            {
              g.setColor(Color.gray);
              g.drawLine(iStartTrack,                iDepth1,
                         iStartTrack + iIncrement,   iDepth1);
              g.drawLine(iStartTrack + 2*iIncrement, iDepth1,
                         iStartTrack + 3*iIncrement, iDepth1);
              g.drawLine(iStartTrack + 4*iIncrement, iDepth1,
                         iStartTrack + 5*iIncrement, iDepth1);
              g.drawLine(iStartTrack + 6*iIncrement, iDepth1,
                         iStartTrack + 7*iIncrement, iDepth1);

              g.setColor(Color.black);
              g.drawString(stStrat.stItem[i].sName, iStart, iDepth1+10);
            }
          }
        }
      }
    }
  }

  /** Method drawData(Graphics g, int iStartTrack, int iWidthTrack)
   * <p> This method will draw the vertical tracks
   * @param g - Graphics Parameter.
   * @param iRank       = The Type of Stratigraphic Unit
   * @param iStartTrack = The starting pixel for the  Track
   * @param iWidthTrack = The width of  Track in pixels
   */

  public void drawData(Graphics g, int iRank, int iStartTrack, int iWidthTrack)
  {
    int    iIncrement = iWidthTrack/_TOTAL;
    int    iEndTrack  = iStartTrack+iWidthTrack;
    int    iContinue  = 0;
    int    iStart     = iStartTrack;
    int    iDepth1    = 0;
    int    iDepth2    = 0;
    int    iDepth     = 0;
    int    iFound     = 0;
    double dRatio     = 0.0;
    double depth1     = 0.0;
    double depth2     = 0.0;

    String sName      = "";
    double dStart     = 0.0;
    double dEnd       = 0.0;

    String sName2     = "";
    double dStart2    = 0.0;
    double dEnd2      = 0.0;

    int    len        = 0;

    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (stStrat != null)
    {
      for (int i=0; i<stStrat.iCount; i++)
      {
        iContinue = 0;
        iFound    = 0;

        switch (iRank)
        {
          case horizon.strat.stratStruct._ERA:
            iStart = iStartTrack + _ERA*iIncrement;
            if (stStrat.stItem[i].sEra.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].sEra);
              sName2 = new String(stStrat.stItem[i].sEra);
            }
            break;
          case horizon.strat.stratStruct._SYSTEM:
            iStart = iStartTrack + _SYSTEM*iIncrement;
            if (stStrat.stItem[i].system.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].system);
              sName2 = new String(stStrat.stItem[i].system);
            }
            break;
          case horizon.strat.stratStruct._SUBSYSTEM:
            iStart = iStartTrack + _SUBSYSTEM*iIncrement;
            if (stStrat.stItem[i].subSystem.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subSystem);
              sName2 = new String(stStrat.stItem[i].subSystem);
            }
            break;
          case horizon.strat.stratStruct._SERIES:
            iStart = iStartTrack + _SERIES*iIncrement;
            if (stStrat.stItem[i].series.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].series);
              sName2 = new String(stStrat.stItem[i].series);
            }
            break;
          case horizon.strat.stratStruct._SUBSERIES:
            iStart = iStartTrack + _SERIES*iIncrement;
            if (stStrat.stItem[i].subSeries.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subSeries);
              sName2 = new String(stStrat.stItem[i].subSeries);
            }
            break;
          case horizon.strat.stratStruct._STAGE:
            iStart = iStartTrack + _STAGE*iIncrement;
            if (stStrat.stItem[i].stage.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].stage);
              sName2 = new String(stStrat.stItem[i].stage);
            }
            break;
          case horizon.strat.stratStruct._GROUP:
            iStart = iStartTrack + _GROUP*iIncrement;
            if (stStrat.stItem[i].sGroup.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].sGroup);
              sName2 = new String(stStrat.stItem[i].sGroup);
            }
            break;
          case horizon.strat.stratStruct._SUBGROUP:
            iStart = iStartTrack + _SUBGROUP*iIncrement;
            if (stStrat.stItem[i].subGroup.length() > 0)
            {
              iFound = 1;
              if (sName.length() == 0)
                sName = new String(stStrat.stItem[i].subGroup);
              sName2 = new String(stStrat.stItem[i].subGroup);
            }
            break;
          case horizon.strat.stratStruct._SUPER_GROUP:
          case horizon.strat.stratStruct._FORMATION:
          case horizon.strat.stratStruct._NONE:
          case horizon.strat.stratStruct._MEMBER:
          case horizon.strat.stratStruct._BED:
            break;
        }

        if ((iFound == 1) || (i == stStrat.iCount-1))
        {
          if ((!sName.equals(sName2)) || (i == stStrat.iCount-1))
          {
            iContinue = 1;

            dStart2 = stStrat.stItem[i].depthStart;
            dEnd2   = stStrat.stItem[i].depthEnd;
          }
          else
          {
            if (dStart == 0.0) dStart = stStrat.stItem[i].depthStart;
            dEnd = stStrat.stItem[i].depthEnd;
          }
        }

        if (iContinue == 1)
        {
          depth1 = dStart;
          depth2 = dEnd;

          if (((depth1 < depthStart) && (depth2 <= depthStart)) ||
              ((depth1 >= depthEnd)   && (depth2 > depthEnd)))
            iContinue = 0;

          if (iContinue == 1)
          {
            if (depth1 < depthStart) depth1 = depthStart;
            if (depth2 > depthEnd)   depth2 = depthEnd;

            dRatio = (double) iLogHeight * (depthStart - depth1) /
                (depthStart - depthEnd);
            iDepth1 = PLOT_TITLES + (int) dRatio;

            dRatio = (double) iLogHeight * (depthStart - depth2) /
                (depthStart - depthEnd);
            iDepth2 = PLOT_TITLES + (int) dRatio;

            iDepth = (iDepth1 + iDepth2) / 2;

            if (iRank > horizon.strat.stratStruct._SUBSERIES)
            {
              g.drawLine(iStartTrack,         iDepth1,
                         iStart + iIncrement, iDepth1);
            }

            Graphics2D g2 = (Graphics2D) g;
            g2.rotate( (Math.PI / 2.0));
            len = sName.length();
            if (iDepth-7*len/2 >= PLOT_TITLES)
              g.drawString(sName, iDepth-7*len/2, -1*iStart);
            g2.rotate( -1.0 * (Math.PI / 2.0));
          }

          if ((i == stStrat.iCount-1) && (!sName.equals(sName2)))
          {
            sName  = new String(sName2);
            dStart = dStart2;
            dEnd   = dEnd2;

            depth1 = dStart;
            depth2 = dEnd;

            if (((depth1 < depthStart) && (depth2 < depthStart)) ||
                ((depth1 > depthEnd)   && (depth2 > depthEnd)))
              iContinue = 0;

            if (iContinue == 1)
            {
              if (depth1 < depthStart) depth1 = depthStart;
              if (depth2 > depthEnd)   depth2 = depthEnd;

              dRatio = (double) iLogHeight * (depthStart - depth1) /
                  (depthStart - depthEnd);
              iDepth1 = PLOT_TITLES + (int) dRatio;

              dRatio = (double) iLogHeight * (depthStart - depth2) /
                  (depthStart - depthEnd);
              iDepth2 = PLOT_TITLES + (int) dRatio;

              iDepth = (iDepth1 + iDepth2) / 2;

              if (iRank > horizon.strat.stratStruct._SUBSERIES)
              {
                g.drawLine(iStartTrack,         iDepth1,
                           iStart + iIncrement, iDepth1);
              }

              Graphics2D g2 = (Graphics2D) g;
              g2.rotate( (Math.PI / 2.0));
              len = sName.length();
              if (iDepth-7*len/2 >= PLOT_TITLES)
                g2.drawString(sName, iDepth-7*len/2, -1*iStart);
              g2.rotate( -1.0 * (Math.PI / 2.0));
            }
          }

          sName  = new String(sName2);
          dStart = dStart2;
          dEnd   = dEnd2;
        }
      }
    }
  }

  /** Method draw()
   * <p> This method will draw all the ICS stratigraphic units.
   *  @param g - Graphics Handle;
   */

  public void draw(Graphics g)
  {
    drawBackground(g, horizon.strat.stratStruct._ERA,       iStartTrack, iWidthTrack);
    drawBackground(g, horizon.strat.stratStruct._SYSTEM,    iStartTrack, iWidthTrack);
    drawBackground(g, horizon.strat.stratStruct._SERIES,    iStartTrack, iWidthTrack);
    drawBackground(g, horizon.strat.stratStruct._SUBSYSTEM, iStartTrack, iWidthTrack);
    drawBackground(g, horizon.strat.stratStruct._SUBSERIES, iStartTrack, iWidthTrack);

    drawGrid(g, iStartTrack, iWidthTrack);

    drawData(g, iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._ERA,       iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._SYSTEM,    iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._SERIES,    iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._SUBSYSTEM, iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._SUBSERIES, iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._STAGE,     iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._GROUP,     iStartTrack, iWidthTrack);
    drawData(g, horizon.strat.stratStruct._SUBGROUP,  iStartTrack, iWidthTrack);
  }

  /** Method paint()
   *  <p> This method will paint the Scale data.
   *  @param  g      - Graphics Handle;
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 10/28/2008
 *  @author  John Victorine
 */
