/*
 * @stratPlotICS.java Version 1.1 10/24/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.plot;

import java.io.*;
import java.lang.*;
import java.util.Observable;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;

import horizon.strat.stratListStruct;

/** Class stratPlotICS
 *  <p> This method will create an interactive Stratigraphic Chart for the user
 *      to select Ages.
 *
 *  @version 1.1 10/24/2008
 *  @author  John R. Victorine
 */

public class stratPlotICS extends Canvas implements MouseListener
{
  private Observable notifier = null; // Observable Object

  private stratListStruct st = null;
  private String          selected = "";

  // Track Widths

  private static final int TRACK = 100;
  private static final int TRK_W = 20;

  // Start Track at:

  public static final int PLOT_TITLES = 55; //75; // 100; // Plot Titles
  public static final int LABELSTART  = 0;  //25;  // Label Start

  private int iWidth     = 800;
  private int iHeight    = PLOT_TITLES + 600;//1350;
  private int iLogHeight = 600;//1350;  // Height of Plot without the Titles

  private static final int START_X   = 10;
  private static final int END_X     = 190;

  private static final int SERIES    = 0;
  private static final int SUBSYSTEM = 100;
  private static final int SYSTEM    = 120;
  private static final int ERA       = 140;
  private static final int EON       = 160;

  private static final int ROW   = 40;  //25;

  private static final int TOTAL1 = 58;
  private static final int TOTAL2 = 73;
  private static final int LOC[][] = {
    { EON,     0, 15},  //37 },  // Phanerozoic
    { ERA,     0,  7 },  // Cenozoic
    { SYSTEM,  0,  2 },  // Quaternary
    { SERIES,  0,  1 },  // Holocene
    { SERIES,  1,  1 },  // Pleistocene

    { SYSTEM,  2,  2 },  // Neogene
    { SERIES,  2,  1 },  // Pliocene
    { SERIES,  3,  1 },  // Miocene

    { SYSTEM,  4,  3 },  //  Paleogene
    { SERIES,  4,  1 },  //  Oligocene
    { SERIES,  5,  1 },  //  Eocene
    { SERIES,  6,  1 },  //  Paleocene

    { ERA,     7,  8 },  //  Mesozoic
    { SYSTEM,  7,  2 },  //  Cretaceous
    { SERIES,  7,  1 },  //  Upper
    { SERIES,  8,  1 },  //  Lower

    { SYSTEM,  9,  3 },  //  Jurassic
    { SERIES,  9,  1 },  //  Upper
    { SERIES, 10,  1 },  //  Middle
    { SERIES, 11,  1 },  //  Lower

    { SYSTEM, 12,  3 },  //  Triassic
    { SERIES, 12,  1 },  //  Upper
    { SERIES, 13,  1 },  //  Middle
    { SERIES, 14,  1 },  //  Lower

    { 200+EON,    0, 12 },  // Phanerozoic
    { 200+ERA,    0, 12 },  //{ ERA,    15, 22 },  //  Paleozoic
    { 200+SYSTEM, 0,  3 },  //{ SUBSYSTEM, 15,  3 },  //  Permian
    { 200+SERIES, 0,  1 },  //{ SERIES, 15,  1 },  //  Lopingian
    { 200+SERIES, 1,  1 },  //{ SERIES, 16,  1 },  //  Guadalupian
    { 200+SERIES, 2,  1 },  //{ SERIES, 17,  1 },  //  Cisuralian

    { 200+SYSTEM, 3,  6 },  //{ SUBSYSTEM, 18,  6 },  //  Carboniferous

    { 200+SUBSYSTEM, 3, 3 }, //{ SUBSYSTEM, 18, 3 }, // Pennsylvanian
    { 200+SERIES, 3,  1 },  //{ SERIES, 18,  1 },  //  Upper
    { 200+SERIES, 4,  1 },  //{ SERIES, 19,  1 },  //  Middle
    { 200+SERIES, 5,  1 },  //{ SERIES, 20,  1 },  //  Lower

    { 200+SUBSYSTEM, 6, 3 }, //{ SUBSYSTEM, 21, 3 }, // Mississippian
    { 200+SERIES, 6,  1 },  //{ SERIES, 21,  1 },  //  Upper
    { 200+SERIES, 7,  1 }, //{ SERIES, 22,  1 },  //  Middle
    { 200+SERIES, 8,  1 },//{ SERIES, 23,  1 },  //  Lower

    { 200+SYSTEM, 9,  3 },  //{ SUBSYSTEM, 24,  3 },  //  Devonian
    { 200+SERIES, 9,  1 },  //{ SERIES, 24,  1 },  //  Upper
    { 200+SERIES, 10,  1 },  //{ SERIES, 25,  1 },  //  Middle
    { 200+SERIES, 11,  1 },  //{ SERIES, 26,  1 },  //  Lower

    { 400+EON,    0, 10 },  // Phanerozoic
    { 400+ERA,    0, 10 },  //{ ERA,    15, 22 },  //  Paleozoic
    { 400+SYSTEM, 0,  4 },  //{ SYSTEM, 27,  4 },  //  Silurian
    { 400+SERIES, 0,  1 },  //{ SERIES, 27,  1 },  //  Pridoli
    { 400+SERIES, 1,  1 },  //{ SERIES, 28,  1 },  //  Ludlow
    { 400+SERIES, 2,  1 },  //{ SERIES, 29,  1 },  //  Wentlock
    { 400+SERIES, 3,  1 },  //{ SERIES, 30,  1 },  //  Llandovery

    { 400+SYSTEM, 4,  3 },  //{ SYSTEM, 31,  3 },  //  Ordovician
    { 400+SERIES, 4,  1 },  //{ SERIES, 31,  1 },  //  Upper
    { 400+SERIES, 5,  1 },  //{ SERIES, 32,  1 },  //  Middle
    { 400+SERIES, 6,  1 },  //{ SERIES, 33,  1 },  //  Lower

    { 400+SYSTEM, 7,  3 },  //{ SYSTEM, 34,  3 },  //  Cambrian
    { 400+SERIES, 7,  1 },  //{ SERIES, 34,  1 },  //  Upper
    { 400+SERIES, 8,  1 },  //{ SERIES, 35,  1 },  //  Middle
    { 400+SERIES, 9,  1 },  //{ SERIES, 36,  1 },  //  Lower

    { 600+EON,    0, 10 },  // Cambrian
    { 600+ERA,    0, 10 },  //{ EON,    37, 10 },  //  Proterozoic
    { 600+SYSTEM, 0,  3 },  //{ ERA,    37,  3 },  //  Neoproterozoic
    { 600+SERIES, 0,  1 },  //{ SYSTEM, 37,  1 },  //  Ediacaran
    { 600+SERIES, 1,  1 },  //{ SYSTEM, 38,  1 },  //  Cryogenian
    { 600+SERIES, 2,  1 }, //{ SYSTEM, 39,  1 },  //  Tonian

    { 600+SYSTEM, 3,  3 },  //{ ERA,    40,  3 },  //  Mesoproterozoic
    { 600+SERIES, 3,  1 },  //{ SYSTEM, 40,  1 },  //  Stenian
    { 600+SERIES, 4,  1 },  //{ SYSTEM, 41,  1 },  //  Ectasian
    { 600+SERIES, 5,  1 },  //{ SYSTEM, 42,  1 },  //  Calymmian

    { 600+SYSTEM, 6,  4 },  //{ ERA,    43,  4 },  //  Paleoproterozoic
    { 600+SERIES, 6,  1 },  //{ SYSTEM, 43,  1 },  //  Statherian
    { 600+SERIES, 7,  1 },  //{ SYSTEM, 44,  1 },  //  Orosirian
    { 600+SERIES, 8,  1 },  //{ SYSTEM, 45,  1 },  //  Rhyacian
    { 600+SERIES, 9,  1 },  //{ SYSTEM, 46,  1 },  //  Siderian
  };

  private static final String NAME[][] = {
    {"10000000", "1",       "EON",    "Phanerozoic" },
    {"10100000", "1000000", "ERA",    "Cenozoic" },
    {"10101000", "1100000", "SYSTEM", "Quaternary" },
    {"10101010", "1110000", "SERIES", "Holocene" },
    {"10101020", "1120000", "SERIES", "Pleistocene" },

    {"10102000", "1200000", "SYSTEM", "Neogene" },
    {"10102010", "1210000", "SERIES", "Pliocene" },
    {"10102020", "1220000", "SERIES", "Miocene" },

    {"10103000", "1300000", "SYSTEM", "Paleogene" },
    {"10103010", "1310000", "SERIES", "Oligocene" },
    {"10103020", "1320000", "SERIES", "Eocene" },
    {"10103030", "1330000", "SERIES", "Paleocene" },

    {"10200000", "2000000", "ERA",    "Mesozoic" },
    {"10201000", "2100000", "SYSTEM", "Cretaceous" },
    {"10201010", "2110000", "SERIES", "Upper" },
    {"10201020", "2120000", "SERIES", "Lower" },

    {"10202000", "2200000", "SYSTEM", "Jurassic" },
    {"10202010", "2210000", "SERIES", "Upper" },
    {"10202020", "2220000", "SERIES", "Middle" },
    {"10202030", "2230000", "SERIES", "Lower" },

    {"10203000", "2300000", "SYSTEM", "Triassic" },
    {"10203010", "2310000", "SERIES", "Upper" },
    {"10203020", "2320000", "SERIES", "Middle" },
    {"10203030", "2330000", "SERIES", "Lower" },

    {"10000000", "1",       "EON",    "Phanerozoic" },
    {"10300000", "3000000", "ERA",    "Paleozoic" },
    {"10301000", "3100000", "SYSTEM", "Permian" },
    {"10301010", "3110000", "SERIES", "Lopingian" },
    {"10301020", "3120000", "SERIES", "Guadalupian" },
    {"10301030", "3130000", "SERIES", "Cisuralian" },

    {"10302000", "3200000", "SYSTEM", "Carboniferous" },

    {"10303000", "3300000", "SUBSYSTEM", "Pennsylvanian" },
    {"10303010", "3310000", "SUBSERIES", "Upper" },
    {"10303020", "3320000", "SUBSERIES", "Middle" },
    {"10303030", "3330000", "SUBSERIES", "Lower" },

    {"10304000", "3400000", "SUBSYSTEM", "Mississippian" },
    {"10304010", "3410000", "SUBSERIES", "Upper" },
    {"10304020", "3420000", "SUBSERIES", "Middle" },
    {"10304030", "3430000", "SUBSERIES", "Lower" },

    {"10305000", "3500000", "SYSTEM", "Devonian" },
    {"10305010", "3510000", "SERIES", "Upper" },
    {"10305020", "3520000", "SERIES", "Middle" },
    {"10305030", "3530000", "SERIES", "Lower" },

    {"10000000", "1",       "EON",    "Phanerozoic" },
    {"10300000", "3000000", "ERA",    "Paleozoic" },
    {"10306000", "3600000", "SYSTEM", "Silurian" },
    {"10306010", "3610000", "SERIES", "Pridoli" },
    {"10306020", "3620000", "SERIES", "Ludlow" },
    {"10306030", "3630000", "SERIES", "Wentlock" },
    {"10306040", "3640000", "SERIES", "Llandovery" },

    {"10307000", "3700000", "SYSTEM", "Ordovician" },
    {"10307010", "3710000", "SERIES", "Upper" },
    {"10307020", "3720000", "SERIES", "Middle" },
    {"10307030", "3730000", "SERIES", "Lower" },

    {"10308000", "3800000", "SYSTEM", "Cambrian" },
    {"10308010", "3810000", "SERIES", "Upper" },
    {"10308020", "3820000", "SERIES", "Middle" },
    {"10308030", "3830000", "SERIES", "Lower" },

    {"20000000", "3900001", "AGE",    "Precambrian"},
    {"20100000", "3900010", "EON",    "Proterozoic" },
    {"20200000", "4000000", "ERA",    "Neoproterozoic" },
    {"20201000", "4100000", "SYSTEM", "Ediacaran" },
    {"20202000", "4200000", "SYSTEM", "Cryogenian" },
    {"20203000", "4300000", "SYSTEM", "Tonian" },

    {"20300000", "5000000", "ERA",    "Mesoproterozoic" },
    {"20301000", "5100000", "SYSTEM", "Stenian" },
    {"20302000", "5200000", "SYSTEM", "Ectasian" },
    {"20303000", "5300000", "SYSTEM", "Calymmian" },

    {"20400000", "6000000", "ERA",    "Paleoproterozoic" },
    {"20401000", "6100000", "SYSTEM", "Statherian" },
    {"20402000", "6200000", "SYSTEM", "Orosirian" },
    {"20403000", "6300000", "SYSTEM", "Rhyacian" },
    {"20404000", "6400000", "SYSTEM", "Siderian" },
  };

  // Close Button

  private static final int iX1Close = 600;
  private static final int iXClose  = 100;
  private static final int iY1Close = 600;
  private static final int iYClose  = 30;

  private static final String CLOSE = "Close";

  /** CONSTRUCTOR stratPlotICS()
   *  <p> This is the constructor for this class.
   *  @param st = the stratigraphic unit list Structure for the International
   *              stratigraphic chart
   */

  public stratPlotICS( stratListStruct st )
  {
    this.st = st;
    this.setBackground(Color.white);
  }

  /** CONSTRUCTOR stratPlotICS()
   *  <p> This is the constructor for this class.
   *  @param notifier = Observable
   *  @param st = the stratigraphic unit list Structure for the International
   *              stratigraphic chart
   */

  public stratPlotICS( Observable notifier, stratListStruct st )
  {
    this.notifier = notifier;
    this.st = st;
    this.setBackground(Color.white);
    addMouseListener(this);
  }

  /** Method close()
   * <p> This method will force Java to deallocate memory
   */

  public void close()
  {
    notifier = null;
    st       = null;
    selected = null;
  }

  /* ======================================================================== *
   * ---------------------------- GET METHODS ------------------------------- *
   * ======================================================================== */

  /** Method getPlotWidth()
   * <p> This method will return the Width of the plot panel.
   * @return WIDTH = the width of the plot panel
   */

  public int getPlotWidth() { return (iWidth); }

  /** Method getPlotHeight()
   * <p> This method will return the Width of the plot panel.
   * @return the height of the plot panel
   */

  public int getPlotHeight() { return (iHeight); }

  /** Method getData()
   * <p> This method will return the ID of the data selected.
   * @return selected = the id for the ICS Stratigraphic unit selected.
   */

  public String getData() { return (selected); }

  /* =============================================================== *
   * -------------------- MouseListener Methods -------------------- *
   * =============================================================== */

  public void mouseMoved(MouseEvent event) { }
  public void mouseEntered(MouseEvent event) { }
  public void mouseExited(MouseEvent event) { }
  public void mousePressed(MouseEvent event) { }
  public void mouseDragged(MouseEvent event) {}
  public void mouseReleased(MouseEvent event) {}

  /** Method mouseClicked()
   *  <p> Mouse Clicked Method retrieve the location of the mouse pointer
   *      location.
   * @param event - Mouse Clicked Event
   */

  public void mouseClicked(MouseEvent event)
  {
    int    iXm     = event.getX();
    int    iYm     = event.getY();
    int    iBegin  = 0;
    int    iEnd    = 0;
    int    iStartTrack = 0;
    int    iEndTrack   = 0;
    int    iRank       = -1;

    selected = new String("");

    if (st != null)
    {
      for (int i=0; i<TOTAL2; i++)
      {
        for (int j=0; j<st.iCount; j++)
        {
          if (st.stItem[j].sKEY.equals(NAME[i][0]))
          {
            iRank  = st.stItem[j].iRank;
          }
        }

        iBegin = PLOT_TITLES + ROW * LOC[i][1];
        iEnd   = PLOT_TITLES + ROW * ( LOC[i][1] + LOC[i][2] );

        if (i < TOTAL1)
        {
          if ((iRank == horizon.strat.stratStruct._SERIES) ||
              (iRank == horizon.strat.stratStruct._SUBSERIES))
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRACK;
          }
          else if (iRank == horizon.strat.stratStruct._SUBSYSTEM)
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRK_W;
          }
          else if ((iRank == horizon.strat.stratStruct._SYSTEM) &&
                   (!NAME[i][3].equals("Carboniferous")))
          {
            iStartTrack = START_X + LOC[i][0] - TRK_W;
            iEndTrack   = 2*TRK_W;
          }
          else
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRK_W;
          }
        }
        else
        {
          if (iRank == horizon.strat.stratStruct._SYSTEM)
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRACK;
          }
          else if (iRank == horizon.strat.stratStruct._ERA)
          {
            iStartTrack = START_X + LOC[i][0] - TRK_W;
            iEndTrack   = 2*TRK_W;
          }
          else
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRK_W;
          }
        }

        if ((iYm >= iBegin) && (iYm < iEnd) &&
            (iXm >= iStartTrack) && (iXm < iStartTrack+iEndTrack))
        {
          selected = new String(NAME[i][0]);
          i = TOTAL2;

          if (notifier != null)
            notifier.notifyObservers(
                new String("ICS Stratigraphic Unit Selected"));
        }
      }
    }

    if ((iYm >= iY1Close) && (iYm < iY1Close+iYClose) &&
        (iXm >= iX1Close) && (iXm < iX1Close+iXClose))
    {
      if (notifier != null)
        notifier.notifyObservers(new String("Close ICS Dialog"));
    }
  }

  /* ======================================================================== *
   * ------------------------------- PAINT ---------------------------------- *
   * ======================================================================== */

  /** Method drawPrecambrianGrid(Graphics g)
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   */

  public void drawPrecambrianGrid(Graphics g)
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iY1    = 0;
    int    iY2    = 0;
    int    iXo     = 0;
    int    iXe     = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

//    for (i=0; i<4; i++)
//    {
      iXo = START_X + 3*(10 + END_X);
      iXe = END_X   + 3*(10 + END_X);

      g.drawLine(iXo+SUBSYSTEM, LABELSTART, iXo+SUBSYSTEM, PLOT_TITLES);
      g.drawLine(iXo+ERA, LABELSTART, iXo+ERA, PLOT_TITLES);
      g.drawLine(iXo+EON, LABELSTART, iXo+EON, PLOT_TITLES);

      g.drawString("System", iXo+30, LABELSTART+20);

      Graphics2D g2 = (Graphics2D) g;
      g2.rotate( (Math.PI / 2.0));
      g2.drawString("EON",    LABELSTART+20, -1*(iXo+ERA+5));
      g2.drawString("Era",    LABELSTART+20, -1*(iXo+SYSTEM+5));
      g2.rotate( -1.0 * (Math.PI / 2.0));

      g.drawLine(iXo,      LABELSTART,    iXe,   LABELSTART);
      g.drawLine(iXo,      PLOT_TITLES,   iXe,   PLOT_TITLES);
      g.drawLine(iXo,      LABELSTART,    iXo,   PLOT_TITLES);
      g.drawLine(iXe,      LABELSTART,    iXe,   PLOT_TITLES);
//    }
  }

  /** Method drawPrecambrianBackground()
   * <p> This method will draw the stratigraphic tracks background colors
   * @param g - Graphics Parameter.
   */

  public void drawPrecambrianBackground(Graphics g)
  {
    int i = 0;
    int j = 0;
    int iStart = 0;
    int iRank  = horizon.strat.stratStruct._NONE;
    int iRed   = 255;
    int iGreen = 255;
    int iBlue  = 255;
    int iBegin = 0;
    int iEnd   = 0;
    int iDepth = 0;
    int iDiff  = 0;
    int iStartTrack = 0;
    int iEndTrack   = 0;
    int loc    = -1;
    Font        fsb   = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (st != null)
    {
      for (i=TOTAL1; i<TOTAL2; i++)
      {
        loc = -1;

        for (j=0; j<st.iCount; j++)
        {
          if (st.stItem[j].sKEY.equals(NAME[i][0]))
          {
            iRank  = st.stItem[j].iRank;
            iRed   = st.stItem[j].iRed;
            iGreen = st.stItem[j].iGreen;
            iBlue  = st.stItem[j].iBlue;
            loc    = i;
          }
        }

        if (loc > -1)
        {
          iBegin = PLOT_TITLES + ROW * LOC[loc][1];
          iEnd   = PLOT_TITLES + ROW * ( LOC[loc][1] + LOC[loc][2] );
          iDepth = ( iBegin + iEnd ) / 2;
          iDiff  = iEnd - iBegin;

          if (iRank == horizon.strat.stratStruct._SYSTEM)
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRACK;
          }
          else if (iRank == horizon.strat.stratStruct._ERA)
          {
            iStartTrack = START_X + LOC[i][0] - TRK_W;
            iEndTrack   = 2*TRK_W;
          }
          else
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRK_W;
          }

          g.setColor(new Color(iRed, iGreen, iBlue));
          g.fillRect(iStartTrack, iBegin, iEndTrack, iDiff);

          g.setColor(Color.black);
          g.drawRect(iStartTrack, iBegin, iEndTrack, iDiff);
        }
      }
    }
  }

  /** Method drawPrecambrianData()
   * <p> This method will draw the Precambrian stratigraphic tracks
   * @param g - Graphics Parameter.
   */

  public void drawPrecambrianData(Graphics g)
  {
    int i = 0;
    int j = 0;
    int iStart = 0;
    int iRank  = horizon.strat.stratStruct._NONE;
    int iRed   = 255;
    int iGreen = 255;
    int iBlue  = 255;
    int iBegin = 0;
    int iEnd   = 0;
    int iDepth = 0;
    int loc    = -1;
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (st != null)
    {
      for (i=TOTAL1; i<TOTAL2; i++)
      {
        loc = -1;

        for (j=0; j<st.iCount; j++)
        {
          if (st.stItem[j].sKEY.equals(NAME[i][0]))
          {
            iRank  = st.stItem[j].iRank;
            iRed   = st.stItem[j].iRed;
            iGreen = st.stItem[j].iGreen;
            iBlue  = st.stItem[j].iBlue;
            loc    = i;
          }
        }

        if (loc > -1)
        {
          iStart = START_X + LOC[i][0];
          iBegin = PLOT_TITLES + ROW * LOC[loc][1];
          iEnd   = PLOT_TITLES + ROW * ( LOC[loc][1] + LOC[loc][2] );
          iDepth = ( iBegin + iEnd ) / 2;

          int len = NAME[loc][3].length();

          if (iRank != horizon.strat.stratStruct._SYSTEM)
          {
            Graphics2D g2 = (Graphics2D) g;
            g2.rotate( (Math.PI / 2.0));
//            g2.drawString(NAME[loc][3], iDepth-4*len, -1 * iStart);
            g2.drawString(NAME[loc][3], iDepth-7*len/2, -1 * (iStart+6));
            g2.rotate( -1.0 * (Math.PI / 2.0));
          }
          else
          {
            g.drawString(NAME[loc][3], iStart+TRACK/2-4*len, iDepth);
          }
        }
      }
    }
  }

  /** Method drawGrid(Graphics g)
   * <p> This method will draw the depth scale
   * @param g - Graphics Parameter.
   */

  public void drawGrid(Graphics g)
  {
    int    i=0;
    int    j=0;
    int    jXinc  = 0;
    int    iLog   = 0;
    int    iCycle = 1;
    double dXinc  = 0.0;
    double dTemp  = 0.0;
    int    iY1    = 0;
    int    iY2    = 0;
    int    iXo     = 0;
    int    iXe     = 0;

    Font        fsb  = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    for (i=0; i<3; i++)
    {
      iXo = START_X + i*(10 + END_X);
      iXe = END_X   + i*(10 + END_X);

      g.drawLine(iXo+SUBSYSTEM, LABELSTART, iXo+SUBSYSTEM, PLOT_TITLES);
      g.drawLine(iXo+ERA, LABELSTART, iXo+ERA, PLOT_TITLES);
      g.drawLine(iXo+EON, LABELSTART, iXo+EON, PLOT_TITLES);

      g.drawString("Series", iXo+30, LABELSTART+20);

      Graphics2D g2 = (Graphics2D) g;
      g2.rotate( (Math.PI / 2.0));
      g2.drawString("EON",       LABELSTART+20, -1*(iXo+EON+5));
      g2.drawString("Era",       LABELSTART+20, -1*(iXo+ERA+5));
      g2.drawString("System",    LABELSTART+10, -1*(iXo+SYSTEM+5));
      g2.rotate( -1.0 * (Math.PI / 2.0));

      g.drawLine(iXo,      LABELSTART,    iXe,   LABELSTART);
      g.drawLine(iXo,      PLOT_TITLES,   iXe,   PLOT_TITLES);
      g.drawLine(iXo,      LABELSTART,    iXo,   PLOT_TITLES);
      g.drawLine(iXe,      LABELSTART,    iXe,   PLOT_TITLES);
    }
  }

  /** Method drawBackground()
   * <p> This method will draw the stratigraphic tracks background colors
   * @param g - Graphics Parameter.
   */

  public void drawBackground(Graphics g)
  {
    int i = 0;
    int j = 0;
    int iStart = 0;
    int iRank  = horizon.strat.stratStruct._NONE;
    int iRed   = 255;
    int iGreen = 255;
    int iBlue  = 255;
    int iBegin = 0;
    int iEnd   = 0;
    int iDepth = 0;
    int iDiff  = 0;
    int iStartTrack = 0;
    int iEndTrack   = 0;
    int loc    = -1;
    Font        fsb   = new Font("Serif", Font.BOLD, 10);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (st != null)
    {
      for (i=0; i<TOTAL1; i++)
      {
        loc = -1;

        for (j=0; j<st.iCount; j++)
        {
          if (st.stItem[j].sKEY.equals(NAME[i][0]))
          {
            iRank  = st.stItem[j].iRank;
            iRed   = st.stItem[j].iRed;
            iGreen = st.stItem[j].iGreen;
            iBlue  = st.stItem[j].iBlue;
            loc    = i;
          }
        }

        if (loc > -1)
        {
          iBegin = PLOT_TITLES + ROW * LOC[loc][1];
          iEnd   = PLOT_TITLES + ROW * ( LOC[loc][1] + LOC[loc][2] );
          iDepth = ( iBegin + iEnd ) / 2;
          iDiff  = iEnd - iBegin;

          if ((iRank == horizon.strat.stratStruct._SERIES) ||
              (iRank == horizon.strat.stratStruct._SUBSERIES))
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRACK;
          }
          else if (iRank == horizon.strat.stratStruct._SYSTEM)
          {
            iStartTrack = START_X + LOC[i][0] - TRK_W;
            iEndTrack   = 2*TRK_W;
          }
          else
          {
            iStartTrack = START_X + LOC[i][0];
            iEndTrack   = TRK_W;
          }

          g.setColor(new Color(iRed, iGreen, iBlue));
          g.fillRect(iStartTrack, iBegin, iEndTrack, iDiff);

          g.setColor(Color.black);
          g.drawRect(iStartTrack, iBegin, iEndTrack, iDiff);
        }
      }
    }
  }

  /** Method drawData()
   * <p> This method will draw the stratigraphic tracks
   * @param g - Graphics Parameter.
   */

  public void drawData(Graphics g)
  {
    int i = 0;
    int j = 0;
    int iStart = 0;
    int iRank  = horizon.strat.stratStruct._NONE;
    int iRed   = 255;
    int iGreen = 255;
    int iBlue  = 255;
    int iBegin = 0;
    int iEnd   = 0;
    int iDepth = 0;
    int loc    = -1;
    Font        fsb   = new Font("Serif", Font.BOLD, 12);
    FontMetrics fsbm  = g.getFontMetrics(fsb);

    g.setFont( fsb );
    g.setColor(Color.black);

    if (st != null)
    {
      for (i=0; i<TOTAL1; i++)
      {
        loc = -1;

        for (j=0; j<st.iCount; j++)
        {
          if (st.stItem[j].sKEY.equals(NAME[i][0]))
          {
            iRank  = st.stItem[j].iRank;
            iRed   = st.stItem[j].iRed;
            iGreen = st.stItem[j].iGreen;
            iBlue  = st.stItem[j].iBlue;
            loc    = i;
          }
        }

        if (loc > -1)
        {
          iStart = START_X + LOC[i][0];
          iBegin = PLOT_TITLES + ROW * LOC[loc][1];
          iEnd   = PLOT_TITLES + ROW * ( LOC[loc][1] + LOC[loc][2] );
          iDepth = ( iBegin + iEnd ) / 2;

          int len = NAME[loc][3].length();

          if ((iRank != horizon.strat.stratStruct._SERIES) &&
              (iRank != horizon.strat.stratStruct._SUBSERIES))
          {
            Graphics2D g2 = (Graphics2D) g;
            g2.rotate( (Math.PI / 2.0));
//            g2.drawString(NAME[loc][3], iDepth-4*len, -1 * iStart);
            g2.drawString(NAME[loc][3], iDepth-7*len/2, -1 * (iStart+6));
            g2.rotate( -1.0 * (Math.PI / 2.0));
          }
          else
          {
            g.drawString(NAME[loc][3], iStart+TRACK/2-4*len, iDepth);
          }
        }
      }
    }
  }

  /** Method drawClose()
   * <p> This method will draw all the ICS stratigraphic units.
   *  @param g - Graphics Handle;
   */

  public void drawClose(Graphics g)
  {
    if (notifier != null)
    {
      g.setColor(Color.black);
      g.draw3DRect(iX1Close, iY1Close, iXClose, iYClose, true);
      int len = CLOSE.length();
      g.setColor(Color.black);
      g.drawString(CLOSE, iX1Close+iXClose/2-4*len, iY1Close+3*iYClose/4);
    }
  }

  /** Method draw()
   * <p> This method will draw all the ICS stratigraphic units.
   *  @param g - Graphics Handle;
   */

  public void draw(Graphics g)
  {
    drawBackground(g);
    drawGrid(g);
    drawData(g);

    drawPrecambrianBackground(g);
    drawPrecambrianGrid(g);
    drawPrecambrianData(g);

    drawClose(g);
  }

  /** Method paint()
   *  <p> This method will paint the Scale data.
   *  @param  g      - Graphics Handle;
   */

  public void paint(Graphics g)
  {
    g.setColor(Color.white);
    g.fillRect(0, 0, iWidth, iHeight);

    draw(g);
  }
}

/*
 *  @version 1.1 10/24/2008
 *  @author  John Victorine
 */
