/*
 * @ReadStratUnitsXMLFile.java Version 1.0 10/17/2008
 *
 * Copyright (c) 2008 John R. Victorine
 * All Rights Reserved.
 */

package horizon.strat.io;

import java.util.*;
import java.io.*;
import java.net.*;
import java.awt.*;
import java.util.zip.*;

import org.xml.sax.*;
import org.xml.sax.helpers.DefaultHandler;

import javax.xml.parsers.SAXParserFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.parsers.SAXParser;

import cmn.cmnString;
import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

/** Class ReadStratUnitsXMLFile()
 * <p> This Class will parse the Regions XML Data Stream
 *     directly to the Regions List Data Structure.
 *
 *  @version 1.1 10/17/2008
 *  @author  John R. Victorine
 */

public class ReadStratUnitsXMLFile extends DefaultHandler
{
  public static final int FILE       = 0;
  public static final int URL        = 1;
  public static final int SERVER     = 2;
  public static final int SERVER_ZIP = 3;

  public static final String ROOTNAME     = "strat";     // Root Name of XML
  public static final String RECORDS      = "records";   // Record Count

  public static final String WELL         = "well";      // ELEMENT info

  public static final String KID          = "kid";       // Well Primary key
  public static final String KEY          = "key";       // Primary Key
  public static final String TYPE         = "type";      // type of data
  public static final String API_NUMBER   = "api";       // API-Number
  public static final String WELL_STATUS  = "status";    // Well Status
  public static final String ERROR_VAL    = "error";     // DB Error
  public static final String LATITUDE     = "latitude";  // Latitude
  public static final String LONGITUDE    = "longitude"; // Longitude
  public static final String DEPTH        = "depth";     // Depth or Height
  public static final String GROUND_LEVEL = "gl";        // Ground Level

  public static final String AGE          = "age";       // ELEMENT age
  public static final String ID           = "id";        // Unique ID
  public static final String RANK         = "rank";      // Rank of formation

  public static final String NAME         = "name";      // Strat Unit Name
  public static final String NAME2        = "alt_name";  // Alternate Region Name
  public static final String SOURCE       = "source";    // Source

  public static final String DEPTH_START  = "top";       // Starting Depth
  public static final String DEPTH_END    = "base";      // Ending Depth

  public static final String AGE_START    = "start";     // Starting date
  public static final String START_CORR   = "s_cor";     // +/- start date
  public static final String AGE_END      = "end";       // Ending Date
  public static final String END_CORR     = "e_cor";     // +/- end date

  public static final String LEVEL        = "level";     // Level of confidence
  public static final String RED          = "red";       // Red Value
  public static final String GREEN        = "green";     // Green Value
  public static final String BLUE         = "blue";      // Blue Value
  public static final String EON          = "eon";       // eon
  public static final String ERA          = "era";       // era
  public static final String SYSTEM       = "system";    // system
  public static final String SERIES       = "series";    // series
  public static final String SUBSYSTEM    = "subsystem"; // subsystem
  public static final String SUBSERIES    = "subseries"; // subseries
  public static final String STAGE        = "stage";     // stage
  public static final String SUPER_GROUP  = "sprgroup";  // super group
  public static final String GROUP        = "group";     // group
  public static final String SUBGROUP     = "subgroup";  // subgroup
  public static final String FORMATION    = "form";      // Formation

  private static final int    SYSTEM_TOTAL = 12;
  private static final String SYSTEM_ID[][] = {
  //     KGS KID,       Name,          KGS ID,       ID
      {"1029095988", "Quaternary",    "100500", "10101000"},
      {"1027710578", "Cretaceous",    "101000", "10201000"},
      {"1027710665", "Jurassic",      "120000", "10202000"},
      {"1027710667", "Triassic",      "122000", "10203000"},
      {"1027710669", "Permian",       "124000", "10301000"},
      {"1027710965", "Pennsylvanian", "194000", "10303000"},
      {"1027711178", "Mississippian", "358000", "10304000"},
      {"1027711206", "Devonian",      "387000", "10305000"},
      {"1027711209", "Silurian",      "390000", "10306000"},
      {"1027711210", "Ordovician",    "391000", "10307000"},
      {"1027711222", "Cambrian",      "405000", "10308000"},
      {"1027711226", "Precambrian",   "409000", "20000000"}};

  private int             iType  = FILE;
  private int             iRows  =  0;
  private int             iCount = -1;
  private int             iError = 0;
  private String          sError = "";
  private stratListStruct stList = new stratListStruct();

  /** Method ReadStratUnitsXMLFile()
   *  This is the Default Constructor for this class.
   */

  public ReadStratUnitsXMLFile() { iType = FILE; }

  /** Method ReadStratUnitsXMLFile()
   *  This is the Constructor for this class.
   *  @param iType = The source identifier,
   *                 0 = File is an absolute Directory Path + Filename
   *                 1 = Source is a URL Location.
   *                 2 = Server Location
   *                 3 = WebStart Location
   */

  public ReadStratUnitsXMLFile(int iType) { this.iType = iType; }

  /** Method Process()
   *  <p> This method will Process the XML File into a Array List.
   *      It is a double pass method in that it Reads the XML File
   *      once to get the number of lines in the XML File and a
   *      second time to actually read the XML File.
   *  @param  filename = The File to be opened.
   *  @return stList   = The XML List Data Structure (Array List).
   */

  public stratListStruct Process(String filename)
  {
    stList.iCount = 0;
    Read(filename);
    stList.iCount = iCount+1;

    if (stList != null)
    {
      for (int i=0; i<stList.iCount; i++)
      {
        if ((stList.stItem[i].sKEY.equals("0")) &&
            (!stList.stItem[i].sKID.equals("0")))
          stList.stItem[i].sKEY = new String(stList.stItem[i].sKID);
      }
    }

    return (stList);
  }

  /** Method Read()
   *  <p> This method will Read & Parse the XML File.
   *  @param filename = the filename of the XML File.
   */

  public void Read(String filename)
  {
    // Use the validating parser

    SAXParserFactory factory = SAXParserFactory.newInstance();
    factory.setValidating(true);

    try
    {
      SAXParser saxParser = factory.newSAXParser();
      if (iType == FILE)
      {
        saxParser.parse( new File(filename), this);
      }
      else if (iType == URL)
      {
        try
        {
          InputSource source = new InputSource(filename);
          saxParser.parse(source, this);
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else if (iType == SERVER)
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          InputStream in = connection.getInputStream();
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
      else
      {
        try
        {
          // Connect to the server

          URL u = new URL(filename);
          URLConnection uc = u.openConnection();
          HttpURLConnection connection = (HttpURLConnection) uc;
          connection.setDoOutput(true);
          connection.setDoInput(true);
          connection.setRequestMethod("POST");

          // Read the response XML Document

          GZIPInputStream in = new GZIPInputStream(connection.getInputStream());
          InputSource source = new InputSource(in);
          saxParser.parse(source, this);
          in.close();
          connection.disconnect();
        }
        catch (Exception e)
        {
          System.err.println(e);
        }
      }
    }
    catch (SAXParseException spe)
    {
      // Error generated by the parser

      System.out.println("\n** Parsing error"            +
                         ", line " + spe.getLineNumber() +
                         ", uri "  + spe.getSystemId());
      System.out.println("   " + spe.getMessage() );

      // Use the contained exception, if any

      Exception  x = spe;
      if (spe.getException() != null)
        x = spe.getException();
      x.printStackTrace();
    }
    catch (SAXException sxe)
    {
      // Error generated by this application (or a parser-initialization error)

      Exception  x = sxe;
      if (sxe.getException() != null)
        x = sxe.getException();
      x.printStackTrace();
    }
    catch (ParserConfigurationException pce)
    {
      // Parser with specified options can't be built

      pce.printStackTrace();
    }
    catch (IOException ioe)
    {
      // I/O error

      ioe.printStackTrace();
    }
  }

  //===========================================================
  // SAX DocumentHandler methods
  //===========================================================

  /** Method setDocumentLocator()
   * <p> This is the Set Document Locator Method
   * @param l Document Locator
   */

  public void setDocumentLocator(Locator l)
  {
    // Ignore errors
  }

  /** Method startDocument()
   * <p> This is the Start Document Method
   * @throws SAXException
   */

  public void startDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method endDocument()
   * <p> This is the End Document Method
   * @throws SAXException
   */

  public void endDocument() throws SAXException
  {
    // Ignore it
  }

  /** Method startElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param lName        = Local Name
   * @param qName        = Qualified Name
   * @param attrs        = Attributes
   * @throws SAXException
   */

  public void startElement(String namespaceURI,
                           String lName, // local name
                           String qName, // qualified name
                           Attributes attrs) throws SAXException
  {
    String eName = lName; // element name

    if ("".equals(eName))
      eName = qName; // namespaceAware = false

    // If the ELEMENT Well is detected then initialize the structure.

    if (eName.equals(AGE))
    {
      iCount++;
      if (iCount < iRows)
      {
        stList.stItem[iCount] = new stratStruct();
      }
    }

    // Add Attributes to the Well Header Data List Structure

    if (attrs != null)
    {
      for (int i=0; i<attrs.getLength(); i++)
      {
        String aName = attrs.getLocalName(i); // Attr name
        if ("".equals(aName))
          aName = attrs.getQName(i);

        String sTemp = new String(attrs.getValue(i));
        String sOut  = removeExcess(sTemp);

        // Get the total number of records available

        if (aName.equals(RECORDS))
        {
          iRows = cmn.cmnString.stringToInt(sOut.trim());

          if (iRows > 0)
          {
            stList.stItem = new stratStruct[iRows];
          }
        }

        // Parse each attribute and add it to the structure

        if (eName.equals(WELL))
        {
          ParseData(aName, sOut.trim());
        }

        if (iCount > -1)
        {
          if (iCount < iRows)
          {
            if (eName.equals(AGE))
            {
              stList.stItem[iCount] =
                  ParseData(aName, sOut.trim(), stList.stItem[iCount]);
            }
          }
        }

        // End Add Attributes to List
      }
    }
  }

  /** Method endElement()
   * <p> This method will parse the ELEMENT from an XML File
   * @param namespaceURI = Name Space URI
   * @param sName        = Simple Name
   * @param qName        = Qualified Name
   * @throws SAXException
   */

  public void endElement(String namespaceURI,
                         String sName, // simple name
                         String qName  // qualified name
                        ) throws SAXException
  {
    // Ignore it
  }

  /** Method characters()
   * <p> This method will parse the CHARACTERS from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void characters(char buf[], int offset, int len) throws SAXException
  {
    // Ignore it
  }

  /** Method ignorableWhitespace()
   * <p> This method will parse the Ignorable White Space from an XML File
   * @param buf    = Buffer Character Array holding the characters
   * @param offset = String Offset
   * @param len    = Length of String
   * @throws SAXException
   */

  public void ignorableWhitespace(char buf[], int offset, int len)
    throws SAXException
  {
    // Ignore it
  }

  /** Method processingInstruction()
   * <p> This method will processing Instructions for a XML File
   * @param target = Target
   * @param data   = Data
   * @throws SAXException
   */

  public void processingInstruction(String target, String data)
    throws SAXException
  {
    // Ignore it
  }

  //===========================================================
  // SAX ErrorHandler methods
  //===========================================================

  /** Method error()
   * <p> This method will treat validation errors as fatal
   * @param e = SAX Parse Exception
   * @throws SAXParseException
   */

  public void error(SAXParseException e) throws SAXParseException { throw e; }

  /** Method warning()
   * <p> This method will treat warnings
   * @param err = SAX Parse Exception
   * @throws SAXParseException
   */

  public void warning(SAXParseException err) throws SAXParseException
  {
    System.out.println("** Warning" +
                       ", line "    + err.getLineNumber() +
                       ", uri "     + err.getSystemId());
    System.out.println("   " + err.getMessage());
  }

  //===========================================================
  // Utility Methods ...
  //===========================================================

  /** Method removeExcess()
   * <p> Remove excess white space within a string
   * @param  sin  = String to be parsed of excess spaces
   * @return sout = String with the excess spaces removed
   */

  private String removeExcess(String sin)
  {
    String sout   = new String("");
    char   chold  = ' ';
    char   ch[];

    ch = sin.toCharArray();

    for (int i=0; i<ch.length; i++)
    {
      if (i==0)
        chold = ch[i];

      if (chold != ' ')
      {
        sout = new String(sout + ch[i]);
      }
      else if (chold == ' ')
      {
        if (ch[i] != ' ')
        {
          sout = new String(sout + ch[i]);
        }
      }

      chold = ch[i];
    }

    return (sout);
  }

  /** Method ParseData()
   * <p> This method will parse the Data Stream
   * @param sIdentifier = The Attribute Identifier
   * @param  sData      = Stratigraphic Units String
   * @param  st         = The Stratigraphic Units Data Structure
   * @return st         = The Stratigraphic Units Data Structure
   */

  private stratStruct ParseData( String sIdentifier,
                                 String sData,
                                 stratStruct st )
  {
    int i=0;

    if (sIdentifier.equals(KID)) { st.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { st.sKEY = new String(sData); }

    if (sIdentifier.equals(ID))             // Unique ID
    {
      st.sid = new String(sData);
//      if (cmn.cmnString.isNumeric(sData))
//      {
//        st.id = cmn.cmnString.stringToInt(sData);
//      }
    }

    if (sIdentifier.equals(RANK))           // Rank of formation
    {
      for (i=0; i<horizon.strat.stratStruct.RANK.length; i++)
      {
        if (sData.equals(horizon.strat.stratStruct.RANK[i])) st.iRank = i;
      }
    }

    if (sIdentifier.equals(NAME))   { st.sName  = new String(sData); }
    if (sIdentifier.equals(NAME2))  { st.sName2 = new String(sData); }
    if (sIdentifier.equals(SOURCE)) { st.source = new String(sData); }

    if (sIdentifier.equals(DEPTH_START))    // Starting Depth
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.depthStart = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(DEPTH_END))      // Ending Depth
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.depthEnd = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(AGE_START))      // Starting date
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dAgeStart = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(START_CORR))     // +/- start date
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dStartCorr = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(AGE_END))        // Ending Date
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dAgeEnd = cmn.cmnString.stringToDouble(sData);
      }
    }
    if (sIdentifier.equals(END_CORR))       // +/- end date
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.dEndCorr = cmn.cmnString.stringToDouble(sData);
      }
    }

    if (sIdentifier.equals(LEVEL))          // Level of confidence
    {
      for (i=0; i<horizon.strat.stratStruct.LEVEL.length; i++)
      {
        if (sData.equals(horizon.strat.stratStruct.LEVEL[i])) st.iLevel = i;
      }
    }

    if (sIdentifier.equals(RED))            // Red Value
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iRed = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(GREEN))          // Green Value
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iGreen = cmn.cmnString.stringToInt(sData);
      }
    }
    if (sIdentifier.equals(BLUE))           // Blue Value
    {
      if (cmn.cmnString.isNumeric(sData))
      {
        st.iBlue = cmn.cmnString.stringToInt(sData);
      }
    }

    if (sIdentifier.equals(EON))         { st.sEon       = new String(sData); }
    if (sIdentifier.equals(ERA))         { st.sEra       = new String(sData); }
    if (sIdentifier.equals(SYSTEM))
    {
      st.system     = new String(sData);

      for (i=0; i<SYSTEM_TOTAL; i++)
      {
        if (st.sKID.equals(SYSTEM_ID[i][0]))
        {
          st.system = new String(SYSTEM_ID[i][1]);
          st.sKEY   = new String(SYSTEM_ID[i][3]);
          st.sName  = new String(st.system);
        }
      }
    }
    if (sIdentifier.equals(SERIES))      { st.series     = new String(sData); }
    if (sIdentifier.equals(SUBSYSTEM))   { st.subSystem  = new String(sData); }
    if (sIdentifier.equals(SUBSERIES))   { st.subSeries  = new String(sData); }
    if (sIdentifier.equals(STAGE))       { st.stage      = new String(sData); }
    if (sIdentifier.equals(SUPER_GROUP)) { st.sprGroup   = new String(sData); }
    if (sIdentifier.equals(GROUP))       { st.sGroup     = new String(sData); }
    if (sIdentifier.equals(SUBGROUP))    { st.subGroup   = new String(sData); }
    if (sIdentifier.equals(FORMATION))   { st.sFormation = new String(sData); }

    return (st);
  }

  /** ParseData()
   * <p> This method will parse the Data Stream for the Individual Well
   *     Header Information
   * @param sIdentifier = The Attribute Identifier
   * @param sData       = Well Header Information String
   */

  private void ParseData(String sIdentifier, String sData)
  {
    // Identification Information

    if (sIdentifier.equals(KID)) { stList.sKID = new String(sData); }
    if (sIdentifier.equals(KEY)) { stList.sKEY = new String(sData); }
    if (sIdentifier.equals(TYPE))
    {
      if (sData.equals("WELL"))    stList.iType = iqstrat.iqstratHeadersStruct.WELL;
      if (sData.equals("OUTCROP")) stList.iType = iqstrat.iqstratHeadersStruct.OUTCROP;
    }
    if (sIdentifier.equals(API_NUMBER))  { stList.sAPI = new String(sData); }
    if (sIdentifier.equals(NAME))
    {
      stList.sName = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(WELL_STATUS))
    {
      stList.status = new String(sData.replaceAll("%20", " "));
    }
    if (sIdentifier.equals(ERROR_VAL))
    {
      iError    = -1;
      sError    = new String(sData);
    }

    // XY Position Information

    if (sIdentifier.equals(LATITUDE))
    {
      stList.dLatitude = cmn.cmnString.stringToDouble(sData);
    }
    if (sIdentifier.equals(LONGITUDE))
    {
      stList.dLongitude = cmn.cmnString.stringToDouble(sData);
    }

    // Z Position Information

    if (sIdentifier.equals(DEPTH))
    {
      stList.depth = cmn.cmnString.stringToDouble(sData);
    }

    if (sIdentifier.equals(GROUND_LEVEL))
    {
      stList.dGL = cmn.cmnString.stringToDouble(sData);
    }
  }

  /* --------------------------------------------------------------- *
   * ------------------- ERROR HANDLING METHODS -------------------- *
   * --------------------------------------------------------------- */

  /** METHOD GetErrorID()
   *  <p> This method will return the error number.
   * @return iError = The Error Identifier
   */

  public int GetErrorID() { return (iError); }

  /** METHOD GetError()
   *  <p> This method will return the error string.
   * @return sError = The Error Text
   */

  public String GetError() { return (sError); }
}