/*
 * @stratTable.java Version 1.1 03/30/2009
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.gui;

import java.awt.*;
import java.lang.*;
import javax.swing.*;
import javax.swing.table.*;

import gui.guiTable;

import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;

/** Class stratTable
 *  <p> This Class will display the Formation Tops.
 *
 *  @version 1.1 03/30/2009
 *  @author  John R. Victorine
 */

public class stratTable
{
  private int    iRows     = 0;
  private int    iColumns  = 13;
  private String sColumn[] = { "Source",
                               "Top",
                               "Base",
                               "Name",
                               "Rank",

                               "System",
                               "Series",
                               "SubSystem",
                               "SubSeries",

                               "Stage",
                               "Group",
                               "Subgroup",
                               "Formation" };

  private int    iColLength[] = {14,
                                  6,
                                  6,
                                 24,
                                 10,

                                 14,
                                 12,
                                 14,
                                 12,

                                 14,
                                 14,
                                 14,
                                 14 };

  private Object   oData[][] = null;
  private guiTable pTable    = null;

  private stratListStruct st        = null;
  private int             iChange   = 0;

  public static final int _ALL_DATA  = 0;
  public static final int _BY_SOURCE = 1;
  public static final int _BY_TABLE  = 2;

  private int             iFilter   = cmn.cmnStruct.ON;
  private String          source    = "";
  private int             iSources  = 0;
  private int             iStatus[] = null;
  private String          sources[] = null;

  /** Constructor stratTable()
   *  <p> This is the Constructor for this class.
   */

  public stratTable()
  {
    pTable = new guiTable( 0, iColumns, sColumn,
                           gui.guiTable._MULTI_SELECTION );

    resetColumnWidths();
    pTable.showHorScroll( true );
  }

  /** Constructor stratTable( stratListStruct st )
   *  <p> This is the Constructor for this class.
   * @param st     = The Stratigraphic Units List Data Structure
   */

  public stratTable( stratListStruct st )
  {
    this.st = st;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        iStatus = new int[st.iCount];
        for (int i=0; i<st.iCount; i++)
        {
          iStatus[i] = -1;
        }
      }
    }

    sources = horizon.strat.stratUtility.getSourceNames( st );
    if (sources != null)
    {
      iSources = sources.length;
      if (iSources > 0) source = new String( sources[0] );
    }

    populateList();

    if (iRows > 0)
      pTable = new guiTable( 0, iRows, iColumns, sColumn, oData,
                             gui.guiTable._MULTI_SELECTION );
    else
      pTable = new guiTable( 0, iColumns, sColumn,
                             gui.guiTable._MULTI_SELECTION );

    resetColumnWidths();
    pTable.showHorScroll( true );

    if (iChange == 0) iRows = 0;
//    if (iRows   > 0)  pTable.setSelectedRow(0);
  }

  /** Method close()
   * <p> This method will force Java to clear allocated memory
   */

  public void close()
  {
    sColumn    = null;
    iColLength = null;
    oData      = null;
    st         = null;
    iStatus    = null;
    sources    = null;

    if (pTable != null)
      pTable.close();
    pTable     = null;
  }

  /** Method resetColumnWidths()
   *  <p> This method will reset the column widths to a specific size.
   */

  private void resetColumnWidths()
  {
    for (int i=0; i<iColumns; i++)
    {
      pTable.resizeColumn( i, iColLength[i] );
    }
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method populateList()
   *  <p> This method will populate the Stratigraphic Units List data.
   */

  private void populateList()
  {
    int i         = 0;
    int iCount    = 0;
    int iTotal    = 0;
    int iContinue = 0;

    iChange = 0;
    iRows   = 1;

    oData = new String[1][iColumns];
    for (i=0; i<iColumns; i++)
    {
      oData[0][i] = new String("");
    }

    if (st != null)
    {
      st = horizon.strat.stratUtility.bubbleSort(st);

      if (st.iCount > 0)
      {
        if (iFilter == cmn.cmnStruct.ON)
        {
          for (i=0; i<st.iCount; i++)
          {
            if (source.equals(st.stItem[i].source))
            {
              iCount++;
            }
          }

          iTotal = iCount;
        }
        else
        {
          iTotal = st.iCount;
        }

        if (iTotal > 0)
        {
          iChange = 1;
          oData   = new Object[iTotal][iColumns];
          iCount  = 0;
        }
      }

      if (iChange > 0)
      {
        for (i=0; i<st.iCount; i++)
        {
          iContinue = 0;

          if (iFilter == cmn.cmnStruct.ON)
          {
            if (source.equals(st.stItem[i].source))
            {
              iContinue = 1;
            }
          }
          else
            iContinue = 1;

          if (iContinue == 1)
          {
            if (iCount < iTotal)
            {
              iStatus[i] = i;

              oData[iCount][0]  = new String( st.stItem[i].source );
              oData[iCount][1]  = new Double( st.stItem[i].depthStart );
              oData[iCount][2]  = new Double( st.stItem[i].depthEnd );
              oData[iCount][3]  = new String( st.stItem[i].sName );
              oData[iCount][4]  = new String(
                  horizon.strat.stratStruct.RANK[st.stItem[i].iRank] );

              oData[iCount][5]  = new String( st.stItem[i].system );
              oData[iCount][6]  = new String( st.stItem[i].series );
              oData[iCount][7]  = new String( st.stItem[i].subSystem );
              oData[iCount][8]  = new String( st.stItem[i].subSeries );

              oData[iCount][9]  = new String( st.stItem[i].stage );
              oData[iCount][10] = new String( st.stItem[i].sGroup );
              oData[iCount][11] = new String( st.stItem[i].subGroup );
              oData[iCount][12] = new String( st.stItem[i].sFormation );

              iCount++;
            }
          }
        }

        iRows = iCount;
      }
    }
  }

  /** Method repopulateList( stratListStruct st )
   *  <p> This method will reload the Stratigraphic Unit List Data
   *      after adding, deleting from the formation name list.
   * @param st     = The Stratigraphic Unit List Data Structure
   */

  public void repopulateList( stratListStruct st )
  {
    this.st = st;

    if (iStatus != null)
      iStatus = null;

    if (st != null)
    {
      if (st.iCount > 0)
      {
        iStatus = new int[st.iCount];
        for (int i=0; i<st.iCount; i++)
        {
          iStatus[i] = -1;
        }
      }
    }

    if (sources != null)
    {
      iSources = 0;
      sources  = null;
    }

    sources = horizon.strat.stratUtility.getSourceNames( st );
    if (sources != null)
    {
      iSources = sources.length;
      if (iSources > 0) source = new String( sources[0] );
    }

    populateList();

    pTable.reload( iRows, oData );

    if (iChange == 0) iRows = 0;
//    if (iRows > 0)    pTable.setSelectedRow(0);
  }

  /** Method setFilter( int iFilter )
   * <p> This method will set the filter ON or OFF
   * @param iFilter = filter ON or OFF
   */

  public void setFilter( int iFilter) { this.iFilter = iFilter; }

  /** Method setSource( String source )
   * <p> This method will set and display the filter option
   * @param source = the source to display
   */

  public void setSource( String source )
  {
    this.source = source;

    for (int i=0; i<st.iCount; i++)
    {
      iStatus[i] = -1;
    }

    populateList();

    pTable.reload( iRows, oData );
  }

  /** Method clear()
   * <p> This method will clear the select settings for multi-select
   */

  public void clear() { pTable.clear(); }

  /* ===================================================================== *
   * -------------------------- GET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method getTotalRows()
   *  <p> This method will return the total rows in the table.
   *  @return Total Number of Rows.
   */

  public int getTotalRows() { return ( iRows ); }

  /** Method getRow()
   *  <p> This method will return the selected data id in the table.
   *  @return selected data id.
   */

  public int getRow() { return ( pTable.getSelectedRow() ); }

  /** Method getRowData()
   *  <p> This method will return the Region Data Structure of the
   *      selected row.
   *  @return stData = Region Data Structure.
   */

  public stratStruct getRowData()
  {
    stratStruct stData = null;

    stData = st.stItem[pTable.getSelectedRow()];

    return ( stData );
  }

  /** Method getAllData()
   *  <p> This method will returns the Region List Data Structure.
   *  @return st = Region List Data Structure.
   */

  public stratListStruct getAllData() { return ( st ); }

  /** Method getSelected( int iSelect )
   * <p> This method will retrieve the selected stratigraphic units from this
   *     table.
   * @param  iSelect = mode of selection
   * @return stList  = Selected Stratigraphic Units.
   */

  public stratListStruct getSelected( int iSelect )
  {
    stratListStruct stList      = null;
    int             i           = 0;
    int             j           = 0;
    int             iCount      = 0;
    int             icnt        = 0;
    int             iSelected[] = null;

    if (st != null)
    {
      switch (iSelect)
      {
        case _ALL_DATA:
          stList = horizon.strat.stratUtility.copyList(st);
          break;

        case _BY_SOURCE:
          for (i=0; i<st.iCount; i++)
          {
            if (source.equals(st.stItem[i].source))
            {
              iCount++;
            }
          }

          if (iCount > 0)
          {
            stList        = new stratListStruct();
            stList.stItem = new stratStruct[iCount];

            iCount = 0;
            for (i=0; i<st.iCount; i++)
            {
              if (source.equals(st.stItem[i].source))
              {
                stList.stItem[iCount] =
                    horizon.strat.stratUtility.copy( st.stItem[i] );
                iCount++;
              }
            }
            stList.iCount = iCount;
          }
          break;

        case _BY_TABLE:
          iSelected = pTable.getSelectedRows();

          if (iSelected != null)
          {
            for (i=0; i<iSelected.length; i++)
            {
              if (iSelected[i] > 0)
                iCount++;
            }
          }

          if (iCount > 0)
          {
            stList        = new stratListStruct();
            stList.stItem = new stratStruct[iCount];

            iCount = 0;
            icnt   = 0;
            for (i=0; i<st.iCount; i++)
            {
              if (iStatus[i] > -1)
              {
                for (j=0; j<iSelected.length; j++)
                {
                  if (icnt == iSelected[j])
                  {
                    stList.stItem[iCount] =
                        horizon.strat.stratUtility.copy(st.stItem[i]);
                    iCount++;
                  }
                }
                icnt++;
              }
            }
            stList.iCount = iCount;
          }
          break;
      }
    }

    return (stList);
  }

  /** Method getScrollPane()
   *  <p> This method will return the scroll pane.
   *  @return jScrollPane - Scroll Pane containing the Table.
   */

  public JScrollPane getScrollPane() { return ( pTable.getScrollPane() ); }
}


/*
 *  @version 1.1 03/30/2009
 *  @author  John Victorine
 */
