/*
 * @stratPlotStateFrame.java Version 1.1 10/28/2008
 *
 * Copyright (c) 2008 Kansas Geological Survey
 * 1930 Constant Avenue, Lawrence, Kansas, 66047, U.S.A.
 * All Rights Reserved.
 */

package horizon.strat.gui;

import java.util.Observable;
import java.awt.*;
import java.awt.image.*;
import java.awt.event.*;

import javax.swing.*;
import javax.swing.event.*;
import javax.swing.border.*;

import horizon.strat.stratListStruct;
import horizon.strat.stratStruct;
import horizon.strat.gui.stratPlotStatePanel;

/** Class stratPlotStateFrame
 *  <p> This Class will display the State Stratigraphic Units available
 *
 *  @version 1.1 10/28/2008
 *  @author  John R. Victorine
 */

public class stratPlotStateFrame extends JFrame implements ActionListener
{
  private Observable      notifier = null; // Observable Object
  private stratListStruct stStrat  = null; // State Stratigraphic Units
  private stratListStruct stICS    = null; // ICS Stratigraphic Units

  private stratPlotStatePanel plot = null;

  private static final int _UNITS   = 30; //17;
  private static final int _LABELS  =  6; //3;
  private static final int _SYSTEMS = 24; //14;

  private static final String NAME[][] = {
    {"10100000", "1000000", "ERA",    "Cenozoic" },
    {"10101000", "1100000", "SYSTEM", "Quaternary" },
    {"10102000", "1200000", "SYSTEM", "Neogene" },
    {"10103000", "1300000", "SYSTEM", "Paleogene" },

    {"10200000", "2000000", "ERA",    "Mesozoic" },
    {"10201000", "2100000", "SYSTEM", "Cretaceous" },
    {"10202000", "2200000", "SYSTEM", "Jurassic" },
    {"10203000", "2300000", "SYSTEM", "Triassic" },

    {"10300000", "3000000", "ERA",    "Paleozoic" },
    {"10301000", "3100000", "SYSTEM", "Permian" },
    {"10302000", "3200000", "SYSTEM", "Carboniferous" },
    {"10303000", "3300000", "SUBSYSTEM", "Pennsylvanian" },
    {"10304000", "3400000", "SUBSYSTEM", "Mississippian" },
    {"10305000", "3500000", "SYSTEM", "Devonian" },
    {"10306000", "3600000", "SYSTEM", "Silurian" },
    {"10307000", "3700000", "SYSTEM", "Ordovician" },
    {"10308000", "3800000", "SYSTEM", "Cambrian" },

    {"20200000", "4000000", "ERA",    "Neoproterozoic" },
    {"20201000", "4100000", "SYSTEM", "Ediacaran" },
    {"20202000", "4200000", "SYSTEM", "Cryogenian" },
    {"20203000", "4300000", "SYSTEM", "Tonian" },

    {"20300000", "5000000", "ERA",    "Mesoproterozoic" },
    {"20301000", "5100000", "SYSTEM", "Stenian" },
    {"20302000", "5200000", "SYSTEM", "Ectasian" },
    {"20303000", "5300000", "SYSTEM", "Calymmian" },

    {"20400000", "6000000", "ERA",    "Paleoproterozoic" },
    {"20401000", "6100000", "SYSTEM", "Statherian" },
    {"20402000", "6200000", "SYSTEM", "Orosirian" },
    {"20403000", "6300000", "SYSTEM", "Rhyacian" },
    {"20404000", "6400000", "SYSTEM", "Siderian" }
  };

  private JButton btnClose   = new JButton();

  private JRadioButton rbAll = new JRadioButton();

  private JPanel       pnlCenter = new JPanel();
  private JPanel       pnl[] = null;
  private JLabel       lbl[] = null;
  private JRadioButton rb[]  = null;

  private JRadioButton rbX5  = new JRadioButton();
  private JRadioButton rbX2  = new JRadioButton();
  private JRadioButton rbX1  = new JRadioButton();

  private JTextField txtRank = new JTextField();
  private JTextField txtName = new JTextField();

  /** Constructor stratPlotStateFrame()
   *  <p> This is the Constructor for this class.
   *  @param notifier = Observable
   *  @param stICS    = ICS Stratigraphic Units List Data Structure
   *  @param stStrat  = the stratigraphic unit list Structure for the State
   *                    stratigraphic chart
   */

  public stratPlotStateFrame( Observable notifier,
                              stratListStruct stICS,
                              stratListStruct stStrat )
  {
    try
    {
      this.notifier = notifier;
      this.stICS    = stICS;
      this.stStrat  = stStrat;

      jbInit();
    }
    catch(Exception e)
    {
      e.printStackTrace();
    }
  }

  /** METHOD jbInit()
   *  <p> This Method will create the Frame for this class.
   *  @throws Exception
   */

  private void jbInit() throws Exception
  {
    int    iLabels   = 0;
    int    iButtons  = 0;
    int    iRed      = 255;
    int    iGreen    = 255;
    int    iBlue     = 255;
    JPanel pnlTop    = new JPanel();
    JPanel pnlLeft   = new JPanel();
    JPanel pnlClose  = new JPanel();
    JPanel pnlSystem = new JPanel();
    JPanel pnlScale  = new JPanel();
    JPanel pnlMouse  = new JPanel();
    JPanel pnlType   = new JPanel();
    JPanel pnlName   = new JPanel();

    JPanel pnlUnits  = new JPanel();
    JPanel pnlAll    = new JPanel();

    JScrollPane scroll = new JScrollPane();

    ButtonGroup groupUnits = new ButtonGroup();
    ButtonGroup groupScale = new ButtonGroup();

    TitledBorder titledBorderScale = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Scale:");

    TitledBorder titledBorderRank = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Rank:");

    TitledBorder titledBorderName = new TitledBorder(
        BorderFactory.createEtchedBorder(Color.white,new Color(165, 163, 151)),
        "Stratigraphic Name:");

    titledBorderScale.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderRank.setTitleFont(new java.awt.Font("Dialog", 1, 11));
    titledBorderName.setTitleFont(new java.awt.Font("Dialog", 1, 11));

    this.setTitle("State Stratigraphic Units");
    this.getContentPane().setLayout(new BorderLayout());

    // Depth Scale & Mouse Position Panel

    pnlTop.setPreferredSize(new Dimension(10, 50));
    pnlTop.setLayout(new BorderLayout());

    pnlScale.setBorder(titledBorderScale);
    pnlScale.setLayout(new GridLayout());

    rbX5.setFont(new java.awt.Font("Dialog", 1, 11));
    rbX5.setHorizontalAlignment(SwingConstants.CENTER);
    rbX5.setText("X 5");
    rbX5.addActionListener(this);

    rbX2.setFont(new java.awt.Font("Dialog", 1, 11));
    rbX2.setHorizontalAlignment(SwingConstants.CENTER);
    rbX2.setText("X 2");
    rbX2.addActionListener(this);

    rbX1.setFont(new java.awt.Font("Dialog", 1, 11));
    rbX1.setHorizontalAlignment(SwingConstants.CENTER);
    rbX1.setSelected(true);
    rbX1.setText("X 1");
    rbX1.addActionListener(this);

    pnlMouse.setLayout(new BorderLayout());

    pnlType.setBorder(titledBorderRank);
    pnlType.setPreferredSize(new Dimension(100, 10));
    pnlType.setLayout(new BorderLayout());

    txtRank.setEditable(false);
    txtRank.setText("");

    pnlName.setBorder(titledBorderName);
    pnlName.setLayout(new BorderLayout());

    txtName.setEditable(false);
    txtName.setText("");

    // Stratigraphic Unit Panel

    pnlLeft.setPreferredSize(new Dimension(180, 10));
    pnlLeft.setLayout(new BorderLayout());

    // Close Button Panel

    pnlClose.setLayout(new BorderLayout());

    btnClose.setFont(new java.awt.Font("Dialog", 1, 11));
    btnClose.setText("Close");
    btnClose.addActionListener(this);

    // State Stratigraphic Units Panel

    pnlCenter.setLayout(new BorderLayout());
    pnlSystem.setLayout(new BorderLayout());

    pnlUnits.setLayout(new GridLayout(_UNITS+1,1));

    // -- Show All Units Panel

    pnlAll.setLayout(new BorderLayout());

    rbAll.setFont(new java.awt.Font("Dialog", 1, 11));
    rbAll.setSelected(true);
    rbAll.setText("Display All");
    rbAll.addActionListener(this);

    groupUnits.add(rbAll);

    pnlUnits.add(pnlAll, null);
    pnlAll.add(rbAll, BorderLayout.WEST);

    // -- Stratigraphic Units Button Panel

    pnl = new JPanel[_UNITS];
    lbl = new JLabel[_LABELS];
    rb  = new JRadioButton[_SYSTEMS];

    for (int i=0; i<_UNITS; i++)
    {
      pnl[i] = new JPanel();
      pnl[i].setLayout(new BorderLayout());

      if (stICS != null)
      {
        for (int j=0; j<stICS.iCount; j++)
        {
          if (NAME[i][3].equals(stICS.stItem[j].sName))
          {
            iRed   = stICS.stItem[j].iRed;
            iGreen = stICS.stItem[j].iGreen;
            iBlue  = stICS.stItem[j].iBlue;

            pnl[i].setBackground(new Color(iRed, iGreen, iBlue));
          }
        }
      }

      if (NAME[i][2].equals("ERA"))
      {
        lbl[iLabels] = new JLabel();
        lbl[iLabels].setFont(new java.awt.Font("Dialog", 1, 11));
        lbl[iLabels].setText(NAME[i][3]);

        pnlUnits.add(pnl[i], null);
        pnl[i].add(lbl[iLabels], BorderLayout.WEST);
        iLabels++;
      }
      else
      {
        rb[iButtons] = new JRadioButton();
        rb[iButtons].setFont(new java.awt.Font("Dialog", 1, 11));
        rb[iButtons].setBackground(new Color(iRed, iGreen, iBlue));
        rb[iButtons].setText(NAME[i][3]);
        rb[iButtons].addActionListener(this);

        rb[iButtons].setEnabled(false);
        rb[iButtons].setForeground(new Color(iRed, iGreen, iBlue));

        if (stStrat != null)
        {
          for (int j=0; j<stStrat.iCount; j++)
          {
            if ((NAME[i][3].equals(stStrat.stItem[j].system)) ||
                (NAME[i][3].equals(stStrat.stItem[j].subSystem)))
            {
              rb[iButtons].setEnabled(true);
              rb[iButtons].setForeground(Color.black);
            }
          }
        }

        groupUnits.add(rb[iButtons]);

        pnlUnits.add(pnl[i], null);
        pnl[i].add(rb[iButtons], BorderLayout.WEST);
        iButtons++;
      }
    }

    // Display State Stratigraphic Units

    plot = new stratPlotStatePanel( this, notifier, stICS, stStrat );
    plot.setTextFields( txtRank, txtName );

    // Attach widgets to frame

    this.getContentPane().add(pnlTop, BorderLayout.NORTH);
    pnlTop.add(pnlScale,   BorderLayout.WEST);
    pnlScale.add(rbX1,     null);
    pnlScale.add(rbX2,     null);
    pnlScale.add(rbX5,     null);

    groupScale.add(rbX1);
    groupScale.add(rbX2);
    groupScale.add(rbX5);

    pnlTop.add(pnlMouse,   BorderLayout.CENTER);
    pnlMouse.add(pnlType,  BorderLayout.WEST);
    pnlType.add(txtRank,   BorderLayout.CENTER);

    pnlMouse.add(pnlName,  BorderLayout.CENTER);
    pnlName.add(txtName,   BorderLayout.SOUTH);

    this.getContentPane().add(pnlLeft, BorderLayout.WEST);
    pnlLeft.add(pnlClose,  BorderLayout.SOUTH);
    pnlClose.add(btnClose, BorderLayout.CENTER);

    pnlLeft.add(pnlSystem, BorderLayout.CENTER);
    pnlSystem.add(scroll,  BorderLayout.CENTER);
    scroll.getViewport().add(pnlUnits, null);

    this.getContentPane().add(pnlCenter, BorderLayout.CENTER);
    pnlCenter.add(plot,    BorderLayout.CENTER);

    // Display the Frame

    this.setSize(new Dimension(525, 700));
    Dimension d = Toolkit.getDefaultToolkit().getScreenSize();

    int iXPos = (d.width  - this.getSize().width) / 2;
//    if (iWidth > 0) iXPos = iXStart + iWidth;
    this.setLocation(iXPos, (d.height - this.getSize().height) / 2);

//    this.setResizable(false);
    this.setVisible(true);
  }

  /* =============================================================== *
   * ------------------------- GET METHODS ------------------------- *
   * =============================================================== */

  /** Method getData()
   * <p> This method will get the State stratigraphic units structure selected
   * @return st = the Stratigraphic unit selected.
   */

  public stratStruct getData()
  {
    stratStruct st = null;

    if (plot != null) st = plot.getData();

    return (st);
  }

  /* ===================================================================== *
   * -------------------------- SET METHODS ------------------------------ *
   * ===================================================================== */

  /** Method setSystem(String sName)
   * <p> This method will set the depth range by system
   * @param sName = name of strat unit
   */

  public void setSystem(String sName)
  {
    int iRank = horizon.strat.stratStruct._NONE;

    for (int i=0; i<_UNITS; i++)
    {
      if (NAME[i][3].equals(sName))
      {
        iRank = horizon.strat.stratStruct._SYSTEM;
        if (NAME[i][2].equals("SUBSYSTEM"))
          iRank = horizon.strat.stratStruct._SUBSYSTEM;
      }
    }

    if (plot != null)
    {
      plot.setSystem(iRank, sName);
    }
    pnlCenter.updateUI();
  }

  /** Method setTextFields()
   * <p> This method will set the textfields for the mouse motion listener
   * @param txtRank = the Rank text field
   * @param txtName = the startigraphic name text field
   */

  public void setTextFields( JTextField txtRank, JTextField txtName )
  {
    if (plot != null)
    {
      plot.setTextFields( txtRank, txtName );
    }
  }

  /* ===================================================================== *
   * ---------------------------- ACTIONS -------------------------------- *
   * ===================================================================== */

  /** Method displayData()
   * <p> This method will display the states stratigraphic chart.
   */

  public void displayData() { this.setVisible(true); }

  /** Method close()
   * <p> This method will close all dialogs that are opened by this class.
   */

  public void close()
  {
    notifier = null;  // Observable Object
    stStrat  = null;  // State Stratigraphic Units
    stICS    = null;  // ICS Stratigraphic Units

    if (plot != null)
      plot.close();
    plot = null;

    btnClose = null;

    rbAll   = null;

    pnl     = null;
    lbl     = null;
    rb      = null;

    rbX5    = null;
    rbX2    = null;
    rbX1    = null;

    txtRank = null;
    txtName = null;

    dispose();
  }

  /** METHOD actionPerformed()
   *  <p> This Method will handle all the actions within the Frame.
   *  @param event - Action Event
   */

  public void actionPerformed(ActionEvent event)
  {
    if (event.getSource() == rbX5)
    {
      plot.setScale(iqstrat.iqstratTracksStruct.SCALE_20);
      pnlCenter.updateUI();
    }
    if (event.getSource() == rbX2)
    {
      plot.setScale(iqstrat.iqstratTracksStruct.SCALE_50);
      pnlCenter.updateUI();
    }
    if (event.getSource() == rbX1)
    {
      plot.setScale(iqstrat.iqstratTracksStruct.SCALE_100);
      pnlCenter.updateUI();
    }

    if (event.getSource() == rbAll)
    {
      setSystem( "ALL" );
    }

    for (int i=0; i<_SYSTEMS; i++)
    {
      if (event.getSource() == rb[i])
      {
        setSystem( rb[i].getText() );
      }
    }

    if (event.getSource() == btnClose)  { this.setVisible(false); }
  }
}

/*
 *  @version 1.1 10/28/2008
 *  @author  John Victorine
 */
